import { PRIMARY_OUTLET } from './shared';
import { UrlSegment, UrlTree } from './url_tree';
import { forEach, shallowEqual } from './utils/collection';
import { TreeNode } from './utils/tree';
export function createUrlTree(route, urlTree, commands, queryParams, fragment) {
    if (commands.length === 0) {
        return tree(urlTree._root, urlTree, queryParams, fragment);
    }
    const normalizedCommands = normalizeCommands(commands);
    if (navigateToRoot(normalizedCommands)) {
        return tree(new TreeNode(urlTree.root, []), urlTree, queryParams, fragment);
    }
    const startingNode = findStartingNode(normalizedCommands, urlTree, route);
    const updated = normalizedCommands.commands.length > 0 ?
        updateMany(startingNode.children.slice(0), normalizedCommands.commands) :
        [];
    const newRoot = constructNewTree(urlTree._root, startingNode, updated);
    return tree(newRoot, urlTree, queryParams, fragment);
}
function tree(root, urlTree, queryParams, fragment) {
    const q = queryParams ? stringify(queryParams) : urlTree.queryParams;
    const f = fragment ? fragment : urlTree.fragment;
    return new UrlTree(root, q, f);
}
function navigateToRoot(normalizedChange) {
    return normalizedChange.isAbsolute && normalizedChange.commands.length === 1 &&
        normalizedChange.commands[0] == '/';
}
class NormalizedNavigationCommands {
    constructor(isAbsolute, numberOfDoubleDots, commands) {
        this.isAbsolute = isAbsolute;
        this.numberOfDoubleDots = numberOfDoubleDots;
        this.commands = commands;
    }
}
function normalizeCommands(commands) {
    if ((typeof commands[0] === 'string') && commands.length === 1 && commands[0] == '/') {
        return new NormalizedNavigationCommands(true, 0, commands);
    }
    let numberOfDoubleDots = 0;
    let isAbsolute = false;
    const res = [];
    for (let i = 0; i < commands.length; ++i) {
        const c = commands[i];
        if (!(typeof c === 'string')) {
            res.push(c);
            continue;
        }
        const parts = c.split('/');
        for (let j = 0; j < parts.length; ++j) {
            let cc = parts[j];
            if (i == 0) {
                if (j == 0 && cc == '.') {
                }
                else if (j == 0 && cc == '') {
                    isAbsolute = true;
                }
                else if (cc == '..') {
                    numberOfDoubleDots++;
                }
                else if (cc != '') {
                    res.push(cc);
                }
            }
            else {
                if (cc != '') {
                    res.push(cc);
                }
            }
        }
    }
    return new NormalizedNavigationCommands(isAbsolute, numberOfDoubleDots, res);
}
function findStartingNode(normalizedChange, urlTree, route) {
    if (normalizedChange.isAbsolute) {
        return urlTree._root;
    }
    else {
        const urlSegment = findUrlSegment(route, urlTree, normalizedChange.numberOfDoubleDots);
        return findMatchingNode(urlSegment, urlTree._root);
    }
}
function findUrlSegment(route, urlTree, numberOfDoubleDots) {
    const urlSegment = route.snapshot._lastUrlSegment;
    const path = urlTree.pathFromRoot(urlSegment);
    if (path.length <= numberOfDoubleDots) {
        throw new Error('Invalid number of \'../\'');
    }
    return path[path.length - 1 - numberOfDoubleDots];
}
function findMatchingNode(segment, node) {
    if (node.value === segment)
        return node;
    for (let c of node.children) {
        const r = findMatchingNode(segment, c);
        if (r)
            return r;
    }
    throw new Error(`Cannot find url segment '${segment}'`);
}
function constructNewTree(node, original, updated) {
    if (node === original) {
        return new TreeNode(node.value, updated);
    }
    else {
        return new TreeNode(node.value, node.children.map(c => constructNewTree(c, original, updated)));
    }
}
function updateMany(nodes, commands) {
    const outlet = getOutlet(commands);
    const nodesInRightOutlet = nodes.filter(c => c.value.outlet === outlet);
    if (nodesInRightOutlet.length > 0) {
        const nodeRightOutlet = nodesInRightOutlet[0];
        nodes[nodes.indexOf(nodeRightOutlet)] = update(nodeRightOutlet, commands);
    }
    else {
        nodes.push(update(null, commands));
    }
    return nodes;
}
function getPath(commands) {
    if (!(typeof commands[0] === 'string'))
        return commands[0];
    const parts = commands[0].toString().split(':');
    return parts.length > 1 ? parts[1] : commands[0];
}
function getOutlet(commands) {
    if (!(typeof commands[0] === 'string'))
        return PRIMARY_OUTLET;
    const parts = commands[0].toString().split(':');
    return parts.length > 1 ? parts[0] : PRIMARY_OUTLET;
}
function update(node, commands) {
    const rest = commands.slice(1);
    const next = rest.length === 0 ? null : rest[0];
    const outlet = getOutlet(commands);
    const path = getPath(commands);
    if (!node && !(typeof next === 'object')) {
        const urlSegment = new UrlSegment(path, {}, outlet);
        const children = rest.length === 0 ? [] : [update(null, rest)];
        return new TreeNode(urlSegment, children);
    }
    else if (!node && typeof next === 'object') {
        const urlSegment = new UrlSegment(path, stringify(next), outlet);
        return recurse(urlSegment, node, rest.slice(1));
    }
    else if (node && outlet !== node.value.outlet) {
        return node;
    }
    else if (node && typeof path === 'object') {
        const newSegment = new UrlSegment(node.value.path, stringify(path), node.value.outlet);
        return recurse(newSegment, node, rest);
    }
    else if (node && typeof next === 'object' && compare(path, stringify(next), node.value)) {
        return recurse(node.value, node, rest.slice(1));
    }
    else if (node && typeof next === 'object') {
        const urlSegment = new UrlSegment(path, stringify(next), outlet);
        return recurse(urlSegment, node, rest.slice(1));
    }
    else if (node && compare(path, {}, node.value)) {
        return recurse(node.value, node, rest);
    }
    else {
        const urlSegment = new UrlSegment(path, {}, outlet);
        return recurse(urlSegment, node, rest);
    }
}
function stringify(params) {
    const res = {};
    forEach(params, (v, k) => res[k] = `${v}`);
    return res;
}
function compare(path, params, segment) {
    return path == segment.path && shallowEqual(params, segment.parameters);
}
function recurse(urlSegment, node, rest) {
    if (rest.length === 0) {
        return new TreeNode(urlSegment, []);
    }
    const children = node ? node.children.slice(0) : [];
    return new TreeNode(urlSegment, updateMany(children, rest));
}
//# sourceMappingURL=data:application/json;base64,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