import { Observable } from 'rxjs/Observable';
import { of } from 'rxjs/observable/of';
import { match } from './apply_redirects';
import { ActivatedRouteSnapshot, RouterStateSnapshot } from './router_state';
import { PRIMARY_OUTLET } from './shared';
import { first, flatten, forEach, merge } from './utils/collection';
import { TreeNode } from './utils/tree';
class CannotRecognize {
}
export function recognize(rootComponentType, config, url) {
    try {
        const match = new MatchResult(rootComponentType, config, [url.root], {}, url._root.children, [], PRIMARY_OUTLET, null, url.root);
        const roots = constructActivatedRoute(match);
        return of(new RouterStateSnapshot(roots[0], url.queryParams, url.fragment));
    }
    catch (e) {
        if (e instanceof CannotRecognize) {
            return new Observable(obs => obs.error(new Error('Cannot match any routes')));
        }
        else {
            return new Observable(obs => obs.error(e));
        }
    }
}
function constructActivatedRoute(match) {
    const activatedRoute = createActivatedRouteSnapshot(match);
    const children = match.leftOverUrl.length > 0 ?
        recognizeMany(match.children, match.leftOverUrl) :
        recognizeLeftOvers(match.children, match.lastUrlSegment);
    checkOutletNameUniqueness(children);
    children.sort((a, b) => {
        if (a.value.outlet === PRIMARY_OUTLET)
            return -1;
        if (b.value.outlet === PRIMARY_OUTLET)
            return 1;
        return a.value.outlet.localeCompare(b.value.outlet);
    });
    return [new TreeNode(activatedRoute, children)];
}
function recognizeLeftOvers(config, lastUrlSegment) {
    if (!config)
        return [];
    const mIndex = matchIndex(config, [], lastUrlSegment);
    return mIndex ? constructActivatedRoute(mIndex) : [];
}
function recognizeMany(config, urls) {
    return flatten(urls.map(url => recognizeOne(config, url)));
}
function createActivatedRouteSnapshot(match) {
    return new ActivatedRouteSnapshot(match.consumedUrlSegments, match.parameters, match.outlet, match.component, match.route, match.lastUrlSegment);
}
function recognizeOne(config, url) {
    const matches = matchNode(config, url);
    for (let match of matches) {
        try {
            const primary = constructActivatedRoute(match);
            const secondary = recognizeMany(config, match.secondary);
            const res = primary.concat(secondary);
            checkOutletNameUniqueness(res);
            return res;
        }
        catch (e) {
            if (!(e instanceof CannotRecognize)) {
                throw e;
            }
        }
    }
    throw new CannotRecognize();
}
function checkOutletNameUniqueness(nodes) {
    let names = {};
    nodes.forEach(n => {
        let routeWithSameOutletName = names[n.value.outlet];
        if (routeWithSameOutletName) {
            const p = routeWithSameOutletName.urlSegments.map(s => s.toString()).join('/');
            const c = n.value.urlSegments.map(s => s.toString()).join('/');
            throw new Error(`Two segments cannot have the same outlet name: '${p}' and '${c}'.`);
        }
        names[n.value.outlet] = n.value;
    });
    return nodes;
}
function matchNode(config, url) {
    const res = [];
    for (let r of config) {
        const m = matchWithParts(r, url);
        if (m) {
            res.push(m);
        }
        else if (r.index) {
            res.push(createIndexMatch(r, [url], url.value));
        }
    }
    return res;
}
function createIndexMatch(r, leftOverUrls, lastUrlSegment) {
    const outlet = r.outlet ? r.outlet : PRIMARY_OUTLET;
    const children = r.children ? r.children : [];
    return new MatchResult(r.component, children, [], lastUrlSegment.parameters, leftOverUrls, [], outlet, r, lastUrlSegment);
}
function matchIndex(config, leftOverUrls, lastUrlSegment) {
    for (let r of config) {
        if (r.index) {
            return createIndexMatch(r, leftOverUrls, lastUrlSegment);
        }
    }
    return null;
}
function matchWithParts(route, url) {
    if (!route.path)
        return null;
    if ((route.outlet ? route.outlet : PRIMARY_OUTLET) !== url.value.outlet)
        return null;
    const path = route.path.startsWith('/') ? route.path.substring(1) : route.path;
    if (path === '**') {
        const consumedUrl = [];
        let u = url;
        while (u) {
            consumedUrl.push(u.value);
            u = first(u.children);
        }
        const last = consumedUrl[consumedUrl.length - 1];
        return new MatchResult(route.component, [], consumedUrl, last.parameters, [], [], PRIMARY_OUTLET, route, last);
    }
    const m = match(route, url);
    if (!m)
        return null;
    const { consumedUrlSegments, lastSegment, lastParent, positionalParamSegments } = m;
    const p = lastSegment.value.parameters;
    const posParams = {};
    forEach(positionalParamSegments, (v, k) => { posParams[k] = v.path; });
    const parameters = merge(p, posParams);
    const secondarySubtrees = lastParent ? lastParent.children.slice(1) : [];
    const children = route.children ? route.children : [];
    const outlet = route.outlet ? route.outlet : PRIMARY_OUTLET;
    return new MatchResult(route.component, children, consumedUrlSegments, parameters, lastSegment.children, secondarySubtrees, outlet, route, lastSegment.value);
}
class MatchResult {
    constructor(component, children, consumedUrlSegments, parameters, leftOverUrl, secondary, outlet, route, lastUrlSegment) {
        this.component = component;
        this.children = children;
        this.consumedUrlSegments = consumedUrlSegments;
        this.parameters = parameters;
        this.leftOverUrl = leftOverUrl;
        this.secondary = secondary;
        this.outlet = outlet;
        this.route = route;
        this.lastUrlSegment = lastUrlSegment;
    }
}
//# sourceMappingURL=data:application/json;base64,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