"use strict";
var Observable_1 = require('rxjs/Observable');
var of_1 = require('rxjs/observable/of');
var apply_redirects_1 = require('./apply_redirects');
var router_state_1 = require('./router_state');
var shared_1 = require('./shared');
var collection_1 = require('./utils/collection');
var tree_1 = require('./utils/tree');
var CannotRecognize = (function () {
    function CannotRecognize() {
    }
    return CannotRecognize;
}());
function recognize(rootComponentType, config, url) {
    try {
        var match_1 = new MatchResult(rootComponentType, config, [url.root], {}, url._root.children, [], shared_1.PRIMARY_OUTLET, null, url.root);
        var roots = constructActivatedRoute(match_1);
        return of_1.of(new router_state_1.RouterStateSnapshot(roots[0], url.queryParams, url.fragment));
    }
    catch (e) {
        if (e instanceof CannotRecognize) {
            return new Observable_1.Observable(function (obs) { return obs.error(new Error('Cannot match any routes')); });
        }
        else {
            return new Observable_1.Observable(function (obs) { return obs.error(e); });
        }
    }
}
exports.recognize = recognize;
function constructActivatedRoute(match) {
    var activatedRoute = createActivatedRouteSnapshot(match);
    var children = match.leftOverUrl.length > 0 ?
        recognizeMany(match.children, match.leftOverUrl) :
        recognizeLeftOvers(match.children, match.lastUrlSegment);
    checkOutletNameUniqueness(children);
    children.sort(function (a, b) {
        if (a.value.outlet === shared_1.PRIMARY_OUTLET)
            return -1;
        if (b.value.outlet === shared_1.PRIMARY_OUTLET)
            return 1;
        return a.value.outlet.localeCompare(b.value.outlet);
    });
    return [new tree_1.TreeNode(activatedRoute, children)];
}
function recognizeLeftOvers(config, lastUrlSegment) {
    if (!config)
        return [];
    var mIndex = matchIndex(config, [], lastUrlSegment);
    return mIndex ? constructActivatedRoute(mIndex) : [];
}
function recognizeMany(config, urls) {
    return collection_1.flatten(urls.map(function (url) { return recognizeOne(config, url); }));
}
function createActivatedRouteSnapshot(match) {
    return new router_state_1.ActivatedRouteSnapshot(match.consumedUrlSegments, match.parameters, match.outlet, match.component, match.route, match.lastUrlSegment);
}
function recognizeOne(config, url) {
    var matches = matchNode(config, url);
    for (var _i = 0, matches_1 = matches; _i < matches_1.length; _i++) {
        var match_2 = matches_1[_i];
        try {
            var primary = constructActivatedRoute(match_2);
            var secondary = recognizeMany(config, match_2.secondary);
            var res = primary.concat(secondary);
            checkOutletNameUniqueness(res);
            return res;
        }
        catch (e) {
            if (!(e instanceof CannotRecognize)) {
                throw e;
            }
        }
    }
    throw new CannotRecognize();
}
function checkOutletNameUniqueness(nodes) {
    var names = {};
    nodes.forEach(function (n) {
        var routeWithSameOutletName = names[n.value.outlet];
        if (routeWithSameOutletName) {
            var p = routeWithSameOutletName.urlSegments.map(function (s) { return s.toString(); }).join('/');
            var c = n.value.urlSegments.map(function (s) { return s.toString(); }).join('/');
            throw new Error("Two segments cannot have the same outlet name: '" + p + "' and '" + c + "'.");
        }
        names[n.value.outlet] = n.value;
    });
    return nodes;
}
function matchNode(config, url) {
    var res = [];
    for (var _i = 0, config_1 = config; _i < config_1.length; _i++) {
        var r = config_1[_i];
        var m = matchWithParts(r, url);
        if (m) {
            res.push(m);
        }
        else if (r.index) {
            res.push(createIndexMatch(r, [url], url.value));
        }
    }
    return res;
}
function createIndexMatch(r, leftOverUrls, lastUrlSegment) {
    var outlet = r.outlet ? r.outlet : shared_1.PRIMARY_OUTLET;
    var children = r.children ? r.children : [];
    return new MatchResult(r.component, children, [], lastUrlSegment.parameters, leftOverUrls, [], outlet, r, lastUrlSegment);
}
function matchIndex(config, leftOverUrls, lastUrlSegment) {
    for (var _i = 0, config_2 = config; _i < config_2.length; _i++) {
        var r = config_2[_i];
        if (r.index) {
            return createIndexMatch(r, leftOverUrls, lastUrlSegment);
        }
    }
    return null;
}
function matchWithParts(route, url) {
    if (!route.path)
        return null;
    if ((route.outlet ? route.outlet : shared_1.PRIMARY_OUTLET) !== url.value.outlet)
        return null;
    var path = route.path.startsWith('/') ? route.path.substring(1) : route.path;
    if (path === '**') {
        var consumedUrl = [];
        var u = url;
        while (u) {
            consumedUrl.push(u.value);
            u = collection_1.first(u.children);
        }
        var last = consumedUrl[consumedUrl.length - 1];
        return new MatchResult(route.component, [], consumedUrl, last.parameters, [], [], shared_1.PRIMARY_OUTLET, route, last);
    }
    var m = apply_redirects_1.match(route, url);
    if (!m)
        return null;
    var consumedUrlSegments = m.consumedUrlSegments, lastSegment = m.lastSegment, lastParent = m.lastParent, positionalParamSegments = m.positionalParamSegments;
    var p = lastSegment.value.parameters;
    var posParams = {};
    collection_1.forEach(positionalParamSegments, function (v, k) { posParams[k] = v.path; });
    var parameters = collection_1.merge(p, posParams);
    var secondarySubtrees = lastParent ? lastParent.children.slice(1) : [];
    var children = route.children ? route.children : [];
    var outlet = route.outlet ? route.outlet : shared_1.PRIMARY_OUTLET;
    return new MatchResult(route.component, children, consumedUrlSegments, parameters, lastSegment.children, secondarySubtrees, outlet, route, lastSegment.value);
}
var MatchResult = (function () {
    function MatchResult(component, children, consumedUrlSegments, parameters, leftOverUrl, secondary, outlet, route, lastUrlSegment) {
        this.component = component;
        this.children = children;
        this.consumedUrlSegments = consumedUrlSegments;
        this.parameters = parameters;
        this.leftOverUrl = leftOverUrl;
        this.secondary = secondary;
        this.outlet = outlet;
        this.route = route;
        this.lastUrlSegment = lastUrlSegment;
    }
    return MatchResult;
}());
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVjb2duaXplLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL3JlY29nbml6ZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQ0EsMkJBQXlCLGlCQUFpQixDQUFDLENBQUE7QUFDM0MsbUJBQWtCLG9CQUFvQixDQUFDLENBQUE7QUFFdkMsZ0NBQW9CLG1CQUFtQixDQUFDLENBQUE7QUFFeEMsNkJBQTBELGdCQUFnQixDQUFDLENBQUE7QUFDM0UsdUJBQTZCLFVBQVUsQ0FBQyxDQUFBO0FBRXhDLDJCQUE2QyxvQkFBb0IsQ0FBQyxDQUFBO0FBQ2xFLHFCQUF1QixjQUFjLENBQUMsQ0FBQTtBQUV0QztJQUFBO0lBQXVCLENBQUM7SUFBRCxzQkFBQztBQUFELENBQUMsQUFBeEIsSUFBd0I7QUFFeEIsbUJBQ0ksaUJBQXVCLEVBQUUsTUFBb0IsRUFBRSxHQUFZO0lBQzdELElBQUksQ0FBQztRQUNILElBQU0sT0FBSyxHQUFHLElBQUksV0FBVyxDQUN6QixpQkFBaUIsRUFBRSxNQUFNLEVBQUUsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUUsRUFBRSxFQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFLEVBQUUsRUFBRSx1QkFBYyxFQUFFLElBQUksRUFDdkYsR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2QsSUFBTSxLQUFLLEdBQUcsdUJBQXVCLENBQUMsT0FBSyxDQUFDLENBQUM7UUFDN0MsTUFBTSxDQUFDLE9BQUUsQ0FBRSxJQUFJLGtDQUFtQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsRUFBRSxHQUFHLENBQUMsV0FBVyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO0lBQy9FLENBQUU7SUFBQSxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ1gsRUFBRSxDQUFDLENBQUMsQ0FBQyxZQUFZLGVBQWUsQ0FBQyxDQUFDLENBQUM7WUFDakMsTUFBTSxDQUFDLElBQUksdUJBQVUsQ0FDakIsVUFBQSxHQUFHLElBQUksT0FBQSxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksS0FBSyxDQUFDLHlCQUF5QixDQUFDLENBQUMsRUFBL0MsQ0FBK0MsQ0FBQyxDQUFDO1FBQzlELENBQUM7UUFBQyxJQUFJLENBQUMsQ0FBQztZQUNOLE1BQU0sQ0FBQyxJQUFJLHVCQUFVLENBQXNCLFVBQUEsR0FBRyxJQUFJLE9BQUEsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsRUFBWixDQUFZLENBQUMsQ0FBQztRQUNsRSxDQUFDO0lBQ0gsQ0FBQztBQUNILENBQUM7QUFoQmUsaUJBQVMsWUFnQnhCLENBQUE7QUFFRCxpQ0FBaUMsS0FBa0I7SUFDakQsSUFBTSxjQUFjLEdBQUcsNEJBQTRCLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0QsSUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQyxNQUFNLEdBQUcsQ0FBQztRQUN6QyxhQUFhLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsV0FBVyxDQUFDO1FBQ2hELGtCQUFrQixDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQzdELHlCQUF5QixDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ3BDLFFBQVEsQ0FBQyxJQUFJLENBQUMsVUFBQyxDQUFDLEVBQUUsQ0FBQztRQUNqQixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLE1BQU0sS0FBSyx1QkFBYyxDQUFDO1lBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ2pELEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxLQUFLLHVCQUFjLENBQUM7WUFBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQ2hELE1BQU0sQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUN0RCxDQUFDLENBQUMsQ0FBQztJQUNILE1BQU0sQ0FBQyxDQUFDLElBQUksZUFBUSxDQUF5QixjQUFjLEVBQUUsUUFBUSxDQUFDLENBQUMsQ0FBQztBQUMxRSxDQUFDO0FBRUQsNEJBQ0ksTUFBZSxFQUFFLGNBQTBCO0lBQzdDLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDO1FBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQztJQUN2QixJQUFNLE1BQU0sR0FBRyxVQUFVLENBQUMsTUFBTSxFQUFFLEVBQUUsRUFBRSxjQUFjLENBQUMsQ0FBQztJQUN0RCxNQUFNLENBQUMsTUFBTSxHQUFHLHVCQUF1QixDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQztBQUN2RCxDQUFDO0FBRUQsdUJBQ0ksTUFBZSxFQUFFLElBQTRCO0lBQy9DLE1BQU0sQ0FBQyxvQkFBTyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsVUFBQSxHQUFHLElBQUksT0FBQSxZQUFZLENBQUMsTUFBTSxFQUFFLEdBQUcsQ0FBQyxFQUF6QixDQUF5QixDQUFDLENBQUMsQ0FBQztBQUM3RCxDQUFDO0FBRUQsc0NBQXNDLEtBQWtCO0lBQ3RELE1BQU0sQ0FBQyxJQUFJLHFDQUFzQixDQUM3QixLQUFLLENBQUMsbUJBQW1CLEVBQUUsS0FBSyxDQUFDLFVBQVUsRUFBRSxLQUFLLENBQUMsTUFBTSxFQUFFLEtBQUssQ0FBQyxTQUFTLEVBQUUsS0FBSyxDQUFDLEtBQUssRUFDdkYsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDO0FBQzVCLENBQUM7QUFFRCxzQkFDSSxNQUFlLEVBQUUsR0FBeUI7SUFDNUMsSUFBTSxPQUFPLEdBQUcsU0FBUyxDQUFDLE1BQU0sRUFBRSxHQUFHLENBQUMsQ0FBQztJQUN2QyxHQUFHLENBQUMsQ0FBYyxVQUFPLEVBQVAsbUJBQU8sRUFBUCxxQkFBTyxFQUFQLElBQU8sQ0FBQztRQUFyQixJQUFJLE9BQUssZ0JBQUE7UUFDWixJQUFJLENBQUM7WUFDSCxJQUFNLE9BQU8sR0FBRyx1QkFBdUIsQ0FBQyxPQUFLLENBQUMsQ0FBQztZQUMvQyxJQUFNLFNBQVMsR0FBRyxhQUFhLENBQUMsTUFBTSxFQUFFLE9BQUssQ0FBQyxTQUFTLENBQUMsQ0FBQztZQUN6RCxJQUFNLEdBQUcsR0FBRyxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBQ3RDLHlCQUF5QixDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQy9CLE1BQU0sQ0FBQyxHQUFHLENBQUM7UUFDYixDQUFFO1FBQUEsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNYLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLFlBQVksZUFBZSxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNwQyxNQUFNLENBQUMsQ0FBQztZQUNWLENBQUM7UUFDSCxDQUFDO0tBQ0Y7SUFDRCxNQUFNLElBQUksZUFBZSxFQUFFLENBQUM7QUFDOUIsQ0FBQztBQUVELG1DQUFtQyxLQUF5QztJQUUxRSxJQUFJLEtBQUssR0FBRyxFQUFFLENBQUM7SUFDZixLQUFLLENBQUMsT0FBTyxDQUFDLFVBQUEsQ0FBQztRQUNiLElBQUksdUJBQXVCLEdBQUcsS0FBSyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDcEQsRUFBRSxDQUFDLENBQUMsdUJBQXVCLENBQUMsQ0FBQyxDQUFDO1lBQzVCLElBQU0sQ0FBQyxHQUFHLHVCQUF1QixDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsVUFBQSxDQUFDLElBQUksT0FBQSxDQUFDLENBQUMsUUFBUSxFQUFFLEVBQVosQ0FBWSxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQy9FLElBQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxVQUFBLENBQUMsSUFBSSxPQUFBLENBQUMsQ0FBQyxRQUFRLEVBQUUsRUFBWixDQUFZLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDL0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxxREFBbUQsQ0FBQyxlQUFVLENBQUMsT0FBSSxDQUFDLENBQUM7UUFDdkYsQ0FBQztRQUNELEtBQUssQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxLQUFLLENBQUM7SUFDbEMsQ0FBQyxDQUFDLENBQUM7SUFDSCxNQUFNLENBQUMsS0FBSyxDQUFDO0FBQ2YsQ0FBQztBQUVELG1CQUFtQixNQUFlLEVBQUUsR0FBeUI7SUFDM0QsSUFBTSxHQUFHLEdBQUcsRUFBRSxDQUFDO0lBQ2YsR0FBRyxDQUFDLENBQVUsVUFBTSxFQUFOLGlCQUFNLEVBQU4sb0JBQU0sRUFBTixJQUFNLENBQUM7UUFBaEIsSUFBSSxDQUFDLGVBQUE7UUFDUixJQUFNLENBQUMsR0FBRyxjQUFjLENBQUMsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQ2pDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDTixHQUFHLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ2QsQ0FBQztRQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztZQUNuQixHQUFHLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQ2xELENBQUM7S0FDRjtJQUNELE1BQU0sQ0FBQyxHQUFHLENBQUM7QUFDYixDQUFDO0FBRUQsMEJBQ0ksQ0FBUSxFQUFFLFlBQW9DLEVBQUUsY0FBMEI7SUFDNUUsSUFBTSxNQUFNLEdBQUcsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsTUFBTSxHQUFHLHVCQUFjLENBQUM7SUFDcEQsSUFBTSxRQUFRLEdBQUcsQ0FBQyxDQUFDLFFBQVEsR0FBRyxDQUFDLENBQUMsUUFBUSxHQUFHLEVBQUUsQ0FBQztJQUM5QyxNQUFNLENBQUMsSUFBSSxXQUFXLENBQ2xCLENBQUMsQ0FBQyxTQUFTLEVBQUUsUUFBUSxFQUFFLEVBQUUsRUFBRSxjQUFjLENBQUMsVUFBVSxFQUFFLFlBQVksRUFBRSxFQUFFLEVBQUUsTUFBTSxFQUFFLENBQUMsRUFDakYsY0FBYyxDQUFDLENBQUM7QUFDdEIsQ0FBQztBQUVELG9CQUNJLE1BQWUsRUFBRSxZQUFvQyxFQUFFLGNBQTBCO0lBRW5GLEdBQUcsQ0FBQyxDQUFVLFVBQU0sRUFBTixpQkFBTSxFQUFOLG9CQUFNLEVBQU4sSUFBTSxDQUFDO1FBQWhCLElBQUksQ0FBQyxlQUFBO1FBQ1IsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7WUFDWixNQUFNLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxFQUFFLFlBQVksRUFBRSxjQUFjLENBQUMsQ0FBQztRQUMzRCxDQUFDO0tBQ0Y7SUFDRCxNQUFNLENBQUMsSUFBSSxDQUFDO0FBQ2QsQ0FBQztBQUVELHdCQUF3QixLQUFZLEVBQUUsR0FBeUI7SUFDN0QsRUFBRSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDO1FBQUMsTUFBTSxDQUFDLElBQUksQ0FBQztJQUM3QixFQUFFLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sR0FBRyx1QkFBYyxDQUFDLEtBQUssR0FBRyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUM7UUFBQyxNQUFNLENBQUMsSUFBSSxDQUFDO0lBRXJGLElBQU0sSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUM7SUFDL0UsRUFBRSxDQUFDLENBQUMsSUFBSSxLQUFLLElBQUksQ0FBQyxDQUFDLENBQUM7UUFDbEIsSUFBTSxXQUFXLEdBQUcsRUFBRSxDQUFDO1FBQ3ZCLElBQUksQ0FBQyxHQUE4QixHQUFHLENBQUM7UUFDdkMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUNULFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQzFCLENBQUMsR0FBRyxrQkFBSyxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUN4QixDQUFDO1FBQ0QsSUFBTSxJQUFJLEdBQUcsV0FBVyxDQUFDLFdBQVcsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDakQsTUFBTSxDQUFDLElBQUksV0FBVyxDQUNsQixLQUFLLENBQUMsU0FBUyxFQUFFLEVBQUUsRUFBRSxXQUFXLEVBQUUsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLHVCQUFjLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQzlGLENBQUM7SUFFRCxJQUFNLENBQUMsR0FBRyx1QkFBSyxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQztJQUM1QixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUM7SUFDYiwrQ0FBbUIsRUFBRSwyQkFBVyxFQUFFLHlCQUFVLEVBQUUsbURBQXVCLENBQU07SUFFbEYsSUFBTSxDQUFDLEdBQUcsV0FBVyxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUM7SUFDdkMsSUFBTSxTQUFTLEdBQUcsRUFBRSxDQUFDO0lBQ3JCLG9CQUFPLENBQUMsdUJBQXVCLEVBQUUsVUFBQyxDQUFDLEVBQUUsQ0FBQyxJQUFPLFNBQVMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDdkUsSUFBTSxVQUFVLEdBQTRCLGtCQUFLLENBQUMsQ0FBQyxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBQ2hFLElBQU0saUJBQWlCLEdBQUcsVUFBVSxHQUFHLFVBQVUsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsQ0FBQztJQUN6RSxJQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLEdBQUcsRUFBRSxDQUFDO0lBQ3RELElBQU0sTUFBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sR0FBRyx1QkFBYyxDQUFDO0lBRTVELE1BQU0sQ0FBQyxJQUFJLFdBQVcsQ0FDbEIsS0FBSyxDQUFDLFNBQVMsRUFBRSxRQUFRLEVBQUUsbUJBQW1CLEVBQUUsVUFBVSxFQUFFLFdBQVcsQ0FBQyxRQUFRLEVBQ2hGLGlCQUFpQixFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsV0FBVyxDQUFDLEtBQUssQ0FBQyxDQUFDO0FBQzNELENBQUM7QUFFRDtJQUNFLHFCQUNXLFNBQXNCLEVBQVMsUUFBaUIsRUFDaEQsbUJBQWlDLEVBQVMsVUFBbUMsRUFDN0UsV0FBbUMsRUFBUyxTQUFpQyxFQUM3RSxNQUFjLEVBQVMsS0FBaUIsRUFBUyxjQUEwQjtRQUgzRSxjQUFTLEdBQVQsU0FBUyxDQUFhO1FBQVMsYUFBUSxHQUFSLFFBQVEsQ0FBUztRQUNoRCx3QkFBbUIsR0FBbkIsbUJBQW1CLENBQWM7UUFBUyxlQUFVLEdBQVYsVUFBVSxDQUF5QjtRQUM3RSxnQkFBVyxHQUFYLFdBQVcsQ0FBd0I7UUFBUyxjQUFTLEdBQVQsU0FBUyxDQUF3QjtRQUM3RSxXQUFNLEdBQU4sTUFBTSxDQUFRO1FBQVMsVUFBSyxHQUFMLEtBQUssQ0FBWTtRQUFTLG1CQUFjLEdBQWQsY0FBYyxDQUFZO0lBQUcsQ0FBQztJQUM1RixrQkFBQztBQUFELENBQUMsQUFORCxJQU1DIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHtUeXBlfSBmcm9tICdAYW5ndWxhci9jb3JlJztcbmltcG9ydCB7T2JzZXJ2YWJsZX0gZnJvbSAncnhqcy9PYnNlcnZhYmxlJztcbmltcG9ydCB7b2YgfSBmcm9tICdyeGpzL29ic2VydmFibGUvb2YnO1xuXG5pbXBvcnQge21hdGNofSBmcm9tICcuL2FwcGx5X3JlZGlyZWN0cyc7XG5pbXBvcnQge1JvdXRlLCBSb3V0ZXJDb25maWd9IGZyb20gJy4vY29uZmlnJztcbmltcG9ydCB7QWN0aXZhdGVkUm91dGVTbmFwc2hvdCwgUm91dGVyU3RhdGVTbmFwc2hvdH0gZnJvbSAnLi9yb3V0ZXJfc3RhdGUnO1xuaW1wb3J0IHtQUklNQVJZX09VVExFVH0gZnJvbSAnLi9zaGFyZWQnO1xuaW1wb3J0IHtVcmxTZWdtZW50LCBVcmxUcmVlfSBmcm9tICcuL3VybF90cmVlJztcbmltcG9ydCB7Zmlyc3QsIGZsYXR0ZW4sIGZvckVhY2gsIG1lcmdlfSBmcm9tICcuL3V0aWxzL2NvbGxlY3Rpb24nO1xuaW1wb3J0IHtUcmVlTm9kZX0gZnJvbSAnLi91dGlscy90cmVlJztcblxuY2xhc3MgQ2Fubm90UmVjb2duaXplIHt9XG5cbmV4cG9ydCBmdW5jdGlvbiByZWNvZ25pemUoXG4gICAgcm9vdENvbXBvbmVudFR5cGU6IFR5cGUsIGNvbmZpZzogUm91dGVyQ29uZmlnLCB1cmw6IFVybFRyZWUpOiBPYnNlcnZhYmxlPFJvdXRlclN0YXRlU25hcHNob3Q+IHtcbiAgdHJ5IHtcbiAgICBjb25zdCBtYXRjaCA9IG5ldyBNYXRjaFJlc3VsdChcbiAgICAgICAgcm9vdENvbXBvbmVudFR5cGUsIGNvbmZpZywgW3VybC5yb290XSwge30sIHVybC5fcm9vdC5jaGlsZHJlbiwgW10sIFBSSU1BUllfT1VUTEVULCBudWxsLFxuICAgICAgICB1cmwucm9vdCk7XG4gICAgY29uc3Qgcm9vdHMgPSBjb25zdHJ1Y3RBY3RpdmF0ZWRSb3V0ZShtYXRjaCk7XG4gICAgcmV0dXJuIG9mIChuZXcgUm91dGVyU3RhdGVTbmFwc2hvdChyb290c1swXSwgdXJsLnF1ZXJ5UGFyYW1zLCB1cmwuZnJhZ21lbnQpKTtcbiAgfSBjYXRjaCAoZSkge1xuICAgIGlmIChlIGluc3RhbmNlb2YgQ2Fubm90UmVjb2duaXplKSB7XG4gICAgICByZXR1cm4gbmV3IE9ic2VydmFibGU8Um91dGVyU3RhdGVTbmFwc2hvdD4oXG4gICAgICAgICAgb2JzID0+IG9icy5lcnJvcihuZXcgRXJyb3IoJ0Nhbm5vdCBtYXRjaCBhbnkgcm91dGVzJykpKTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIG5ldyBPYnNlcnZhYmxlPFJvdXRlclN0YXRlU25hcHNob3Q+KG9icyA9PiBvYnMuZXJyb3IoZSkpO1xuICAgIH1cbiAgfVxufVxuXG5mdW5jdGlvbiBjb25zdHJ1Y3RBY3RpdmF0ZWRSb3V0ZShtYXRjaDogTWF0Y2hSZXN1bHQpOiBUcmVlTm9kZTxBY3RpdmF0ZWRSb3V0ZVNuYXBzaG90PltdIHtcbiAgY29uc3QgYWN0aXZhdGVkUm91dGUgPSBjcmVhdGVBY3RpdmF0ZWRSb3V0ZVNuYXBzaG90KG1hdGNoKTtcbiAgY29uc3QgY2hpbGRyZW4gPSBtYXRjaC5sZWZ0T3ZlclVybC5sZW5ndGggPiAwID9cbiAgICAgIHJlY29nbml6ZU1hbnkobWF0Y2guY2hpbGRyZW4sIG1hdGNoLmxlZnRPdmVyVXJsKSA6XG4gICAgICByZWNvZ25pemVMZWZ0T3ZlcnMobWF0Y2guY2hpbGRyZW4sIG1hdGNoLmxhc3RVcmxTZWdtZW50KTtcbiAgY2hlY2tPdXRsZXROYW1lVW5pcXVlbmVzcyhjaGlsZHJlbik7XG4gIGNoaWxkcmVuLnNvcnQoKGEsIGIpID0+IHtcbiAgICBpZiAoYS52YWx1ZS5vdXRsZXQgPT09IFBSSU1BUllfT1VUTEVUKSByZXR1cm4gLTE7XG4gICAgaWYgKGIudmFsdWUub3V0bGV0ID09PSBQUklNQVJZX09VVExFVCkgcmV0dXJuIDE7XG4gICAgcmV0dXJuIGEudmFsdWUub3V0bGV0LmxvY2FsZUNvbXBhcmUoYi52YWx1ZS5vdXRsZXQpO1xuICB9KTtcbiAgcmV0dXJuIFtuZXcgVHJlZU5vZGU8QWN0aXZhdGVkUm91dGVTbmFwc2hvdD4oYWN0aXZhdGVkUm91dGUsIGNoaWxkcmVuKV07XG59XG5cbmZ1bmN0aW9uIHJlY29nbml6ZUxlZnRPdmVycyhcbiAgICBjb25maWc6IFJvdXRlW10sIGxhc3RVcmxTZWdtZW50OiBVcmxTZWdtZW50KTogVHJlZU5vZGU8QWN0aXZhdGVkUm91dGVTbmFwc2hvdD5bXSB7XG4gIGlmICghY29uZmlnKSByZXR1cm4gW107XG4gIGNvbnN0IG1JbmRleCA9IG1hdGNoSW5kZXgoY29uZmlnLCBbXSwgbGFzdFVybFNlZ21lbnQpO1xuICByZXR1cm4gbUluZGV4ID8gY29uc3RydWN0QWN0aXZhdGVkUm91dGUobUluZGV4KSA6IFtdO1xufVxuXG5mdW5jdGlvbiByZWNvZ25pemVNYW55KFxuICAgIGNvbmZpZzogUm91dGVbXSwgdXJsczogVHJlZU5vZGU8VXJsU2VnbWVudD5bXSk6IFRyZWVOb2RlPEFjdGl2YXRlZFJvdXRlU25hcHNob3Q+W10ge1xuICByZXR1cm4gZmxhdHRlbih1cmxzLm1hcCh1cmwgPT4gcmVjb2duaXplT25lKGNvbmZpZywgdXJsKSkpO1xufVxuXG5mdW5jdGlvbiBjcmVhdGVBY3RpdmF0ZWRSb3V0ZVNuYXBzaG90KG1hdGNoOiBNYXRjaFJlc3VsdCk6IEFjdGl2YXRlZFJvdXRlU25hcHNob3Qge1xuICByZXR1cm4gbmV3IEFjdGl2YXRlZFJvdXRlU25hcHNob3QoXG4gICAgICBtYXRjaC5jb25zdW1lZFVybFNlZ21lbnRzLCBtYXRjaC5wYXJhbWV0ZXJzLCBtYXRjaC5vdXRsZXQsIG1hdGNoLmNvbXBvbmVudCwgbWF0Y2gucm91dGUsXG4gICAgICBtYXRjaC5sYXN0VXJsU2VnbWVudCk7XG59XG5cbmZ1bmN0aW9uIHJlY29nbml6ZU9uZShcbiAgICBjb25maWc6IFJvdXRlW10sIHVybDogVHJlZU5vZGU8VXJsU2VnbWVudD4pOiBUcmVlTm9kZTxBY3RpdmF0ZWRSb3V0ZVNuYXBzaG90PltdIHtcbiAgY29uc3QgbWF0Y2hlcyA9IG1hdGNoTm9kZShjb25maWcsIHVybCk7XG4gIGZvciAobGV0IG1hdGNoIG9mIG1hdGNoZXMpIHtcbiAgICB0cnkge1xuICAgICAgY29uc3QgcHJpbWFyeSA9IGNvbnN0cnVjdEFjdGl2YXRlZFJvdXRlKG1hdGNoKTtcbiAgICAgIGNvbnN0IHNlY29uZGFyeSA9IHJlY29nbml6ZU1hbnkoY29uZmlnLCBtYXRjaC5zZWNvbmRhcnkpO1xuICAgICAgY29uc3QgcmVzID0gcHJpbWFyeS5jb25jYXQoc2Vjb25kYXJ5KTtcbiAgICAgIGNoZWNrT3V0bGV0TmFtZVVuaXF1ZW5lc3MocmVzKTtcbiAgICAgIHJldHVybiByZXM7XG4gICAgfSBjYXRjaCAoZSkge1xuICAgICAgaWYgKCEoZSBpbnN0YW5jZW9mIENhbm5vdFJlY29nbml6ZSkpIHtcbiAgICAgICAgdGhyb3cgZTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbiAgdGhyb3cgbmV3IENhbm5vdFJlY29nbml6ZSgpO1xufVxuXG5mdW5jdGlvbiBjaGVja091dGxldE5hbWVVbmlxdWVuZXNzKG5vZGVzOiBUcmVlTm9kZTxBY3RpdmF0ZWRSb3V0ZVNuYXBzaG90PltdKTpcbiAgICBUcmVlTm9kZTxBY3RpdmF0ZWRSb3V0ZVNuYXBzaG90PltdIHtcbiAgbGV0IG5hbWVzID0ge307XG4gIG5vZGVzLmZvckVhY2gobiA9PiB7XG4gICAgbGV0IHJvdXRlV2l0aFNhbWVPdXRsZXROYW1lID0gbmFtZXNbbi52YWx1ZS5vdXRsZXRdO1xuICAgIGlmIChyb3V0ZVdpdGhTYW1lT3V0bGV0TmFtZSkge1xuICAgICAgY29uc3QgcCA9IHJvdXRlV2l0aFNhbWVPdXRsZXROYW1lLnVybFNlZ21lbnRzLm1hcChzID0+IHMudG9TdHJpbmcoKSkuam9pbignLycpO1xuICAgICAgY29uc3QgYyA9IG4udmFsdWUudXJsU2VnbWVudHMubWFwKHMgPT4gcy50b1N0cmluZygpKS5qb2luKCcvJyk7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFR3byBzZWdtZW50cyBjYW5ub3QgaGF2ZSB0aGUgc2FtZSBvdXRsZXQgbmFtZTogJyR7cH0nIGFuZCAnJHtjfScuYCk7XG4gICAgfVxuICAgIG5hbWVzW24udmFsdWUub3V0bGV0XSA9IG4udmFsdWU7XG4gIH0pO1xuICByZXR1cm4gbm9kZXM7XG59XG5cbmZ1bmN0aW9uIG1hdGNoTm9kZShjb25maWc6IFJvdXRlW10sIHVybDogVHJlZU5vZGU8VXJsU2VnbWVudD4pOiBNYXRjaFJlc3VsdFtdIHtcbiAgY29uc3QgcmVzID0gW107XG4gIGZvciAobGV0IHIgb2YgY29uZmlnKSB7XG4gICAgY29uc3QgbSA9IG1hdGNoV2l0aFBhcnRzKHIsIHVybCk7XG4gICAgaWYgKG0pIHtcbiAgICAgIHJlcy5wdXNoKG0pO1xuICAgIH0gZWxzZSBpZiAoci5pbmRleCkge1xuICAgICAgcmVzLnB1c2goY3JlYXRlSW5kZXhNYXRjaChyLCBbdXJsXSwgdXJsLnZhbHVlKSk7XG4gICAgfVxuICB9XG4gIHJldHVybiByZXM7XG59XG5cbmZ1bmN0aW9uIGNyZWF0ZUluZGV4TWF0Y2goXG4gICAgcjogUm91dGUsIGxlZnRPdmVyVXJsczogVHJlZU5vZGU8VXJsU2VnbWVudD5bXSwgbGFzdFVybFNlZ21lbnQ6IFVybFNlZ21lbnQpOiBNYXRjaFJlc3VsdCB7XG4gIGNvbnN0IG91dGxldCA9IHIub3V0bGV0ID8gci5vdXRsZXQgOiBQUklNQVJZX09VVExFVDtcbiAgY29uc3QgY2hpbGRyZW4gPSByLmNoaWxkcmVuID8gci5jaGlsZHJlbiA6IFtdO1xuICByZXR1cm4gbmV3IE1hdGNoUmVzdWx0KFxuICAgICAgci5jb21wb25lbnQsIGNoaWxkcmVuLCBbXSwgbGFzdFVybFNlZ21lbnQucGFyYW1ldGVycywgbGVmdE92ZXJVcmxzLCBbXSwgb3V0bGV0LCByLFxuICAgICAgbGFzdFVybFNlZ21lbnQpO1xufVxuXG5mdW5jdGlvbiBtYXRjaEluZGV4KFxuICAgIGNvbmZpZzogUm91dGVbXSwgbGVmdE92ZXJVcmxzOiBUcmVlTm9kZTxVcmxTZWdtZW50PltdLCBsYXN0VXJsU2VnbWVudDogVXJsU2VnbWVudCk6IE1hdGNoUmVzdWx0fFxuICAgIG51bGwge1xuICBmb3IgKGxldCByIG9mIGNvbmZpZykge1xuICAgIGlmIChyLmluZGV4KSB7XG4gICAgICByZXR1cm4gY3JlYXRlSW5kZXhNYXRjaChyLCBsZWZ0T3ZlclVybHMsIGxhc3RVcmxTZWdtZW50KTtcbiAgICB9XG4gIH1cbiAgcmV0dXJuIG51bGw7XG59XG5cbmZ1bmN0aW9uIG1hdGNoV2l0aFBhcnRzKHJvdXRlOiBSb3V0ZSwgdXJsOiBUcmVlTm9kZTxVcmxTZWdtZW50Pik6IE1hdGNoUmVzdWx0fG51bGwge1xuICBpZiAoIXJvdXRlLnBhdGgpIHJldHVybiBudWxsO1xuICBpZiAoKHJvdXRlLm91dGxldCA/IHJvdXRlLm91dGxldCA6IFBSSU1BUllfT1VUTEVUKSAhPT0gdXJsLnZhbHVlLm91dGxldCkgcmV0dXJuIG51bGw7XG5cbiAgY29uc3QgcGF0aCA9IHJvdXRlLnBhdGguc3RhcnRzV2l0aCgnLycpID8gcm91dGUucGF0aC5zdWJzdHJpbmcoMSkgOiByb3V0ZS5wYXRoO1xuICBpZiAocGF0aCA9PT0gJyoqJykge1xuICAgIGNvbnN0IGNvbnN1bWVkVXJsID0gW107XG4gICAgbGV0IHU6IFRyZWVOb2RlPFVybFNlZ21lbnQ+fG51bGwgPSB1cmw7XG4gICAgd2hpbGUgKHUpIHtcbiAgICAgIGNvbnN1bWVkVXJsLnB1c2godS52YWx1ZSk7XG4gICAgICB1ID0gZmlyc3QodS5jaGlsZHJlbik7XG4gICAgfVxuICAgIGNvbnN0IGxhc3QgPSBjb25zdW1lZFVybFtjb25zdW1lZFVybC5sZW5ndGggLSAxXTtcbiAgICByZXR1cm4gbmV3IE1hdGNoUmVzdWx0KFxuICAgICAgICByb3V0ZS5jb21wb25lbnQsIFtdLCBjb25zdW1lZFVybCwgbGFzdC5wYXJhbWV0ZXJzLCBbXSwgW10sIFBSSU1BUllfT1VUTEVULCByb3V0ZSwgbGFzdCk7XG4gIH1cblxuICBjb25zdCBtID0gbWF0Y2gocm91dGUsIHVybCk7XG4gIGlmICghbSkgcmV0dXJuIG51bGw7XG4gIGNvbnN0IHtjb25zdW1lZFVybFNlZ21lbnRzLCBsYXN0U2VnbWVudCwgbGFzdFBhcmVudCwgcG9zaXRpb25hbFBhcmFtU2VnbWVudHN9ID0gbTtcblxuICBjb25zdCBwID0gbGFzdFNlZ21lbnQudmFsdWUucGFyYW1ldGVycztcbiAgY29uc3QgcG9zUGFyYW1zID0ge307XG4gIGZvckVhY2gocG9zaXRpb25hbFBhcmFtU2VnbWVudHMsICh2LCBrKSA9PiB7IHBvc1BhcmFtc1trXSA9IHYucGF0aDsgfSk7XG4gIGNvbnN0IHBhcmFtZXRlcnMgPSA8e1trZXk6IHN0cmluZ106IHN0cmluZ30+bWVyZ2UocCwgcG9zUGFyYW1zKTtcbiAgY29uc3Qgc2Vjb25kYXJ5U3VidHJlZXMgPSBsYXN0UGFyZW50ID8gbGFzdFBhcmVudC5jaGlsZHJlbi5zbGljZSgxKSA6IFtdO1xuICBjb25zdCBjaGlsZHJlbiA9IHJvdXRlLmNoaWxkcmVuID8gcm91dGUuY2hpbGRyZW4gOiBbXTtcbiAgY29uc3Qgb3V0bGV0ID0gcm91dGUub3V0bGV0ID8gcm91dGUub3V0bGV0IDogUFJJTUFSWV9PVVRMRVQ7XG5cbiAgcmV0dXJuIG5ldyBNYXRjaFJlc3VsdChcbiAgICAgIHJvdXRlLmNvbXBvbmVudCwgY2hpbGRyZW4sIGNvbnN1bWVkVXJsU2VnbWVudHMsIHBhcmFtZXRlcnMsIGxhc3RTZWdtZW50LmNoaWxkcmVuLFxuICAgICAgc2Vjb25kYXJ5U3VidHJlZXMsIG91dGxldCwgcm91dGUsIGxhc3RTZWdtZW50LnZhbHVlKTtcbn1cblxuY2xhc3MgTWF0Y2hSZXN1bHQge1xuICBjb25zdHJ1Y3RvcihcbiAgICAgIHB1YmxpYyBjb21wb25lbnQ6IFR5cGV8c3RyaW5nLCBwdWJsaWMgY2hpbGRyZW46IFJvdXRlW10sXG4gICAgICBwdWJsaWMgY29uc3VtZWRVcmxTZWdtZW50czogVXJsU2VnbWVudFtdLCBwdWJsaWMgcGFyYW1ldGVyczoge1trZXk6IHN0cmluZ106IHN0cmluZ30sXG4gICAgICBwdWJsaWMgbGVmdE92ZXJVcmw6IFRyZWVOb2RlPFVybFNlZ21lbnQ+W10sIHB1YmxpYyBzZWNvbmRhcnk6IFRyZWVOb2RlPFVybFNlZ21lbnQ+W10sXG4gICAgICBwdWJsaWMgb3V0bGV0OiBzdHJpbmcsIHB1YmxpYyByb3V0ZTogUm91dGV8bnVsbCwgcHVibGljIGxhc3RVcmxTZWdtZW50OiBVcmxTZWdtZW50KSB7fVxufSJdfQ==