"use strict";
var Observable_1 = require('rxjs/Observable');
var of_1 = require('rxjs/observable/of');
var shared_1 = require('./shared');
var url_tree_1 = require('./url_tree');
var NoMatch = (function () {
    function NoMatch(segment) {
        if (segment === void 0) { segment = null; }
        this.segment = segment;
    }
    return NoMatch;
}());
var GlobalRedirect = (function () {
    function GlobalRedirect(paths) {
        this.paths = paths;
    }
    return GlobalRedirect;
}());
function applyRedirects(urlTree, config) {
    try {
        return createUrlTree(urlTree, expandSegment(config, urlTree.root, shared_1.PRIMARY_OUTLET));
    }
    catch (e) {
        if (e instanceof GlobalRedirect) {
            return createUrlTree(urlTree, new url_tree_1.UrlSegment([], (_a = {}, _a[shared_1.PRIMARY_OUTLET] = new url_tree_1.UrlSegment(e.paths, {}), _a)));
        }
        else if (e instanceof NoMatch) {
            return new Observable_1.Observable(function (obs) { return obs.error(new Error("Cannot match any routes: '" + e.segment + "'")); });
        }
        else {
            return new Observable_1.Observable(function (obs) { return obs.error(e); });
        }
    }
    var _a;
}
exports.applyRedirects = applyRedirects;
function createUrlTree(urlTree, root) {
    return of_1.of(new url_tree_1.UrlTree(root, urlTree.queryParams, urlTree.fragment));
}
function expandSegment(routes, segment, outlet) {
    if (segment.pathsWithParams.length === 0 && Object.keys(segment.children).length > 0) {
        return new url_tree_1.UrlSegment([], expandSegmentChildren(routes, segment));
    }
    else {
        return expandPathsWithParams(segment, routes, segment.pathsWithParams, outlet, true);
    }
}
function expandSegmentChildren(routes, segment) {
    return url_tree_1.mapChildren(segment, function (child, childOutlet) { return expandSegment(routes, child, childOutlet); });
}
function expandPathsWithParams(segment, routes, paths, outlet, allowRedirects) {
    for (var _i = 0, routes_1 = routes; _i < routes_1.length; _i++) {
        var r = routes_1[_i];
        try {
            return expandPathsWithParamsAgainstRoute(segment, routes, r, paths, outlet, allowRedirects);
        }
        catch (e) {
            if (!(e instanceof NoMatch))
                throw e;
        }
    }
    throw new NoMatch(segment);
}
function expandPathsWithParamsAgainstRoute(segment, routes, route, paths, outlet, allowRedirects) {
    if ((route.outlet ? route.outlet : shared_1.PRIMARY_OUTLET) !== outlet)
        throw new NoMatch();
    if (route.redirectTo && !allowRedirects)
        throw new NoMatch();
    if (route.redirectTo) {
        return expandPathsWithParamsAgainstRouteUsingRedirect(segment, routes, route, paths, outlet);
    }
    else {
        return matchPathsWithParamsAgainstRoute(segment, route, paths);
    }
}
function expandPathsWithParamsAgainstRouteUsingRedirect(segment, routes, route, paths, outlet) {
    if (route.path === '**') {
        return expandWildCardWithParamsAgainstRouteUsingRedirect(route);
    }
    else {
        return expandRegularPathWithParamsAgainstRouteUsingRedirect(segment, routes, route, paths, outlet);
    }
}
function expandWildCardWithParamsAgainstRouteUsingRedirect(route) {
    var newPaths = applyRedirectCommands([], route.redirectTo, {});
    if (route.redirectTo.startsWith('/')) {
        throw new GlobalRedirect(newPaths);
    }
    else {
        return new url_tree_1.UrlSegment(newPaths, {});
    }
}
function expandRegularPathWithParamsAgainstRouteUsingRedirect(segment, routes, route, paths, outlet) {
    var _a = match(segment, route, paths), consumedPaths = _a.consumedPaths, lastChild = _a.lastChild, positionalParamSegments = _a.positionalParamSegments;
    var newPaths = applyRedirectCommands(consumedPaths, route.redirectTo, positionalParamSegments);
    if (route.redirectTo.startsWith('/')) {
        throw new GlobalRedirect(newPaths);
    }
    else {
        return expandPathsWithParams(segment, routes, newPaths.concat(paths.slice(lastChild)), outlet, false);
    }
}
function matchPathsWithParamsAgainstRoute(segment, route, paths) {
    if (route.path === '**') {
        return new url_tree_1.UrlSegment(paths, {});
    }
    else {
        var _a = match(segment, route, paths), consumedPaths = _a.consumedPaths, lastChild = _a.lastChild;
        var childConfig = route.children ? route.children : [];
        var slicedPath = paths.slice(lastChild);
        if (childConfig.length === 0 && slicedPath.length === 0) {
            return new url_tree_1.UrlSegment(consumedPaths, {});
        }
        else if (slicedPath.length === 0 && Object.keys(segment.children).length > 0) {
            var children = expandSegmentChildren(childConfig, segment);
            return new url_tree_1.UrlSegment(consumedPaths, children);
        }
        else {
            var cs = expandPathsWithParams(segment, childConfig, slicedPath, shared_1.PRIMARY_OUTLET, true);
            return new url_tree_1.UrlSegment(consumedPaths.concat(cs.pathsWithParams), cs.children);
        }
    }
}
function match(segment, route, paths) {
    if (route.index || route.path === '' || route.path === '/') {
        if (route.terminal && (Object.keys(segment.children).length > 0 || paths.length > 0)) {
            throw new NoMatch();
        }
        else {
            return { consumedPaths: [], lastChild: 0, positionalParamSegments: {} };
        }
    }
    var path = route.path.startsWith('/') ? route.path.substring(1) : route.path;
    var parts = path.split('/');
    var positionalParamSegments = {};
    var consumedPaths = [];
    var currentIndex = 0;
    for (var i = 0; i < parts.length; ++i) {
        if (currentIndex >= paths.length)
            throw new NoMatch();
        var current = paths[currentIndex];
        var p = parts[i];
        var isPosParam = p.startsWith(':');
        if (!isPosParam && p !== current.path)
            throw new NoMatch();
        if (isPosParam) {
            positionalParamSegments[p.substring(1)] = current;
        }
        consumedPaths.push(current);
        currentIndex++;
    }
    if (route.terminal && (Object.keys(segment.children).length > 0 || currentIndex < paths.length)) {
        throw new NoMatch();
    }
    return { consumedPaths: consumedPaths, lastChild: currentIndex, positionalParamSegments: positionalParamSegments };
}
function applyRedirectCommands(paths, redirectTo, posParams) {
    if (redirectTo.startsWith('/')) {
        var parts = redirectTo.substring(1).split('/');
        return createPaths(redirectTo, parts, paths, posParams);
    }
    else {
        var parts = redirectTo.split('/');
        return createPaths(redirectTo, parts, paths, posParams);
    }
}
function createPaths(redirectTo, parts, segments, posParams) {
    return parts.map(function (p) { return p.startsWith(':') ? findPosParam(p, posParams, redirectTo) :
        findOrCreatePath(p, segments); });
}
function findPosParam(part, posParams, redirectTo) {
    var paramName = part.substring(1);
    var pos = posParams[paramName];
    if (!pos)
        throw new Error("Cannot redirect to '" + redirectTo + "'. Cannot find '" + part + "'.");
    return pos;
}
function findOrCreatePath(part, paths) {
    var matchingIndex = paths.findIndex(function (s) { return s.path === part; });
    if (matchingIndex > -1) {
        var r = paths[matchingIndex];
        paths.splice(matchingIndex);
        return r;
    }
    else {
        return new url_tree_1.UrlPathWithParams(part, {});
    }
}
//# sourceMappingURL=data:application/json;base64,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