import { Observable } from 'rxjs/Observable';
import { of } from 'rxjs/observable/of';
import { PRIMARY_OUTLET } from './shared';
import { UrlPathWithParams, UrlSegment, UrlTree, mapChildren } from './url_tree';
class NoMatch {
    constructor(segment = null) {
        this.segment = segment;
    }
}
class GlobalRedirect {
    constructor(paths) {
        this.paths = paths;
    }
}
export function applyRedirects(urlTree, config) {
    try {
        return createUrlTree(urlTree, expandSegment(config, urlTree.root, PRIMARY_OUTLET));
    }
    catch (e) {
        if (e instanceof GlobalRedirect) {
            return createUrlTree(urlTree, new UrlSegment([], { [PRIMARY_OUTLET]: new UrlSegment(e.paths, {}) }));
        }
        else if (e instanceof NoMatch) {
            return new Observable(obs => obs.error(new Error(`Cannot match any routes: '${e.segment}'`)));
        }
        else {
            return new Observable(obs => obs.error(e));
        }
    }
}
function createUrlTree(urlTree, root) {
    return of(new UrlTree(root, urlTree.queryParams, urlTree.fragment));
}
function expandSegment(routes, segment, outlet) {
    if (segment.pathsWithParams.length === 0 && Object.keys(segment.children).length > 0) {
        return new UrlSegment([], expandSegmentChildren(routes, segment));
    }
    else {
        return expandPathsWithParams(segment, routes, segment.pathsWithParams, outlet, true);
    }
}
function expandSegmentChildren(routes, segment) {
    return mapChildren(segment, (child, childOutlet) => expandSegment(routes, child, childOutlet));
}
function expandPathsWithParams(segment, routes, paths, outlet, allowRedirects) {
    for (let r of routes) {
        try {
            return expandPathsWithParamsAgainstRoute(segment, routes, r, paths, outlet, allowRedirects);
        }
        catch (e) {
            if (!(e instanceof NoMatch))
                throw e;
        }
    }
    throw new NoMatch(segment);
}
function expandPathsWithParamsAgainstRoute(segment, routes, route, paths, outlet, allowRedirects) {
    if ((route.outlet ? route.outlet : PRIMARY_OUTLET) !== outlet)
        throw new NoMatch();
    if (route.redirectTo && !allowRedirects)
        throw new NoMatch();
    if (route.redirectTo) {
        return expandPathsWithParamsAgainstRouteUsingRedirect(segment, routes, route, paths, outlet);
    }
    else {
        return matchPathsWithParamsAgainstRoute(segment, route, paths);
    }
}
function expandPathsWithParamsAgainstRouteUsingRedirect(segment, routes, route, paths, outlet) {
    if (route.path === '**') {
        return expandWildCardWithParamsAgainstRouteUsingRedirect(route);
    }
    else {
        return expandRegularPathWithParamsAgainstRouteUsingRedirect(segment, routes, route, paths, outlet);
    }
}
function expandWildCardWithParamsAgainstRouteUsingRedirect(route) {
    const newPaths = applyRedirectCommands([], route.redirectTo, {});
    if (route.redirectTo.startsWith('/')) {
        throw new GlobalRedirect(newPaths);
    }
    else {
        return new UrlSegment(newPaths, {});
    }
}
function expandRegularPathWithParamsAgainstRouteUsingRedirect(segment, routes, route, paths, outlet) {
    const { consumedPaths, lastChild, positionalParamSegments } = match(segment, route, paths);
    const newPaths = applyRedirectCommands(consumedPaths, route.redirectTo, positionalParamSegments);
    if (route.redirectTo.startsWith('/')) {
        throw new GlobalRedirect(newPaths);
    }
    else {
        return expandPathsWithParams(segment, routes, newPaths.concat(paths.slice(lastChild)), outlet, false);
    }
}
function matchPathsWithParamsAgainstRoute(segment, route, paths) {
    if (route.path === '**') {
        return new UrlSegment(paths, {});
    }
    else {
        const { consumedPaths, lastChild } = match(segment, route, paths);
        const childConfig = route.children ? route.children : [];
        const slicedPath = paths.slice(lastChild);
        if (childConfig.length === 0 && slicedPath.length === 0) {
            return new UrlSegment(consumedPaths, {});
        }
        else if (slicedPath.length === 0 && Object.keys(segment.children).length > 0) {
            const children = expandSegmentChildren(childConfig, segment);
            return new UrlSegment(consumedPaths, children);
        }
        else {
            const cs = expandPathsWithParams(segment, childConfig, slicedPath, PRIMARY_OUTLET, true);
            return new UrlSegment(consumedPaths.concat(cs.pathsWithParams), cs.children);
        }
    }
}
function match(segment, route, paths) {
    if (route.index || route.path === '' || route.path === '/') {
        if (route.terminal && (Object.keys(segment.children).length > 0 || paths.length > 0)) {
            throw new NoMatch();
        }
        else {
            return { consumedPaths: [], lastChild: 0, positionalParamSegments: {} };
        }
    }
    const path = route.path.startsWith('/') ? route.path.substring(1) : route.path;
    const parts = path.split('/');
    const positionalParamSegments = {};
    const consumedPaths = [];
    let currentIndex = 0;
    for (let i = 0; i < parts.length; ++i) {
        if (currentIndex >= paths.length)
            throw new NoMatch();
        const current = paths[currentIndex];
        const p = parts[i];
        const isPosParam = p.startsWith(':');
        if (!isPosParam && p !== current.path)
            throw new NoMatch();
        if (isPosParam) {
            positionalParamSegments[p.substring(1)] = current;
        }
        consumedPaths.push(current);
        currentIndex++;
    }
    if (route.terminal && (Object.keys(segment.children).length > 0 || currentIndex < paths.length)) {
        throw new NoMatch();
    }
    return { consumedPaths, lastChild: currentIndex, positionalParamSegments };
}
function applyRedirectCommands(paths, redirectTo, posParams) {
    if (redirectTo.startsWith('/')) {
        const parts = redirectTo.substring(1).split('/');
        return createPaths(redirectTo, parts, paths, posParams);
    }
    else {
        const parts = redirectTo.split('/');
        return createPaths(redirectTo, parts, paths, posParams);
    }
}
function createPaths(redirectTo, parts, segments, posParams) {
    return parts.map(p => p.startsWith(':') ? findPosParam(p, posParams, redirectTo) :
        findOrCreatePath(p, segments));
}
function findPosParam(part, posParams, redirectTo) {
    const paramName = part.substring(1);
    const pos = posParams[paramName];
    if (!pos)
        throw new Error(`Cannot redirect to '${redirectTo}'. Cannot find '${part}'.`);
    return pos;
}
function findOrCreatePath(part, paths) {
    const matchingIndex = paths.findIndex(s => s.path === part);
    if (matchingIndex > -1) {
        const r = paths[matchingIndex];
        paths.splice(matchingIndex);
        return r;
    }
    else {
        return new UrlPathWithParams(part, {});
    }
}
//# sourceMappingURL=data:application/json;base64,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