/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { Location, LocationStrategy } from '@angular/common';
import { MockLocationStrategy, SpyLocation } from '@angular/common/testing';
import { Compiler, Injectable, Injector, NgModule, NgModuleFactoryLoader } from '@angular/core';
import { Router, RouterModule, RouterOutletMap, UrlSerializer, provideRoutes } from '@angular/router';
import { ROUTER_PROVIDERS, ROUTES, flatten } from './private_import_router';
/**
 * A spy for {@link NgModuleFactoryLoader} that allows tests to simulate the loading of ng module
 * factories.
 *
 * @stable
 */
export var SpyNgModuleFactoryLoader = (function () {
    function SpyNgModuleFactoryLoader(compiler) {
        this.compiler = compiler;
        this.stubbedModules = {};
    }
    SpyNgModuleFactoryLoader.prototype.load = function (path) {
        if (this.stubbedModules[path]) {
            return this.compiler.compileModuleAsync(this.stubbedModules[path]);
        }
        else {
            return Promise.reject(new Error("Cannot find module " + path));
        }
    };
    SpyNgModuleFactoryLoader.decorators = [
        { type: Injectable },
    ];
    /** @nocollapse */
    SpyNgModuleFactoryLoader.ctorParameters = [
        { type: Compiler, },
    ];
    return SpyNgModuleFactoryLoader;
}());
/**
 * Router setup factory function used for testing.
 *
 * @stable
 */
export function setupTestingRouter(urlSerializer, outletMap, location, loader, compiler, injector, routes) {
    return new Router(null, urlSerializer, outletMap, location, injector, loader, compiler, flatten(routes));
}
/**
 * A module setting up the router that should be used for testing.
 * It provides spy implementations of Location, LocationStrategy, and NgModuleFactoryLoader.
 *
 * # Example:
 *
 * ```
 * beforeEach(() => {
 *   TestBed.configureTestModule({
 *     modules: [
 *       RouterTestingModule.withRoutes(
 *         [{path: '', component: BlankCmp}, {path: 'simple', component: SimpleCmp}])]
 *       )
 *     ]
 *   });
 * });
 * ```
 *
 * @stable
 */
export var RouterTestingModule = (function () {
    function RouterTestingModule() {
    }
    RouterTestingModule.withRoutes = function (routes) {
        return { ngModule: RouterTestingModule, providers: [provideRoutes(routes)] };
    };
    RouterTestingModule.decorators = [
        { type: NgModule, args: [{
                    exports: [RouterModule],
                    providers: [
                        ROUTER_PROVIDERS, { provide: Location, useClass: SpyLocation },
                        { provide: LocationStrategy, useClass: MockLocationStrategy },
                        { provide: NgModuleFactoryLoader, useClass: SpyNgModuleFactoryLoader }, {
                            provide: Router,
                            useFactory: setupTestingRouter,
                            deps: [
                                UrlSerializer, RouterOutletMap, Location, NgModuleFactoryLoader, Compiler, Injector, ROUTES
                            ]
                        },
                        provideRoutes([])
                    ]
                },] },
    ];
    /** @nocollapse */
    RouterTestingModule.ctorParameters = [];
    return RouterTestingModule;
}());
//# sourceMappingURL=router_testing_module.js.map