import { nextTick, inject, defineComponent } from 'vue';
import TransitionEvents from './css-animation/Event';
import raf from './raf';
import { defaultConfigProvider } from '../config-provider';
import { findDOMNode } from './props-util';
var styleForPesudo; // Where el is the DOM element you'd like to test for visibility

function isHidden(element) {
  if (process.env.NODE_ENV === 'test') {
    return false;
  }

  return !element || element.offsetParent === null;
}

function isNotGrey(color) {
  // eslint-disable-next-line no-useless-escape
  var match = (color || '').match(/rgba?\((\d*), (\d*), (\d*)(, [\.\d]*)?\)/);

  if (match && match[1] && match[2] && match[3]) {
    return !(match[1] === match[2] && match[2] === match[3]);
  }

  return true;
}

export default defineComponent({
  name: 'Wave',
  props: ['insertExtraNode'],
  setup: function setup() {
    var configProvider = inject('configProvider', defaultConfigProvider);
    return {
      configProvider: configProvider
    };
  },
  mounted: function mounted() {
    var _this = this;

    nextTick(function () {
      var node = findDOMNode(_this);

      if (node.nodeType !== 1) {
        return;
      }

      _this.instance = _this.bindAnimationEvent(node);
    });
  },
  beforeUnmount: function beforeUnmount() {
    if (this.instance) {
      this.instance.cancel();
    }

    if (this.clickWaveTimeoutId) {
      clearTimeout(this.clickWaveTimeoutId);
    }
  },
  methods: {
    onClick: function onClick(node, waveColor) {
      if (!node || isHidden(node) || node.className.indexOf('-leave') >= 0) {
        return;
      }

      var insertExtraNode = this.$props.insertExtraNode;
      this.extraNode = document.createElement('div');
      var extraNode = this.extraNode;
      extraNode.className = 'ant-click-animating-node';
      var attributeName = this.getAttributeName();
      node.removeAttribute(attributeName);
      node.setAttribute(attributeName, 'true'); // Not white or transparent or grey

      styleForPesudo = styleForPesudo || document.createElement('style');

      if (waveColor && waveColor !== '#ffffff' && waveColor !== 'rgb(255, 255, 255)' && isNotGrey(waveColor) && !/rgba\(\d*, \d*, \d*, 0\)/.test(waveColor) && // any transparent rgba color
      waveColor !== 'transparent') {
        // Add nonce if CSP exist
        if (this.csp && this.csp.nonce) {
          styleForPesudo.nonce = this.csp.nonce;
        }

        extraNode.style.borderColor = waveColor;
        styleForPesudo.innerHTML = "\n        [ant-click-animating-without-extra-node='true']::after, .ant-click-animating-node {\n          --antd-wave-shadow-color: ".concat(waveColor, ";\n        }");

        if (!document.body.contains(styleForPesudo)) {
          document.body.appendChild(styleForPesudo);
        }
      }

      if (insertExtraNode) {
        node.appendChild(extraNode);
      }

      TransitionEvents.addStartEventListener(node, this.onTransitionStart);
      TransitionEvents.addEndEventListener(node, this.onTransitionEnd);
    },
    onTransitionStart: function onTransitionStart(e) {
      if (this._.isUnmounted) return;
      var node = findDOMNode(this);

      if (!e || e.target !== node) {
        return;
      }

      if (!this.animationStart) {
        this.resetEffect(node);
      }
    },
    onTransitionEnd: function onTransitionEnd(e) {
      if (!e || e.animationName !== 'fadeEffect') {
        return;
      }

      this.resetEffect(e.target);
    },
    getAttributeName: function getAttributeName() {
      var insertExtraNode = this.$props.insertExtraNode;
      return insertExtraNode ? 'ant-click-animating' : 'ant-click-animating-without-extra-node';
    },
    bindAnimationEvent: function bindAnimationEvent(node) {
      var _this2 = this;

      if (!node || !node.getAttribute || node.getAttribute('disabled') || node.className.indexOf('disabled') >= 0) {
        return;
      }

      var onClick = function onClick(e) {
        // Fix radio button click twice
        if (e.target.tagName === 'INPUT' || isHidden(e.target)) {
          return;
        }

        _this2.resetEffect(node); // Get wave color from target


        var waveColor = getComputedStyle(node).getPropertyValue('border-top-color') || // Firefox Compatible
        getComputedStyle(node).getPropertyValue('border-color') || getComputedStyle(node).getPropertyValue('background-color');
        _this2.clickWaveTimeoutId = window.setTimeout(function () {
          return _this2.onClick(node, waveColor);
        }, 0);
        raf.cancel(_this2.animationStartId);
        _this2.animationStart = true; // Render to trigger transition event cost 3 frames. Let's delay 10 frames to reset this.

        _this2.animationStartId = raf(function () {
          _this2.animationStart = false;
        }, 10);
      };

      node.addEventListener('click', onClick, true);
      return {
        cancel: function cancel() {
          node.removeEventListener('click', onClick, true);
        }
      };
    },
    resetEffect: function resetEffect(node) {
      if (!node || node === this.extraNode || !(node instanceof Element)) {
        return;
      }

      var insertExtraNode = this.$props.insertExtraNode;
      var attributeName = this.getAttributeName();
      node.setAttribute(attributeName, 'false'); // edge has bug on `removeAttribute` #14466

      if (styleForPesudo) {
        styleForPesudo.innerHTML = '';
      }

      if (insertExtraNode && this.extraNode && node.contains(this.extraNode)) {
        node.removeChild(this.extraNode);
      }

      TransitionEvents.removeStartEventListener(node, this.onTransitionStart);
      TransitionEvents.removeEndEventListener(node, this.onTransitionEnd);
    }
  },
  render: function render() {
    var _a, _b;

    var csp = this.configProvider.csp;

    if (csp) {
      this.csp = csp;
    }

    return (_b = (_a = this.$slots).default) === null || _b === void 0 ? void 0 : _b.call(_a)[0];
  }
});