import _objectSpread from "@babel/runtime/helpers/esm/objectSpread2";
import _slicedToArray from "@babel/runtime/helpers/esm/slicedToArray";
import _toConsumableArray from "@babel/runtime/helpers/esm/toConsumableArray";
import _extends from "@babel/runtime/helpers/esm/extends";
import { createVNode as _createVNode } from "vue";
import { computed, defineComponent, onBeforeUnmount, onMounted, shallowRef, watch } from 'vue';
import { getStyleStr, getPixelRatio, rotateWatermark, reRendering } from './utils';
import { arrayType, objectType, someType, withInstall } from '../_util/type';
import { useMutationObserver } from '../_util/hooks/_vueuse/useMutationObserver';
import { initDefaultProps } from '../_util/props-util';
/**
 * Base size of the canvas, 1 for parallel layout and 2 for alternate layout
 * Only alternate layout is currently supported
 */
var BaseSize = 2;
var FontGap = 3;
export var watermarkProps = function watermarkProps() {
  return {
    zIndex: Number,
    rotate: Number,
    width: Number,
    height: Number,
    image: String,
    content: someType([String, Array]),
    font: objectType(),
    rootClassName: String,
    gap: arrayType(),
    offset: arrayType()
  };
};
var Watermark = defineComponent({
  name: 'AWatermark',
  inheritAttrs: false,
  props: initDefaultProps(watermarkProps(), {
    zIndex: 9,
    rotate: -22,
    font: {},
    gap: [100, 100]
  }),
  setup: function setup(props, _ref) {
    var slots = _ref.slots,
      attrs = _ref.attrs;
    var containerRef = shallowRef();
    var watermarkRef = shallowRef();
    var stopObservation = shallowRef(false);
    var gapX = computed(function () {
      var _a, _b;
      return (_b = (_a = props.gap) === null || _a === void 0 ? void 0 : _a[0]) !== null && _b !== void 0 ? _b : 100;
    });
    var gapY = computed(function () {
      var _a, _b;
      return (_b = (_a = props.gap) === null || _a === void 0 ? void 0 : _a[1]) !== null && _b !== void 0 ? _b : 100;
    });
    var gapXCenter = computed(function () {
      return gapX.value / 2;
    });
    var gapYCenter = computed(function () {
      return gapY.value / 2;
    });
    var offsetLeft = computed(function () {
      var _a, _b;
      return (_b = (_a = props.offset) === null || _a === void 0 ? void 0 : _a[0]) !== null && _b !== void 0 ? _b : gapXCenter.value;
    });
    var offsetTop = computed(function () {
      var _a, _b;
      return (_b = (_a = props.offset) === null || _a === void 0 ? void 0 : _a[1]) !== null && _b !== void 0 ? _b : gapYCenter.value;
    });
    var fontSize = computed(function () {
      var _a, _b;
      return (_b = (_a = props.font) === null || _a === void 0 ? void 0 : _a.fontSize) !== null && _b !== void 0 ? _b : 16;
    });
    var fontWeight = computed(function () {
      var _a, _b;
      return (_b = (_a = props.font) === null || _a === void 0 ? void 0 : _a.fontWeight) !== null && _b !== void 0 ? _b : 'normal';
    });
    var fontStyle = computed(function () {
      var _a, _b;
      return (_b = (_a = props.font) === null || _a === void 0 ? void 0 : _a.fontStyle) !== null && _b !== void 0 ? _b : 'normal';
    });
    var fontFamily = computed(function () {
      var _a, _b;
      return (_b = (_a = props.font) === null || _a === void 0 ? void 0 : _a.fontFamily) !== null && _b !== void 0 ? _b : 'sans-serif';
    });
    var color = computed(function () {
      var _a, _b;
      return (_b = (_a = props.font) === null || _a === void 0 ? void 0 : _a.color) !== null && _b !== void 0 ? _b : 'rgba(0, 0, 0, 0.15)';
    });
    var markStyle = computed(function () {
      var _a;
      var markStyle = {
        zIndex: (_a = props.zIndex) !== null && _a !== void 0 ? _a : 9,
        position: 'absolute',
        left: 0,
        top: 0,
        width: '100%',
        height: '100%',
        pointerEvents: 'none',
        backgroundRepeat: 'repeat'
      };
      /** Calculate the style of the offset */
      var positionLeft = offsetLeft.value - gapXCenter.value;
      var positionTop = offsetTop.value - gapYCenter.value;
      if (positionLeft > 0) {
        markStyle.left = "".concat(positionLeft, "px");
        markStyle.width = "calc(100% - ".concat(positionLeft, "px)");
        positionLeft = 0;
      }
      if (positionTop > 0) {
        markStyle.top = "".concat(positionTop, "px");
        markStyle.height = "calc(100% - ".concat(positionTop, "px)");
        positionTop = 0;
      }
      markStyle.backgroundPosition = "".concat(positionLeft, "px ").concat(positionTop, "px");
      return markStyle;
    });
    var destroyWatermark = function destroyWatermark() {
      if (watermarkRef.value) {
        watermarkRef.value.remove();
        watermarkRef.value = undefined;
      }
    };
    var appendWatermark = function appendWatermark(base64Url, markWidth) {
      var _a;
      if (containerRef.value && watermarkRef.value) {
        stopObservation.value = true;
        watermarkRef.value.setAttribute('style', getStyleStr(_extends(_extends({}, markStyle.value), {
          backgroundImage: "url('".concat(base64Url, "')"),
          backgroundSize: "".concat((gapX.value + markWidth) * BaseSize, "px")
        })));
        (_a = containerRef.value) === null || _a === void 0 ? void 0 : _a.append(watermarkRef.value);
        // Delayed execution
        setTimeout(function () {
          stopObservation.value = false;
        });
      }
    };
    /**
     * Get the width and height of the watermark. The default values are as follows
     * Image: [120, 64]; Content: It's calculated by content;
     */
    var getMarkSize = function getMarkSize(ctx) {
      var defaultWidth = 120;
      var defaultHeight = 64;
      var content = props.content;
      var image = props.image;
      var width = props.width;
      var height = props.height;
      if (!image && ctx.measureText) {
        ctx.font = "".concat(Number(fontSize.value), "px ").concat(fontFamily.value);
        var contents = Array.isArray(content) ? content : [content];
        var widths = contents.map(function (item) {
          return ctx.measureText(item).width;
        });
        defaultWidth = Math.ceil(Math.max.apply(Math, _toConsumableArray(widths)));
        defaultHeight = Number(fontSize.value) * contents.length + (contents.length - 1) * FontGap;
      }
      return [width !== null && width !== void 0 ? width : defaultWidth, height !== null && height !== void 0 ? height : defaultHeight];
    };
    var fillTexts = function fillTexts(ctx, drawX, drawY, drawWidth, drawHeight) {
      var ratio = getPixelRatio();
      var content = props.content;
      var mergedFontSize = Number(fontSize.value) * ratio;
      ctx.font = "".concat(fontStyle.value, " normal ").concat(fontWeight.value, " ").concat(mergedFontSize, "px/").concat(drawHeight, "px ").concat(fontFamily.value);
      ctx.fillStyle = color.value;
      ctx.textAlign = 'center';
      ctx.textBaseline = 'top';
      ctx.translate(drawWidth / 2, 0);
      var contents = Array.isArray(content) ? content : [content];
      contents === null || contents === void 0 ? void 0 : contents.forEach(function (item, index) {
        ctx.fillText(item !== null && item !== void 0 ? item : '', drawX, drawY + index * (mergedFontSize + FontGap * ratio));
      });
    };
    var renderWatermark = function renderWatermark() {
      var _a;
      var canvas = document.createElement('canvas');
      var ctx = canvas.getContext('2d');
      var image = props.image;
      var rotate = (_a = props.rotate) !== null && _a !== void 0 ? _a : -22;
      if (ctx) {
        if (!watermarkRef.value) {
          watermarkRef.value = document.createElement('div');
        }
        var ratio = getPixelRatio();
        var _getMarkSize = getMarkSize(ctx),
          _getMarkSize2 = _slicedToArray(_getMarkSize, 2),
          markWidth = _getMarkSize2[0],
          markHeight = _getMarkSize2[1];
        var canvasWidth = (gapX.value + markWidth) * ratio;
        var canvasHeight = (gapY.value + markHeight) * ratio;
        canvas.setAttribute('width', "".concat(canvasWidth * BaseSize, "px"));
        canvas.setAttribute('height', "".concat(canvasHeight * BaseSize, "px"));
        var drawX = gapX.value * ratio / 2;
        var drawY = gapY.value * ratio / 2;
        var drawWidth = markWidth * ratio;
        var drawHeight = markHeight * ratio;
        var rotateX = (drawWidth + gapX.value * ratio) / 2;
        var rotateY = (drawHeight + gapY.value * ratio) / 2;
        /** Alternate drawing parameters */
        var alternateDrawX = drawX + canvasWidth;
        var alternateDrawY = drawY + canvasHeight;
        var alternateRotateX = rotateX + canvasWidth;
        var alternateRotateY = rotateY + canvasHeight;
        ctx.save();
        rotateWatermark(ctx, rotateX, rotateY, rotate);
        if (image) {
          var img = new Image();
          img.onload = function () {
            ctx.drawImage(img, drawX, drawY, drawWidth, drawHeight);
            /** Draw interleaved pictures after rotation */
            ctx.restore();
            rotateWatermark(ctx, alternateRotateX, alternateRotateY, rotate);
            ctx.drawImage(img, alternateDrawX, alternateDrawY, drawWidth, drawHeight);
            appendWatermark(canvas.toDataURL(), markWidth);
          };
          img.crossOrigin = 'anonymous';
          img.referrerPolicy = 'no-referrer';
          img.src = image;
        } else {
          fillTexts(ctx, drawX, drawY, drawWidth, drawHeight);
          /** Fill the interleaved text after rotation */
          ctx.restore();
          rotateWatermark(ctx, alternateRotateX, alternateRotateY, rotate);
          fillTexts(ctx, alternateDrawX, alternateDrawY, drawWidth, drawHeight);
          appendWatermark(canvas.toDataURL(), markWidth);
        }
      }
    };
    onMounted(function () {
      renderWatermark();
    });
    watch(function () {
      return props;
    }, function () {
      renderWatermark();
    }, {
      deep: true,
      flush: 'post'
    });
    onBeforeUnmount(function () {
      destroyWatermark();
    });
    var onMutate = function onMutate(mutations) {
      if (stopObservation.value) {
        return;
      }
      mutations.forEach(function (mutation) {
        if (reRendering(mutation, watermarkRef.value)) {
          destroyWatermark();
          renderWatermark();
        }
      });
    };
    useMutationObserver(containerRef, onMutate, {
      attributes: true
    });
    return function () {
      var _a;
      return _createVNode("div", _objectSpread(_objectSpread({}, attrs), {}, {
        "ref": containerRef,
        "class": [attrs.class, props.rootClassName],
        "style": [{
          position: 'relative'
        }, attrs.style]
      }), [(_a = slots.default) === null || _a === void 0 ? void 0 : _a.call(slots)]);
    };
  }
});
export default withInstall(Watermark);