"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.QRCodeCanvas = void 0;
var _vue = require("vue");
var _objectSpread2 = _interopRequireDefault(require("@babel/runtime/helpers/objectSpread2"));
var _extends2 = _interopRequireDefault(require("@babel/runtime/helpers/extends"));
var _interface = require("./interface");
var _qrcodegen = _interopRequireDefault(require("./qrcodegen"));
var ERROR_LEVEL_MAP = {
  L: _qrcodegen.default.QrCode.Ecc.LOW,
  M: _qrcodegen.default.QrCode.Ecc.MEDIUM,
  Q: _qrcodegen.default.QrCode.Ecc.QUARTILE,
  H: _qrcodegen.default.QrCode.Ecc.HIGH
};
var DEFAULT_SIZE = 128;
var DEFAULT_LEVEL = 'L';
var DEFAULT_BGCOLOR = '#FFFFFF';
var DEFAULT_FGCOLOR = '#000000';
var DEFAULT_INCLUDEMARGIN = false;
var SPEC_MARGIN_SIZE = 4;
var DEFAULT_MARGIN_SIZE = 0;
// This is *very* rough estimate of max amount of QRCode allowed to be covered.
// It is "wrong" in a lot of ways (area is a terrible way to estimate, it
// really should be number of modules covered), but if for some reason we don't
// get an explicit height or width, I'd rather default to something than throw.
var DEFAULT_IMG_SCALE = 0.1;
function generatePath(modules) {
  var margin = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
  var ops = [];
  modules.forEach(function (row, y) {
    var start = null;
    row.forEach(function (cell, x) {
      if (!cell && start !== null) {
        // M0 0h7v1H0z injects the space with the move and drops the comma,
        // saving a char per operation
        ops.push("M".concat(start + margin, " ").concat(y + margin, "h").concat(x - start, "v1H").concat(start + margin, "z"));
        start = null;
        return;
      }
      // end of row, clean up or skip
      if (x === row.length - 1) {
        if (!cell) {
          // We would have closed the op above already so this can only mean
          // 2+ light modules in a row.
          return;
        }
        if (start === null) {
          // Just a single dark module.
          ops.push("M".concat(x + margin, ",").concat(y + margin, " h1v1H").concat(x + margin, "z"));
        } else {
          // Otherwise finish the current line.
          ops.push("M".concat(start + margin, ",").concat(y + margin, " h").concat(x + 1 - start, "v1H").concat(start + margin, "z"));
        }
        return;
      }
      if (cell && start === null) {
        start = x;
      }
    });
  });
  return ops.join('');
}
// We could just do this in generatePath, except that we want to support
// non-Path2D canvas, so we need to keep it an explicit step.
function excavateModules(modules, excavation) {
  return modules.slice().map(function (row, y) {
    if (y < excavation.y || y >= excavation.y + excavation.h) {
      return row;
    }
    return row.map(function (cell, x) {
      if (x < excavation.x || x >= excavation.x + excavation.w) {
        return cell;
      }
      return false;
    });
  });
}
function getImageSettings(cells, size, margin, imageSettings) {
  if (imageSettings == null) {
    return null;
  }
  var numCells = cells.length + margin * 2;
  var defaultSize = Math.floor(size * DEFAULT_IMG_SCALE);
  var scale = numCells / size;
  var w = (imageSettings.width || defaultSize) * scale;
  var h = (imageSettings.height || defaultSize) * scale;
  var x = imageSettings.x == null ? cells.length / 2 - w / 2 : imageSettings.x * scale;
  var y = imageSettings.y == null ? cells.length / 2 - h / 2 : imageSettings.y * scale;
  var excavation = null;
  if (imageSettings.excavate) {
    var floorX = Math.floor(x);
    var floorY = Math.floor(y);
    var ceilW = Math.ceil(w + x - floorX);
    var ceilH = Math.ceil(h + y - floorY);
    excavation = {
      x: floorX,
      y: floorY,
      w: ceilW,
      h: ceilH
    };
  }
  return {
    x: x,
    y: y,
    h: h,
    w: w,
    excavation: excavation
  };
}
function getMarginSize(includeMargin, marginSize) {
  if (marginSize != null) {
    return Math.floor(marginSize);
  }
  return includeMargin ? SPEC_MARGIN_SIZE : DEFAULT_MARGIN_SIZE;
}
// For canvas we're going to switch our drawing mode based on whether or not
// the environment supports Path2D. We only need the constructor to be
// supported, but Edge doesn't actually support the path (string) type
// argument. Luckily it also doesn't support the addPath() method. We can
// treat that as the same thing.
var SUPPORTS_PATH2D = function () {
  try {
    new Path2D().addPath(new Path2D());
  } catch (e) {
    return false;
  }
  return true;
}();
var QRCodeCanvas = (0, _vue.defineComponent)({
  name: 'QRCodeCanvas',
  inheritAttrs: false,
  props: (0, _extends2.default)((0, _extends2.default)({}, (0, _interface.qrProps)()), {
    level: String,
    bgColor: String,
    fgColor: String,
    marginSize: Number
  }),
  setup: function setup(props, _ref) {
    var attrs = _ref.attrs,
      expose = _ref.expose;
    var imgSrc = (0, _vue.computed)(function () {
      var _a;
      return (_a = props.imageSettings) === null || _a === void 0 ? void 0 : _a.src;
    });
    var _canvas = (0, _vue.shallowRef)(null);
    var _image = (0, _vue.shallowRef)(null);
    var isImgLoaded = (0, _vue.shallowRef)(false);
    expose({
      toDataURL: function toDataURL(type, quality) {
        var _a;
        return (_a = _canvas.value) === null || _a === void 0 ? void 0 : _a.toDataURL(type, quality);
      }
    });
    (0, _vue.watchEffect)(function () {
      var value = props.value,
        _props$size = props.size,
        size = _props$size === void 0 ? DEFAULT_SIZE : _props$size,
        _props$level = props.level,
        level = _props$level === void 0 ? DEFAULT_LEVEL : _props$level,
        _props$bgColor = props.bgColor,
        bgColor = _props$bgColor === void 0 ? DEFAULT_BGCOLOR : _props$bgColor,
        _props$fgColor = props.fgColor,
        fgColor = _props$fgColor === void 0 ? DEFAULT_FGCOLOR : _props$fgColor,
        _props$includeMargin = props.includeMargin,
        includeMargin = _props$includeMargin === void 0 ? DEFAULT_INCLUDEMARGIN : _props$includeMargin,
        marginSize = props.marginSize,
        imageSettings = props.imageSettings;
      if (_canvas.value != null) {
        var canvas = _canvas.value;
        var ctx = canvas.getContext('2d');
        if (!ctx) {
          return;
        }
        var cells = _qrcodegen.default.QrCode.encodeText(value, ERROR_LEVEL_MAP[level]).getModules();
        var margin = getMarginSize(includeMargin, marginSize);
        var numCells = cells.length + margin * 2;
        var calculatedImageSettings = getImageSettings(cells, size, margin, imageSettings);
        var image = _image.value;
        var haveImageToRender = isImgLoaded.value && calculatedImageSettings != null && image !== null && image.complete && image.naturalHeight !== 0 && image.naturalWidth !== 0;
        if (haveImageToRender) {
          if (calculatedImageSettings.excavation != null) {
            cells = excavateModules(cells, calculatedImageSettings.excavation);
          }
        }
        // We're going to scale this so that the number of drawable units
        // matches the number of cells. This avoids rounding issues, but does
        // result in some potentially unwanted single pixel issues between
        // blocks, only in environments that don't support Path2D.
        var pixelRatio = window.devicePixelRatio || 1;
        canvas.height = canvas.width = size * pixelRatio;
        var scale = size / numCells * pixelRatio;
        ctx.scale(scale, scale);
        // Draw solid background, only paint dark modules.
        ctx.fillStyle = bgColor;
        ctx.fillRect(0, 0, numCells, numCells);
        ctx.fillStyle = fgColor;
        if (SUPPORTS_PATH2D) {
          // $FlowFixMe: Path2D c'tor doesn't support args yet.
          ctx.fill(new Path2D(generatePath(cells, margin)));
        } else {
          cells.forEach(function (row, rdx) {
            row.forEach(function (cell, cdx) {
              if (cell) {
                ctx.fillRect(cdx + margin, rdx + margin, 1, 1);
              }
            });
          });
        }
        if (haveImageToRender) {
          ctx.drawImage(image, calculatedImageSettings.x + margin, calculatedImageSettings.y + margin, calculatedImageSettings.w, calculatedImageSettings.h);
        }
      }
    }, {
      flush: 'post'
    });
    (0, _vue.watch)(imgSrc, function () {
      isImgLoaded.value = false;
    });
    return function () {
      var _a;
      var size = (_a = props.size) !== null && _a !== void 0 ? _a : DEFAULT_SIZE;
      var canvasStyle = {
        height: "".concat(size, "px"),
        width: "".concat(size, "px")
      };
      var img = null;
      if (imgSrc.value != null) {
        img = (0, _vue.createVNode)("img", {
          "src": imgSrc.value,
          "key": imgSrc.value,
          "style": {
            display: 'none'
          },
          "onLoad": function onLoad() {
            isImgLoaded.value = true;
          },
          "ref": _image
        }, null);
      }
      return (0, _vue.createVNode)(_vue.Fragment, null, [(0, _vue.createVNode)("canvas", (0, _objectSpread2.default)((0, _objectSpread2.default)({}, attrs), {}, {
        "style": [canvasStyle, attrs.style],
        "ref": _canvas
      }), null), img]);
    };
  }
});
exports.QRCodeCanvas = QRCodeCanvas;