"use strict";
/*
 * This file is released under the MIT license.
 * Copyright (c) 2017, 2021, Mike Lischke
 *
 * See LICENSE file for more info.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SymbolTable = void 0;
const BaseSymbol_1 = require("./BaseSymbol");
const ScopedSymbol_1 = require("./ScopedSymbol");
const NamespaceSymbol_1 = require("./NamespaceSymbol");
/** The main class managing all the symbols for a top level entity like a file, library or similar. */
class SymbolTable extends ScopedSymbol_1.ScopedSymbol {
    constructor(name, options) {
        super(name);
        this.options = options;
        /**  Other symbol information available to this instance. */
        this.dependencies = new Set();
    }
    get info() {
        return {
            dependencyCount: this.dependencies.size,
            symbolCount: this.children.length,
        };
    }
    clear() {
        super.clear();
        this.dependencies.clear();
    }
    addDependencies(...tables) {
        tables.forEach((value, key) => {
            this.dependencies.add(value);
        });
    }
    removeDependency(table) {
        if (this.dependencies.has(table)) {
            this.dependencies.delete(table);
        }
    }
    addNewSymbolOfType(t, parent, ...args) {
        const result = new t(...args);
        if (!parent || parent === this) {
            this.addSymbol(result);
        }
        else {
            parent.addSymbol(result);
        }
        return result;
    }
    addNewNamespaceFromPath(parent, path, delimiter = ".") {
        return __awaiter(this, void 0, void 0, function* () {
            const parts = path.split(delimiter);
            let i = 0;
            let currentParent = (parent === undefined) ? this : parent;
            while (i < parts.length - 1) {
                let namespace = yield currentParent.resolve(parts[i], true);
                if (namespace === undefined) {
                    namespace = this.addNewSymbolOfType(NamespaceSymbol_1.NamespaceSymbol, currentParent, parts[i]);
                }
                currentParent = namespace;
                ++i;
            }
            return this.addNewSymbolOfType(NamespaceSymbol_1.NamespaceSymbol, currentParent, parts[parts.length - 1]);
        });
    }
    addNewNamespaceFromPathSync(parent, path, delimiter = ".") {
        const parts = path.split(delimiter);
        let i = 0;
        let currentParent = (parent === undefined) ? this : parent;
        while (i < parts.length - 1) {
            let namespace = currentParent.resolveSync(parts[i], true);
            if (namespace === undefined) {
                namespace = this.addNewSymbolOfType(NamespaceSymbol_1.NamespaceSymbol, currentParent, parts[i]);
            }
            currentParent = namespace;
            ++i;
        }
        return this.addNewSymbolOfType(NamespaceSymbol_1.NamespaceSymbol, currentParent, parts[parts.length - 1]);
    }
    getAllSymbols(t, localOnly = false) {
        const _super = Object.create(null, {
            getAllSymbols: { get: () => super.getAllSymbols }
        });
        return __awaiter(this, void 0, void 0, function* () {
            const result = yield _super.getAllSymbols.call(this, t, localOnly);
            if (!localOnly) {
                const dependencyResults = yield Promise.all([...this.dependencies].map((dependency) => {
                    return (dependency.getAllSymbols(t, localOnly));
                }));
                dependencyResults.forEach((value) => {
                    result.push(...value);
                });
            }
            return result;
        });
    }
    getAllSymbolsSync(t, localOnly = false) {
        const result = super.getAllSymbolsSync(t, localOnly);
        if (!localOnly) {
            this.dependencies.forEach((dependency) => {
                result.push(...dependency.getAllSymbolsSync(t, localOnly));
            });
        }
        return result;
    }
    symbolWithContext(context) {
        return __awaiter(this, void 0, void 0, function* () {
            /**
             * Local function to find a symbol recursively.
             *
             * @param symbol The symbol to search through.
             *
             * @returns The symbol with the given context, if found.
             */
            const findRecursive = (symbol) => {
                if (symbol.context === context) {
                    return symbol;
                }
                if (symbol instanceof ScopedSymbol_1.ScopedSymbol) {
                    for (const child of symbol.children) {
                        const result = findRecursive(child);
                        if (result) {
                            return result;
                        }
                    }
                }
                return undefined;
            };
            let symbols = yield this.getAllSymbols(BaseSymbol_1.BaseSymbol);
            for (const symbol of symbols) {
                const result = findRecursive(symbol);
                if (result) {
                    return result;
                }
            }
            for (const dependency of this.dependencies) {
                symbols = yield dependency.getAllSymbols(BaseSymbol_1.BaseSymbol);
                for (const symbol of symbols) {
                    const result = findRecursive(symbol);
                    if (result) {
                        return result;
                    }
                }
            }
            return undefined;
        });
    }
    symbolWithContextSync(context) {
        /**
         * Local function to find a symbol recursively.
         *
         * @param symbol The symbol to search through.
         *
         * @returns The symbol with the given context, if found.
         */
        const findRecursive = (symbol) => {
            if (symbol.context === context) {
                return symbol;
            }
            if (symbol instanceof ScopedSymbol_1.ScopedSymbol) {
                for (const child of symbol.children) {
                    const result = findRecursive(child);
                    if (result) {
                        return result;
                    }
                }
            }
            return undefined;
        };
        let symbols = this.getAllSymbolsSync(BaseSymbol_1.BaseSymbol);
        for (const symbol of symbols) {
            const result = findRecursive(symbol);
            if (result) {
                return result;
            }
        }
        for (const dependency of this.dependencies) {
            symbols = dependency.getAllSymbolsSync(BaseSymbol_1.BaseSymbol);
            for (const symbol of symbols) {
                const result = findRecursive(symbol);
                if (result) {
                    return result;
                }
            }
        }
        return undefined;
    }
    resolve(name, localOnly = false) {
        const _super = Object.create(null, {
            resolve: { get: () => super.resolve }
        });
        return __awaiter(this, void 0, void 0, function* () {
            let result = yield _super.resolve.call(this, name, localOnly);
            if (!result && !localOnly) {
                for (const dependency of this.dependencies) {
                    result = yield dependency.resolve(name, false);
                    if (result) {
                        return result;
                    }
                }
            }
            return result;
        });
    }
    resolveSync(name, localOnly = false) {
        let result = super.resolveSync(name, localOnly);
        if (!result && !localOnly) {
            for (const dependency of this.dependencies) {
                result = dependency.resolveSync(name, false);
                if (result) {
                    return result;
                }
            }
        }
        return result;
    }
}
exports.SymbolTable = SymbolTable;
//# sourceMappingURL=SymbolTable.js.map