/**
 * @fileOverview random layout
 * @author shiwu.wyy@antfin.com
 */
import dagre from "dagrejs";
import { isArray, isNumber, isObject, getEdgeTerminal } from "../util";
import { Base } from "./base";
/**
 * 层次布局
 */
export class DagreLayout extends Base {
    constructor(options) {
        super();
        /** layout 方向, 可选 TB, BT, LR, RL */
        this.rankdir = "TB";
        /** 节点水平间距(px) */
        this.nodesep = 50;
        /** 每一层节点之间间距 */
        this.ranksep = 50;
        /** 是否保留布局连线的控制点 */
        this.controlPoints = false;
        /** 每层节点是否根据节点数据中的 comboId 进行排序，以防止同层 combo 重叠 */
        this.sortByCombo = false;
        /** 是否保留每条边上的dummy node */
        this.edgeLabelSpace = true;
        /** 是否按照给定的节点顺序排序 */
        this.keepNodeOrder = false;
        this.nodes = [];
        this.edges = [];
        /** 迭代结束的回调函数 */
        this.onLayoutEnd = () => { };
        this.updateCfg(options);
    }
    getDefaultCfg() {
        return {
            rankdir: "TB",
            align: undefined,
            nodeSize: undefined,
            nodesepFunc: undefined,
            ranksepFunc: undefined,
            nodesep: 50,
            ranksep: 50,
            controlPoints: false, // 是否保留布局连线的控制点
        };
    }
    /**
     * 执行布局
     */
    execute() {
        const self = this;
        const { nodes, nodeSize, rankdir, combos } = self;
        if (!nodes)
            return;
        const edges = self.edges || [];
        const g = new dagre.graphlib.Graph({
            multigraph: true,
            compound: true,
        });
        let nodeSizeFunc;
        if (!nodeSize) {
            nodeSizeFunc = (d) => {
                if (d.size) {
                    if (isArray(d.size)) {
                        return d.size;
                    }
                    if (isObject(d.size)) {
                        return [d.size.width || 40, d.size.height || 40];
                    }
                    return [d.size, d.size];
                }
                return [40, 40];
            };
        }
        else if (isArray(nodeSize)) {
            nodeSizeFunc = () => nodeSize;
        }
        else {
            nodeSizeFunc = () => [nodeSize, nodeSize];
        }
        let horisep = getFunc(self.nodesepFunc, self.nodesep, 50);
        let vertisep = getFunc(self.ranksepFunc, self.ranksep, 50);
        if (rankdir === "LR" || rankdir === "RL") {
            horisep = getFunc(self.ranksepFunc, self.ranksep, 50);
            vertisep = getFunc(self.nodesepFunc, self.nodesep, 50);
        }
        g.setDefaultEdgeLabel(() => ({}));
        g.setGraph(self);
        const comboMap = {};
        nodes.filter((node) => node.layout !== false).forEach((node) => {
            const size = nodeSizeFunc(node);
            const verti = vertisep(node);
            const hori = horisep(node);
            const width = size[0] + 2 * hori;
            const height = size[1] + 2 * verti;
            const layer = node.layer;
            if (isNumber(layer)) {
                // 如果有layer属性，加入到node的label中
                g.setNode(node.id, { width, height, layer });
            }
            else {
                g.setNode(node.id, { width, height });
            }
            if (this.sortByCombo && node.comboId) {
                if (!comboMap[node.comboId]) {
                    comboMap[node.comboId] = true;
                    g.setNode(node.comboId, {});
                }
                g.setParent(node.id, node.comboId);
            }
        });
        if (this.sortByCombo && combos) {
            combos.forEach((combo) => {
                if (!combo.parentId)
                    return;
                if (!comboMap[combo.parentId]) {
                    comboMap[combo.parentId] = true;
                    g.setNode(combo.parentId, {});
                }
                g.setParent(combo.id, combo.parentId);
            });
        }
        edges.forEach((edge) => {
            // dagrejs Wiki https://github.com/dagrejs/dagre/wiki#configuring-the-layout
            const source = getEdgeTerminal(edge, 'source');
            const target = getEdgeTerminal(edge, 'target');
            g.setEdge(source, target, {
                weight: edge.weight || 1,
            });
        });
        dagre.layout(g, {
            edgeLabelSpace: self.edgeLabelSpace,
            keepNodeOrder: self.keepNodeOrder,
            nodeOrder: self.nodeOrder,
        });
        let coord;
        g.nodes().forEach((node) => {
            coord = g.node(node);
            const i = nodes.findIndex((it) => it.id === node);
            if (!nodes[i])
                return;
            nodes[i].x = coord.x;
            nodes[i].y = coord.y;
        });
        g.edges().forEach((edge) => {
            coord = g.edge(edge);
            const i = edges.findIndex((it) => {
                const source = getEdgeTerminal(it, 'source');
                const target = getEdgeTerminal(it, 'target');
                return source === edge.v && target === edge.w;
            });
            if ((self.edgeLabelSpace !== false) && self.controlPoints && edges[i].type !== "loop") {
                edges[i].controlPoints = coord.points.slice(1, coord.points.length - 1);
            }
        });
        if (self.onLayoutEnd)
            self.onLayoutEnd();
        return {
            nodes,
            edges,
        };
    }
    getType() {
        return "dagre";
    }
}
function getFunc(func, value, defaultValue) {
    let resultFunc;
    if (func) {
        resultFunc = func;
    }
    else if (isNumber(value)) {
        resultFunc = () => value;
    }
    else {
        resultFunc = () => defaultValue;
    }
    return resultFunc;
}
//# sourceMappingURL=dagre.js.map