"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.arrayToTextureData = exports.attributesToTextureData = exports.buildTextureDataWithTwoEdgeAttr = exports.buildTextureData = exports.proccessToFunc = void 0;
var _1 = require("./");
var math_1 = require("./math");
/**
 * 将 number | Function 类型的参数转换为 return number 的 Function
 * @param  {number | Function}  value 需要被转换的值
 * @param  {number}  defaultV 返回函数的默认返回值
 * @return {Function} 转换后的函数
 */
var proccessToFunc = function (value, defaultV) {
    var func;
    if (!value) {
        func = function () {
            return defaultV || 1;
        };
    }
    else if (_1.isNumber(value)) {
        func = function () {
            return value;
        };
    }
    else {
        func = value;
    }
    return func;
};
exports.proccessToFunc = proccessToFunc;
/**
 * 将节点和边数据转换为 GPU 可读的数组。并返回 maxEdgePerVetex，每个节点上最多的边数
 * @param  {NodeConfig[]}  nodes 需要被转换的值
 * @param  {EdgeConfig[]}  edges 返回函数的默认返回值
 * @return {Object} 转换后的数组及 maxEdgePerVetex 组成的对象
 */
var buildTextureData = function (nodes, edges) {
    var dataArray = [];
    var nodeDict = [];
    var mapIdPos = {};
    var i = 0;
    for (i = 0; i < nodes.length; i++) {
        var n = nodes[i];
        mapIdPos[n.id] = i;
        dataArray.push(n.x);
        dataArray.push(n.y);
        dataArray.push(0);
        dataArray.push(0);
        nodeDict.push([]);
    }
    for (i = 0; i < edges.length; i++) {
        var e = edges[i];
        var source = math_1.getEdgeTerminal(e, 'source');
        var target = math_1.getEdgeTerminal(e, 'target');
        nodeDict[mapIdPos[source]].push(mapIdPos[target]);
        nodeDict[mapIdPos[target]].push(mapIdPos[source]);
    }
    var maxEdgePerVetex = 0;
    for (i = 0; i < nodes.length; i++) {
        var offset = dataArray.length;
        var dests = nodeDict[i];
        var len = dests.length;
        dataArray[i * 4 + 2] = offset;
        dataArray[i * 4 + 3] = dests.length;
        maxEdgePerVetex = Math.max(maxEdgePerVetex, dests.length);
        for (var j = 0; j < len; ++j) {
            var dest = dests[j];
            dataArray.push(+dest);
        }
    }
    while (dataArray.length % 4 !== 0) {
        dataArray.push(0);
    }
    return {
        maxEdgePerVetex: maxEdgePerVetex,
        array: new Float32Array(dataArray),
    };
};
exports.buildTextureData = buildTextureData;
/**
* 将节点和边数据转换为 GPU 可读的数组，每条边带有一个属性。并返回 maxEdgePerVetex，每个节点上最多的边数
* @param  {NodeConfig[]}  nodes 节点数组
* @param  {EdgeConfig[]}  edges 边数组
* @param  {Function}  attrs 读取边属性的函数
* @return {Object} 转换后的数组及 maxEdgePerVetex 组成的对象
*/
// export const buildTextureDataWithOneEdgeAttr = (nodes: OutNode[], edges: Edge[], attrs: Function): {
//   array: Float32Array,
//   maxEdgePerVetex: number
// } => {
//   const dataArray = [];
//   const nodeDict: any = [];
//   const mapIdPos: IndexMap = {};
//   let i = 0;
//   for (i = 0; i < nodes.length; i++) {
//       const n = nodes[i];
//       mapIdPos[n.id] = i;
//       dataArray.push(n.x);
//       dataArray.push(n.y);
//       dataArray.push(0);
//       dataArray.push(0);
//       nodeDict.push([]);
//   }
//   for (i = 0; i < edges.length; i++) {
//       const e = edges[i];
//       nodeDict[mapIdPos[e.source]].push(mapIdPos[e.target]);
//       nodeDict[mapIdPos[e.source]].push(attrs(e)); // 理想边长，后续可以改成每条边不同
//       nodeDict[mapIdPos[e.target]].push(mapIdPos[e.source]);
//       nodeDict[mapIdPos[e.target]].push(attrs(e)); // 理想边长，后续可以改成每条边不同
//   }
//   let maxEdgePerVetex = 0;
//   for (i = 0; i < nodes.length; i++) {
//       const offset: number = dataArray.length;
//       const dests = nodeDict[i]; // dest 中节点 id 与边长间隔存储，即一位节点 id，一位边长……
//       const len = dests.length;
//       dataArray[i * 4 + 2] = offset;
//       dataArray[i * 4 + 3] = len / 2; // 第四位存储与该节点相关的所有节点个数
//       maxEdgePerVetex = Math.max(maxEdgePerVetex, len / 2);
//       for (let j = 0; j < len; ++j) {
//           const dest = dests[j];
//           dataArray.push(+dest);
//       }
//   }
//   // 不是 4 的倍数，填充 0
//   while (dataArray.length % 4 !== 0) {
//       dataArray.push(0);
//   }
//   return {
//       array: new Float32Array(dataArray),
//       maxEdgePerVetex
//   }
// }
/**
* 将节点和边数据转换为 GPU 可读的数组，每条边带有一个以上属性。并返回 maxEdgePerVetex，每个节点上最多的边数
* @param  {NodeConfig[]}  nodes 节点数组
* @param  {EdgeConfig[]}  edges 边数组
* @param  {Function}  attrs 读取边属性的函数
* @return {Object} 转换后的数组及 maxEdgePerVetex 组成的对象
*/
var buildTextureDataWithTwoEdgeAttr = function (nodes, edges, attrs1, attrs2) {
    var dataArray = [];
    var nodeDict = [];
    var mapIdPos = {};
    var i = 0;
    for (i = 0; i < nodes.length; i++) {
        var n = nodes[i];
        mapIdPos[n.id] = i;
        dataArray.push(n.x);
        dataArray.push(n.y);
        dataArray.push(0);
        dataArray.push(0);
        nodeDict.push([]);
    }
    for (i = 0; i < edges.length; i++) {
        var e = edges[i];
        var source = math_1.getEdgeTerminal(e, 'source');
        var target = math_1.getEdgeTerminal(e, 'target');
        nodeDict[mapIdPos[source]].push(mapIdPos[target]);
        nodeDict[mapIdPos[source]].push(attrs1(e));
        nodeDict[mapIdPos[source]].push(attrs2(e));
        nodeDict[mapIdPos[source]].push(0);
        nodeDict[mapIdPos[target]].push(mapIdPos[source]);
        nodeDict[mapIdPos[target]].push(attrs1(e));
        nodeDict[mapIdPos[target]].push(attrs2(e));
        nodeDict[mapIdPos[target]].push(0);
    }
    var maxEdgePerVetex = 0;
    for (i = 0; i < nodes.length; i++) {
        var offset = dataArray.length;
        var dests = nodeDict[i]; // dest 中节点 id 与边长间隔存储，即一位节点 id，一位边长……
        var len = dests.length;
        // dataArray[i * 4 + 2] = offset;
        // dataArray[i * 4 + 3] = len / 4; // 第四位存储与该节点相关的所有节点个数
        // pack offset & length into float32: offset 20bit, length 12bit
        dataArray[i * 4 + 2] = offset + 1048576 * len / 4;
        dataArray[i * 4 + 3] = 0; // 第四位存储与上一次的距离差值
        maxEdgePerVetex = Math.max(maxEdgePerVetex, len / 4);
        for (var j = 0; j < len; ++j) {
            var dest = dests[j];
            dataArray.push(+dest);
        }
    }
    // 不是 4 的倍数，填充 0
    while (dataArray.length % 4 !== 0) {
        dataArray.push(0);
    }
    return {
        maxEdgePerVetex: maxEdgePerVetex,
        array: new Float32Array(dataArray),
    };
};
exports.buildTextureDataWithTwoEdgeAttr = buildTextureDataWithTwoEdgeAttr;
/**
* transform the extended attributes of nodes or edges to a texture array
* @param  {string[]}  attributeNames attributes' name to be read from items and put into output array
* @param  {ModelConfig[]}  items the items to be read
* @return {Float32Array} the attributes' value array to be read by GPU
*/
var attributesToTextureData = function (attributeNames, items) {
    var dataArray = [];
    var attributeNum = attributeNames.length;
    var attributteStringMap = {};
    items.forEach(function (item) {
        attributeNames.forEach(function (name, i) {
            if (attributteStringMap[item[name]] === undefined) {
                attributteStringMap[item[name]] = Object.keys(attributteStringMap).length;
            }
            dataArray.push(attributteStringMap[item[name]]);
            // insure each node's attributes take inter number of grids
            if (i === attributeNum - 1) {
                while (dataArray.length % 4 !== 0) {
                    dataArray.push(0);
                }
            }
        });
    });
    return {
        array: new Float32Array(dataArray),
        count: Object.keys(attributteStringMap).length
    };
};
exports.attributesToTextureData = attributesToTextureData;
/**
* transform the number array format of extended attributes of nodes or edges to a texture array
* @param  {string[]}  attributeNames attributes' name to be read from items and put into output array
* @return {Float32Array} the attributes' value array to be read by GPU
*/
var arrayToTextureData = function (valueArrays) {
    var dataArray = [];
    var attributeNum = valueArrays.length;
    var itemNum = valueArrays[0].length;
    var _loop_1 = function (j) {
        valueArrays.forEach(function (valueArray, i) {
            dataArray.push(valueArray[j]);
            // insure each node's attributes take inter number of grids
            if (i === attributeNum - 1) {
                while (dataArray.length % 4 !== 0) {
                    dataArray.push(0);
                }
            }
        });
    };
    for (var j = 0; j < itemNum; j++) {
        _loop_1(j);
    }
    return new Float32Array(dataArray);
};
exports.arrayToTextureData = arrayToTextureData;
//# sourceMappingURL=gpu.js.map