"use strict";
// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.AsyncByteStream = exports.ByteStream = exports.AsyncByteQueue = void 0;
const tslib_1 = require("tslib");
const adapters_1 = require("./adapters");
const utf8_1 = require("../util/utf8");
const interfaces_1 = require("./interfaces");
const buffer_1 = require("../util/buffer");
const compat_1 = require("../util/compat");
/** @ignore */
class AsyncByteQueue extends interfaces_1.AsyncQueue {
    write(value) {
        if ((value = buffer_1.toUint8Array(value)).byteLength > 0) {
            return super.write(value);
        }
    }
    toString(sync = false) {
        return sync
            ? utf8_1.decodeUtf8(this.toUint8Array(true))
            : this.toUint8Array(false).then(utf8_1.decodeUtf8);
    }
    toUint8Array(sync = false) {
        return sync ? buffer_1.joinUint8Arrays(this._values)[0] : (() => tslib_1.__awaiter(this, void 0, void 0, function* () {
            var e_1, _a;
            const buffers = [];
            let byteLength = 0;
            try {
                for (var _b = tslib_1.__asyncValues(this), _c; _c = yield _b.next(), !_c.done;) {
                    const chunk = _c.value;
                    buffers.push(chunk);
                    byteLength += chunk.byteLength;
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) yield _a.call(_b);
                }
                finally { if (e_1) throw e_1.error; }
            }
            return buffer_1.joinUint8Arrays(buffers, byteLength)[0];
        }))();
    }
}
exports.AsyncByteQueue = AsyncByteQueue;
/** @ignore */
class ByteStream {
    constructor(source) {
        if (source) {
            this.source = new ByteStreamSource(adapters_1.default.fromIterable(source));
        }
    }
    [Symbol.iterator]() { return this; }
    next(value) { return this.source.next(value); }
    throw(value) { return this.source.throw(value); }
    return(value) { return this.source.return(value); }
    peek(size) { return this.source.peek(size); }
    read(size) { return this.source.read(size); }
}
exports.ByteStream = ByteStream;
/** @ignore */
class AsyncByteStream {
    constructor(source) {
        if (source instanceof AsyncByteStream) {
            this.source = source.source;
        }
        else if (source instanceof AsyncByteQueue) {
            this.source = new AsyncByteStreamSource(adapters_1.default.fromAsyncIterable(source));
        }
        else if (compat_1.isReadableNodeStream(source)) {
            this.source = new AsyncByteStreamSource(adapters_1.default.fromNodeStream(source));
        }
        else if (compat_1.isReadableDOMStream(source)) {
            this.source = new AsyncByteStreamSource(adapters_1.default.fromDOMStream(source));
        }
        else if (compat_1.isFetchResponse(source)) {
            this.source = new AsyncByteStreamSource(adapters_1.default.fromDOMStream(source.body));
        }
        else if (compat_1.isIterable(source)) {
            this.source = new AsyncByteStreamSource(adapters_1.default.fromIterable(source));
        }
        else if (compat_1.isPromise(source)) {
            this.source = new AsyncByteStreamSource(adapters_1.default.fromAsyncIterable(source));
        }
        else if (compat_1.isAsyncIterable(source)) {
            this.source = new AsyncByteStreamSource(adapters_1.default.fromAsyncIterable(source));
        }
    }
    [Symbol.asyncIterator]() { return this; }
    next(value) { return this.source.next(value); }
    throw(value) { return this.source.throw(value); }
    return(value) { return this.source.return(value); }
    get closed() { return this.source.closed; }
    cancel(reason) { return this.source.cancel(reason); }
    peek(size) { return this.source.peek(size); }
    read(size) { return this.source.read(size); }
}
exports.AsyncByteStream = AsyncByteStream;
/** @ignore */
class ByteStreamSource {
    constructor(source) {
        this.source = source;
    }
    cancel(reason) { this.return(reason); }
    peek(size) { return this.next(size, 'peek').value; }
    read(size) { return this.next(size, 'read').value; }
    next(size, cmd = 'read') { return this.source.next({ cmd, size }); }
    throw(value) { return Object.create((this.source.throw && this.source.throw(value)) || interfaces_1.ITERATOR_DONE); }
    return(value) { return Object.create((this.source.return && this.source.return(value)) || interfaces_1.ITERATOR_DONE); }
}
/** @ignore */
class AsyncByteStreamSource {
    constructor(source) {
        this.source = source;
        this._closedPromise = new Promise((r) => this._closedPromiseResolve = r);
    }
    cancel(reason) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () { yield this.return(reason); });
    }
    get closed() { return this._closedPromise; }
    read(size) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () { return (yield this.next(size, 'read')).value; });
    }
    peek(size) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () { return (yield this.next(size, 'peek')).value; });
    }
    next(size, cmd = 'read') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () { return (yield this.source.next({ cmd, size })); });
    }
    throw(value) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const result = (this.source.throw && (yield this.source.throw(value))) || interfaces_1.ITERATOR_DONE;
            this._closedPromiseResolve && this._closedPromiseResolve();
            this._closedPromiseResolve = undefined;
            return Object.create(result);
        });
    }
    return(value) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const result = (this.source.return && (yield this.source.return(value))) || interfaces_1.ITERATOR_DONE;
            this._closedPromiseResolve && this._closedPromiseResolve();
            this._closedPromiseResolve = undefined;
            return Object.create(result);
        });
    }
}

//# sourceMappingURL=stream.js.map
