"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var Observable_1 = require('./util/Observable');
var errorHandling_1 = require('./util/errorHandling');
var assign = require('lodash.assign');
var isEqual = require('lodash.isequal');
var ObservableQuery = (function (_super) {
    __extends(ObservableQuery, _super);
    function ObservableQuery(_a) {
        var _this = this;
        var scheduler = _a.scheduler, options = _a.options, _b = _a.shouldSubscribe, shouldSubscribe = _b === void 0 ? true : _b;
        var queryManager = scheduler.queryManager;
        var queryId = queryManager.generateQueryId();
        var isPollingQuery = !!options.pollInterval;
        var subscriberFunction = function (observer) {
            var retQuerySubscription = {
                unsubscribe: function () {
                    if (isPollingQuery) {
                        scheduler.stopPollingQuery(queryId);
                    }
                    queryManager.stopQuery(queryId);
                },
            };
            if (shouldSubscribe) {
                queryManager.addObservableQuery(queryId, _this);
                queryManager.addQuerySubscription(queryId, retQuerySubscription);
            }
            if (isPollingQuery) {
                if (_this.options.noFetch) {
                    throw new Error('noFetch option should not use query polling.');
                }
                _this.scheduler.startPollingQuery(_this.options, queryId);
            }
            queryManager.startQuery(queryId, _this.options, queryManager.queryListenerForObserver(queryId, _this.options, observer));
            return retQuerySubscription;
        };
        _super.call(this, subscriberFunction);
        this.options = options;
        this.variables = this.options.variables || {};
        this.scheduler = scheduler;
        this.queryManager = queryManager;
        this.queryId = queryId;
        this.refetch = function (variables) {
            _this.variables = assign({}, _this.variables, variables);
            if (_this.options.noFetch) {
                throw new Error('noFetch option should not use query refetch.');
            }
            return _this.queryManager.fetchQuery(_this.queryId, assign(_this.options, {
                forceFetch: true,
                variables: _this.variables,
            }))
                .then(function (result) { return _this.queryManager.transformResult(result); });
        };
        this.setOptions = function (opts) {
            _this.options = assign({}, _this.options, opts);
            if (opts.pollInterval) {
                _this.startPolling(opts.pollInterval);
            }
            else if (opts.pollInterval === 0) {
                _this.stopPolling();
            }
            return _this.setVariables(opts.variables);
        };
        this.setVariables = function (variables) {
            var newVariables = assign({}, _this.variables, variables);
            if (isEqual(newVariables, _this.variables)) {
                return _this.result();
            }
            else {
                _this.variables = newVariables;
                return _this.queryManager.fetchQuery(_this.queryId, assign(_this.options, {
                    variables: _this.variables,
                }))
                    .then(function (result) { return _this.queryManager.transformResult(result); });
            }
        };
        this.fetchMore = function (fetchMoreOptions) {
            return Promise.resolve()
                .then(function () {
                var qid = _this.queryManager.generateQueryId();
                var combinedOptions = null;
                if (fetchMoreOptions.query) {
                    combinedOptions = fetchMoreOptions;
                }
                else {
                    var variables = assign({}, _this.variables, fetchMoreOptions.variables);
                    combinedOptions = assign({}, _this.options, fetchMoreOptions, {
                        variables: variables,
                    });
                }
                combinedOptions = assign({}, combinedOptions, {
                    forceFetch: true,
                });
                return _this.queryManager.fetchQuery(qid, combinedOptions);
            })
                .then(function (fetchMoreResult) {
                var reducer = fetchMoreOptions.updateQuery;
                var mapFn = function (previousResult, _a) {
                    var queryVariables = _a.queryVariables;
                    return reducer(previousResult, {
                        fetchMoreResult: fetchMoreResult,
                        queryVariables: queryVariables,
                    });
                };
                _this.updateQuery(mapFn);
                return fetchMoreResult;
            });
        };
        this.updateQuery = function (mapFn) {
            var _a = _this.queryManager.getQueryWithPreviousResult(_this.queryId), previousResult = _a.previousResult, queryVariables = _a.queryVariables, querySelectionSet = _a.querySelectionSet, _b = _a.queryFragments, queryFragments = _b === void 0 ? [] : _b;
            var newResult = errorHandling_1.tryFunctionOrLogError(function () { return mapFn(previousResult, { queryVariables: queryVariables }); });
            if (newResult) {
                _this.queryManager.store.dispatch({
                    type: 'APOLLO_UPDATE_QUERY_RESULT',
                    newResult: newResult,
                    queryVariables: queryVariables,
                    querySelectionSet: querySelectionSet,
                    queryFragments: queryFragments,
                });
            }
        };
        this.stopPolling = function () {
            if (isPollingQuery) {
                _this.scheduler.stopPollingQuery(_this.queryId);
            }
        };
        this.startPolling = function (pollInterval) {
            if (_this.options.noFetch) {
                throw new Error('noFetch option should not use query polling.');
            }
            if (isPollingQuery) {
                _this.scheduler.stopPollingQuery(_this.queryId);
            }
            _this.options.pollInterval = pollInterval;
            _this.scheduler.startPollingQuery(_this.options, _this.queryId, false);
        };
    }
    ObservableQuery.prototype.result = function () {
        var _this = this;
        return new Promise(function (resolve, reject) {
            var subscription = _this.subscribe({
                next: function (result) {
                    resolve(result);
                    setTimeout(function () {
                        subscription.unsubscribe();
                    }, 0);
                },
                error: function (error) {
                    reject(error);
                },
            });
        });
    };
    ObservableQuery.prototype.currentResult = function () {
        var _a = this.queryManager.getCurrentQueryResult(this), data = _a.data, partial = _a.partial;
        var queryStoreValue = this.queryManager.getApolloState().queries[this.queryId];
        var queryLoading = !queryStoreValue || queryStoreValue.loading;
        var loading = (this.options.forceFetch && queryLoading)
            || (partial && !this.options.noFetch);
        return { data: data, loading: loading };
    };
    return ObservableQuery;
}(Observable_1.Observable));
exports.ObservableQuery = ObservableQuery;
//# sourceMappingURL=ObservableQuery.js.map