"use strict";
var getFromAST_1 = require('../queries/getFromAST');
var storeUtils_1 = require('../data/storeUtils');
var assign = require('lodash.assign');
var cloneDeep = require('lodash.clonedeep');
var isArray = require('lodash.isarray');
var isNull = require('lodash.isnull');
var isUndefined = require('lodash.isundefined');
function mergeRequests(requests) {
    var rootQueryDoc = createEmptyRootQueryDoc();
    var rootVariables;
    requests.forEach(function (request, requestIndex) {
        request = cloneDeep(request);
        rootQueryDoc = addQueryToRoot(rootQueryDoc, request.query, requestIndex);
        if (request.variables) {
            rootVariables = addVariablesToRoot(rootVariables, request.variables, request.query, requestIndex);
        }
    });
    var rootRequest = {
        debugName: '___composed',
        query: rootQueryDoc,
        variables: rootVariables,
    };
    return rootRequest;
}
exports.mergeRequests = mergeRequests;
function unpackMergedResult(result, childRequests) {
    var resultArray = childRequests.map(function (request, index) {
        var unpackedData = unpackDataForRequest({
            request: request,
            data: result.data,
            selectionSet: getFromAST_1.getQueryDefinition(request.query).selectionSet,
            queryIndex: index,
            startIndex: 0,
            fragmentMap: getFromAST_1.createFragmentMap(getFromAST_1.getFragmentDefinitions(request.query)),
            topLevel: true,
        }).unpackedData;
        return assign({}, result, { data: unpackedData });
    });
    return resultArray;
}
exports.unpackMergedResult = unpackMergedResult;
function unpackDataForRequest(_a) {
    var request = _a.request, data = _a.data, selectionSet = _a.selectionSet, queryIndex = _a.queryIndex, startIndex = _a.startIndex, fragmentMap = _a.fragmentMap, topLevel = _a.topLevel;
    if (!selectionSet) {
        return {
            newIndex: startIndex,
            unpackedData: {},
        };
    }
    var unpackedData = {};
    var currIndex = startIndex;
    selectionSet.selections.forEach(function (selection) {
        if (selection.kind === 'Field') {
            var field = selection;
            var realName = storeUtils_1.resultKeyNameFromField(field);
            var aliasName = getOperationDefinitionName(getFromAST_1.getQueryDefinition(request.query), queryIndex);
            var stringKey = topLevel ? aliasName + "___fieldIndex_" + currIndex : realName;
            if (topLevel) {
                currIndex += 1;
            }
            var childData = isNull(data) ? null : data[stringKey];
            var resData = childData;
            if (field.selectionSet && field.selectionSet.selections.length > 0) {
                var fieldOpts_1 = {
                    request: request,
                    data: childData,
                    selectionSet: field.selectionSet,
                    queryIndex: queryIndex,
                    fragmentMap: fragmentMap,
                    startIndex: currIndex,
                    topLevel: false,
                };
                if (isNull(childData)) {
                    var selectionRet = unpackDataForRequest(assign(fieldOpts_1, {
                        startIndex: currIndex,
                    }));
                    currIndex = selectionRet.newIndex;
                    resData = childData;
                }
                else if (isArray(childData)) {
                    var resUnpacked_1 = [];
                    var newIndex_1 = 0;
                    childData.forEach(function (dataObject) {
                        var selectionRet = unpackDataForRequest(assign(fieldOpts_1, {
                            data: dataObject,
                            startIndex: currIndex,
                        }));
                        newIndex_1 = selectionRet.newIndex;
                        resUnpacked_1.push(selectionRet.unpackedData);
                    });
                    currIndex = newIndex_1;
                    resData = resUnpacked_1;
                }
                else {
                    var selectionRet = unpackDataForRequest(assign(fieldOpts_1, { startIndex: currIndex }));
                    resData = selectionRet.unpackedData;
                    currIndex = selectionRet.newIndex;
                }
            }
            if (!isUndefined(childData)) {
                unpackedData[realName] = resData;
            }
        }
        else if (selection.kind === 'InlineFragment') {
            var inlineFragment = selection;
            var ret = unpackDataForRequest({
                request: request,
                data: data,
                selectionSet: inlineFragment.selectionSet,
                queryIndex: queryIndex,
                startIndex: currIndex,
                fragmentMap: fragmentMap,
                topLevel: topLevel,
            });
            assign(unpackedData, ret.unpackedData);
            currIndex = ret.newIndex;
        }
        else if (selection.kind === 'FragmentSpread') {
            var fragmentSpread = selection;
            var fragment = fragmentMap[fragmentSpread.name.value];
            var fragmentRet = unpackDataForRequest({
                request: request,
                data: data,
                selectionSet: fragment.selectionSet,
                queryIndex: queryIndex,
                startIndex: currIndex,
                fragmentMap: fragmentMap,
                topLevel: true,
            });
            assign(unpackedData, fragmentRet.unpackedData);
            currIndex = fragmentRet.newIndex;
        }
    });
    return {
        newIndex: currIndex,
        unpackedData: unpackedData,
    };
}
exports.unpackDataForRequest = unpackDataForRequest;
function mergeQueryDocuments(childQueryDocs) {
    var rootQueryDoc = createEmptyRootQueryDoc();
    childQueryDocs.forEach(function (childQueryDoc, childQueryDocIndex) {
        rootQueryDoc = addQueryToRoot(rootQueryDoc, childQueryDoc, childQueryDocIndex);
    });
    return rootQueryDoc;
}
exports.mergeQueryDocuments = mergeQueryDocuments;
function addVariablesToRoot(rootVariables, childVariables, childQueryDoc, childQueryDocIndex) {
    var aliasName = getOperationDefinitionName(getFromAST_1.getQueryDefinition(childQueryDoc), childQueryDocIndex);
    var aliasedChildVariables = addPrefixToVariables(aliasName + '___', childVariables);
    return assign({}, rootVariables, aliasedChildVariables);
}
exports.addVariablesToRoot = addVariablesToRoot;
function addQueryToRoot(rootQueryDoc, childQueryDoc, childQueryDocIndex) {
    var aliasName = getOperationDefinitionName(getFromAST_1.getQueryDefinition(childQueryDoc), childQueryDocIndex);
    var aliasedChild = applyAliasNameToDocument(childQueryDoc, aliasName);
    var aliasedChildQueryDef = getFromAST_1.getQueryDefinition(aliasedChild);
    var aliasedChildFragmentDefs = getFromAST_1.getFragmentDefinitions(aliasedChild);
    var rootQueryDef = getFromAST_1.getQueryDefinition(rootQueryDoc);
    rootQueryDoc.definitions = rootQueryDoc.definitions.concat(aliasedChildFragmentDefs);
    rootQueryDef.selectionSet.selections =
        rootQueryDef.selectionSet.selections.concat(aliasedChildQueryDef.selectionSet.selections);
    rootQueryDef.variableDefinitions =
        rootQueryDef.variableDefinitions.concat(aliasedChildQueryDef.variableDefinitions);
    return rootQueryDoc;
}
exports.addQueryToRoot = addQueryToRoot;
function createEmptyRootQueryDoc(rootQueryName) {
    if (!rootQueryName) {
        rootQueryName = '___composed';
    }
    return {
        kind: 'Document',
        definitions: [
            {
                kind: 'OperationDefinition',
                operation: 'query',
                name: {
                    kind: 'Name',
                    value: rootQueryName,
                },
                variableDefinitions: [],
                directives: [],
                selectionSet: {
                    kind: 'SelectionSet',
                    selections: [],
                },
            },
        ],
    };
}
exports.createEmptyRootQueryDoc = createEmptyRootQueryDoc;
function renameFragmentSpreads(selSet, aliasName) {
    if (selSet && selSet.selections) {
        selSet.selections = selSet.selections.map(function (selection) {
            if (selection.kind === 'FragmentSpread') {
                var fragmentSpread = selection;
                fragmentSpread.name.value = getFragmentAliasName(fragmentSpread, aliasName);
                return fragmentSpread;
            }
            else {
                var withSelSet = selection;
                withSelSet.selectionSet = renameFragmentSpreads(withSelSet.selectionSet, aliasName);
                return selection;
            }
        });
    }
    return selSet;
}
exports.renameFragmentSpreads = renameFragmentSpreads;
function renameVariablesInArgument(argument, aliasName) {
    if (argument.kind === 'Argument' &&
        argument.value.kind === 'Variable') {
        var varx = argument.value;
        argument.value.name.value = getVariableAliasName(varx, aliasName);
    }
    return argument;
}
function renameVariables(selSet, aliasName) {
    if (selSet && selSet.selections) {
        selSet.selections = selSet.selections.map(function (selection) {
            if (selection.kind === 'Field') {
                var field = selection;
                if (field.arguments) {
                    field.arguments = field.arguments.map(function (argument) {
                        return renameVariablesInArgument(argument, aliasName);
                    });
                }
                if (field.directives) {
                    field.directives = field.directives.map(function (directive) {
                        if (directive.arguments) {
                            directive.arguments = directive.arguments.map(function (argument) {
                                return renameVariablesInArgument(argument, aliasName);
                            });
                        }
                        return directive;
                    });
                }
                field.selectionSet = renameVariables(field.selectionSet, aliasName);
                return field;
            }
            else if (selection.kind === 'InlineFragment') {
                var inlineFragment = selection;
                inlineFragment.selectionSet = renameVariables(inlineFragment.selectionSet, aliasName);
                return inlineFragment;
            }
            return selection;
        });
    }
    return selSet;
}
exports.renameVariables = renameVariables;
function applyAliasNameToVariableDefinition(vDef, aliasName) {
    if (containsMarker(vDef.variable.name.value)) {
        throw new Error("Variable definition for " + vDef.variable.name.value + " contains \"___\"");
    }
    vDef.variable.name.value = getVariableAliasName(vDef.variable, aliasName);
    return vDef;
}
exports.applyAliasNameToVariableDefinition = applyAliasNameToVariableDefinition;
function applyAliasNameToDocument(document, aliasName) {
    document.definitions = document.definitions.map(function (definition) {
        var operationOrFragmentDef = definition;
        operationOrFragmentDef.selectionSet =
            renameFragmentSpreads(operationOrFragmentDef.selectionSet, aliasName);
        operationOrFragmentDef.selectionSet =
            renameVariables(operationOrFragmentDef.selectionSet, aliasName);
        return operationOrFragmentDef;
    });
    var currStartIndex = 0;
    document.definitions = document.definitions.map(function (definition) {
        if (definition.kind === 'OperationDefinition' &&
            definition.operation === 'query') {
            var operationDef = definition;
            if (operationDef.variableDefinitions) {
                operationDef.variableDefinitions =
                    operationDef.variableDefinitions.map(function (vDef) {
                        return applyAliasNameToVariableDefinition(vDef, aliasName);
                    });
            }
            var retDef = applyAliasNameToTopLevelFields(operationDef, aliasName, currStartIndex);
            currStartIndex += operationDef.selectionSet.selections.length;
            return retDef;
        }
        else if (definition.kind === 'FragmentDefinition') {
            var fragmentDef = definition;
            var retDef = applyAliasNameToFragment(fragmentDef, aliasName, currStartIndex);
            currStartIndex += fragmentDef.selectionSet.selections.length;
            return retDef;
        }
        else {
            throw new Error('Cannot apply alias names to documents that contain mutations.');
        }
    });
    return document;
}
exports.applyAliasNameToDocument = applyAliasNameToDocument;
function applyAliasNameToFragment(fragment, aliasName, startIndex) {
    if (containsMarker(fragment.name.value)) {
        throw new Error("Fragment " + fragment.name.value + " contains \"___\"");
    }
    fragment.name.value = getFragmentAliasName(fragment, aliasName);
    fragment.selectionSet.selections =
        applyAliasNameToSelections(fragment.selectionSet.selections, aliasName, startIndex).res;
    return fragment;
}
exports.applyAliasNameToFragment = applyAliasNameToFragment;
function applyAliasNameToTopLevelFields(childQuery, aliasName, startIndex) {
    childQuery.selectionSet.selections =
        applyAliasNameToSelections(childQuery.selectionSet.selections, aliasName, startIndex).res;
    return childQuery;
}
exports.applyAliasNameToTopLevelFields = applyAliasNameToTopLevelFields;
function getVariableAliasName(varNode, aliasName) {
    return aliasName + "___" + varNode.name.value;
}
exports.getVariableAliasName = getVariableAliasName;
function getFragmentAliasName(fragment, queryAliasName) {
    return queryAliasName + "___" + fragment.name.value;
}
exports.getFragmentAliasName = getFragmentAliasName;
function getOperationDefinitionName(operationDef, requestIndex) {
    var operationDefName = '';
    if (operationDef.name) {
        operationDefName = operationDef.name.value;
    }
    return "___" + operationDefName + "___requestIndex_" + requestIndex;
}
exports.getOperationDefinitionName = getOperationDefinitionName;
function aliasField(field, alias) {
    if (containsMarker(field.name.value)) {
        throw new Error("Field " + field.name.value + " contains \"___\".");
    }
    field.alias = {
        kind: 'Name',
        value: alias,
    };
    return field;
}
exports.aliasField = aliasField;
function addPrefixToQuery(prefix, query) {
    if (query.name) {
        query.name.value = prefix + query.name.value;
    }
    return query;
}
exports.addPrefixToQuery = addPrefixToQuery;
function addPrefixToVariables(prefix, variables) {
    var newVariables = {};
    Object.keys(variables).forEach(function (variableName) {
        newVariables[prefix + variableName] = variables[variableName];
    });
    return newVariables;
}
exports.addPrefixToVariables = addPrefixToVariables;
function applyAliasNameToSelections(selections, aliasName, startIndex) {
    var currIndex = startIndex;
    var res = selections.map(function (selection) {
        if (selection.kind === 'Field') {
            var aliasedField = aliasField(selection, aliasName + "___fieldIndex_" + currIndex);
            currIndex += 1;
            return aliasedField;
        }
        else if (selection.kind === 'InlineFragment') {
            var inlineFragment = selection;
            var ret = applyAliasNameToSelections(inlineFragment.selectionSet.selections, aliasName, currIndex);
            inlineFragment.selectionSet.selections = ret.res;
            currIndex = ret.newIndex;
            return inlineFragment;
        }
        else {
            return selection;
        }
    });
    return {
        res: res,
        newIndex: currIndex,
    };
}
function containsMarker(name) {
    return name.indexOf('___') > -1;
}
//# sourceMappingURL=queryMerging.js.map