"use strict";
var isArray = require('lodash.isarray');
var isNull = require('lodash.isnull');
var isObject = require('lodash.isobject');
var has = require('lodash.has');
var merge = require('lodash.merge');
var storeUtils_1 = require('./storeUtils');
var store_1 = require('./store');
var getFromAST_1 = require('../queries/getFromAST');
var directives_1 = require('../queries/directives');
var errors_1 = require('../errors');
var flatten = require('lodash.flatten');
function diffQueryAgainstStore(_a) {
    var store = _a.store, query = _a.query, variables = _a.variables;
    var queryDef = getFromAST_1.getQueryDefinition(query);
    return diffSelectionSetAgainstStore({
        store: store,
        rootId: 'ROOT_QUERY',
        selectionSet: queryDef.selectionSet,
        throwOnMissingField: false,
        variables: variables,
    });
}
exports.diffQueryAgainstStore = diffQueryAgainstStore;
function diffFragmentAgainstStore(_a) {
    var store = _a.store, fragment = _a.fragment, rootId = _a.rootId, variables = _a.variables;
    var fragmentDef = getFromAST_1.getFragmentDefinition(fragment);
    return diffSelectionSetAgainstStore({
        store: store,
        rootId: rootId,
        selectionSet: fragmentDef.selectionSet,
        throwOnMissingField: false,
        variables: variables,
    });
}
exports.diffFragmentAgainstStore = diffFragmentAgainstStore;
function handleFragmentErrors(fragmentErrors) {
    var typenames = Object.keys(fragmentErrors);
    if (typenames.length === 0) {
        return;
    }
    var errorTypes = typenames.filter(function (typename) {
        return (fragmentErrors[typename] !== null);
    });
    if (errorTypes.length === Object.keys(fragmentErrors).length) {
        throw fragmentErrors[errorTypes[0]];
    }
}
exports.handleFragmentErrors = handleFragmentErrors;
function diffSelectionSetAgainstStore(_a) {
    var selectionSet = _a.selectionSet, store = _a.store, rootId = _a.rootId, _b = _a.throwOnMissingField, throwOnMissingField = _b === void 0 ? false : _b, variables = _a.variables, fragmentMap = _a.fragmentMap;
    if (selectionSet.kind !== 'SelectionSet') {
        throw new Error('Must be a selection set.');
    }
    if (!fragmentMap) {
        fragmentMap = {};
    }
    var result = {};
    var missingFields = [];
    var fragmentErrors = {};
    selectionSet.selections.forEach(function (selection) {
        var missingFieldPushed = false;
        var fieldResult;
        var fieldIsMissing;
        function pushMissingField(missingField) {
            if (!missingFieldPushed) {
                missingFields.push(missingField);
                missingFieldPushed = true;
            }
        }
        var included = directives_1.shouldInclude(selection, variables);
        if (storeUtils_1.isField(selection)) {
            var diffResult = diffFieldAgainstStore({
                field: selection,
                throwOnMissingField: throwOnMissingField,
                variables: variables,
                rootId: rootId,
                store: store,
                fragmentMap: fragmentMap,
                included: included,
            });
            fieldIsMissing = diffResult.isMissing;
            fieldResult = diffResult.result;
            var resultFieldKey = storeUtils_1.resultKeyNameFromField(selection);
            if (fieldIsMissing) {
                pushMissingField(selection);
            }
            if (included && fieldResult !== undefined) {
                result[resultFieldKey] = fieldResult;
            }
        }
        else if (storeUtils_1.isInlineFragment(selection)) {
            var typename = selection.typeCondition.name.value;
            if (included) {
                try {
                    var diffResult = diffSelectionSetAgainstStore({
                        selectionSet: selection.selectionSet,
                        throwOnMissingField: throwOnMissingField,
                        variables: variables,
                        rootId: rootId,
                        store: store,
                        fragmentMap: fragmentMap,
                    });
                    fieldIsMissing = diffResult.isMissing;
                    fieldResult = diffResult.result;
                    if (fieldIsMissing) {
                        pushMissingField(selection);
                    }
                    if (isObject(fieldResult)) {
                        merge(result, fieldResult);
                    }
                    if (!fragmentErrors[typename]) {
                        fragmentErrors[typename] = null;
                    }
                }
                catch (e) {
                    if (e.extraInfo && e.extraInfo.isFieldError) {
                        fragmentErrors[typename] = e;
                    }
                    else {
                        throw e;
                    }
                }
            }
        }
        else {
            var fragment = fragmentMap[selection.name.value];
            if (!fragment) {
                throw new Error("No fragment named " + selection.name.value);
            }
            var typename = fragment.typeCondition.name.value;
            if (included) {
                try {
                    var diffResult = diffSelectionSetAgainstStore({
                        selectionSet: fragment.selectionSet,
                        throwOnMissingField: throwOnMissingField,
                        variables: variables,
                        rootId: rootId,
                        store: store,
                        fragmentMap: fragmentMap,
                    });
                    fieldIsMissing = diffResult.isMissing;
                    fieldResult = diffResult.result;
                    if (fieldIsMissing) {
                        pushMissingField(selection);
                    }
                    if (isObject(fieldResult)) {
                        merge(result, fieldResult);
                    }
                    if (!fragmentErrors[typename]) {
                        fragmentErrors[typename] = null;
                    }
                }
                catch (e) {
                    if (e.extraInfo && e.extraInfo.isFieldError) {
                        fragmentErrors[typename] = e;
                    }
                    else {
                        throw e;
                    }
                }
            }
        }
    });
    if (throwOnMissingField) {
        handleFragmentErrors(fragmentErrors);
    }
    var isMissing;
    var missingSelectionSets;
    if (missingFields.length) {
        if (rootId === 'ROOT_QUERY') {
            var typeName = 'Query';
            missingSelectionSets = [
                {
                    id: rootId,
                    typeName: typeName,
                    selectionSet: {
                        kind: 'SelectionSet',
                        selections: missingFields,
                    },
                },
            ];
        }
        else {
            isMissing = 'true';
        }
    }
    return {
        result: result,
        isMissing: isMissing,
        missingSelectionSets: missingSelectionSets,
    };
}
exports.diffSelectionSetAgainstStore = diffSelectionSetAgainstStore;
function diffFieldAgainstStore(_a) {
    var field = _a.field, throwOnMissingField = _a.throwOnMissingField, variables = _a.variables, rootId = _a.rootId, store = _a.store, fragmentMap = _a.fragmentMap, _b = _a.included, included = _b === void 0 ? true : _b;
    var storeObj = store[rootId] || {};
    var storeFieldKey = storeUtils_1.storeKeyNameFromField(field, variables);
    if (!has(storeObj, storeFieldKey)) {
        if (throwOnMissingField && included) {
            throw new errors_1.ApolloError({
                errorMessage: "Can't find field " + storeFieldKey + " on object (" + rootId + ") " + JSON.stringify(storeObj, null, 2) + ".\nPerhaps you want to use the `returnPartialData` option?",
                extraInfo: {
                    isFieldError: true,
                },
            });
        }
        return {
            isMissing: 'true',
        };
    }
    var storeValue = storeObj[storeFieldKey];
    if (!field.selectionSet) {
        if (store_1.isJsonValue(storeValue)) {
            return {
                result: storeValue.json,
            };
        }
        else {
            return {
                result: storeValue,
            };
        }
    }
    if (isNull(storeValue)) {
        return {
            result: null,
        };
    }
    if (isArray(storeValue)) {
        var isMissing_1;
        var result = storeValue.map(function (id) {
            if (isNull(id)) {
                return null;
            }
            var itemDiffResult = diffSelectionSetAgainstStore({
                store: store,
                throwOnMissingField: throwOnMissingField,
                rootId: id,
                selectionSet: field.selectionSet,
                variables: variables,
                fragmentMap: fragmentMap,
            });
            if (itemDiffResult.isMissing) {
                isMissing_1 = 'true';
            }
            return itemDiffResult.result;
        });
        return {
            result: result,
            isMissing: isMissing_1,
        };
    }
    if (store_1.isIdValue(storeValue)) {
        var unescapedId = storeValue.id;
        return diffSelectionSetAgainstStore({
            store: store,
            throwOnMissingField: throwOnMissingField,
            rootId: unescapedId,
            selectionSet: field.selectionSet,
            variables: variables,
            fragmentMap: fragmentMap,
        });
    }
    throw new Error('Unexpected value in the store where the query had a subselection.');
}
function collectUsedVariablesFromSelectionSet(selectionSet) {
    return uniq(flatten(selectionSet.selections.map(function (selection) {
        if (storeUtils_1.isField(selection)) {
            return collectUsedVariablesFromField(selection);
        }
        else if (storeUtils_1.isInlineFragment(selection)) {
            return collectUsedVariablesFromSelectionSet(selection.selectionSet);
        }
        else {
            return [];
        }
    })));
}
function collectUsedVariablesFromDirectives(directives) {
    return flatten(directives.map(function (directive) {
        if (directive.arguments) {
            return flatten(directive.arguments.map(function (arg) {
                if (arg.kind === 'Argument' && arg.value.kind === 'Variable') {
                    return [arg.value.name.value];
                }
                return [];
            }));
        }
        return [];
    }));
}
function collectUsedVariablesFromField(field) {
    var variables = [];
    if (field.arguments) {
        variables = flatten(field.arguments.map(function (arg) {
            if (arg.value.kind === 'Variable') {
                return [arg.value.name.value];
            }
            return [];
        }));
    }
    if (field.selectionSet) {
        variables = variables.concat(collectUsedVariablesFromSelectionSet(field.selectionSet));
    }
    if (field.directives) {
        variables = variables.concat(collectUsedVariablesFromDirectives(field.directives));
    }
    return uniq(variables);
}
function removeUnusedVariablesFromQuery(query) {
    var queryDef = getFromAST_1.getQueryDefinition(query);
    var usedVariables = flatten(query.definitions.map(function (def) { return collectUsedVariablesFromSelectionSet(def.selectionSet); }));
    if (!queryDef.variableDefinitions) {
        return;
    }
    var diffedVariableDefinitions = queryDef.variableDefinitions.filter(function (variableDefinition) {
        return usedVariables.indexOf(variableDefinition.variable.name.value) !== -1;
    });
    queryDef.variableDefinitions = diffedVariableDefinitions;
}
exports.removeUnusedVariablesFromQuery = removeUnusedVariablesFromQuery;
function uniq(array) {
    return array.filter(function (item, index, arr) {
        return arr.indexOf(item) === index;
    });
}
//# sourceMappingURL=diffAgainstStore.js.map