"use strict";
var isString = require('lodash.isstring');
var assign = require('lodash.assign');
var mapValues = require('lodash.mapvalues');
require('whatwg-fetch');
var printer_1 = require('graphql-tag/printer');
var queryMerging_1 = require('./batching/queryMerging');
function addQueryMerging(networkInterface) {
    return assign(networkInterface, {
        batchQuery: function (requests) {
            if (requests.length === 1) {
                return this.query(requests[0]).then(function (result) {
                    return Promise.resolve([result]);
                });
            }
            var composedRequest = queryMerging_1.mergeRequests(requests);
            return this.query(composedRequest).then(function (composedResult) {
                return queryMerging_1.unpackMergedResult(composedResult, requests);
            });
        },
    });
}
exports.addQueryMerging = addQueryMerging;
function printRequest(request) {
    return mapValues(request, function (val, key) {
        return key === 'query' ? printer_1.print(val) : val;
    });
}
exports.printRequest = printRequest;
var HTTPFetchNetworkInterface = (function () {
    function HTTPFetchNetworkInterface(uri, opts) {
        if (opts === void 0) { opts = {}; }
        if (!uri) {
            throw new Error('A remote enpdoint is required for a network layer');
        }
        if (!isString(uri)) {
            throw new Error('Remote endpoint must be a string');
        }
        this._uri = uri;
        this._opts = assign({}, opts);
        this._middlewares = [];
        this._afterwares = [];
    }
    HTTPFetchNetworkInterface.prototype.applyMiddlewares = function (_a) {
        var _this = this;
        var request = _a.request, options = _a.options;
        return new Promise(function (resolve, reject) {
            var queue = function (funcs, scope) {
                var next = function () {
                    if (funcs.length > 0) {
                        var f = funcs.shift();
                        f.applyMiddleware.apply(scope, [{ request: request, options: options }, next]);
                    }
                    else {
                        resolve({
                            request: request,
                            options: options,
                        });
                    }
                };
                next();
            };
            queue(_this._middlewares.slice(), _this);
        });
    };
    HTTPFetchNetworkInterface.prototype.applyAfterwares = function (_a) {
        var _this = this;
        var response = _a.response, options = _a.options;
        return new Promise(function (resolve, reject) {
            var queue = function (funcs, scope) {
                var next = function () {
                    if (funcs.length > 0) {
                        var f = funcs.shift();
                        f.applyAfterware.apply(scope, [{ response: response, options: options }, next]);
                    }
                    else {
                        resolve({
                            response: response,
                            options: options,
                        });
                    }
                };
                next();
            };
            queue(_this._afterwares.slice(), _this);
        });
    };
    HTTPFetchNetworkInterface.prototype.fetchFromRemoteEndpoint = function (_a) {
        var request = _a.request, options = _a.options;
        return fetch(this._uri, assign({}, this._opts, {
            body: JSON.stringify(printRequest(request)),
            method: 'POST',
        }, options, {
            headers: assign({}, {
                Accept: '*/*',
                'Content-Type': 'application/json',
            }, options.headers),
        }));
    };
    ;
    HTTPFetchNetworkInterface.prototype.query = function (request) {
        var _this = this;
        var options = assign({}, this._opts);
        return this.applyMiddlewares({
            request: request,
            options: options,
        }).then(this.fetchFromRemoteEndpoint.bind(this))
            .then(function (response) {
            _this.applyAfterwares({
                response: response,
                options: options,
            });
            return response;
        })
            .then(function (result) { return result.json(); })
            .then(function (payload) {
            if (!payload.hasOwnProperty('data') && !payload.hasOwnProperty('errors')) {
                throw new Error("Server response was missing for query '" + request.debugName + "'.");
            }
            else {
                return payload;
            }
        });
    };
    ;
    HTTPFetchNetworkInterface.prototype.use = function (middlewares) {
        var _this = this;
        middlewares.map(function (middleware) {
            if (typeof middleware.applyMiddleware === 'function') {
                _this._middlewares.push(middleware);
            }
            else {
                throw new Error('Middleware must implement the applyMiddleware function');
            }
        });
    };
    HTTPFetchNetworkInterface.prototype.useAfter = function (afterwares) {
        var _this = this;
        afterwares.map(function (afterware) {
            if (typeof afterware.applyAfterware === 'function') {
                _this._afterwares.push(afterware);
            }
            else {
                throw new Error('Afterware must implement the applyAfterware function');
            }
        });
    };
    return HTTPFetchNetworkInterface;
}());
exports.HTTPFetchNetworkInterface = HTTPFetchNetworkInterface;
var batchedNetworkInterface_1 = require('./batchedNetworkInterface');
function createNetworkInterface(interfaceOpts, backOpts) {
    if (backOpts === void 0) { backOpts = {}; }
    if (isString(interfaceOpts) || !interfaceOpts) {
        var uri = interfaceOpts;
        return addQueryMerging(new HTTPFetchNetworkInterface(uri, backOpts));
    }
    else {
        var _a = interfaceOpts, _b = _a.transportBatching, transportBatching = _b === void 0 ? false : _b, _c = _a.opts, opts = _c === void 0 ? {} : _c, uri = _a.uri;
        if (transportBatching) {
            return new batchedNetworkInterface_1.HTTPBatchedNetworkInterface(uri, opts);
        }
        else {
            return addQueryMerging(new HTTPFetchNetworkInterface(uri, opts));
        }
    }
}
exports.createNetworkInterface = createNetworkInterface;
//# sourceMappingURL=networkInterface.js.map