"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFieldDef = exports.getOperationRootType = exports.isTypeProperSuperTypeOf = exports.valueFromValueNode = exports.filePathForNode = exports.sourceAt = exports.removeClientDirectives = exports.removeConnectionDirectives = exports.isMetaFieldName = exports.isList = exports.sortEnumValues = void 0;
const graphql_1 = require("graphql");
function sortEnumValues(values) {
    return values.sort((a, b) => a.value < b.value ? -1 : a.value > b.value ? 1 : 0);
}
exports.sortEnumValues = sortEnumValues;
function isList(type) {
    return graphql_1.isListType(type) || (graphql_1.isNonNullType(type) && graphql_1.isListType(type.ofType));
}
exports.isList = isList;
function isMetaFieldName(name) {
    return name.startsWith("__");
}
exports.isMetaFieldName = isMetaFieldName;
function removeConnectionDirectives(ast) {
    return graphql_1.visit(ast, {
        Directive(node) {
            if (node.name.value === "connection")
                return null;
            return node;
        }
    });
}
exports.removeConnectionDirectives = removeConnectionDirectives;
function removeClientDirectives(ast) {
    return graphql_1.visit(ast, {
        Field(node) {
            if (node.directives &&
                node.directives.find(directive => directive.name.value === "client"))
                return null;
            return node;
        },
        OperationDefinition: {
            leave(node) {
                if (!node.selectionSet.selections.length)
                    return null;
                return node;
            }
        }
    });
}
exports.removeClientDirectives = removeClientDirectives;
const typenameField = {
    kind: graphql_1.Kind.FIELD,
    name: { kind: graphql_1.Kind.NAME, value: "__typename" }
};
function sourceAt(location) {
    return location.source.body.slice(location.start, location.end);
}
exports.sourceAt = sourceAt;
function filePathForNode(node) {
    const name = node.loc && node.loc.source && node.loc.source.name;
    if (!name || name === "GraphQL") {
        throw new Error("Node does not seem to have a file path");
    }
    return name;
}
exports.filePathForNode = filePathForNode;
function valueFromValueNode(valueNode) {
    switch (valueNode.kind) {
        case "IntValue":
        case "FloatValue":
            return Number(valueNode.value);
        case "NullValue":
            return null;
        case "ListValue":
            return valueNode.values.map(valueFromValueNode);
        case "ObjectValue":
            return valueNode.fields.reduce((object, field) => {
                object[field.name.value] = valueFromValueNode(field.value);
                return object;
            }, {});
        case "Variable":
            return { kind: "Variable", variableName: valueNode.name.value };
        default:
            return valueNode.value;
    }
}
exports.valueFromValueNode = valueFromValueNode;
function isTypeProperSuperTypeOf(schema, maybeSuperType, subType) {
    return (graphql_1.isEqualType(maybeSuperType, subType) ||
        (graphql_1.isObjectType(subType) &&
            (graphql_1.isAbstractType(maybeSuperType) &&
                schema.isPossibleType(maybeSuperType, subType))));
}
exports.isTypeProperSuperTypeOf = isTypeProperSuperTypeOf;
function getOperationRootType(schema, operation) {
    switch (operation.operation) {
        case "query":
            return schema.getQueryType();
        case "mutation":
            const mutationType = schema.getMutationType();
            if (!mutationType) {
                throw new graphql_1.GraphQLError("Schema is not configured for mutations", [
                    operation
                ]);
            }
            return mutationType;
        case "subscription":
            const subscriptionType = schema.getSubscriptionType();
            if (!subscriptionType) {
                throw new graphql_1.GraphQLError("Schema is not configured for subscriptions", [
                    operation
                ]);
            }
            return subscriptionType;
        default:
            throw new graphql_1.GraphQLError("Can only compile queries, mutations and subscriptions", [operation]);
    }
}
exports.getOperationRootType = getOperationRootType;
function getFieldDef(schema, parentType, fieldAST) {
    const name = fieldAST.name.value;
    if (name === graphql_1.SchemaMetaFieldDef.name &&
        schema.getQueryType() === parentType) {
        return graphql_1.SchemaMetaFieldDef;
    }
    if (name === graphql_1.TypeMetaFieldDef.name && schema.getQueryType() === parentType) {
        return graphql_1.TypeMetaFieldDef;
    }
    if (name === graphql_1.TypeNameMetaFieldDef.name &&
        (graphql_1.isObjectType(parentType) ||
            graphql_1.isInterfaceType(parentType) ||
            graphql_1.isUnionType(parentType))) {
        return graphql_1.TypeNameMetaFieldDef;
    }
    if (graphql_1.isObjectType(parentType) || graphql_1.isInterfaceType(parentType)) {
        return parentType.getFields()[name];
    }
    return undefined;
}
exports.getFieldDef = getFieldDef;
//# sourceMappingURL=graphql.js.map