import { __assign, __extends } from 'tslib';
import { isReference } from '../../utilities/graphql/storeUtils.js';
import { dep } from 'optimism';

var hasOwn = Object.prototype.hasOwnProperty;
var EntityCache = (function () {
    function EntityCache() {
        this.data = Object.create(null);
        this.depend = null;
        this.rootIds = Object.create(null);
        this.refs = Object.create(null);
    }
    EntityCache.prototype.toObject = function () {
        return __assign({}, this.data);
    };
    EntityCache.prototype.has = function (dataId) {
        return hasOwn.call(this.data, dataId);
    };
    EntityCache.prototype.get = function (dataId) {
        if (this.depend)
            this.depend(dataId);
        return this.data[dataId];
    };
    EntityCache.prototype.set = function (dataId, value) {
        if (!hasOwn.call(this.data, dataId) || value !== this.data[dataId]) {
            this.data[dataId] = value;
            delete this.refs[dataId];
            if (this.depend)
                this.depend.dirty(dataId);
        }
    };
    EntityCache.prototype.delete = function (dataId) {
        delete this.data[dataId];
        delete this.refs[dataId];
        if (this.depend)
            this.depend.dirty(dataId);
    };
    EntityCache.prototype.clear = function () {
        this.replace(null);
    };
    EntityCache.prototype.replace = function (newData) {
        var _this = this;
        Object.keys(this.data).forEach(function (dataId) {
            if (!(newData && hasOwn.call(newData, dataId))) {
                _this.delete(dataId);
            }
        });
        if (newData) {
            Object.keys(newData).forEach(function (dataId) {
                _this.set(dataId, newData[dataId]);
            });
        }
    };
    EntityCache.prototype.retain = function (rootId) {
        return this.rootIds[rootId] = (this.rootIds[rootId] || 0) + 1;
    };
    EntityCache.prototype.release = function (rootId) {
        if (this.rootIds[rootId] > 0) {
            var count = --this.rootIds[rootId];
            if (!count)
                delete this.rootIds[rootId];
            return count;
        }
        return 0;
    };
    EntityCache.prototype.getRootIdSet = function () {
        return new Set(Object.keys(this.rootIds));
    };
    EntityCache.prototype.gc = function () {
        var _this = this;
        var ids = this.getRootIdSet();
        var snapshot = this.toObject();
        ids.forEach(function (id) {
            if (hasOwn.call(snapshot, id)) {
                Object.keys(_this.findChildRefIds(id)).forEach(ids.add, ids);
                delete snapshot[id];
            }
        });
        var idsToRemove = Object.keys(snapshot);
        if (idsToRemove.length) {
            var root = this;
            while (root instanceof Layer)
                root = root.parent;
            idsToRemove.forEach(root.delete, root);
        }
        return idsToRemove;
    };
    EntityCache.prototype.findChildRefIds = function (dataId) {
        if (!hasOwn.call(this.refs, dataId)) {
            var found_1 = this.refs[dataId] = Object.create(null);
            var workSet_1 = new Set([this.data[dataId]]);
            var canTraverse_1 = function (obj) { return obj !== null && typeof obj === 'object'; };
            workSet_1.forEach(function (obj) {
                if (isReference(obj)) {
                    found_1[obj.__ref] = true;
                }
                else if (canTraverse_1(obj)) {
                    Object.values(obj)
                        .filter(canTraverse_1)
                        .forEach(workSet_1.add, workSet_1);
                }
            });
        }
        return this.refs[dataId];
    };
    return EntityCache;
}());
(function (EntityCache) {
    var Root = (function (_super) {
        __extends(Root, _super);
        function Root(_a) {
            var _b = _a.resultCaching, resultCaching = _b === void 0 ? true : _b, seed = _a.seed;
            var _this = _super.call(this) || this;
            _this.sharedLayerDepend = null;
            if (resultCaching) {
                _this.depend = dep();
                _this.sharedLayerDepend = dep();
            }
            if (seed)
                _this.replace(seed);
            return _this;
        }
        Root.prototype.addLayer = function (layerId, replay) {
            return new Layer(layerId, this, replay, this.sharedLayerDepend);
        };
        Root.prototype.removeLayer = function (layerId) {
            return this;
        };
        return Root;
    }(EntityCache));
    EntityCache.Root = Root;
})(EntityCache || (EntityCache = {}));
var Layer = (function (_super) {
    __extends(Layer, _super);
    function Layer(id, parent, replay, depend) {
        var _this = _super.call(this) || this;
        _this.id = id;
        _this.parent = parent;
        _this.replay = replay;
        _this.depend = depend;
        replay(_this);
        return _this;
    }
    Layer.prototype.addLayer = function (layerId, replay) {
        return new Layer(layerId, this, replay, this.depend);
    };
    Layer.prototype.removeLayer = function (layerId) {
        var _this = this;
        var parent = this.parent.removeLayer(layerId);
        if (layerId === this.id) {
            if (this.depend) {
                Object.keys(this.data).forEach(function (dataId) { return _this.depend.dirty(dataId); });
            }
            return parent;
        }
        if (parent === this.parent)
            return this;
        return parent.addLayer(this.id, this.replay);
    };
    Layer.prototype.toObject = function () {
        return __assign(__assign({}, this.parent.toObject()), this.data);
    };
    Layer.prototype.has = function (dataId) {
        if (hasOwn.call(this.data, dataId) && this.data[dataId] === void 0) {
            return false;
        }
        return this.parent.has(dataId);
    };
    Layer.prototype.delete = function (dataId) {
        _super.prototype.delete.call(this, dataId);
        this.data[dataId] = void 0;
    };
    Layer.prototype.get = function (dataId) {
        if (hasOwn.call(this.data, dataId)) {
            return _super.prototype.get.call(this, dataId);
        }
        if (this.depend && this.depend !== this.parent.depend) {
            this.depend(dataId);
        }
        return this.parent.get(dataId);
    };
    Layer.prototype.getRootIdSet = function () {
        var ids = this.parent.getRootIdSet();
        _super.prototype.getRootIdSet.call(this).forEach(ids.add, ids);
        return ids;
    };
    Layer.prototype.findChildRefIds = function (dataId) {
        var fromParent = this.parent.findChildRefIds(dataId);
        return hasOwn.call(this.data, dataId) ? __assign(__assign({}, fromParent), _super.prototype.findChildRefIds.call(this, dataId)) : fromParent;
    };
    return Layer;
}(EntityCache));
function supportsResultCaching(store) {
    return !!(store instanceof EntityCache && store.depend);
}
function defaultNormalizedCacheFactory(seed) {
    return new EntityCache.Root({ resultCaching: true, seed: seed });
}

export { EntityCache, defaultNormalizedCacheFactory, supportsResultCaching };
//# sourceMappingURL=entityCache.js.map
