import { __assign } from 'tslib';
import invariant from 'ts-invariant';
import { getFragmentFromSelection } from '../../utilities/graphql/fragments.js';
import { makeReference, getTypenameFromResult, valueToObjectRepresentation, storeKeyNameFromField, argumentsObjectFromField, isField } from '../../utilities/graphql/storeUtils.js';
import { canUseWeakMap } from '../../utilities/common/canUse.js';
import { KeyTrie } from 'optimism';

var hasOwn = Object.prototype.hasOwnProperty;
function defaultDataIdFromObject(object) {
    var __typename = object.__typename, id = object.id, _id = object._id;
    if (typeof __typename === "string") {
        if (typeof id !== "undefined")
            return __typename + ":" + id;
        if (typeof _id !== "undefined")
            return __typename + ":" + _id;
    }
    return null;
}
var nullKeyFn = function () { return null; };
var Policies = (function () {
    function Policies(config) {
        var _this = this;
        if (config === void 0) { config = {}; }
        this.config = config;
        this.typePolicies = Object.create(null);
        this.rootTypenamesById = {
            __proto__: null,
            ROOT_QUERY: "Query",
            ROOT_MUTATION: "Mutation",
            ROOT_SUBSCRIPTION: "Subscription",
        };
        this.usingPossibleTypes = false;
        this.toReference = function (object, selectionSet, fragmentMap) {
            var id = _this.identify(object, selectionSet, fragmentMap);
            return id && makeReference(id);
        };
        this.config = __assign({ dataIdFromObject: defaultDataIdFromObject }, config);
        if (config.possibleTypes) {
            this.addPossibleTypes(config.possibleTypes);
        }
        if (config.typePolicies) {
            this.addTypePolicies(config.typePolicies);
        }
    }
    Policies.prototype.identify = function (object, selectionSet, fragmentMap) {
        var typename = selectionSet && fragmentMap
            ? getTypenameFromResult(object, selectionSet, fragmentMap)
            : object.__typename;
        var context = {
            typename: typename,
            selectionSet: selectionSet,
            fragmentMap: fragmentMap,
        };
        var id;
        var policy = this.getTypePolicy(typename, false);
        if (policy && policy.keyFn) {
            id = policy.keyFn.call(this, object, context);
        }
        else {
            id = this.config.dataIdFromObject
                ? this.config.dataIdFromObject.call(this, object, context)
                : null;
        }
        return id && String(id);
    };
    Policies.prototype.addTypePolicies = function (typePolicies) {
        var _this = this;
        Object.keys(typePolicies).forEach(function (typename) {
            var existing = _this.getTypePolicy(typename, true);
            var incoming = typePolicies[typename];
            var keyFields = incoming.keyFields, fields = incoming.fields;
            if (incoming.queryType)
                _this.setRootTypename("Query", typename);
            if (incoming.mutationType)
                _this.setRootTypename("Mutation", typename);
            if (incoming.subscriptionType)
                _this.setRootTypename("Subscription", typename);
            existing.keyFn =
                keyFields === false ? nullKeyFn :
                    Array.isArray(keyFields) ? keyFieldsFnFromSpecifier(keyFields) :
                        typeof keyFields === "function" ? keyFields : void 0;
            if (fields) {
                Object.keys(fields).forEach(function (fieldName) {
                    var existing = _this.getFieldPolicy(typename, fieldName, true);
                    var incoming = fields[fieldName];
                    if (typeof incoming === "function") {
                        existing.read = incoming;
                    }
                    else {
                        var keyArgs = incoming.keyArgs, read = incoming.read, merge = incoming.merge;
                        existing.keyFn = Array.isArray(keyArgs)
                            ? keyArgsFnFromSpecifier(keyArgs)
                            : typeof keyArgs === "function" ? keyArgs : void 0;
                        if (typeof read === "function")
                            existing.read = read;
                        if (typeof merge === "function")
                            existing.merge = merge;
                    }
                });
            }
        });
    };
    Policies.prototype.setRootTypename = function (which, typename) {
        var rootId = "ROOT_" + which.toUpperCase();
        var old = this.rootTypenamesById[rootId];
        if (typename !== old) {
            process.env.NODE_ENV === "production" ? invariant(old === which, 25) : invariant(old === which, "Cannot change root " + which + " __typename more than once");
            this.rootTypenamesById[rootId] = typename;
        }
    };
    Policies.prototype.addPossibleTypes = function (possibleTypes) {
        var _this = this;
        this.usingPossibleTypes = true;
        Object.keys(possibleTypes).forEach(function (supertype) {
            var subtypeSet = _this.getSubtypeSet(supertype, true);
            possibleTypes[supertype].forEach(subtypeSet.add, subtypeSet);
        });
    };
    Policies.prototype.getTypePolicy = function (typename, createIfMissing) {
        var typePolicies = this.typePolicies;
        return typePolicies[typename] || (createIfMissing && (typePolicies[typename] = Object.create(null)));
    };
    Policies.prototype.getSubtypeSet = function (supertype, createIfMissing) {
        var policy = this.getTypePolicy(supertype, createIfMissing);
        if (policy) {
            return policy.subtypes || (createIfMissing && (policy.subtypes = new Set()));
        }
    };
    Policies.prototype.getFieldPolicy = function (typename, fieldName, createIfMissing) {
        var typePolicy = this.getTypePolicy(typename, createIfMissing);
        if (typePolicy) {
            var fieldPolicies = typePolicy.fields || (createIfMissing && (typePolicy.fields = Object.create(null)));
            if (fieldPolicies) {
                return fieldPolicies[fieldName] || (createIfMissing && (fieldPolicies[fieldName] = Object.create(null)));
            }
        }
    };
    Policies.prototype.fragmentMatches = function (fragment, typename) {
        var _this = this;
        if (!fragment.typeCondition)
            return true;
        var supertype = fragment.typeCondition.name.value;
        if (typename === supertype)
            return true;
        if (this.usingPossibleTypes) {
            var workQueue_1 = [this.getSubtypeSet(supertype, false)];
            for (var i = 0; i < workQueue_1.length; ++i) {
                var subtypes = workQueue_1[i];
                if (subtypes) {
                    if (subtypes.has(typename))
                        return true;
                    subtypes.forEach(function (subtype) {
                        var subsubtypes = _this.getSubtypeSet(subtype, false);
                        if (subsubtypes && workQueue_1.indexOf(subsubtypes) < 0) {
                            workQueue_1.push(subsubtypes);
                        }
                    });
                }
            }
            return false;
        }
        return "heuristic";
    };
    Policies.prototype.getStoreFieldName = function (typename, field, variables) {
        if (typeof typename === "string") {
            var policy = this.getFieldPolicy(typename, field.name.value, false);
            if (policy && policy.keyFn) {
                return policy.keyFn.call(this, field, {
                    typename: typename,
                    variables: variables,
                });
            }
        }
        return storeKeyNameFromField(field, variables);
    };
    Policies.prototype.readFieldFromStoreObject = function (parentObject, field, typename, variables) {
        if (typename === void 0) { typename = parentObject.__typename; }
        var storeFieldName = this.getStoreFieldName(typename, field, variables);
        var existing = parentObject[storeFieldName];
        var policy = this.getFieldPolicy(typename, field.name.value, false);
        if (policy && policy.read) {
            return policy.read.call(this, existing, {
                args: argumentsObjectFromField(field, variables),
                parentObject: parentObject,
                field: field,
                variables: variables,
                toReference: this.toReference,
            });
        }
        return existing;
    };
    Policies.prototype.getFieldMergeFunction = function (typename, field, variables) {
        var _this = this;
        var policy = this.getFieldPolicy(typename, field.name.value, false);
        if (policy && policy.merge) {
            return function (existing, incoming, parentObject) { return policy.merge.call(_this, existing, incoming, {
                args: argumentsObjectFromField(field, variables),
                parentObject: parentObject,
                field: field,
                variables: variables,
                toReference: _this.toReference,
            }); };
        }
    };
    return Policies;
}());
function keyArgsFnFromSpecifier(specifier) {
    var topLevelArgNames = Object.create(null);
    specifier.forEach(function (name) {
        if (typeof name === "string") {
            topLevelArgNames[name] = true;
        }
    });
    return function (field, context) {
        var fieldName = field.name.value;
        if (field.arguments && field.arguments.length > 0) {
            var args_1 = Object.create(null);
            field.arguments.forEach(function (arg) {
                if (topLevelArgNames[arg.name.value] === true) {
                    valueToObjectRepresentation(args_1, arg.name, arg.value, context.variables);
                }
            });
            return fieldName + ":" + JSON.stringify(computeKeyObject(args_1, specifier));
        }
        return fieldName;
    };
}
function keyFieldsFnFromSpecifier(specifier) {
    var trie = new KeyTrie(canUseWeakMap);
    return function (object, context) {
        var aliasMap;
        if (context.selectionSet && context.fragmentMap) {
            var info = trie.lookupArray([
                context.selectionSet,
                context.fragmentMap,
            ]);
            aliasMap = info.aliasMap || (info.aliasMap = makeAliasMap(context.selectionSet, context.fragmentMap));
        }
        return context.typename + ":" + JSON.stringify(computeKeyObject(object, specifier, aliasMap));
    };
}
function makeAliasMap(selectionSet, fragmentMap) {
    var map = Object.create(null);
    var workQueue = new Set([selectionSet]);
    workQueue.forEach(function (selectionSet) {
        selectionSet.selections.forEach(function (selection) {
            if (isField(selection)) {
                if (selection.alias) {
                    var responseKey = selection.alias.value;
                    var storeKey = selection.name.value;
                    if (storeKey !== responseKey) {
                        var aliases = map.aliases || (map.aliases = Object.create(null));
                        aliases[storeKey] = responseKey;
                    }
                }
                if (selection.selectionSet) {
                    var subsets = map.subsets || (map.subsets = Object.create(null));
                    subsets[selection.name.value] =
                        makeAliasMap(selection.selectionSet, fragmentMap);
                }
            }
            else {
                var fragment = getFragmentFromSelection(selection, fragmentMap);
                workQueue.add(fragment.selectionSet);
            }
        });
    });
    return map;
}
function computeKeyObject(response, specifier, aliasMap) {
    var keyObj = Object.create(null);
    var prevKey;
    specifier.forEach(function (s) {
        if (Array.isArray(s)) {
            if (typeof prevKey === "string") {
                var subsets = aliasMap && aliasMap.subsets;
                var subset = subsets && subsets[prevKey];
                keyObj[prevKey] = computeKeyObject(response[prevKey], s, subset);
            }
        }
        else {
            var aliases = aliasMap && aliasMap.aliases;
            var responseName = aliases && aliases[s] || s;
            process.env.NODE_ENV === "production" ? invariant(hasOwn.call(response, responseName), 26) : invariant(hasOwn.call(response, responseName), "Missing field " + responseName + " while computing key fields");
            keyObj[prevKey = s] = response[responseName];
        }
    });
    return keyObj;
}

export { Policies, defaultDataIdFromObject };
//# sourceMappingURL=policies.js.map
