import { __assign } from 'tslib';
import { InvariantError } from 'ts-invariant';
import { createFragmentMap } from '../../utilities/graphql/fragments.js';
import { isReference, makeReference, isField, resultKeyNameFromField, isInlineFragment } from '../../utilities/graphql/storeUtils.js';
import { getMainDefinition, getDefaultValues, getQueryDefinition, getFragmentDefinitions } from '../../utilities/graphql/getFromAST.js';
import { shouldInclude } from '../../utilities/graphql/directives.js';
import { canUseWeakMap } from '../../utilities/common/canUse.js';
import { equal } from '@wry/equality';
import { mergeDeepArray } from '../../utilities/common/mergeDeep.js';
import { KeyTrie, wrap } from 'optimism';
import { maybeDeepFreeze } from '../../utilities/common/maybeDeepFreeze.js';
import { supportsResultCaching } from './entityCache.js';
import { getTypenameFromStoreObject } from './helpers.js';

var StoreReader = (function () {
    function StoreReader(config) {
        var _this = this;
        this.config = config;
        var cacheKeyRoot = config && config.cacheKeyRoot || new KeyTrie(canUseWeakMap);
        this.config = __assign({ addTypename: true, cacheKeyRoot: cacheKeyRoot }, config);
        var _a = this, executeSelectionSet = _a.executeSelectionSet, executeSubSelectedArray = _a.executeSubSelectedArray;
        this.executeSelectionSet = wrap(function (options) {
            return executeSelectionSet.call(_this, options);
        }, {
            makeCacheKey: function (_a) {
                var selectionSet = _a.selectionSet, objectOrReference = _a.objectOrReference, context = _a.context;
                if (supportsResultCaching(context.store)) {
                    return cacheKeyRoot.lookup(context.store, selectionSet, JSON.stringify(context.variables), isReference(objectOrReference) ? objectOrReference.__ref : objectOrReference);
                }
            }
        });
        this.executeSubSelectedArray = wrap(function (options) {
            return executeSubSelectedArray.call(_this, options);
        }, {
            makeCacheKey: function (_a) {
                var field = _a.field, array = _a.array, context = _a.context;
                if (supportsResultCaching(context.store)) {
                    return cacheKeyRoot.lookup(context.store, field, array, JSON.stringify(context.variables));
                }
            }
        });
    }
    StoreReader.prototype.readQueryFromStore = function (options) {
        return this.diffQueryAgainstStore(__assign(__assign({}, options), { returnPartialData: false })).result;
    };
    StoreReader.prototype.diffQueryAgainstStore = function (_a) {
        var store = _a.store, query = _a.query, variables = _a.variables, previousResult = _a.previousResult, _b = _a.returnPartialData, returnPartialData = _b === void 0 ? true : _b, _c = _a.rootId, rootId = _c === void 0 ? 'ROOT_QUERY' : _c, config = _a.config;
        var policies = this.config.policies;
        var execResult = this.executeSelectionSet({
            selectionSet: getMainDefinition(query).selectionSet,
            objectOrReference: makeReference(rootId),
            context: {
                store: store,
                query: query,
                policies: policies,
                variables: __assign(__assign({}, getDefaultValues(getQueryDefinition(query))), variables),
                fragmentMap: createFragmentMap(getFragmentDefinitions(query)),
            },
        });
        var hasMissingFields = execResult.missing && execResult.missing.length > 0;
        if (hasMissingFields && !returnPartialData) {
            execResult.missing.forEach(function (info) {
                if (info.tolerable)
                    return;
                throw process.env.NODE_ENV === "production" ? new InvariantError(35) : new InvariantError("Can't find field " + info.fieldName + " on object " + JSON.stringify(info.object, null, 2) + ".");
            });
        }
        if (previousResult) {
            if (equal(previousResult, execResult.result)) {
                execResult.result = previousResult;
            }
        }
        return {
            result: execResult.result,
            complete: !hasMissingFields,
        };
    };
    StoreReader.prototype.executeSelectionSet = function (_a) {
        var _this = this;
        var selectionSet = _a.selectionSet, objectOrReference = _a.objectOrReference, context = _a.context;
        var store = context.store, fragmentMap = context.fragmentMap, variables = context.variables, policies = context.policies;
        var finalResult = { result: null };
        var objectsToMerge = [];
        var object;
        var typename;
        if (isReference(objectOrReference)) {
            object = store.get(objectOrReference.__ref);
            typename =
                (object && object.__typename) ||
                    policies.rootTypenamesById[objectOrReference.__ref];
        }
        else {
            object = objectOrReference;
            typename = object && object.__typename;
        }
        if (this.config.addTypename) {
            var typenameFromStore = object && object.__typename;
            if (typeof typenameFromStore === "string" &&
                Object.values(policies.rootTypenamesById).indexOf(typenameFromStore) < 0) {
                objectsToMerge.push({ __typename: typenameFromStore });
            }
        }
        function handleMissing(result) {
            var _a;
            if (result.missing) {
                finalResult.missing = finalResult.missing || [];
                (_a = finalResult.missing).push.apply(_a, result.missing);
            }
            return result.result;
        }
        selectionSet.selections.forEach(function (selection) {
            var _a;
            if (!shouldInclude(selection, variables)) {
                return;
            }
            if (isField(selection)) {
                var fieldResult = handleMissing(_this.executeField(object, typename, selection, context));
                if (typeof fieldResult !== 'undefined') {
                    objectsToMerge.push((_a = {},
                        _a[resultKeyNameFromField(selection)] = fieldResult,
                        _a));
                }
            }
            else {
                var fragment = void 0;
                if (isInlineFragment(selection)) {
                    fragment = selection;
                }
                else {
                    fragment = fragmentMap[selection.name.value];
                    if (!fragment) {
                        throw process.env.NODE_ENV === "production" ? new InvariantError(36) : new InvariantError("No fragment named " + selection.name.value);
                    }
                }
                var match = policies.fragmentMatches(fragment, typename);
                if (match) {
                    var fragmentExecResult = _this.executeSelectionSet({
                        selectionSet: fragment.selectionSet,
                        objectOrReference: objectOrReference,
                        context: context,
                    });
                    if (match === 'heuristic' && fragmentExecResult.missing) {
                        fragmentExecResult = __assign(__assign({}, fragmentExecResult), { missing: fragmentExecResult.missing.map(function (info) {
                                return __assign(__assign({}, info), { tolerable: true });
                            }) });
                    }
                    objectsToMerge.push(handleMissing(fragmentExecResult));
                }
            }
        });
        finalResult.result = mergeDeepArray(objectsToMerge);
        if (process.env.NODE_ENV !== 'production') {
            Object.freeze(finalResult.result);
        }
        return finalResult;
    };
    StoreReader.prototype.executeField = function (object, typename, field, context) {
        var variables = context.variables, store = context.store, policies = context.policies;
        var fieldValue = object &&
            policies.readFieldFromStoreObject(object, field, typename, variables);
        var readStoreResult = typeof fieldValue === "undefined" ? {
            result: fieldValue,
            missing: [{
                    object: object,
                    fieldName: field.name.value,
                    tolerable: false,
                }],
        } : {
            result: fieldValue,
        };
        if (Array.isArray(readStoreResult.result)) {
            return this.combineExecResults(readStoreResult, this.executeSubSelectedArray({
                field: field,
                array: readStoreResult.result,
                context: context,
            }));
        }
        if (!field.selectionSet) {
            if (process.env.NODE_ENV !== 'production') {
                assertSelectionSetForIdValue(store, field, readStoreResult.result);
                maybeDeepFreeze(readStoreResult);
            }
            return readStoreResult;
        }
        if (readStoreResult.result == null) {
            return readStoreResult;
        }
        return this.combineExecResults(readStoreResult, this.executeSelectionSet({
            selectionSet: field.selectionSet,
            objectOrReference: readStoreResult.result,
            context: context,
        }));
    };
    StoreReader.prototype.combineExecResults = function () {
        var execResults = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            execResults[_i] = arguments[_i];
        }
        var missing;
        execResults.forEach(function (execResult) {
            if (execResult.missing) {
                missing = missing || [];
                missing.push.apply(missing, execResult.missing);
            }
        });
        return {
            result: execResults.pop().result,
            missing: missing,
        };
    };
    StoreReader.prototype.executeSubSelectedArray = function (_a) {
        var _this = this;
        var field = _a.field, array = _a.array, context = _a.context;
        var missing;
        function handleMissing(childResult) {
            if (childResult.missing) {
                missing = missing || [];
                missing.push.apply(missing, childResult.missing);
            }
            return childResult.result;
        }
        array = array.map(function (item) {
            if (item === null) {
                return null;
            }
            if (Array.isArray(item)) {
                return handleMissing(_this.executeSubSelectedArray({
                    field: field,
                    array: item,
                    context: context,
                }));
            }
            if (field.selectionSet) {
                return handleMissing(_this.executeSelectionSet({
                    selectionSet: field.selectionSet,
                    objectOrReference: item,
                    context: context,
                }));
            }
            if (process.env.NODE_ENV !== 'production') {
                assertSelectionSetForIdValue(context.store, field, item);
            }
            return item;
        });
        if (process.env.NODE_ENV !== 'production') {
            Object.freeze(array);
        }
        return { result: array, missing: missing };
    };
    return StoreReader;
}());
function assertSelectionSetForIdValue(store, field, fieldValue) {
    if (!field.selectionSet) {
        var workSet_1 = new Set([fieldValue]);
        workSet_1.forEach(function (value) {
            if (value && typeof value === "object") {
                if (isReference(value)) {
                    throw process.env.NODE_ENV === "production" ? new InvariantError(37) : new InvariantError("Missing selection set for object of type " + getTypenameFromStoreObject(store, value) + " returned for query field " + field.name.value);
                }
                Object.values(value).forEach(workSet_1.add, workSet_1);
            }
        });
    }
}

export { StoreReader };
//# sourceMappingURL=readFromStore.js.map
