import { __assign } from 'tslib';
import { invariant, InvariantError } from 'ts-invariant';
import { createFragmentMap, getFragmentFromSelection } from '../../utilities/graphql/fragments.js';
import { getTypenameFromResult, isField, resultKeyNameFromField, makeReference, isReference } from '../../utilities/graphql/storeUtils.js';
import { getOperationDefinition, getDefaultValues, getFragmentDefinitions } from '../../utilities/graphql/getFromAST.js';
import { shouldInclude } from '../../utilities/graphql/directives.js';
import { cloneDeep } from '../../utilities/common/cloneDeep.js';
import { DeepMerger } from '../../utilities/common/mergeDeep.js';
import { defaultNormalizedCacheFactory } from './entityCache.js';
import { getTypenameFromStoreObject } from './helpers.js';

var StoreWriter = (function () {
    function StoreWriter(config) {
        this.policies = config.policies;
    }
    StoreWriter.prototype.writeQueryToStore = function (_a) {
        var query = _a.query, result = _a.result, _b = _a.dataId, dataId = _b === void 0 ? 'ROOT_QUERY' : _b, _c = _a.store, store = _c === void 0 ? defaultNormalizedCacheFactory() : _c, variables = _a.variables;
        var operationDefinition = getOperationDefinition(query);
        store.retain(dataId);
        var simpleFieldsMerger = new DeepMerger;
        var storeObjectMerger = new DeepMerger(storeObjectReconciler);
        return this.writeSelectionSetToStore({
            result: result || Object.create(null),
            dataId: dataId,
            selectionSet: operationDefinition.selectionSet,
            context: {
                store: store,
                written: Object.create(null),
                mergeFields: function (existing, incoming) {
                    return simpleFieldsMerger.merge(existing, incoming);
                },
                mergeStoreObjects: function (existing, incoming, overrides) {
                    return storeObjectMerger.merge(existing, incoming, store, overrides);
                },
                variables: __assign(__assign({}, getDefaultValues(operationDefinition)), variables),
                fragmentMap: createFragmentMap(getFragmentDefinitions(query)),
            },
        });
    };
    StoreWriter.prototype.writeSelectionSetToStore = function (_a) {
        var dataId = _a.dataId, result = _a.result, selectionSet = _a.selectionSet, context = _a.context;
        var store = context.store, written = context.written;
        var sets = written[dataId] || (written[dataId] = []);
        if (sets.indexOf(selectionSet) >= 0)
            return store;
        sets.push(selectionSet);
        var processed = this.processSelectionSet({
            result: result,
            selectionSet: selectionSet,
            context: context,
            typename: this.policies.rootTypenamesById[dataId],
        });
        var existing = store.get(dataId) || Object.create(null);
        if (processed.mergeOverrides) {
            walkWithMergeOverrides(existing, processed.result, processed.mergeOverrides);
        }
        store.set(dataId, context.mergeStoreObjects(existing, processed.result, processed.mergeOverrides));
        return store;
    };
    StoreWriter.prototype.processSelectionSet = function (_a) {
        var _this = this;
        var result = _a.result, selectionSet = _a.selectionSet, context = _a.context, mergeOverrides = _a.mergeOverrides, _b = _a.typename, typename = _b === void 0 ? getTypenameFromResult(result, selectionSet, context.fragmentMap) : _b;
        var mergedFields = Object.create(null);
        if (typeof typename === "string") {
            mergedFields.__typename = typename;
        }
        selectionSet.selections.forEach(function (selection) {
            var _a;
            if (!shouldInclude(selection, context.variables)) {
                return;
            }
            if (isField(selection)) {
                var resultFieldKey = resultKeyNameFromField(selection);
                var value = result[resultFieldKey];
                if (typeof value !== 'undefined') {
                    var storeFieldName = _this.policies.getStoreFieldName(typename, selection, context.variables);
                    var processed = _this.processFieldValue(value, selection, context);
                    var merge = _this.policies.getFieldMergeFunction(typename, selection, context.variables);
                    if (merge || processed.mergeOverrides) {
                        mergeOverrides = mergeOverrides || Object.create(null);
                        mergeOverrides[storeFieldName] = context.mergeFields(mergeOverrides[storeFieldName], { merge: merge, child: processed.mergeOverrides });
                    }
                    mergedFields = context.mergeFields(mergedFields, (_a = {},
                        _a[storeFieldName] = processed.result,
                        _a));
                }
                else if (_this.policies.usingPossibleTypes &&
                    !(selection.directives &&
                        selection.directives.some(function (_a) {
                            var name = _a.name;
                            return name && (name.value === 'defer' || name.value === 'client');
                        }))) {
                    process.env.NODE_ENV === "production" || invariant.warn("Missing field " + resultFieldKey + " in " + JSON.stringify(result, null, 2).substring(0, 100));
                }
            }
            else {
                var fragment = getFragmentFromSelection(selection, context.fragmentMap);
                if (_this.policies.fragmentMatches(fragment, typename)) {
                    mergedFields = context.mergeFields(mergedFields, _this.processSelectionSet({
                        result: result,
                        selectionSet: fragment.selectionSet,
                        context: context,
                        mergeOverrides: mergeOverrides,
                        typename: typename,
                    }).result);
                }
            }
        });
        return {
            result: mergedFields,
            mergeOverrides: mergeOverrides,
        };
    };
    StoreWriter.prototype.processFieldValue = function (value, field, context) {
        var _this = this;
        if (!field.selectionSet || value === null) {
            return {
                result: process.env.NODE_ENV === 'production' ? value : cloneDeep(value),
            };
        }
        if (Array.isArray(value)) {
            var overrides_1;
            var result = value.map(function (item, i) {
                var _a = _this.processFieldValue(item, field, context), result = _a.result, mergeOverrides = _a.mergeOverrides;
                if (mergeOverrides) {
                    overrides_1 = overrides_1 || [];
                    overrides_1[i] = mergeOverrides;
                }
                return result;
            });
            return { result: result, mergeOverrides: overrides_1 };
        }
        if (value) {
            var dataId = this.policies.identify(value, field.selectionSet, context.fragmentMap);
            if (typeof dataId === 'string') {
                this.writeSelectionSetToStore({
                    dataId: dataId,
                    result: value,
                    selectionSet: field.selectionSet,
                    context: context,
                });
                return { result: makeReference(dataId) };
            }
        }
        return this.processSelectionSet({
            result: value,
            selectionSet: field.selectionSet,
            context: context,
        });
    };
    return StoreWriter;
}());
function walkWithMergeOverrides(existingObject, incomingObject, overrides) {
    Object.keys(overrides).forEach(function (name) {
        var _a = overrides[name], merge = _a.merge, child = _a.child;
        var existingValue = existingObject && existingObject[name];
        var incomingValue = incomingObject && incomingObject[name];
        if (child) {
            walkWithMergeOverrides(existingValue, incomingValue, child);
        }
        if (merge) {
            incomingObject[name] = merge(existingValue, incomingValue, existingObject);
        }
    });
}
var storeObjectReconciler = function (existingObject, incomingObject, property, store, mergeOverrides) {
    var existing = existingObject[property];
    var incoming = incomingObject[property];
    var mergeChildObj = mergeOverrides && mergeOverrides[property];
    if (mergeChildObj && typeof mergeChildObj.merge === "function") {
        return incoming;
    }
    if (existing !== incoming &&
        this.isObject(existing) &&
        this.isObject(incoming)) {
        var eType = getTypenameFromStoreObject(store, existing);
        var iType = getTypenameFromStoreObject(store, incoming);
        if (typeof eType === 'string' &&
            typeof iType === 'string' &&
            eType !== iType) {
            return incoming;
        }
        var childMergeOverrides = mergeChildObj && mergeChildObj.child;
        if (isReference(incoming)) {
            if (isReference(existing)) {
                return incoming.__ref === existing.__ref ? existing : incoming;
            }
            store.set(incoming.__ref, this.merge(existing, store.get(incoming.__ref), store, childMergeOverrides));
            return incoming;
        }
        else if (isReference(existing)) {
            throw process.env.NODE_ENV === "production" ? new InvariantError(38) : new InvariantError("Store error: the application attempted to write an object with no provided id but the store already contains an id of " + existing.__ref + " for this object.");
        }
        if (Array.isArray(incoming)) {
            if (!Array.isArray(existing))
                return incoming;
            if (existing.length > incoming.length) {
                return this.merge(existing.slice(0, incoming.length), incoming, store, childMergeOverrides);
            }
        }
        return this.merge(existing, incoming, store, childMergeOverrides);
    }
    return incoming;
};

export { StoreWriter };
//# sourceMappingURL=writeToStore.js.map
