import { __assign, __rest } from "tslib";
import { useContext, useEffect, useMemo, useRef, useState, } from 'react';
import { equal } from '@wry/equality';
import { getApolloContext } from "../context/index.js";
import { ApolloError } from "../../errors/index.js";
import { NetworkStatus, } from "../../core/index.js";
import { DocumentType, verifyDocumentType } from "../parser/index.js";
import { useApolloClient } from "./useApolloClient.js";
import { canUseWeakMap, isNonEmptyArray } from "../../utilities/index.js";
import { useNormalizedOptions } from "./options.js";
var hasOwnProperty = Object.prototype.hasOwnProperty;
export function useQuery(query, optionsOrFunction) {
    var options = useNormalizedOptions(optionsOrFunction);
    return useInternalState(useApolloClient(options.client), query).useQuery(options);
}
function useInternalState(client, query) {
    var stateRef = useRef();
    if (!stateRef.current ||
        client !== stateRef.current.client ||
        query !== stateRef.current.query) {
        stateRef.current = new InternalState(client, query);
    }
    var state = stateRef.current;
    var _a = useState(0), _tick = _a[0], setTick = _a[1];
    state.forceUpdate = function () {
        setTick(function (tick) { return tick + 1; });
    };
    return state;
}
var InternalState = (function () {
    function InternalState(client, query) {
        this.client = client;
        this.query = query;
        this.toQueryResultCache = new (canUseWeakMap ? WeakMap : Map)();
        verifyDocumentType(query, DocumentType.Query);
    }
    InternalState.prototype.forceUpdate = function () {
    };
    InternalState.prototype.useQuery = function (options) {
        this.useOptions(options);
        var obsQuery = this.useObservableQuery();
        this.useSubscriptionEffect(obsQuery);
        var result = this.getCurrentResult();
        this.unsafeHandlePartialRefetch(result);
        return this.toQueryResult(result);
    };
    InternalState.prototype.useOptions = function (options) {
        this.renderPromises = useContext(getApolloContext()).renderPromises;
        var watchQueryOptions = this.createWatchQueryOptions(this.queryHookOptions = options);
        if (!equal(watchQueryOptions, this.watchQueryOptions)) {
            this.watchQueryOptions = watchQueryOptions;
        }
        this.ssrDisabled = !!(options && (options.ssr === false ||
            options.skip));
        this.onCompleted = options
            && options.onCompleted
            || InternalState.prototype.onCompleted;
        this.onError = options
            && options.onError
            || InternalState.prototype.onError;
    };
    InternalState.prototype.createWatchQueryOptions = function (_a) {
        var _b;
        if (_a === void 0) { _a = {}; }
        var skip = _a.skip, ssr = _a.ssr, onCompleted = _a.onCompleted, onError = _a.onError, displayName = _a.displayName, otherOptions = __rest(_a, ["skip", "ssr", "onCompleted", "onError", "displayName"]);
        var watchQueryOptions = Object.assign(otherOptions, { query: this.query });
        if (skip) {
            watchQueryOptions.fetchPolicy = 'standby';
        }
        else if (((_b = watchQueryOptions.context) === null || _b === void 0 ? void 0 : _b.renderPromises) &&
            (watchQueryOptions.fetchPolicy === 'network-only' ||
                watchQueryOptions.fetchPolicy === 'cache-and-network')) {
            watchQueryOptions.fetchPolicy = 'cache-first';
        }
        else if (!watchQueryOptions.fetchPolicy) {
            var defaultOptions = this.client.defaultOptions.watchQuery;
            watchQueryOptions.fetchPolicy =
                defaultOptions && defaultOptions.fetchPolicy || 'cache-first';
        }
        if (!watchQueryOptions.variables) {
            watchQueryOptions.variables = {};
        }
        return watchQueryOptions;
    };
    InternalState.prototype.onCompleted = function (data) { };
    InternalState.prototype.onError = function (error) { };
    InternalState.prototype.useObservableQuery = function () {
        var _this = this;
        var obsQuery = this.observable =
            this.renderPromises
                && this.renderPromises.getSSRObservable(this.watchQueryOptions)
                || this.observable
                || this.client.watchQuery(this.watchQueryOptions);
        this.obsQueryFields = useMemo(function () { return ({
            refetch: obsQuery.refetch.bind(obsQuery),
            fetchMore: obsQuery.fetchMore.bind(obsQuery),
            updateQuery: obsQuery.updateQuery.bind(obsQuery),
            startPolling: obsQuery.startPolling.bind(obsQuery),
            stopPolling: obsQuery.stopPolling.bind(obsQuery),
            subscribeToMore: obsQuery.subscribeToMore.bind(obsQuery),
        }); }, [obsQuery]);
        if (this.renderPromises) {
            this.renderPromises.registerSSRObservable(obsQuery);
            if (!this.ssrDisabled && obsQuery.getCurrentResult().loading) {
                this.renderPromises.addQueryPromise({
                    getOptions: function () { return obsQuery.options; },
                    fetchData: function () { return new Promise(function (resolve) {
                        var sub = obsQuery.subscribe({
                            next: function (result) {
                                if (!result.loading) {
                                    resolve();
                                    sub.unsubscribe();
                                }
                            },
                            error: function () {
                                resolve();
                                sub.unsubscribe();
                            },
                            complete: function () {
                                resolve();
                            },
                        });
                    }); },
                }, function () { return null; });
                obsQuery.setOptions(this.watchQueryOptions).catch(function () { });
            }
        }
        var prevOptionsRef = useRef({
            watchQueryOptions: this.watchQueryOptions,
        });
        useEffect(function () {
            if (_this.renderPromises) {
            }
            else if (_this.watchQueryOptions !== prevOptionsRef.current.watchQueryOptions) {
                obsQuery.setOptions(_this.watchQueryOptions).catch(function () { });
                prevOptionsRef.current.watchQueryOptions = _this.watchQueryOptions;
                _this.setResult(obsQuery.getCurrentResult());
            }
        }, [obsQuery, this.watchQueryOptions]);
        return obsQuery;
    };
    InternalState.prototype.useSubscriptionEffect = function (obsQuery) {
        var _this = this;
        useEffect(function () {
            if (_this.renderPromises) {
                return;
            }
            var onNext = function () {
                var previousResult = _this.result;
                var result = obsQuery.getCurrentResult();
                if (previousResult &&
                    previousResult.loading === result.loading &&
                    previousResult.networkStatus === result.networkStatus &&
                    equal(previousResult.data, result.data)) {
                    return;
                }
                _this.setResult(result);
            };
            var onError = function (error) {
                var last = obsQuery["last"];
                subscription.unsubscribe();
                try {
                    obsQuery.resetLastResults();
                    subscription = obsQuery.subscribe(onNext, onError);
                }
                finally {
                    obsQuery["last"] = last;
                }
                if (!hasOwnProperty.call(error, 'graphQLErrors')) {
                    throw error;
                }
                var previousResult = _this.result;
                if (!previousResult ||
                    (previousResult && previousResult.loading) ||
                    !equal(error, previousResult.error)) {
                    _this.setResult({
                        data: (previousResult && previousResult.data),
                        error: error,
                        loading: false,
                        networkStatus: NetworkStatus.error,
                    });
                }
            };
            var subscription = obsQuery.subscribe(onNext, onError);
            return function () { return subscription.unsubscribe(); };
        }, [
            obsQuery,
            this.renderPromises,
            this.client.disableNetworkFetches,
        ]);
    };
    InternalState.prototype.setResult = function (nextResult) {
        var previousResult = this.result;
        if (previousResult && previousResult.data) {
            this.previousData = previousResult.data;
        }
        this.result = nextResult;
        this.forceUpdate();
        this.handleErrorOrCompleted(nextResult);
    };
    InternalState.prototype.handleErrorOrCompleted = function (result) {
        if (!result.loading) {
            if (result.error) {
                this.onError(result.error);
            }
            else if (result.data) {
                this.onCompleted(result.data);
            }
        }
    };
    InternalState.prototype.getCurrentResult = function () {
        var result = this.result;
        if (!result) {
            result = this.result = this.observable.getCurrentResult();
            this.handleErrorOrCompleted(result);
        }
        if ((this.renderPromises || this.client.disableNetworkFetches) &&
            this.queryHookOptions.ssr === false) {
            result = {
                loading: true,
                data: void 0,
                error: void 0,
                networkStatus: NetworkStatus.loading,
            };
        }
        else if (this.queryHookOptions.skip ||
            this.queryHookOptions.fetchPolicy === 'standby') {
            result = {
                loading: false,
                data: void 0,
                error: void 0,
                networkStatus: NetworkStatus.ready,
            };
        }
        return result;
    };
    InternalState.prototype.toQueryResult = function (result) {
        var queryResult = this.toQueryResultCache.get(result);
        if (queryResult)
            return queryResult;
        var data = result.data, partial = result.partial, resultWithoutPartial = __rest(result, ["data", "partial"]);
        this.toQueryResultCache.set(result, queryResult = __assign(__assign(__assign({ data: data }, resultWithoutPartial), this.obsQueryFields), { client: this.client, observable: this.observable, variables: this.observable.variables, called: true, previousData: this.previousData }));
        if (!queryResult.error && isNonEmptyArray(result.errors)) {
            queryResult.error = new ApolloError({ graphQLErrors: result.errors });
        }
        return queryResult;
    };
    InternalState.prototype.unsafeHandlePartialRefetch = function (result) {
        if (result.partial &&
            this.queryHookOptions.partialRefetch &&
            !result.loading &&
            (!result.data || Object.keys(result.data).length === 0) &&
            this.observable.options.fetchPolicy !== 'cache-only') {
            Object.assign(result, {
                loading: true,
                networkStatus: NetworkStatus.refetch,
            });
            this.observable.refetch();
        }
    };
    return InternalState;
}());
//# sourceMappingURL=useQuery.js.map