import MIMEType from 'whatwg-mimetype';
import { Observable } from 'zen-observable-ts';
import { createClient } from 'graphql-ws';
import { SubscriptionClient as SubscriptionClient$1 } from 'subscriptions-transport-ws';

// URL for any embedded Explorer iframe
const EMBEDDABLE_EXPLORER_URL = (__testLocal__ = false) => __testLocal__ ? 'https://embed.apollo.local:3000' : 'https://explorer.embed.apollographql.com'; // Message types for Explorer state

const EXPLORER_LISTENING_FOR_SCHEMA = 'ExplorerListeningForSchema';
const SCHEMA_RESPONSE = 'SchemaResponse'; // Message types for queries and mutations

const EXPLORER_QUERY_MUTATION_REQUEST = 'ExplorerRequest';
const EXPLORER_QUERY_MUTATION_RESPONSE = 'ExplorerResponse';

const EXPLORER_SUBSCRIPTION_REQUEST = 'ExplorerSubscriptionRequest';
const EXPLORER_SUBSCRIPTION_RESPONSE = 'ExplorerSubscriptionResponse';
const EXPLORER_SUBSCRIPTION_TERMINATION = 'ExplorerSubscriptionTermination';
const EXPLORER_SET_SOCKET_ERROR = 'ExplorerSetSocketError';
const EXPLORER_SET_SOCKET_STATUS = 'ExplorerSetSocketStatus';
const IFRAME_DOM_ID = uniqueId => `apollo-embed-${uniqueId}`; // Message types for authentication

const EXPLORER_LISTENING_FOR_HANDSHAKE = 'ExplorerListeningForHandshake';
const HANDSHAKE_RESPONSE = 'HandshakeResponse';
const SET_PARTIAL_AUTHENTICATION_TOKEN_FOR_PARENT = 'SetPartialAuthenticationTokenForParent';
const TRIGGER_LOGOUT_IN_PARENT = 'TriggerLogoutInParent';
const EXPLORER_LISTENING_FOR_PARTIAL_TOKEN = 'ExplorerListeningForPartialToken';
const PARTIAL_AUTHENTICATION_TOKEN_RESPONSE = 'PartialAuthenticationTokenResponse';
const PARENT_LOGOUT_SUCCESS = 'ParentLogoutSuccess';

const defaultHandleRequest = ({
  includeCookies
}) => {
  const handleRequestWithCookiePref = (endpointUrl, options) => fetch(endpointUrl, Object.assign(Object.assign({}, options), includeCookies ? {
    credentials: 'include'
  } : {
    credentials: 'omit'
  }));

  return handleRequestWithCookiePref;
};

/*! *****************************************************************************
Copyright (c) Microsoft Corporation.

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.
***************************************************************************** */
function __awaiter(thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function (resolve) {
      resolve(value);
    });
  }

  return new (P || (P = Promise))(function (resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }

    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }

    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }

    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
}

function readMultipartWebStream(response, mimeType) {
  return new Observable(observer => {
    if (response.body === null) {
      throw new Error('Missing body');
    } else if (typeof response.body.tee !== 'function') {
      // not sure if we actually need this check in explorer?
      throw new Error('Streaming bodies not supported by provided fetch implementation');
    }

    const decoder = new TextDecoder('utf-8');
    let buffer = '';
    const messageBoundary = `--${mimeType.parameters.get('boundary') || '-'}`;
    const reader = response.body.getReader();

    function readMultipartStream() {
      reader.read().then(iteration => {
        var _a, _b, _c;

        if (iteration.done) {
          (_a = observer.complete) === null || _a === void 0 ? void 0 : _a.call(observer);
          return;
        }

        const chunk = decoder.decode(iteration.value);
        buffer += chunk;
        let boundaryIndex = buffer.indexOf(messageBoundary);

        while (boundaryIndex > -1) {
          const message = buffer.slice(0, boundaryIndex);
          buffer = buffer.slice(boundaryIndex + messageBoundary.length);

          if (message.trim()) {
            const messageStartIndex = message.indexOf('\r\n\r\n');
            const chunkHeaders = Object.fromEntries(message.slice(0, messageStartIndex).split('\n').map(line => {
              const i = line.indexOf(':');

              if (i > -1) {
                const name = line.slice(0, i).trim();
                const value = line.slice(i + 1).trim();
                return [name, value];
              } else {
                return null;
              }
            }).filter(h => !!h));

            if (((_b = chunkHeaders['content-type']) === null || _b === void 0 ? void 0 : _b.toLowerCase().indexOf('application/json')) === -1) {
              throw new Error('Unsupported patch content type');
            }

            const bodyText = message.slice(messageStartIndex);

            try {
              (_c = observer.next) === null || _c === void 0 ? void 0 : _c.call(observer, {
                data: JSON.parse(bodyText),
                headers: chunkHeaders,
                size: chunk.length
              });
            } catch (err) {
              // const parseError = err as ServerParseError;
              // parseError.name = 'ServerParseError';
              // parseError.response = response;
              // parseError.statusCode = response.status;
              // parseError.bodyText = bodyText;
              throw err;
            }
          }

          boundaryIndex = buffer.indexOf(messageBoundary);
        }

        readMultipartStream();
      }).catch(err => {
        var _a, _b;

        if (err.name === 'AbortError') return; // if it is a network error, BUT there is graphql result info fire
        // the next observer before calling error this gives apollo-client
        // (and react-apollo) the `graphqlErrors` and `networkErrors` to
        // pass to UI this should only happen if we *also* have data as
        // part of the response key per the spec

        if (err.result && err.result.errors && err.result.data) {
          // if we don't call next, the UI can only show networkError
          // because AC didn't get any graphqlErrors this is graphql
          // execution result info (i.e errors and possibly data) this is
          // because there is no formal spec how errors should translate to
          // http status codes. So an auth error (401) could have both data
          // from a public field, errors from a private field, and a status
          // of 401
          // {
          //  user { // this will have errors
          //    firstName
          //  }
          //  products { // this is public so will have data
          //    cost
          //  }
          // }
          //
          // the result of above *could* look like this:
          // {
          //   data: { products: [{ cost: "$10" }] },
          //   errors: [{
          //      message: 'your session has timed out',
          //      path: []
          //   }]
          // }
          // status code of above would be a 401
          // in the UI you want to show data where you can, errors as data where you can
          // and use correct http status codes
          (_a = observer.next) === null || _a === void 0 ? void 0 : _a.call(observer, {
            data: err.result,
            size: Infinity
          });
        }

        (_b = observer.error) === null || _b === void 0 ? void 0 : _b.call(observer, err);
      });
    }

    readMultipartStream();
  });
}

// to each request's headers if not present

function getHeadersWithContentType(headers) {
  const headersWithContentType = headers !== null && headers !== void 0 ? headers : {};

  if (Object.keys(headersWithContentType).every(key => key.toLowerCase() !== 'content-type')) {
    headersWithContentType['content-type'] = 'application/json';
  }

  return headersWithContentType;
}

function sendPostMessageToEmbed({
  message,
  embeddedIFrameElement,
  embedUrl
}) {
  var _a;

  (_a = embeddedIFrameElement === null || embeddedIFrameElement === void 0 ? void 0 : embeddedIFrameElement.contentWindow) === null || _a === void 0 ? void 0 : _a.postMessage(message, embedUrl);
}
function executeOperation({
  endpointUrl,
  handleRequest,
  operation,
  operationName,
  variables,
  headers,
  embeddedIFrameElement,
  operationId,
  embedUrl
}) {
  return handleRequest(endpointUrl, {
    method: 'POST',
    headers: getHeadersWithContentType(headers),
    body: JSON.stringify({
      query: operation,
      variables,
      operationName
    })
  }).then(response => __awaiter(this, void 0, void 0, function* () {
    var _a;

    const responseHeaders = {};
    response.headers.forEach((value, key) => {
      responseHeaders[key] = value;
    });
    const contentType = (_a = response.headers) === null || _a === void 0 ? void 0 : _a.get('content-type');
    const mimeType = contentType && new MIMEType(contentType);

    if (mimeType && mimeType.type === 'multipart' && mimeType.subtype === 'mixed') {
      const observable = readMultipartWebStream(response, mimeType);
      let isFirst = true;
      observable.subscribe({
        next(data) {
          sendPostMessageToEmbed({
            message: {
              // Include the same operation ID in the response message's name
              // so the Explorer knows which operation it's associated with
              name: EXPLORER_QUERY_MUTATION_RESPONSE,
              operationId,
              response: {
                incremental: data.data.incremental,
                data: data.data.data,
                errors: data.data.errors,
                extensions: data.data.extensions,
                path: data.data.path,
                status: response.status,
                headers: isFirst ? [responseHeaders, ...(data.headers ? [data.headers] : [])] : data.headers,
                hasNext: true,
                size: data.size
              }
            },
            embeddedIFrameElement,
            embedUrl
          });
          isFirst = false;
        },

        error(err) {
          sendPostMessageToEmbed({
            message: {
              // Include the same operation ID in the response message's name
              // so the Explorer knows which operation it's associated with
              name: EXPLORER_QUERY_MUTATION_RESPONSE,
              operationId,
              response: {
                data: null,
                error: Object.assign({
                  message: err.message
                }, err.stack ? {
                  stack: err.stack
                } : {}),
                size: 0,
                hasNext: false
              }
            },
            embeddedIFrameElement,
            embedUrl
          });
        },

        complete() {
          sendPostMessageToEmbed({
            message: {
              // Include the same operation ID in the response message's name
              // so the Explorer knows which operation it's associated with
              name: EXPLORER_QUERY_MUTATION_RESPONSE,
              operationId,
              response: {
                data: null,
                size: 0,
                status: response.status,
                headers: isFirst ? responseHeaders : undefined,
                hasNext: false
              }
            },
            embeddedIFrameElement,
            embedUrl
          });
        }

      });
    } else {
      const json = yield response.json();
      sendPostMessageToEmbed({
        message: {
          // Include the same operation ID in the response message's name
          // so the Explorer knows which operation it's associated with
          name: EXPLORER_QUERY_MUTATION_RESPONSE,
          operationId,
          response: Object.assign(Object.assign({}, json), {
            status: response.status,
            headers: responseHeaders,
            hasNext: false
          })
        },
        embeddedIFrameElement,
        embedUrl
      });
    }
  })).catch(response => {
    sendPostMessageToEmbed({
      message: {
        // Include the same operation ID in the response message's name
        // so the Explorer knows which operation it's associated with
        name: EXPLORER_QUERY_MUTATION_RESPONSE,
        operationId,
        response: {
          error: Object.assign({
            message: response.message
          }, response.stack ? {
            stack: response.stack
          } : {}),
          hasNext: false
        }
      },
      embeddedIFrameElement,
      embedUrl
    });
  });
}
const handleAuthenticationPostMessage = ({
  event,
  embeddedIFrameElement,
  embedUrl
}) => {
  const {
    data
  } = event; // When the embed authenticates, save the partial token in local storage

  if (data.name === SET_PARTIAL_AUTHENTICATION_TOKEN_FOR_PARENT) {
    const partialEmbedApiKeysString = window.localStorage.getItem('apolloStudioEmbeddedExplorerEncodedApiKey');
    const partialEmbedApiKeys = partialEmbedApiKeysString ? JSON.parse(partialEmbedApiKeysString) : {};
    partialEmbedApiKeys[data.localStorageKey] = data.partialToken;
    window.localStorage.setItem('apolloStudioEmbeddedExplorerEncodedApiKey', JSON.stringify(partialEmbedApiKeys));
  } // When the embed logs out, remove the partial token in local storage


  if (data.name === TRIGGER_LOGOUT_IN_PARENT) {
    const partialEmbedApiKeysString = window.localStorage.getItem('apolloStudioEmbeddedExplorerEncodedApiKey');
    const partialEmbedApiKeys = partialEmbedApiKeysString ? JSON.parse(partialEmbedApiKeysString) : {};
    delete partialEmbedApiKeys[data.localStorageKey];
    window.localStorage.setItem('apolloStudioEmbeddedExplorerEncodedApiKey', JSON.stringify(partialEmbedApiKeys));
    sendPostMessageToEmbed({
      message: {
        name: PARENT_LOGOUT_SUCCESS
      },
      embeddedIFrameElement,
      embedUrl
    });
  }

  if (data.name === EXPLORER_LISTENING_FOR_PARTIAL_TOKEN && data.localStorageKey) {
    const partialEmbedApiKeysString = window.localStorage.getItem('apolloStudioEmbeddedExplorerEncodedApiKey');
    const partialEmbedApiKeys = partialEmbedApiKeysString ? JSON.parse(partialEmbedApiKeysString) : {};

    if (partialEmbedApiKeys && partialEmbedApiKeys[data.localStorageKey]) {
      sendPostMessageToEmbed({
        message: {
          name: PARTIAL_AUTHENTICATION_TOKEN_RESPONSE,
          partialToken: partialEmbedApiKeys[data.localStorageKey]
        },
        embeddedIFrameElement,
        embedUrl
      });
    }
  }
};

function assertUnreachable(x) {
  throw new Error(`Didn't expect to get here ${x}`);
}

class SubscriptionClient {
  constructor(url, headers, protocol) {
    this.unsubscribeFunctions = [];
    this.protocol = protocol;
    this.url = url;
    this.headers = headers;
  }

  get graphWsClient() {
    var _a, _b;

    const client = (_a = this._graphWsClient) !== null && _a !== void 0 ? _a : createClient({
      url: this.url,
      lazy: true,
      connectionParams: (_b = this.headers) !== null && _b !== void 0 ? _b : {},
      keepAlive: 10000
    });
    this._graphWsClient = client;
    return client;
  }

  get transportSubscriptionClient() {
    var _a, _b;

    const client = (_a = this._transportSubscriptionClient) !== null && _a !== void 0 ? _a : new SubscriptionClient$1(this.url, {
      reconnect: true,
      lazy: true,
      connectionParams: (_b = this.headers) !== null && _b !== void 0 ? _b : {}
    });
    this._transportSubscriptionClient = client;
    return client;
  }

  onConnected(callback) {
    if (this.protocol === 'graphql-ws') {
      return this.graphWsClient.on('connected', callback);
    }

    if (this.protocol === 'subscriptions-transport-ws') {
      return this.transportSubscriptionClient.onConnected(callback);
    }

    assertUnreachable(this.protocol);
  }

  onConnecting(callback) {
    if (this.protocol === 'graphql-ws') {
      return this.graphWsClient.on('connecting', callback);
    }

    if (this.protocol === 'subscriptions-transport-ws') {
      return this.transportSubscriptionClient.onConnecting(callback);
    }

    assertUnreachable(this.protocol);
  }

  onError(callback) {
    if (this.protocol === 'graphql-ws') {
      return this.graphWsClient.on('error', error => callback(error));
    }

    if (this.protocol === 'subscriptions-transport-ws') {
      return this.transportSubscriptionClient.onError(e => callback(e));
    }

    assertUnreachable(this.protocol);
  }

  onReconnecting(callback) {
    if (this.protocol === 'graphql-ws') {
      return;
    }

    if (this.protocol === 'subscriptions-transport-ws') {
      return this.transportSubscriptionClient.onReconnecting(callback);
    }

    assertUnreachable(this.protocol);
  }

  onReconnected(callback) {
    if (this.protocol === 'graphql-ws') {
      return;
    }

    if (this.protocol === 'subscriptions-transport-ws') {
      return this.transportSubscriptionClient.onReconnected(callback);
    }

    assertUnreachable(this.protocol);
  }

  onDisconnected(callback) {
    if (this.protocol === 'graphql-ws') {
      return this.graphWsClient.on('closed', callback);
    }

    if (this.protocol === 'subscriptions-transport-ws') {
      return this.transportSubscriptionClient.onDisconnected(callback);
    }

    assertUnreachable(this.protocol);
  }

  request(params) {
    return {
      subscribe: subscribeParams => {
        if (this.protocol === 'graphql-ws') {
          this.unsubscribeFunctions.push(this.graphWsClient.subscribe(params, Object.assign(Object.assign({}, subscribeParams), {
            next: data => {
              var _a;

              return (_a = subscribeParams.next) === null || _a === void 0 ? void 0 : _a.call(subscribeParams, data);
            },
            error: error => {
              var _a;

              return (_a = subscribeParams.error) === null || _a === void 0 ? void 0 : _a.call(subscribeParams, error);
            },
            complete: () => {}
          })));
        }

        if (this.protocol === 'subscriptions-transport-ws') {
          return this.transportSubscriptionClient.request(params).subscribe(subscribeParams);
        } else {
          return undefined;
        }
      }
    };
  }

  unsubscribeAll() {
    if (this.protocol === 'graphql-ws') {
      this.unsubscribeFunctions.forEach(off => {
        off();
      });
      this.unsubscribeFunctions = [];
    }

    if (this.protocol === 'subscriptions-transport-ws') {
      return this.transportSubscriptionClient.unsubscribeAll();
    }
  }

}

function setParentSocketError({
  error,
  embeddedIFrameElement,
  embedUrl
}) {
  sendPostMessageToEmbed({
    message: {
      name: EXPLORER_SET_SOCKET_ERROR,
      error
    },
    embeddedIFrameElement,
    embedUrl
  });
}

function setParentSocketStatus({
  status,
  embeddedIFrameElement,
  embedUrl
}) {
  sendPostMessageToEmbed({
    message: {
      name: EXPLORER_SET_SOCKET_STATUS,
      status
    },
    embeddedIFrameElement,
    embedUrl
  });
}

function executeSubscription({
  operation,
  operationName,
  variables,
  headers,
  embeddedIFrameElement,
  operationId,
  embedUrl,
  subscriptionUrl,
  protocol
}) {
  const client = new SubscriptionClient(subscriptionUrl, headers !== null && headers !== void 0 ? headers : {}, protocol);

  const checkForSubscriptionTermination = event => {
    if (event.data.name === EXPLORER_SUBSCRIPTION_TERMINATION) {
      client.unsubscribeAll();
      window.removeEventListener('message', checkForSubscriptionTermination);
    }
  };

  window.addEventListener('message', checkForSubscriptionTermination);
  client.onError(e => setParentSocketError({
    error: JSON.parse(JSON.stringify(e)),
    embeddedIFrameElement,
    embedUrl
  }));
  client.onConnected(() => {
    setParentSocketError({
      error: undefined,
      embeddedIFrameElement,
      embedUrl
    });
    setParentSocketStatus({
      status: 'connected',
      embeddedIFrameElement,
      embedUrl
    });
  });
  client.onReconnected(() => {
    setParentSocketError({
      error: undefined,
      embeddedIFrameElement,
      embedUrl
    });
    setParentSocketStatus({
      status: 'connected',
      embeddedIFrameElement,
      embedUrl
    });
  });
  client.onConnecting(() => setParentSocketStatus({
    status: 'connecting',
    embeddedIFrameElement,
    embedUrl
  }));
  client.onReconnecting(() => setParentSocketStatus({
    status: 'connecting',
    embeddedIFrameElement,
    embedUrl
  }));
  client.onDisconnected(() => setParentSocketStatus({
    status: 'disconnected',
    embeddedIFrameElement,
    embedUrl
  }));
  client.request({
    query: operation,
    variables: variables !== null && variables !== void 0 ? variables : {},
    operationName
  }).subscribe({
    next(data) {
      sendPostMessageToEmbed({
        message: {
          // Include the same operation ID in the response message's name
          // so the Explorer knows which operation it's associated with
          name: EXPLORER_SUBSCRIPTION_RESPONSE,
          operationId,
          response: {
            data
          }
        },
        embeddedIFrameElement,
        embedUrl
      });
    },

    error: error => {
      sendPostMessageToEmbed({
        message: {
          // Include the same operation ID in the response message's name
          // so the Explorer knows which operation it's associated with
          name: EXPLORER_SUBSCRIPTION_RESPONSE,
          operationId,
          response: {
            error: JSON.parse(JSON.stringify(error))
          }
        },
        embeddedIFrameElement,
        embedUrl
      });
    }
  });
}

function setupEmbedRelay({
  endpointUrl,
  handleRequest,
  embeddedExplorerIFrameElement,
  updateSchemaInEmbed,
  schema,
  graphRef,
  autoInviteOptions,
  __testLocal__
}) {
  const embedUrl = EMBEDDABLE_EXPLORER_URL(__testLocal__); // Callback definition

  const onPostMessageReceived = event => {
    handleAuthenticationPostMessage({
      event,
      embedUrl,
      embeddedIFrameElement: embeddedExplorerIFrameElement
    }); // Any pm can be listened for here, not just the ones we know the
    // structure of. Some have a data field that is not an object

    const data = typeof event.data === 'object' ? event.data : undefined;

    if (data && 'name' in data) {
      // When embed connects, send a handshake message
      if (data.name === EXPLORER_LISTENING_FOR_HANDSHAKE) {
        sendPostMessageToEmbed({
          message: {
            name: HANDSHAKE_RESPONSE,
            graphRef,
            inviteToken: autoInviteOptions === null || autoInviteOptions === void 0 ? void 0 : autoInviteOptions.inviteToken,
            accountId: autoInviteOptions === null || autoInviteOptions === void 0 ? void 0 : autoInviteOptions.accountId,
            parentHref: window.location.href
          },
          embeddedIFrameElement: embeddedExplorerIFrameElement,
          embedUrl
        });
      } // Embedded Explorer sends us a PM when it is ready for a schema


      if (data.name === EXPLORER_LISTENING_FOR_SCHEMA && !!schema) {
        updateSchemaInEmbed({
          schema
        });
      } // Check to see if the posted message indicates that the user is
      // executing a query or mutation or subscription in the Explorer


      const isQueryOrMutation = data.name === EXPLORER_QUERY_MUTATION_REQUEST;
      const isSubscription = data.name === EXPLORER_SUBSCRIPTION_REQUEST; // If the user is executing a query or mutation or subscription...

      if ((isQueryOrMutation || isSubscription) && data.operation && data.operationId) {
        // Extract the operation details from the event.data object
        const {
          operation,
          operationId,
          operationName,
          variables,
          headers
        } = data;

        if (isQueryOrMutation) {
          // we support the old way of using the embed when we didn't require folks to have
          // studio graphs, which is to pass in an endpoint manually. However, we use the
          // endpoint sent to us from studio if there is no endpoint passed in manually
          const endpointUrlToUseInExecution = endpointUrl !== null && endpointUrl !== void 0 ? endpointUrl : data.endpointUrl;

          if (!endpointUrlToUseInExecution) {
            throw new Error('Something went wrong, we should not have gotten here. Please double check that you are passing `endpointUrl` in your config if you are using an older version of embedded Explorer');
          }

          executeOperation({
            endpointUrl: endpointUrlToUseInExecution,
            handleRequest,
            operation,
            operationName,
            variables,
            headers,
            embeddedIFrameElement: embeddedExplorerIFrameElement,
            operationId,
            embedUrl
          });
        } else if (isSubscription) {
          if (!!schema) {
            setParentSocketError({
              error: new Error('you cannot run subscriptions from this embed, since you are not embedding with a registered Studio graph'),
              embeddedIFrameElement: embeddedExplorerIFrameElement,
              embedUrl
            });
          } else {
            executeSubscription({
              operation,
              operationName,
              variables,
              headers,
              embeddedIFrameElement: embeddedExplorerIFrameElement,
              operationId,
              embedUrl,
              subscriptionUrl: data.subscriptionUrl,
              protocol: data.protocol
            });
          }
        }
      }
    }
  }; // Execute our callback whenever window.postMessage is called


  window.addEventListener('message', onPostMessageReceived);
  return {
    dispose: () => window.removeEventListener('message', onPostMessageReceived)
  };
}

var name = "@apollo/explorer";
var version = "2.0.2";
var author = "packages@apollographql.com";
var license = "MIT";
var repository = {
	type: "git",
	url: "https://github.com/apollographql/embeddable-explorer"
};
var homepage = "https://github.com/apollographql/embeddable-explorer#readme";
var main = "dist/index.cjs";
var module = "dist/index.mjs";
var typings = "dist/src/index.d.ts";
var engines = {
	node: ">=12.0",
	npm: ">=7.0"
};
var volta = {
	node: "16.13.0",
	npm: "8.3.1"
};
var scripts = {
	build: "npm run build:cjs-esm",
	"build:cjs-esm": "rm -rf dist && rollup -c buildHelpers/rollup.cjs-esm.config.js && cp src/index.cjs dist/index.cjs && cp src/react/index.cjs dist/react/index.cjs && node ./buildHelpers/prepareDist.js explorer",
	"build:umd": "rm -rf dist && rollup -c buildHelpers/rollup.umd.config.js",
	lint: "eslint --ext .js,.jsx,.ts,.tsx .",
	size: "size-limit",
	analyze: "size-limit --why",
	"typescript:check": "tsc --noEmit",
	"prettier:check": "prettier --check .",
	"prettier:fix": "prettier --write ."
};
var husky = {
	hooks: {
		"pre-commit": "npm run lint"
	}
};
var prettier = {
	printWidth: 80,
	semi: true,
	singleQuote: true,
	trailingComma: "es5",
	endOfLine: "auto"
};
var peerDependencies = {
	react: "^16.8.0 || ^17.0.0 || ^18.0.0",
	"react-dom": "^16.8.0 || ^17.0.0 || ^18.0.0",
	"use-deep-compare-effect": "^1.8.1"
};
var peerDependenciesMeta = {
	react: {
		optional: true
	},
	"react-dom": {
		optional: true
	},
	"use-deep-compare-effect": {
		optional: true
	}
};
var dependencies = {
	"@types/whatwg-mimetype": "^3.0.0",
	"graphql-ws": "^5.9.0",
	"subscriptions-transport-ws": "^0.11.0",
	"whatwg-mimetype": "^3.0.0",
	"zen-observable-ts": "^1.1.0"
};
var packageJSON = {
	name: name,
	version: version,
	author: author,
	license: license,
	repository: repository,
	homepage: homepage,
	main: main,
	module: module,
	typings: typings,
	engines: engines,
	volta: volta,
	scripts: scripts,
	husky: husky,
	prettier: prettier,
	"size-limit": [
	{
		path: "dist/index.production.min.js",
		limit: "10 KB"
	},
	{
		path: "dist/index.mjs",
		limit: "10 KB"
	}
],
	peerDependencies: peerDependencies,
	peerDependenciesMeta: peerDependenciesMeta,
	dependencies: dependencies
};

let idCounter = 0;
class EmbeddedExplorer {
  constructor(options) {
    var _a;

    this.getEmbeddedExplorerURL = () => {
      var _a, _b;

      const {
        document,
        variables,
        headers,
        displayOptions
      } = this.options.initialState || {};
      const {
        persistExplorerState
      } = this.options;
      const graphRef = 'graphRef' in this.options ? this.options.graphRef : undefined;
      const queryParams = {
        graphRef,
        defaultDocument: document ? encodeURIComponent(document) : undefined,
        defaultVariables: variables ? encodeURIComponent(JSON.stringify(variables, null, 2)) : undefined,
        defaultHeaders: headers ? encodeURIComponent(JSON.stringify(headers)) : undefined,
        shouldPersistState: !!persistExplorerState,
        docsPanelState: (_a = displayOptions === null || displayOptions === void 0 ? void 0 : displayOptions.docsPanelState) !== null && _a !== void 0 ? _a : 'open',
        showHeadersAndEnvVars: (displayOptions === null || displayOptions === void 0 ? void 0 : displayOptions.showHeadersAndEnvVars) !== false,
        theme: (_b = displayOptions === null || displayOptions === void 0 ? void 0 : displayOptions.theme) !== null && _b !== void 0 ? _b : 'dark',
        shouldShowGlobalHeader: true,
        parentSupportsSubscriptions: !!graphRef,
        version: packageJSON.version,
        runTelemetry: true
      };
      const queryString = Object.entries(queryParams).filter(([_, value]) => value !== undefined).map(([key, value]) => `${key}=${value}`).join('&');
      return `${EMBEDDABLE_EXPLORER_URL(this.__testLocal__)}?${queryString}`;
    };

    this.options = options;
    this.__testLocal__ = !!this.options.__testLocal__;
    this.validateOptions();
    this.handleRequest = (_a = this.options.handleRequest) !== null && _a !== void 0 ? _a : defaultHandleRequest({
      includeCookies: !!this.options.includeCookies
    });
    this.uniqueEmbedInstanceId = idCounter++;
    this.embeddedExplorerURL = this.getEmbeddedExplorerURL();
    this.embeddedExplorerIFrameElement = this.injectEmbed();
    this.disposable = setupEmbedRelay({
      embeddedExplorerIFrameElement: this.embeddedExplorerIFrameElement,
      endpointUrl: this.options.endpointUrl,
      handleRequest: this.handleRequest,
      updateSchemaInEmbed: this.updateSchemaInEmbed.bind(this),
      schema: 'schema' in this.options ? this.options.schema : undefined,
      graphRef: 'graphRef' in this.options ? this.options.graphRef : undefined,
      autoInviteOptions: this.options.autoInviteOptions,
      __testLocal__: this.__testLocal__
    });
  }

  dispose() {
    var _a; // remove the dom element


    (_a = document.getElementById(IFRAME_DOM_ID(this.uniqueEmbedInstanceId))) === null || _a === void 0 ? void 0 : _a.remove(); // remove the listener

    this.disposable.dispose();
  }

  injectEmbed() {
    var _a;

    let element;
    const {
      target
    } = this.options;

    if (typeof target === 'string') {
      element = (_a = document === null || document === void 0 ? void 0 : document.querySelector) === null || _a === void 0 ? void 0 : _a.call(document, target);
    } else {
      element = target;
    }

    const iframeElement = document.createElement('iframe');
    iframeElement.src = this.embeddedExplorerURL;
    iframeElement.id = IFRAME_DOM_ID(this.uniqueEmbedInstanceId);
    iframeElement.setAttribute('style', 'height: 100%; width: 100%; border: none;');
    element === null || element === void 0 ? void 0 : element.appendChild(iframeElement); // inject the Apollo favicon if there is not one on this page

    fetch(`${window.location.origin}/favicon.ico`).then(response => {
      if (response.status === 404) {
        var existingLink = document.querySelector('link[rel*="icon"]');

        if (!existingLink) {
          var darkMode = window.matchMedia('(prefers-color-scheme: dark)').matches;
          ['icon', 'apple-touch-icon'].forEach(id => {
            var link = document.createElement('link');
            link.rel = id;
            link.href = `https://embeddable-explorer.cdn.apollographql.com/_latest/public/assets/favicon${darkMode ? '-dark' : ''}.png`;
            document.head.appendChild(link);
          });
        }
      }
    }).catch(() => {// do nothing with the error
    });
    return iframeElement;
  }

  validateOptions() {
    if (!this.options.target) {
      throw new Error('"target" is required');
    }

    if ('endpointUrl' in this.options && 'graphRef' in this.options) {
      // we can't throw here for backwards compat reasons. Folks on the cdn _latest bundle
      // will still be passing this
      console.warn('You may only specify an endpointUrl if you are manually passing a `schema`. If you pass a `graphRef`, you must configure your endpoint on your Studio graph.');
    }

    if ('schema' in this.options && 'graphRef' in this.options) {
      throw new Error('Both `schema` and `graphRef` cannot be set. You can either send your schema as an IntrospectionQuery or string via the `schema` field, or specifiy a public graphRef.');
    }

    if (!('schema' in this.options || 'graphRef' in this.options)) {
      throw new Error('You must set either `schema` or `graphRef`.');
    }
  }

  updateSchemaInEmbed({
    schema
  }) {
    sendPostMessageToEmbed({
      message: {
        name: SCHEMA_RESPONSE,
        schema
      },
      embeddedIFrameElement: this.embeddedExplorerIFrameElement,
      embedUrl: EMBEDDABLE_EXPLORER_URL(this.__testLocal__)
    });
  }

}

export { EmbeddedExplorer as E };
