(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
  typeof define === 'function' && define.amd ? define(['exports'], factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.aq = {}));
})(this, (function (exports) { 'use strict';

  const ONE = 0x80000000;
  const ALL = 0xFFFFFFFF;

  /**
   * Represent an indexable set of bits.
   */
  class BitSet {
    /**
     * Instantiate a new BitSet instance.
     * @param {number} size The number of bits.
     */
    constructor(size) {
      this._size = size;
      this._bits = new Uint32Array(Math.ceil(size / 32));
    }

    /**
     * The number of bits.
     * @return {number}
     */
    get length() {
      return this._size;
    }

    /**
     * The number of bits set to one.
     * https://graphics.stanford.edu/~seander/bithacks.html#CountBitsSetKernighan
     * @return {number}
     */
    count() {
      const n = this._bits.length;
      let count = 0;
      for (let i = 0; i < n; ++i) {
        for (let b = this._bits[i]; b; ++count) {
          b &= b - 1;
        }
      }
      return count;
    }

    /**
     * Get the bit at a given index.
     * @param {number} i The bit index.
     */
    get(i) {
      return this._bits[i >> 5] & (ONE >>> i);
    }

    /**
     * Set the bit at a given index to one.
     * @param {number} i The bit index.
     */
    set(i) {
      this._bits[i >> 5] |= (ONE >>> i);
    }

    /**
     * Clear the bit at a given index to zero.
     * @param {number} i The bit index.
     */
    clear(i) {
      this._bits[i >> 5] &= ~(ONE >>> i);
    }

    /**
     * Scan the bits, invoking a callback function with the index of
     * each non-zero bit.
     * @param {(i: number) => void} fn A callback function.
     */
    scan(fn) {
      for (let i = this.next(0); i >= 0; i = this.next(i + 1)) {
        fn(i);
      }
    }

    /**
     * Get the next non-zero bit starting from a given index.
     * @param {number} i The bit index.
     */
    next(i) {
      const bits = this._bits;
      const n = bits.length;

      let index = i >> 5;
      let curr = bits[index] & (ALL >>> i);

      for (; index < n; curr = bits[++index]) {
        if (curr !== 0) {
          return (index << 5) + Math.clz32(curr);
        }
      }

      return -1;
    }

    /**
     * Return the index of the nth non-zero bit.
     * @param {number} n The number of non-zero bits to advance.
     * @return {number} The index of the nth non-zero bit.
     */
    nth(n) {
      let i = this.next(0);
      while (n-- && i >= 0) i = this.next(i + 1);
      return i;
    }

    /**
     * Negate all bits in this bitset.
     * Modifies this BitSet in place.
     * @return {this}
     */
    not() {
      const bits = this._bits;
      const n = bits.length;

      // invert all bits
      for (let i = 0; i < n; ++i) {
        bits[i] = ~bits[i];
      }

      // unset extraneous trailing bits
      const tail = this._size % 32;
      if (tail) {
        bits[n - 1] &= ONE >> (tail - 1);
      }

      return this;
    }

    /**
     * Compute the logical AND of this BitSet and another.
     * @param {BitSet} bitset The BitSet to combine with.
     * @return {BitSet} This BitSet updated with the logical AND.
     */
    and(bitset) {
      if (bitset) {
        const a = this._bits;
        const b = bitset._bits;
        const n = a.length;

        for (let i = 0; i < n; ++i) {
          a[i] &= b[i];
        }
      }
      return this;
    }

    /**
     * Compute the logical OR of this BitSet and another.
     * @param {BitSet} bitset The BitSet to combine with.
     * @return {BitSet} This BitSet updated with the logical OR.
     */
    or(bitset) {
      if (bitset) {
        const a = this._bits;
        const b = bitset._bits;
        const n = a.length;

        for (let i = 0; i < n; ++i) {
          a[i] |= b[i];
        }
      }
      return this;
    }
  }

  /**
   * Default NULL (missing) value to use.
   */
  var NULL = undefined;

  var isArray$2 = Array.isArray;

  const TypedArray$1 = Object.getPrototypeOf(Int8Array);

  /**
   * @param {*} value
   * @return {value is import("../table/types.js").TypedArray}
   */
  function isTypedArray$1(value) {
    return value instanceof TypedArray$1;
  }

  /**
   * @param {*} value
   * @return {value is (any[] | import('../table/types.js').TypedArray)}
   */
  function isArrayType(value) {
    return isArray$2(value) || isTypedArray$1(value);
  }

  /**
   * @param {*} value
   * @return {value is String}
   */
  function isString(value) {
    return typeof value === 'string';
  }

  function isValid(value) {
    return value != null && value === value;
  }

  const isSeq = (seq) => isArrayType(seq) || isString(seq);

  var array$1 = {
    /**
     * Returns a new compacted array with invalid values
     * (`null`, `undefined`, `NaN`) removed.
     * @template T
     * @param {T[]} array The input array.
     * @return {T[]} A compacted array.
     */
    compact: (array) => isArrayType(array)
      ? array.filter(v => isValid(v))
      : array,

    /**
     * Merges two or more arrays in sequence, returning a new array.
     * @template T
     * @param {...(T|T[])} values The arrays to merge.
     * @return {T[]} The merged array.
     */
    concat: (...values) => [].concat(...values),

    /**
     * Determines whether an *array* includes a certain *value* among its
     * entries, returning `true` or `false` as appropriate.
     * @template T
     * @param {T[]} sequence The input array value.
     * @param {T} value The value to search for.
     * @param {number} [index=0] The integer index to start searching
     *  from (default `0`).
     * @return {boolean} True if the value is included, false otherwise.
     */
    includes: (sequence, value, index) => isSeq(sequence)
      ? sequence.includes(value, index)
      : false,

    /**
     * Returns the first index at which a given *value* can be found in the
     * *sequence* (array or string), or -1 if it is not present.
     * @template T
     * @param {T[]|string} sequence The input array or string value.
     * @param {T} value The value to search for.
     * @return {number} The index of the value, or -1 if not present.
     */
    indexof: (sequence, value) => isSeq(sequence)
      // @ts-ignore
      ? sequence.indexOf(value)
      : -1,

    /**
     * Creates and returns a new string by concatenating all of the elements
     * in an *array* (or an array-like object), separated by commas or a
     * specified *delimiter* string. If the *array* has only one item, then
     * that item will be returned without using the delimiter.
     * @template T
     * @param {T[]} array The input array value.
     * @param {string} delim The delimiter string (default `','`).
     * @return {string} The joined string.
     */
    join: (array, delim) => isArrayType(array) ? array.join(delim) : NULL,

    /**
     * Returns the last index at which a given *value* can be found in the
     * *sequence* (array or string), or -1 if it is not present.
     * @template T
     * @param {T[]|string} sequence The input array or string value.
     * @param {T} value The value to search for.
     * @return {number} The last index of the value, or -1 if not present.
     */
    lastindexof: (sequence, value) => isSeq(sequence)
      // @ts-ignore
      ? sequence.lastIndexOf(value)
      : -1,

    /**
     * Returns the length of the input *sequence* (array or string).
     * @param {Array|string} sequence The input array or string value.
     * @return {number} The length of the sequence.
     */
    length: (sequence) => isSeq(sequence) ? sequence.length : 0,

    /**
     * Returns a new array in which the given *property* has been extracted
     * for each element in the input *array*.
     * @param {Array} array The input array value.
     * @param {string} property The property name string to extract. Nested
     *  properties are not supported: the input `"a.b"` will indicates a
     *  property with that exact name, *not* a nested property `"b"` of
     *  the object `"a"`.
     * @return {Array} An array of plucked properties.
     */
    pluck: (array, property) => isArrayType(array)
      ? array.map(v => isValid(v) ? v[property] : NULL)
      : NULL,

    /**
     * Returns a new array or string with the element order reversed: the first
     * *sequence* element becomes the last, and the last *sequence* element
     * becomes the first. The input *sequence* is unchanged.
     * @template T
     * @param {T[]|string} sequence The input array or string value.
     * @return {T[]|string} The reversed sequence.
     */
    reverse: (sequence) => isArrayType(sequence) ? sequence.slice().reverse()
      : isString(sequence) ? sequence.split('').reverse().join('')
      : NULL,

    /**
     * Returns a copy of a portion of the input *sequence* (array or string)
     * selected from *start* to *end* (*end* not included) where *start* and
     * *end* represent the index of items in the sequence.
     * @template T
     * @param {T[]|string} sequence The input array or string value.
     * @param {number} [start=0] The starting integer index to copy from
     *  (inclusive, default `0`).
     * @param {number} [end] The ending integer index to copy from (exclusive,
     *  default `sequence.length`).
     * @return {T[]|string} The sliced sequence.
     */
    slice: (sequence, start, end) => isSeq(sequence)
      ? sequence.slice(start, end)
      : NULL
  };

  /**
   * Truncate a value to a bin boundary.
   * Useful for creating equal-width histograms.
   * Values outside the [min, max] range will be mapped to
   * -Infinity (< min) or +Infinity (> max).
   * @param {number} value The value to bin.
   * @param {number} min The minimum bin boundary.
   * @param {number} max The maximum bin boundary.
   * @param {number} step The step size between bin boundaries.
   * @param {number} [offset=0] Offset in steps by which to adjust
   *  the bin value. An offset of 1 will return the next boundary.
   */
  function bin$1(value, min, max, step, offset) {
    return value == null ? null
      : value < min ? -Infinity
      : value > max ? +Infinity
      : (
          value = Math.max(min, Math.min(value, max)),
          min + step * Math.floor(1e-14 + (value - min) / step + (offset || 0))
        );
  }

  function pad(value, width, char = '0') {
    const s = value + '';
    const len = s.length;
    return len < width ? Array(width - len + 1).join(char) + s : s;
  }

  const pad2 = v => (v < 10 ? '0' : '') + v;

  const formatYear = year => year < 0 ? '-' + pad(-year, 6)
    : year > 9999 ? '+' + pad(year, 6)
    : pad(year, 4);

  function formatISO(year, month, date, hours, min, sec, ms, utc, short) {
    const suffix = utc ? 'Z' : '';
    return formatYear(year) + '-' + pad2(month + 1) + '-' + pad2(date) + (
      !short || ms ? 'T' + pad2(hours) + ':' + pad2(min) + ':' + pad2(sec) + '.' + pad(ms, 3) + suffix
      : sec ? 'T' + pad2(hours) + ':' + pad2(min) + ':' + pad2(sec) + suffix
      : min || hours || !utc ? 'T' + pad2(hours) + ':' + pad2(min) + suffix
      : ''
    );
  }

  function formatDate(d, short) {
    return isNaN(d)
      ? 'Invalid Date'
      : formatISO(
        d.getFullYear(),
        d.getMonth(),
        d.getDate(),
        d.getHours(),
        d.getMinutes(),
        d.getSeconds(),
        d.getMilliseconds(),
        false, short
      );
  }

  function formatUTCDate(d, short) {
    return isNaN(d)
      ? 'Invalid Date'
      : formatISO(
        d.getUTCFullYear(),
        d.getUTCMonth(),
        d.getUTCDate(),
        d.getUTCHours(),
        d.getUTCMinutes(),
        d.getUTCSeconds(),
        d.getUTCMilliseconds(),
        true, short
      );
  }

  const iso_re = /^([-+]\d{2})?\d{4}(-\d{2}(-\d{2})?)?(T\d{2}:\d{2}(:\d{2}(\.\d{3})?)?(Z|[-+]\d{2}:\d{2})?)?$/;

  function isISODateString(value) {
    return value.match(iso_re) && !isNaN(Date.parse(value));
  }

  function parseIsoDate(value, parse = Date.parse) {
    return isISODateString(value) ? parse(value) : value;
  }

  const msMinute = 6e4;
  const msDay = 864e5;
  const msWeek = 6048e5;

  const t0 = new Date();
  const t1 = new Date();
  const t = d => (
    t0.setTime(typeof d === 'string' ? parseIsoDate(d) : d),
    t0
  );

  /**
   * Function to create a new Date value.
   * If no arguments are provided, the current time is used.
   * @param {number} [year] The year.
   * @param {number} [month=0] The (zero-based) month.
   * @param {number} [date=1] The date within the month.
   * @param {number} [hours=0] The hour within the day.
   * @param {number} [minutes=0] The minute within the hour.
   * @param {number} [seconds=0] The second within the minute.
   * @param {number} [milliseconds=0] The milliseconds within the second.
   * @return {Date} The resuting Date value.
   */
  function datetime(year, month, date, hours, minutes, seconds, milliseconds) {
    return !arguments.length
      ? new Date(Date.now())
      : new Date(
          year,
          month || 0,
          date == null ? 1 : date,
          hours || 0,
          minutes || 0,
          seconds || 0,
          milliseconds || 0
        );
  }

  /**
   * Function to create a new Date value according to UTC time.
   * If no arguments are provided, the current time is used.
   * @param {number} [year] The year.
   * @param {number} [month=0] The (zero-based) month.
   * @param {number} [date=1] The date within the month.
   * @param {number} [hours=0] The hour within the day.
   * @param {number} [minutes=0] The minute within the hour.
   * @param {number} [seconds=0] The second within the minute.
   * @param {number} [milliseconds=0] The milliseconds within the second.
   * @return {Date} The resuting Date value.
   */
  function utcdatetime(year, month, date, hours, minutes, seconds, milliseconds) {
    return !arguments.length
      ? new Date(Date.now())
      : new Date(Date.UTC(
          year,
          month || 0,
          date == null ? 1 : date,
          hours || 0,
          minutes || 0,
          seconds || 0,
          milliseconds || 0
        ));
  }

  /**
   * Return the current day of the year in local time as a number
   * between 1 and 366.
   * @param {Date|number} date A date or timestamp.
   * @return {number} The day of the year in local time.
   */
  function dayofyear(date) {
    t1.setTime(+date);
    t1.setHours(0, 0, 0, 0);
    t0.setTime(+t1);
    t0.setMonth(0);
    t0.setDate(1);
    const tz = (t1.getTimezoneOffset() - t0.getTimezoneOffset()) * msMinute;
    return Math.floor(1 + ((+t1 - +t0) - tz) / msDay);
  }

  /**
   * Return the current day of the year in UTC time as a number
   * between 1 and 366.
   * @param {Date|number} date A date or timestamp.
   * @return {number} The day of the year in UTC time.
   */
  function utcdayofyear(date) {
    t1.setTime(+date);
    t1.setUTCHours(0, 0, 0, 0);
    const t0 = Date.UTC(t1.getUTCFullYear(), 0, 1);
    return Math.floor(1 + (+t1 - t0) / msDay);
  }

  /**
   * Return the current week of the year in local time as a number
   * between 1 and 52.
   * @param {Date|number} date A date or timestamp.
   * @return {number} The week of the year in local time.
   */
  function week(date, firstday) {
    const i = firstday || 0;
    t1.setTime(+date);
    t1.setDate(t1.getDate() - (t1.getDay() + 7 - i) % 7);
    t1.setHours(0, 0, 0, 0);
    t0.setTime(+date);
    t0.setMonth(0);
    t0.setDate(1);
    t0.setDate(1 - (t0.getDay() + 7 - i) % 7);
    t0.setHours(0, 0, 0, 0);
    const tz = (t1.getTimezoneOffset() - t0.getTimezoneOffset()) * msMinute;
    return Math.floor((1 + (+t1 - +t0) - tz) / msWeek);
  }

  /**
   * Return the current week of the year in UTC time as a number
   * between 1 and 52.
   * @param {Date|number} date A date or timestamp.
   * @return {number} The week of the year in UTC time.
   */
  function utcweek(date, firstday) {
    const i = firstday || 0;
    t1.setTime(+date);
    t1.setUTCDate(t1.getUTCDate() - (t1.getUTCDay() + 7 - i) % 7);
    t1.setUTCHours(0, 0, 0, 0);
    t0.setTime(+date);
    t0.setUTCMonth(0);
    t0.setUTCDate(1);
    t0.setUTCDate(1 - (t0.getUTCDay() + 7 - i) % 7);
    t0.setUTCHours(0, 0, 0, 0);
    return Math.floor((1 + (+t1 - +t0)) / msWeek);
  }

  var date$1 = {
    /**
     * Returns an [ISO 8601](https://en.wikipedia.org/wiki/ISO_8601) formatted
     * string for the given *date* in local timezone. The resulting string is
     * compatible with *parse_date* and JavaScript's built-in *Date.parse*.
     * @param {Date | number} date The input Date or timestamp value.
     * @param {boolean} [shorten=false] A boolean flag (default `false`)
     *  indicating if the formatted string should be shortened if possible.
     *  For example, the local date `2001-01-01` will shorten from
     *  `"2001-01-01T00:00:00.000"` to `"2001-01-01T00:00"`.
     * @return {string} The formatted date string in local time.
     */
    format_date: (date, shorten) => formatDate(t(date), !shorten),

    /**
     * Returns an [ISO 8601](https://en.wikipedia.org/wiki/ISO_8601) formatted
     * string for the given *date* in Coordinated Universal Time (UTC). The
     * resulting string is compatible with *parse_date* and JavaScript's
     * built-in *Date.parse*.
     * @param {Date | number} date The input Date or timestamp value.
     * @param {boolean} [shorten=false] A boolean flag (default `false`)
     *  indicating if the formatted string should be shortened if possible.
     *  For example, the the UTC date `2001-01-01` will shorten from
     *  `"2001-01-01T00:00:00.000Z"` to `"2001-01-01"`
     * @return {string} The formatted date string in UTC time.
     */
    format_utcdate: (date, shorten) => formatUTCDate(t(date), !shorten),

    /**
     * Returns the number of milliseconds elapsed since midnight, January 1,
     * 1970 Universal Coordinated Time (UTC).
     * @return {number} The timestamp for now.
     */
    now: Date.now,

    /**
     * Returns the timestamp for a *date* as the number of milliseconds elapsed
     * since January 1, 1970 00:00:00 UTC.
     * @param {Date | number} date The input Date value.
     * @return {number} The timestamp value.
     */
    timestamp: (date) => +t(date),

    /**
     * Creates and returns a new Date value. If no arguments are provided,
     * the current date and time are used.
     * @param {number} [year] The year.
     * @param {number} [month=0] The (zero-based) month.
     * @param {number} [date=1] The date within the month.
     * @param {number} [hours=0] The hour within the day.
     * @param {number} [minutes=0] The minute within the hour.
     * @param {number} [seconds=0] The second within the minute.
     * @param {number} [milliseconds=0] The milliseconds within the second.
     * @return {Date} The Date value.
     */
    datetime,

    /**
     * Returns the year of the specified *date* according to local time.
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The year value in local time.
     */
    year: (date) => t(date).getFullYear(),

    /**
     * Returns the zero-based quarter of the specified *date* according to
     * local time.
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The quarter value in local time.
     */
    quarter: (date) => Math.floor(t(date).getMonth() / 3),

    /**
     * Returns the zero-based month of the specified *date* according to local
     * time. A value of `0` indicates January, `1` indicates February, and so on.
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The month value in local time.
     */
    month: (date) => t(date).getMonth(),

    /**
     * Returns the week number of the year (0-53) for the specified *date*
     * according to local time. By default, Sunday is used as the first day
     * of the week. All days in a new year preceding the first Sunday are
     * considered to be in week 0.
     * @param {Date | number} date The input Date or timestamp value.
     * @param {number} firstday The number of first day of the week (default
     *  `0` for Sunday, `1` for Monday and so on).
     * @return {number} The week of the year in local time.
     */
    week,

    /**
     * Returns the date (day of month) of the specified *date* according
     * to local time.
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The date (day of month) value.
     */
    date: (date) => t(date).getDate(),

    /**
     * Returns the day of the year (1-366) of the specified *date* according
     * to local time.
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The day of the year in local time.
     */
    dayofyear,

    /**
     * Returns the Sunday-based day of the week (0-6) of the specified *date*
     * according to local time. A value of `0` indicates Sunday, `1` indicates
     * Monday, and so on.
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The day of the week value in local time.
     */
    dayofweek: (date) => t(date).getDay(),

    /**
     * Returns the hour of the day for the specified *date* according
     * to local time.
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The hour value in local time.
     */
    hours: (date) => t(date).getHours(),

    /**
     * Returns the minute of the hour for the specified *date* according
     * to local time.
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The minutes value in local time.
     */
    minutes: (date) => t(date).getMinutes(),

    /**
     * Returns the seconds of the minute for the specified *date* according
     * to local time.
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The seconds value in local time.
     */
    seconds: (date) => t(date).getSeconds(),

    /**
     * Returns the milliseconds of the second for the specified *date* according
     * to local time.
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The milliseconds value in local time.
     */
    milliseconds: (date) => t(date).getMilliseconds(),

    /**
     * Creates and returns a new Date value using Coordinated Universal Time
     * (UTC). If no arguments are provided, the current date and time are used.
     * @param {number} [year] The year.
     * @param {number} [month=0] The (zero-based) month.
     * @param {number} [date=1] The date within the month.
     * @param {number} [hours=0] The hour within the day.
     * @param {number} [minutes=0] The minute within the hour.
     * @param {number} [seconds=0] The second within the minute.
     * @param {number} [milliseconds=0] The milliseconds within the second.
     * @return {Date} The Date value.
     */
    utcdatetime,

    /**
     * Returns the year of the specified *date* according to Coordinated
     * Universal Time (UTC).
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The year value in UTC time.
     */
    utcyear: (date) => t(date).getUTCFullYear(),

    /**
     * Returns the zero-based quarter of the specified *date* according to
     * Coordinated Universal Time (UTC)
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The quarter value in UTC time.
     */
    utcquarter: (date) => Math.floor(t(date).getUTCMonth() / 3),

    /**
     * Returns the zero-based month of the specified *date* according to
     * Coordinated Universal Time (UTC). A value of `0` indicates January,
     * `1` indicates February, and so on.
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The month value in UTC time.
     */
    utcmonth: (date) => t(date).getUTCMonth(),

    /**
     * Returns the week number of the year (0-53) for the specified *date*
     * according to Coordinated Universal Time (UTC). By default, Sunday is
     * used as the first day of the week. All days in a new year preceding the
     * first Sunday are considered to be in week 0.
     * @param {Date | number} date The input Date or timestamp value.
     * @param {number} firstday The number of first day of the week (default
     *  `0` for Sunday, `1` for Monday and so on).
     * @return {number} The week of the year in UTC time.
     */
    utcweek,

    /**
     * Returns the date (day of month) of the specified *date* according to
     * Coordinated Universal Time (UTC).
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The date (day of month) value in UTC time.
     */
    utcdate: (date) => t(date).getUTCDate(),

    /**
     * Returns the day of the year (1-366) of the specified *date* according
     * to Coordinated Universal Time (UTC).
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The day of the year in UTC time.
     */
    utcdayofyear,

    /**
     * Returns the Sunday-based day of the week (0-6) of the specified *date*
     * according to Coordinated Universal Time (UTC). A value of `0` indicates
     * Sunday, `1` indicates Monday, and so on.
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The day of the week in UTC time.
     */
    utcdayofweek: (date) => t(date).getUTCDay(),

    /**
     * Returns the hour of the day for the specified *date* according to
     * Coordinated Universal Time (UTC).
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The hours value in UTC time.
     */
    utchours: (date) => t(date).getUTCHours(),

    /**
     * Returns the minute of the hour for the specified *date* according to
     * Coordinated Universal Time (UTC).
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The minutes value in UTC time.
     */
    utcminutes: (date) => t(date).getUTCMinutes(),

    /**
     * Returns the seconds of the minute for the specified *date* according to
     * Coordinated Universal Time (UTC).
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The seconds value in UTC time.
     */
    utcseconds: (date) => t(date).getUTCSeconds(),

    /**
     * Returns the milliseconds of the second for the specified *date* according to
     * Coordinated Universal Time (UTC).
     * @param {Date | number} date The input Date or timestamp value.
     * @return {number} The milliseconds value in UTC time.
     */
    utcmilliseconds: (date) => t(date).getUTCMilliseconds()
  };

  function isDate$1(value) {
    return value instanceof Date;
  }

  function isRegExp(value) {
    return value instanceof RegExp;
  }

  function isObject(value) {
    return value === Object(value);
  }

  /**
   * Compare two values for equality, using join semantics in which null
   * !== null. If the inputs are object-valued, a deep equality check
   * of array entries or object key-value pairs is performed.
   * @param {*} a The first input.
   * @param {*} b The second input.
   * @return {boolean} True if equal, false if not.
   */
  function equal(a, b) {
    return (a == null || b == null || a !== a || b !== b) ? false
      : a === b ? true
      : (isDate$1(a) || isDate$1(b)) ? +a === +b
      : (isRegExp(a) && isRegExp(b)) ? a + '' === b + ''
      : (isObject(a) && isObject(b)) ? deepEqual(a, b)
      : false;
  }

  function deepEqual(a, b) {
    if (Object.getPrototypeOf(a) !== Object.getPrototypeOf(b)) {
      return false;
    }

    if (a.length || b.length) {
      return arrayEqual(a, b);
    }

    const keysA = Object.keys(a);
    const keysB = Object.keys(b);
    if (keysA.length !== keysB.length) {
      return false;
    }
    keysA.sort();
    keysB.sort();

    if (!arrayEqual(keysA, keysB, (a, b) => a === b)) {
      return false;
    }

    const n = keysA.length;
    for (let i = 0; i < n; ++i) {
      const k = keysA[i];
      if (!equal(a[k], b[k])) {
        return false;
      }
    }

    return true;
  }

  function arrayEqual(a, b, test = equal) {
    const n = a.length;
    if (n !== b.length) return false;

    for (let i = 0; i < n; ++i) {
      if (!test(a[i], b[i])) {
        return false;
      }
    }

    return true;
  }

  var json = {
    /**
     * Parses a string *value* in JSON format, constructing the JavaScript
     * value or object described by the string.
     * @param {string} value The input string value.
     * @return {any} The parsed JSON.
     */
    parse_json: (value) => JSON.parse(value),

    /**
     * Converts a JavaScript object or value to a JSON string.
     * @param {*} value The value to convert to a JSON string.
     * @return {string} The JSON string.
     */
    to_json: (value) => JSON.stringify(value)
  };

  let source = Math.random;

  function random() {
    return source();
  }

  /**
   * Set a seed value for random number generation.
   * If the seed is a valid number, a 32-bit linear congruential generator
   * with the given seed will be used to generate random values.
   * If the seed is null, undefined, or not a valid number, the random
   * number generator will revert to Math.random.
   * @param {number} seed The random seed value. Should either be an
   *  integer or a fraction between 0 and 1.
   */
  function seed(seed) {
    source = isValid(seed) && isFinite(seed = +seed) ? lcg(seed) : Math.random;
  }

  function lcg(seed) {
    const a = 0x19660D;
    const c = 0x3C6EF35F;
    const m = 1 / 0x100000000;
    seed = (0 <= seed && seed < 1 ? seed / m : Math.abs(seed)) | 0;

    // Random numbers using a Linear Congruential Generator with seed value
    // https://en.wikipedia.org/wiki/Linear_congruential_generator
    return () => (seed = a * seed + c | 0, m * (seed >>> 0));
  }

  var math = {
    /**
     * Return a random floating point number between 0 (inclusive) and 1
     * (exclusive). By default uses *Math.random*. Use the *seed* method
     * to instead use a seeded random number generator.
     * @return {number} A pseudorandom number between 0 and 1.
     */
    random,

    /**
     * Tests if the input *value* is not a number (`NaN`); equivalent
     * to *Number.isNaN*.
     * @param {*} value The value to test.
     * @return {boolean} True if the value is not a number, false otherwise.
     */
    is_nan: Number.isNaN,

    /**
     * Tests if the input *value* is finite; equivalent to *Number.isFinite*.
     * @param {*} value The value to test.
     * @return {boolean} True if the value is finite, false otherwise.
     */
    is_finite: Number.isFinite,

    /**
     * Returns the absolute value of the input *value*; equivalent to *Math.abs*.
     * @param {number} value The input number value.
     * @return {number} The absolute value.
     */
    abs: Math.abs,

    /**
     * Returns the cube root value of the input *value*; equivalent to
     * *Math.cbrt*.
     * @param {number} value The input number value.
     * @return {number} The cube root value.
     */
    cbrt: Math.cbrt,

    /**
     * Returns the ceiling of the input *value*, the nearest integer equal to
     * or greater than the input; equivalent to *Math.ceil*.
     * @param {number} value The input number value.
     * @return {number} The ceiling value.
     */
    ceil: Math.ceil,

    /**
     * Returns the number of leading zero bits in the 32-bit binary
     * representation of a number *value*; equivalent to *Math.clz32*.
     * @param {number} value The input number value.
     * @return {number} The leading zero bits value.
     */
    clz32: Math.clz32,

    /**
     * Returns *e<sup>value</sup>*, where *e* is Euler's number, the base of the
     * natural logarithm; equivalent to *Math.exp*.
     * @param {number} value The input number value.
     * @return {number} The base-e exponentiated value.
     */
    exp: Math.exp,

    /**
     * Returns *e<sup>value</sup> - 1*, where *e* is Euler's number, the base of
     * the natural logarithm; equivalent to *Math.expm1*.
     * @param {number} value The input number value.
     * @return {number} The base-e exponentiated value minus 1.
     */
    expm1: Math.expm1,

    /**
     * Returns the floor of the input *value*, the nearest integer equal to or
     * less than the input; equivalent to *Math.floor*.
     * @param {number} value The input number value.
     * @return {number} The floor value.
     */
    floor: Math.floor,

    /**
     * Returns the nearest 32-bit single precision float representation of the
     * input number *value*; equivalent to *Math.fround*. Useful for translating
     * between 64-bit `Number` values and values from a `Float32Array`.
     * @param {number} value The input number value.
     * @return {number} The rounded value.
     */
    fround: Math.fround,

    /**
     * Returns the greatest (maximum) value among the input *values*; equivalent
     * to *Math.max*. This is _not_ an aggregate function, see *op.max* to
     * compute a maximum value across multiple rows.
     * @param {...number} values The input number values.
     * @return {number} The greatest (maximum) value among the inputs.
     */
    greatest: Math.max,

    /**
     * Returns the least (minimum) value among the input *values*; equivalent
     * to *Math.min*. This is _not_ an aggregate function, see *op.min* to
     * compute a minimum value across multiple rows.
     * @param {...number} values The input number values.
     * @return {number} The least (minimum) value among the inputs.
     */
    least: Math.min,

    /**
     * Returns the natural logarithm (base *e*) of a number *value*; equivalent
     * to *Math.log*.
     * @param {number} value The input number value.
     * @return {number} The base-e log value.
     */
    log: Math.log,

    /**
     * Returns the base 10 logarithm of a number *value*; equivalent
     * to *Math.log10*.
     * @param {number} value The input number value.
     * @return {number} The base-10 log value.
     */
    log10: Math.log10,

    /**
     * Returns the natural logarithm (base *e*) of 1 + a number *value*;
     * equivalent to *Math.log1p*.
     * @param {number} value The input number value.
     * @return {number} The base-e log of value + 1.
     */
    log1p: Math.log1p,

    /**
     * Returns the base 2 logarithm of a number *value*; equivalent
     * to *Math.log2*.
     * @param {number} value The input number value.
     * @return {number} The base-2 log value.
     */
    log2: Math.log2,

    /**
     * Returns the *base* raised to the *exponent* power, that is,
     * *base*<sup>*exponent*</sup>; equivalent to *Math.pow*.
     * @param {number} base The base number value.
     * @param {number} exponent The exponent number value.
     * @return {number} The exponentiated value.
     */
    pow: Math.pow,

    /**
     * Returns the value of a number rounded to the nearest integer;
     * equivalent to *Math.round*.
     * @param {number} value The input number value.
     * @return {number} The rounded value.
     */
    round: Math.round,

    /**
     * Returns either a positive or negative +/- 1, indicating the sign of the
     * input *value*; equivalent to *Math.sign*.
     * @param {number} value The input number value.
     * @return {number} The sign of the value.
     */
    sign: Math.sign,

    /**
     * Returns the square root of the input *value*; equivalent to *Math.sqrt*.
     * @param {number} value The input number value.
     * @return {number} The square root value.
     */
    sqrt: Math.sqrt,

    /**
     * Returns the integer part of a number by removing any fractional digits;
     * equivalent to *Math.trunc*.
     * @param {number} value The input number value.
     * @return {number} The truncated value.
     */
    trunc: Math.trunc,

    /**
     * Converts the input *radians* value to degrees.
     * @param {number} radians The input radians value.
     * @return {number} The value in degrees
     */
    degrees: (radians) => 180 * radians / Math.PI,

    /**
     * Converts the input *degrees* value to radians.
     * @param {number} degrees The input degrees value.
     * @return {number} The value in radians.
     */
    radians: (degrees) => Math.PI * degrees / 180,

    /**
     * Returns the arc-cosine (in radians) of a number *value*;
     * equivalent to *Math.acos*.
     * @param {number} value The input number value.
     * @return {number} The arc-cosine value.
     */
    acos: Math.acos,

    /**
     * Returns the hyperbolic arc-cosine of a number *value*;
     * equivalent to *Math.acosh*.
     * @param {number} value The input number value.
     * @return {number} The hyperbolic arc-cosine value.
     */
    acosh: Math.acosh,

    /**
     * Returns the arc-sine (in radians) of a number *value*;
     * equivalent to *Math.asin*.
     * @param {number} value The input number value.
     * @return {number} The arc-sine value.
     */
    asin: Math.asin,

    /**
     * Returns the hyperbolic arc-sine of a number *value*;
     * equivalent to *Math.asinh*.
     * @param {number} value The input number value.
     * @return {number} The hyperbolic arc-sine value.
     */
    asinh: Math.asinh,

    /**
     * Returns the arc-tangent (in radians) of a number *value*;
     * equivalent to *Math.atan*.
     * @param {number} value The input number value.
     * @return {number} The arc-tangent value.
     */
    atan: Math.atan,

    /**
     * Returns the angle in the plane (in radians) between the positive x-axis
     * and the ray from (0, 0) to the point (*x*, *y*);
     * equivalent to *Math.atan2*.
     * @param {number} y The y coordinate of the point.
     * @param {number} x The x coordinate of the point.
     * @return {number} The arc-tangent angle.
     */
    atan2: Math.atan2,

    /**
     * Returns the hyperbolic arc-tangent of a number *value*;
     * equivalent to *Math.atanh*.
     * @param {number} value The input number value.
     * @return {number} The hyperbolic arc-tangent value.
     */
    atanh: Math.atanh,

    /**
     * Returns the cosine (in radians) of a number *value*;
     * equivalent to *Math.cos*.
     * @param {number} value The input number value.
     * @return {number} The cosine value.
     */
    cos: Math.cos,

    /**
     * Returns the hyperbolic cosine (in radians) of a number *value*;
     * equivalent to *Math.cosh*.
     * @param {number} value The input number value.
     * @return {number} The hyperbolic cosine value.
     */
    cosh: Math.cosh,

    /**
     * Returns the sine (in radians) of a number *value*;
     * equivalent to *Math.sin*.
     * @param {number} value The input number value.
     * @return {number} The sine value.
     */
    sin: Math.sin,

    /**
     * Returns the hyperbolic sine (in radians) of a number *value*;
     * equivalent to *Math.sinh*.
     * @param {number} value The input number value.
     * @return {number} The hyperbolic sine value.
     */
    sinh: Math.sinh,

    /**
     * Returns the tangent (in radians) of a number *value*;
     * equivalent to *Math.tan*.
     * @param {number} value The input number value.
     * @return {number} The tangent value.
     */
    tan: Math.tan,

    /**
     * Returns the hyperbolic tangent (in radians) of a number *value*;
     * equivalent to *Math.tanh*.
     * @param {number} value The input number value.
     * @return {number} The hyperbolic tangent value.
     */
    tanh: Math.tanh
  };

  var has = Object.hasOwn;

  /**
   * @param {*} value
   * @return {value is Map}
   */
  function isMap(value) {
    return value instanceof Map;
  }

  /**
   * @param {*} value
   * @return {value is Set}
   */
  function isSet(value) {
    return value instanceof Set;
  }

  /**
   * @param {*} value
   * @return {value is Map | Set}
   */
  function isMapOrSet(value) {
    return isMap(value) || isSet(value);
  }

  function array(iter) {
    return Array.from(iter);
  }

  var object = {
    /**
     * Returns a boolean indicating whether the *object* has the specified *key*
     * as its own property (as opposed to inheriting it). If the *object* is a
     * *Map* or *Set* instance, the *has* method will be invoked directly on the
     * object, otherwise *Object.hasOwnProperty* is used.
     * @template K, V
     * @param {Map<K, V>|Set<K>|Record<K, V>} object The object, Map, or Set to
     *  test for property membership.
     * @param {K} key The property key to test for.
     * @return {boolean} True if the object has the given key, false otherwise.
     */
    has: (object, key) => isMapOrSet(object) ? object.has(key)
      : object != null ? has(object, `${key}`)
      : false,

    /**
     * Returns an array of a given *object*'s own enumerable property names. If
     * the *object* is a *Map* instance, the *keys* method will be invoked
     * directly on the object, otherwise *Object.keys* is used.
     * @template K, V
     * @param {Map<K, V>|Record<K, V>} object The input object or Map value.
     * @return {K[]} An array of property key name strings.
     */
    keys: (object) => isMap(object) ? array(object.keys())
      : object != null ? Object.keys(object)
      : [],

    /**
     * Returns an array of a given *object*'s own enumerable property values. If
     * the *object* is a *Map* or *Set* instance, the *values* method will be
     * invoked directly on the object, otherwise *Object.values* is used.
     * @template K, V
     * @param {Map<K, V>|Set<V>|Record<K, V>} object The input
     *  object, Map, or Set value.
     * @return {V[]} An array of property values.
     */
    values: (object) => isMapOrSet(object) ? array(object.values())
      : object != null ? Object.values(object)
      : [],

    /**
     * Returns an array of a given *object*'s own enumerable keyed property
     * `[key, value]` pairs. If the *object* is a *Map* or *Set* instance, the
     * *entries* method will be invoked directly on the object, otherwise
     * *Object.entries* is used.
     * @template K, V
     * @param {Map<K, V>|Set<V>|Record<K, V>} object The input
     *  object, Map, or Set value.
     * @return {[K, V][]} An array of property values.
     */
    entries: (object) => isMapOrSet(object) ? array(object.entries())
      : object != null ? Object.entries(object)
      : [],

    /**
     * Returns a new object given iterable *entries* of `[key, value]` pairs.
     * This method is Arquero's version of the *Object.fromEntries* method.
     * @template K, V
     * @param {Iterable<[K, V]>} entries An iterable collection of `[key, value]`
     *  pairs, such as an array of two-element arrays or a *Map*.
     * @return {Record<K, V>} An object of consolidated key-value pairs.
     */
    object: (entries) => entries ? Object.fromEntries(entries) : NULL
  };

  /**
   * Recodes an input value to an alternative value, based on a provided
   * value map. If a fallback value is specified, it will be returned when
   * a matching value is not found in the map; otherwise, the input value
   * is returned unchanged.
   * @template T
   * @param {T} value The value to recode. The value must be safely
   *  coercible to a string for lookup against the value map.
   * @param {Map|Record<string,any>} map An object or Map with input values
   *  for keys and output recoded values as values. If a non-Map object, only
   *  the object's own properties will be considered.
   * @param {T} [fallback] A default fallback value to use if the input
   *  value is not found in the value map.
   * @return {T} The recoded value.
   */
  function recode(value, map, fallback) {
    if (map instanceof Map) {
      if (map.has(value)) return map.get(value);
    } else {
      const key = `${value}`;
      if (has(map, key)) return map[key];
    }
    return fallback !== undefined ? fallback : value;
  }

  /**
   * Returns an array containing an arithmetic sequence from the start value
   * to the stop value, in step increments. If step is positive, the last
   * element is the largest start + i * step less than stop; if step is
   * negative, the last element is the smallest start + i * step greater
   * than stop. If the returned array would contain an infinite number of
   * values, an empty range is returned.
   * @param {number} [start=0] The starting value of the sequence.
   * @param {number} [stop] The stopping value of the sequence.
   *  The stop value is exclusive; it is not included in the result.
   * @param {number} [step=1] The step increment between sequence values.
   * @return {number[]} The generated sequence.
   */
  function sequence(start, stop, step) {
    let n = arguments.length;
    start = +start;
    stop = +stop;
    step = n < 2
      ? (stop = start, start = 0, 1)
      : n < 3 ? 1 : +step;

    n = Math.max(0, Math.ceil((stop - start) / step)) | 0;
    const seq = new Array(n);

    for (let i = 0; i < n; ++i) {
      seq[i] = start + i * step;
    }

    return seq;
  }

  var string = {
    /**
     * Parses a string *value* and returns a Date instance. Beware: this method
     * uses JavaScript's *Date.parse()* functionality, which is inconsistently
     * implemented across browsers. That said,
     * [ISO 8601](https://en.wikipedia.org/wiki/ISO_8601) formatted strings such
     * as those produced by *op.format_date* and *op.format_utcdate* should be
     * supported across platforms. Note that "bare" ISO date strings such as
     * `"2001-01-01"` are interpreted by JavaScript as indicating midnight of
     * that day in Coordinated Universal Time (UTC), *not* local time. To
     * indicate the local timezone, an ISO string can include additional time
     * components and no `Z` suffix: `"2001-01-01T00:00"`.
     * @param {*} value The input value.
     * @return {Date} The parsed date value.
     */
    parse_date: (value) => value == null ? value : new Date(value),

    /**
     * Parses a string *value* and returns a floating point number.
     * @param {*} value The input value.
     * @return {number} The parsed number value.
     */
    parse_float: (value) => value == null ? value : Number.parseFloat(value),

    /**
     * Parses a string *value* and returns an integer of the specified radix
     * (the base in mathematical numeral systems).
     * @param {*} value The input value.
     * @param {number} [radix] An integer between 2 and 36 that represents the
     *  radix (the base in mathematical numeral systems) of the string. Be
     *  careful: this does not default to 10! If *radix* is `undefined`, `0`,
     *  or unspecified, JavaScript assumes the following: If the input string
     *  begins with `"0x"` or `"0X"` (a zero, followed by lowercase or
     *  uppercase X), the radix is assumed to be 16 and the rest of the string
     *  is parsed as a hexidecimal number. If the input string begins with `"0"`
     *  (a zero), the radix is assumed to be 8 (octal) or 10 (decimal). Exactly
     *  which radix is chosen is implementation-dependent.  If the input string
     *  begins with any other value, the radix is 10 (decimal).
     * @return {number} The parsed integer value.
     */
    parse_int: (value, radix) => value == null ? value
      : Number.parseInt(value, radix),

    /**
     * Determines whether a string *value* ends with the characters of a
     * specified *search* string, returning `true` or `false` as appropriate.
     * @param {any} value The input string value.
     * @param {string} search The search string to test for.
     * @param {number} [length] If provided, used as the length of *value*
     *  (default `value.length`).
     * @return {boolean} True if the value ends with the search string,
     *  false otherwise.
     */
    endswith: (value, search, length) => value == null ? false
      : String(value).endsWith(search, length),

    /**
     * Retrieves the result of matching a string *value* against a regular
     * expression *regexp*. If no *index* is specified, returns an array
     * whose contents depend on the presence or absence of the regular
     * expression global (`g`) flag, or `null` if no matches are found. If the
     * `g` flag is used, all results matching the complete regular expression
     * will be returned, but capturing groups will not. If the `g` flag is not
     * used, only the first complete match and its related capturing groups are
     * returned.
     *
     * If specified, the *index* looks up a value of the resulting match. If
     * *index* is a number, the corresponding index of the result array is
     * returned. If *index* is a string, the value of the corresponding
     * named capture group is returned, or `null` if there is no such group.
     * @param {*} value The input string value.
     * @param {*} regexp The regular expression to match against.
     * @param {number|string} index The index into the match result array
     *  or capture group.
     * @return {string|string[]} The match result.
     */
    match: (value, regexp, index) => {
      const m = value == null ? value : String(value).match(regexp);
      return index == null || m == null ? m
        : typeof index === 'number' ? m[index]
        : m.groups ? m.groups[index]
        : null;
    },

    /**
     * Returns the Unicode normalization form of the string *value*.
     * @param {*} value The input value to normalize.
     * @param {string} form The Unicode normalization form, one of
     *  `'NFC'` (default, canonical decomposition, followed by canonical
     *  composition), `'NFD'` (canonical decomposition), `'NFKC'` (compatibility
     *  decomposition, followed by canonical composition),
     *  or `'NFKD'` (compatibility decomposition).
     * @return {string} The normalized string value.
     */
    normalize: (value, form) => value == null ? value
      : String(value).normalize(form),

    /**
     * Pad a string *value* with a given *fill* string (applied from the end of
     * *value* and repeated, if needed) so that the resulting string reaches a
     * given *length*.
     * @param {*} value The input value to pad.
     * @param {number} length The length of the resulting string once the
     *  *value* string has been padded. If the length is lower than
     *  `value.length`, the *value* string will be returned as-is.
     * @param {string} [fill] The string to pad the *value* string with
     *  (default `''`). If *fill* is too long to stay within the target
     *  *length*, it will be truncated: for left-to-right languages the
     *  left-most part and for right-to-left languages the right-most will
     *  be applied.
     * @return {string} The padded string.
     */
    padend: (value, length, fill) => value == null ? value
      : String(value).padEnd(length, fill),

    /**
     * Pad a string *value* with a given *fill* string (applied from the start
     * of *value* and repeated, if needed) so that the resulting string reaches
     * a given *length*.
     * @param {*} value The input value to pad.
     * @param {number} length The length of the resulting string once the
     *  *value* string has been padded. If the length is lower than
     *  `value.length`, the *value* string will be returned as-is.
     * @param {string} [fill] The string to pad the *value* string with
     *  (default `''`). If *fill* is too long to stay within the target
     *  *length*, it will be truncated: for left-to-right languages the
     *  left-most part and for right-to-left languages the right-most will
     *  be applied.
     * @return {string} The padded string.
     */
    padstart: (value, length, fill) => value == null ? value
      : String(value).padStart(length, fill),

    /**
     * Returns the string *value* converted to upper case.
     * @param {*} value The input string value.
     * @return {string} The upper case string.
     */
    upper: (value) => value == null ? value : String(value).toUpperCase(),

    /**
     * Returns the string *value* converted to lower case.
     * @param {*} value The input string value.
     * @return {string} The lower case string.
     */
    lower: (value) => value == null ? value : String(value).toLowerCase(),

    /**
     * Returns a new string which contains the specified *number* of copies of
     * the *value* string concatenated together.
     * @param {*} value The input string to repeat.
     * @param {*} number An integer between `0` and `+Infinity`, indicating the
     *  number of times to repeat the string.
     * @return {string} The repeated string.
     */
    repeat: (value, number) => value == null ? value
      : String(value).repeat(number),

    /**
     * Returns a new string with some or all matches of a *pattern* replaced by
     * a *replacement*. The *pattern* can be a string or a regular expression,
     * and the *replacement* must be a string. If *pattern* is a string, only
     * the first occurrence will be replaced; to make multiple replacements, use
     * a regular expression *pattern* with a `g` (global) flag.
     * @param {*} value The input string value.
     * @param {*} pattern The pattern string or regular expression to replace.
     * @param {*} replacement The replacement string to use.
     * @return {string} The string with patterns replaced.
     */
    replace: (value, pattern, replacement) => value == null ? value
      : String(value).replace(pattern, String(replacement)),

    /**
     * Divides a string *value* into an ordered list of substrings based on a
     * *separator* pattern, puts these substrings into an array, and returns the
     * array.
     * @param {*} value The input string value.
     * @param {*} separator A string or regular expression pattern describing
     *  where each split should occur.
     * @param {number} [limit] An integer specifying a limit on the number of
     *  substrings to be included in the array.
     * @return {string[]}
     */
    split: (value, separator, limit) => value == null ? []
      : String(value).split(separator, limit),

    /**
     * Determines whether a string *value* starts with the characters of a
     * specified *search* string, returning `true` or `false` as appropriate.
     * @param {*} value The input string value.
     * @param {string} search The search string to test for.
     * @param {number} [position=0] The position in the *value* string at which
     *  to begin searching (default `0`).
     * @return {boolean} True if the string starts with the search pattern,
     *  false otherwise.
     */
    startswith: (value, search, position) => value == null ? false
      : String(value).startsWith(search, position),

    /**
     * Returns the part of the string *value* between the *start* and *end*
     * indexes, or to the end of the string.
     * @param {*} value The input string value.
     * @param {number} [start=0] The index of the first character to include in
     *  the returned substring (default `0`).
     * @param {number} [end] The index of the first character to exclude from
     *  the returned substring (default `value.length`).
     * @return {string} The substring.
     */
    substring: (value, start, end) => value == null ? value
      : String(value).substring(start, end),

    /**
     * Returns a new string with whitespace removed from both ends of the input
     * *value* string. Whitespace in this context is all the whitespace
     * characters (space, tab, no-break space, etc.) and all the line terminator
     * characters (LF, CR, etc.).
     * @param {*} value The input string value to trim.
     * @return {string} The trimmed string.
     */
    trim: (value) => value == null ? value : String(value).trim()
  };

  var functions = {
    bin: bin$1,
    equal,
    recode,
    sequence,
    ...array$1,
    ...date$1,
    ...json,
    ...math,
    ...object,
    ...string
  };

  function toArray(value) {
    return value != null
      ? (isArray$2(value) ? value : [value])
      : [];
  }

  function isBigInt(value) {
    return typeof value === 'bigint';
  }

  function toString$1(v) {
    return v === undefined ? v + ''
      : isBigInt(v) ? v + 'n'
      : JSON.stringify(v);
  }

  let Op$1 = class Op {
    constructor(name, fields, params) {
      this.name = name;
      this.fields = fields;
      this.params = params;
    }
    toString() {
      const args = [
        ...this.fields.map(f => `d[${toString$1(f)}]`),
        ...this.params.map(toString$1)
      ];
      return `d => op.${this.name}(${args})`;
    }
    toObject() {
      return { expr: this.toString(), func: true };
    }
  };

  /**
   * @param {string} name
   * @param {any | any[]} [fields]
   * @param {any | any[]} [params]
   */
  function op(name, fields = [], params = []) {
    return new Op$1(name, toArray(fields), toArray(params));
  }

  const any = (field) => op('any', field);
  const count = () => op('count');
  const array_agg = (field) => op('array_agg', field);
  const array_agg_distinct = (field) => op('array_agg_distinct', field);
  const map_agg = (key, value) => op('map_agg', [key, value]);
  const object_agg = (key, value) => op('object_agg', [key, value]);
  const entries_agg = (key, value) => op('entries_agg', [key, value]);

  /**
   * @typedef {import('../table/types.js').Struct} Struct
   */

  /**
   * All table expression operations including normal functions,
   * aggregate functions, and window functions.
   */
  var ops = {
    ...functions,

    /**
     * Generate an object representing the current table row.
     * @param {...string} names The column names to include in the object.
     *  If unspecified, all columns are included.
     * @return {Struct} The generated row object.
     */
    row_object: (...names) => op('row_object', null, names.flat()),

    /**
     * Aggregate function to count the number of records (rows).
     * @returns {number} The count of records.
     */
    count,

    /**
     * Aggregate function returning an arbitrary observed value.
     * @template T
     * @param {T} field The data field.
     * @return {T} An arbitrary observed value.
     */
    any,

    /**
     * Aggregate function to collect an array of values.
     * @template T
     * @param {T} field The data field.
     * @return {Array<T>} A list of values.
     */
    array_agg,

    /**
     * Aggregate function to collect an array of distinct (unique) values.
     * @template T
     * @param {T} field The data field.
     * @return {Array<T>} An array of unique values.
     */
    array_agg_distinct,

    /**
     * Aggregate function to create an object given input key and value fields.
     * @template K, V
     * @param {K} key The object key field.
     * @param {V} value The object value field.
     * @return {Record<K, V>} An object of key-value pairs.
     */
    object_agg,

    /**
     * Aggregate function to create a Map given input key and value fields.
     * @template K, V
     * @param {K} key The object key field.
     * @param {V} value The object value field.
     * @return {Map<K, V>} A Map of key-value pairs.
     */
    map_agg,

    /**
     * Aggregate function to create an array in the style of Object.entries()
     * given input key and value fields.
     * @template K, V
     * @param {K} key The object key field.
     * @param {V} value The object value field.
     * @return {[K, V][]} An array of [key, value] arrays.
     */
    entries_agg,

    /**
     * Aggregate function to count the number of valid values.
     * Invalid values are null, undefined, or NaN.
     * @param {*} field The data field.
     * @return {number} The count of valid values.
     */
    // @ts-ignore
    valid: (field) => op('valid', field),

    /**
     * Aggregate function to count the number of invalid values.
     * Invalid values are null, undefined, or NaN.
     * @param {*} field The data field.
     * @return {number} The count of invalid values.
     */
    // @ts-ignore
    invalid: (field) => op('invalid', field),

    /**
     * Aggregate function to count the number of distinct values.
     * @param {*} field The data field.
     * @return {number} The count of distinct values.
     */
    // @ts-ignore
    distinct: (field) => op('distinct', field),

    /**
     * Aggregate function to determine the mode (most frequent) value.
     * @template T
     * @param {T} field The data field.
     * @return {T} The mode value.
     */
    // @ts-ignore
    mode: (field) => op('mode', field),

    /**
     * Aggregate function to sum values.
     * @param {*} field The data field.
     * @return {number} The sum of the values.
     */
    // @ts-ignore
    sum: (field) => op('sum', field),

    /**
     * Aggregate function to multiply values.
     * @param {*} field The data field.
     * @return {number} The product of the values.
     */
    // @ts-ignore
    product: (field) => op('product', field),

    /**
     * Aggregate function for the mean (average) value.
     * @param {*} field The data field.
     * @return {number} The mean (average) of the values.
     */
    // @ts-ignore
    mean: (field) => op('mean', field),

    /**
     * Aggregate function for the average (mean) value.
     * @param {*} field The data field.
     * @return {number} The average (mean) of the values.
     */
    // @ts-ignore
    average: (field) => op('average', field),

    /**
     * Aggregate function for the sample variance.
     * @param {*} field The data field.
     * @return {number} The sample variance of the values.
     */
    // @ts-ignore
    variance: (field) => op('variance', field),

    /**
     * Aggregate function for the population variance.
     * @param {*} field The data field.
     * @return {number} The population variance of the values.
     */
    // @ts-ignore
    variancep: (field) => op('variancep', field),

    /**
     * Aggregate function for the sample standard deviation.
     * @param {*} field The data field.
     * @return {number} The sample standard deviation of the values.
     */
    // @ts-ignore
    stdev: (field) => op('stdev', field),

    /**
     * Aggregate function for the population standard deviation.
     * @param {*} field The data field.
     * @return {number} The population standard deviation of the values.
     */
    // @ts-ignore
    stdevp: (field) => op('stdevp', field),

    /**
     * Aggregate function for the minimum value.
     * @template T
     * @param {T} field The data field.
     * @return {T} The minimum value.
     */
    // @ts-ignore
    min: (field) => op('min', field),

    /**
     * Aggregate function for the maximum value.
     * @template T
     * @param {T} field The data field.
     * @return {T} The maximum value.
     */
    // @ts-ignore
    max: (field) => op('max', field),

    /**
     * Aggregate function to compute the quantile boundary
     * of a data field for a probability threshold.
     * @param {*} field The data field.
     * @param {number} p The probability threshold.
     * @return {number} The quantile value.
     */
    // @ts-ignore
    quantile: (field, p) => op('quantile', field, p),

    /**
     * Aggregate function for the median value.
     * This is a shorthand for the 0.5 quantile value.
     * @param {*} field The data field.
     * @return {number} The median value.
     */
    // @ts-ignore
    median: (field) => op('median', field),

    /**
     * Aggregate function for the sample covariance between two variables.
     * @param {*} field1 The first data field.
     * @param {*} field2 The second data field.
     * @return {number} The sample covariance of the values.
     */
    // @ts-ignore
    covariance: (field1, field2) => op('covariance', [field1, field2]),

    /**
     * Aggregate function for the population covariance between two variables.
     * @param {*} field1 The first data field.
     * @param {*} field2 The second data field.
     * @return {number} The population covariance of the values.
     */
    // @ts-ignore
    covariancep: (field1, field2) => op('covariancep', [field1, field2]),

    /**
     * Aggregate function for the product-moment correlation between two variables.
     * To instead compute a rank correlation, compute the average ranks for each
     * variable and then apply this function to the result.
     * @param {*} field1 The first data field.
     * @param {*} field2 The second data field.
     * @return {number} The correlation between the field values.
     */
    // @ts-ignore
    corr: (field1, field2) => op('corr', [field1, field2]),

    /**
     * Aggregate function for calculating a binning scheme in terms of
     * the minimum bin boundary, maximum bin boundary, and step size.
     * @param {*} field The data field.
     * @param {number} [maxbins=15] The maximum number of allowed bins.
     * @param {boolean} [nice=true] Flag indicating if the bin min and max
     *  should snap to "nice" human-friendly values.
     * @param {number} [minstep] The minimum allowed step size between bins.
     * @param {number} [step] The exact step size to use between bins.
     *  If specified, the maxbins and minstep arguments are ignored.
     * @return {[number, number, number]} The bin [min, max, and step] values.
     */
    // @ts-ignore
    bins: (field, maxbins, nice, minstep, step) => op(
      'bins',
      field,
      [maxbins, nice, minstep, step]
    ),

    /**
     * Window function to assign consecutive row numbers, starting from 1.
     * @return {number} The row number value.
     */
    // @ts-ignore
    row_number: () => op('row_number'),

    /**
     * Window function to assign a rank to each value in a group, starting
     * from 1. Peer values are assigned the same rank. Subsequent ranks
     * reflect the number of prior values: if the first two values tie for
     * rank 1, the third value is assigned rank 3.
     * @return {number} The rank value.
     */
    // @ts-ignore
    rank: () => op('rank'),

    /**
     * Window function to assign a fractional (average) rank to each value in
     * a group, starting from 1. Peer values are assigned the average of their
     * indices: if the first two values tie, both will be assigned rank 1.5.
     * @return {number} The peer-averaged rank value.
     */
    // @ts-ignore
    avg_rank: () => op('avg_rank'),

    /**
     * Window function to assign a dense rank to each value in a group,
     * starting from 1. Peer values are assigned the same rank. Subsequent
     * ranks do not reflect the number of prior values: if the first two
     * values tie for rank 1, the third value is assigned rank 2.
     * @return {number} The dense rank value.
     */
    // @ts-ignore
    dense_rank: () => op('dense_rank'),

    /**
     * Window function to assign a percentage rank to each value in a group.
     * The percent is calculated as (rank - 1) / (group_size - 1).
     * @return {number} The percentage rank value.
     */
    // @ts-ignore
    percent_rank: () => op('percent_rank'),

    /**
     * Window function to assign a cumulative distribution value between 0 and 1
     * to each value in a group.
     * @return {number} The cumulative distribution value.
     */
    // @ts-ignore
    cume_dist: () => op('cume_dist'),

    /**
     * Window function to assign a quantile (e.g., percentile) value to each
     * value in a group. Accepts an integer parameter indicating the number of
     * buckets to use (e.g., 100 for percentiles, 5 for quintiles).
     * @param {number} num The number of buckets for ntile calculation.
     * @return {number} The quantile value.
     */
    // @ts-ignore
    ntile: (num) => op('ntile', null, num),

    /**
     * Window function to assign a value that precedes the current value by
     * a specified number of positions. If no such value exists, returns a
     * default value instead.
     * @template T
     * @param {T} field The data field.
     * @param {number} [offset=1] The lag offset from the current value.
     * @param {T} [defaultValue=undefined] The default value.
     * @return {T} The lagging value.
     */
    // @ts-ignore
    lag: (field, offset, defaultValue) => op('lag', field, [offset, defaultValue]),

    /**
     * Window function to assign a value that follows the current value by
     * a specified number of positions. If no such value exists, returns a
     * default value instead.
     * @template T
     * @param {T} field The data field.
     * @param {number} [offset=1] The lead offset from the current value.
     * @param {T} [defaultValue=undefined] The default value.
     * @return {T} The leading value.
     */
    // @ts-ignore
    lead: (field, offset, defaultValue) => op('lead', field, [offset, defaultValue]),

    /**
     * Window function to assign the first value in a sliding window frame.
     * @template T
     * @param {T} field The data field.
     * @return {T} The first value in the current frame.
     */
    // @ts-ignore
    first_value: (field) => op('first_value', field),

    /**
     * Window function to assign the last value in a sliding window frame.
     * @template T
     * @param {T} field The data field.
     * @return {T} The last value in the current frame.
     */
    // @ts-ignore
    last_value: (field) => op('last_value', field),

    /**
     * Window function to assign the nth value in a sliding window frame
     * (counting from 1), or undefined if no such value exists.
     * @template T
     * @param {T} field The data field.
     * @param {number} nth The nth position, starting from 1.
     * @return {T} The nth value in the current frame.
     */
    // @ts-ignore
    nth_value: (field, nth) => op('nth_value', field, nth),

    /**
     * Window function to fill in missing values with preceding values.
     * @template T
     * @param {T} field The data field.
     * @param {T} [defaultValue=undefined] The default value.
     * @return {T} The current value if valid, otherwise the first preceding
     *  valid value. If no such value exists, returns the default value.
     */
    // @ts-ignore
    fill_down: (field, defaultValue) => op('fill_down', field, defaultValue),

    /**
     * Window function to fill in missing values with subsequent values.
     * @template T
     * @param {T} field The data field.
     * @param {T} [defaultValue=undefined] The default value.
     * @return {T} The current value if valid, otherwise the first subsequent
     *  valid value. If no such value exists, returns the default value.
     */
    // @ts-ignore
    fill_up: (field, defaultValue) => op('fill_up', field, defaultValue)
  };

  function error(message, cause) {
    // @ts-ignore
    throw Error(message, { cause });
  }

  function uniqueName(names, name) {
    names = isMapOrSet(names) ? names : new Set(names);
    let uname = name;
    let index = 0;

    while (names.has(uname)) {
      uname = name + ++index;
    }

    return uname;
  }

  /**
   * Abstract class for custom aggregation operations.
   */
  class Reducer {
    constructor(outputs) {
      this._outputs = outputs;
    }

    size() {
      return this._outputs.length;
    }

    outputs() {
      return this._outputs;
    }

    // eslint-disable-next-line no-unused-vars
    init(columns) {
      return {};
    }

    // eslint-disable-next-line no-unused-vars
    add(state, row, data) {
      // no-op, subclasses should override
    }

    // eslint-disable-next-line no-unused-vars
    rem(state, row, data) {
      // no-op, subclasses should override
    }

    // eslint-disable-next-line no-unused-vars
    write(state, values, index) {
    }
  }

  function bins(min, max, maxbins = 15, nice = true, minstep = 0, step) {
    const base = 10;
    const logb = Math.LN10;

    if (step == null) {
      const level = Math.ceil(Math.log(maxbins) / logb);
      const span = (max - min) || Math.abs(min) || 1;
      const div = [5, 2];

      step = Math.max(
        minstep,
        Math.pow(base, Math.round(Math.log(span) / logb) - level)
      );

      // increase step size if too many bins
      while (Math.ceil(span / step) > maxbins) {
        step *= base;
      }

      // decrease step size if it stays within maxbins
      const n = div.length;
      for (let i = 0; i < n; ++i) {
        const v = step / div[i];
        if (v >= minstep && span / v <= maxbins) {
          step = v;
        }
      }
    }

    // snap to "nice" boundaries
    if (nice) {
      let v = Math.log(step);
      const precision = v >= 0 ? 0 : ~~(-v / logb) + 1;
      const eps = Math.pow(base, -precision - 1);
      v = Math.floor(min / step + eps) * step;
      min = min < v ? v - step : v;
      max = Math.ceil(max / step) * step;
    }

    return [
      min,
      max === min ? min + step : max,
      step
    ];
  }

  function key(value) {
    const type = typeof value;
    return type === 'string' ? `"${value}"`
      : type !== 'object' || !value ? value
      : isDate$1(value) ? +value
      : isArray$2(value) || isTypedArray$1(value) ? `[${value.map(key)}]`
      : isRegExp(value) ? value + ''
      : objectKey$1(value);
  }

  function objectKey$1(value) {
    let s = '{';
    let i = -1;
    for (const k in value) {
      if (++i > 0) s += ',';
      s += `"${k}":${key(value[k])}`;
    }
    s += '}';
    return s;
  }

  function keyFunction(get, nulls) {
    const n = get.length;
    return n === 1
      ? (row, data) => key(get[0](row, data))
      : (row, data) => {
          let s = '';
          for (let i = 0; i < n; ++i) {
            if (i > 0) s += '|';
            const v = get[i](row, data);
            if (nulls && (v == null || v !== v)) return null;
            s += key(v);
          }
          return s;
        };
  }

  function distinctMap() {
    const map = new Map();
    return {
      count() {
        return map.size;
      },
      values() {
        return Array.from(map.values(), _ => _.v);
      },
      increment(v) {
        const k = key(v);
        const e = map.get(k);
        e ? ++e.n : map.set(k, { v, n: 1 });
      },
      decrement(v) {
        const k = key(v);
        const e = map.get(k);
        e.n === 1 ? map.delete(k) : --e.n;
      },
      forEach(fn) {
        map.forEach(({ v, n }) => fn(v, n));
      }
    };
  }

  function noop() {}

  function product(values, start = 0, stop = values.length) {
    let prod = values[start++];

    for (let i = start; i < stop; ++i) {
      prod *= values[i];
    }

    return prod;
  }

  /**
   * Initialize an aggregate operator.
   */
  function initOp(op) {
    op.init = op.init || noop;
    op.add = op.add || noop;
    op.rem = op.rem || noop;
    return op;
  }

  function initProduct(s, value) {
    s.product_v = false;
    return s.product = value;
  }

  /**
   * Initialize an aggregate operator.
   * @callback AggregateInit
   * @param {object} state The aggregate state object.
   * @return {void}
   */

  /**
   * Add a value to an aggregate operator.
   * @callback AggregateAdd
   * @param {object} state The aggregate state object.
   * @param {*} value The value to add.
   * @return {void}
   */

  /**
   * Remove a value from an aggregate operator.
   * @callback AggregateRem
   * @param {object} state The aggregate state object.
   * @param {*} value The value to remove.
   * @return {void}
   */

  /**
   * Retrive an output value from an aggregate operator.
   * @callback AggregateValue
   * @param {object} state The aggregate state object.
   * @return {*} The output value.
   */

  /**
   * An operator instance for an aggregate function.
   * @typedef {object} AggregateOperator
   * @property {AggregateInit} init Initialize the operator.
   * @property {AggregateAdd} [add] Add a value to the operator state.
   * @property {AggregateRem} [rem] Remove a value from the operator state.
   * @property {AggregateValue} value Retrieve an output value.
   */

  /**
   * Create a new aggregate operator instance.
   * @callback AggregateCreate
   * @param {...any} params The aggregate operator parameters.
   * @return {AggregateOperator} The instantiated aggregate operator.
   */

  /**
   * An operator definition for an aggregate function.
   * @typedef {object} AggregateDef
   * @property {AggregateCreate} create Create a new operator instance.
   * @property {number[]} param Two-element array containing the
   *  counts of input fields and additional parameters.
   * @property {string[]} [req] Names of operators required by this one.
   * @property {string[]} [stream] Names of operators required by this one
   *  for streaming operations (value removes).
   */

  /**
   * Aggregate operator definitions.
   */
  var aggregateFunctions = {
    /** @type {AggregateDef} */
    count: {
      create: () => initOp({
        value: s => s.count
      }),
      param: []
    },

    /** @type {AggregateDef} */
    array_agg: {
      create: () => initOp({
        init: s => s.values = true,
        value: s => s.list.values(s.stream)
      }),
      param: [1]
    },

    /** @type {AggregateDef} */
    object_agg: {
      create: () => initOp({
        init:  s => s.values = true,
        value: s => Object.fromEntries(s.list.values())
      }),
      param: [2]
    },

    /** @type {AggregateDef} */
    map_agg: {
      create: () => initOp({
        init:  s => s.values = true,
        value: s => new Map(s.list.values())
      }),
      param: [2]
    },

    /** @type {AggregateDef} */
    entries_agg: {
      create: () => initOp({
        init:  s => s.values = true,
        value: s => s.list.values(s.stream)
      }),
      param: [2]
    },

    /** @type {AggregateDef} */
    any: {
      create: () => initOp({
        add: (s, v) => { if (s.any == null) s.any = v; },
        value: s => s.valid ? s.any : NULL
      }),
      param: [1]
    },

    /** @type {AggregateDef} */
    valid: {
      create: () => initOp({
        value: s => s.valid
      }),
      param: [1]
    },

    /** @type {AggregateDef} */
    invalid: {
      create: () => initOp({
        value: s => s.count - s.valid
      }),
      param: [1]
    },

    /** @type {AggregateDef} */
    distinct: {
      create: () => ({
        init: s => s.distinct = distinctMap(),
        value: s => s.distinct.count() + (s.valid === s.count ? 0 : 1),
        add: (s, v) => s.distinct.increment(v),
        rem: (s, v) => s.distinct.decrement(v)
      }),
      param: [1]
    },

    /** @type {AggregateDef} */
    array_agg_distinct: {
      create: () => initOp({
        value: s => s.distinct.values()
      }),
      param: [1],
      req: ['distinct']
    },

    /** @type {AggregateDef} */
    mode: {
      create: () => initOp({
        value: s => {
          let mode = NULL;
          let max = 0;
          s.distinct.forEach((value, count) => {
            if (count > max) {
              max = count;
              mode = value;
            }
          });
          return mode;
        }
      }),
      param: [1],
      req: ['distinct']
    },

    /** @type {AggregateDef} */
    sum: {
      create: () => ({
        init:  s => s.sum = 0,
        value: s => s.valid ? s.sum : NULL,
        add: (s, v) => isBigInt(v)
          ? (s.sum === 0 ? s.sum = v : s.sum += v)
          : s.sum += +v,
        rem: (s, v) => s.sum -= v
      }),
      param: [1]
    },

    /** @type {AggregateDef} */
    product: {
      create: () => ({
        init:  s => initProduct(s, 1),
        value: s => s.valid
          ? (
              s.product_v
                ? initProduct(s, product(s.list.values()))
                : s.product
            )
          : undefined,
        add: (s, v) => isBigInt(v)
          ? (s.product === 1 ? s.product = v : s.product *= v)
          : s.product *= v,
        rem: (s, v) => (v == 0 || v === Infinity || v === -Infinity)
          ? s.product_v = true
          : s.product /= v
      }),
      param: [1],
      stream: ['array_agg']
    },

    /** @type {AggregateDef} */
    mean: {
      create: () => ({
        init: s => s.mean = 0,
        value: s => s.valid ? s.mean : NULL,
        add: (s, v) => {
          s.mean_d = v - s.mean;
          s.mean += s.mean_d / s.valid;
        },
        rem: (s, v) => {
          s.mean_d = v - s.mean;
          s.mean -= s.valid ? s.mean_d / s.valid : s.mean;
        }
      }),
      param: [1]
    },

    /** @type {AggregateDef} */
    average: {
      create: () => initOp({
        value: s => s.valid ? s.mean : NULL
      }),
      param: [1],
      req: ['mean']
    },

    /** @type {AggregateDef} */
    variance: {
      create: () => ({
        init:  s => s.dev = 0,
        value: s => s.valid > 1 ? s.dev / (s.valid - 1) : NULL,
        add: (s, v) => s.dev += s.mean_d * (v - s.mean),
        rem: (s, v) => s.dev -= s.mean_d * (v - s.mean)
      }),
      param: [1],
      req: ['mean']
    },

    /** @type {AggregateDef} */
    variancep: {
      create: () => initOp({
        value: s => s.valid > 1 ? s.dev / s.valid : NULL
      }),
      param: [1],
      req: ['variance']
    },

    /** @type {AggregateDef} */
    stdev: {
      create: () => initOp({
        value: s => s.valid > 1 ? Math.sqrt(s.dev / (s.valid - 1)) : NULL
      }),
      param: [1],
      req: ['variance']
    },

    /** @type {AggregateDef} */
    stdevp: {
      create: () => initOp({
        value: s => s.valid > 1 ? Math.sqrt(s.dev / s.valid) : NULL
      }),
      param: [1],
      req: ['variance']
    },

    /** @type {AggregateDef} */
    min: {
      create: () => ({
        init:  s => s.min = NULL,
        value: s => s.min = (Number.isNaN(s.min) ? s.list.min() : s.min),
        add: (s, v) => { if (v < s.min || s.min === NULL) s.min = v; },
        rem: (s, v) => { if (v <= s.min) s.min = NaN; }
      }),
      param: [1],
      stream: ['array_agg']
    },

    /** @type {AggregateDef} */
    max: {
      create: () => ({
        init:  s => s.max = NULL,
        value: s => s.max = (Number.isNaN(s.max) ? s.list.max() : s.max),
        add: (s, v) => { if (v > s.max || s.max === NULL) s.max = v; },
        rem: (s, v) => { if (v >= s.max) s.max = NaN; }
      }),
      param: [1],
      stream: ['array_agg']
    },

    /** @type {AggregateDef} */
    quantile: {
      create: (p) => initOp({
        value: s => s.list.quantile(p)
      }),
      param: [1, 1],
      req: ['array_agg']
    },

    /** @type {AggregateDef} */
    median: {
      create: () => initOp({
        value: s => s.list.quantile(0.5)
      }),
      param: [1],
      req: ['array_agg']
    },

    /** @type {AggregateDef} */
    covariance: {
      create: () => ({
        init:  s => {
          s.cov = s.mean_x = s.mean_y = s.dev_x = s.dev_y = 0;
        },
        value: s => s.valid > 1 ? s.cov / (s.valid - 1) : NULL,
        add: (s, x, y) => {
          const dx = x - s.mean_x;
          const dy = y - s.mean_y;
          s.mean_x += dx / s.valid;
          s.mean_y += dy / s.valid;
          const dy2 = y - s.mean_y;
          s.dev_x += dx * (x - s.mean_x);
          s.dev_y += dy * dy2;
          s.cov += dx * dy2;
        },
        rem: (s, x, y) => {
          const dx = x - s.mean_x;
          const dy = y - s.mean_y;
          s.mean_x -= s.valid ? dx / s.valid : s.mean_x;
          s.mean_y -= s.valid ? dy / s.valid : s.mean_y;
          const dy2 = y - s.mean_y;
          s.dev_x -= dx * (x - s.mean_x);
          s.dev_y -= dy * dy2;
          s.cov -= dx * dy2;
        }
      }),
      param: [2]
    },

    /** @type {AggregateDef} */
    covariancep: {
      create: () => initOp({
        value: s => s.valid > 1 ? s.cov / s.valid : NULL
      }),
      param: [2],
      req: ['covariance']
    },

    /** @type {AggregateDef} */
    corr: {
      create: () => initOp({
        value: s => s.valid > 1
          ? s.cov / (Math.sqrt(s.dev_x) * Math.sqrt(s.dev_y))
          : NULL
      }),
      param: [2],
      req: ['covariance']
    },

    /** @type {AggregateDef} */
    bins: {
      create: (maxbins, nice, minstep, step) => initOp({
        value: s => bins(s.min, s.max, maxbins, nice, minstep, step)
      }),
      param: [1, 4],
      req: ['min', 'max']
    }
  };

  /**
   * Initialize a window operator.
   * @callback WindowInit
   * @return {void}
   */

  /**
   * A storage object for the state of the window.
   * @typedef {import('../verbs/window/window-state.js').default} WindowState
   */

  /**
   * Retrieve an output value from a window operator.
   * @callback WindowValue
   * @param {WindowState} state The window state object.
   * @return {*} The output value.
   */

  /**
   * Initialize an aggregate operator.
   * @typedef {import('./aggregate-functions.js').AggregateInit} AggregateInit
   */

  /**
   * Retrive an output value from an aggregate operator.
   * @typedef {import('./aggregate-functions.js').AggregateValue} AggregateValue
   */

  /**
   * An operator instance for a window function.
   * @typedef {object} WindowOperator
   * @property {AggregateInit} init Initialize the operator.
   * @property {AggregateValue} value Retrieve an output value.
   */

  /**
   * Create a new window operator instance.
   * @callback WindowCreate
   * @param {...any} params The aggregate operator parameters.
   * @return {WindowOperator} The instantiated window operator.
   */

  /**
   * Create a new aggregate operator instance.
   * @typedef {import('./aggregate-functions.js').AggregateCreate} AggregateCreate
   */

  /**
   * An operator definition for a window function.
   * @typedef {object} WindowDef
   * @property {AggregateCreate} create Create a new operator instance.
   * @property {number[]} param Two-element array containing the
   *  counts of input fields and additional parameters.
   */

  const rank = {
    create() {
      let rank;
      return {
        init: () => rank = 1,
        value: w => {
          const i = w.index;
          return (i && !w.peer(i)) ? (rank = i + 1) : rank;
        }
      };
    },
    param: []
  };

  const cume_dist = {
    create() {
      let cume;
      return {
        init: () => cume = 0,
        value: w => {
          const { index, peer, size } = w;
          let i = index;
          if (cume < i) {
            while (i + 1 < size && peer(i + 1)) ++i;
            cume = i;
          }
          return (1 + cume) / size;
        }
      };
    },
    param: []
  };

  /**
   * Window operator definitions.
   */
  var windowFunctions = {
    /** @type {WindowDef} */
    row_number: {
      create() {
        return {
          init: noop,
          value: w => w.index + 1
        };
      },
      param: []
    },

    /** @type {WindowDef} */
    rank,

    /** @type {WindowDef} */
    avg_rank: {
      create() {
        let j, rank;
        return {
          init: () => (j = -1, rank = 1),
          value: w => {
            const i = w.index;
            if (i >= j) {
              for (rank = j = i + 1; w.peer(j); rank += ++j);
              rank /= (j - i);
            }
            return rank;
          }
        };
      },
      param: []
    },

    /** @type {WindowDef} */
    dense_rank: {
      create() {
        let drank;
        return {
          init: () => drank = 1,
          value: w => {
            const i = w.index;
            return (i && !w.peer(i)) ? ++drank : drank;
          }
        };
      },
      param: []
    },

    /** @type {WindowDef} */
    percent_rank: {
      create() {
        const { init, value } = rank.create();
        return {
          init,
          value: w => (value(w) - 1) / (w.size - 1)
        };
      },
      param: []
    },

    /** @type {WindowDef} */
    cume_dist,

    /** @type {WindowDef} */
    ntile: {
      create(num) {
        num = +num;
        if (!(num > 0)) error('ntile num must be greater than zero.');
        const { init, value } = cume_dist.create();
        return {
          init,
          value: w => Math.ceil(num * value(w))
        };
      },
      param: [0, 1]
    },

    /** @type {WindowDef} */
    lag: {
      create(offset, defaultValue = NULL) {
        offset = +offset || 1;
        return {
          init: noop,
          value: (w, f) => {
            const i = w.index - offset;
            return i >= 0 ? w.value(i, f) : defaultValue;
          }
        };
      },
      param: [1, 2]
    },

    /** @type {WindowDef} */
    lead: {
      create(offset, defaultValue = NULL) {
        offset = +offset || 1;
        return {
          init: noop,
          value: (w, f) => {
            const i = w.index + offset;
            return i < w.size ? w.value(i, f) : defaultValue;
          }
        };
      },
      param: [1, 2]
    },

    /** @type {WindowDef} */
    first_value: {
      create() {
        return {
          init: noop,
          value: (w, f) => w.value(w.i0, f)
        };
      },
      param: [1]
    },

    /** @type {WindowDef} */
    last_value: {
      create() {
        return {
          init: noop,
          value: (w, f) => w.value(w.i1 - 1, f)
        };
      },
      param: [1]
    },

    /** @type {WindowDef} */
    nth_value: {
      create(nth) {
        nth = +nth;
        if (!(nth > 0)) error('nth_value nth must be greater than zero.');
        return {
          init: noop,
          value: (w, f) => {
            const i = w.i0 + (nth - 1);
            return i < w.i1 ? w.value(i, f) : NULL;
          }
        };
      },
      param: [1, 1]
    },

    /** @type {WindowDef} */
    fill_down: {
      create(defaultValue = NULL) {
        let value;
        return {
          init: () => value = defaultValue,
          value: (w, f) => {
            const v = w.value(w.index, f);
            return isValid(v) ? (value = v) : value;
          }
        };
      },
      param: [1, 1]
    },

    /** @type {WindowDef} */
    fill_up: {
      create(defaultValue = NULL) {
        let value, idx;
        return {
          init: () => (value = defaultValue, idx = -1),
          value: (w, f) => w.index <= idx ? value
            : (idx = find(w, f, w.index)) >= 0 ? (value = w.value(idx, f))
            : (idx = w.size, value = defaultValue)
        };
      },
      param: [1, 1]
    }
  };

  function find(w, f, i) {
    for (const n = w.size; i < n; ++i) {
      if (isValid(w.value(i, f))) return i;
    }
    return -1;
  }

  /**
   * Check if an aggregate function with the given name exists.
   * @param {string} name The name of the aggregate function.
   * @return {boolean} True if found, false otherwise.
   */
  function hasAggregate(name) {
    return has(aggregateFunctions, name);
  }

  /**
   * Check if a window function with the given name exists.
   * @param {string} name The name of the window function.
   * @return {boolean} True if found, false otherwise.
   */
  function hasWindow(name) {
    return has(windowFunctions, name);
  }

  /**
   * Check if an expression function with the given name exists.
   * @param {string} name The name of the function.
   * @return {boolean} True if found, false otherwise.
   */
   function hasFunction(name) {
    return has(functions, name) || name === 'row_object';
  }

  /**
   * Get an aggregate function definition.
   * @param {string} name The name of the aggregate function.
   * @return {import('./aggregate-functions.js').AggregateDef}
   *  The aggregate function definition, or undefined if not found.
   */
  function getAggregate(name) {
    return hasAggregate(name) && aggregateFunctions[name];
  }

  /**
   * Get a window function definition.
   * @param {string} name The name of the window function.
   * @return {import('./window-functions.js').WindowDef}
   *  The window function definition, or undefined if not found.
   */
  function getWindow(name) {
    return hasWindow(name) && windowFunctions[name];
  }

  // eslint-disable-next-line no-unused-vars
  function concat$1(list, fn = ((x, i) => x), delim = '') {
    const n = list.length;
    if (!n) return '';

    let s = fn(list[0], 0);
    for (let i = 1; i < n; ++i) {
      s += delim + fn(list[i], i);
    }

    return s;
  }

  function unroll$1(args, code, ...lists) {
    const v = ['_', '$'];
    const a = v.slice(0, lists.length);
    a.push('"use strict"; const '
      + lists
          .map((l, j) => l.map((_, i) => `${v[j]}${i} = ${v[j]}[${i}]`).join(', '))
          .join(', ')
      + `; return (${args}) => ${code};`
    );
    return Function(...a)(...lists);
  }

  function ascending(a, b) {
    return a < b ? -1 : a > b ? 1 : a >= b ? 0 : NaN;
  }

  function max(values, start = 0, stop = values.length) {
    let max = stop ? values[start++] : NULL;

    for (let i = start; i < stop; ++i) {
      if (max < values[i]) {
        max = values[i];
      }
    }

    return max;
  }

  function min(values, start = 0, stop = values.length) {
    let min = stop ? values[start++] : NULL;

    for (let i = start; i < stop; ++i) {
      if (min > values[i]) {
        min = values[i];
      }
    }

    return min;
  }

  function toNumeric(value) {
    return isBigInt(value) ? value : +value;
  }

  function quantile(values, p) {
    const n = values.length;

    if (!n) return NULL;
    if ((p = +p) <= 0 || n < 2) return toNumeric(values[0]);
    if (p >= 1) return toNumeric(values[n - 1]);

    const i = (n - 1) * p;
    const i0 = Math.floor(i);
    const v0 = toNumeric(values[i0]);
    return isBigInt(v0)
      ? v0
      // @ts-ignore
      : v0 + (toNumeric(values[i0 + 1]) - v0) * (i - i0);
  }

  class ValueList {
    constructor(values) {
      this._values = values || [];
      this._sorted = null;
      this._start = 0;
    }

    values(copy) {
      if (this._start) {
        this._values = this._values.slice(this._start);
        this._start = 0;
      }
      return copy
        ? this._values.slice()
        : this._values;
    }

    add(value) {
      this._values.push(value);
      this._sorted = null;
    }

    rem() {
      this._start += 1;
      this._sorted = null;
    }

    min() {
      return this._sorted && this._sorted.length
        ? this._sorted[0]
        : min(this._values, this._start);
    }

    max() {
      return this._sorted && this._sorted.length
        ? this._sorted[this._sorted.length - 1]
        : max(this._values, this._start);
    }

    quantile(p) {
      if (!this._sorted) {
        this._sorted = this.values(true);
        this._sorted.sort(ascending);
      }
      return quantile(this._sorted, p);
    }
  }

  const update = (ops, args, fn) => unroll$1(
    args,
    '{' + concat$1(ops, (_, i) => `_${i}.${fn}(${args});`) + '}',
    ops
  );

  function fieldReducer(oplist, stream) {
    const { ops, output } = expand$1(oplist, stream);
    const fields = oplist[0].fields;
    const n = fields.length;
    const cls = n === 0 ? FieldReducer
      : n === 1 ? Field1Reducer
      : n === 2 ? Field2Reducer
      : error('Unsupported field count: ' + n);
    // @ts-ignore
    return new cls(fields, ops, output, stream);
  }

  function expand$1(oplist, stream) {
    const has = {};
    const ops = [];

    function add(name, params = []) {
      // check key
      const key = name + ':' + params;
      if (has[key]) return has[key];

      // get op instance
      const def = getAggregate(name);
      const op = def.create(...params);

      // add required dependencies
      if (stream < 0 && def.stream) {
        def.stream.forEach(name => add(name, []));
      }
      if (def.req) {
        def.req.forEach(name => add(name, []));
      }

      // update state
      has[key] = op;
      ops.push(op);

      return op;
    }

    const output = oplist.map(item => {
      const op = add(item.name, item.params);
      op.output = item.id;
      return op;
    });

    return { ops, output };
  }

  class FieldReducer extends Reducer {
    constructor(fields, ops, outputs, stream) {
      super(outputs);
      this._op = ops;
      this._fields = fields;
      this._stream = !!stream;
    }

    init() {
      const state = { count: 0, valid: 0, stream: this._stream };
      this._op.forEach(op => op.init(state));

      // value list requested
      if (state.values) {
        state.list = new ValueList();
      }

      return state;
    }

    write(state, values, index) {
      const op = this._outputs;
      const n = op.length;
      for (let i = 0; i < n; ++i) {
        values[op[i].output][index] = op[i].value(state);
      }
      return 1;
    }

    _add() {
    }

    _rem() {
    }

    add(state) {
      ++state.count;
    }

    rem(state) {
      --state.count;
    }
  }

  class Field1Reducer extends FieldReducer {
    constructor(fields, ops, outputs, stream) {
      super(fields, ops, outputs, stream);

      // unroll op invocations for performance
      const args = ['state', 'v1', 'v2'];
      this._add = update(ops, args, 'add');
      this._rem = update(ops, args, 'rem');
    }

    add(state, row, data) {
      const value = this._fields[0](row, data);
      ++state.count;
      if (isValid(value)) {
        ++state.valid;
        if (state.list) state.list.add(value);
        this._add(state, value);
      }
    }

    rem(state, row, data) {
      const value = this._fields[0](row, data);
      --state.count;
      if (isValid(value)) {
        --state.valid;
        if (state.list) state.list.rem();
        this._rem(state, value);
      }
    }
  }

  class Field2Reducer extends FieldReducer {
    constructor(fields, ops, outputs, stream) {
      super(fields, ops, outputs, stream);

      // unroll op invocations for performance
      const args = ['state', 'v1', 'v2'];
      this._add = update(ops, args, 'add');
      this._rem = update(ops, args, 'rem');
    }

    add(state, row, data) {
      const value1 = this._fields[0](row, data);
      const value2 = this._fields[1](row, data);
      ++state.count;
      if (isValid(value1) && isValid(value2)) {
        ++state.valid;
        if (state.list) state.list.add([value1, value2]);
        this._add(state, value1, value2);
      }
    }

    rem(state, row, data) {
      const value1 = this._fields[0](row, data);
      const value2 = this._fields[1](row, data);
      --state.count;
      if (isValid(value1) && isValid(value2)) {
        --state.valid;
        if (state.list) state.list.rem();
        this._rem(state, value1, value2);
      }
    }
  }

  /**
   * @returns {value is Function}
   */
  function isFunction(value) {
    return typeof value === 'function';
  }

  function repeat(reps, value) {
    const result = Array(reps);
    if (isFunction(value)) {
      for (let i = 0; i < reps; ++i) {
        result[i] = value(i);
      }
    } else {
      result.fill(value);
    }
    return result;
  }

  function aggregateGet(table, ops, get) {
    if (ops.length) {
      const data = table.data();
      const { keys } = table.groups() || {};
      const result = aggregate(table, ops);
      const op = keys
        ? (name, row) => result[name][keys[row]]
        : name => result[name][0];
      get = get.map(f => row => f(row, data, op));
    }

    return get;
  }

  function aggregate(table, ops, result) {
    if (!ops.length) return result; // early exit

    // instantiate aggregators and result store
    const aggrs = reducers(ops);
    const groups = table.groups();
    const size = groups ? groups.size : 1;
    result = result || repeat(ops.length, () => Array(size));

    // compute aggregates, extract results
    if (size > 1) {
      aggrs.forEach(aggr => {
        const cells = reduceGroups(table, aggr, groups);
        for (let i = 0; i < size; ++i) {
          aggr.write(cells[i], result, i);
        }
      });
    } else {
      aggrs.forEach(aggr => {
        const cell = reduceFlat(table, aggr);
        aggr.write(cell, result, 0);
      });
    }

    return result;
  }

  function reducers(ops, stream) {
    const aggrs = [];
    const fields = {};

    // group operators by field inputs
    for (const op of ops) {
      const key = op.fields.map(f => f + '').join(',');
      (fields[key] || (fields[key] = [])).push(op);
    }

    // generate a field reducer for each field
    for (const key in fields) {
      aggrs.push(fieldReducer(fields[key], stream));
    }

    return aggrs;
  }

  function reduceFlat(table, reducer) {
    // initialize aggregation cell
    const cell = reducer.init();

    // compute aggregate values
    // inline the following for performance:
    // table.scan((row, data) => reducer.add(cell, row, data));
    const n = table.totalRows();
    const data = table.data();
    const bits = table.mask();

    if (table.isOrdered()) {
      const idx = table.indices();
      for (let i = 0; i < n; ++i) {
        reducer.add(cell, idx[i], data);
      }
    } else if (bits) {
      for (let i = bits.next(0); i >= 0; i = bits.next(i + 1)) {
        reducer.add(cell, i, data);
      }
    } else {
      for (let i = 0; i < n; ++i) {
        reducer.add(cell, i, data);
      }
    }

    return cell;
  }

  function reduceGroups(table, reducer, groups) {
    const { keys, size } = groups;

    // initialize aggregation cells
    const cells = repeat(size, () => reducer.init());

    // compute aggregate values
    // inline the following for performance:
    // table.scan((row, data) => reducer.add(cells[keys[row]], row, data));
    const data = table.data();

    if (table.isOrdered()) {
      const idx = table.indices();
      const m = idx.length;
      for (let i = 0; i < m; ++i) {
        const row = idx[i];
        reducer.add(cells[keys[row]], row, data);
      }
    } else if (table.isFiltered()) {
      const bits = table.mask();
      for (let i = bits.next(0); i >= 0; i = bits.next(i + 1)) {
        reducer.add(cells[keys[i]], i, data);
      }
    } else {
      const n = table.totalRows();
      for (let i = 0; i < n; ++i) {
        reducer.add(cells[keys[i]], i, data);
      }
    }

    return cells;
  }

  function groupOutput(cols, groups) {
    const { get, names, rows, size } = groups;

    // write group values to output columns
    const m = names.length;
    for (let j = 0; j < m; ++j) {
      const col = cols.add(names[j], Array(size));
      const val = get[j];
      for (let i = 0; i < size; ++i) {
        col[i] = val(rows[i]);
      }
    }
  }

  const ArrayPattern = 'ArrayPattern';
  const ArrowFunctionExpression = 'ArrowFunctionExpression';
  const FunctionExpression = 'FunctionExpression';
  const Identifier = 'Identifier';
  const Literal = 'Literal';
  const MemberExpression = 'MemberExpression';
  const ObjectExpression = 'ObjectExpression';
  const ObjectPattern = 'ObjectPattern';
  const Property = 'Property';

  const Column$1 = 'Column';
  const Constant = 'Constant';
  const Dictionary = 'Dictionary';
  const Function$1 = 'Function';
  const Parameter = 'Parameter';
  const Op = 'Op';

  function walk(node, ctx, visitors, parent) {
    const visit = visitors[node.type] || visitors['Default'];
    if (visit && visit(node, ctx, parent) === false) return;

    const walker = walkers[node.type];
    if (walker) walker(node, ctx, visitors);
  }

  const unary = (node, ctx, visitors) => {
    walk(node.argument, ctx, visitors, node);
  };

  const binary$2 = (node, ctx, visitors) => {
    walk(node.left, ctx, visitors, node);
    walk(node.right, ctx, visitors, node);
  };

  const ternary = (node, ctx, visitors) => {
    walk(node.test, ctx, visitors, node);
    walk(node.consequent, ctx, visitors, node);
    if (node.alternate) walk(node.alternate, ctx, visitors, node);
  };

  const func$1 = (node, ctx, visitors) => {
    list$3(node.params, ctx, visitors, node);
    walk(node.body, ctx, visitors, node);
  };

  const call$1 = (node, ctx, visitors) => {
    walk(node.callee, ctx, visitors, node);
    list$3(node.arguments, ctx, visitors, node);
  };

  const list$3 = (nodes, ctx, visitors, node) => {
    nodes.forEach(item => walk(item, ctx, visitors, node));
  };

  const walkers = {
    TemplateLiteral: (node, ctx, visitors) => {
      list$3(node.expressions, ctx, visitors, node);
      list$3(node.quasis, ctx, visitors, node);
    },
    MemberExpression: (node, ctx, visitors) => {
      walk(node.object, ctx, visitors, node);
      walk(node.property, ctx, visitors, node);
    },
    CallExpression: call$1,
    NewExpression: call$1,
    ArrayExpression: (node, ctx, visitors) => {
      list$3(node.elements, ctx, visitors, node);
    },
    AssignmentExpression: binary$2,
    AwaitExpression: unary,
    BinaryExpression: binary$2,
    LogicalExpression: binary$2,
    UnaryExpression: unary,
    UpdateExpression: unary,
    ConditionalExpression: ternary,
    ObjectExpression: (node, ctx, visitors) => {
      list$3(node.properties, ctx, visitors, node);
    },
    Property: (node, ctx, visitors) => {
      walk(node.key, ctx, visitors, node);
      walk(node.value, ctx, visitors, node);
    },

    ArrowFunctionExpression: func$1,
    FunctionExpression: func$1,
    FunctionDeclaration: func$1,

    VariableDeclaration: (node, ctx, visitors) => {
      list$3(node.declarations, ctx, visitors, node);
    },
    VariableDeclarator: (node, ctx, visitors) => {
      walk(node.id, ctx, visitors, node);
      walk(node.init, ctx, visitors, node);
    },
    SpreadElement: (node, ctx, visitors) => {
      walk(node.argument, ctx, visitors, node);
    },

    BlockStatement: (node, ctx, visitors) => {
      list$3(node.body, ctx, visitors, node);
    },
    ExpressionStatement: (node, ctx, visitors) => {
      walk(node.expression, ctx, visitors, node);
    },
    IfStatement: ternary,
    ForStatement: (node, ctx, visitors) => {
      walk(node.init, ctx, visitors, node);
      walk(node.test, ctx, visitors, node);
      walk(node.update, ctx, visitors, node);
      walk(node.body, ctx, visitors, node);
    },
    WhileStatement: (node, ctx, visitors) => {
      walk(node.test, ctx, visitors, node);
      walk(node.body, ctx, visitors, node);
    },
    DoWhileStatement: (node, ctx, visitors) => {
      walk(node.body, ctx, visitors, node);
      walk(node.test, ctx, visitors, node);
    },
    SwitchStatement: (node, ctx, visitors) => {
      walk(node.discriminant, ctx, visitors, node);
      list$3(node.cases, ctx, visitors, node);
    },
    SwitchCase: (node, ctx, visitors) => {
      if (node.test) walk(node.test, ctx, visitors, node);
      list$3(node.consequent, ctx, visitors, node);
    },
    ReturnStatement: unary,

    Program: (node, ctx, visitors) => {
      walk(node.body[0], ctx, visitors, node);
    }
  };

  function strip(node) {
    delete node.start;
    delete node.end;
    delete node.optional;
  }

  function stripMember(node) {
    strip(node);
    delete node.object;
    delete node.property;
    delete node.computed;
    if (!node.table) delete node.table;
  }

  function clean(ast) {
    walk(ast, null, {
      Column: stripMember,
      Constant: stripMember,
      Default: strip
    });
    return ast;
  }

  function is(type, node) {
    return node && node.type === type;
  }

  function isFunctionExpression(node) {
    return is(FunctionExpression, node)
      || is(ArrowFunctionExpression, node);
  }

  const visit$2 = (node, opt) => {
    const f = visitors$1[node.type];
    return f
      ? f(node, opt)
      : error(`Unsupported expression construct: ${node.type}`);
  };

  const binary$1 = (node, opt) => {
    return '(' + visit$2(node.left, opt) + ' ' + node.operator + ' ' + visit$2(node.right, opt) + ')';
  };

  const func = (node, opt) => {
    return '(' + list$2(node.params, opt) + ')=>' + visit$2(node.body, opt);
  };

  const call = (node, opt) => {
    return visit$2(node.callee, opt) + '(' + list$2(node.arguments, opt) + ')';
  };

  const list$2 = (array, opt, delim = ',') => {
    return array.map(node => visit$2(node, opt)).join(delim);
  };

  const name = node => node.computed
    ? `[${toString$1(node.name)}]`
    : `.${node.name}`;

  const ref$1 = (node, opt, method) => {
    const table = node.table || '';
    return `data${table}${name(node)}.${method}(${opt.index}${table})`;
  };

  const get$1 = (node, opt) => {
    const table = node.table || '';
    return `data${table}${name(node)}[${opt.index}${table}]`;
  };

  const visitors$1 = {
    Constant: node => node.raw,
    Column: (node, opt) => node.array ? get$1(node, opt) : ref$1(node, opt, 'at'),
    Dictionary: (node, opt) => ref$1(node, opt, 'key'),
    Function: node => `fn.${node.name}`,
    Parameter: node => `$${name(node)}`,
    Op: (node, opt) => `op(${toString$1(node.name)},${opt.op || opt.index})`,
    Literal: node => node.raw,
    Identifier: node => node.name,
    TemplateLiteral: (node, opt) => {
      const { quasis, expressions } = node;
      const n = expressions.length;
      let t = quasis[0].value.raw;
      for (let i = 0; i < n;) {
        t += '${' + visit$2(expressions[i], opt) + '}' + quasis[++i].value.raw;
      }
      return '`' + t + '`';
    },
    MemberExpression: (node, opt) => {
      const d = !node.computed;
      const o = visit$2(node.object, opt);
      const p = visit$2(node.property, opt);
      return o + (d ? '.' + p : '[' + p + ']');
    },
    CallExpression: call,
    NewExpression: (node, opt) => {
      return 'new ' + call(node, opt);
    },
    ArrayExpression: (node, opt) => {
      return '[' + list$2(node.elements, opt) + ']';
    },
    AssignmentExpression: binary$1,
    BinaryExpression: binary$1,
    LogicalExpression: binary$1,
    UnaryExpression: (node, opt) => {
      return '(' + node.operator + visit$2(node.argument, opt) + ')';
    },
    ConditionalExpression: (node, opt) => {
      return '(' + visit$2(node.test, opt) +
        '?' + visit$2(node.consequent, opt) +
        ':' + visit$2(node.alternate, opt) + ')';
    },
    ObjectExpression: (node, opt) => {
      return '({' + list$2(node.properties, opt) + '})';
    },
    Property: (node, opt) => {
      const key = visit$2(node.key, opt);
      return (node.computed ? `[${key}]` : key) + ':' + visit$2(node.value, opt);
    },

    ArrowFunctionExpression: func,
    FunctionExpression: func,
    FunctionDeclaration: func,

    ArrayPattern: (node, opt) => {
      return '[' + list$2(node.elements, opt) + ']';
    },
    ObjectPattern: (node, opt) => {
      return '{' + list$2(node.properties, opt) + '}';
    },
    VariableDeclaration: (node, opt) => {
      return node.kind + ' ' + list$2(node.declarations, opt, ',');
    },
    VariableDeclarator: (node, opt) => {
      return visit$2(node.id, opt) + '=' + visit$2(node.init, opt);
    },
    SpreadElement: (node, opt) => {
      return '...' + visit$2(node.argument, opt);
    },

    BlockStatement: (node, opt) => {
      return '{' + list$2(node.body, opt, ';') + ';}';
    },
    BreakStatement: () => {
      return 'break';
    },
    ExpressionStatement: (node, opt) => {
      return visit$2(node.expression, opt);
    },
    IfStatement: (node, opt) => {
      return 'if (' + visit$2(node.test, opt) + ')'
        + visit$2(node.consequent, opt)
        + (node.alternate ? ' else ' + visit$2(node.alternate, opt) : '');
    },
    SwitchStatement: (node, opt) => {
      return 'switch (' + visit$2(node.discriminant, opt) + ') {'
       + list$2(node.cases, opt, '')
       + '}';
    },
    SwitchCase: (node, opt) => {
      return (node.test ? 'case ' + visit$2(node.test, opt) : 'default')
        + ': '
        + list$2(node.consequent, opt, ';') + ';';
    },
    ReturnStatement: (node, opt) => {
      return 'return ' + visit$2(node.argument, opt);
    },
    Program: (node, opt) => visit$2(node.body[0], opt)
  };

  function codegen(node, opt = { index: 'row' }) {
    return visit$2(node, opt);
  }

  function compile(code, fn, params) {
    code = `"use strict"; return ${code};`;
    return (Function('fn', '$', code))(fn, params);
  }

  var compile$1 = {
    escape: (code, func, params) => compile(code, func, params),
    expr:   (code, params) => compile(`(row,data,op)=>${code}`, functions, params),
    expr2:  (code, params) => compile(`(row0,data0,row,data)=>${code}`, functions, params),
    join:   (code, params) => compile(`(row1,data1,row2,data2)=>${code}`, functions, params),
    param:  (code, params) => compile(code, functions, params)
  };

  function entries(value) {
    return isArray$2(value) ? value
      : isMap(value) ? value.entries()
      : value ? Object.entries(value)
      : [];
  }

  const dictOps = {
    '==': 1,
    '!=': 1,
    '===': 1,
    '!==': 1
  };

  /**
   * Rewrite AST node to be a table column reference.
   * Additionally optimizes dictionary column operations.
   * @param {object} ref AST node to rewrite to a column reference.
   * @param {string} name The name of the column.
   * @param {number} [index] The table index of the column.
   * @param {object} [col] The actual table column instance.
   * @param {object} [op] Parent AST node operating on the column reference.
   */
  function rewrite(ref, name, index = 0, col = undefined, op = undefined) {
    ref.type = Column$1;
    ref.name = name;
    ref.table = index;

    // annotate arrays as such for optimized access
    if (isArrayType(col)) {
      ref.array = true;
    }

    // proceed only if has parent op and is a dictionary column
    if (op && col && isFunction(col.keyFor)) {
      // get other arg if op is an optimizeable operation
      const lit = dictOps[op.operator]
        ? op.left === ref ? op.right : op.left
        : op.callee && op.callee.name === 'equal'
        ? op.arguments[op.arguments[0] === ref ? 1 : 0]
        : null;

      // rewrite as dictionary lookup if other arg is a literal
      if (lit && lit.type === Literal) {
        rewriteDictionary(op, ref, lit, col.keyFor(lit.value));
      }
    }

    return ref;
  }

  function rewriteDictionary(op, ref, lit, key) {
    if (key < 0) {
      // value not in dictionary, rewrite op as false literal
      op.type = Literal;
      op.value = false;
      op.raw = 'false';
    } else {
      // rewrite ref as dict key access
      ref.type = Dictionary;

      // rewrite literal as target dict key
      lit.value = key;
      lit.raw = key + '';
    }

    return true;
  }

  const ROW_OBJECT = 'row_object';

  function rowObjectExpression(
    node,
    table,
    props = table.columnNames())
  {
    node.type = ObjectExpression;

    const p = node.properties = [];
    for (const prop of entries(props)) {
      const [name, key] = isArray$2(prop) ? prop : [prop, prop];
      p.push({
        type: Property,
        key: { type: Literal, raw: toString$1(key) },
        value: rewrite({ computed: true }, name, 0, table.column(name))
      });
    }

    return node;
  }

  function rowObjectCode(table, props) {
    return codegen(rowObjectExpression({}, table, props));
  }

  function rowObjectBuilder(table, props) {
    return compile$1.expr(rowObjectCode(table, props));
  }

  function toFunction(value) {
    return isFunction(value) ? value : () => value;
  }

  const ERROR_ESC_AGGRONLY = 'Escaped functions are not valid as rollup or pivot values.';

  function parseEscape(ctx, spec, params) {
    if (ctx.aggronly) error(ERROR_ESC_AGGRONLY);

    // generate escaped function invocation code
    const code = `(row,data)=>fn(${rowObjectCode(ctx.table)},$)`;

    return { escape: compile$1.escape(code, toFunction(spec.expr), params) };
  }

  // This file was generated. Do not modify manually!
  var astralIdentifierCodes = [509, 0, 227, 0, 150, 4, 294, 9, 1368, 2, 2, 1, 6, 3, 41, 2, 5, 0, 166, 1, 574, 3, 9, 9, 370, 1, 81, 2, 71, 10, 50, 3, 123, 2, 54, 14, 32, 10, 3, 1, 11, 3, 46, 10, 8, 0, 46, 9, 7, 2, 37, 13, 2, 9, 6, 1, 45, 0, 13, 2, 49, 13, 9, 3, 2, 11, 83, 11, 7, 0, 3, 0, 158, 11, 6, 9, 7, 3, 56, 1, 2, 6, 3, 1, 3, 2, 10, 0, 11, 1, 3, 6, 4, 4, 193, 17, 10, 9, 5, 0, 82, 19, 13, 9, 214, 6, 3, 8, 28, 1, 83, 16, 16, 9, 82, 12, 9, 9, 84, 14, 5, 9, 243, 14, 166, 9, 71, 5, 2, 1, 3, 3, 2, 0, 2, 1, 13, 9, 120, 6, 3, 6, 4, 0, 29, 9, 41, 6, 2, 3, 9, 0, 10, 10, 47, 15, 406, 7, 2, 7, 17, 9, 57, 21, 2, 13, 123, 5, 4, 0, 2, 1, 2, 6, 2, 0, 9, 9, 49, 4, 2, 1, 2, 4, 9, 9, 330, 3, 10, 1, 2, 0, 49, 6, 4, 4, 14, 9, 5351, 0, 7, 14, 13835, 9, 87, 9, 39, 4, 60, 6, 26, 9, 1014, 0, 2, 54, 8, 3, 82, 0, 12, 1, 19628, 1, 4706, 45, 3, 22, 543, 4, 4, 5, 9, 7, 3, 6, 31, 3, 149, 2, 1418, 49, 513, 54, 5, 49, 9, 0, 15, 0, 23, 4, 2, 14, 1361, 6, 2, 16, 3, 6, 2, 1, 2, 4, 101, 0, 161, 6, 10, 9, 357, 0, 62, 13, 499, 13, 983, 6, 110, 6, 6, 9, 4759, 9, 787719, 239];

  // This file was generated. Do not modify manually!
  var astralIdentifierStartCodes = [0, 11, 2, 25, 2, 18, 2, 1, 2, 14, 3, 13, 35, 122, 70, 52, 268, 28, 4, 48, 48, 31, 14, 29, 6, 37, 11, 29, 3, 35, 5, 7, 2, 4, 43, 157, 19, 35, 5, 35, 5, 39, 9, 51, 13, 10, 2, 14, 2, 6, 2, 1, 2, 10, 2, 14, 2, 6, 2, 1, 68, 310, 10, 21, 11, 7, 25, 5, 2, 41, 2, 8, 70, 5, 3, 0, 2, 43, 2, 1, 4, 0, 3, 22, 11, 22, 10, 30, 66, 18, 2, 1, 11, 21, 11, 25, 71, 55, 7, 1, 65, 0, 16, 3, 2, 2, 2, 28, 43, 28, 4, 28, 36, 7, 2, 27, 28, 53, 11, 21, 11, 18, 14, 17, 111, 72, 56, 50, 14, 50, 14, 35, 349, 41, 7, 1, 79, 28, 11, 0, 9, 21, 43, 17, 47, 20, 28, 22, 13, 52, 58, 1, 3, 0, 14, 44, 33, 24, 27, 35, 30, 0, 3, 0, 9, 34, 4, 0, 13, 47, 15, 3, 22, 0, 2, 0, 36, 17, 2, 24, 20, 1, 64, 6, 2, 0, 2, 3, 2, 14, 2, 9, 8, 46, 39, 7, 3, 1, 3, 21, 2, 6, 2, 1, 2, 4, 4, 0, 19, 0, 13, 4, 159, 52, 19, 3, 21, 2, 31, 47, 21, 1, 2, 0, 185, 46, 42, 3, 37, 47, 21, 0, 60, 42, 14, 0, 72, 26, 38, 6, 186, 43, 117, 63, 32, 7, 3, 0, 3, 7, 2, 1, 2, 23, 16, 0, 2, 0, 95, 7, 3, 38, 17, 0, 2, 0, 29, 0, 11, 39, 8, 0, 22, 0, 12, 45, 20, 0, 19, 72, 264, 8, 2, 36, 18, 0, 50, 29, 113, 6, 2, 1, 2, 37, 22, 0, 26, 5, 2, 1, 2, 31, 15, 0, 328, 18, 16, 0, 2, 12, 2, 33, 125, 0, 80, 921, 103, 110, 18, 195, 2637, 96, 16, 1071, 18, 5, 4026, 582, 8634, 568, 8, 30, 18, 78, 18, 29, 19, 47, 17, 3, 32, 20, 6, 18, 689, 63, 129, 74, 6, 0, 67, 12, 65, 1, 2, 0, 29, 6135, 9, 1237, 43, 8, 8936, 3, 2, 6, 2, 1, 2, 290, 16, 0, 30, 2, 3, 0, 15, 3, 9, 395, 2309, 106, 6, 12, 4, 8, 8, 9, 5991, 84, 2, 70, 2, 1, 3, 0, 3, 1, 3, 3, 2, 11, 2, 0, 2, 6, 2, 64, 2, 3, 3, 7, 2, 6, 2, 27, 2, 3, 2, 4, 2, 0, 4, 6, 2, 339, 3, 24, 2, 24, 2, 30, 2, 24, 2, 30, 2, 24, 2, 30, 2, 24, 2, 30, 2, 24, 2, 7, 1845, 30, 7, 5, 262, 61, 147, 44, 11, 6, 17, 0, 322, 29, 19, 43, 485, 27, 757, 6, 2, 3, 2, 1, 2, 14, 2, 196, 60, 67, 8, 0, 1205, 3, 2, 26, 2, 1, 2, 0, 3, 0, 2, 9, 2, 3, 2, 0, 2, 0, 7, 0, 5, 0, 2, 0, 2, 0, 2, 2, 2, 1, 2, 0, 3, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 1, 2, 0, 3, 3, 2, 6, 2, 3, 2, 3, 2, 0, 2, 9, 2, 16, 6, 2, 2, 4, 2, 16, 4421, 42719, 33, 4153, 7, 221, 3, 5761, 15, 7472, 16, 621, 2467, 541, 1507, 4938, 6, 4191];

  // This file was generated. Do not modify manually!
  var nonASCIIidentifierChars = "\u200c\u200d\xb7\u0300-\u036f\u0387\u0483-\u0487\u0591-\u05bd\u05bf\u05c1\u05c2\u05c4\u05c5\u05c7\u0610-\u061a\u064b-\u0669\u0670\u06d6-\u06dc\u06df-\u06e4\u06e7\u06e8\u06ea-\u06ed\u06f0-\u06f9\u0711\u0730-\u074a\u07a6-\u07b0\u07c0-\u07c9\u07eb-\u07f3\u07fd\u0816-\u0819\u081b-\u0823\u0825-\u0827\u0829-\u082d\u0859-\u085b\u0898-\u089f\u08ca-\u08e1\u08e3-\u0903\u093a-\u093c\u093e-\u094f\u0951-\u0957\u0962\u0963\u0966-\u096f\u0981-\u0983\u09bc\u09be-\u09c4\u09c7\u09c8\u09cb-\u09cd\u09d7\u09e2\u09e3\u09e6-\u09ef\u09fe\u0a01-\u0a03\u0a3c\u0a3e-\u0a42\u0a47\u0a48\u0a4b-\u0a4d\u0a51\u0a66-\u0a71\u0a75\u0a81-\u0a83\u0abc\u0abe-\u0ac5\u0ac7-\u0ac9\u0acb-\u0acd\u0ae2\u0ae3\u0ae6-\u0aef\u0afa-\u0aff\u0b01-\u0b03\u0b3c\u0b3e-\u0b44\u0b47\u0b48\u0b4b-\u0b4d\u0b55-\u0b57\u0b62\u0b63\u0b66-\u0b6f\u0b82\u0bbe-\u0bc2\u0bc6-\u0bc8\u0bca-\u0bcd\u0bd7\u0be6-\u0bef\u0c00-\u0c04\u0c3c\u0c3e-\u0c44\u0c46-\u0c48\u0c4a-\u0c4d\u0c55\u0c56\u0c62\u0c63\u0c66-\u0c6f\u0c81-\u0c83\u0cbc\u0cbe-\u0cc4\u0cc6-\u0cc8\u0cca-\u0ccd\u0cd5\u0cd6\u0ce2\u0ce3\u0ce6-\u0cef\u0cf3\u0d00-\u0d03\u0d3b\u0d3c\u0d3e-\u0d44\u0d46-\u0d48\u0d4a-\u0d4d\u0d57\u0d62\u0d63\u0d66-\u0d6f\u0d81-\u0d83\u0dca\u0dcf-\u0dd4\u0dd6\u0dd8-\u0ddf\u0de6-\u0def\u0df2\u0df3\u0e31\u0e34-\u0e3a\u0e47-\u0e4e\u0e50-\u0e59\u0eb1\u0eb4-\u0ebc\u0ec8-\u0ece\u0ed0-\u0ed9\u0f18\u0f19\u0f20-\u0f29\u0f35\u0f37\u0f39\u0f3e\u0f3f\u0f71-\u0f84\u0f86\u0f87\u0f8d-\u0f97\u0f99-\u0fbc\u0fc6\u102b-\u103e\u1040-\u1049\u1056-\u1059\u105e-\u1060\u1062-\u1064\u1067-\u106d\u1071-\u1074\u1082-\u108d\u108f-\u109d\u135d-\u135f\u1369-\u1371\u1712-\u1715\u1732-\u1734\u1752\u1753\u1772\u1773\u17b4-\u17d3\u17dd\u17e0-\u17e9\u180b-\u180d\u180f-\u1819\u18a9\u1920-\u192b\u1930-\u193b\u1946-\u194f\u19d0-\u19da\u1a17-\u1a1b\u1a55-\u1a5e\u1a60-\u1a7c\u1a7f-\u1a89\u1a90-\u1a99\u1ab0-\u1abd\u1abf-\u1ace\u1b00-\u1b04\u1b34-\u1b44\u1b50-\u1b59\u1b6b-\u1b73\u1b80-\u1b82\u1ba1-\u1bad\u1bb0-\u1bb9\u1be6-\u1bf3\u1c24-\u1c37\u1c40-\u1c49\u1c50-\u1c59\u1cd0-\u1cd2\u1cd4-\u1ce8\u1ced\u1cf4\u1cf7-\u1cf9\u1dc0-\u1dff\u200c\u200d\u203f\u2040\u2054\u20d0-\u20dc\u20e1\u20e5-\u20f0\u2cef-\u2cf1\u2d7f\u2de0-\u2dff\u302a-\u302f\u3099\u309a\u30fb\ua620-\ua629\ua66f\ua674-\ua67d\ua69e\ua69f\ua6f0\ua6f1\ua802\ua806\ua80b\ua823-\ua827\ua82c\ua880\ua881\ua8b4-\ua8c5\ua8d0-\ua8d9\ua8e0-\ua8f1\ua8ff-\ua909\ua926-\ua92d\ua947-\ua953\ua980-\ua983\ua9b3-\ua9c0\ua9d0-\ua9d9\ua9e5\ua9f0-\ua9f9\uaa29-\uaa36\uaa43\uaa4c\uaa4d\uaa50-\uaa59\uaa7b-\uaa7d\uaab0\uaab2-\uaab4\uaab7\uaab8\uaabe\uaabf\uaac1\uaaeb-\uaaef\uaaf5\uaaf6\uabe3-\uabea\uabec\uabed\uabf0-\uabf9\ufb1e\ufe00-\ufe0f\ufe20-\ufe2f\ufe33\ufe34\ufe4d-\ufe4f\uff10-\uff19\uff3f\uff65";

  // This file was generated. Do not modify manually!
  var nonASCIIidentifierStartChars = "\xaa\xb5\xba\xc0-\xd6\xd8-\xf6\xf8-\u02c1\u02c6-\u02d1\u02e0-\u02e4\u02ec\u02ee\u0370-\u0374\u0376\u0377\u037a-\u037d\u037f\u0386\u0388-\u038a\u038c\u038e-\u03a1\u03a3-\u03f5\u03f7-\u0481\u048a-\u052f\u0531-\u0556\u0559\u0560-\u0588\u05d0-\u05ea\u05ef-\u05f2\u0620-\u064a\u066e\u066f\u0671-\u06d3\u06d5\u06e5\u06e6\u06ee\u06ef\u06fa-\u06fc\u06ff\u0710\u0712-\u072f\u074d-\u07a5\u07b1\u07ca-\u07ea\u07f4\u07f5\u07fa\u0800-\u0815\u081a\u0824\u0828\u0840-\u0858\u0860-\u086a\u0870-\u0887\u0889-\u088e\u08a0-\u08c9\u0904-\u0939\u093d\u0950\u0958-\u0961\u0971-\u0980\u0985-\u098c\u098f\u0990\u0993-\u09a8\u09aa-\u09b0\u09b2\u09b6-\u09b9\u09bd\u09ce\u09dc\u09dd\u09df-\u09e1\u09f0\u09f1\u09fc\u0a05-\u0a0a\u0a0f\u0a10\u0a13-\u0a28\u0a2a-\u0a30\u0a32\u0a33\u0a35\u0a36\u0a38\u0a39\u0a59-\u0a5c\u0a5e\u0a72-\u0a74\u0a85-\u0a8d\u0a8f-\u0a91\u0a93-\u0aa8\u0aaa-\u0ab0\u0ab2\u0ab3\u0ab5-\u0ab9\u0abd\u0ad0\u0ae0\u0ae1\u0af9\u0b05-\u0b0c\u0b0f\u0b10\u0b13-\u0b28\u0b2a-\u0b30\u0b32\u0b33\u0b35-\u0b39\u0b3d\u0b5c\u0b5d\u0b5f-\u0b61\u0b71\u0b83\u0b85-\u0b8a\u0b8e-\u0b90\u0b92-\u0b95\u0b99\u0b9a\u0b9c\u0b9e\u0b9f\u0ba3\u0ba4\u0ba8-\u0baa\u0bae-\u0bb9\u0bd0\u0c05-\u0c0c\u0c0e-\u0c10\u0c12-\u0c28\u0c2a-\u0c39\u0c3d\u0c58-\u0c5a\u0c5d\u0c60\u0c61\u0c80\u0c85-\u0c8c\u0c8e-\u0c90\u0c92-\u0ca8\u0caa-\u0cb3\u0cb5-\u0cb9\u0cbd\u0cdd\u0cde\u0ce0\u0ce1\u0cf1\u0cf2\u0d04-\u0d0c\u0d0e-\u0d10\u0d12-\u0d3a\u0d3d\u0d4e\u0d54-\u0d56\u0d5f-\u0d61\u0d7a-\u0d7f\u0d85-\u0d96\u0d9a-\u0db1\u0db3-\u0dbb\u0dbd\u0dc0-\u0dc6\u0e01-\u0e30\u0e32\u0e33\u0e40-\u0e46\u0e81\u0e82\u0e84\u0e86-\u0e8a\u0e8c-\u0ea3\u0ea5\u0ea7-\u0eb0\u0eb2\u0eb3\u0ebd\u0ec0-\u0ec4\u0ec6\u0edc-\u0edf\u0f00\u0f40-\u0f47\u0f49-\u0f6c\u0f88-\u0f8c\u1000-\u102a\u103f\u1050-\u1055\u105a-\u105d\u1061\u1065\u1066\u106e-\u1070\u1075-\u1081\u108e\u10a0-\u10c5\u10c7\u10cd\u10d0-\u10fa\u10fc-\u1248\u124a-\u124d\u1250-\u1256\u1258\u125a-\u125d\u1260-\u1288\u128a-\u128d\u1290-\u12b0\u12b2-\u12b5\u12b8-\u12be\u12c0\u12c2-\u12c5\u12c8-\u12d6\u12d8-\u1310\u1312-\u1315\u1318-\u135a\u1380-\u138f\u13a0-\u13f5\u13f8-\u13fd\u1401-\u166c\u166f-\u167f\u1681-\u169a\u16a0-\u16ea\u16ee-\u16f8\u1700-\u1711\u171f-\u1731\u1740-\u1751\u1760-\u176c\u176e-\u1770\u1780-\u17b3\u17d7\u17dc\u1820-\u1878\u1880-\u18a8\u18aa\u18b0-\u18f5\u1900-\u191e\u1950-\u196d\u1970-\u1974\u1980-\u19ab\u19b0-\u19c9\u1a00-\u1a16\u1a20-\u1a54\u1aa7\u1b05-\u1b33\u1b45-\u1b4c\u1b83-\u1ba0\u1bae\u1baf\u1bba-\u1be5\u1c00-\u1c23\u1c4d-\u1c4f\u1c5a-\u1c7d\u1c80-\u1c88\u1c90-\u1cba\u1cbd-\u1cbf\u1ce9-\u1cec\u1cee-\u1cf3\u1cf5\u1cf6\u1cfa\u1d00-\u1dbf\u1e00-\u1f15\u1f18-\u1f1d\u1f20-\u1f45\u1f48-\u1f4d\u1f50-\u1f57\u1f59\u1f5b\u1f5d\u1f5f-\u1f7d\u1f80-\u1fb4\u1fb6-\u1fbc\u1fbe\u1fc2-\u1fc4\u1fc6-\u1fcc\u1fd0-\u1fd3\u1fd6-\u1fdb\u1fe0-\u1fec\u1ff2-\u1ff4\u1ff6-\u1ffc\u2071\u207f\u2090-\u209c\u2102\u2107\u210a-\u2113\u2115\u2118-\u211d\u2124\u2126\u2128\u212a-\u2139\u213c-\u213f\u2145-\u2149\u214e\u2160-\u2188\u2c00-\u2ce4\u2ceb-\u2cee\u2cf2\u2cf3\u2d00-\u2d25\u2d27\u2d2d\u2d30-\u2d67\u2d6f\u2d80-\u2d96\u2da0-\u2da6\u2da8-\u2dae\u2db0-\u2db6\u2db8-\u2dbe\u2dc0-\u2dc6\u2dc8-\u2dce\u2dd0-\u2dd6\u2dd8-\u2dde\u3005-\u3007\u3021-\u3029\u3031-\u3035\u3038-\u303c\u3041-\u3096\u309b-\u309f\u30a1-\u30fa\u30fc-\u30ff\u3105-\u312f\u3131-\u318e\u31a0-\u31bf\u31f0-\u31ff\u3400-\u4dbf\u4e00-\ua48c\ua4d0-\ua4fd\ua500-\ua60c\ua610-\ua61f\ua62a\ua62b\ua640-\ua66e\ua67f-\ua69d\ua6a0-\ua6ef\ua717-\ua71f\ua722-\ua788\ua78b-\ua7ca\ua7d0\ua7d1\ua7d3\ua7d5-\ua7d9\ua7f2-\ua801\ua803-\ua805\ua807-\ua80a\ua80c-\ua822\ua840-\ua873\ua882-\ua8b3\ua8f2-\ua8f7\ua8fb\ua8fd\ua8fe\ua90a-\ua925\ua930-\ua946\ua960-\ua97c\ua984-\ua9b2\ua9cf\ua9e0-\ua9e4\ua9e6-\ua9ef\ua9fa-\ua9fe\uaa00-\uaa28\uaa40-\uaa42\uaa44-\uaa4b\uaa60-\uaa76\uaa7a\uaa7e-\uaaaf\uaab1\uaab5\uaab6\uaab9-\uaabd\uaac0\uaac2\uaadb-\uaadd\uaae0-\uaaea\uaaf2-\uaaf4\uab01-\uab06\uab09-\uab0e\uab11-\uab16\uab20-\uab26\uab28-\uab2e\uab30-\uab5a\uab5c-\uab69\uab70-\uabe2\uac00-\ud7a3\ud7b0-\ud7c6\ud7cb-\ud7fb\uf900-\ufa6d\ufa70-\ufad9\ufb00-\ufb06\ufb13-\ufb17\ufb1d\ufb1f-\ufb28\ufb2a-\ufb36\ufb38-\ufb3c\ufb3e\ufb40\ufb41\ufb43\ufb44\ufb46-\ufbb1\ufbd3-\ufd3d\ufd50-\ufd8f\ufd92-\ufdc7\ufdf0-\ufdfb\ufe70-\ufe74\ufe76-\ufefc\uff21-\uff3a\uff41-\uff5a\uff66-\uffbe\uffc2-\uffc7\uffca-\uffcf\uffd2-\uffd7\uffda-\uffdc";

  // These are a run-length and offset encoded representation of the
  // >0xffff code points that are a valid part of identifiers. The
  // offset starts at 0x10000, and each pair of numbers represents an
  // offset to the next range, and then a size of the range.

  // Reserved word lists for various dialects of the language

  var reservedWords = {
    3: "abstract boolean byte char class double enum export extends final float goto implements import int interface long native package private protected public short static super synchronized throws transient volatile",
    5: "class enum extends super const export import",
    6: "enum",
    strict: "implements interface let package private protected public static yield",
    strictBind: "eval arguments"
  };

  // And the keywords

  var ecma5AndLessKeywords = "break case catch continue debugger default do else finally for function if return switch throw try var while with null true false instanceof typeof void delete new in this";

  var keywords$1 = {
    5: ecma5AndLessKeywords,
    "5module": ecma5AndLessKeywords + " export import",
    6: ecma5AndLessKeywords + " const class extends export import super"
  };

  var keywordRelationalOperator = /^in(stanceof)?$/;

  // ## Character categories

  var nonASCIIidentifierStart = new RegExp("[" + nonASCIIidentifierStartChars + "]");
  var nonASCIIidentifier = new RegExp("[" + nonASCIIidentifierStartChars + nonASCIIidentifierChars + "]");

  // This has a complexity linear to the value of the code. The
  // assumption is that looking up astral identifier characters is
  // rare.
  function isInAstralSet(code, set) {
    var pos = 0x10000;
    for (var i = 0; i < set.length; i += 2) {
      pos += set[i];
      if (pos > code) { return false }
      pos += set[i + 1];
      if (pos >= code) { return true }
    }
    return false
  }

  // Test whether a given character code starts an identifier.

  function isIdentifierStart(code, astral) {
    if (code < 65) { return code === 36 }
    if (code < 91) { return true }
    if (code < 97) { return code === 95 }
    if (code < 123) { return true }
    if (code <= 0xffff) { return code >= 0xaa && nonASCIIidentifierStart.test(String.fromCharCode(code)) }
    if (astral === false) { return false }
    return isInAstralSet(code, astralIdentifierStartCodes)
  }

  // Test whether a given character is part of an identifier.

  function isIdentifierChar(code, astral) {
    if (code < 48) { return code === 36 }
    if (code < 58) { return true }
    if (code < 65) { return false }
    if (code < 91) { return true }
    if (code < 97) { return code === 95 }
    if (code < 123) { return true }
    if (code <= 0xffff) { return code >= 0xaa && nonASCIIidentifier.test(String.fromCharCode(code)) }
    if (astral === false) { return false }
    return isInAstralSet(code, astralIdentifierStartCodes) || isInAstralSet(code, astralIdentifierCodes)
  }

  // ## Token types

  // The assignment of fine-grained, information-carrying type objects
  // allows the tokenizer to store the information it has about a
  // token in a way that is very cheap for the parser to look up.

  // All token type variables start with an underscore, to make them
  // easy to recognize.

  // The `beforeExpr` property is used to disambiguate between regular
  // expressions and divisions. It is set on all token types that can
  // be followed by an expression (thus, a slash after them would be a
  // regular expression).
  //
  // The `startsExpr` property is used to check if the token ends a
  // `yield` expression. It is set on all token types that either can
  // directly start an expression (like a quotation mark) or can
  // continue an expression (like the body of a string).
  //
  // `isLoop` marks a keyword as starting a loop, which is important
  // to know when parsing a label, in order to allow or disallow
  // continue jumps to that label.

  var TokenType = function TokenType(label, conf) {
    if ( conf === void 0 ) conf = {};

    this.label = label;
    this.keyword = conf.keyword;
    this.beforeExpr = !!conf.beforeExpr;
    this.startsExpr = !!conf.startsExpr;
    this.isLoop = !!conf.isLoop;
    this.isAssign = !!conf.isAssign;
    this.prefix = !!conf.prefix;
    this.postfix = !!conf.postfix;
    this.binop = conf.binop || null;
    this.updateContext = null;
  };

  function binop(name, prec) {
    return new TokenType(name, {beforeExpr: true, binop: prec})
  }
  var beforeExpr = {beforeExpr: true}, startsExpr = {startsExpr: true};

  // Map keyword names to token types.

  var keywords = {};

  // Succinct definitions of keyword token types
  function kw(name, options) {
    if ( options === void 0 ) options = {};

    options.keyword = name;
    return keywords[name] = new TokenType(name, options)
  }

  var types$1 = {
    num: new TokenType("num", startsExpr),
    regexp: new TokenType("regexp", startsExpr),
    string: new TokenType("string", startsExpr),
    name: new TokenType("name", startsExpr),
    privateId: new TokenType("privateId", startsExpr),
    eof: new TokenType("eof"),

    // Punctuation token types.
    bracketL: new TokenType("[", {beforeExpr: true, startsExpr: true}),
    bracketR: new TokenType("]"),
    braceL: new TokenType("{", {beforeExpr: true, startsExpr: true}),
    braceR: new TokenType("}"),
    parenL: new TokenType("(", {beforeExpr: true, startsExpr: true}),
    parenR: new TokenType(")"),
    comma: new TokenType(",", beforeExpr),
    semi: new TokenType(";", beforeExpr),
    colon: new TokenType(":", beforeExpr),
    dot: new TokenType("."),
    question: new TokenType("?", beforeExpr),
    questionDot: new TokenType("?."),
    arrow: new TokenType("=>", beforeExpr),
    template: new TokenType("template"),
    invalidTemplate: new TokenType("invalidTemplate"),
    ellipsis: new TokenType("...", beforeExpr),
    backQuote: new TokenType("`", startsExpr),
    dollarBraceL: new TokenType("${", {beforeExpr: true, startsExpr: true}),

    // Operators. These carry several kinds of properties to help the
    // parser use them properly (the presence of these properties is
    // what categorizes them as operators).
    //
    // `binop`, when present, specifies that this operator is a binary
    // operator, and will refer to its precedence.
    //
    // `prefix` and `postfix` mark the operator as a prefix or postfix
    // unary operator.
    //
    // `isAssign` marks all of `=`, `+=`, `-=` etcetera, which act as
    // binary operators with a very low precedence, that should result
    // in AssignmentExpression nodes.

    eq: new TokenType("=", {beforeExpr: true, isAssign: true}),
    assign: new TokenType("_=", {beforeExpr: true, isAssign: true}),
    incDec: new TokenType("++/--", {prefix: true, postfix: true, startsExpr: true}),
    prefix: new TokenType("!/~", {beforeExpr: true, prefix: true, startsExpr: true}),
    logicalOR: binop("||", 1),
    logicalAND: binop("&&", 2),
    bitwiseOR: binop("|", 3),
    bitwiseXOR: binop("^", 4),
    bitwiseAND: binop("&", 5),
    equality: binop("==/!=/===/!==", 6),
    relational: binop("</>/<=/>=", 7),
    bitShift: binop("<</>>/>>>", 8),
    plusMin: new TokenType("+/-", {beforeExpr: true, binop: 9, prefix: true, startsExpr: true}),
    modulo: binop("%", 10),
    star: binop("*", 10),
    slash: binop("/", 10),
    starstar: new TokenType("**", {beforeExpr: true}),
    coalesce: binop("??", 1),

    // Keyword token types.
    _break: kw("break"),
    _case: kw("case", beforeExpr),
    _catch: kw("catch"),
    _continue: kw("continue"),
    _debugger: kw("debugger"),
    _default: kw("default", beforeExpr),
    _do: kw("do", {isLoop: true, beforeExpr: true}),
    _else: kw("else", beforeExpr),
    _finally: kw("finally"),
    _for: kw("for", {isLoop: true}),
    _function: kw("function", startsExpr),
    _if: kw("if"),
    _return: kw("return", beforeExpr),
    _switch: kw("switch"),
    _throw: kw("throw", beforeExpr),
    _try: kw("try"),
    _var: kw("var"),
    _const: kw("const"),
    _while: kw("while", {isLoop: true}),
    _with: kw("with"),
    _new: kw("new", {beforeExpr: true, startsExpr: true}),
    _this: kw("this", startsExpr),
    _super: kw("super", startsExpr),
    _class: kw("class", startsExpr),
    _extends: kw("extends", beforeExpr),
    _export: kw("export"),
    _import: kw("import", startsExpr),
    _null: kw("null", startsExpr),
    _true: kw("true", startsExpr),
    _false: kw("false", startsExpr),
    _in: kw("in", {beforeExpr: true, binop: 7}),
    _instanceof: kw("instanceof", {beforeExpr: true, binop: 7}),
    _typeof: kw("typeof", {beforeExpr: true, prefix: true, startsExpr: true}),
    _void: kw("void", {beforeExpr: true, prefix: true, startsExpr: true}),
    _delete: kw("delete", {beforeExpr: true, prefix: true, startsExpr: true})
  };

  // Matches a whole line break (where CRLF is considered a single
  // line break). Used to count lines.

  var lineBreak = /\r\n?|\n|\u2028|\u2029/;
  var lineBreakG = new RegExp(lineBreak.source, "g");

  function isNewLine(code) {
    return code === 10 || code === 13 || code === 0x2028 || code === 0x2029
  }

  function nextLineBreak(code, from, end) {
    if ( end === void 0 ) end = code.length;

    for (var i = from; i < end; i++) {
      var next = code.charCodeAt(i);
      if (isNewLine(next))
        { return i < end - 1 && next === 13 && code.charCodeAt(i + 1) === 10 ? i + 2 : i + 1 }
    }
    return -1
  }

  var nonASCIIwhitespace = /[\u1680\u2000-\u200a\u202f\u205f\u3000\ufeff]/;

  var skipWhiteSpace = /(?:\s|\/\/.*|\/\*[^]*?\*\/)*/g;

  var ref = Object.prototype;
  var hasOwnProperty = ref.hasOwnProperty;
  var toString = ref.toString;

  var hasOwn = Object.hasOwn || (function (obj, propName) { return (
    hasOwnProperty.call(obj, propName)
  ); });

  var isArray$1 = Array.isArray || (function (obj) { return (
    toString.call(obj) === "[object Array]"
  ); });

  var regexpCache = Object.create(null);

  function wordsRegexp(words) {
    return regexpCache[words] || (regexpCache[words] = new RegExp("^(?:" + words.replace(/ /g, "|") + ")$"))
  }

  function codePointToString(code) {
    // UTF-16 Decoding
    if (code <= 0xFFFF) { return String.fromCharCode(code) }
    code -= 0x10000;
    return String.fromCharCode((code >> 10) + 0xD800, (code & 1023) + 0xDC00)
  }

  var loneSurrogate = /(?:[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF])/;

  // These are used when `options.locations` is on, for the
  // `startLoc` and `endLoc` properties.

  var Position = function Position(line, col) {
    this.line = line;
    this.column = col;
  };

  Position.prototype.offset = function offset (n) {
    return new Position(this.line, this.column + n)
  };

  var SourceLocation = function SourceLocation(p, start, end) {
    this.start = start;
    this.end = end;
    if (p.sourceFile !== null) { this.source = p.sourceFile; }
  };

  // The `getLineInfo` function is mostly useful when the
  // `locations` option is off (for performance reasons) and you
  // want to find the line/column position for a given character
  // offset. `input` should be the code string that the offset refers
  // into.

  function getLineInfo(input, offset) {
    for (var line = 1, cur = 0;;) {
      var nextBreak = nextLineBreak(input, cur, offset);
      if (nextBreak < 0) { return new Position(line, offset - cur) }
      ++line;
      cur = nextBreak;
    }
  }

  // A second argument must be given to configure the parser process.
  // These options are recognized (only `ecmaVersion` is required):

  var defaultOptions = {
    // `ecmaVersion` indicates the ECMAScript version to parse. Must be
    // either 3, 5, 6 (or 2015), 7 (2016), 8 (2017), 9 (2018), 10
    // (2019), 11 (2020), 12 (2021), 13 (2022), 14 (2023), or `"latest"`
    // (the latest version the library supports). This influences
    // support for strict mode, the set of reserved words, and support
    // for new syntax features.
    ecmaVersion: null,
    // `sourceType` indicates the mode the code should be parsed in.
    // Can be either `"script"` or `"module"`. This influences global
    // strict mode and parsing of `import` and `export` declarations.
    sourceType: "script",
    // `onInsertedSemicolon` can be a callback that will be called when
    // a semicolon is automatically inserted. It will be passed the
    // position of the inserted semicolon as an offset, and if
    // `locations` is enabled, it is given the location as a `{line,
    // column}` object as second argument.
    onInsertedSemicolon: null,
    // `onTrailingComma` is similar to `onInsertedSemicolon`, but for
    // trailing commas.
    onTrailingComma: null,
    // By default, reserved words are only enforced if ecmaVersion >= 5.
    // Set `allowReserved` to a boolean value to explicitly turn this on
    // an off. When this option has the value "never", reserved words
    // and keywords can also not be used as property names.
    allowReserved: null,
    // When enabled, a return at the top level is not considered an
    // error.
    allowReturnOutsideFunction: false,
    // When enabled, import/export statements are not constrained to
    // appearing at the top of the program, and an import.meta expression
    // in a script isn't considered an error.
    allowImportExportEverywhere: false,
    // By default, await identifiers are allowed to appear at the top-level scope only if ecmaVersion >= 2022.
    // When enabled, await identifiers are allowed to appear at the top-level scope,
    // but they are still not allowed in non-async functions.
    allowAwaitOutsideFunction: null,
    // When enabled, super identifiers are not constrained to
    // appearing in methods and do not raise an error when they appear elsewhere.
    allowSuperOutsideMethod: null,
    // When enabled, hashbang directive in the beginning of file is
    // allowed and treated as a line comment. Enabled by default when
    // `ecmaVersion` >= 2023.
    allowHashBang: false,
    // By default, the parser will verify that private properties are
    // only used in places where they are valid and have been declared.
    // Set this to false to turn such checks off.
    checkPrivateFields: true,
    // When `locations` is on, `loc` properties holding objects with
    // `start` and `end` properties in `{line, column}` form (with
    // line being 1-based and column 0-based) will be attached to the
    // nodes.
    locations: false,
    // A function can be passed as `onToken` option, which will
    // cause Acorn to call that function with object in the same
    // format as tokens returned from `tokenizer().getToken()`. Note
    // that you are not allowed to call the parser from the
    // callback—that will corrupt its internal state.
    onToken: null,
    // A function can be passed as `onComment` option, which will
    // cause Acorn to call that function with `(block, text, start,
    // end)` parameters whenever a comment is skipped. `block` is a
    // boolean indicating whether this is a block (`/* */`) comment,
    // `text` is the content of the comment, and `start` and `end` are
    // character offsets that denote the start and end of the comment.
    // When the `locations` option is on, two more parameters are
    // passed, the full `{line, column}` locations of the start and
    // end of the comments. Note that you are not allowed to call the
    // parser from the callback—that will corrupt its internal state.
    // When this option has an array as value, objects representing the
    // comments are pushed to it.
    onComment: null,
    // Nodes have their start and end characters offsets recorded in
    // `start` and `end` properties (directly on the node, rather than
    // the `loc` object, which holds line/column data. To also add a
    // [semi-standardized][range] `range` property holding a `[start,
    // end]` array with the same numbers, set the `ranges` option to
    // `true`.
    //
    // [range]: https://bugzilla.mozilla.org/show_bug.cgi?id=745678
    ranges: false,
    // It is possible to parse multiple files into a single AST by
    // passing the tree produced by parsing the first file as
    // `program` option in subsequent parses. This will add the
    // toplevel forms of the parsed file to the `Program` (top) node
    // of an existing parse tree.
    program: null,
    // When `locations` is on, you can pass this to record the source
    // file in every node's `loc` object.
    sourceFile: null,
    // This value, if given, is stored in every node, whether
    // `locations` is on or off.
    directSourceFile: null,
    // When enabled, parenthesized expressions are represented by
    // (non-standard) ParenthesizedExpression nodes
    preserveParens: false
  };

  // Interpret and default an options object

  var warnedAboutEcmaVersion = false;

  function getOptions(opts) {
    var options = {};

    for (var opt in defaultOptions)
      { options[opt] = opts && hasOwn(opts, opt) ? opts[opt] : defaultOptions[opt]; }

    if (options.ecmaVersion === "latest") {
      options.ecmaVersion = 1e8;
    } else if (options.ecmaVersion == null) {
      if (!warnedAboutEcmaVersion && typeof console === "object" && console.warn) {
        warnedAboutEcmaVersion = true;
        console.warn("Since Acorn 8.0.0, options.ecmaVersion is required.\nDefaulting to 2020, but this will stop working in the future.");
      }
      options.ecmaVersion = 11;
    } else if (options.ecmaVersion >= 2015) {
      options.ecmaVersion -= 2009;
    }

    if (options.allowReserved == null)
      { options.allowReserved = options.ecmaVersion < 5; }

    if (!opts || opts.allowHashBang == null)
      { options.allowHashBang = options.ecmaVersion >= 14; }

    if (isArray$1(options.onToken)) {
      var tokens = options.onToken;
      options.onToken = function (token) { return tokens.push(token); };
    }
    if (isArray$1(options.onComment))
      { options.onComment = pushComment(options, options.onComment); }

    return options
  }

  function pushComment(options, array) {
    return function(block, text, start, end, startLoc, endLoc) {
      var comment = {
        type: block ? "Block" : "Line",
        value: text,
        start: start,
        end: end
      };
      if (options.locations)
        { comment.loc = new SourceLocation(this, startLoc, endLoc); }
      if (options.ranges)
        { comment.range = [start, end]; }
      array.push(comment);
    }
  }

  // Each scope gets a bitset that may contain these flags
  var
      SCOPE_TOP = 1,
      SCOPE_FUNCTION = 2,
      SCOPE_ASYNC = 4,
      SCOPE_GENERATOR = 8,
      SCOPE_ARROW = 16,
      SCOPE_SIMPLE_CATCH = 32,
      SCOPE_SUPER = 64,
      SCOPE_DIRECT_SUPER = 128,
      SCOPE_CLASS_STATIC_BLOCK = 256,
      SCOPE_VAR = SCOPE_TOP | SCOPE_FUNCTION | SCOPE_CLASS_STATIC_BLOCK;

  function functionFlags(async, generator) {
    return SCOPE_FUNCTION | (async ? SCOPE_ASYNC : 0) | (generator ? SCOPE_GENERATOR : 0)
  }

  // Used in checkLVal* and declareName to determine the type of a binding
  var
      BIND_NONE = 0, // Not a binding
      BIND_VAR = 1, // Var-style binding
      BIND_LEXICAL = 2, // Let- or const-style binding
      BIND_FUNCTION = 3, // Function declaration
      BIND_SIMPLE_CATCH = 4, // Simple (identifier pattern) catch binding
      BIND_OUTSIDE = 5; // Special case for function names as bound inside the function

  var Parser = function Parser(options, input, startPos) {
    this.options = options = getOptions(options);
    this.sourceFile = options.sourceFile;
    this.keywords = wordsRegexp(keywords$1[options.ecmaVersion >= 6 ? 6 : options.sourceType === "module" ? "5module" : 5]);
    var reserved = "";
    if (options.allowReserved !== true) {
      reserved = reservedWords[options.ecmaVersion >= 6 ? 6 : options.ecmaVersion === 5 ? 5 : 3];
      if (options.sourceType === "module") { reserved += " await"; }
    }
    this.reservedWords = wordsRegexp(reserved);
    var reservedStrict = (reserved ? reserved + " " : "") + reservedWords.strict;
    this.reservedWordsStrict = wordsRegexp(reservedStrict);
    this.reservedWordsStrictBind = wordsRegexp(reservedStrict + " " + reservedWords.strictBind);
    this.input = String(input);

    // Used to signal to callers of `readWord1` whether the word
    // contained any escape sequences. This is needed because words with
    // escape sequences must not be interpreted as keywords.
    this.containsEsc = false;

    // Set up token state

    // The current position of the tokenizer in the input.
    if (startPos) {
      this.pos = startPos;
      this.lineStart = this.input.lastIndexOf("\n", startPos - 1) + 1;
      this.curLine = this.input.slice(0, this.lineStart).split(lineBreak).length;
    } else {
      this.pos = this.lineStart = 0;
      this.curLine = 1;
    }

    // Properties of the current token:
    // Its type
    this.type = types$1.eof;
    // For tokens that include more information than their type, the value
    this.value = null;
    // Its start and end offset
    this.start = this.end = this.pos;
    // And, if locations are used, the {line, column} object
    // corresponding to those offsets
    this.startLoc = this.endLoc = this.curPosition();

    // Position information for the previous token
    this.lastTokEndLoc = this.lastTokStartLoc = null;
    this.lastTokStart = this.lastTokEnd = this.pos;

    // The context stack is used to superficially track syntactic
    // context to predict whether a regular expression is allowed in a
    // given position.
    this.context = this.initialContext();
    this.exprAllowed = true;

    // Figure out if it's a module code.
    this.inModule = options.sourceType === "module";
    this.strict = this.inModule || this.strictDirective(this.pos);

    // Used to signify the start of a potential arrow function
    this.potentialArrowAt = -1;
    this.potentialArrowInForAwait = false;

    // Positions to delayed-check that yield/await does not exist in default parameters.
    this.yieldPos = this.awaitPos = this.awaitIdentPos = 0;
    // Labels in scope.
    this.labels = [];
    // Thus-far undefined exports.
    this.undefinedExports = Object.create(null);

    // If enabled, skip leading hashbang line.
    if (this.pos === 0 && options.allowHashBang && this.input.slice(0, 2) === "#!")
      { this.skipLineComment(2); }

    // Scope tracking for duplicate variable names (see scope.js)
    this.scopeStack = [];
    this.enterScope(SCOPE_TOP);

    // For RegExp validation
    this.regexpState = null;

    // The stack of private names.
    // Each element has two properties: 'declared' and 'used'.
    // When it exited from the outermost class definition, all used private names must be declared.
    this.privateNameStack = [];
  };

  var prototypeAccessors = { inFunction: { configurable: true },inGenerator: { configurable: true },inAsync: { configurable: true },canAwait: { configurable: true },allowSuper: { configurable: true },allowDirectSuper: { configurable: true },treatFunctionsAsVar: { configurable: true },allowNewDotTarget: { configurable: true },inClassStaticBlock: { configurable: true } };

  Parser.prototype.parse = function parse () {
    var node = this.options.program || this.startNode();
    this.nextToken();
    return this.parseTopLevel(node)
  };

  prototypeAccessors.inFunction.get = function () { return (this.currentVarScope().flags & SCOPE_FUNCTION) > 0 };

  prototypeAccessors.inGenerator.get = function () { return (this.currentVarScope().flags & SCOPE_GENERATOR) > 0 && !this.currentVarScope().inClassFieldInit };

  prototypeAccessors.inAsync.get = function () { return (this.currentVarScope().flags & SCOPE_ASYNC) > 0 && !this.currentVarScope().inClassFieldInit };

  prototypeAccessors.canAwait.get = function () {
    for (var i = this.scopeStack.length - 1; i >= 0; i--) {
      var scope = this.scopeStack[i];
      if (scope.inClassFieldInit || scope.flags & SCOPE_CLASS_STATIC_BLOCK) { return false }
      if (scope.flags & SCOPE_FUNCTION) { return (scope.flags & SCOPE_ASYNC) > 0 }
    }
    return (this.inModule && this.options.ecmaVersion >= 13) || this.options.allowAwaitOutsideFunction
  };

  prototypeAccessors.allowSuper.get = function () {
    var ref = this.currentThisScope();
      var flags = ref.flags;
      var inClassFieldInit = ref.inClassFieldInit;
    return (flags & SCOPE_SUPER) > 0 || inClassFieldInit || this.options.allowSuperOutsideMethod
  };

  prototypeAccessors.allowDirectSuper.get = function () { return (this.currentThisScope().flags & SCOPE_DIRECT_SUPER) > 0 };

  prototypeAccessors.treatFunctionsAsVar.get = function () { return this.treatFunctionsAsVarInScope(this.currentScope()) };

  prototypeAccessors.allowNewDotTarget.get = function () {
    var ref = this.currentThisScope();
      var flags = ref.flags;
      var inClassFieldInit = ref.inClassFieldInit;
    return (flags & (SCOPE_FUNCTION | SCOPE_CLASS_STATIC_BLOCK)) > 0 || inClassFieldInit
  };

  prototypeAccessors.inClassStaticBlock.get = function () {
    return (this.currentVarScope().flags & SCOPE_CLASS_STATIC_BLOCK) > 0
  };

  Parser.extend = function extend () {
      var plugins = [], len = arguments.length;
      while ( len-- ) plugins[ len ] = arguments[ len ];

    var cls = this;
    for (var i = 0; i < plugins.length; i++) { cls = plugins[i](cls); }
    return cls
  };

  Parser.parse = function parse (input, options) {
    return new this(options, input).parse()
  };

  Parser.parseExpressionAt = function parseExpressionAt (input, pos, options) {
    var parser = new this(options, input, pos);
    parser.nextToken();
    return parser.parseExpression()
  };

  Parser.tokenizer = function tokenizer (input, options) {
    return new this(options, input)
  };

  Object.defineProperties( Parser.prototype, prototypeAccessors );

  var pp$9 = Parser.prototype;

  // ## Parser utilities

  var literal = /^(?:'((?:\\[^]|[^'\\])*?)'|"((?:\\[^]|[^"\\])*?)")/;
  pp$9.strictDirective = function(start) {
    if (this.options.ecmaVersion < 5) { return false }
    for (;;) {
      // Try to find string literal.
      skipWhiteSpace.lastIndex = start;
      start += skipWhiteSpace.exec(this.input)[0].length;
      var match = literal.exec(this.input.slice(start));
      if (!match) { return false }
      if ((match[1] || match[2]) === "use strict") {
        skipWhiteSpace.lastIndex = start + match[0].length;
        var spaceAfter = skipWhiteSpace.exec(this.input), end = spaceAfter.index + spaceAfter[0].length;
        var next = this.input.charAt(end);
        return next === ";" || next === "}" ||
          (lineBreak.test(spaceAfter[0]) &&
           !(/[(`.[+\-/*%<>=,?^&]/.test(next) || next === "!" && this.input.charAt(end + 1) === "="))
      }
      start += match[0].length;

      // Skip semicolon, if any.
      skipWhiteSpace.lastIndex = start;
      start += skipWhiteSpace.exec(this.input)[0].length;
      if (this.input[start] === ";")
        { start++; }
    }
  };

  // Predicate that tests whether the next token is of the given
  // type, and if yes, consumes it as a side effect.

  pp$9.eat = function(type) {
    if (this.type === type) {
      this.next();
      return true
    } else {
      return false
    }
  };

  // Tests whether parsed token is a contextual keyword.

  pp$9.isContextual = function(name) {
    return this.type === types$1.name && this.value === name && !this.containsEsc
  };

  // Consumes contextual keyword if possible.

  pp$9.eatContextual = function(name) {
    if (!this.isContextual(name)) { return false }
    this.next();
    return true
  };

  // Asserts that following token is given contextual keyword.

  pp$9.expectContextual = function(name) {
    if (!this.eatContextual(name)) { this.unexpected(); }
  };

  // Test whether a semicolon can be inserted at the current position.

  pp$9.canInsertSemicolon = function() {
    return this.type === types$1.eof ||
      this.type === types$1.braceR ||
      lineBreak.test(this.input.slice(this.lastTokEnd, this.start))
  };

  pp$9.insertSemicolon = function() {
    if (this.canInsertSemicolon()) {
      if (this.options.onInsertedSemicolon)
        { this.options.onInsertedSemicolon(this.lastTokEnd, this.lastTokEndLoc); }
      return true
    }
  };

  // Consume a semicolon, or, failing that, see if we are allowed to
  // pretend that there is a semicolon at this position.

  pp$9.semicolon = function() {
    if (!this.eat(types$1.semi) && !this.insertSemicolon()) { this.unexpected(); }
  };

  pp$9.afterTrailingComma = function(tokType, notNext) {
    if (this.type === tokType) {
      if (this.options.onTrailingComma)
        { this.options.onTrailingComma(this.lastTokStart, this.lastTokStartLoc); }
      if (!notNext)
        { this.next(); }
      return true
    }
  };

  // Expect a token of a given type. If found, consume it, otherwise,
  // raise an unexpected token error.

  pp$9.expect = function(type) {
    this.eat(type) || this.unexpected();
  };

  // Raise an unexpected token error.

  pp$9.unexpected = function(pos) {
    this.raise(pos != null ? pos : this.start, "Unexpected token");
  };

  var DestructuringErrors = function DestructuringErrors() {
    this.shorthandAssign =
    this.trailingComma =
    this.parenthesizedAssign =
    this.parenthesizedBind =
    this.doubleProto =
      -1;
  };

  pp$9.checkPatternErrors = function(refDestructuringErrors, isAssign) {
    if (!refDestructuringErrors) { return }
    if (refDestructuringErrors.trailingComma > -1)
      { this.raiseRecoverable(refDestructuringErrors.trailingComma, "Comma is not permitted after the rest element"); }
    var parens = isAssign ? refDestructuringErrors.parenthesizedAssign : refDestructuringErrors.parenthesizedBind;
    if (parens > -1) { this.raiseRecoverable(parens, isAssign ? "Assigning to rvalue" : "Parenthesized pattern"); }
  };

  pp$9.checkExpressionErrors = function(refDestructuringErrors, andThrow) {
    if (!refDestructuringErrors) { return false }
    var shorthandAssign = refDestructuringErrors.shorthandAssign;
    var doubleProto = refDestructuringErrors.doubleProto;
    if (!andThrow) { return shorthandAssign >= 0 || doubleProto >= 0 }
    if (shorthandAssign >= 0)
      { this.raise(shorthandAssign, "Shorthand property assignments are valid only in destructuring patterns"); }
    if (doubleProto >= 0)
      { this.raiseRecoverable(doubleProto, "Redefinition of __proto__ property"); }
  };

  pp$9.checkYieldAwaitInDefaultParams = function() {
    if (this.yieldPos && (!this.awaitPos || this.yieldPos < this.awaitPos))
      { this.raise(this.yieldPos, "Yield expression cannot be a default value"); }
    if (this.awaitPos)
      { this.raise(this.awaitPos, "Await expression cannot be a default value"); }
  };

  pp$9.isSimpleAssignTarget = function(expr) {
    if (expr.type === "ParenthesizedExpression")
      { return this.isSimpleAssignTarget(expr.expression) }
    return expr.type === "Identifier" || expr.type === "MemberExpression"
  };

  var pp$8 = Parser.prototype;

  // ### Statement parsing

  // Parse a program. Initializes the parser, reads any number of
  // statements, and wraps them in a Program node.  Optionally takes a
  // `program` argument.  If present, the statements will be appended
  // to its body instead of creating a new node.

  pp$8.parseTopLevel = function(node) {
    var exports = Object.create(null);
    if (!node.body) { node.body = []; }
    while (this.type !== types$1.eof) {
      var stmt = this.parseStatement(null, true, exports);
      node.body.push(stmt);
    }
    if (this.inModule)
      { for (var i = 0, list = Object.keys(this.undefinedExports); i < list.length; i += 1)
        {
          var name = list[i];

          this.raiseRecoverable(this.undefinedExports[name].start, ("Export '" + name + "' is not defined"));
        } }
    this.adaptDirectivePrologue(node.body);
    this.next();
    node.sourceType = this.options.sourceType;
    return this.finishNode(node, "Program")
  };

  var loopLabel = {kind: "loop"}, switchLabel = {kind: "switch"};

  pp$8.isLet = function(context) {
    if (this.options.ecmaVersion < 6 || !this.isContextual("let")) { return false }
    skipWhiteSpace.lastIndex = this.pos;
    var skip = skipWhiteSpace.exec(this.input);
    var next = this.pos + skip[0].length, nextCh = this.input.charCodeAt(next);
    // For ambiguous cases, determine if a LexicalDeclaration (or only a
    // Statement) is allowed here. If context is not empty then only a Statement
    // is allowed. However, `let [` is an explicit negative lookahead for
    // ExpressionStatement, so special-case it first.
    if (nextCh === 91 || nextCh === 92) { return true } // '[', '\'
    if (context) { return false }

    if (nextCh === 123 || nextCh > 0xd7ff && nextCh < 0xdc00) { return true } // '{', astral
    if (isIdentifierStart(nextCh, true)) {
      var pos = next + 1;
      while (isIdentifierChar(nextCh = this.input.charCodeAt(pos), true)) { ++pos; }
      if (nextCh === 92 || nextCh > 0xd7ff && nextCh < 0xdc00) { return true }
      var ident = this.input.slice(next, pos);
      if (!keywordRelationalOperator.test(ident)) { return true }
    }
    return false
  };

  // check 'async [no LineTerminator here] function'
  // - 'async /*foo*/ function' is OK.
  // - 'async /*\n*/ function' is invalid.
  pp$8.isAsyncFunction = function() {
    if (this.options.ecmaVersion < 8 || !this.isContextual("async"))
      { return false }

    skipWhiteSpace.lastIndex = this.pos;
    var skip = skipWhiteSpace.exec(this.input);
    var next = this.pos + skip[0].length, after;
    return !lineBreak.test(this.input.slice(this.pos, next)) &&
      this.input.slice(next, next + 8) === "function" &&
      (next + 8 === this.input.length ||
       !(isIdentifierChar(after = this.input.charCodeAt(next + 8)) || after > 0xd7ff && after < 0xdc00))
  };

  // Parse a single statement.
  //
  // If expecting a statement and finding a slash operator, parse a
  // regular expression literal. This is to handle cases like
  // `if (foo) /blah/.exec(foo)`, where looking at the previous token
  // does not help.

  pp$8.parseStatement = function(context, topLevel, exports) {
    var starttype = this.type, node = this.startNode(), kind;

    if (this.isLet(context)) {
      starttype = types$1._var;
      kind = "let";
    }

    // Most types of statements are recognized by the keyword they
    // start with. Many are trivial to parse, some require a bit of
    // complexity.

    switch (starttype) {
    case types$1._break: case types$1._continue: return this.parseBreakContinueStatement(node, starttype.keyword)
    case types$1._debugger: return this.parseDebuggerStatement(node)
    case types$1._do: return this.parseDoStatement(node)
    case types$1._for: return this.parseForStatement(node)
    case types$1._function:
      // Function as sole body of either an if statement or a labeled statement
      // works, but not when it is part of a labeled statement that is the sole
      // body of an if statement.
      if ((context && (this.strict || context !== "if" && context !== "label")) && this.options.ecmaVersion >= 6) { this.unexpected(); }
      return this.parseFunctionStatement(node, false, !context)
    case types$1._class:
      if (context) { this.unexpected(); }
      return this.parseClass(node, true)
    case types$1._if: return this.parseIfStatement(node)
    case types$1._return: return this.parseReturnStatement(node)
    case types$1._switch: return this.parseSwitchStatement(node)
    case types$1._throw: return this.parseThrowStatement(node)
    case types$1._try: return this.parseTryStatement(node)
    case types$1._const: case types$1._var:
      kind = kind || this.value;
      if (context && kind !== "var") { this.unexpected(); }
      return this.parseVarStatement(node, kind)
    case types$1._while: return this.parseWhileStatement(node)
    case types$1._with: return this.parseWithStatement(node)
    case types$1.braceL: return this.parseBlock(true, node)
    case types$1.semi: return this.parseEmptyStatement(node)
    case types$1._export:
    case types$1._import:
      if (this.options.ecmaVersion > 10 && starttype === types$1._import) {
        skipWhiteSpace.lastIndex = this.pos;
        var skip = skipWhiteSpace.exec(this.input);
        var next = this.pos + skip[0].length, nextCh = this.input.charCodeAt(next);
        if (nextCh === 40 || nextCh === 46) // '(' or '.'
          { return this.parseExpressionStatement(node, this.parseExpression()) }
      }

      if (!this.options.allowImportExportEverywhere) {
        if (!topLevel)
          { this.raise(this.start, "'import' and 'export' may only appear at the top level"); }
        if (!this.inModule)
          { this.raise(this.start, "'import' and 'export' may appear only with 'sourceType: module'"); }
      }
      return starttype === types$1._import ? this.parseImport(node) : this.parseExport(node, exports)

      // If the statement does not start with a statement keyword or a
      // brace, it's an ExpressionStatement or LabeledStatement. We
      // simply start parsing an expression, and afterwards, if the
      // next token is a colon and the expression was a simple
      // Identifier node, we switch to interpreting it as a label.
    default:
      if (this.isAsyncFunction()) {
        if (context) { this.unexpected(); }
        this.next();
        return this.parseFunctionStatement(node, true, !context)
      }

      var maybeName = this.value, expr = this.parseExpression();
      if (starttype === types$1.name && expr.type === "Identifier" && this.eat(types$1.colon))
        { return this.parseLabeledStatement(node, maybeName, expr, context) }
      else { return this.parseExpressionStatement(node, expr) }
    }
  };

  pp$8.parseBreakContinueStatement = function(node, keyword) {
    var isBreak = keyword === "break";
    this.next();
    if (this.eat(types$1.semi) || this.insertSemicolon()) { node.label = null; }
    else if (this.type !== types$1.name) { this.unexpected(); }
    else {
      node.label = this.parseIdent();
      this.semicolon();
    }

    // Verify that there is an actual destination to break or
    // continue to.
    var i = 0;
    for (; i < this.labels.length; ++i) {
      var lab = this.labels[i];
      if (node.label == null || lab.name === node.label.name) {
        if (lab.kind != null && (isBreak || lab.kind === "loop")) { break }
        if (node.label && isBreak) { break }
      }
    }
    if (i === this.labels.length) { this.raise(node.start, "Unsyntactic " + keyword); }
    return this.finishNode(node, isBreak ? "BreakStatement" : "ContinueStatement")
  };

  pp$8.parseDebuggerStatement = function(node) {
    this.next();
    this.semicolon();
    return this.finishNode(node, "DebuggerStatement")
  };

  pp$8.parseDoStatement = function(node) {
    this.next();
    this.labels.push(loopLabel);
    node.body = this.parseStatement("do");
    this.labels.pop();
    this.expect(types$1._while);
    node.test = this.parseParenExpression();
    if (this.options.ecmaVersion >= 6)
      { this.eat(types$1.semi); }
    else
      { this.semicolon(); }
    return this.finishNode(node, "DoWhileStatement")
  };

  // Disambiguating between a `for` and a `for`/`in` or `for`/`of`
  // loop is non-trivial. Basically, we have to parse the init `var`
  // statement or expression, disallowing the `in` operator (see
  // the second parameter to `parseExpression`), and then check
  // whether the next token is `in` or `of`. When there is no init
  // part (semicolon immediately after the opening parenthesis), it
  // is a regular `for` loop.

  pp$8.parseForStatement = function(node) {
    this.next();
    var awaitAt = (this.options.ecmaVersion >= 9 && this.canAwait && this.eatContextual("await")) ? this.lastTokStart : -1;
    this.labels.push(loopLabel);
    this.enterScope(0);
    this.expect(types$1.parenL);
    if (this.type === types$1.semi) {
      if (awaitAt > -1) { this.unexpected(awaitAt); }
      return this.parseFor(node, null)
    }
    var isLet = this.isLet();
    if (this.type === types$1._var || this.type === types$1._const || isLet) {
      var init$1 = this.startNode(), kind = isLet ? "let" : this.value;
      this.next();
      this.parseVar(init$1, true, kind);
      this.finishNode(init$1, "VariableDeclaration");
      if ((this.type === types$1._in || (this.options.ecmaVersion >= 6 && this.isContextual("of"))) && init$1.declarations.length === 1) {
        if (this.options.ecmaVersion >= 9) {
          if (this.type === types$1._in) {
            if (awaitAt > -1) { this.unexpected(awaitAt); }
          } else { node.await = awaitAt > -1; }
        }
        return this.parseForIn(node, init$1)
      }
      if (awaitAt > -1) { this.unexpected(awaitAt); }
      return this.parseFor(node, init$1)
    }
    var startsWithLet = this.isContextual("let"), isForOf = false;
    var containsEsc = this.containsEsc;
    var refDestructuringErrors = new DestructuringErrors;
    var initPos = this.start;
    var init = awaitAt > -1
      ? this.parseExprSubscripts(refDestructuringErrors, "await")
      : this.parseExpression(true, refDestructuringErrors);
    if (this.type === types$1._in || (isForOf = this.options.ecmaVersion >= 6 && this.isContextual("of"))) {
      if (awaitAt > -1) { // implies `ecmaVersion >= 9` (see declaration of awaitAt)
        if (this.type === types$1._in) { this.unexpected(awaitAt); }
        node.await = true;
      } else if (isForOf && this.options.ecmaVersion >= 8) {
        if (init.start === initPos && !containsEsc && init.type === "Identifier" && init.name === "async") { this.unexpected(); }
        else if (this.options.ecmaVersion >= 9) { node.await = false; }
      }
      if (startsWithLet && isForOf) { this.raise(init.start, "The left-hand side of a for-of loop may not start with 'let'."); }
      this.toAssignable(init, false, refDestructuringErrors);
      this.checkLValPattern(init);
      return this.parseForIn(node, init)
    } else {
      this.checkExpressionErrors(refDestructuringErrors, true);
    }
    if (awaitAt > -1) { this.unexpected(awaitAt); }
    return this.parseFor(node, init)
  };

  pp$8.parseFunctionStatement = function(node, isAsync, declarationPosition) {
    this.next();
    return this.parseFunction(node, FUNC_STATEMENT | (declarationPosition ? 0 : FUNC_HANGING_STATEMENT), false, isAsync)
  };

  pp$8.parseIfStatement = function(node) {
    this.next();
    node.test = this.parseParenExpression();
    // allow function declarations in branches, but only in non-strict mode
    node.consequent = this.parseStatement("if");
    node.alternate = this.eat(types$1._else) ? this.parseStatement("if") : null;
    return this.finishNode(node, "IfStatement")
  };

  pp$8.parseReturnStatement = function(node) {
    if (!this.inFunction && !this.options.allowReturnOutsideFunction)
      { this.raise(this.start, "'return' outside of function"); }
    this.next();

    // In `return` (and `break`/`continue`), the keywords with
    // optional arguments, we eagerly look for a semicolon or the
    // possibility to insert one.

    if (this.eat(types$1.semi) || this.insertSemicolon()) { node.argument = null; }
    else { node.argument = this.parseExpression(); this.semicolon(); }
    return this.finishNode(node, "ReturnStatement")
  };

  pp$8.parseSwitchStatement = function(node) {
    this.next();
    node.discriminant = this.parseParenExpression();
    node.cases = [];
    this.expect(types$1.braceL);
    this.labels.push(switchLabel);
    this.enterScope(0);

    // Statements under must be grouped (by label) in SwitchCase
    // nodes. `cur` is used to keep the node that we are currently
    // adding statements to.

    var cur;
    for (var sawDefault = false; this.type !== types$1.braceR;) {
      if (this.type === types$1._case || this.type === types$1._default) {
        var isCase = this.type === types$1._case;
        if (cur) { this.finishNode(cur, "SwitchCase"); }
        node.cases.push(cur = this.startNode());
        cur.consequent = [];
        this.next();
        if (isCase) {
          cur.test = this.parseExpression();
        } else {
          if (sawDefault) { this.raiseRecoverable(this.lastTokStart, "Multiple default clauses"); }
          sawDefault = true;
          cur.test = null;
        }
        this.expect(types$1.colon);
      } else {
        if (!cur) { this.unexpected(); }
        cur.consequent.push(this.parseStatement(null));
      }
    }
    this.exitScope();
    if (cur) { this.finishNode(cur, "SwitchCase"); }
    this.next(); // Closing brace
    this.labels.pop();
    return this.finishNode(node, "SwitchStatement")
  };

  pp$8.parseThrowStatement = function(node) {
    this.next();
    if (lineBreak.test(this.input.slice(this.lastTokEnd, this.start)))
      { this.raise(this.lastTokEnd, "Illegal newline after throw"); }
    node.argument = this.parseExpression();
    this.semicolon();
    return this.finishNode(node, "ThrowStatement")
  };

  // Reused empty array added for node fields that are always empty.

  var empty$1 = [];

  pp$8.parseCatchClauseParam = function() {
    var param = this.parseBindingAtom();
    var simple = param.type === "Identifier";
    this.enterScope(simple ? SCOPE_SIMPLE_CATCH : 0);
    this.checkLValPattern(param, simple ? BIND_SIMPLE_CATCH : BIND_LEXICAL);
    this.expect(types$1.parenR);

    return param
  };

  pp$8.parseTryStatement = function(node) {
    this.next();
    node.block = this.parseBlock();
    node.handler = null;
    if (this.type === types$1._catch) {
      var clause = this.startNode();
      this.next();
      if (this.eat(types$1.parenL)) {
        clause.param = this.parseCatchClauseParam();
      } else {
        if (this.options.ecmaVersion < 10) { this.unexpected(); }
        clause.param = null;
        this.enterScope(0);
      }
      clause.body = this.parseBlock(false);
      this.exitScope();
      node.handler = this.finishNode(clause, "CatchClause");
    }
    node.finalizer = this.eat(types$1._finally) ? this.parseBlock() : null;
    if (!node.handler && !node.finalizer)
      { this.raise(node.start, "Missing catch or finally clause"); }
    return this.finishNode(node, "TryStatement")
  };

  pp$8.parseVarStatement = function(node, kind, allowMissingInitializer) {
    this.next();
    this.parseVar(node, false, kind, allowMissingInitializer);
    this.semicolon();
    return this.finishNode(node, "VariableDeclaration")
  };

  pp$8.parseWhileStatement = function(node) {
    this.next();
    node.test = this.parseParenExpression();
    this.labels.push(loopLabel);
    node.body = this.parseStatement("while");
    this.labels.pop();
    return this.finishNode(node, "WhileStatement")
  };

  pp$8.parseWithStatement = function(node) {
    if (this.strict) { this.raise(this.start, "'with' in strict mode"); }
    this.next();
    node.object = this.parseParenExpression();
    node.body = this.parseStatement("with");
    return this.finishNode(node, "WithStatement")
  };

  pp$8.parseEmptyStatement = function(node) {
    this.next();
    return this.finishNode(node, "EmptyStatement")
  };

  pp$8.parseLabeledStatement = function(node, maybeName, expr, context) {
    for (var i$1 = 0, list = this.labels; i$1 < list.length; i$1 += 1)
      {
      var label = list[i$1];

      if (label.name === maybeName)
        { this.raise(expr.start, "Label '" + maybeName + "' is already declared");
    } }
    var kind = this.type.isLoop ? "loop" : this.type === types$1._switch ? "switch" : null;
    for (var i = this.labels.length - 1; i >= 0; i--) {
      var label$1 = this.labels[i];
      if (label$1.statementStart === node.start) {
        // Update information about previous labels on this node
        label$1.statementStart = this.start;
        label$1.kind = kind;
      } else { break }
    }
    this.labels.push({name: maybeName, kind: kind, statementStart: this.start});
    node.body = this.parseStatement(context ? context.indexOf("label") === -1 ? context + "label" : context : "label");
    this.labels.pop();
    node.label = expr;
    return this.finishNode(node, "LabeledStatement")
  };

  pp$8.parseExpressionStatement = function(node, expr) {
    node.expression = expr;
    this.semicolon();
    return this.finishNode(node, "ExpressionStatement")
  };

  // Parse a semicolon-enclosed block of statements, handling `"use
  // strict"` declarations when `allowStrict` is true (used for
  // function bodies).

  pp$8.parseBlock = function(createNewLexicalScope, node, exitStrict) {
    if ( createNewLexicalScope === void 0 ) createNewLexicalScope = true;
    if ( node === void 0 ) node = this.startNode();

    node.body = [];
    this.expect(types$1.braceL);
    if (createNewLexicalScope) { this.enterScope(0); }
    while (this.type !== types$1.braceR) {
      var stmt = this.parseStatement(null);
      node.body.push(stmt);
    }
    if (exitStrict) { this.strict = false; }
    this.next();
    if (createNewLexicalScope) { this.exitScope(); }
    return this.finishNode(node, "BlockStatement")
  };

  // Parse a regular `for` loop. The disambiguation code in
  // `parseStatement` will already have parsed the init statement or
  // expression.

  pp$8.parseFor = function(node, init) {
    node.init = init;
    this.expect(types$1.semi);
    node.test = this.type === types$1.semi ? null : this.parseExpression();
    this.expect(types$1.semi);
    node.update = this.type === types$1.parenR ? null : this.parseExpression();
    this.expect(types$1.parenR);
    node.body = this.parseStatement("for");
    this.exitScope();
    this.labels.pop();
    return this.finishNode(node, "ForStatement")
  };

  // Parse a `for`/`in` and `for`/`of` loop, which are almost
  // same from parser's perspective.

  pp$8.parseForIn = function(node, init) {
    var isForIn = this.type === types$1._in;
    this.next();

    if (
      init.type === "VariableDeclaration" &&
      init.declarations[0].init != null &&
      (
        !isForIn ||
        this.options.ecmaVersion < 8 ||
        this.strict ||
        init.kind !== "var" ||
        init.declarations[0].id.type !== "Identifier"
      )
    ) {
      this.raise(
        init.start,
        ((isForIn ? "for-in" : "for-of") + " loop variable declaration may not have an initializer")
      );
    }
    node.left = init;
    node.right = isForIn ? this.parseExpression() : this.parseMaybeAssign();
    this.expect(types$1.parenR);
    node.body = this.parseStatement("for");
    this.exitScope();
    this.labels.pop();
    return this.finishNode(node, isForIn ? "ForInStatement" : "ForOfStatement")
  };

  // Parse a list of variable declarations.

  pp$8.parseVar = function(node, isFor, kind, allowMissingInitializer) {
    node.declarations = [];
    node.kind = kind;
    for (;;) {
      var decl = this.startNode();
      this.parseVarId(decl, kind);
      if (this.eat(types$1.eq)) {
        decl.init = this.parseMaybeAssign(isFor);
      } else if (!allowMissingInitializer && kind === "const" && !(this.type === types$1._in || (this.options.ecmaVersion >= 6 && this.isContextual("of")))) {
        this.unexpected();
      } else if (!allowMissingInitializer && decl.id.type !== "Identifier" && !(isFor && (this.type === types$1._in || this.isContextual("of")))) {
        this.raise(this.lastTokEnd, "Complex binding patterns require an initialization value");
      } else {
        decl.init = null;
      }
      node.declarations.push(this.finishNode(decl, "VariableDeclarator"));
      if (!this.eat(types$1.comma)) { break }
    }
    return node
  };

  pp$8.parseVarId = function(decl, kind) {
    decl.id = this.parseBindingAtom();
    this.checkLValPattern(decl.id, kind === "var" ? BIND_VAR : BIND_LEXICAL, false);
  };

  var FUNC_STATEMENT = 1, FUNC_HANGING_STATEMENT = 2, FUNC_NULLABLE_ID = 4;

  // Parse a function declaration or literal (depending on the
  // `statement & FUNC_STATEMENT`).

  // Remove `allowExpressionBody` for 7.0.0, as it is only called with false
  pp$8.parseFunction = function(node, statement, allowExpressionBody, isAsync, forInit) {
    this.initFunction(node);
    if (this.options.ecmaVersion >= 9 || this.options.ecmaVersion >= 6 && !isAsync) {
      if (this.type === types$1.star && (statement & FUNC_HANGING_STATEMENT))
        { this.unexpected(); }
      node.generator = this.eat(types$1.star);
    }
    if (this.options.ecmaVersion >= 8)
      { node.async = !!isAsync; }

    if (statement & FUNC_STATEMENT) {
      node.id = (statement & FUNC_NULLABLE_ID) && this.type !== types$1.name ? null : this.parseIdent();
      if (node.id && !(statement & FUNC_HANGING_STATEMENT))
        // If it is a regular function declaration in sloppy mode, then it is
        // subject to Annex B semantics (BIND_FUNCTION). Otherwise, the binding
        // mode depends on properties of the current scope (see
        // treatFunctionsAsVar).
        { this.checkLValSimple(node.id, (this.strict || node.generator || node.async) ? this.treatFunctionsAsVar ? BIND_VAR : BIND_LEXICAL : BIND_FUNCTION); }
    }

    var oldYieldPos = this.yieldPos, oldAwaitPos = this.awaitPos, oldAwaitIdentPos = this.awaitIdentPos;
    this.yieldPos = 0;
    this.awaitPos = 0;
    this.awaitIdentPos = 0;
    this.enterScope(functionFlags(node.async, node.generator));

    if (!(statement & FUNC_STATEMENT))
      { node.id = this.type === types$1.name ? this.parseIdent() : null; }

    this.parseFunctionParams(node);
    this.parseFunctionBody(node, allowExpressionBody, false, forInit);

    this.yieldPos = oldYieldPos;
    this.awaitPos = oldAwaitPos;
    this.awaitIdentPos = oldAwaitIdentPos;
    return this.finishNode(node, (statement & FUNC_STATEMENT) ? "FunctionDeclaration" : "FunctionExpression")
  };

  pp$8.parseFunctionParams = function(node) {
    this.expect(types$1.parenL);
    node.params = this.parseBindingList(types$1.parenR, false, this.options.ecmaVersion >= 8);
    this.checkYieldAwaitInDefaultParams();
  };

  // Parse a class declaration or literal (depending on the
  // `isStatement` parameter).

  pp$8.parseClass = function(node, isStatement) {
    this.next();

    // ecma-262 14.6 Class Definitions
    // A class definition is always strict mode code.
    var oldStrict = this.strict;
    this.strict = true;

    this.parseClassId(node, isStatement);
    this.parseClassSuper(node);
    var privateNameMap = this.enterClassBody();
    var classBody = this.startNode();
    var hadConstructor = false;
    classBody.body = [];
    this.expect(types$1.braceL);
    while (this.type !== types$1.braceR) {
      var element = this.parseClassElement(node.superClass !== null);
      if (element) {
        classBody.body.push(element);
        if (element.type === "MethodDefinition" && element.kind === "constructor") {
          if (hadConstructor) { this.raiseRecoverable(element.start, "Duplicate constructor in the same class"); }
          hadConstructor = true;
        } else if (element.key && element.key.type === "PrivateIdentifier" && isPrivateNameConflicted(privateNameMap, element)) {
          this.raiseRecoverable(element.key.start, ("Identifier '#" + (element.key.name) + "' has already been declared"));
        }
      }
    }
    this.strict = oldStrict;
    this.next();
    node.body = this.finishNode(classBody, "ClassBody");
    this.exitClassBody();
    return this.finishNode(node, isStatement ? "ClassDeclaration" : "ClassExpression")
  };

  pp$8.parseClassElement = function(constructorAllowsSuper) {
    if (this.eat(types$1.semi)) { return null }

    var ecmaVersion = this.options.ecmaVersion;
    var node = this.startNode();
    var keyName = "";
    var isGenerator = false;
    var isAsync = false;
    var kind = "method";
    var isStatic = false;

    if (this.eatContextual("static")) {
      // Parse static init block
      if (ecmaVersion >= 13 && this.eat(types$1.braceL)) {
        this.parseClassStaticBlock(node);
        return node
      }
      if (this.isClassElementNameStart() || this.type === types$1.star) {
        isStatic = true;
      } else {
        keyName = "static";
      }
    }
    node.static = isStatic;
    if (!keyName && ecmaVersion >= 8 && this.eatContextual("async")) {
      if ((this.isClassElementNameStart() || this.type === types$1.star) && !this.canInsertSemicolon()) {
        isAsync = true;
      } else {
        keyName = "async";
      }
    }
    if (!keyName && (ecmaVersion >= 9 || !isAsync) && this.eat(types$1.star)) {
      isGenerator = true;
    }
    if (!keyName && !isAsync && !isGenerator) {
      var lastValue = this.value;
      if (this.eatContextual("get") || this.eatContextual("set")) {
        if (this.isClassElementNameStart()) {
          kind = lastValue;
        } else {
          keyName = lastValue;
        }
      }
    }

    // Parse element name
    if (keyName) {
      // 'async', 'get', 'set', or 'static' were not a keyword contextually.
      // The last token is any of those. Make it the element name.
      node.computed = false;
      node.key = this.startNodeAt(this.lastTokStart, this.lastTokStartLoc);
      node.key.name = keyName;
      this.finishNode(node.key, "Identifier");
    } else {
      this.parseClassElementName(node);
    }

    // Parse element value
    if (ecmaVersion < 13 || this.type === types$1.parenL || kind !== "method" || isGenerator || isAsync) {
      var isConstructor = !node.static && checkKeyName(node, "constructor");
      var allowsDirectSuper = isConstructor && constructorAllowsSuper;
      // Couldn't move this check into the 'parseClassMethod' method for backward compatibility.
      if (isConstructor && kind !== "method") { this.raise(node.key.start, "Constructor can't have get/set modifier"); }
      node.kind = isConstructor ? "constructor" : kind;
      this.parseClassMethod(node, isGenerator, isAsync, allowsDirectSuper);
    } else {
      this.parseClassField(node);
    }

    return node
  };

  pp$8.isClassElementNameStart = function() {
    return (
      this.type === types$1.name ||
      this.type === types$1.privateId ||
      this.type === types$1.num ||
      this.type === types$1.string ||
      this.type === types$1.bracketL ||
      this.type.keyword
    )
  };

  pp$8.parseClassElementName = function(element) {
    if (this.type === types$1.privateId) {
      if (this.value === "constructor") {
        this.raise(this.start, "Classes can't have an element named '#constructor'");
      }
      element.computed = false;
      element.key = this.parsePrivateIdent();
    } else {
      this.parsePropertyName(element);
    }
  };

  pp$8.parseClassMethod = function(method, isGenerator, isAsync, allowsDirectSuper) {
    // Check key and flags
    var key = method.key;
    if (method.kind === "constructor") {
      if (isGenerator) { this.raise(key.start, "Constructor can't be a generator"); }
      if (isAsync) { this.raise(key.start, "Constructor can't be an async method"); }
    } else if (method.static && checkKeyName(method, "prototype")) {
      this.raise(key.start, "Classes may not have a static property named prototype");
    }

    // Parse value
    var value = method.value = this.parseMethod(isGenerator, isAsync, allowsDirectSuper);

    // Check value
    if (method.kind === "get" && value.params.length !== 0)
      { this.raiseRecoverable(value.start, "getter should have no params"); }
    if (method.kind === "set" && value.params.length !== 1)
      { this.raiseRecoverable(value.start, "setter should have exactly one param"); }
    if (method.kind === "set" && value.params[0].type === "RestElement")
      { this.raiseRecoverable(value.params[0].start, "Setter cannot use rest params"); }

    return this.finishNode(method, "MethodDefinition")
  };

  pp$8.parseClassField = function(field) {
    if (checkKeyName(field, "constructor")) {
      this.raise(field.key.start, "Classes can't have a field named 'constructor'");
    } else if (field.static && checkKeyName(field, "prototype")) {
      this.raise(field.key.start, "Classes can't have a static field named 'prototype'");
    }

    if (this.eat(types$1.eq)) {
      // To raise SyntaxError if 'arguments' exists in the initializer.
      var scope = this.currentThisScope();
      var inClassFieldInit = scope.inClassFieldInit;
      scope.inClassFieldInit = true;
      field.value = this.parseMaybeAssign();
      scope.inClassFieldInit = inClassFieldInit;
    } else {
      field.value = null;
    }
    this.semicolon();

    return this.finishNode(field, "PropertyDefinition")
  };

  pp$8.parseClassStaticBlock = function(node) {
    node.body = [];

    var oldLabels = this.labels;
    this.labels = [];
    this.enterScope(SCOPE_CLASS_STATIC_BLOCK | SCOPE_SUPER);
    while (this.type !== types$1.braceR) {
      var stmt = this.parseStatement(null);
      node.body.push(stmt);
    }
    this.next();
    this.exitScope();
    this.labels = oldLabels;

    return this.finishNode(node, "StaticBlock")
  };

  pp$8.parseClassId = function(node, isStatement) {
    if (this.type === types$1.name) {
      node.id = this.parseIdent();
      if (isStatement)
        { this.checkLValSimple(node.id, BIND_LEXICAL, false); }
    } else {
      if (isStatement === true)
        { this.unexpected(); }
      node.id = null;
    }
  };

  pp$8.parseClassSuper = function(node) {
    node.superClass = this.eat(types$1._extends) ? this.parseExprSubscripts(null, false) : null;
  };

  pp$8.enterClassBody = function() {
    var element = {declared: Object.create(null), used: []};
    this.privateNameStack.push(element);
    return element.declared
  };

  pp$8.exitClassBody = function() {
    var ref = this.privateNameStack.pop();
    var declared = ref.declared;
    var used = ref.used;
    if (!this.options.checkPrivateFields) { return }
    var len = this.privateNameStack.length;
    var parent = len === 0 ? null : this.privateNameStack[len - 1];
    for (var i = 0; i < used.length; ++i) {
      var id = used[i];
      if (!hasOwn(declared, id.name)) {
        if (parent) {
          parent.used.push(id);
        } else {
          this.raiseRecoverable(id.start, ("Private field '#" + (id.name) + "' must be declared in an enclosing class"));
        }
      }
    }
  };

  function isPrivateNameConflicted(privateNameMap, element) {
    var name = element.key.name;
    var curr = privateNameMap[name];

    var next = "true";
    if (element.type === "MethodDefinition" && (element.kind === "get" || element.kind === "set")) {
      next = (element.static ? "s" : "i") + element.kind;
    }

    // `class { get #a(){}; static set #a(_){} }` is also conflict.
    if (
      curr === "iget" && next === "iset" ||
      curr === "iset" && next === "iget" ||
      curr === "sget" && next === "sset" ||
      curr === "sset" && next === "sget"
    ) {
      privateNameMap[name] = "true";
      return false
    } else if (!curr) {
      privateNameMap[name] = next;
      return false
    } else {
      return true
    }
  }

  function checkKeyName(node, name) {
    var computed = node.computed;
    var key = node.key;
    return !computed && (
      key.type === "Identifier" && key.name === name ||
      key.type === "Literal" && key.value === name
    )
  }

  // Parses module export declaration.

  pp$8.parseExportAllDeclaration = function(node, exports) {
    if (this.options.ecmaVersion >= 11) {
      if (this.eatContextual("as")) {
        node.exported = this.parseModuleExportName();
        this.checkExport(exports, node.exported, this.lastTokStart);
      } else {
        node.exported = null;
      }
    }
    this.expectContextual("from");
    if (this.type !== types$1.string) { this.unexpected(); }
    node.source = this.parseExprAtom();
    this.semicolon();
    return this.finishNode(node, "ExportAllDeclaration")
  };

  pp$8.parseExport = function(node, exports) {
    this.next();
    // export * from '...'
    if (this.eat(types$1.star)) {
      return this.parseExportAllDeclaration(node, exports)
    }
    if (this.eat(types$1._default)) { // export default ...
      this.checkExport(exports, "default", this.lastTokStart);
      node.declaration = this.parseExportDefaultDeclaration();
      return this.finishNode(node, "ExportDefaultDeclaration")
    }
    // export var|const|let|function|class ...
    if (this.shouldParseExportStatement()) {
      node.declaration = this.parseExportDeclaration(node);
      if (node.declaration.type === "VariableDeclaration")
        { this.checkVariableExport(exports, node.declaration.declarations); }
      else
        { this.checkExport(exports, node.declaration.id, node.declaration.id.start); }
      node.specifiers = [];
      node.source = null;
    } else { // export { x, y as z } [from '...']
      node.declaration = null;
      node.specifiers = this.parseExportSpecifiers(exports);
      if (this.eatContextual("from")) {
        if (this.type !== types$1.string) { this.unexpected(); }
        node.source = this.parseExprAtom();
      } else {
        for (var i = 0, list = node.specifiers; i < list.length; i += 1) {
          // check for keywords used as local names
          var spec = list[i];

          this.checkUnreserved(spec.local);
          // check if export is defined
          this.checkLocalExport(spec.local);

          if (spec.local.type === "Literal") {
            this.raise(spec.local.start, "A string literal cannot be used as an exported binding without `from`.");
          }
        }

        node.source = null;
      }
      this.semicolon();
    }
    return this.finishNode(node, "ExportNamedDeclaration")
  };

  pp$8.parseExportDeclaration = function(node) {
    return this.parseStatement(null)
  };

  pp$8.parseExportDefaultDeclaration = function() {
    var isAsync;
    if (this.type === types$1._function || (isAsync = this.isAsyncFunction())) {
      var fNode = this.startNode();
      this.next();
      if (isAsync) { this.next(); }
      return this.parseFunction(fNode, FUNC_STATEMENT | FUNC_NULLABLE_ID, false, isAsync)
    } else if (this.type === types$1._class) {
      var cNode = this.startNode();
      return this.parseClass(cNode, "nullableID")
    } else {
      var declaration = this.parseMaybeAssign();
      this.semicolon();
      return declaration
    }
  };

  pp$8.checkExport = function(exports, name, pos) {
    if (!exports) { return }
    if (typeof name !== "string")
      { name = name.type === "Identifier" ? name.name : name.value; }
    if (hasOwn(exports, name))
      { this.raiseRecoverable(pos, "Duplicate export '" + name + "'"); }
    exports[name] = true;
  };

  pp$8.checkPatternExport = function(exports, pat) {
    var type = pat.type;
    if (type === "Identifier")
      { this.checkExport(exports, pat, pat.start); }
    else if (type === "ObjectPattern")
      { for (var i = 0, list = pat.properties; i < list.length; i += 1)
        {
          var prop = list[i];

          this.checkPatternExport(exports, prop);
        } }
    else if (type === "ArrayPattern")
      { for (var i$1 = 0, list$1 = pat.elements; i$1 < list$1.length; i$1 += 1) {
        var elt = list$1[i$1];

          if (elt) { this.checkPatternExport(exports, elt); }
      } }
    else if (type === "Property")
      { this.checkPatternExport(exports, pat.value); }
    else if (type === "AssignmentPattern")
      { this.checkPatternExport(exports, pat.left); }
    else if (type === "RestElement")
      { this.checkPatternExport(exports, pat.argument); }
  };

  pp$8.checkVariableExport = function(exports, decls) {
    if (!exports) { return }
    for (var i = 0, list = decls; i < list.length; i += 1)
      {
      var decl = list[i];

      this.checkPatternExport(exports, decl.id);
    }
  };

  pp$8.shouldParseExportStatement = function() {
    return this.type.keyword === "var" ||
      this.type.keyword === "const" ||
      this.type.keyword === "class" ||
      this.type.keyword === "function" ||
      this.isLet() ||
      this.isAsyncFunction()
  };

  // Parses a comma-separated list of module exports.

  pp$8.parseExportSpecifier = function(exports) {
    var node = this.startNode();
    node.local = this.parseModuleExportName();

    node.exported = this.eatContextual("as") ? this.parseModuleExportName() : node.local;
    this.checkExport(
      exports,
      node.exported,
      node.exported.start
    );

    return this.finishNode(node, "ExportSpecifier")
  };

  pp$8.parseExportSpecifiers = function(exports) {
    var nodes = [], first = true;
    // export { x, y as z } [from '...']
    this.expect(types$1.braceL);
    while (!this.eat(types$1.braceR)) {
      if (!first) {
        this.expect(types$1.comma);
        if (this.afterTrailingComma(types$1.braceR)) { break }
      } else { first = false; }

      nodes.push(this.parseExportSpecifier(exports));
    }
    return nodes
  };

  // Parses import declaration.

  pp$8.parseImport = function(node) {
    this.next();

    // import '...'
    if (this.type === types$1.string) {
      node.specifiers = empty$1;
      node.source = this.parseExprAtom();
    } else {
      node.specifiers = this.parseImportSpecifiers();
      this.expectContextual("from");
      node.source = this.type === types$1.string ? this.parseExprAtom() : this.unexpected();
    }
    this.semicolon();
    return this.finishNode(node, "ImportDeclaration")
  };

  // Parses a comma-separated list of module imports.

  pp$8.parseImportSpecifier = function() {
    var node = this.startNode();
    node.imported = this.parseModuleExportName();

    if (this.eatContextual("as")) {
      node.local = this.parseIdent();
    } else {
      this.checkUnreserved(node.imported);
      node.local = node.imported;
    }
    this.checkLValSimple(node.local, BIND_LEXICAL);

    return this.finishNode(node, "ImportSpecifier")
  };

  pp$8.parseImportDefaultSpecifier = function() {
    // import defaultObj, { x, y as z } from '...'
    var node = this.startNode();
    node.local = this.parseIdent();
    this.checkLValSimple(node.local, BIND_LEXICAL);
    return this.finishNode(node, "ImportDefaultSpecifier")
  };

  pp$8.parseImportNamespaceSpecifier = function() {
    var node = this.startNode();
    this.next();
    this.expectContextual("as");
    node.local = this.parseIdent();
    this.checkLValSimple(node.local, BIND_LEXICAL);
    return this.finishNode(node, "ImportNamespaceSpecifier")
  };

  pp$8.parseImportSpecifiers = function() {
    var nodes = [], first = true;
    if (this.type === types$1.name) {
      nodes.push(this.parseImportDefaultSpecifier());
      if (!this.eat(types$1.comma)) { return nodes }
    }
    if (this.type === types$1.star) {
      nodes.push(this.parseImportNamespaceSpecifier());
      return nodes
    }
    this.expect(types$1.braceL);
    while (!this.eat(types$1.braceR)) {
      if (!first) {
        this.expect(types$1.comma);
        if (this.afterTrailingComma(types$1.braceR)) { break }
      } else { first = false; }

      nodes.push(this.parseImportSpecifier());
    }
    return nodes
  };

  pp$8.parseModuleExportName = function() {
    if (this.options.ecmaVersion >= 13 && this.type === types$1.string) {
      var stringLiteral = this.parseLiteral(this.value);
      if (loneSurrogate.test(stringLiteral.value)) {
        this.raise(stringLiteral.start, "An export name cannot include a lone surrogate.");
      }
      return stringLiteral
    }
    return this.parseIdent(true)
  };

  // Set `ExpressionStatement#directive` property for directive prologues.
  pp$8.adaptDirectivePrologue = function(statements) {
    for (var i = 0; i < statements.length && this.isDirectiveCandidate(statements[i]); ++i) {
      statements[i].directive = statements[i].expression.raw.slice(1, -1);
    }
  };
  pp$8.isDirectiveCandidate = function(statement) {
    return (
      this.options.ecmaVersion >= 5 &&
      statement.type === "ExpressionStatement" &&
      statement.expression.type === "Literal" &&
      typeof statement.expression.value === "string" &&
      // Reject parenthesized strings.
      (this.input[statement.start] === "\"" || this.input[statement.start] === "'")
    )
  };

  var pp$7 = Parser.prototype;

  // Convert existing expression atom to assignable pattern
  // if possible.

  pp$7.toAssignable = function(node, isBinding, refDestructuringErrors) {
    if (this.options.ecmaVersion >= 6 && node) {
      switch (node.type) {
      case "Identifier":
        if (this.inAsync && node.name === "await")
          { this.raise(node.start, "Cannot use 'await' as identifier inside an async function"); }
        break

      case "ObjectPattern":
      case "ArrayPattern":
      case "AssignmentPattern":
      case "RestElement":
        break

      case "ObjectExpression":
        node.type = "ObjectPattern";
        if (refDestructuringErrors) { this.checkPatternErrors(refDestructuringErrors, true); }
        for (var i = 0, list = node.properties; i < list.length; i += 1) {
          var prop = list[i];

        this.toAssignable(prop, isBinding);
          // Early error:
          //   AssignmentRestProperty[Yield, Await] :
          //     `...` DestructuringAssignmentTarget[Yield, Await]
          //
          //   It is a Syntax Error if |DestructuringAssignmentTarget| is an |ArrayLiteral| or an |ObjectLiteral|.
          if (
            prop.type === "RestElement" &&
            (prop.argument.type === "ArrayPattern" || prop.argument.type === "ObjectPattern")
          ) {
            this.raise(prop.argument.start, "Unexpected token");
          }
        }
        break

      case "Property":
        // AssignmentProperty has type === "Property"
        if (node.kind !== "init") { this.raise(node.key.start, "Object pattern can't contain getter or setter"); }
        this.toAssignable(node.value, isBinding);
        break

      case "ArrayExpression":
        node.type = "ArrayPattern";
        if (refDestructuringErrors) { this.checkPatternErrors(refDestructuringErrors, true); }
        this.toAssignableList(node.elements, isBinding);
        break

      case "SpreadElement":
        node.type = "RestElement";
        this.toAssignable(node.argument, isBinding);
        if (node.argument.type === "AssignmentPattern")
          { this.raise(node.argument.start, "Rest elements cannot have a default value"); }
        break

      case "AssignmentExpression":
        if (node.operator !== "=") { this.raise(node.left.end, "Only '=' operator can be used for specifying default value."); }
        node.type = "AssignmentPattern";
        delete node.operator;
        this.toAssignable(node.left, isBinding);
        break

      case "ParenthesizedExpression":
        this.toAssignable(node.expression, isBinding, refDestructuringErrors);
        break

      case "ChainExpression":
        this.raiseRecoverable(node.start, "Optional chaining cannot appear in left-hand side");
        break

      case "MemberExpression":
        if (!isBinding) { break }

      default:
        this.raise(node.start, "Assigning to rvalue");
      }
    } else if (refDestructuringErrors) { this.checkPatternErrors(refDestructuringErrors, true); }
    return node
  };

  // Convert list of expression atoms to binding list.

  pp$7.toAssignableList = function(exprList, isBinding) {
    var end = exprList.length;
    for (var i = 0; i < end; i++) {
      var elt = exprList[i];
      if (elt) { this.toAssignable(elt, isBinding); }
    }
    if (end) {
      var last = exprList[end - 1];
      if (this.options.ecmaVersion === 6 && isBinding && last && last.type === "RestElement" && last.argument.type !== "Identifier")
        { this.unexpected(last.argument.start); }
    }
    return exprList
  };

  // Parses spread element.

  pp$7.parseSpread = function(refDestructuringErrors) {
    var node = this.startNode();
    this.next();
    node.argument = this.parseMaybeAssign(false, refDestructuringErrors);
    return this.finishNode(node, "SpreadElement")
  };

  pp$7.parseRestBinding = function() {
    var node = this.startNode();
    this.next();

    // RestElement inside of a function parameter must be an identifier
    if (this.options.ecmaVersion === 6 && this.type !== types$1.name)
      { this.unexpected(); }

    node.argument = this.parseBindingAtom();

    return this.finishNode(node, "RestElement")
  };

  // Parses lvalue (assignable) atom.

  pp$7.parseBindingAtom = function() {
    if (this.options.ecmaVersion >= 6) {
      switch (this.type) {
      case types$1.bracketL:
        var node = this.startNode();
        this.next();
        node.elements = this.parseBindingList(types$1.bracketR, true, true);
        return this.finishNode(node, "ArrayPattern")

      case types$1.braceL:
        return this.parseObj(true)
      }
    }
    return this.parseIdent()
  };

  pp$7.parseBindingList = function(close, allowEmpty, allowTrailingComma, allowModifiers) {
    var elts = [], first = true;
    while (!this.eat(close)) {
      if (first) { first = false; }
      else { this.expect(types$1.comma); }
      if (allowEmpty && this.type === types$1.comma) {
        elts.push(null);
      } else if (allowTrailingComma && this.afterTrailingComma(close)) {
        break
      } else if (this.type === types$1.ellipsis) {
        var rest = this.parseRestBinding();
        this.parseBindingListItem(rest);
        elts.push(rest);
        if (this.type === types$1.comma) { this.raiseRecoverable(this.start, "Comma is not permitted after the rest element"); }
        this.expect(close);
        break
      } else {
        elts.push(this.parseAssignableListItem(allowModifiers));
      }
    }
    return elts
  };

  pp$7.parseAssignableListItem = function(allowModifiers) {
    var elem = this.parseMaybeDefault(this.start, this.startLoc);
    this.parseBindingListItem(elem);
    return elem
  };

  pp$7.parseBindingListItem = function(param) {
    return param
  };

  // Parses assignment pattern around given atom if possible.

  pp$7.parseMaybeDefault = function(startPos, startLoc, left) {
    left = left || this.parseBindingAtom();
    if (this.options.ecmaVersion < 6 || !this.eat(types$1.eq)) { return left }
    var node = this.startNodeAt(startPos, startLoc);
    node.left = left;
    node.right = this.parseMaybeAssign();
    return this.finishNode(node, "AssignmentPattern")
  };

  // The following three functions all verify that a node is an lvalue —
  // something that can be bound, or assigned to. In order to do so, they perform
  // a variety of checks:
  //
  // - Check that none of the bound/assigned-to identifiers are reserved words.
  // - Record name declarations for bindings in the appropriate scope.
  // - Check duplicate argument names, if checkClashes is set.
  //
  // If a complex binding pattern is encountered (e.g., object and array
  // destructuring), the entire pattern is recursively checked.
  //
  // There are three versions of checkLVal*() appropriate for different
  // circumstances:
  //
  // - checkLValSimple() shall be used if the syntactic construct supports
  //   nothing other than identifiers and member expressions. Parenthesized
  //   expressions are also correctly handled. This is generally appropriate for
  //   constructs for which the spec says
  //
  //   > It is a Syntax Error if AssignmentTargetType of [the production] is not
  //   > simple.
  //
  //   It is also appropriate for checking if an identifier is valid and not
  //   defined elsewhere, like import declarations or function/class identifiers.
  //
  //   Examples where this is used include:
  //     a += …;
  //     import a from '…';
  //   where a is the node to be checked.
  //
  // - checkLValPattern() shall be used if the syntactic construct supports
  //   anything checkLValSimple() supports, as well as object and array
  //   destructuring patterns. This is generally appropriate for constructs for
  //   which the spec says
  //
  //   > It is a Syntax Error if [the production] is neither an ObjectLiteral nor
  //   > an ArrayLiteral and AssignmentTargetType of [the production] is not
  //   > simple.
  //
  //   Examples where this is used include:
  //     (a = …);
  //     const a = …;
  //     try { … } catch (a) { … }
  //   where a is the node to be checked.
  //
  // - checkLValInnerPattern() shall be used if the syntactic construct supports
  //   anything checkLValPattern() supports, as well as default assignment
  //   patterns, rest elements, and other constructs that may appear within an
  //   object or array destructuring pattern.
  //
  //   As a special case, function parameters also use checkLValInnerPattern(),
  //   as they also support defaults and rest constructs.
  //
  // These functions deliberately support both assignment and binding constructs,
  // as the logic for both is exceedingly similar. If the node is the target of
  // an assignment, then bindingType should be set to BIND_NONE. Otherwise, it
  // should be set to the appropriate BIND_* constant, like BIND_VAR or
  // BIND_LEXICAL.
  //
  // If the function is called with a non-BIND_NONE bindingType, then
  // additionally a checkClashes object may be specified to allow checking for
  // duplicate argument names. checkClashes is ignored if the provided construct
  // is an assignment (i.e., bindingType is BIND_NONE).

  pp$7.checkLValSimple = function(expr, bindingType, checkClashes) {
    if ( bindingType === void 0 ) bindingType = BIND_NONE;

    var isBind = bindingType !== BIND_NONE;

    switch (expr.type) {
    case "Identifier":
      if (this.strict && this.reservedWordsStrictBind.test(expr.name))
        { this.raiseRecoverable(expr.start, (isBind ? "Binding " : "Assigning to ") + expr.name + " in strict mode"); }
      if (isBind) {
        if (bindingType === BIND_LEXICAL && expr.name === "let")
          { this.raiseRecoverable(expr.start, "let is disallowed as a lexically bound name"); }
        if (checkClashes) {
          if (hasOwn(checkClashes, expr.name))
            { this.raiseRecoverable(expr.start, "Argument name clash"); }
          checkClashes[expr.name] = true;
        }
        if (bindingType !== BIND_OUTSIDE) { this.declareName(expr.name, bindingType, expr.start); }
      }
      break

    case "ChainExpression":
      this.raiseRecoverable(expr.start, "Optional chaining cannot appear in left-hand side");
      break

    case "MemberExpression":
      if (isBind) { this.raiseRecoverable(expr.start, "Binding member expression"); }
      break

    case "ParenthesizedExpression":
      if (isBind) { this.raiseRecoverable(expr.start, "Binding parenthesized expression"); }
      return this.checkLValSimple(expr.expression, bindingType, checkClashes)

    default:
      this.raise(expr.start, (isBind ? "Binding" : "Assigning to") + " rvalue");
    }
  };

  pp$7.checkLValPattern = function(expr, bindingType, checkClashes) {
    if ( bindingType === void 0 ) bindingType = BIND_NONE;

    switch (expr.type) {
    case "ObjectPattern":
      for (var i = 0, list = expr.properties; i < list.length; i += 1) {
        var prop = list[i];

      this.checkLValInnerPattern(prop, bindingType, checkClashes);
      }
      break

    case "ArrayPattern":
      for (var i$1 = 0, list$1 = expr.elements; i$1 < list$1.length; i$1 += 1) {
        var elem = list$1[i$1];

      if (elem) { this.checkLValInnerPattern(elem, bindingType, checkClashes); }
      }
      break

    default:
      this.checkLValSimple(expr, bindingType, checkClashes);
    }
  };

  pp$7.checkLValInnerPattern = function(expr, bindingType, checkClashes) {
    if ( bindingType === void 0 ) bindingType = BIND_NONE;

    switch (expr.type) {
    case "Property":
      // AssignmentProperty has type === "Property"
      this.checkLValInnerPattern(expr.value, bindingType, checkClashes);
      break

    case "AssignmentPattern":
      this.checkLValPattern(expr.left, bindingType, checkClashes);
      break

    case "RestElement":
      this.checkLValPattern(expr.argument, bindingType, checkClashes);
      break

    default:
      this.checkLValPattern(expr, bindingType, checkClashes);
    }
  };

  // The algorithm used to determine whether a regexp can appear at a
  // given point in the program is loosely based on sweet.js' approach.
  // See https://github.com/mozilla/sweet.js/wiki/design


  var TokContext = function TokContext(token, isExpr, preserveSpace, override, generator) {
    this.token = token;
    this.isExpr = !!isExpr;
    this.preserveSpace = !!preserveSpace;
    this.override = override;
    this.generator = !!generator;
  };

  var types = {
    b_stat: new TokContext("{", false),
    b_expr: new TokContext("{", true),
    b_tmpl: new TokContext("${", false),
    p_stat: new TokContext("(", false),
    p_expr: new TokContext("(", true),
    q_tmpl: new TokContext("`", true, true, function (p) { return p.tryReadTemplateToken(); }),
    f_stat: new TokContext("function", false),
    f_expr: new TokContext("function", true),
    f_expr_gen: new TokContext("function", true, false, null, true),
    f_gen: new TokContext("function", false, false, null, true)
  };

  var pp$6 = Parser.prototype;

  pp$6.initialContext = function() {
    return [types.b_stat]
  };

  pp$6.curContext = function() {
    return this.context[this.context.length - 1]
  };

  pp$6.braceIsBlock = function(prevType) {
    var parent = this.curContext();
    if (parent === types.f_expr || parent === types.f_stat)
      { return true }
    if (prevType === types$1.colon && (parent === types.b_stat || parent === types.b_expr))
      { return !parent.isExpr }

    // The check for `tt.name && exprAllowed` detects whether we are
    // after a `yield` or `of` construct. See the `updateContext` for
    // `tt.name`.
    if (prevType === types$1._return || prevType === types$1.name && this.exprAllowed)
      { return lineBreak.test(this.input.slice(this.lastTokEnd, this.start)) }
    if (prevType === types$1._else || prevType === types$1.semi || prevType === types$1.eof || prevType === types$1.parenR || prevType === types$1.arrow)
      { return true }
    if (prevType === types$1.braceL)
      { return parent === types.b_stat }
    if (prevType === types$1._var || prevType === types$1._const || prevType === types$1.name)
      { return false }
    return !this.exprAllowed
  };

  pp$6.inGeneratorContext = function() {
    for (var i = this.context.length - 1; i >= 1; i--) {
      var context = this.context[i];
      if (context.token === "function")
        { return context.generator }
    }
    return false
  };

  pp$6.updateContext = function(prevType) {
    var update, type = this.type;
    if (type.keyword && prevType === types$1.dot)
      { this.exprAllowed = false; }
    else if (update = type.updateContext)
      { update.call(this, prevType); }
    else
      { this.exprAllowed = type.beforeExpr; }
  };

  // Used to handle edge cases when token context could not be inferred correctly during tokenization phase

  pp$6.overrideContext = function(tokenCtx) {
    if (this.curContext() !== tokenCtx) {
      this.context[this.context.length - 1] = tokenCtx;
    }
  };

  // Token-specific context update code

  types$1.parenR.updateContext = types$1.braceR.updateContext = function() {
    if (this.context.length === 1) {
      this.exprAllowed = true;
      return
    }
    var out = this.context.pop();
    if (out === types.b_stat && this.curContext().token === "function") {
      out = this.context.pop();
    }
    this.exprAllowed = !out.isExpr;
  };

  types$1.braceL.updateContext = function(prevType) {
    this.context.push(this.braceIsBlock(prevType) ? types.b_stat : types.b_expr);
    this.exprAllowed = true;
  };

  types$1.dollarBraceL.updateContext = function() {
    this.context.push(types.b_tmpl);
    this.exprAllowed = true;
  };

  types$1.parenL.updateContext = function(prevType) {
    var statementParens = prevType === types$1._if || prevType === types$1._for || prevType === types$1._with || prevType === types$1._while;
    this.context.push(statementParens ? types.p_stat : types.p_expr);
    this.exprAllowed = true;
  };

  types$1.incDec.updateContext = function() {
    // tokExprAllowed stays unchanged
  };

  types$1._function.updateContext = types$1._class.updateContext = function(prevType) {
    if (prevType.beforeExpr && prevType !== types$1._else &&
        !(prevType === types$1.semi && this.curContext() !== types.p_stat) &&
        !(prevType === types$1._return && lineBreak.test(this.input.slice(this.lastTokEnd, this.start))) &&
        !((prevType === types$1.colon || prevType === types$1.braceL) && this.curContext() === types.b_stat))
      { this.context.push(types.f_expr); }
    else
      { this.context.push(types.f_stat); }
    this.exprAllowed = false;
  };

  types$1.colon.updateContext = function() {
    if (this.curContext().token === "function") { this.context.pop(); }
    this.exprAllowed = true;
  };

  types$1.backQuote.updateContext = function() {
    if (this.curContext() === types.q_tmpl)
      { this.context.pop(); }
    else
      { this.context.push(types.q_tmpl); }
    this.exprAllowed = false;
  };

  types$1.star.updateContext = function(prevType) {
    if (prevType === types$1._function) {
      var index = this.context.length - 1;
      if (this.context[index] === types.f_expr)
        { this.context[index] = types.f_expr_gen; }
      else
        { this.context[index] = types.f_gen; }
    }
    this.exprAllowed = true;
  };

  types$1.name.updateContext = function(prevType) {
    var allowed = false;
    if (this.options.ecmaVersion >= 6 && prevType !== types$1.dot) {
      if (this.value === "of" && !this.exprAllowed ||
          this.value === "yield" && this.inGeneratorContext())
        { allowed = true; }
    }
    this.exprAllowed = allowed;
  };

  // A recursive descent parser operates by defining functions for all
  // syntactic elements, and recursively calling those, each function
  // advancing the input stream and returning an AST node. Precedence
  // of constructs (for example, the fact that `!x[1]` means `!(x[1])`
  // instead of `(!x)[1]` is handled by the fact that the parser
  // function that parses unary prefix operators is called first, and
  // in turn calls the function that parses `[]` subscripts — that
  // way, it'll receive the node for `x[1]` already parsed, and wraps
  // *that* in the unary operator node.
  //
  // Acorn uses an [operator precedence parser][opp] to handle binary
  // operator precedence, because it is much more compact than using
  // the technique outlined above, which uses different, nesting
  // functions to specify precedence, for all of the ten binary
  // precedence levels that JavaScript defines.
  //
  // [opp]: http://en.wikipedia.org/wiki/Operator-precedence_parser


  var pp$5 = Parser.prototype;

  // Check if property name clashes with already added.
  // Object/class getters and setters are not allowed to clash —
  // either with each other or with an init property — and in
  // strict mode, init properties are also not allowed to be repeated.

  pp$5.checkPropClash = function(prop, propHash, refDestructuringErrors) {
    if (this.options.ecmaVersion >= 9 && prop.type === "SpreadElement")
      { return }
    if (this.options.ecmaVersion >= 6 && (prop.computed || prop.method || prop.shorthand))
      { return }
    var key = prop.key;
    var name;
    switch (key.type) {
    case "Identifier": name = key.name; break
    case "Literal": name = String(key.value); break
    default: return
    }
    var kind = prop.kind;
    if (this.options.ecmaVersion >= 6) {
      if (name === "__proto__" && kind === "init") {
        if (propHash.proto) {
          if (refDestructuringErrors) {
            if (refDestructuringErrors.doubleProto < 0) {
              refDestructuringErrors.doubleProto = key.start;
            }
          } else {
            this.raiseRecoverable(key.start, "Redefinition of __proto__ property");
          }
        }
        propHash.proto = true;
      }
      return
    }
    name = "$" + name;
    var other = propHash[name];
    if (other) {
      var redefinition;
      if (kind === "init") {
        redefinition = this.strict && other.init || other.get || other.set;
      } else {
        redefinition = other.init || other[kind];
      }
      if (redefinition)
        { this.raiseRecoverable(key.start, "Redefinition of property"); }
    } else {
      other = propHash[name] = {
        init: false,
        get: false,
        set: false
      };
    }
    other[kind] = true;
  };

  // ### Expression parsing

  // These nest, from the most general expression type at the top to
  // 'atomic', nondivisible expression types at the bottom. Most of
  // the functions will simply let the function(s) below them parse,
  // and, *if* the syntactic construct they handle is present, wrap
  // the AST node that the inner parser gave them in another node.

  // Parse a full expression. The optional arguments are used to
  // forbid the `in` operator (in for loops initalization expressions)
  // and provide reference for storing '=' operator inside shorthand
  // property assignment in contexts where both object expression
  // and object pattern might appear (so it's possible to raise
  // delayed syntax error at correct position).

  pp$5.parseExpression = function(forInit, refDestructuringErrors) {
    var startPos = this.start, startLoc = this.startLoc;
    var expr = this.parseMaybeAssign(forInit, refDestructuringErrors);
    if (this.type === types$1.comma) {
      var node = this.startNodeAt(startPos, startLoc);
      node.expressions = [expr];
      while (this.eat(types$1.comma)) { node.expressions.push(this.parseMaybeAssign(forInit, refDestructuringErrors)); }
      return this.finishNode(node, "SequenceExpression")
    }
    return expr
  };

  // Parse an assignment expression. This includes applications of
  // operators like `+=`.

  pp$5.parseMaybeAssign = function(forInit, refDestructuringErrors, afterLeftParse) {
    if (this.isContextual("yield")) {
      if (this.inGenerator) { return this.parseYield(forInit) }
      // The tokenizer will assume an expression is allowed after
      // `yield`, but this isn't that kind of yield
      else { this.exprAllowed = false; }
    }

    var ownDestructuringErrors = false, oldParenAssign = -1, oldTrailingComma = -1, oldDoubleProto = -1;
    if (refDestructuringErrors) {
      oldParenAssign = refDestructuringErrors.parenthesizedAssign;
      oldTrailingComma = refDestructuringErrors.trailingComma;
      oldDoubleProto = refDestructuringErrors.doubleProto;
      refDestructuringErrors.parenthesizedAssign = refDestructuringErrors.trailingComma = -1;
    } else {
      refDestructuringErrors = new DestructuringErrors;
      ownDestructuringErrors = true;
    }

    var startPos = this.start, startLoc = this.startLoc;
    if (this.type === types$1.parenL || this.type === types$1.name) {
      this.potentialArrowAt = this.start;
      this.potentialArrowInForAwait = forInit === "await";
    }
    var left = this.parseMaybeConditional(forInit, refDestructuringErrors);
    if (afterLeftParse) { left = afterLeftParse.call(this, left, startPos, startLoc); }
    if (this.type.isAssign) {
      var node = this.startNodeAt(startPos, startLoc);
      node.operator = this.value;
      if (this.type === types$1.eq)
        { left = this.toAssignable(left, false, refDestructuringErrors); }
      if (!ownDestructuringErrors) {
        refDestructuringErrors.parenthesizedAssign = refDestructuringErrors.trailingComma = refDestructuringErrors.doubleProto = -1;
      }
      if (refDestructuringErrors.shorthandAssign >= left.start)
        { refDestructuringErrors.shorthandAssign = -1; } // reset because shorthand default was used correctly
      if (this.type === types$1.eq)
        { this.checkLValPattern(left); }
      else
        { this.checkLValSimple(left); }
      node.left = left;
      this.next();
      node.right = this.parseMaybeAssign(forInit);
      if (oldDoubleProto > -1) { refDestructuringErrors.doubleProto = oldDoubleProto; }
      return this.finishNode(node, "AssignmentExpression")
    } else {
      if (ownDestructuringErrors) { this.checkExpressionErrors(refDestructuringErrors, true); }
    }
    if (oldParenAssign > -1) { refDestructuringErrors.parenthesizedAssign = oldParenAssign; }
    if (oldTrailingComma > -1) { refDestructuringErrors.trailingComma = oldTrailingComma; }
    return left
  };

  // Parse a ternary conditional (`?:`) operator.

  pp$5.parseMaybeConditional = function(forInit, refDestructuringErrors) {
    var startPos = this.start, startLoc = this.startLoc;
    var expr = this.parseExprOps(forInit, refDestructuringErrors);
    if (this.checkExpressionErrors(refDestructuringErrors)) { return expr }
    if (this.eat(types$1.question)) {
      var node = this.startNodeAt(startPos, startLoc);
      node.test = expr;
      node.consequent = this.parseMaybeAssign();
      this.expect(types$1.colon);
      node.alternate = this.parseMaybeAssign(forInit);
      return this.finishNode(node, "ConditionalExpression")
    }
    return expr
  };

  // Start the precedence parser.

  pp$5.parseExprOps = function(forInit, refDestructuringErrors) {
    var startPos = this.start, startLoc = this.startLoc;
    var expr = this.parseMaybeUnary(refDestructuringErrors, false, false, forInit);
    if (this.checkExpressionErrors(refDestructuringErrors)) { return expr }
    return expr.start === startPos && expr.type === "ArrowFunctionExpression" ? expr : this.parseExprOp(expr, startPos, startLoc, -1, forInit)
  };

  // Parse binary operators with the operator precedence parsing
  // algorithm. `left` is the left-hand side of the operator.
  // `minPrec` provides context that allows the function to stop and
  // defer further parser to one of its callers when it encounters an
  // operator that has a lower precedence than the set it is parsing.

  pp$5.parseExprOp = function(left, leftStartPos, leftStartLoc, minPrec, forInit) {
    var prec = this.type.binop;
    if (prec != null && (!forInit || this.type !== types$1._in)) {
      if (prec > minPrec) {
        var logical = this.type === types$1.logicalOR || this.type === types$1.logicalAND;
        var coalesce = this.type === types$1.coalesce;
        if (coalesce) {
          // Handle the precedence of `tt.coalesce` as equal to the range of logical expressions.
          // In other words, `node.right` shouldn't contain logical expressions in order to check the mixed error.
          prec = types$1.logicalAND.binop;
        }
        var op = this.value;
        this.next();
        var startPos = this.start, startLoc = this.startLoc;
        var right = this.parseExprOp(this.parseMaybeUnary(null, false, false, forInit), startPos, startLoc, prec, forInit);
        var node = this.buildBinary(leftStartPos, leftStartLoc, left, right, op, logical || coalesce);
        if ((logical && this.type === types$1.coalesce) || (coalesce && (this.type === types$1.logicalOR || this.type === types$1.logicalAND))) {
          this.raiseRecoverable(this.start, "Logical expressions and coalesce expressions cannot be mixed. Wrap either by parentheses");
        }
        return this.parseExprOp(node, leftStartPos, leftStartLoc, minPrec, forInit)
      }
    }
    return left
  };

  pp$5.buildBinary = function(startPos, startLoc, left, right, op, logical) {
    if (right.type === "PrivateIdentifier") { this.raise(right.start, "Private identifier can only be left side of binary expression"); }
    var node = this.startNodeAt(startPos, startLoc);
    node.left = left;
    node.operator = op;
    node.right = right;
    return this.finishNode(node, logical ? "LogicalExpression" : "BinaryExpression")
  };

  // Parse unary operators, both prefix and postfix.

  pp$5.parseMaybeUnary = function(refDestructuringErrors, sawUnary, incDec, forInit) {
    var startPos = this.start, startLoc = this.startLoc, expr;
    if (this.isContextual("await") && this.canAwait) {
      expr = this.parseAwait(forInit);
      sawUnary = true;
    } else if (this.type.prefix) {
      var node = this.startNode(), update = this.type === types$1.incDec;
      node.operator = this.value;
      node.prefix = true;
      this.next();
      node.argument = this.parseMaybeUnary(null, true, update, forInit);
      this.checkExpressionErrors(refDestructuringErrors, true);
      if (update) { this.checkLValSimple(node.argument); }
      else if (this.strict && node.operator === "delete" && isLocalVariableAccess(node.argument))
        { this.raiseRecoverable(node.start, "Deleting local variable in strict mode"); }
      else if (node.operator === "delete" && isPrivateFieldAccess(node.argument))
        { this.raiseRecoverable(node.start, "Private fields can not be deleted"); }
      else { sawUnary = true; }
      expr = this.finishNode(node, update ? "UpdateExpression" : "UnaryExpression");
    } else if (!sawUnary && this.type === types$1.privateId) {
      if ((forInit || this.privateNameStack.length === 0) && this.options.checkPrivateFields) { this.unexpected(); }
      expr = this.parsePrivateIdent();
      // only could be private fields in 'in', such as #x in obj
      if (this.type !== types$1._in) { this.unexpected(); }
    } else {
      expr = this.parseExprSubscripts(refDestructuringErrors, forInit);
      if (this.checkExpressionErrors(refDestructuringErrors)) { return expr }
      while (this.type.postfix && !this.canInsertSemicolon()) {
        var node$1 = this.startNodeAt(startPos, startLoc);
        node$1.operator = this.value;
        node$1.prefix = false;
        node$1.argument = expr;
        this.checkLValSimple(expr);
        this.next();
        expr = this.finishNode(node$1, "UpdateExpression");
      }
    }

    if (!incDec && this.eat(types$1.starstar)) {
      if (sawUnary)
        { this.unexpected(this.lastTokStart); }
      else
        { return this.buildBinary(startPos, startLoc, expr, this.parseMaybeUnary(null, false, false, forInit), "**", false) }
    } else {
      return expr
    }
  };

  function isLocalVariableAccess(node) {
    return (
      node.type === "Identifier" ||
      node.type === "ParenthesizedExpression" && isLocalVariableAccess(node.expression)
    )
  }

  function isPrivateFieldAccess(node) {
    return (
      node.type === "MemberExpression" && node.property.type === "PrivateIdentifier" ||
      node.type === "ChainExpression" && isPrivateFieldAccess(node.expression) ||
      node.type === "ParenthesizedExpression" && isPrivateFieldAccess(node.expression)
    )
  }

  // Parse call, dot, and `[]`-subscript expressions.

  pp$5.parseExprSubscripts = function(refDestructuringErrors, forInit) {
    var startPos = this.start, startLoc = this.startLoc;
    var expr = this.parseExprAtom(refDestructuringErrors, forInit);
    if (expr.type === "ArrowFunctionExpression" && this.input.slice(this.lastTokStart, this.lastTokEnd) !== ")")
      { return expr }
    var result = this.parseSubscripts(expr, startPos, startLoc, false, forInit);
    if (refDestructuringErrors && result.type === "MemberExpression") {
      if (refDestructuringErrors.parenthesizedAssign >= result.start) { refDestructuringErrors.parenthesizedAssign = -1; }
      if (refDestructuringErrors.parenthesizedBind >= result.start) { refDestructuringErrors.parenthesizedBind = -1; }
      if (refDestructuringErrors.trailingComma >= result.start) { refDestructuringErrors.trailingComma = -1; }
    }
    return result
  };

  pp$5.parseSubscripts = function(base, startPos, startLoc, noCalls, forInit) {
    var maybeAsyncArrow = this.options.ecmaVersion >= 8 && base.type === "Identifier" && base.name === "async" &&
        this.lastTokEnd === base.end && !this.canInsertSemicolon() && base.end - base.start === 5 &&
        this.potentialArrowAt === base.start;
    var optionalChained = false;

    while (true) {
      var element = this.parseSubscript(base, startPos, startLoc, noCalls, maybeAsyncArrow, optionalChained, forInit);

      if (element.optional) { optionalChained = true; }
      if (element === base || element.type === "ArrowFunctionExpression") {
        if (optionalChained) {
          var chainNode = this.startNodeAt(startPos, startLoc);
          chainNode.expression = element;
          element = this.finishNode(chainNode, "ChainExpression");
        }
        return element
      }

      base = element;
    }
  };

  pp$5.shouldParseAsyncArrow = function() {
    return !this.canInsertSemicolon() && this.eat(types$1.arrow)
  };

  pp$5.parseSubscriptAsyncArrow = function(startPos, startLoc, exprList, forInit) {
    return this.parseArrowExpression(this.startNodeAt(startPos, startLoc), exprList, true, forInit)
  };

  pp$5.parseSubscript = function(base, startPos, startLoc, noCalls, maybeAsyncArrow, optionalChained, forInit) {
    var optionalSupported = this.options.ecmaVersion >= 11;
    var optional = optionalSupported && this.eat(types$1.questionDot);
    if (noCalls && optional) { this.raise(this.lastTokStart, "Optional chaining cannot appear in the callee of new expressions"); }

    var computed = this.eat(types$1.bracketL);
    if (computed || (optional && this.type !== types$1.parenL && this.type !== types$1.backQuote) || this.eat(types$1.dot)) {
      var node = this.startNodeAt(startPos, startLoc);
      node.object = base;
      if (computed) {
        node.property = this.parseExpression();
        this.expect(types$1.bracketR);
      } else if (this.type === types$1.privateId && base.type !== "Super") {
        node.property = this.parsePrivateIdent();
      } else {
        node.property = this.parseIdent(this.options.allowReserved !== "never");
      }
      node.computed = !!computed;
      if (optionalSupported) {
        node.optional = optional;
      }
      base = this.finishNode(node, "MemberExpression");
    } else if (!noCalls && this.eat(types$1.parenL)) {
      var refDestructuringErrors = new DestructuringErrors, oldYieldPos = this.yieldPos, oldAwaitPos = this.awaitPos, oldAwaitIdentPos = this.awaitIdentPos;
      this.yieldPos = 0;
      this.awaitPos = 0;
      this.awaitIdentPos = 0;
      var exprList = this.parseExprList(types$1.parenR, this.options.ecmaVersion >= 8, false, refDestructuringErrors);
      if (maybeAsyncArrow && !optional && this.shouldParseAsyncArrow()) {
        this.checkPatternErrors(refDestructuringErrors, false);
        this.checkYieldAwaitInDefaultParams();
        if (this.awaitIdentPos > 0)
          { this.raise(this.awaitIdentPos, "Cannot use 'await' as identifier inside an async function"); }
        this.yieldPos = oldYieldPos;
        this.awaitPos = oldAwaitPos;
        this.awaitIdentPos = oldAwaitIdentPos;
        return this.parseSubscriptAsyncArrow(startPos, startLoc, exprList, forInit)
      }
      this.checkExpressionErrors(refDestructuringErrors, true);
      this.yieldPos = oldYieldPos || this.yieldPos;
      this.awaitPos = oldAwaitPos || this.awaitPos;
      this.awaitIdentPos = oldAwaitIdentPos || this.awaitIdentPos;
      var node$1 = this.startNodeAt(startPos, startLoc);
      node$1.callee = base;
      node$1.arguments = exprList;
      if (optionalSupported) {
        node$1.optional = optional;
      }
      base = this.finishNode(node$1, "CallExpression");
    } else if (this.type === types$1.backQuote) {
      if (optional || optionalChained) {
        this.raise(this.start, "Optional chaining cannot appear in the tag of tagged template expressions");
      }
      var node$2 = this.startNodeAt(startPos, startLoc);
      node$2.tag = base;
      node$2.quasi = this.parseTemplate({isTagged: true});
      base = this.finishNode(node$2, "TaggedTemplateExpression");
    }
    return base
  };

  // Parse an atomic expression — either a single token that is an
  // expression, an expression started by a keyword like `function` or
  // `new`, or an expression wrapped in punctuation like `()`, `[]`,
  // or `{}`.

  pp$5.parseExprAtom = function(refDestructuringErrors, forInit, forNew) {
    // If a division operator appears in an expression position, the
    // tokenizer got confused, and we force it to read a regexp instead.
    if (this.type === types$1.slash) { this.readRegexp(); }

    var node, canBeArrow = this.potentialArrowAt === this.start;
    switch (this.type) {
    case types$1._super:
      if (!this.allowSuper)
        { this.raise(this.start, "'super' keyword outside a method"); }
      node = this.startNode();
      this.next();
      if (this.type === types$1.parenL && !this.allowDirectSuper)
        { this.raise(node.start, "super() call outside constructor of a subclass"); }
      // The `super` keyword can appear at below:
      // SuperProperty:
      //     super [ Expression ]
      //     super . IdentifierName
      // SuperCall:
      //     super ( Arguments )
      if (this.type !== types$1.dot && this.type !== types$1.bracketL && this.type !== types$1.parenL)
        { this.unexpected(); }
      return this.finishNode(node, "Super")

    case types$1._this:
      node = this.startNode();
      this.next();
      return this.finishNode(node, "ThisExpression")

    case types$1.name:
      var startPos = this.start, startLoc = this.startLoc, containsEsc = this.containsEsc;
      var id = this.parseIdent(false);
      if (this.options.ecmaVersion >= 8 && !containsEsc && id.name === "async" && !this.canInsertSemicolon() && this.eat(types$1._function)) {
        this.overrideContext(types.f_expr);
        return this.parseFunction(this.startNodeAt(startPos, startLoc), 0, false, true, forInit)
      }
      if (canBeArrow && !this.canInsertSemicolon()) {
        if (this.eat(types$1.arrow))
          { return this.parseArrowExpression(this.startNodeAt(startPos, startLoc), [id], false, forInit) }
        if (this.options.ecmaVersion >= 8 && id.name === "async" && this.type === types$1.name && !containsEsc &&
            (!this.potentialArrowInForAwait || this.value !== "of" || this.containsEsc)) {
          id = this.parseIdent(false);
          if (this.canInsertSemicolon() || !this.eat(types$1.arrow))
            { this.unexpected(); }
          return this.parseArrowExpression(this.startNodeAt(startPos, startLoc), [id], true, forInit)
        }
      }
      return id

    case types$1.regexp:
      var value = this.value;
      node = this.parseLiteral(value.value);
      node.regex = {pattern: value.pattern, flags: value.flags};
      return node

    case types$1.num: case types$1.string:
      return this.parseLiteral(this.value)

    case types$1._null: case types$1._true: case types$1._false:
      node = this.startNode();
      node.value = this.type === types$1._null ? null : this.type === types$1._true;
      node.raw = this.type.keyword;
      this.next();
      return this.finishNode(node, "Literal")

    case types$1.parenL:
      var start = this.start, expr = this.parseParenAndDistinguishExpression(canBeArrow, forInit);
      if (refDestructuringErrors) {
        if (refDestructuringErrors.parenthesizedAssign < 0 && !this.isSimpleAssignTarget(expr))
          { refDestructuringErrors.parenthesizedAssign = start; }
        if (refDestructuringErrors.parenthesizedBind < 0)
          { refDestructuringErrors.parenthesizedBind = start; }
      }
      return expr

    case types$1.bracketL:
      node = this.startNode();
      this.next();
      node.elements = this.parseExprList(types$1.bracketR, true, true, refDestructuringErrors);
      return this.finishNode(node, "ArrayExpression")

    case types$1.braceL:
      this.overrideContext(types.b_expr);
      return this.parseObj(false, refDestructuringErrors)

    case types$1._function:
      node = this.startNode();
      this.next();
      return this.parseFunction(node, 0)

    case types$1._class:
      return this.parseClass(this.startNode(), false)

    case types$1._new:
      return this.parseNew()

    case types$1.backQuote:
      return this.parseTemplate()

    case types$1._import:
      if (this.options.ecmaVersion >= 11) {
        return this.parseExprImport(forNew)
      } else {
        return this.unexpected()
      }

    default:
      return this.parseExprAtomDefault()
    }
  };

  pp$5.parseExprAtomDefault = function() {
    this.unexpected();
  };

  pp$5.parseExprImport = function(forNew) {
    var node = this.startNode();

    // Consume `import` as an identifier for `import.meta`.
    // Because `this.parseIdent(true)` doesn't check escape sequences, it needs the check of `this.containsEsc`.
    if (this.containsEsc) { this.raiseRecoverable(this.start, "Escape sequence in keyword import"); }
    this.next();

    if (this.type === types$1.parenL && !forNew) {
      return this.parseDynamicImport(node)
    } else if (this.type === types$1.dot) {
      var meta = this.startNodeAt(node.start, node.loc && node.loc.start);
      meta.name = "import";
      node.meta = this.finishNode(meta, "Identifier");
      return this.parseImportMeta(node)
    } else {
      this.unexpected();
    }
  };

  pp$5.parseDynamicImport = function(node) {
    this.next(); // skip `(`

    // Parse node.source.
    node.source = this.parseMaybeAssign();

    // Verify ending.
    if (!this.eat(types$1.parenR)) {
      var errorPos = this.start;
      if (this.eat(types$1.comma) && this.eat(types$1.parenR)) {
        this.raiseRecoverable(errorPos, "Trailing comma is not allowed in import()");
      } else {
        this.unexpected(errorPos);
      }
    }

    return this.finishNode(node, "ImportExpression")
  };

  pp$5.parseImportMeta = function(node) {
    this.next(); // skip `.`

    var containsEsc = this.containsEsc;
    node.property = this.parseIdent(true);

    if (node.property.name !== "meta")
      { this.raiseRecoverable(node.property.start, "The only valid meta property for import is 'import.meta'"); }
    if (containsEsc)
      { this.raiseRecoverable(node.start, "'import.meta' must not contain escaped characters"); }
    if (this.options.sourceType !== "module" && !this.options.allowImportExportEverywhere)
      { this.raiseRecoverable(node.start, "Cannot use 'import.meta' outside a module"); }

    return this.finishNode(node, "MetaProperty")
  };

  pp$5.parseLiteral = function(value) {
    var node = this.startNode();
    node.value = value;
    node.raw = this.input.slice(this.start, this.end);
    if (node.raw.charCodeAt(node.raw.length - 1) === 110) { node.bigint = node.raw.slice(0, -1).replace(/_/g, ""); }
    this.next();
    return this.finishNode(node, "Literal")
  };

  pp$5.parseParenExpression = function() {
    this.expect(types$1.parenL);
    var val = this.parseExpression();
    this.expect(types$1.parenR);
    return val
  };

  pp$5.shouldParseArrow = function(exprList) {
    return !this.canInsertSemicolon()
  };

  pp$5.parseParenAndDistinguishExpression = function(canBeArrow, forInit) {
    var startPos = this.start, startLoc = this.startLoc, val, allowTrailingComma = this.options.ecmaVersion >= 8;
    if (this.options.ecmaVersion >= 6) {
      this.next();

      var innerStartPos = this.start, innerStartLoc = this.startLoc;
      var exprList = [], first = true, lastIsComma = false;
      var refDestructuringErrors = new DestructuringErrors, oldYieldPos = this.yieldPos, oldAwaitPos = this.awaitPos, spreadStart;
      this.yieldPos = 0;
      this.awaitPos = 0;
      // Do not save awaitIdentPos to allow checking awaits nested in parameters
      while (this.type !== types$1.parenR) {
        first ? first = false : this.expect(types$1.comma);
        if (allowTrailingComma && this.afterTrailingComma(types$1.parenR, true)) {
          lastIsComma = true;
          break
        } else if (this.type === types$1.ellipsis) {
          spreadStart = this.start;
          exprList.push(this.parseParenItem(this.parseRestBinding()));
          if (this.type === types$1.comma) {
            this.raiseRecoverable(
              this.start,
              "Comma is not permitted after the rest element"
            );
          }
          break
        } else {
          exprList.push(this.parseMaybeAssign(false, refDestructuringErrors, this.parseParenItem));
        }
      }
      var innerEndPos = this.lastTokEnd, innerEndLoc = this.lastTokEndLoc;
      this.expect(types$1.parenR);

      if (canBeArrow && this.shouldParseArrow(exprList) && this.eat(types$1.arrow)) {
        this.checkPatternErrors(refDestructuringErrors, false);
        this.checkYieldAwaitInDefaultParams();
        this.yieldPos = oldYieldPos;
        this.awaitPos = oldAwaitPos;
        return this.parseParenArrowList(startPos, startLoc, exprList, forInit)
      }

      if (!exprList.length || lastIsComma) { this.unexpected(this.lastTokStart); }
      if (spreadStart) { this.unexpected(spreadStart); }
      this.checkExpressionErrors(refDestructuringErrors, true);
      this.yieldPos = oldYieldPos || this.yieldPos;
      this.awaitPos = oldAwaitPos || this.awaitPos;

      if (exprList.length > 1) {
        val = this.startNodeAt(innerStartPos, innerStartLoc);
        val.expressions = exprList;
        this.finishNodeAt(val, "SequenceExpression", innerEndPos, innerEndLoc);
      } else {
        val = exprList[0];
      }
    } else {
      val = this.parseParenExpression();
    }

    if (this.options.preserveParens) {
      var par = this.startNodeAt(startPos, startLoc);
      par.expression = val;
      return this.finishNode(par, "ParenthesizedExpression")
    } else {
      return val
    }
  };

  pp$5.parseParenItem = function(item) {
    return item
  };

  pp$5.parseParenArrowList = function(startPos, startLoc, exprList, forInit) {
    return this.parseArrowExpression(this.startNodeAt(startPos, startLoc), exprList, false, forInit)
  };

  // New's precedence is slightly tricky. It must allow its argument to
  // be a `[]` or dot subscript expression, but not a call — at least,
  // not without wrapping it in parentheses. Thus, it uses the noCalls
  // argument to parseSubscripts to prevent it from consuming the
  // argument list.

  var empty = [];

  pp$5.parseNew = function() {
    if (this.containsEsc) { this.raiseRecoverable(this.start, "Escape sequence in keyword new"); }
    var node = this.startNode();
    this.next();
    if (this.options.ecmaVersion >= 6 && this.type === types$1.dot) {
      var meta = this.startNodeAt(node.start, node.loc && node.loc.start);
      meta.name = "new";
      node.meta = this.finishNode(meta, "Identifier");
      this.next();
      var containsEsc = this.containsEsc;
      node.property = this.parseIdent(true);
      if (node.property.name !== "target")
        { this.raiseRecoverable(node.property.start, "The only valid meta property for new is 'new.target'"); }
      if (containsEsc)
        { this.raiseRecoverable(node.start, "'new.target' must not contain escaped characters"); }
      if (!this.allowNewDotTarget)
        { this.raiseRecoverable(node.start, "'new.target' can only be used in functions and class static block"); }
      return this.finishNode(node, "MetaProperty")
    }
    var startPos = this.start, startLoc = this.startLoc;
    node.callee = this.parseSubscripts(this.parseExprAtom(null, false, true), startPos, startLoc, true, false);
    if (this.eat(types$1.parenL)) { node.arguments = this.parseExprList(types$1.parenR, this.options.ecmaVersion >= 8, false); }
    else { node.arguments = empty; }
    return this.finishNode(node, "NewExpression")
  };

  // Parse template expression.

  pp$5.parseTemplateElement = function(ref) {
    var isTagged = ref.isTagged;

    var elem = this.startNode();
    if (this.type === types$1.invalidTemplate) {
      if (!isTagged) {
        this.raiseRecoverable(this.start, "Bad escape sequence in untagged template literal");
      }
      elem.value = {
        raw: this.value.replace(/\r\n?/g, "\n"),
        cooked: null
      };
    } else {
      elem.value = {
        raw: this.input.slice(this.start, this.end).replace(/\r\n?/g, "\n"),
        cooked: this.value
      };
    }
    this.next();
    elem.tail = this.type === types$1.backQuote;
    return this.finishNode(elem, "TemplateElement")
  };

  pp$5.parseTemplate = function(ref) {
    if ( ref === void 0 ) ref = {};
    var isTagged = ref.isTagged; if ( isTagged === void 0 ) isTagged = false;

    var node = this.startNode();
    this.next();
    node.expressions = [];
    var curElt = this.parseTemplateElement({isTagged: isTagged});
    node.quasis = [curElt];
    while (!curElt.tail) {
      if (this.type === types$1.eof) { this.raise(this.pos, "Unterminated template literal"); }
      this.expect(types$1.dollarBraceL);
      node.expressions.push(this.parseExpression());
      this.expect(types$1.braceR);
      node.quasis.push(curElt = this.parseTemplateElement({isTagged: isTagged}));
    }
    this.next();
    return this.finishNode(node, "TemplateLiteral")
  };

  pp$5.isAsyncProp = function(prop) {
    return !prop.computed && prop.key.type === "Identifier" && prop.key.name === "async" &&
      (this.type === types$1.name || this.type === types$1.num || this.type === types$1.string || this.type === types$1.bracketL || this.type.keyword || (this.options.ecmaVersion >= 9 && this.type === types$1.star)) &&
      !lineBreak.test(this.input.slice(this.lastTokEnd, this.start))
  };

  // Parse an object literal or binding pattern.

  pp$5.parseObj = function(isPattern, refDestructuringErrors) {
    var node = this.startNode(), first = true, propHash = {};
    node.properties = [];
    this.next();
    while (!this.eat(types$1.braceR)) {
      if (!first) {
        this.expect(types$1.comma);
        if (this.options.ecmaVersion >= 5 && this.afterTrailingComma(types$1.braceR)) { break }
      } else { first = false; }

      var prop = this.parseProperty(isPattern, refDestructuringErrors);
      if (!isPattern) { this.checkPropClash(prop, propHash, refDestructuringErrors); }
      node.properties.push(prop);
    }
    return this.finishNode(node, isPattern ? "ObjectPattern" : "ObjectExpression")
  };

  pp$5.parseProperty = function(isPattern, refDestructuringErrors) {
    var prop = this.startNode(), isGenerator, isAsync, startPos, startLoc;
    if (this.options.ecmaVersion >= 9 && this.eat(types$1.ellipsis)) {
      if (isPattern) {
        prop.argument = this.parseIdent(false);
        if (this.type === types$1.comma) {
          this.raiseRecoverable(this.start, "Comma is not permitted after the rest element");
        }
        return this.finishNode(prop, "RestElement")
      }
      // Parse argument.
      prop.argument = this.parseMaybeAssign(false, refDestructuringErrors);
      // To disallow trailing comma via `this.toAssignable()`.
      if (this.type === types$1.comma && refDestructuringErrors && refDestructuringErrors.trailingComma < 0) {
        refDestructuringErrors.trailingComma = this.start;
      }
      // Finish
      return this.finishNode(prop, "SpreadElement")
    }
    if (this.options.ecmaVersion >= 6) {
      prop.method = false;
      prop.shorthand = false;
      if (isPattern || refDestructuringErrors) {
        startPos = this.start;
        startLoc = this.startLoc;
      }
      if (!isPattern)
        { isGenerator = this.eat(types$1.star); }
    }
    var containsEsc = this.containsEsc;
    this.parsePropertyName(prop);
    if (!isPattern && !containsEsc && this.options.ecmaVersion >= 8 && !isGenerator && this.isAsyncProp(prop)) {
      isAsync = true;
      isGenerator = this.options.ecmaVersion >= 9 && this.eat(types$1.star);
      this.parsePropertyName(prop);
    } else {
      isAsync = false;
    }
    this.parsePropertyValue(prop, isPattern, isGenerator, isAsync, startPos, startLoc, refDestructuringErrors, containsEsc);
    return this.finishNode(prop, "Property")
  };

  pp$5.parseGetterSetter = function(prop) {
    prop.kind = prop.key.name;
    this.parsePropertyName(prop);
    prop.value = this.parseMethod(false);
    var paramCount = prop.kind === "get" ? 0 : 1;
    if (prop.value.params.length !== paramCount) {
      var start = prop.value.start;
      if (prop.kind === "get")
        { this.raiseRecoverable(start, "getter should have no params"); }
      else
        { this.raiseRecoverable(start, "setter should have exactly one param"); }
    } else {
      if (prop.kind === "set" && prop.value.params[0].type === "RestElement")
        { this.raiseRecoverable(prop.value.params[0].start, "Setter cannot use rest params"); }
    }
  };

  pp$5.parsePropertyValue = function(prop, isPattern, isGenerator, isAsync, startPos, startLoc, refDestructuringErrors, containsEsc) {
    if ((isGenerator || isAsync) && this.type === types$1.colon)
      { this.unexpected(); }

    if (this.eat(types$1.colon)) {
      prop.value = isPattern ? this.parseMaybeDefault(this.start, this.startLoc) : this.parseMaybeAssign(false, refDestructuringErrors);
      prop.kind = "init";
    } else if (this.options.ecmaVersion >= 6 && this.type === types$1.parenL) {
      if (isPattern) { this.unexpected(); }
      prop.kind = "init";
      prop.method = true;
      prop.value = this.parseMethod(isGenerator, isAsync);
    } else if (!isPattern && !containsEsc &&
               this.options.ecmaVersion >= 5 && !prop.computed && prop.key.type === "Identifier" &&
               (prop.key.name === "get" || prop.key.name === "set") &&
               (this.type !== types$1.comma && this.type !== types$1.braceR && this.type !== types$1.eq)) {
      if (isGenerator || isAsync) { this.unexpected(); }
      this.parseGetterSetter(prop);
    } else if (this.options.ecmaVersion >= 6 && !prop.computed && prop.key.type === "Identifier") {
      if (isGenerator || isAsync) { this.unexpected(); }
      this.checkUnreserved(prop.key);
      if (prop.key.name === "await" && !this.awaitIdentPos)
        { this.awaitIdentPos = startPos; }
      prop.kind = "init";
      if (isPattern) {
        prop.value = this.parseMaybeDefault(startPos, startLoc, this.copyNode(prop.key));
      } else if (this.type === types$1.eq && refDestructuringErrors) {
        if (refDestructuringErrors.shorthandAssign < 0)
          { refDestructuringErrors.shorthandAssign = this.start; }
        prop.value = this.parseMaybeDefault(startPos, startLoc, this.copyNode(prop.key));
      } else {
        prop.value = this.copyNode(prop.key);
      }
      prop.shorthand = true;
    } else { this.unexpected(); }
  };

  pp$5.parsePropertyName = function(prop) {
    if (this.options.ecmaVersion >= 6) {
      if (this.eat(types$1.bracketL)) {
        prop.computed = true;
        prop.key = this.parseMaybeAssign();
        this.expect(types$1.bracketR);
        return prop.key
      } else {
        prop.computed = false;
      }
    }
    return prop.key = this.type === types$1.num || this.type === types$1.string ? this.parseExprAtom() : this.parseIdent(this.options.allowReserved !== "never")
  };

  // Initialize empty function node.

  pp$5.initFunction = function(node) {
    node.id = null;
    if (this.options.ecmaVersion >= 6) { node.generator = node.expression = false; }
    if (this.options.ecmaVersion >= 8) { node.async = false; }
  };

  // Parse object or class method.

  pp$5.parseMethod = function(isGenerator, isAsync, allowDirectSuper) {
    var node = this.startNode(), oldYieldPos = this.yieldPos, oldAwaitPos = this.awaitPos, oldAwaitIdentPos = this.awaitIdentPos;

    this.initFunction(node);
    if (this.options.ecmaVersion >= 6)
      { node.generator = isGenerator; }
    if (this.options.ecmaVersion >= 8)
      { node.async = !!isAsync; }

    this.yieldPos = 0;
    this.awaitPos = 0;
    this.awaitIdentPos = 0;
    this.enterScope(functionFlags(isAsync, node.generator) | SCOPE_SUPER | (allowDirectSuper ? SCOPE_DIRECT_SUPER : 0));

    this.expect(types$1.parenL);
    node.params = this.parseBindingList(types$1.parenR, false, this.options.ecmaVersion >= 8);
    this.checkYieldAwaitInDefaultParams();
    this.parseFunctionBody(node, false, true, false);

    this.yieldPos = oldYieldPos;
    this.awaitPos = oldAwaitPos;
    this.awaitIdentPos = oldAwaitIdentPos;
    return this.finishNode(node, "FunctionExpression")
  };

  // Parse arrow function expression with given parameters.

  pp$5.parseArrowExpression = function(node, params, isAsync, forInit) {
    var oldYieldPos = this.yieldPos, oldAwaitPos = this.awaitPos, oldAwaitIdentPos = this.awaitIdentPos;

    this.enterScope(functionFlags(isAsync, false) | SCOPE_ARROW);
    this.initFunction(node);
    if (this.options.ecmaVersion >= 8) { node.async = !!isAsync; }

    this.yieldPos = 0;
    this.awaitPos = 0;
    this.awaitIdentPos = 0;

    node.params = this.toAssignableList(params, true);
    this.parseFunctionBody(node, true, false, forInit);

    this.yieldPos = oldYieldPos;
    this.awaitPos = oldAwaitPos;
    this.awaitIdentPos = oldAwaitIdentPos;
    return this.finishNode(node, "ArrowFunctionExpression")
  };

  // Parse function body and check parameters.

  pp$5.parseFunctionBody = function(node, isArrowFunction, isMethod, forInit) {
    var isExpression = isArrowFunction && this.type !== types$1.braceL;
    var oldStrict = this.strict, useStrict = false;

    if (isExpression) {
      node.body = this.parseMaybeAssign(forInit);
      node.expression = true;
      this.checkParams(node, false);
    } else {
      var nonSimple = this.options.ecmaVersion >= 7 && !this.isSimpleParamList(node.params);
      if (!oldStrict || nonSimple) {
        useStrict = this.strictDirective(this.end);
        // If this is a strict mode function, verify that argument names
        // are not repeated, and it does not try to bind the words `eval`
        // or `arguments`.
        if (useStrict && nonSimple)
          { this.raiseRecoverable(node.start, "Illegal 'use strict' directive in function with non-simple parameter list"); }
      }
      // Start a new scope with regard to labels and the `inFunction`
      // flag (restore them to their old value afterwards).
      var oldLabels = this.labels;
      this.labels = [];
      if (useStrict) { this.strict = true; }

      // Add the params to varDeclaredNames to ensure that an error is thrown
      // if a let/const declaration in the function clashes with one of the params.
      this.checkParams(node, !oldStrict && !useStrict && !isArrowFunction && !isMethod && this.isSimpleParamList(node.params));
      // Ensure the function name isn't a forbidden identifier in strict mode, e.g. 'eval'
      if (this.strict && node.id) { this.checkLValSimple(node.id, BIND_OUTSIDE); }
      node.body = this.parseBlock(false, undefined, useStrict && !oldStrict);
      node.expression = false;
      this.adaptDirectivePrologue(node.body.body);
      this.labels = oldLabels;
    }
    this.exitScope();
  };

  pp$5.isSimpleParamList = function(params) {
    for (var i = 0, list = params; i < list.length; i += 1)
      {
      var param = list[i];

      if (param.type !== "Identifier") { return false
    } }
    return true
  };

  // Checks function params for various disallowed patterns such as using "eval"
  // or "arguments" and duplicate parameters.

  pp$5.checkParams = function(node, allowDuplicates) {
    var nameHash = Object.create(null);
    for (var i = 0, list = node.params; i < list.length; i += 1)
      {
      var param = list[i];

      this.checkLValInnerPattern(param, BIND_VAR, allowDuplicates ? null : nameHash);
    }
  };

  // Parses a comma-separated list of expressions, and returns them as
  // an array. `close` is the token type that ends the list, and
  // `allowEmpty` can be turned on to allow subsequent commas with
  // nothing in between them to be parsed as `null` (which is needed
  // for array literals).

  pp$5.parseExprList = function(close, allowTrailingComma, allowEmpty, refDestructuringErrors) {
    var elts = [], first = true;
    while (!this.eat(close)) {
      if (!first) {
        this.expect(types$1.comma);
        if (allowTrailingComma && this.afterTrailingComma(close)) { break }
      } else { first = false; }

      var elt = (void 0);
      if (allowEmpty && this.type === types$1.comma)
        { elt = null; }
      else if (this.type === types$1.ellipsis) {
        elt = this.parseSpread(refDestructuringErrors);
        if (refDestructuringErrors && this.type === types$1.comma && refDestructuringErrors.trailingComma < 0)
          { refDestructuringErrors.trailingComma = this.start; }
      } else {
        elt = this.parseMaybeAssign(false, refDestructuringErrors);
      }
      elts.push(elt);
    }
    return elts
  };

  pp$5.checkUnreserved = function(ref) {
    var start = ref.start;
    var end = ref.end;
    var name = ref.name;

    if (this.inGenerator && name === "yield")
      { this.raiseRecoverable(start, "Cannot use 'yield' as identifier inside a generator"); }
    if (this.inAsync && name === "await")
      { this.raiseRecoverable(start, "Cannot use 'await' as identifier inside an async function"); }
    if (this.currentThisScope().inClassFieldInit && name === "arguments")
      { this.raiseRecoverable(start, "Cannot use 'arguments' in class field initializer"); }
    if (this.inClassStaticBlock && (name === "arguments" || name === "await"))
      { this.raise(start, ("Cannot use " + name + " in class static initialization block")); }
    if (this.keywords.test(name))
      { this.raise(start, ("Unexpected keyword '" + name + "'")); }
    if (this.options.ecmaVersion < 6 &&
      this.input.slice(start, end).indexOf("\\") !== -1) { return }
    var re = this.strict ? this.reservedWordsStrict : this.reservedWords;
    if (re.test(name)) {
      if (!this.inAsync && name === "await")
        { this.raiseRecoverable(start, "Cannot use keyword 'await' outside an async function"); }
      this.raiseRecoverable(start, ("The keyword '" + name + "' is reserved"));
    }
  };

  // Parse the next token as an identifier. If `liberal` is true (used
  // when parsing properties), it will also convert keywords into
  // identifiers.

  pp$5.parseIdent = function(liberal) {
    var node = this.parseIdentNode();
    this.next(!!liberal);
    this.finishNode(node, "Identifier");
    if (!liberal) {
      this.checkUnreserved(node);
      if (node.name === "await" && !this.awaitIdentPos)
        { this.awaitIdentPos = node.start; }
    }
    return node
  };

  pp$5.parseIdentNode = function() {
    var node = this.startNode();
    if (this.type === types$1.name) {
      node.name = this.value;
    } else if (this.type.keyword) {
      node.name = this.type.keyword;

      // To fix https://github.com/acornjs/acorn/issues/575
      // `class` and `function` keywords push new context into this.context.
      // But there is no chance to pop the context if the keyword is consumed as an identifier such as a property name.
      // If the previous token is a dot, this does not apply because the context-managing code already ignored the keyword
      if ((node.name === "class" || node.name === "function") &&
        (this.lastTokEnd !== this.lastTokStart + 1 || this.input.charCodeAt(this.lastTokStart) !== 46)) {
        this.context.pop();
      }
      this.type = types$1.name;
    } else {
      this.unexpected();
    }
    return node
  };

  pp$5.parsePrivateIdent = function() {
    var node = this.startNode();
    if (this.type === types$1.privateId) {
      node.name = this.value;
    } else {
      this.unexpected();
    }
    this.next();
    this.finishNode(node, "PrivateIdentifier");

    // For validating existence
    if (this.options.checkPrivateFields) {
      if (this.privateNameStack.length === 0) {
        this.raise(node.start, ("Private field '#" + (node.name) + "' must be declared in an enclosing class"));
      } else {
        this.privateNameStack[this.privateNameStack.length - 1].used.push(node);
      }
    }

    return node
  };

  // Parses yield expression inside generator.

  pp$5.parseYield = function(forInit) {
    if (!this.yieldPos) { this.yieldPos = this.start; }

    var node = this.startNode();
    this.next();
    if (this.type === types$1.semi || this.canInsertSemicolon() || (this.type !== types$1.star && !this.type.startsExpr)) {
      node.delegate = false;
      node.argument = null;
    } else {
      node.delegate = this.eat(types$1.star);
      node.argument = this.parseMaybeAssign(forInit);
    }
    return this.finishNode(node, "YieldExpression")
  };

  pp$5.parseAwait = function(forInit) {
    if (!this.awaitPos) { this.awaitPos = this.start; }

    var node = this.startNode();
    this.next();
    node.argument = this.parseMaybeUnary(null, true, false, forInit);
    return this.finishNode(node, "AwaitExpression")
  };

  var pp$4 = Parser.prototype;

  // This function is used to raise exceptions on parse errors. It
  // takes an offset integer (into the current `input`) to indicate
  // the location of the error, attaches the position to the end
  // of the error message, and then raises a `SyntaxError` with that
  // message.

  pp$4.raise = function(pos, message) {
    var loc = getLineInfo(this.input, pos);
    message += " (" + loc.line + ":" + loc.column + ")";
    var err = new SyntaxError(message);
    err.pos = pos; err.loc = loc; err.raisedAt = this.pos;
    throw err
  };

  pp$4.raiseRecoverable = pp$4.raise;

  pp$4.curPosition = function() {
    if (this.options.locations) {
      return new Position(this.curLine, this.pos - this.lineStart)
    }
  };

  var pp$3 = Parser.prototype;

  var Scope = function Scope(flags) {
    this.flags = flags;
    // A list of var-declared names in the current lexical scope
    this.var = [];
    // A list of lexically-declared names in the current lexical scope
    this.lexical = [];
    // A list of lexically-declared FunctionDeclaration names in the current lexical scope
    this.functions = [];
    // A switch to disallow the identifier reference 'arguments'
    this.inClassFieldInit = false;
  };

  // The functions in this module keep track of declared variables in the current scope in order to detect duplicate variable names.

  pp$3.enterScope = function(flags) {
    this.scopeStack.push(new Scope(flags));
  };

  pp$3.exitScope = function() {
    this.scopeStack.pop();
  };

  // The spec says:
  // > At the top level of a function, or script, function declarations are
  // > treated like var declarations rather than like lexical declarations.
  pp$3.treatFunctionsAsVarInScope = function(scope) {
    return (scope.flags & SCOPE_FUNCTION) || !this.inModule && (scope.flags & SCOPE_TOP)
  };

  pp$3.declareName = function(name, bindingType, pos) {
    var redeclared = false;
    if (bindingType === BIND_LEXICAL) {
      var scope = this.currentScope();
      redeclared = scope.lexical.indexOf(name) > -1 || scope.functions.indexOf(name) > -1 || scope.var.indexOf(name) > -1;
      scope.lexical.push(name);
      if (this.inModule && (scope.flags & SCOPE_TOP))
        { delete this.undefinedExports[name]; }
    } else if (bindingType === BIND_SIMPLE_CATCH) {
      var scope$1 = this.currentScope();
      scope$1.lexical.push(name);
    } else if (bindingType === BIND_FUNCTION) {
      var scope$2 = this.currentScope();
      if (this.treatFunctionsAsVar)
        { redeclared = scope$2.lexical.indexOf(name) > -1; }
      else
        { redeclared = scope$2.lexical.indexOf(name) > -1 || scope$2.var.indexOf(name) > -1; }
      scope$2.functions.push(name);
    } else {
      for (var i = this.scopeStack.length - 1; i >= 0; --i) {
        var scope$3 = this.scopeStack[i];
        if (scope$3.lexical.indexOf(name) > -1 && !((scope$3.flags & SCOPE_SIMPLE_CATCH) && scope$3.lexical[0] === name) ||
            !this.treatFunctionsAsVarInScope(scope$3) && scope$3.functions.indexOf(name) > -1) {
          redeclared = true;
          break
        }
        scope$3.var.push(name);
        if (this.inModule && (scope$3.flags & SCOPE_TOP))
          { delete this.undefinedExports[name]; }
        if (scope$3.flags & SCOPE_VAR) { break }
      }
    }
    if (redeclared) { this.raiseRecoverable(pos, ("Identifier '" + name + "' has already been declared")); }
  };

  pp$3.checkLocalExport = function(id) {
    // scope.functions must be empty as Module code is always strict.
    if (this.scopeStack[0].lexical.indexOf(id.name) === -1 &&
        this.scopeStack[0].var.indexOf(id.name) === -1) {
      this.undefinedExports[id.name] = id;
    }
  };

  pp$3.currentScope = function() {
    return this.scopeStack[this.scopeStack.length - 1]
  };

  pp$3.currentVarScope = function() {
    for (var i = this.scopeStack.length - 1;; i--) {
      var scope = this.scopeStack[i];
      if (scope.flags & SCOPE_VAR) { return scope }
    }
  };

  // Could be useful for `this`, `new.target`, `super()`, `super.property`, and `super[property]`.
  pp$3.currentThisScope = function() {
    for (var i = this.scopeStack.length - 1;; i--) {
      var scope = this.scopeStack[i];
      if (scope.flags & SCOPE_VAR && !(scope.flags & SCOPE_ARROW)) { return scope }
    }
  };

  var Node = function Node(parser, pos, loc) {
    this.type = "";
    this.start = pos;
    this.end = 0;
    if (parser.options.locations)
      { this.loc = new SourceLocation(parser, loc); }
    if (parser.options.directSourceFile)
      { this.sourceFile = parser.options.directSourceFile; }
    if (parser.options.ranges)
      { this.range = [pos, 0]; }
  };

  // Start an AST node, attaching a start offset.

  var pp$2 = Parser.prototype;

  pp$2.startNode = function() {
    return new Node(this, this.start, this.startLoc)
  };

  pp$2.startNodeAt = function(pos, loc) {
    return new Node(this, pos, loc)
  };

  // Finish an AST node, adding `type` and `end` properties.

  function finishNodeAt(node, type, pos, loc) {
    node.type = type;
    node.end = pos;
    if (this.options.locations)
      { node.loc.end = loc; }
    if (this.options.ranges)
      { node.range[1] = pos; }
    return node
  }

  pp$2.finishNode = function(node, type) {
    return finishNodeAt.call(this, node, type, this.lastTokEnd, this.lastTokEndLoc)
  };

  // Finish node at given position

  pp$2.finishNodeAt = function(node, type, pos, loc) {
    return finishNodeAt.call(this, node, type, pos, loc)
  };

  pp$2.copyNode = function(node) {
    var newNode = new Node(this, node.start, this.startLoc);
    for (var prop in node) { newNode[prop] = node[prop]; }
    return newNode
  };

  // This file contains Unicode properties extracted from the ECMAScript specification.
  // The lists are extracted like so:
  // $$('#table-binary-unicode-properties > figure > table > tbody > tr > td:nth-child(1) code').map(el => el.innerText)

  // #table-binary-unicode-properties
  var ecma9BinaryProperties = "ASCII ASCII_Hex_Digit AHex Alphabetic Alpha Any Assigned Bidi_Control Bidi_C Bidi_Mirrored Bidi_M Case_Ignorable CI Cased Changes_When_Casefolded CWCF Changes_When_Casemapped CWCM Changes_When_Lowercased CWL Changes_When_NFKC_Casefolded CWKCF Changes_When_Titlecased CWT Changes_When_Uppercased CWU Dash Default_Ignorable_Code_Point DI Deprecated Dep Diacritic Dia Emoji Emoji_Component Emoji_Modifier Emoji_Modifier_Base Emoji_Presentation Extender Ext Grapheme_Base Gr_Base Grapheme_Extend Gr_Ext Hex_Digit Hex IDS_Binary_Operator IDSB IDS_Trinary_Operator IDST ID_Continue IDC ID_Start IDS Ideographic Ideo Join_Control Join_C Logical_Order_Exception LOE Lowercase Lower Math Noncharacter_Code_Point NChar Pattern_Syntax Pat_Syn Pattern_White_Space Pat_WS Quotation_Mark QMark Radical Regional_Indicator RI Sentence_Terminal STerm Soft_Dotted SD Terminal_Punctuation Term Unified_Ideograph UIdeo Uppercase Upper Variation_Selector VS White_Space space XID_Continue XIDC XID_Start XIDS";
  var ecma10BinaryProperties = ecma9BinaryProperties + " Extended_Pictographic";
  var ecma11BinaryProperties = ecma10BinaryProperties;
  var ecma12BinaryProperties = ecma11BinaryProperties + " EBase EComp EMod EPres ExtPict";
  var ecma13BinaryProperties = ecma12BinaryProperties;
  var ecma14BinaryProperties = ecma13BinaryProperties;

  var unicodeBinaryProperties = {
    9: ecma9BinaryProperties,
    10: ecma10BinaryProperties,
    11: ecma11BinaryProperties,
    12: ecma12BinaryProperties,
    13: ecma13BinaryProperties,
    14: ecma14BinaryProperties
  };

  // #table-binary-unicode-properties-of-strings
  var ecma14BinaryPropertiesOfStrings = "Basic_Emoji Emoji_Keycap_Sequence RGI_Emoji_Modifier_Sequence RGI_Emoji_Flag_Sequence RGI_Emoji_Tag_Sequence RGI_Emoji_ZWJ_Sequence RGI_Emoji";

  var unicodeBinaryPropertiesOfStrings = {
    9: "",
    10: "",
    11: "",
    12: "",
    13: "",
    14: ecma14BinaryPropertiesOfStrings
  };

  // #table-unicode-general-category-values
  var unicodeGeneralCategoryValues = "Cased_Letter LC Close_Punctuation Pe Connector_Punctuation Pc Control Cc cntrl Currency_Symbol Sc Dash_Punctuation Pd Decimal_Number Nd digit Enclosing_Mark Me Final_Punctuation Pf Format Cf Initial_Punctuation Pi Letter L Letter_Number Nl Line_Separator Zl Lowercase_Letter Ll Mark M Combining_Mark Math_Symbol Sm Modifier_Letter Lm Modifier_Symbol Sk Nonspacing_Mark Mn Number N Open_Punctuation Ps Other C Other_Letter Lo Other_Number No Other_Punctuation Po Other_Symbol So Paragraph_Separator Zp Private_Use Co Punctuation P punct Separator Z Space_Separator Zs Spacing_Mark Mc Surrogate Cs Symbol S Titlecase_Letter Lt Unassigned Cn Uppercase_Letter Lu";

  // #table-unicode-script-values
  var ecma9ScriptValues = "Adlam Adlm Ahom Anatolian_Hieroglyphs Hluw Arabic Arab Armenian Armn Avestan Avst Balinese Bali Bamum Bamu Bassa_Vah Bass Batak Batk Bengali Beng Bhaiksuki Bhks Bopomofo Bopo Brahmi Brah Braille Brai Buginese Bugi Buhid Buhd Canadian_Aboriginal Cans Carian Cari Caucasian_Albanian Aghb Chakma Cakm Cham Cham Cherokee Cher Common Zyyy Coptic Copt Qaac Cuneiform Xsux Cypriot Cprt Cyrillic Cyrl Deseret Dsrt Devanagari Deva Duployan Dupl Egyptian_Hieroglyphs Egyp Elbasan Elba Ethiopic Ethi Georgian Geor Glagolitic Glag Gothic Goth Grantha Gran Greek Grek Gujarati Gujr Gurmukhi Guru Han Hani Hangul Hang Hanunoo Hano Hatran Hatr Hebrew Hebr Hiragana Hira Imperial_Aramaic Armi Inherited Zinh Qaai Inscriptional_Pahlavi Phli Inscriptional_Parthian Prti Javanese Java Kaithi Kthi Kannada Knda Katakana Kana Kayah_Li Kali Kharoshthi Khar Khmer Khmr Khojki Khoj Khudawadi Sind Lao Laoo Latin Latn Lepcha Lepc Limbu Limb Linear_A Lina Linear_B Linb Lisu Lisu Lycian Lyci Lydian Lydi Mahajani Mahj Malayalam Mlym Mandaic Mand Manichaean Mani Marchen Marc Masaram_Gondi Gonm Meetei_Mayek Mtei Mende_Kikakui Mend Meroitic_Cursive Merc Meroitic_Hieroglyphs Mero Miao Plrd Modi Mongolian Mong Mro Mroo Multani Mult Myanmar Mymr Nabataean Nbat New_Tai_Lue Talu Newa Newa Nko Nkoo Nushu Nshu Ogham Ogam Ol_Chiki Olck Old_Hungarian Hung Old_Italic Ital Old_North_Arabian Narb Old_Permic Perm Old_Persian Xpeo Old_South_Arabian Sarb Old_Turkic Orkh Oriya Orya Osage Osge Osmanya Osma Pahawh_Hmong Hmng Palmyrene Palm Pau_Cin_Hau Pauc Phags_Pa Phag Phoenician Phnx Psalter_Pahlavi Phlp Rejang Rjng Runic Runr Samaritan Samr Saurashtra Saur Sharada Shrd Shavian Shaw Siddham Sidd SignWriting Sgnw Sinhala Sinh Sora_Sompeng Sora Soyombo Soyo Sundanese Sund Syloti_Nagri Sylo Syriac Syrc Tagalog Tglg Tagbanwa Tagb Tai_Le Tale Tai_Tham Lana Tai_Viet Tavt Takri Takr Tamil Taml Tangut Tang Telugu Telu Thaana Thaa Thai Thai Tibetan Tibt Tifinagh Tfng Tirhuta Tirh Ugaritic Ugar Vai Vaii Warang_Citi Wara Yi Yiii Zanabazar_Square Zanb";
  var ecma10ScriptValues = ecma9ScriptValues + " Dogra Dogr Gunjala_Gondi Gong Hanifi_Rohingya Rohg Makasar Maka Medefaidrin Medf Old_Sogdian Sogo Sogdian Sogd";
  var ecma11ScriptValues = ecma10ScriptValues + " Elymaic Elym Nandinagari Nand Nyiakeng_Puachue_Hmong Hmnp Wancho Wcho";
  var ecma12ScriptValues = ecma11ScriptValues + " Chorasmian Chrs Diak Dives_Akuru Khitan_Small_Script Kits Yezi Yezidi";
  var ecma13ScriptValues = ecma12ScriptValues + " Cypro_Minoan Cpmn Old_Uyghur Ougr Tangsa Tnsa Toto Vithkuqi Vith";
  var ecma14ScriptValues = ecma13ScriptValues + " Hrkt Katakana_Or_Hiragana Kawi Nag_Mundari Nagm Unknown Zzzz";

  var unicodeScriptValues = {
    9: ecma9ScriptValues,
    10: ecma10ScriptValues,
    11: ecma11ScriptValues,
    12: ecma12ScriptValues,
    13: ecma13ScriptValues,
    14: ecma14ScriptValues
  };

  var data = {};
  function buildUnicodeData(ecmaVersion) {
    var d = data[ecmaVersion] = {
      binary: wordsRegexp(unicodeBinaryProperties[ecmaVersion] + " " + unicodeGeneralCategoryValues),
      binaryOfStrings: wordsRegexp(unicodeBinaryPropertiesOfStrings[ecmaVersion]),
      nonBinary: {
        General_Category: wordsRegexp(unicodeGeneralCategoryValues),
        Script: wordsRegexp(unicodeScriptValues[ecmaVersion])
      }
    };
    d.nonBinary.Script_Extensions = d.nonBinary.Script;

    d.nonBinary.gc = d.nonBinary.General_Category;
    d.nonBinary.sc = d.nonBinary.Script;
    d.nonBinary.scx = d.nonBinary.Script_Extensions;
  }

  for (var i = 0, list$1 = [9, 10, 11, 12, 13, 14]; i < list$1.length; i += 1) {
    var ecmaVersion = list$1[i];

    buildUnicodeData(ecmaVersion);
  }

  var pp$1 = Parser.prototype;

  // Track disjunction structure to determine whether a duplicate
  // capture group name is allowed because it is in a separate branch.
  var BranchID = function BranchID(parent, base) {
    // Parent disjunction branch
    this.parent = parent;
    // Identifies this set of sibling branches
    this.base = base || this;
  };

  BranchID.prototype.separatedFrom = function separatedFrom (alt) {
    // A branch is separate from another branch if they or any of
    // their parents are siblings in a given disjunction
    for (var self = this; self; self = self.parent) {
      for (var other = alt; other; other = other.parent) {
        if (self.base === other.base && self !== other) { return true }
      }
    }
    return false
  };

  BranchID.prototype.sibling = function sibling () {
    return new BranchID(this.parent, this.base)
  };

  var RegExpValidationState = function RegExpValidationState(parser) {
    this.parser = parser;
    this.validFlags = "gim" + (parser.options.ecmaVersion >= 6 ? "uy" : "") + (parser.options.ecmaVersion >= 9 ? "s" : "") + (parser.options.ecmaVersion >= 13 ? "d" : "") + (parser.options.ecmaVersion >= 15 ? "v" : "");
    this.unicodeProperties = data[parser.options.ecmaVersion >= 14 ? 14 : parser.options.ecmaVersion];
    this.source = "";
    this.flags = "";
    this.start = 0;
    this.switchU = false;
    this.switchV = false;
    this.switchN = false;
    this.pos = 0;
    this.lastIntValue = 0;
    this.lastStringValue = "";
    this.lastAssertionIsQuantifiable = false;
    this.numCapturingParens = 0;
    this.maxBackReference = 0;
    this.groupNames = Object.create(null);
    this.backReferenceNames = [];
    this.branchID = null;
  };

  RegExpValidationState.prototype.reset = function reset (start, pattern, flags) {
    var unicodeSets = flags.indexOf("v") !== -1;
    var unicode = flags.indexOf("u") !== -1;
    this.start = start | 0;
    this.source = pattern + "";
    this.flags = flags;
    if (unicodeSets && this.parser.options.ecmaVersion >= 15) {
      this.switchU = true;
      this.switchV = true;
      this.switchN = true;
    } else {
      this.switchU = unicode && this.parser.options.ecmaVersion >= 6;
      this.switchV = false;
      this.switchN = unicode && this.parser.options.ecmaVersion >= 9;
    }
  };

  RegExpValidationState.prototype.raise = function raise (message) {
    this.parser.raiseRecoverable(this.start, ("Invalid regular expression: /" + (this.source) + "/: " + message));
  };

  // If u flag is given, this returns the code point at the index (it combines a surrogate pair).
  // Otherwise, this returns the code unit of the index (can be a part of a surrogate pair).
  RegExpValidationState.prototype.at = function at (i, forceU) {
      if ( forceU === void 0 ) forceU = false;

    var s = this.source;
    var l = s.length;
    if (i >= l) {
      return -1
    }
    var c = s.charCodeAt(i);
    if (!(forceU || this.switchU) || c <= 0xD7FF || c >= 0xE000 || i + 1 >= l) {
      return c
    }
    var next = s.charCodeAt(i + 1);
    return next >= 0xDC00 && next <= 0xDFFF ? (c << 10) + next - 0x35FDC00 : c
  };

  RegExpValidationState.prototype.nextIndex = function nextIndex (i, forceU) {
      if ( forceU === void 0 ) forceU = false;

    var s = this.source;
    var l = s.length;
    if (i >= l) {
      return l
    }
    var c = s.charCodeAt(i), next;
    if (!(forceU || this.switchU) || c <= 0xD7FF || c >= 0xE000 || i + 1 >= l ||
        (next = s.charCodeAt(i + 1)) < 0xDC00 || next > 0xDFFF) {
      return i + 1
    }
    return i + 2
  };

  RegExpValidationState.prototype.current = function current (forceU) {
      if ( forceU === void 0 ) forceU = false;

    return this.at(this.pos, forceU)
  };

  RegExpValidationState.prototype.lookahead = function lookahead (forceU) {
      if ( forceU === void 0 ) forceU = false;

    return this.at(this.nextIndex(this.pos, forceU), forceU)
  };

  RegExpValidationState.prototype.advance = function advance (forceU) {
      if ( forceU === void 0 ) forceU = false;

    this.pos = this.nextIndex(this.pos, forceU);
  };

  RegExpValidationState.prototype.eat = function eat (ch, forceU) {
      if ( forceU === void 0 ) forceU = false;

    if (this.current(forceU) === ch) {
      this.advance(forceU);
      return true
    }
    return false
  };

  RegExpValidationState.prototype.eatChars = function eatChars (chs, forceU) {
      if ( forceU === void 0 ) forceU = false;

    var pos = this.pos;
    for (var i = 0, list = chs; i < list.length; i += 1) {
      var ch = list[i];

        var current = this.at(pos, forceU);
      if (current === -1 || current !== ch) {
        return false
      }
      pos = this.nextIndex(pos, forceU);
    }
    this.pos = pos;
    return true
  };

  /**
   * Validate the flags part of a given RegExpLiteral.
   *
   * @param {RegExpValidationState} state The state to validate RegExp.
   * @returns {void}
   */
  pp$1.validateRegExpFlags = function(state) {
    var validFlags = state.validFlags;
    var flags = state.flags;

    var u = false;
    var v = false;

    for (var i = 0; i < flags.length; i++) {
      var flag = flags.charAt(i);
      if (validFlags.indexOf(flag) === -1) {
        this.raise(state.start, "Invalid regular expression flag");
      }
      if (flags.indexOf(flag, i + 1) > -1) {
        this.raise(state.start, "Duplicate regular expression flag");
      }
      if (flag === "u") { u = true; }
      if (flag === "v") { v = true; }
    }
    if (this.options.ecmaVersion >= 15 && u && v) {
      this.raise(state.start, "Invalid regular expression flag");
    }
  };

  function hasProp(obj) {
    for (var _ in obj) { return true }
    return false
  }

  /**
   * Validate the pattern part of a given RegExpLiteral.
   *
   * @param {RegExpValidationState} state The state to validate RegExp.
   * @returns {void}
   */
  pp$1.validateRegExpPattern = function(state) {
    this.regexp_pattern(state);

    // The goal symbol for the parse is |Pattern[~U, ~N]|. If the result of
    // parsing contains a |GroupName|, reparse with the goal symbol
    // |Pattern[~U, +N]| and use this result instead. Throw a *SyntaxError*
    // exception if _P_ did not conform to the grammar, if any elements of _P_
    // were not matched by the parse, or if any Early Error conditions exist.
    if (!state.switchN && this.options.ecmaVersion >= 9 && hasProp(state.groupNames)) {
      state.switchN = true;
      this.regexp_pattern(state);
    }
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-Pattern
  pp$1.regexp_pattern = function(state) {
    state.pos = 0;
    state.lastIntValue = 0;
    state.lastStringValue = "";
    state.lastAssertionIsQuantifiable = false;
    state.numCapturingParens = 0;
    state.maxBackReference = 0;
    state.groupNames = Object.create(null);
    state.backReferenceNames.length = 0;
    state.branchID = null;

    this.regexp_disjunction(state);

    if (state.pos !== state.source.length) {
      // Make the same messages as V8.
      if (state.eat(0x29 /* ) */)) {
        state.raise("Unmatched ')'");
      }
      if (state.eat(0x5D /* ] */) || state.eat(0x7D /* } */)) {
        state.raise("Lone quantifier brackets");
      }
    }
    if (state.maxBackReference > state.numCapturingParens) {
      state.raise("Invalid escape");
    }
    for (var i = 0, list = state.backReferenceNames; i < list.length; i += 1) {
      var name = list[i];

      if (!state.groupNames[name]) {
        state.raise("Invalid named capture referenced");
      }
    }
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-Disjunction
  pp$1.regexp_disjunction = function(state) {
    var trackDisjunction = this.options.ecmaVersion >= 16;
    if (trackDisjunction) { state.branchID = new BranchID(state.branchID, null); }
    this.regexp_alternative(state);
    while (state.eat(0x7C /* | */)) {
      if (trackDisjunction) { state.branchID = state.branchID.sibling(); }
      this.regexp_alternative(state);
    }
    if (trackDisjunction) { state.branchID = state.branchID.parent; }

    // Make the same message as V8.
    if (this.regexp_eatQuantifier(state, true)) {
      state.raise("Nothing to repeat");
    }
    if (state.eat(0x7B /* { */)) {
      state.raise("Lone quantifier brackets");
    }
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-Alternative
  pp$1.regexp_alternative = function(state) {
    while (state.pos < state.source.length && this.regexp_eatTerm(state)) {}
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-annexB-Term
  pp$1.regexp_eatTerm = function(state) {
    if (this.regexp_eatAssertion(state)) {
      // Handle `QuantifiableAssertion Quantifier` alternative.
      // `state.lastAssertionIsQuantifiable` is true if the last eaten Assertion
      // is a QuantifiableAssertion.
      if (state.lastAssertionIsQuantifiable && this.regexp_eatQuantifier(state)) {
        // Make the same message as V8.
        if (state.switchU) {
          state.raise("Invalid quantifier");
        }
      }
      return true
    }

    if (state.switchU ? this.regexp_eatAtom(state) : this.regexp_eatExtendedAtom(state)) {
      this.regexp_eatQuantifier(state);
      return true
    }

    return false
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-annexB-Assertion
  pp$1.regexp_eatAssertion = function(state) {
    var start = state.pos;
    state.lastAssertionIsQuantifiable = false;

    // ^, $
    if (state.eat(0x5E /* ^ */) || state.eat(0x24 /* $ */)) {
      return true
    }

    // \b \B
    if (state.eat(0x5C /* \ */)) {
      if (state.eat(0x42 /* B */) || state.eat(0x62 /* b */)) {
        return true
      }
      state.pos = start;
    }

    // Lookahead / Lookbehind
    if (state.eat(0x28 /* ( */) && state.eat(0x3F /* ? */)) {
      var lookbehind = false;
      if (this.options.ecmaVersion >= 9) {
        lookbehind = state.eat(0x3C /* < */);
      }
      if (state.eat(0x3D /* = */) || state.eat(0x21 /* ! */)) {
        this.regexp_disjunction(state);
        if (!state.eat(0x29 /* ) */)) {
          state.raise("Unterminated group");
        }
        state.lastAssertionIsQuantifiable = !lookbehind;
        return true
      }
    }

    state.pos = start;
    return false
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-Quantifier
  pp$1.regexp_eatQuantifier = function(state, noError) {
    if ( noError === void 0 ) noError = false;

    if (this.regexp_eatQuantifierPrefix(state, noError)) {
      state.eat(0x3F /* ? */);
      return true
    }
    return false
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-QuantifierPrefix
  pp$1.regexp_eatQuantifierPrefix = function(state, noError) {
    return (
      state.eat(0x2A /* * */) ||
      state.eat(0x2B /* + */) ||
      state.eat(0x3F /* ? */) ||
      this.regexp_eatBracedQuantifier(state, noError)
    )
  };
  pp$1.regexp_eatBracedQuantifier = function(state, noError) {
    var start = state.pos;
    if (state.eat(0x7B /* { */)) {
      var min = 0, max = -1;
      if (this.regexp_eatDecimalDigits(state)) {
        min = state.lastIntValue;
        if (state.eat(0x2C /* , */) && this.regexp_eatDecimalDigits(state)) {
          max = state.lastIntValue;
        }
        if (state.eat(0x7D /* } */)) {
          // SyntaxError in https://www.ecma-international.org/ecma-262/8.0/#sec-term
          if (max !== -1 && max < min && !noError) {
            state.raise("numbers out of order in {} quantifier");
          }
          return true
        }
      }
      if (state.switchU && !noError) {
        state.raise("Incomplete quantifier");
      }
      state.pos = start;
    }
    return false
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-Atom
  pp$1.regexp_eatAtom = function(state) {
    return (
      this.regexp_eatPatternCharacters(state) ||
      state.eat(0x2E /* . */) ||
      this.regexp_eatReverseSolidusAtomEscape(state) ||
      this.regexp_eatCharacterClass(state) ||
      this.regexp_eatUncapturingGroup(state) ||
      this.regexp_eatCapturingGroup(state)
    )
  };
  pp$1.regexp_eatReverseSolidusAtomEscape = function(state) {
    var start = state.pos;
    if (state.eat(0x5C /* \ */)) {
      if (this.regexp_eatAtomEscape(state)) {
        return true
      }
      state.pos = start;
    }
    return false
  };
  pp$1.regexp_eatUncapturingGroup = function(state) {
    var start = state.pos;
    if (state.eat(0x28 /* ( */)) {
      if (state.eat(0x3F /* ? */) && state.eat(0x3A /* : */)) {
        this.regexp_disjunction(state);
        if (state.eat(0x29 /* ) */)) {
          return true
        }
        state.raise("Unterminated group");
      }
      state.pos = start;
    }
    return false
  };
  pp$1.regexp_eatCapturingGroup = function(state) {
    if (state.eat(0x28 /* ( */)) {
      if (this.options.ecmaVersion >= 9) {
        this.regexp_groupSpecifier(state);
      } else if (state.current() === 0x3F /* ? */) {
        state.raise("Invalid group");
      }
      this.regexp_disjunction(state);
      if (state.eat(0x29 /* ) */)) {
        state.numCapturingParens += 1;
        return true
      }
      state.raise("Unterminated group");
    }
    return false
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-annexB-ExtendedAtom
  pp$1.regexp_eatExtendedAtom = function(state) {
    return (
      state.eat(0x2E /* . */) ||
      this.regexp_eatReverseSolidusAtomEscape(state) ||
      this.regexp_eatCharacterClass(state) ||
      this.regexp_eatUncapturingGroup(state) ||
      this.regexp_eatCapturingGroup(state) ||
      this.regexp_eatInvalidBracedQuantifier(state) ||
      this.regexp_eatExtendedPatternCharacter(state)
    )
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-annexB-InvalidBracedQuantifier
  pp$1.regexp_eatInvalidBracedQuantifier = function(state) {
    if (this.regexp_eatBracedQuantifier(state, true)) {
      state.raise("Nothing to repeat");
    }
    return false
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-SyntaxCharacter
  pp$1.regexp_eatSyntaxCharacter = function(state) {
    var ch = state.current();
    if (isSyntaxCharacter(ch)) {
      state.lastIntValue = ch;
      state.advance();
      return true
    }
    return false
  };
  function isSyntaxCharacter(ch) {
    return (
      ch === 0x24 /* $ */ ||
      ch >= 0x28 /* ( */ && ch <= 0x2B /* + */ ||
      ch === 0x2E /* . */ ||
      ch === 0x3F /* ? */ ||
      ch >= 0x5B /* [ */ && ch <= 0x5E /* ^ */ ||
      ch >= 0x7B /* { */ && ch <= 0x7D /* } */
    )
  }

  // https://www.ecma-international.org/ecma-262/8.0/#prod-PatternCharacter
  // But eat eager.
  pp$1.regexp_eatPatternCharacters = function(state) {
    var start = state.pos;
    var ch = 0;
    while ((ch = state.current()) !== -1 && !isSyntaxCharacter(ch)) {
      state.advance();
    }
    return state.pos !== start
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-annexB-ExtendedPatternCharacter
  pp$1.regexp_eatExtendedPatternCharacter = function(state) {
    var ch = state.current();
    if (
      ch !== -1 &&
      ch !== 0x24 /* $ */ &&
      !(ch >= 0x28 /* ( */ && ch <= 0x2B /* + */) &&
      ch !== 0x2E /* . */ &&
      ch !== 0x3F /* ? */ &&
      ch !== 0x5B /* [ */ &&
      ch !== 0x5E /* ^ */ &&
      ch !== 0x7C /* | */
    ) {
      state.advance();
      return true
    }
    return false
  };

  // GroupSpecifier ::
  //   [empty]
  //   `?` GroupName
  pp$1.regexp_groupSpecifier = function(state) {
    if (state.eat(0x3F /* ? */)) {
      if (!this.regexp_eatGroupName(state)) { state.raise("Invalid group"); }
      var trackDisjunction = this.options.ecmaVersion >= 16;
      var known = state.groupNames[state.lastStringValue];
      if (known) {
        if (trackDisjunction) {
          for (var i = 0, list = known; i < list.length; i += 1) {
            var altID = list[i];

            if (!altID.separatedFrom(state.branchID))
              { state.raise("Duplicate capture group name"); }
          }
        } else {
          state.raise("Duplicate capture group name");
        }
      }
      if (trackDisjunction) {
        (known || (state.groupNames[state.lastStringValue] = [])).push(state.branchID);
      } else {
        state.groupNames[state.lastStringValue] = true;
      }
    }
  };

  // GroupName ::
  //   `<` RegExpIdentifierName `>`
  // Note: this updates `state.lastStringValue` property with the eaten name.
  pp$1.regexp_eatGroupName = function(state) {
    state.lastStringValue = "";
    if (state.eat(0x3C /* < */)) {
      if (this.regexp_eatRegExpIdentifierName(state) && state.eat(0x3E /* > */)) {
        return true
      }
      state.raise("Invalid capture group name");
    }
    return false
  };

  // RegExpIdentifierName ::
  //   RegExpIdentifierStart
  //   RegExpIdentifierName RegExpIdentifierPart
  // Note: this updates `state.lastStringValue` property with the eaten name.
  pp$1.regexp_eatRegExpIdentifierName = function(state) {
    state.lastStringValue = "";
    if (this.regexp_eatRegExpIdentifierStart(state)) {
      state.lastStringValue += codePointToString(state.lastIntValue);
      while (this.regexp_eatRegExpIdentifierPart(state)) {
        state.lastStringValue += codePointToString(state.lastIntValue);
      }
      return true
    }
    return false
  };

  // RegExpIdentifierStart ::
  //   UnicodeIDStart
  //   `$`
  //   `_`
  //   `\` RegExpUnicodeEscapeSequence[+U]
  pp$1.regexp_eatRegExpIdentifierStart = function(state) {
    var start = state.pos;
    var forceU = this.options.ecmaVersion >= 11;
    var ch = state.current(forceU);
    state.advance(forceU);

    if (ch === 0x5C /* \ */ && this.regexp_eatRegExpUnicodeEscapeSequence(state, forceU)) {
      ch = state.lastIntValue;
    }
    if (isRegExpIdentifierStart(ch)) {
      state.lastIntValue = ch;
      return true
    }

    state.pos = start;
    return false
  };
  function isRegExpIdentifierStart(ch) {
    return isIdentifierStart(ch, true) || ch === 0x24 /* $ */ || ch === 0x5F /* _ */
  }

  // RegExpIdentifierPart ::
  //   UnicodeIDContinue
  //   `$`
  //   `_`
  //   `\` RegExpUnicodeEscapeSequence[+U]
  //   <ZWNJ>
  //   <ZWJ>
  pp$1.regexp_eatRegExpIdentifierPart = function(state) {
    var start = state.pos;
    var forceU = this.options.ecmaVersion >= 11;
    var ch = state.current(forceU);
    state.advance(forceU);

    if (ch === 0x5C /* \ */ && this.regexp_eatRegExpUnicodeEscapeSequence(state, forceU)) {
      ch = state.lastIntValue;
    }
    if (isRegExpIdentifierPart(ch)) {
      state.lastIntValue = ch;
      return true
    }

    state.pos = start;
    return false
  };
  function isRegExpIdentifierPart(ch) {
    return isIdentifierChar(ch, true) || ch === 0x24 /* $ */ || ch === 0x5F /* _ */ || ch === 0x200C /* <ZWNJ> */ || ch === 0x200D /* <ZWJ> */
  }

  // https://www.ecma-international.org/ecma-262/8.0/#prod-annexB-AtomEscape
  pp$1.regexp_eatAtomEscape = function(state) {
    if (
      this.regexp_eatBackReference(state) ||
      this.regexp_eatCharacterClassEscape(state) ||
      this.regexp_eatCharacterEscape(state) ||
      (state.switchN && this.regexp_eatKGroupName(state))
    ) {
      return true
    }
    if (state.switchU) {
      // Make the same message as V8.
      if (state.current() === 0x63 /* c */) {
        state.raise("Invalid unicode escape");
      }
      state.raise("Invalid escape");
    }
    return false
  };
  pp$1.regexp_eatBackReference = function(state) {
    var start = state.pos;
    if (this.regexp_eatDecimalEscape(state)) {
      var n = state.lastIntValue;
      if (state.switchU) {
        // For SyntaxError in https://www.ecma-international.org/ecma-262/8.0/#sec-atomescape
        if (n > state.maxBackReference) {
          state.maxBackReference = n;
        }
        return true
      }
      if (n <= state.numCapturingParens) {
        return true
      }
      state.pos = start;
    }
    return false
  };
  pp$1.regexp_eatKGroupName = function(state) {
    if (state.eat(0x6B /* k */)) {
      if (this.regexp_eatGroupName(state)) {
        state.backReferenceNames.push(state.lastStringValue);
        return true
      }
      state.raise("Invalid named reference");
    }
    return false
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-annexB-CharacterEscape
  pp$1.regexp_eatCharacterEscape = function(state) {
    return (
      this.regexp_eatControlEscape(state) ||
      this.regexp_eatCControlLetter(state) ||
      this.regexp_eatZero(state) ||
      this.regexp_eatHexEscapeSequence(state) ||
      this.regexp_eatRegExpUnicodeEscapeSequence(state, false) ||
      (!state.switchU && this.regexp_eatLegacyOctalEscapeSequence(state)) ||
      this.regexp_eatIdentityEscape(state)
    )
  };
  pp$1.regexp_eatCControlLetter = function(state) {
    var start = state.pos;
    if (state.eat(0x63 /* c */)) {
      if (this.regexp_eatControlLetter(state)) {
        return true
      }
      state.pos = start;
    }
    return false
  };
  pp$1.regexp_eatZero = function(state) {
    if (state.current() === 0x30 /* 0 */ && !isDecimalDigit(state.lookahead())) {
      state.lastIntValue = 0;
      state.advance();
      return true
    }
    return false
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-ControlEscape
  pp$1.regexp_eatControlEscape = function(state) {
    var ch = state.current();
    if (ch === 0x74 /* t */) {
      state.lastIntValue = 0x09; /* \t */
      state.advance();
      return true
    }
    if (ch === 0x6E /* n */) {
      state.lastIntValue = 0x0A; /* \n */
      state.advance();
      return true
    }
    if (ch === 0x76 /* v */) {
      state.lastIntValue = 0x0B; /* \v */
      state.advance();
      return true
    }
    if (ch === 0x66 /* f */) {
      state.lastIntValue = 0x0C; /* \f */
      state.advance();
      return true
    }
    if (ch === 0x72 /* r */) {
      state.lastIntValue = 0x0D; /* \r */
      state.advance();
      return true
    }
    return false
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-ControlLetter
  pp$1.regexp_eatControlLetter = function(state) {
    var ch = state.current();
    if (isControlLetter(ch)) {
      state.lastIntValue = ch % 0x20;
      state.advance();
      return true
    }
    return false
  };
  function isControlLetter(ch) {
    return (
      (ch >= 0x41 /* A */ && ch <= 0x5A /* Z */) ||
      (ch >= 0x61 /* a */ && ch <= 0x7A /* z */)
    )
  }

  // https://www.ecma-international.org/ecma-262/8.0/#prod-RegExpUnicodeEscapeSequence
  pp$1.regexp_eatRegExpUnicodeEscapeSequence = function(state, forceU) {
    if ( forceU === void 0 ) forceU = false;

    var start = state.pos;
    var switchU = forceU || state.switchU;

    if (state.eat(0x75 /* u */)) {
      if (this.regexp_eatFixedHexDigits(state, 4)) {
        var lead = state.lastIntValue;
        if (switchU && lead >= 0xD800 && lead <= 0xDBFF) {
          var leadSurrogateEnd = state.pos;
          if (state.eat(0x5C /* \ */) && state.eat(0x75 /* u */) && this.regexp_eatFixedHexDigits(state, 4)) {
            var trail = state.lastIntValue;
            if (trail >= 0xDC00 && trail <= 0xDFFF) {
              state.lastIntValue = (lead - 0xD800) * 0x400 + (trail - 0xDC00) + 0x10000;
              return true
            }
          }
          state.pos = leadSurrogateEnd;
          state.lastIntValue = lead;
        }
        return true
      }
      if (
        switchU &&
        state.eat(0x7B /* { */) &&
        this.regexp_eatHexDigits(state) &&
        state.eat(0x7D /* } */) &&
        isValidUnicode(state.lastIntValue)
      ) {
        return true
      }
      if (switchU) {
        state.raise("Invalid unicode escape");
      }
      state.pos = start;
    }

    return false
  };
  function isValidUnicode(ch) {
    return ch >= 0 && ch <= 0x10FFFF
  }

  // https://www.ecma-international.org/ecma-262/8.0/#prod-annexB-IdentityEscape
  pp$1.regexp_eatIdentityEscape = function(state) {
    if (state.switchU) {
      if (this.regexp_eatSyntaxCharacter(state)) {
        return true
      }
      if (state.eat(0x2F /* / */)) {
        state.lastIntValue = 0x2F; /* / */
        return true
      }
      return false
    }

    var ch = state.current();
    if (ch !== 0x63 /* c */ && (!state.switchN || ch !== 0x6B /* k */)) {
      state.lastIntValue = ch;
      state.advance();
      return true
    }

    return false
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-DecimalEscape
  pp$1.regexp_eatDecimalEscape = function(state) {
    state.lastIntValue = 0;
    var ch = state.current();
    if (ch >= 0x31 /* 1 */ && ch <= 0x39 /* 9 */) {
      do {
        state.lastIntValue = 10 * state.lastIntValue + (ch - 0x30 /* 0 */);
        state.advance();
      } while ((ch = state.current()) >= 0x30 /* 0 */ && ch <= 0x39 /* 9 */)
      return true
    }
    return false
  };

  // Return values used by character set parsing methods, needed to
  // forbid negation of sets that can match strings.
  var CharSetNone = 0; // Nothing parsed
  var CharSetOk = 1; // Construct parsed, cannot contain strings
  var CharSetString = 2; // Construct parsed, can contain strings

  // https://www.ecma-international.org/ecma-262/8.0/#prod-CharacterClassEscape
  pp$1.regexp_eatCharacterClassEscape = function(state) {
    var ch = state.current();

    if (isCharacterClassEscape(ch)) {
      state.lastIntValue = -1;
      state.advance();
      return CharSetOk
    }

    var negate = false;
    if (
      state.switchU &&
      this.options.ecmaVersion >= 9 &&
      ((negate = ch === 0x50 /* P */) || ch === 0x70 /* p */)
    ) {
      state.lastIntValue = -1;
      state.advance();
      var result;
      if (
        state.eat(0x7B /* { */) &&
        (result = this.regexp_eatUnicodePropertyValueExpression(state)) &&
        state.eat(0x7D /* } */)
      ) {
        if (negate && result === CharSetString) { state.raise("Invalid property name"); }
        return result
      }
      state.raise("Invalid property name");
    }

    return CharSetNone
  };

  function isCharacterClassEscape(ch) {
    return (
      ch === 0x64 /* d */ ||
      ch === 0x44 /* D */ ||
      ch === 0x73 /* s */ ||
      ch === 0x53 /* S */ ||
      ch === 0x77 /* w */ ||
      ch === 0x57 /* W */
    )
  }

  // UnicodePropertyValueExpression ::
  //   UnicodePropertyName `=` UnicodePropertyValue
  //   LoneUnicodePropertyNameOrValue
  pp$1.regexp_eatUnicodePropertyValueExpression = function(state) {
    var start = state.pos;

    // UnicodePropertyName `=` UnicodePropertyValue
    if (this.regexp_eatUnicodePropertyName(state) && state.eat(0x3D /* = */)) {
      var name = state.lastStringValue;
      if (this.regexp_eatUnicodePropertyValue(state)) {
        var value = state.lastStringValue;
        this.regexp_validateUnicodePropertyNameAndValue(state, name, value);
        return CharSetOk
      }
    }
    state.pos = start;

    // LoneUnicodePropertyNameOrValue
    if (this.regexp_eatLoneUnicodePropertyNameOrValue(state)) {
      var nameOrValue = state.lastStringValue;
      return this.regexp_validateUnicodePropertyNameOrValue(state, nameOrValue)
    }
    return CharSetNone
  };

  pp$1.regexp_validateUnicodePropertyNameAndValue = function(state, name, value) {
    if (!hasOwn(state.unicodeProperties.nonBinary, name))
      { state.raise("Invalid property name"); }
    if (!state.unicodeProperties.nonBinary[name].test(value))
      { state.raise("Invalid property value"); }
  };

  pp$1.regexp_validateUnicodePropertyNameOrValue = function(state, nameOrValue) {
    if (state.unicodeProperties.binary.test(nameOrValue)) { return CharSetOk }
    if (state.switchV && state.unicodeProperties.binaryOfStrings.test(nameOrValue)) { return CharSetString }
    state.raise("Invalid property name");
  };

  // UnicodePropertyName ::
  //   UnicodePropertyNameCharacters
  pp$1.regexp_eatUnicodePropertyName = function(state) {
    var ch = 0;
    state.lastStringValue = "";
    while (isUnicodePropertyNameCharacter(ch = state.current())) {
      state.lastStringValue += codePointToString(ch);
      state.advance();
    }
    return state.lastStringValue !== ""
  };

  function isUnicodePropertyNameCharacter(ch) {
    return isControlLetter(ch) || ch === 0x5F /* _ */
  }

  // UnicodePropertyValue ::
  //   UnicodePropertyValueCharacters
  pp$1.regexp_eatUnicodePropertyValue = function(state) {
    var ch = 0;
    state.lastStringValue = "";
    while (isUnicodePropertyValueCharacter(ch = state.current())) {
      state.lastStringValue += codePointToString(ch);
      state.advance();
    }
    return state.lastStringValue !== ""
  };
  function isUnicodePropertyValueCharacter(ch) {
    return isUnicodePropertyNameCharacter(ch) || isDecimalDigit(ch)
  }

  // LoneUnicodePropertyNameOrValue ::
  //   UnicodePropertyValueCharacters
  pp$1.regexp_eatLoneUnicodePropertyNameOrValue = function(state) {
    return this.regexp_eatUnicodePropertyValue(state)
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-CharacterClass
  pp$1.regexp_eatCharacterClass = function(state) {
    if (state.eat(0x5B /* [ */)) {
      var negate = state.eat(0x5E /* ^ */);
      var result = this.regexp_classContents(state);
      if (!state.eat(0x5D /* ] */))
        { state.raise("Unterminated character class"); }
      if (negate && result === CharSetString)
        { state.raise("Negated character class may contain strings"); }
      return true
    }
    return false
  };

  // https://tc39.es/ecma262/#prod-ClassContents
  // https://www.ecma-international.org/ecma-262/8.0/#prod-ClassRanges
  pp$1.regexp_classContents = function(state) {
    if (state.current() === 0x5D /* ] */) { return CharSetOk }
    if (state.switchV) { return this.regexp_classSetExpression(state) }
    this.regexp_nonEmptyClassRanges(state);
    return CharSetOk
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-NonemptyClassRanges
  // https://www.ecma-international.org/ecma-262/8.0/#prod-NonemptyClassRangesNoDash
  pp$1.regexp_nonEmptyClassRanges = function(state) {
    while (this.regexp_eatClassAtom(state)) {
      var left = state.lastIntValue;
      if (state.eat(0x2D /* - */) && this.regexp_eatClassAtom(state)) {
        var right = state.lastIntValue;
        if (state.switchU && (left === -1 || right === -1)) {
          state.raise("Invalid character class");
        }
        if (left !== -1 && right !== -1 && left > right) {
          state.raise("Range out of order in character class");
        }
      }
    }
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-ClassAtom
  // https://www.ecma-international.org/ecma-262/8.0/#prod-ClassAtomNoDash
  pp$1.regexp_eatClassAtom = function(state) {
    var start = state.pos;

    if (state.eat(0x5C /* \ */)) {
      if (this.regexp_eatClassEscape(state)) {
        return true
      }
      if (state.switchU) {
        // Make the same message as V8.
        var ch$1 = state.current();
        if (ch$1 === 0x63 /* c */ || isOctalDigit(ch$1)) {
          state.raise("Invalid class escape");
        }
        state.raise("Invalid escape");
      }
      state.pos = start;
    }

    var ch = state.current();
    if (ch !== 0x5D /* ] */) {
      state.lastIntValue = ch;
      state.advance();
      return true
    }

    return false
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-annexB-ClassEscape
  pp$1.regexp_eatClassEscape = function(state) {
    var start = state.pos;

    if (state.eat(0x62 /* b */)) {
      state.lastIntValue = 0x08; /* <BS> */
      return true
    }

    if (state.switchU && state.eat(0x2D /* - */)) {
      state.lastIntValue = 0x2D; /* - */
      return true
    }

    if (!state.switchU && state.eat(0x63 /* c */)) {
      if (this.regexp_eatClassControlLetter(state)) {
        return true
      }
      state.pos = start;
    }

    return (
      this.regexp_eatCharacterClassEscape(state) ||
      this.regexp_eatCharacterEscape(state)
    )
  };

  // https://tc39.es/ecma262/#prod-ClassSetExpression
  // https://tc39.es/ecma262/#prod-ClassUnion
  // https://tc39.es/ecma262/#prod-ClassIntersection
  // https://tc39.es/ecma262/#prod-ClassSubtraction
  pp$1.regexp_classSetExpression = function(state) {
    var result = CharSetOk, subResult;
    if (this.regexp_eatClassSetRange(state)) ; else if (subResult = this.regexp_eatClassSetOperand(state)) {
      if (subResult === CharSetString) { result = CharSetString; }
      // https://tc39.es/ecma262/#prod-ClassIntersection
      var start = state.pos;
      while (state.eatChars([0x26, 0x26] /* && */)) {
        if (
          state.current() !== 0x26 /* & */ &&
          (subResult = this.regexp_eatClassSetOperand(state))
        ) {
          if (subResult !== CharSetString) { result = CharSetOk; }
          continue
        }
        state.raise("Invalid character in character class");
      }
      if (start !== state.pos) { return result }
      // https://tc39.es/ecma262/#prod-ClassSubtraction
      while (state.eatChars([0x2D, 0x2D] /* -- */)) {
        if (this.regexp_eatClassSetOperand(state)) { continue }
        state.raise("Invalid character in character class");
      }
      if (start !== state.pos) { return result }
    } else {
      state.raise("Invalid character in character class");
    }
    // https://tc39.es/ecma262/#prod-ClassUnion
    for (;;) {
      if (this.regexp_eatClassSetRange(state)) { continue }
      subResult = this.regexp_eatClassSetOperand(state);
      if (!subResult) { return result }
      if (subResult === CharSetString) { result = CharSetString; }
    }
  };

  // https://tc39.es/ecma262/#prod-ClassSetRange
  pp$1.regexp_eatClassSetRange = function(state) {
    var start = state.pos;
    if (this.regexp_eatClassSetCharacter(state)) {
      var left = state.lastIntValue;
      if (state.eat(0x2D /* - */) && this.regexp_eatClassSetCharacter(state)) {
        var right = state.lastIntValue;
        if (left !== -1 && right !== -1 && left > right) {
          state.raise("Range out of order in character class");
        }
        return true
      }
      state.pos = start;
    }
    return false
  };

  // https://tc39.es/ecma262/#prod-ClassSetOperand
  pp$1.regexp_eatClassSetOperand = function(state) {
    if (this.regexp_eatClassSetCharacter(state)) { return CharSetOk }
    return this.regexp_eatClassStringDisjunction(state) || this.regexp_eatNestedClass(state)
  };

  // https://tc39.es/ecma262/#prod-NestedClass
  pp$1.regexp_eatNestedClass = function(state) {
    var start = state.pos;
    if (state.eat(0x5B /* [ */)) {
      var negate = state.eat(0x5E /* ^ */);
      var result = this.regexp_classContents(state);
      if (state.eat(0x5D /* ] */)) {
        if (negate && result === CharSetString) {
          state.raise("Negated character class may contain strings");
        }
        return result
      }
      state.pos = start;
    }
    if (state.eat(0x5C /* \ */)) {
      var result$1 = this.regexp_eatCharacterClassEscape(state);
      if (result$1) {
        return result$1
      }
      state.pos = start;
    }
    return null
  };

  // https://tc39.es/ecma262/#prod-ClassStringDisjunction
  pp$1.regexp_eatClassStringDisjunction = function(state) {
    var start = state.pos;
    if (state.eatChars([0x5C, 0x71] /* \q */)) {
      if (state.eat(0x7B /* { */)) {
        var result = this.regexp_classStringDisjunctionContents(state);
        if (state.eat(0x7D /* } */)) {
          return result
        }
      } else {
        // Make the same message as V8.
        state.raise("Invalid escape");
      }
      state.pos = start;
    }
    return null
  };

  // https://tc39.es/ecma262/#prod-ClassStringDisjunctionContents
  pp$1.regexp_classStringDisjunctionContents = function(state) {
    var result = this.regexp_classString(state);
    while (state.eat(0x7C /* | */)) {
      if (this.regexp_classString(state) === CharSetString) { result = CharSetString; }
    }
    return result
  };

  // https://tc39.es/ecma262/#prod-ClassString
  // https://tc39.es/ecma262/#prod-NonEmptyClassString
  pp$1.regexp_classString = function(state) {
    var count = 0;
    while (this.regexp_eatClassSetCharacter(state)) { count++; }
    return count === 1 ? CharSetOk : CharSetString
  };

  // https://tc39.es/ecma262/#prod-ClassSetCharacter
  pp$1.regexp_eatClassSetCharacter = function(state) {
    var start = state.pos;
    if (state.eat(0x5C /* \ */)) {
      if (
        this.regexp_eatCharacterEscape(state) ||
        this.regexp_eatClassSetReservedPunctuator(state)
      ) {
        return true
      }
      if (state.eat(0x62 /* b */)) {
        state.lastIntValue = 0x08; /* <BS> */
        return true
      }
      state.pos = start;
      return false
    }
    var ch = state.current();
    if (ch < 0 || ch === state.lookahead() && isClassSetReservedDoublePunctuatorCharacter(ch)) { return false }
    if (isClassSetSyntaxCharacter(ch)) { return false }
    state.advance();
    state.lastIntValue = ch;
    return true
  };

  // https://tc39.es/ecma262/#prod-ClassSetReservedDoublePunctuator
  function isClassSetReservedDoublePunctuatorCharacter(ch) {
    return (
      ch === 0x21 /* ! */ ||
      ch >= 0x23 /* # */ && ch <= 0x26 /* & */ ||
      ch >= 0x2A /* * */ && ch <= 0x2C /* , */ ||
      ch === 0x2E /* . */ ||
      ch >= 0x3A /* : */ && ch <= 0x40 /* @ */ ||
      ch === 0x5E /* ^ */ ||
      ch === 0x60 /* ` */ ||
      ch === 0x7E /* ~ */
    )
  }

  // https://tc39.es/ecma262/#prod-ClassSetSyntaxCharacter
  function isClassSetSyntaxCharacter(ch) {
    return (
      ch === 0x28 /* ( */ ||
      ch === 0x29 /* ) */ ||
      ch === 0x2D /* - */ ||
      ch === 0x2F /* / */ ||
      ch >= 0x5B /* [ */ && ch <= 0x5D /* ] */ ||
      ch >= 0x7B /* { */ && ch <= 0x7D /* } */
    )
  }

  // https://tc39.es/ecma262/#prod-ClassSetReservedPunctuator
  pp$1.regexp_eatClassSetReservedPunctuator = function(state) {
    var ch = state.current();
    if (isClassSetReservedPunctuator(ch)) {
      state.lastIntValue = ch;
      state.advance();
      return true
    }
    return false
  };

  // https://tc39.es/ecma262/#prod-ClassSetReservedPunctuator
  function isClassSetReservedPunctuator(ch) {
    return (
      ch === 0x21 /* ! */ ||
      ch === 0x23 /* # */ ||
      ch === 0x25 /* % */ ||
      ch === 0x26 /* & */ ||
      ch === 0x2C /* , */ ||
      ch === 0x2D /* - */ ||
      ch >= 0x3A /* : */ && ch <= 0x3E /* > */ ||
      ch === 0x40 /* @ */ ||
      ch === 0x60 /* ` */ ||
      ch === 0x7E /* ~ */
    )
  }

  // https://www.ecma-international.org/ecma-262/8.0/#prod-annexB-ClassControlLetter
  pp$1.regexp_eatClassControlLetter = function(state) {
    var ch = state.current();
    if (isDecimalDigit(ch) || ch === 0x5F /* _ */) {
      state.lastIntValue = ch % 0x20;
      state.advance();
      return true
    }
    return false
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-HexEscapeSequence
  pp$1.regexp_eatHexEscapeSequence = function(state) {
    var start = state.pos;
    if (state.eat(0x78 /* x */)) {
      if (this.regexp_eatFixedHexDigits(state, 2)) {
        return true
      }
      if (state.switchU) {
        state.raise("Invalid escape");
      }
      state.pos = start;
    }
    return false
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-DecimalDigits
  pp$1.regexp_eatDecimalDigits = function(state) {
    var start = state.pos;
    var ch = 0;
    state.lastIntValue = 0;
    while (isDecimalDigit(ch = state.current())) {
      state.lastIntValue = 10 * state.lastIntValue + (ch - 0x30 /* 0 */);
      state.advance();
    }
    return state.pos !== start
  };
  function isDecimalDigit(ch) {
    return ch >= 0x30 /* 0 */ && ch <= 0x39 /* 9 */
  }

  // https://www.ecma-international.org/ecma-262/8.0/#prod-HexDigits
  pp$1.regexp_eatHexDigits = function(state) {
    var start = state.pos;
    var ch = 0;
    state.lastIntValue = 0;
    while (isHexDigit(ch = state.current())) {
      state.lastIntValue = 16 * state.lastIntValue + hexToInt(ch);
      state.advance();
    }
    return state.pos !== start
  };
  function isHexDigit(ch) {
    return (
      (ch >= 0x30 /* 0 */ && ch <= 0x39 /* 9 */) ||
      (ch >= 0x41 /* A */ && ch <= 0x46 /* F */) ||
      (ch >= 0x61 /* a */ && ch <= 0x66 /* f */)
    )
  }
  function hexToInt(ch) {
    if (ch >= 0x41 /* A */ && ch <= 0x46 /* F */) {
      return 10 + (ch - 0x41 /* A */)
    }
    if (ch >= 0x61 /* a */ && ch <= 0x66 /* f */) {
      return 10 + (ch - 0x61 /* a */)
    }
    return ch - 0x30 /* 0 */
  }

  // https://www.ecma-international.org/ecma-262/8.0/#prod-annexB-LegacyOctalEscapeSequence
  // Allows only 0-377(octal) i.e. 0-255(decimal).
  pp$1.regexp_eatLegacyOctalEscapeSequence = function(state) {
    if (this.regexp_eatOctalDigit(state)) {
      var n1 = state.lastIntValue;
      if (this.regexp_eatOctalDigit(state)) {
        var n2 = state.lastIntValue;
        if (n1 <= 3 && this.regexp_eatOctalDigit(state)) {
          state.lastIntValue = n1 * 64 + n2 * 8 + state.lastIntValue;
        } else {
          state.lastIntValue = n1 * 8 + n2;
        }
      } else {
        state.lastIntValue = n1;
      }
      return true
    }
    return false
  };

  // https://www.ecma-international.org/ecma-262/8.0/#prod-OctalDigit
  pp$1.regexp_eatOctalDigit = function(state) {
    var ch = state.current();
    if (isOctalDigit(ch)) {
      state.lastIntValue = ch - 0x30; /* 0 */
      state.advance();
      return true
    }
    state.lastIntValue = 0;
    return false
  };
  function isOctalDigit(ch) {
    return ch >= 0x30 /* 0 */ && ch <= 0x37 /* 7 */
  }

  // https://www.ecma-international.org/ecma-262/8.0/#prod-Hex4Digits
  // https://www.ecma-international.org/ecma-262/8.0/#prod-HexDigit
  // And HexDigit HexDigit in https://www.ecma-international.org/ecma-262/8.0/#prod-HexEscapeSequence
  pp$1.regexp_eatFixedHexDigits = function(state, length) {
    var start = state.pos;
    state.lastIntValue = 0;
    for (var i = 0; i < length; ++i) {
      var ch = state.current();
      if (!isHexDigit(ch)) {
        state.pos = start;
        return false
      }
      state.lastIntValue = 16 * state.lastIntValue + hexToInt(ch);
      state.advance();
    }
    return true
  };

  // Object type used to represent tokens. Note that normally, tokens
  // simply exist as properties on the parser object. This is only
  // used for the onToken callback and the external tokenizer.

  var Token = function Token(p) {
    this.type = p.type;
    this.value = p.value;
    this.start = p.start;
    this.end = p.end;
    if (p.options.locations)
      { this.loc = new SourceLocation(p, p.startLoc, p.endLoc); }
    if (p.options.ranges)
      { this.range = [p.start, p.end]; }
  };

  // ## Tokenizer

  var pp = Parser.prototype;

  // Move to the next token

  pp.next = function(ignoreEscapeSequenceInKeyword) {
    if (!ignoreEscapeSequenceInKeyword && this.type.keyword && this.containsEsc)
      { this.raiseRecoverable(this.start, "Escape sequence in keyword " + this.type.keyword); }
    if (this.options.onToken)
      { this.options.onToken(new Token(this)); }

    this.lastTokEnd = this.end;
    this.lastTokStart = this.start;
    this.lastTokEndLoc = this.endLoc;
    this.lastTokStartLoc = this.startLoc;
    this.nextToken();
  };

  pp.getToken = function() {
    this.next();
    return new Token(this)
  };

  // If we're in an ES6 environment, make parsers iterable
  if (typeof Symbol !== "undefined")
    { pp[Symbol.iterator] = function() {
      var this$1$1 = this;

      return {
        next: function () {
          var token = this$1$1.getToken();
          return {
            done: token.type === types$1.eof,
            value: token
          }
        }
      }
    }; }

  // Toggle strict mode. Re-reads the next number or string to please
  // pedantic tests (`"use strict"; 010;` should fail).

  // Read a single token, updating the parser object's token-related
  // properties.

  pp.nextToken = function() {
    var curContext = this.curContext();
    if (!curContext || !curContext.preserveSpace) { this.skipSpace(); }

    this.start = this.pos;
    if (this.options.locations) { this.startLoc = this.curPosition(); }
    if (this.pos >= this.input.length) { return this.finishToken(types$1.eof) }

    if (curContext.override) { return curContext.override(this) }
    else { this.readToken(this.fullCharCodeAtPos()); }
  };

  pp.readToken = function(code) {
    // Identifier or keyword. '\uXXXX' sequences are allowed in
    // identifiers, so '\' also dispatches to that.
    if (isIdentifierStart(code, this.options.ecmaVersion >= 6) || code === 92 /* '\' */)
      { return this.readWord() }

    return this.getTokenFromCode(code)
  };

  pp.fullCharCodeAtPos = function() {
    var code = this.input.charCodeAt(this.pos);
    if (code <= 0xd7ff || code >= 0xdc00) { return code }
    var next = this.input.charCodeAt(this.pos + 1);
    return next <= 0xdbff || next >= 0xe000 ? code : (code << 10) + next - 0x35fdc00
  };

  pp.skipBlockComment = function() {
    var startLoc = this.options.onComment && this.curPosition();
    var start = this.pos, end = this.input.indexOf("*/", this.pos += 2);
    if (end === -1) { this.raise(this.pos - 2, "Unterminated comment"); }
    this.pos = end + 2;
    if (this.options.locations) {
      for (var nextBreak = (void 0), pos = start; (nextBreak = nextLineBreak(this.input, pos, this.pos)) > -1;) {
        ++this.curLine;
        pos = this.lineStart = nextBreak;
      }
    }
    if (this.options.onComment)
      { this.options.onComment(true, this.input.slice(start + 2, end), start, this.pos,
                             startLoc, this.curPosition()); }
  };

  pp.skipLineComment = function(startSkip) {
    var start = this.pos;
    var startLoc = this.options.onComment && this.curPosition();
    var ch = this.input.charCodeAt(this.pos += startSkip);
    while (this.pos < this.input.length && !isNewLine(ch)) {
      ch = this.input.charCodeAt(++this.pos);
    }
    if (this.options.onComment)
      { this.options.onComment(false, this.input.slice(start + startSkip, this.pos), start, this.pos,
                             startLoc, this.curPosition()); }
  };

  // Called at the start of the parse and after every token. Skips
  // whitespace and comments, and.

  pp.skipSpace = function() {
    loop: while (this.pos < this.input.length) {
      var ch = this.input.charCodeAt(this.pos);
      switch (ch) {
      case 32: case 160: // ' '
        ++this.pos;
        break
      case 13:
        if (this.input.charCodeAt(this.pos + 1) === 10) {
          ++this.pos;
        }
      case 10: case 8232: case 8233:
        ++this.pos;
        if (this.options.locations) {
          ++this.curLine;
          this.lineStart = this.pos;
        }
        break
      case 47: // '/'
        switch (this.input.charCodeAt(this.pos + 1)) {
        case 42: // '*'
          this.skipBlockComment();
          break
        case 47:
          this.skipLineComment(2);
          break
        default:
          break loop
        }
        break
      default:
        if (ch > 8 && ch < 14 || ch >= 5760 && nonASCIIwhitespace.test(String.fromCharCode(ch))) {
          ++this.pos;
        } else {
          break loop
        }
      }
    }
  };

  // Called at the end of every token. Sets `end`, `val`, and
  // maintains `context` and `exprAllowed`, and skips the space after
  // the token, so that the next one's `start` will point at the
  // right position.

  pp.finishToken = function(type, val) {
    this.end = this.pos;
    if (this.options.locations) { this.endLoc = this.curPosition(); }
    var prevType = this.type;
    this.type = type;
    this.value = val;

    this.updateContext(prevType);
  };

  // ### Token reading

  // This is the function that is called to fetch the next token. It
  // is somewhat obscure, because it works in character codes rather
  // than characters, and because operator parsing has been inlined
  // into it.
  //
  // All in the name of speed.
  //
  pp.readToken_dot = function() {
    var next = this.input.charCodeAt(this.pos + 1);
    if (next >= 48 && next <= 57) { return this.readNumber(true) }
    var next2 = this.input.charCodeAt(this.pos + 2);
    if (this.options.ecmaVersion >= 6 && next === 46 && next2 === 46) { // 46 = dot '.'
      this.pos += 3;
      return this.finishToken(types$1.ellipsis)
    } else {
      ++this.pos;
      return this.finishToken(types$1.dot)
    }
  };

  pp.readToken_slash = function() { // '/'
    var next = this.input.charCodeAt(this.pos + 1);
    if (this.exprAllowed) { ++this.pos; return this.readRegexp() }
    if (next === 61) { return this.finishOp(types$1.assign, 2) }
    return this.finishOp(types$1.slash, 1)
  };

  pp.readToken_mult_modulo_exp = function(code) { // '%*'
    var next = this.input.charCodeAt(this.pos + 1);
    var size = 1;
    var tokentype = code === 42 ? types$1.star : types$1.modulo;

    // exponentiation operator ** and **=
    if (this.options.ecmaVersion >= 7 && code === 42 && next === 42) {
      ++size;
      tokentype = types$1.starstar;
      next = this.input.charCodeAt(this.pos + 2);
    }

    if (next === 61) { return this.finishOp(types$1.assign, size + 1) }
    return this.finishOp(tokentype, size)
  };

  pp.readToken_pipe_amp = function(code) { // '|&'
    var next = this.input.charCodeAt(this.pos + 1);
    if (next === code) {
      if (this.options.ecmaVersion >= 12) {
        var next2 = this.input.charCodeAt(this.pos + 2);
        if (next2 === 61) { return this.finishOp(types$1.assign, 3) }
      }
      return this.finishOp(code === 124 ? types$1.logicalOR : types$1.logicalAND, 2)
    }
    if (next === 61) { return this.finishOp(types$1.assign, 2) }
    return this.finishOp(code === 124 ? types$1.bitwiseOR : types$1.bitwiseAND, 1)
  };

  pp.readToken_caret = function() { // '^'
    var next = this.input.charCodeAt(this.pos + 1);
    if (next === 61) { return this.finishOp(types$1.assign, 2) }
    return this.finishOp(types$1.bitwiseXOR, 1)
  };

  pp.readToken_plus_min = function(code) { // '+-'
    var next = this.input.charCodeAt(this.pos + 1);
    if (next === code) {
      if (next === 45 && !this.inModule && this.input.charCodeAt(this.pos + 2) === 62 &&
          (this.lastTokEnd === 0 || lineBreak.test(this.input.slice(this.lastTokEnd, this.pos)))) {
        // A `-->` line comment
        this.skipLineComment(3);
        this.skipSpace();
        return this.nextToken()
      }
      return this.finishOp(types$1.incDec, 2)
    }
    if (next === 61) { return this.finishOp(types$1.assign, 2) }
    return this.finishOp(types$1.plusMin, 1)
  };

  pp.readToken_lt_gt = function(code) { // '<>'
    var next = this.input.charCodeAt(this.pos + 1);
    var size = 1;
    if (next === code) {
      size = code === 62 && this.input.charCodeAt(this.pos + 2) === 62 ? 3 : 2;
      if (this.input.charCodeAt(this.pos + size) === 61) { return this.finishOp(types$1.assign, size + 1) }
      return this.finishOp(types$1.bitShift, size)
    }
    if (next === 33 && code === 60 && !this.inModule && this.input.charCodeAt(this.pos + 2) === 45 &&
        this.input.charCodeAt(this.pos + 3) === 45) {
      // `<!--`, an XML-style comment that should be interpreted as a line comment
      this.skipLineComment(4);
      this.skipSpace();
      return this.nextToken()
    }
    if (next === 61) { size = 2; }
    return this.finishOp(types$1.relational, size)
  };

  pp.readToken_eq_excl = function(code) { // '=!'
    var next = this.input.charCodeAt(this.pos + 1);
    if (next === 61) { return this.finishOp(types$1.equality, this.input.charCodeAt(this.pos + 2) === 61 ? 3 : 2) }
    if (code === 61 && next === 62 && this.options.ecmaVersion >= 6) { // '=>'
      this.pos += 2;
      return this.finishToken(types$1.arrow)
    }
    return this.finishOp(code === 61 ? types$1.eq : types$1.prefix, 1)
  };

  pp.readToken_question = function() { // '?'
    var ecmaVersion = this.options.ecmaVersion;
    if (ecmaVersion >= 11) {
      var next = this.input.charCodeAt(this.pos + 1);
      if (next === 46) {
        var next2 = this.input.charCodeAt(this.pos + 2);
        if (next2 < 48 || next2 > 57) { return this.finishOp(types$1.questionDot, 2) }
      }
      if (next === 63) {
        if (ecmaVersion >= 12) {
          var next2$1 = this.input.charCodeAt(this.pos + 2);
          if (next2$1 === 61) { return this.finishOp(types$1.assign, 3) }
        }
        return this.finishOp(types$1.coalesce, 2)
      }
    }
    return this.finishOp(types$1.question, 1)
  };

  pp.readToken_numberSign = function() { // '#'
    var ecmaVersion = this.options.ecmaVersion;
    var code = 35; // '#'
    if (ecmaVersion >= 13) {
      ++this.pos;
      code = this.fullCharCodeAtPos();
      if (isIdentifierStart(code, true) || code === 92 /* '\' */) {
        return this.finishToken(types$1.privateId, this.readWord1())
      }
    }

    this.raise(this.pos, "Unexpected character '" + codePointToString(code) + "'");
  };

  pp.getTokenFromCode = function(code) {
    switch (code) {
    // The interpretation of a dot depends on whether it is followed
    // by a digit or another two dots.
    case 46: // '.'
      return this.readToken_dot()

    // Punctuation tokens.
    case 40: ++this.pos; return this.finishToken(types$1.parenL)
    case 41: ++this.pos; return this.finishToken(types$1.parenR)
    case 59: ++this.pos; return this.finishToken(types$1.semi)
    case 44: ++this.pos; return this.finishToken(types$1.comma)
    case 91: ++this.pos; return this.finishToken(types$1.bracketL)
    case 93: ++this.pos; return this.finishToken(types$1.bracketR)
    case 123: ++this.pos; return this.finishToken(types$1.braceL)
    case 125: ++this.pos; return this.finishToken(types$1.braceR)
    case 58: ++this.pos; return this.finishToken(types$1.colon)

    case 96: // '`'
      if (this.options.ecmaVersion < 6) { break }
      ++this.pos;
      return this.finishToken(types$1.backQuote)

    case 48: // '0'
      var next = this.input.charCodeAt(this.pos + 1);
      if (next === 120 || next === 88) { return this.readRadixNumber(16) } // '0x', '0X' - hex number
      if (this.options.ecmaVersion >= 6) {
        if (next === 111 || next === 79) { return this.readRadixNumber(8) } // '0o', '0O' - octal number
        if (next === 98 || next === 66) { return this.readRadixNumber(2) } // '0b', '0B' - binary number
      }

    // Anything else beginning with a digit is an integer, octal
    // number, or float.
    case 49: case 50: case 51: case 52: case 53: case 54: case 55: case 56: case 57: // 1-9
      return this.readNumber(false)

    // Quotes produce strings.
    case 34: case 39: // '"', "'"
      return this.readString(code)

    // Operators are parsed inline in tiny state machines. '=' (61) is
    // often referred to. `finishOp` simply skips the amount of
    // characters it is given as second argument, and returns a token
    // of the type given by its first argument.
    case 47: // '/'
      return this.readToken_slash()

    case 37: case 42: // '%*'
      return this.readToken_mult_modulo_exp(code)

    case 124: case 38: // '|&'
      return this.readToken_pipe_amp(code)

    case 94: // '^'
      return this.readToken_caret()

    case 43: case 45: // '+-'
      return this.readToken_plus_min(code)

    case 60: case 62: // '<>'
      return this.readToken_lt_gt(code)

    case 61: case 33: // '=!'
      return this.readToken_eq_excl(code)

    case 63: // '?'
      return this.readToken_question()

    case 126: // '~'
      return this.finishOp(types$1.prefix, 1)

    case 35: // '#'
      return this.readToken_numberSign()
    }

    this.raise(this.pos, "Unexpected character '" + codePointToString(code) + "'");
  };

  pp.finishOp = function(type, size) {
    var str = this.input.slice(this.pos, this.pos + size);
    this.pos += size;
    return this.finishToken(type, str)
  };

  pp.readRegexp = function() {
    var escaped, inClass, start = this.pos;
    for (;;) {
      if (this.pos >= this.input.length) { this.raise(start, "Unterminated regular expression"); }
      var ch = this.input.charAt(this.pos);
      if (lineBreak.test(ch)) { this.raise(start, "Unterminated regular expression"); }
      if (!escaped) {
        if (ch === "[") { inClass = true; }
        else if (ch === "]" && inClass) { inClass = false; }
        else if (ch === "/" && !inClass) { break }
        escaped = ch === "\\";
      } else { escaped = false; }
      ++this.pos;
    }
    var pattern = this.input.slice(start, this.pos);
    ++this.pos;
    var flagsStart = this.pos;
    var flags = this.readWord1();
    if (this.containsEsc) { this.unexpected(flagsStart); }

    // Validate pattern
    var state = this.regexpState || (this.regexpState = new RegExpValidationState(this));
    state.reset(start, pattern, flags);
    this.validateRegExpFlags(state);
    this.validateRegExpPattern(state);

    // Create Literal#value property value.
    var value = null;
    try {
      value = new RegExp(pattern, flags);
    } catch (e) {
      // ESTree requires null if it failed to instantiate RegExp object.
      // https://github.com/estree/estree/blob/a27003adf4fd7bfad44de9cef372a2eacd527b1c/es5.md#regexpliteral
    }

    return this.finishToken(types$1.regexp, {pattern: pattern, flags: flags, value: value})
  };

  // Read an integer in the given radix. Return null if zero digits
  // were read, the integer value otherwise. When `len` is given, this
  // will return `null` unless the integer has exactly `len` digits.

  pp.readInt = function(radix, len, maybeLegacyOctalNumericLiteral) {
    // `len` is used for character escape sequences. In that case, disallow separators.
    var allowSeparators = this.options.ecmaVersion >= 12 && len === undefined;

    // `maybeLegacyOctalNumericLiteral` is true if it doesn't have prefix (0x,0o,0b)
    // and isn't fraction part nor exponent part. In that case, if the first digit
    // is zero then disallow separators.
    var isLegacyOctalNumericLiteral = maybeLegacyOctalNumericLiteral && this.input.charCodeAt(this.pos) === 48;

    var start = this.pos, total = 0, lastCode = 0;
    for (var i = 0, e = len == null ? Infinity : len; i < e; ++i, ++this.pos) {
      var code = this.input.charCodeAt(this.pos), val = (void 0);

      if (allowSeparators && code === 95) {
        if (isLegacyOctalNumericLiteral) { this.raiseRecoverable(this.pos, "Numeric separator is not allowed in legacy octal numeric literals"); }
        if (lastCode === 95) { this.raiseRecoverable(this.pos, "Numeric separator must be exactly one underscore"); }
        if (i === 0) { this.raiseRecoverable(this.pos, "Numeric separator is not allowed at the first of digits"); }
        lastCode = code;
        continue
      }

      if (code >= 97) { val = code - 97 + 10; } // a
      else if (code >= 65) { val = code - 65 + 10; } // A
      else if (code >= 48 && code <= 57) { val = code - 48; } // 0-9
      else { val = Infinity; }
      if (val >= radix) { break }
      lastCode = code;
      total = total * radix + val;
    }

    if (allowSeparators && lastCode === 95) { this.raiseRecoverable(this.pos - 1, "Numeric separator is not allowed at the last of digits"); }
    if (this.pos === start || len != null && this.pos - start !== len) { return null }

    return total
  };

  function stringToNumber(str, isLegacyOctalNumericLiteral) {
    if (isLegacyOctalNumericLiteral) {
      return parseInt(str, 8)
    }

    // `parseFloat(value)` stops parsing at the first numeric separator then returns a wrong value.
    return parseFloat(str.replace(/_/g, ""))
  }

  function stringToBigInt(str) {
    if (typeof BigInt !== "function") {
      return null
    }

    // `BigInt(value)` throws syntax error if the string contains numeric separators.
    return BigInt(str.replace(/_/g, ""))
  }

  pp.readRadixNumber = function(radix) {
    var start = this.pos;
    this.pos += 2; // 0x
    var val = this.readInt(radix);
    if (val == null) { this.raise(this.start + 2, "Expected number in radix " + radix); }
    if (this.options.ecmaVersion >= 11 && this.input.charCodeAt(this.pos) === 110) {
      val = stringToBigInt(this.input.slice(start, this.pos));
      ++this.pos;
    } else if (isIdentifierStart(this.fullCharCodeAtPos())) { this.raise(this.pos, "Identifier directly after number"); }
    return this.finishToken(types$1.num, val)
  };

  // Read an integer, octal integer, or floating-point number.

  pp.readNumber = function(startsWithDot) {
    var start = this.pos;
    if (!startsWithDot && this.readInt(10, undefined, true) === null) { this.raise(start, "Invalid number"); }
    var octal = this.pos - start >= 2 && this.input.charCodeAt(start) === 48;
    if (octal && this.strict) { this.raise(start, "Invalid number"); }
    var next = this.input.charCodeAt(this.pos);
    if (!octal && !startsWithDot && this.options.ecmaVersion >= 11 && next === 110) {
      var val$1 = stringToBigInt(this.input.slice(start, this.pos));
      ++this.pos;
      if (isIdentifierStart(this.fullCharCodeAtPos())) { this.raise(this.pos, "Identifier directly after number"); }
      return this.finishToken(types$1.num, val$1)
    }
    if (octal && /[89]/.test(this.input.slice(start, this.pos))) { octal = false; }
    if (next === 46 && !octal) { // '.'
      ++this.pos;
      this.readInt(10);
      next = this.input.charCodeAt(this.pos);
    }
    if ((next === 69 || next === 101) && !octal) { // 'eE'
      next = this.input.charCodeAt(++this.pos);
      if (next === 43 || next === 45) { ++this.pos; } // '+-'
      if (this.readInt(10) === null) { this.raise(start, "Invalid number"); }
    }
    if (isIdentifierStart(this.fullCharCodeAtPos())) { this.raise(this.pos, "Identifier directly after number"); }

    var val = stringToNumber(this.input.slice(start, this.pos), octal);
    return this.finishToken(types$1.num, val)
  };

  // Read a string value, interpreting backslash-escapes.

  pp.readCodePoint = function() {
    var ch = this.input.charCodeAt(this.pos), code;

    if (ch === 123) { // '{'
      if (this.options.ecmaVersion < 6) { this.unexpected(); }
      var codePos = ++this.pos;
      code = this.readHexChar(this.input.indexOf("}", this.pos) - this.pos);
      ++this.pos;
      if (code > 0x10FFFF) { this.invalidStringToken(codePos, "Code point out of bounds"); }
    } else {
      code = this.readHexChar(4);
    }
    return code
  };

  pp.readString = function(quote) {
    var out = "", chunkStart = ++this.pos;
    for (;;) {
      if (this.pos >= this.input.length) { this.raise(this.start, "Unterminated string constant"); }
      var ch = this.input.charCodeAt(this.pos);
      if (ch === quote) { break }
      if (ch === 92) { // '\'
        out += this.input.slice(chunkStart, this.pos);
        out += this.readEscapedChar(false);
        chunkStart = this.pos;
      } else if (ch === 0x2028 || ch === 0x2029) {
        if (this.options.ecmaVersion < 10) { this.raise(this.start, "Unterminated string constant"); }
        ++this.pos;
        if (this.options.locations) {
          this.curLine++;
          this.lineStart = this.pos;
        }
      } else {
        if (isNewLine(ch)) { this.raise(this.start, "Unterminated string constant"); }
        ++this.pos;
      }
    }
    out += this.input.slice(chunkStart, this.pos++);
    return this.finishToken(types$1.string, out)
  };

  // Reads template string tokens.

  var INVALID_TEMPLATE_ESCAPE_ERROR = {};

  pp.tryReadTemplateToken = function() {
    this.inTemplateElement = true;
    try {
      this.readTmplToken();
    } catch (err) {
      if (err === INVALID_TEMPLATE_ESCAPE_ERROR) {
        this.readInvalidTemplateToken();
      } else {
        throw err
      }
    }

    this.inTemplateElement = false;
  };

  pp.invalidStringToken = function(position, message) {
    if (this.inTemplateElement && this.options.ecmaVersion >= 9) {
      throw INVALID_TEMPLATE_ESCAPE_ERROR
    } else {
      this.raise(position, message);
    }
  };

  pp.readTmplToken = function() {
    var out = "", chunkStart = this.pos;
    for (;;) {
      if (this.pos >= this.input.length) { this.raise(this.start, "Unterminated template"); }
      var ch = this.input.charCodeAt(this.pos);
      if (ch === 96 || ch === 36 && this.input.charCodeAt(this.pos + 1) === 123) { // '`', '${'
        if (this.pos === this.start && (this.type === types$1.template || this.type === types$1.invalidTemplate)) {
          if (ch === 36) {
            this.pos += 2;
            return this.finishToken(types$1.dollarBraceL)
          } else {
            ++this.pos;
            return this.finishToken(types$1.backQuote)
          }
        }
        out += this.input.slice(chunkStart, this.pos);
        return this.finishToken(types$1.template, out)
      }
      if (ch === 92) { // '\'
        out += this.input.slice(chunkStart, this.pos);
        out += this.readEscapedChar(true);
        chunkStart = this.pos;
      } else if (isNewLine(ch)) {
        out += this.input.slice(chunkStart, this.pos);
        ++this.pos;
        switch (ch) {
        case 13:
          if (this.input.charCodeAt(this.pos) === 10) { ++this.pos; }
        case 10:
          out += "\n";
          break
        default:
          out += String.fromCharCode(ch);
          break
        }
        if (this.options.locations) {
          ++this.curLine;
          this.lineStart = this.pos;
        }
        chunkStart = this.pos;
      } else {
        ++this.pos;
      }
    }
  };

  // Reads a template token to search for the end, without validating any escape sequences
  pp.readInvalidTemplateToken = function() {
    for (; this.pos < this.input.length; this.pos++) {
      switch (this.input[this.pos]) {
      case "\\":
        ++this.pos;
        break

      case "$":
        if (this.input[this.pos + 1] !== "{") { break }
        // fall through
      case "`":
        return this.finishToken(types$1.invalidTemplate, this.input.slice(this.start, this.pos))

      case "\r":
        if (this.input[this.pos + 1] === "\n") { ++this.pos; }
        // fall through
      case "\n": case "\u2028": case "\u2029":
        ++this.curLine;
        this.lineStart = this.pos + 1;
        break
      }
    }
    this.raise(this.start, "Unterminated template");
  };

  // Used to read escaped characters

  pp.readEscapedChar = function(inTemplate) {
    var ch = this.input.charCodeAt(++this.pos);
    ++this.pos;
    switch (ch) {
    case 110: return "\n" // 'n' -> '\n'
    case 114: return "\r" // 'r' -> '\r'
    case 120: return String.fromCharCode(this.readHexChar(2)) // 'x'
    case 117: return codePointToString(this.readCodePoint()) // 'u'
    case 116: return "\t" // 't' -> '\t'
    case 98: return "\b" // 'b' -> '\b'
    case 118: return "\u000b" // 'v' -> '\u000b'
    case 102: return "\f" // 'f' -> '\f'
    case 13: if (this.input.charCodeAt(this.pos) === 10) { ++this.pos; } // '\r\n'
    case 10: // ' \n'
      if (this.options.locations) { this.lineStart = this.pos; ++this.curLine; }
      return ""
    case 56:
    case 57:
      if (this.strict) {
        this.invalidStringToken(
          this.pos - 1,
          "Invalid escape sequence"
        );
      }
      if (inTemplate) {
        var codePos = this.pos - 1;

        this.invalidStringToken(
          codePos,
          "Invalid escape sequence in template string"
        );
      }
    default:
      if (ch >= 48 && ch <= 55) {
        var octalStr = this.input.substr(this.pos - 1, 3).match(/^[0-7]+/)[0];
        var octal = parseInt(octalStr, 8);
        if (octal > 255) {
          octalStr = octalStr.slice(0, -1);
          octal = parseInt(octalStr, 8);
        }
        this.pos += octalStr.length - 1;
        ch = this.input.charCodeAt(this.pos);
        if ((octalStr !== "0" || ch === 56 || ch === 57) && (this.strict || inTemplate)) {
          this.invalidStringToken(
            this.pos - 1 - octalStr.length,
            inTemplate
              ? "Octal literal in template string"
              : "Octal literal in strict mode"
          );
        }
        return String.fromCharCode(octal)
      }
      if (isNewLine(ch)) {
        // Unicode new line characters after \ get removed from output in both
        // template literals and strings
        if (this.options.locations) { this.lineStart = this.pos; ++this.curLine; }
        return ""
      }
      return String.fromCharCode(ch)
    }
  };

  // Used to read character escape sequences ('\x', '\u', '\U').

  pp.readHexChar = function(len) {
    var codePos = this.pos;
    var n = this.readInt(16, len);
    if (n === null) { this.invalidStringToken(codePos, "Bad character escape sequence"); }
    return n
  };

  // Read an identifier, and return it as a string. Sets `this.containsEsc`
  // to whether the word contained a '\u' escape.
  //
  // Incrementally adds only escaped chars, adding other chunks as-is
  // as a micro-optimization.

  pp.readWord1 = function() {
    this.containsEsc = false;
    var word = "", first = true, chunkStart = this.pos;
    var astral = this.options.ecmaVersion >= 6;
    while (this.pos < this.input.length) {
      var ch = this.fullCharCodeAtPos();
      if (isIdentifierChar(ch, astral)) {
        this.pos += ch <= 0xffff ? 1 : 2;
      } else if (ch === 92) { // "\"
        this.containsEsc = true;
        word += this.input.slice(chunkStart, this.pos);
        var escStart = this.pos;
        if (this.input.charCodeAt(++this.pos) !== 117) // "u"
          { this.invalidStringToken(this.pos, "Expecting Unicode escape sequence \\uXXXX"); }
        ++this.pos;
        var esc = this.readCodePoint();
        if (!(first ? isIdentifierStart : isIdentifierChar)(esc, astral))
          { this.invalidStringToken(escStart, "Invalid Unicode escape"); }
        word += codePointToString(esc);
        chunkStart = this.pos;
      } else {
        break
      }
      first = false;
    }
    return word + this.input.slice(chunkStart, this.pos)
  };

  // Read an identifier or keyword token. Will check for reserved
  // words when necessary.

  pp.readWord = function() {
    var word = this.readWord1();
    var type = types$1.name;
    if (this.keywords.test(word)) {
      type = keywords[word];
    }
    return this.finishToken(type, word)
  };

  // Acorn is a tiny, fast JavaScript parser written in JavaScript.
  //
  // Acorn was written by Marijn Haverbeke, Ingvar Stepanyan, and
  // various contributors and released under an MIT license.
  //
  // Git repositories for Acorn are available at
  //
  //     http://marijnhaverbeke.nl/git/acorn
  //     https://github.com/acornjs/acorn.git
  //
  // Please use the [github bug tracker][ghbt] to report issues.
  //
  // [ghbt]: https://github.com/acornjs/acorn/issues
  //
  // [walk]: util/walk.js


  var version = "8.12.1";

  Parser.acorn = {
    Parser: Parser,
    version: version,
    defaultOptions: defaultOptions,
    Position: Position,
    SourceLocation: SourceLocation,
    getLineInfo: getLineInfo,
    Node: Node,
    TokenType: TokenType,
    tokTypes: types$1,
    keywordTypes: keywords,
    TokContext: TokContext,
    tokContexts: types,
    isIdentifierChar: isIdentifierChar,
    isIdentifierStart: isIdentifierStart,
    Token: Token,
    isNewLine: isNewLine,
    lineBreak: lineBreak,
    lineBreakG: lineBreakG,
    nonASCIIwhitespace: nonASCIIwhitespace
  };

  // The main exported interface (under `self.acorn` when in the
  // browser) is a `parse` function that takes a code string and returns
  // an abstract syntax tree as specified by the [ESTree spec][estree].
  //
  // [estree]: https://github.com/estree/estree

  function parse$3(input, options) {
    return Parser.parse(input, options)
  }

  var constants = {
    undefined: 'void(0)',
    Infinity:  'Number.POSITIVE_INFINITY',
    NaN:       'Number.NaN',
    E:         'Math.E',
    LN2:       'Math.LN2',
    LN10:      'Math.LN10',
    LOG2E:     'Math.LOG2E',
    LOG10E:    'Math.LOG10E',
    PI:        'Math.PI',
    SQRT1_2:   'Math.SQRT1_2',
    SQRT2:     'Math.SQRT2'
  };

  function isNumber(value) {
    return typeof value === 'number';
  }

  const PARSER_OPT = { ecmaVersion: 11 };
  const DEFAULT_PARAM_ID = '$';
  const DEFAULT_TUPLE_ID = 'd';
  const DEFAULT_TUPLE_ID1 = 'd1';
  const DEFAULT_TUPLE_ID2 = 'd2';

  const NO = msg => (node, ctx) => ctx.error(node, msg + ' not allowed');
  const ERROR_AGGREGATE = NO('Aggregate function');
  const ERROR_WINDOW = NO('Window function');
  const ERROR_ARGUMENT = 'Invalid argument';
  const ERROR_COLUMN = 'Invalid column reference';
  const ERROR_AGGRONLY = ERROR_COLUMN + ' (must be input to an aggregate function)';
  const ERROR_FUNCTION = 'Invalid function call';
  const ERROR_MEMBER = 'Invalid member expression';
  const ERROR_OP_PARAMETER = 'Invalid operator parameter';
  const ERROR_PARAM = 'Invalid param reference';
  const ERROR_VARIABLE = 'Invalid variable reference';
  const ERROR_VARIABLE_OP = 'Variable not accessible in operator call';
  const ERROR_DECLARATION = 'Unsupported variable declaration';
  const ERROR_DESTRUCTURE = 'Unsupported destructuring pattern';
  const ERROR_CLOSURE = 'Table expressions do not support closures';
  const ERROR_ESCAPE = 'Use aq.escape(fn) to use a function as-is (including closures)';
  const ERROR_USE_PARAMS = 'use table.params({ name: value }) to define dynamic parameters';
  const ERROR_ADD_FUNCTION = 'use aq.addFunction(name, fn) to add new op functions';
  const ERROR_VARIABLE_NOTE = `\nNote: ${ERROR_CLOSURE}. ${ERROR_ESCAPE}, or ${ERROR_USE_PARAMS}.`;
  const ERROR_FUNCTION_NOTE = `\nNote: ${ERROR_CLOSURE}. ${ERROR_ESCAPE}, or ${ERROR_ADD_FUNCTION}.`;
  const ERROR_ROW_OBJECT = `The ${ROW_OBJECT} method is not valid in multi-table expressions.`;

  function parseExpression(ctx, spec) {
    const ast = parseAST(spec);
    let node = ctx.root = ast;
    ctx.spec = spec;
    ctx.tuple = null;
    ctx.tuple1 = null;
    ctx.tuple2 = null;
    ctx.$param = null;
    ctx.$op = 0;
    ctx.scope = new Set();
    ctx.paramsRef = new Map();
    ctx.columnRef = new Map();

    // parse input column parameters
    // if no function def, assume default tuple identifiers
    if (isFunctionExpression(node)) {
      parseFunction(node, ctx);
      node = node.body;
    } else if (ctx.join) {
      ctx.scope.add(ctx.tuple1 = DEFAULT_TUPLE_ID1);
      ctx.scope.add(ctx.tuple2 = DEFAULT_TUPLE_ID2);
      ctx.scope.add(ctx.$param = DEFAULT_PARAM_ID);
    } else {
      ctx.scope.add(ctx.tuple = DEFAULT_TUPLE_ID);
      ctx.scope.add(ctx.$param = DEFAULT_PARAM_ID);
    }

    // rewrite column references & function calls
    walk(node, ctx, visitors);

    return ctx.root;
  }

  function parseAST(expr) {
    try {
      const code = expr.field ? fieldRef(expr)
        : isArray$2(expr) ? toString$1(expr)
        : expr;
      // @ts-ignore
      return parse$3(`expr=(${code})`, PARSER_OPT).body[0].expression.right;
    } catch (err) { // eslint-disable-line no-unused-vars
      error(`Expression parse error: ${expr+''}`);
    }
  }

  function fieldRef(expr) {
    const col = JSON.stringify(expr+'');
    return !(expr.table || 0) ? `d=>d[${col}]` : `(a,b)=>b[${col}]`;
  }

  const visitors = {
    FunctionDeclaration: NO('Function definitions'),
    ForStatement: NO('For loops'),
    ForOfStatement: NO('For-of loops'),
    ForInStatement: NO('For-in loops'),
    WhileStatement: NO('While loops'),
    DoWhileStatement: NO('Do-while loops'),
    AwaitExpression: NO('Await expressions'),
    ArrowFunctionExpression: NO('Function definitions'),
    AssignmentExpression: NO('Assignments'),
    FunctionExpression: NO('Function definitions'),
    NewExpression: NO('Use of "new"'),
    UpdateExpression: NO('Update expressions'),

    VariableDeclarator(node, ctx) {
      handleDeclaration(node.id, ctx);
    },
    Identifier(node, ctx, parent) {
      if (handleIdentifier(node, ctx, parent) && !ctx.scope.has(node.name)) {
        // handle identifier passed responsibility here
        // raise error if identifier not defined in scope
        ctx.error(node, ERROR_VARIABLE, ERROR_VARIABLE_NOTE);
      }
    },
    CallExpression(node, ctx) {
      const name = functionName(node.callee);
      const def = getAggregate(name) || getWindow(name);

      // parse operator and rewrite invocation
      if (def) {
        if ((ctx.join || ctx.aggregate === false) && hasAggregate(name)) {
          ERROR_AGGREGATE(node, ctx);
        }
        if ((ctx.join || ctx.window === false) && hasWindow(name)) {
          ERROR_WINDOW(node, ctx);
        }

        ctx.$op = 1;
        if (ctx.ast) {
          updateFunctionNode(node, name, ctx);
          node.arguments.forEach(arg => walk(arg, ctx, opVisitors));
        } else {
          const op = ctx.op(parseOperator(ctx, def, name, node.arguments));
          Object.assign(node, { type: Op, name: op.id });
        }
        ctx.$op = 0;
        return false;
      } else if (hasFunction(name)) {
        updateFunctionNode(node, name, ctx);
      } else {
        ctx.error(node, ERROR_FUNCTION, ERROR_FUNCTION_NOTE);
      }
    },
    MemberExpression(node, ctx, parent) {
      const { object, property } = node;

      // bail if left head is not an identifier
      // in this case we will recurse and handle it later
      if (!is(Identifier, object)) return;
      const { name } = object;

      // allow use of Math prefix to access constant values
      if (isMath(node) && is(Identifier, property)
          && has(constants, property.name)) {
        updateConstantNode(node, property.name);
        return;
      }

      const index = name === ctx.tuple ? 0
        : name === ctx.tuple1 ? 1
        : name === ctx.tuple2 ? 2
        : -1;

      if (index >= 0) {
        // replace member expression with column ref
        return spliceMember(node, index, ctx, checkColumn, parent);
      } else if (name === ctx.$param) {
        // replace member expression with param ref
        return spliceMember(node, index, ctx, checkParam);
      } else if (ctx.paramsRef.has(name)) {
        updateParameterNode(node, ctx.paramsRef.get(name));
      } else if (ctx.columnRef.has(name)) {
        updateColumnNode(object, name, ctx, node);
      } else if (has(ctx.params, name)) {
        updateParameterNode(object, name);
      }
    }
  };

  function spliceMember(node, index, ctx, check, parent) {
    const { property, computed } = node;
    let name;

    if (!computed) {
      name = property.name;
    } else if (is(Literal, property)) {
      name = property.value;
    } else try {
      // visit subtree to ensure nodes are rewritten as needed
      // then compile the code to compute the property name
      walk(property, ctx, visitors, node);
      name = ctx.param(property);
    } catch (e) { // eslint-disable-line no-unused-vars
      ctx.error(node, ERROR_MEMBER);
    }

    check(node, name, index, ctx, parent);
    return false;
  }

  const opVisitors = {
    ...visitors,
    VariableDeclarator: NO('Variable declaration in operator call'),
    Identifier(node, ctx, parent) {
      if (handleIdentifier(node, ctx, parent)) {
        ctx.error(node, ERROR_VARIABLE_OP);
      }
    },
    CallExpression(node, ctx) {
      const name = functionName(node.callee);

      // rewrite if built-in function
      if (hasFunction(name)) {
        updateFunctionNode(node, name, ctx);
      } else {
        ctx.error(node, ERROR_FUNCTION, ERROR_FUNCTION_NOTE);
      }
    }
  };

  function parseFunction(node, ctx) {
    if (node.generator) NO('Generator functions')(node, ctx);
    if (node.async) NO('Async functions')(node, ctx);

    const { params } = node;
    const len = params.length;
    const setc = index => (name, key) => ctx.columnRef.set(name, [key, index]);
    const setp = (name, key) => ctx.paramsRef.set(name, key);

    if (!len) ; else if (ctx.join) {
      parseRef(ctx, params[0], 'tuple1', setc(1));
      if (len > 1) parseRef(ctx, params[1], 'tuple2', setc(2));
      if (len > 2) parseRef(ctx, params[2], '$param', setp);
    } else {
      parseRef(ctx, params[0], 'tuple', setc(0));
      if (len > 1) parseRef(ctx, params[1], '$param', setp);
    }

    ctx.root = node.body;
  }

  function parseRef(ctx, node, refName, alias) {
    if (is(Identifier, node)) {
      ctx.scope.add(node.name);
      ctx[refName] = node.name;
    } else if (is(ObjectPattern, node)) {
      node.properties.forEach(p => {
        const key = is(Identifier, p.key) ? p.key.name
          : is(Literal, p.key) ? p.key.value
          : ctx.error(p, ERROR_ARGUMENT);
        if (!is(Identifier, p.value)) {
          ctx.error(p.value, ERROR_DESTRUCTURE);
        }
        alias(p.value.name, key);
      });
    }
  }

  function parseOperator(ctx, def, name, args) {
    const fields = [];
    const params = [];
    const idxFields = def.param[0] || 0;
    const idxParams = idxFields + (def.param[1] || 0);

    args.forEach((arg, index) => {
      if (index < idxFields) {
        walk(arg, ctx, opVisitors);
        fields.push(ctx.field(arg));
      } else if (index < idxParams) {
        walk(arg, ctx, opVisitors);
        params.push(ctx.param(arg));
      } else {
        ctx.error(arg, ERROR_OP_PARAMETER);
      }
    });

    return { name, fields, params, ...(ctx.spec.window || {}) };
  }

  function functionName(node) {
    return is(Identifier, node) ? node.name
      : !is(MemberExpression, node) ? null
      : isMath(node) ? rewriteMath(node.property.name)
      : node.property.name;
  }

  function isMath(node) {
    return is(Identifier, node.object) && node.object.name === 'Math';
  }

  function rewriteMath(name) {
    return name === 'max' ? 'greatest'
      : name === 'min' ? 'least'
      : name;
  }

  function handleIdentifier(node, ctx, parent) {
    const { name } = node;

    if (is(MemberExpression, parent) && parent.property === node) ; else if (is(Property, parent) && parent.key === node) ; else if (ctx.paramsRef.has(name)) {
      updateParameterNode(node, ctx.paramsRef.get(name));
    } else if (ctx.columnRef.has(name)) {
      updateColumnNode(node, name, ctx, parent);
    } else if (has(ctx.params, name)) {
      updateParameterNode(node, name);
    } else if (has(constants, name)) {
      updateConstantNode(node, name);
    } else {
      return true;
    }
  }

  function checkColumn(node, name, index, ctx, parent) {
    // check column existence if we have a backing table
    const table = index === 0 ? ctx.table
      : index > 0 ? ctx.join[index - 1]
      : null;
    const col = table && table.column(name);
    if (table && !col) {
      ctx.error(node, ERROR_COLUMN);
    }

    // check if column reference is valid in current context
    if (ctx.aggronly && !ctx.$op) {
      ctx.error(node, ERROR_AGGRONLY);
    }

    // rewrite ast node as a column access
    rewrite(node, name, index, col, parent);
  }

  function updateColumnNode(node, key, ctx, parent) {
    const [name, index] = ctx.columnRef.get(key);
    checkColumn(node, name, index, ctx, parent);
  }

  function checkParam(node, name, index, ctx) {
    if (ctx.params && !has(ctx.params, name)) {
      ctx.error(node, ERROR_PARAM);
    }
    updateParameterNode(node, name);
  }

  function updateParameterNode(node, name) {
    node.type = Parameter;
    node.name = name;
  }

  function updateConstantNode(node, name) {
    node.type = Constant;
    node.name = name;
    node.raw = constants[name];
  }

  function updateFunctionNode(node, name, ctx) {
    if (name === ROW_OBJECT) {
      const t = ctx.table;
      if (!t) ctx.error(node, ERROR_ROW_OBJECT);
      rowObjectExpression(node, t,
        node.arguments.length
          ? node.arguments.map(node => {
              const col = ctx.param(node);
              const name = isNumber(col) ? t.columnName(col) : col;
              if (!t.column(name)) ctx.error(node, ERROR_COLUMN);
              return name;
            })
          : t.columnNames()
      );
    } else {
      node.callee = { type: Function$1, name };
    }
  }

  function handleDeclaration(node, ctx) {
    if (is(Identifier, node)) {
      ctx.scope.add(node.name);
    } else if (is(ArrayPattern, node)) {
      node.elements.forEach(elm => handleDeclaration(elm, ctx));
    } else if (is(ObjectPattern, node)) {
      node.properties.forEach(prop => handleDeclaration(prop.value, ctx));
    } else {
      ctx.error(node.id, ERROR_DECLARATION);
    }
  }

  const ANNOTATE = { [Column$1]: 1, [Op]: 1 };

  function parse$2(input, opt = {}) {
    const generate = opt.generate || codegen;
    const compiler = opt.compiler || compile$1;
    const params = getParams(opt);
    const fields = {};
    const opcall = {};
    const names = [];
    const exprs = [];
    let fieldId = 0;
    let opId = -1;

    const compileExpr = opt.join ? compiler.join
      : opt.index == 1 ? compiler.expr2
      : compiler.expr;

    // parser context
    const ctx = {
      op(op) {
        const key = opKey(op);
        return opcall[key] || (op.id = ++opId, opcall[key] = op);
      },
      field(node) {
        const code = generate(node);
        return fields[code] || (fields[code] = ++fieldId);
      },
      param(node) {
        return is(Literal, node)
          ? node.value
          : compiler.param(generate(node), params);
      },
      value(name, node) {
        names.push(name);
        const e = node.escape || (opt.ast
          ? clean(node)
          : compileExpr(generate(node), params));
        exprs.push(e);
        // annotate expression if it is a direct column or op access
        // this permits downstream optimizations
        if (ANNOTATE[node.type] && e !== node && isObject(e)) {
          e.field = node.name;
        }
      },
      error(node, msg, note = '') {
        // both expresions and fields are parsed
        // with added code prefixes of length 6!
        const i = node.start - 6;
        const j = node.end - 6;
        const snippet = String(ctx.spec).slice(i, j);
        error(`${msg}: "${snippet}"${note}`);
      }
    };

    // copy all options to context, potentially overwriting methods
    Object.assign(ctx, opt, { params });

    // parse each expression
    for (const [name, value] of entries(input)) {
      ctx.value(
        name + '',
        value.escape
          ? parseEscape(ctx, value, params)
          : parseExpression(ctx, value)
      );
    }

    // return expression asts if requested
    if (opt.ast) {
      return { names, exprs };
    }

    // compile input field accessors
    const f = [];
    for (const key in fields) {
      f[fields[key]] = compiler.expr(key, params);
    }

    // resolve input fields to operations
    const ops = Object.values(opcall);
    ops.forEach(op => op.fields = op.fields.map(id => f[id]));

    return { names, exprs, ops };
  }

  function opKey(op) {
    let key = `${op.name}(${op.fields.concat(op.params).join(',')})`;
    if (op.frame) {
      const frame = op.frame.map(v => Number.isFinite(v) ? Math.abs(v) : -1);
      key += `[${frame},${!!op.peers}]`;
    }
    return key;
  }

  function getParams(opt) {
    return (opt.table ? getTableParams(opt.table)
      : opt.join ? {
          ...getTableParams(opt.join[1]),
          ...getTableParams(opt.join[0])
        }
      : {}) || {};
  }

  function getTableParams(table) {
    return table && isFunction(table.params) ? table.params() : {};
  }

  /**
   * Annotate an expression in an object wrapper.
   * @param {string|Function|object} expr An expression to annotate.
   * @param {object} properties The properties to annotate with.
   * @return {object} A new wrapped expression object.
   */
  function wrap$1(expr, properties) {
    return expr && expr.expr
      ? new Wrapper({ ...expr, ...properties })
      : new Wrapper(properties, expr);
  }

  class Wrapper {
    constructor(properties, expr) {
      this.expr = expr;
      Object.assign(this, properties);
    }
    toString() {
      return String(this.expr);
    }
    toObject() {
      return {
        ...this,
        expr: this.toString(),
        ...(isFunction(this.expr) ? { func: true } : {})
      };
    }
  }

  /**
   * Annotate an expression to indicate it is a string field reference.
   * @param {string|object} expr The column name, or an existing wrapped
   *  expression for a column name.
   * @param {string} [name] The column name to use. If provided, will
   *  overwrite the input expression value.
   * @param {number} [table=0] The table index of the field, in case of
   *  expressions over multiple tables.
   * @return A wrapped expression for a named column.
   * @example field('colA')
   */
  function field$1(expr, name, table = 0) {
    const props = table ? { field: true, table } : { field: true };
    return wrap$1(
      expr,
      name ? { expr: name, ...props } : props
    );
  }

  function assign$1(map, pairs) {
    for (const [key, value] of entries(pairs)) {
      map.set(key, value);
    }
    return map;
  }

  function escapeRegExp(str) {
    return str.replace(/[.*+\-?^${}()|[\]\\]/g, '\\$&');
  }

  function resolve(table, sel, map = new Map()) {
    sel = isNumber(sel) ? table.columnName(sel) : sel;

    if (isString(sel)) {
      map.set(sel, sel);
    } else if (isArray$2(sel)) {
      sel.forEach(r => resolve(table, r, map));
    } else if (isFunction(sel)) {
      resolve(table, sel(table), map);
    } else if (isObject(sel)) {
      assign$1(map, sel);
    } else {
      error(`Invalid column selection: ${toString$1(sel)}`);
    }

    return map;
  }

  function decorate(value, toObject) {
    value.toObject = toObject;
    return value;
  }

  function toObject(value) {
    return isArray$2(value) ? value.map(toObject)
      : value && value.toObject ? value.toObject()
      : value;
  }

  /**
   * Proxy type for SelectHelper function.
   * @typedef {import('../table/types.js').SelectHelper} SelectHelper
   */

  /**
   * Select all columns in a table.
   * Returns a function-valued selection compatible with {@link Table#select}.
   * @return {SelectHelper} Selection function compatible with select().
   */
  function all() {
    return decorate(
      table => table.columnNames(),
      () => ({ all: [] })
    );
  }

  /**
   * Negate a column selection, selecting all other columns in a table.
   * Returns a function-valued selection compatible with {@link Table#select}.
   * @param {...any} selection The selection to negate. May be a column name,
   *  column index, array of either, or a selection function (e.g., from range).
   * @return {SelectHelper} Selection function compatible with select().
   */
  function not(...selection) {
    selection = selection.flat();
    return decorate(
      table => {
        const drop = resolve(table, selection);
        return table.columnNames(name => !drop.has(name));
      },
      () => ({ not: toObject(selection) })
    );
  }

  /**
   * Select a contiguous range of columns.
   * @param {string|number} start The name/index of the first selected column.
   * @param {string|number} end The name/index of the last selected column.
   * @return {SelectHelper} Selection function compatible with select().
   */
  function range(start, end) {
    return decorate(
      table => {
        let i = isNumber(start) ? start : table.columnIndex(start);
        let j = isNumber(end) ? end : table.columnIndex(end);
        if (j < i) { const t = j; j = i; i = t; }
        return table.columnNames().slice(i, j + 1);
      },
      () => ({ range: [start, end] })
    );
  }

  /**
   * Select all columns whose names match a pattern.
   * @param {string|RegExp} pattern A string or regular expression pattern to match.
   * @return {SelectHelper} Selection function compatible with select().
   */
  function matches(pattern) {
    if (isString(pattern)) pattern = RegExp(escapeRegExp(pattern));
    return decorate(
      // @ts-ignore
      table => table.columnNames(name => pattern.test(name)),
      // @ts-ignore
      () => ({ matches: [pattern.source, pattern.flags] })
    );
  }

  /**
   * Select all columns whose names start with a string.
   * @param {string} string The string to match at the start of the column name.
   * @return {SelectHelper} Selection function compatible with select().
   */
  function startswith(string) {
    return matches(RegExp('^' + escapeRegExp(string)));
  }

  /**
   * Select all columns whose names end with a string.
   * @param {string} string The string to match at the end of the column name.
   * @return {SelectHelper} Selection function compatible with select().
   */
  function endswith(string) {
    return matches(RegExp(escapeRegExp(string) + '$'));
  }

  function parse$1(name, table, params, options = { window: false }) {
    const exprs = new Map();

    const marshal = param => {
      param = isNumber(param) ? table.columnName(param) : param;
      isString(param) ? exprs.set(param, field$1(param))
        : isFunction(param) ? resolve(table, param).forEach(marshal)
        : isObject(param) ? assign$1(exprs, param)
        : error(`Invalid ${name} value: ${param+''}`);
    };

    toArray(params).forEach(marshal);

    if (options.preparse) {
      options.preparse(exprs);
    }

    return parse$2(exprs, { table, ...options });
  }

  function groupby(table, ...values) {
    return _groupby(table, parse$1('groupby', table, values.flat()));
  }

  function _groupby(table, exprs) {
    return table.create({
      groups: createGroups(table, exprs)
    });
  }

  function createGroups(table, { names = [], exprs = [], ops = [] }) {
    const n = names.length;
    if (n === 0) return null;

    // check for optimized path when grouping by a single field
    // use pre-calculated groups if available
    if (n === 1 && !table.isFiltered() && exprs[0].field) {
      const col = table.column(exprs[0].field);
      if (col.groups) return col.groups(names);
    }

    let get = aggregateGet(table, ops, exprs);
    const getKey = keyFunction(get);
    const nrows = table.totalRows();
    const keys = new Uint32Array(nrows);
    const index = {};
    const rows = [];

    // inline table scan for performance
    const data = table.data();
    const bits = table.mask();
    if (bits) {
      for (let i = bits.next(0); i >= 0; i = bits.next(i + 1)) {
        const key = getKey(i, data) + '';
        keys[i] = (index[key] ??= rows.push(i) - 1);
      }
    } else {
      for (let i = 0; i < nrows; ++i) {
        const key = getKey(i, data) + '';
        keys[i] = (index[key] ??= rows.push(i) - 1);
      }
    }

    if (!ops.length) {
      // capture data in closure, so no interaction with select
      get = get.map(f => row => f(row, data));
    }

    return { keys, get, names, rows, size: rows.length };
  }

  /**
   * Return a new column set instance.
   * @param {import('./Table.js').Table} [table] A base table whose columns
   *  should populate the returned set. If unspecified, create an empty set.
   * @return {ColumnSet} The column set.
   */
  function columnSet(table) {
    return table
      ? new ColumnSet({ ...table.data() }, table.columnNames())
      : new ColumnSet();
  }

  /** An editable collection of named columns. */
  class ColumnSet {
    /**
     * Create a new column set instance.
     * @param {import('./types.js').ColumnData} [data] Initial column data.
     * @param {string[]} [names] Initial column names.
     */
    constructor(data, names) {
      this.data = data || {};
      this.names = names || [];
    }

    /**
     * Add a new column to this set and return the column values.
     * @template {import('./types.js').ColumnType} T
     * @param {string} name The column name.
     * @param {T} values The column values.
     * @return {T} The provided column values.
     */
    add(name, values) {
      if (!this.has(name)) this.names.push(name + '');
      return this.data[name] = values;
    }

    /**
     * Test if this column set has a columns with the given name.
     * @param {string} name A column name
     * @return {boolean} True if this set contains a column with the given name,
     *  false otherwise.
     */
    has(name) {
      return has(this.data, name);
    }

    /**
     * Add a groupby specification to this column set.
     * @param {import('./types.js').GroupBySpec} groups A groupby specification.
     * @return {this} This column set.
     */
    groupby(groups) {
      this.groups = groups;
      return this;
    }

    /**
     * Create a new table with the contents of this column set, using the same
     * type as a given prototype table. The new table does not inherit the
     * filter, groupby, or orderby state of the prototype.
     * @template {import('./Table.js').Table} T
     * @param {T} proto A prototype table
     * @return {T} The new table.
     */
    new(proto) {
      const { data, names, groups = null } = this;
      return proto.create({ data, names, groups, filter: null, order: null });
    }

    /**
     * Create a derived table with the contents of this column set, using the same
     * type as a given prototype table. The new table will inherit the filter,
     * groupby, and orderby state of the prototype.
     * @template {import('./Table.js').Table} T
     * @param {T} proto A prototype table
     * @return {T} The new table.
     */
    derive(proto) {
      return proto.create(this);
    }
  }

  function rollup(table, values) {
    return _rollup(table, parse$2(values, { table, aggronly: true, window: false }));
  }

  function _rollup(table, { names, exprs, ops = [] }) {
    // output data
    const cols = columnSet();
    const groups = table.groups();

    // write groupby fields to output
    if (groups) groupOutput(cols, groups);

    // compute and write aggregate output
    output$2(names, exprs, groups, aggregate(table, ops), cols);

    // return output table
    return cols.new(table);
  }

  function output$2(names, exprs, groups, result = [], cols) {
    if (!exprs.length) return;
    const size = groups ? groups.size : 1;
    const op = (id, row) => result[id][row];
    const n = names.length;

    for (let i = 0; i < n; ++i) {
      const get = exprs[i];
      if (get.field != null) {
        // if expression is op only, use aggregates directly
        cols.add(names[i], result[get.field]);
      } else if (size > 1) {
        // if multiple groups, evaluate expression for each
        const col = cols.add(names[i], Array(size));
        for (let j = 0; j < size; ++j) {
          col[j] = get(j, null, op);
        }
      } else {
        // if only one group, no need to loop
        cols.add(names[i], [ get(0, null, op) ]);
      }
    }
  }

  function select(table, ...columns) {
    return _select(table, resolve(table, columns.flat()));
  }

  function _select(table, columns) {
    const cols = columnSet();

    columns.forEach((value, curr) => {
      const next = isString(value) ? value : curr;
      if (next) {
        const col = table.column(curr) || error(`Unrecognized column: ${curr}`);
        cols.add(next, col);
      }
    });

    return cols.derive(table);
  }

  /**
   * Regroup table rows in response to a BitSet filter.
   * @param {import('./types.js').GroupBySpec} groups The current groupby specification.
   * @param {import('./BitSet.js').BitSet} filter The filter to apply.
   */
  function regroup(groups, filter) {
    if (!groups || !filter) return groups;

    // check for presence of rows for each group
    const { keys, rows, size } = groups;
    const map = new Uint32Array(size);
    filter.scan(row => map[keys[row]] = 1);

    // check sum, exit early if all groups occur
    const sum = map.reduce((sum, val) => sum + val, 0);
    if (sum === size) return groups;

    // create group index map, filter exemplar rows
    const _rows = Array(sum);
    let _size = 0;
    for (let i = 0; i < size; ++i) {
      if (map[i]) _rows[map[i] = _size++] = rows[i];
    }

    // re-index the group keys
    const _keys = new Uint32Array(keys.length);
    filter.scan(row => _keys[row] = map[keys[row]]);

    return { ...groups, keys: _keys, rows: _rows, size: _size };
  }

  /**
   * Regroup table rows in response to a re-indexing.
   * This operation may or may not involve filtering of rows.
   * @param {import('./types.js').GroupBySpec} groups
   *  The current groupby specification.
   * @param {Function} scan Function to scan new row indices.
   * @param {boolean} filter Flag indicating if filtering may occur.
   * @param {number} nrows The number of rows in the new table.
   */
  function reindex(groups, scan, filter, nrows) {
    const { keys, rows, size } = groups;
    let _rows = rows;
    let _size = size;
    let map = null;

    if (filter) {
      // check for presence of rows for each group
      map = new Int32Array(size);
      scan(row => map[keys[row]] = 1);

      // check sum, regroup if not all groups occur
      const sum = map.reduce((sum, val) => sum + val, 0);
      if (sum !== size) {
        // create group index map, filter exemplar rows
        _rows = Array(sum);
        _size = 0;
        for (let i = 0; i < size; ++i) {
          if (map[i]) _rows[map[i] = _size++] = rows[i];
        }
      }
    }

    // re-index the group keys
    let r = -1;
    const _keys = new Uint32Array(nrows);
    const fn = _size !== size
      ? row => _keys[++r] = map[keys[row]]
      : row => _keys[++r] = keys[row];
    scan(fn);

    return { ...groups, keys: _keys, rows: _rows, size: _size };
  }

  function nest(table, idx, obj, type) {
    const agg = type === 'map' || type === true ? map_agg
      : type === 'entries' ? entries_agg
      : type === 'object' ? object_agg
      : error('groups option must be "map", "entries", or "object".');

    const { names } = table.groups();
    const col = uniqueName(table.columnNames(), '_');

    // create table with one column of row objects
    // then aggregate into per-group arrays
    let t = select(table, {}).reify(idx).create({ data: { [col]: obj } });
    t = rollup(t, { [col]: array_agg(col) });

    // create nested structures for each level of grouping
    for (let i = names.length; --i >= 0;) {
      t = rollup(
          groupby(t, names.slice(0, i)),
          // @ts-ignore
          { [col]: agg(names[i], col) }
        );
    }

    // return the final aggregated structure
    return t.get(col);
  }

  /**
   * @param {*} column
   * @returns {ArrayConstructor | import('../table/types.js').TypedArrayConstructor}
   */
  function arrayType$1(column) {
    // @ts-ignore
    return isTypedArray$1(column) ? column.constructor : Array;
  }

  /**
   * Base class representing a column-oriented data table.
   */
  let Table$1 = class Table {
    /**
     * Instantiate a Table instance.
     * @param {import('./types.js').ColumnData} columns
     *  An object mapping column names to values.
     * @param {string[]} [names]
     *  An ordered list of column names.
     * @param {import('./BitSet.js').BitSet} [filter]
     *  A filtering BitSet.
     * @param {import('./types.js').GroupBySpec} [group]
     *  A groupby specification.
     * @param {import('./types.js').RowComparator} [order]
     *  A row comparator function.
     * @param {import('./types.js').Params} [params]
     *  An object mapping parameter names to values.
     */
    constructor(columns, names, filter, group, order, params) {
      const data = Object.freeze({ ...columns });
      names = names?.slice() ?? Object.keys(data);
      const nrows = names.length ? data[names[0]].length : 0;
      /**
       * @private
       * @type {readonly string[]}
       */
      this._names = Object.freeze(names);
      /**
       * @private
       * @type {import('./types.js').ColumnData}
       */
      this._data = data;
      /**
       * @private
       * @type {number}
       */
      this._total = nrows;
      /**
       * @private
       * @type {number}
       */
      this._nrows = filter?.count() ?? nrows;
      /**
       * @private
       * @type {import('./BitSet.js').BitSet}
       */
      this._mask = filter ?? null;
      /**
       * @private
       * @type {import('./types.js').GroupBySpec}
       */
      this._group = group ?? null;
      /**
       * @private
       * @type {import('./types.js').RowComparator}
       */
      this._order = order ?? null;
      /**
       * @private
       * @type {import('./types.js').Params}
       */
      this._params = params;
      /**
       * @private
       * @type {Uint32Array}
       */
      this._index = null;
      /**
       * @private
       * @type {number[][] | Uint32Array[]}
       */
      this._partitions = null;
    }

    /**
     * Create a new table with the same type as this table.
     * The new table may have different data, filter, grouping, or ordering
     * based on the values of the optional configuration argument. If a
     * setting is not specified, it is inherited from the current table.
     * @param {import('./types.js').CreateOptions} [options]
     *  Creation options for the new table.
     * @return {this} A newly created table.
     */
    create({
      data = undefined,
      names = undefined,
      filter = undefined,
      groups = undefined,
      order = undefined
    } = {}) {
      const f = filter !== undefined ? filter : this.mask();
      // @ts-ignore
      return new this.constructor(
        data || this._data,
        names || (!data ? this._names : null),
        f,
        groups !== undefined ? groups : regroup(this._group, filter && f),
        order !== undefined ? order : this._order,
        this._params
      );
    }

    /**
     * Get or set table expression parameter values.
     * If called with no arguments, returns the current parameter values
     * as an object. Otherwise, adds the provided parameters to this
     * table's parameter set and returns the table. Any prior parameters
     * with names matching the input parameters are overridden.
     * @param {import('./types.js').Params} [values]
     *  The parameter values.
     * @return {this|import('./types.js').Params}
     *  The current parameter values (if called with no arguments) or this table.
     */
    params(values) {
      if (arguments.length) {
        if (values) {
          this._params = { ...this._params, ...values };
        }
        return this;
      } else {
        return this._params;
      }
    }

    /**
     * Provide an informative object string tag.
     */
    get [Symbol.toStringTag]() {
      if (!this._names) return 'Object'; // bail if called on prototype
      const nr = this.numRows();
      const nc = this.numCols();
      const plural = v => v !== 1 ? 's' : '';
      return `Table: ${nc} col${plural(nc)} x ${nr} row${plural(nr)}`
        + (this.isFiltered() ? ` (${this.totalRows()} backing)` : '')
        + (this.isGrouped() ? `, ${this._group.size} groups` : '')
        + (this.isOrdered() ? ', ordered' : '');
    }

    /**
     * Indicates if the table has a filter applied.
     * @return {boolean} True if filtered, false otherwise.
     */
    isFiltered() {
      return !!this._mask;
    }

    /**
     * Indicates if the table has a groupby specification.
     * @return {boolean} True if grouped, false otherwise.
     */
    isGrouped() {
      return !!this._group;
    }

    /**
     * Indicates if the table has a row order comparator.
     * @return {boolean} True if ordered, false otherwise.
     */
    isOrdered() {
      return !!this._order;
    }

    /**
     * Get the backing column data for this table.
     * @return {import('./types.js').ColumnData}
     *  Object of named column instances.
     */
    data() {
      return this._data;
    }

    /**
     * Returns the filter bitset mask, if defined.
     * @return {import('./BitSet.js').BitSet} The filter bitset mask.
     */
    mask() {
      return this._mask;
    }

    /**
     * Returns the groupby specification, if defined.
     * @return {import('./types.js').GroupBySpec} The groupby specification.
     */
    groups() {
      return this._group;
    }

    /**
     * Returns the row order comparator function, if specified.
     * @return {import('./types.js').RowComparator}
     *  The row order comparator function.
     */
    comparator() {
      return this._order;
    }

    /**
     * The total number of rows in this table, counting both
     * filtered and unfiltered rows.
     * @return {number} The number of total rows.
     */
    totalRows() {
      return this._total;
    }

    /**
     * The number of active rows in this table. This number may be
     * less than the *totalRows* if the table has been filtered.
     * @return {number} The number of rows.
     */
    numRows() {
      return this._nrows;
    }

    /**
     * The number of active rows in this table. This number may be
     * less than the *totalRows* if the table has been filtered.
     * @return {number} The number of rows.
     */
    get size() {
      return this._nrows;
    }

    /**
     * The number of columns in this table.
     * @return {number} The number of columns.
     */
    numCols() {
      return this._names.length;
    }

    /**
     * Filter function invoked for each column name.
     * @callback NameFilter
     * @param {string} name The column name.
     * @param {number} index The column index.
     * @param {string[]} array The array of names.
     * @return {boolean} Returns true to retain the column name.
     */

    /**
     * The table column names, optionally filtered.
     * @param {NameFilter} [filter] An optional filter function.
     *  If unspecified, all column names are returned.
     * @return {string[]} An array of matching column names.
     */
    columnNames(filter) {
      return filter ? this._names.filter(filter) : this._names.slice();
    }

    /**
     * The column name at the given index.
     * @param {number} index The column index.
     * @return {string} The column name,
     *  or undefined if the index is out of range.
     */
    columnName(index) {
      return this._names[index];
    }

    /**
     * The column index for the given name.
     * @param {string} name The column name.
     * @return {number} The column index, or -1 if the name is not found.
     */
    columnIndex(name) {
      return this._names.indexOf(name);
    }

    /**
     * Get the column instance with the given name.
     * @param {string} name The column name.
     * @return {import('./types.js').ColumnType | undefined}
     *  The named column, or undefined if it does not exist.
     */
    column(name) {
      return this._data[name];
    }

    /**
     * Get the column instance at the given index position.
     * @param {number} index The zero-based column index.
     * @return {import('./types.js').ColumnType | undefined}
     *  The column, or undefined if it does not exist.
     */
    columnAt(index) {
      return this._data[this._names[index]];
    }

    /**
     * Get an array of values contained in a column. The resulting array
     * respects any table filter or orderby criteria.
     * @param {string} name The column name.
     * @param {ArrayConstructor | import('./types.js').TypedArrayConstructor} [constructor=Array]
     *  The array constructor for instantiating the output array.
     * @return {import('./types.js').DataValue[] | import('./types.js').TypedArray}
     *  The array of column values.
     */
    array(name, constructor = Array) {
      const column = this.column(name);
      const array = new constructor(this.numRows());
      let idx = -1;
      this.scan(row => array[++idx] = column.at(row), true);
      return array;
    }

    /**
     * Get the value for the given column and row.
     * @param {string} name The column name.
     * @param {number} [row=0] The row index, defaults to zero if not specified.
     * @return {import('./types.js').DataValue} The table value at (column, row).
     */
    get(name, row = 0) {
      const column = this.column(name);
      return this.isFiltered() || this.isOrdered()
        ? column.at(this.indices()[row])
        : column.at(row);
    }

    /**
     * Returns an accessor ("getter") function for a column. The returned
     * function takes a row index as its single argument and returns the
     * corresponding column value.
     * @param {string} name The column name.
     * @return {import('./types.js').ColumnGetter} The column getter function.
     */
    getter(name) {
      const column = this.column(name);
      const indices = this.isFiltered() || this.isOrdered() ? this.indices() : null;
      if (indices) {
        return row => column.at(indices[row]);
      } else if (column) {
        return row => column.at(row);
      } else {
        error(`Unrecognized column: ${name}`);
      }
    }

    /**
     * Returns an object representing a table row.
     * @param {number} [row=0] The row index, defaults to zero if not specified.
     * @return {object} A row object with named properties for each column.
     */
    object(row = 0) {
      return objectBuilder$1(this)(row);
    }

    /**
     * Returns an array of objects representing table rows.
     * @param {import('./types.js').ObjectsOptions} [options]
     *  The options for row object generation.
     * @return {object[]} An array of row objects.
     */
    objects(options = {}) {
      const { grouped, limit, offset } = options;

      // generate array of row objects
      const names = resolve(this, options.columns || all());
      const createRow = rowObjectBuilder(this, names);
      const obj = [];
      this.scan(
        (row, data) => obj.push(createRow(row, data)),
        true, limit, offset
      );

      // produce nested output as requested
      if (grouped && this.isGrouped()) {
        const idx = [];
        this.scan(row => idx.push(row), true, limit, offset);
        return nest(this, idx, obj, grouped);
      }

      return obj;
    }

    /**
     * Returns an iterator over objects representing table rows.
     * @return {Iterator<object>} An iterator over row objects.
     */
    *[Symbol.iterator]() {
      const createRow = objectBuilder$1(this);
      const n = this.numRows();
      for (let i = 0; i < n; ++i) {
        yield createRow(i);
      }
    }

    /**
     * Returns an iterator over column values.
     * @return {Iterator<object>} An iterator over row objects.
     */
    *values(name) {
      const get = this.getter(name);
      const n = this.numRows();
      for (let i = 0; i < n; ++i) {
        yield get(i);
      }
    }

    /**
     * Print the contents of this table using the console.table() method.
     * @param {import('./types.js').PrintOptions|number} options
     *  The options for row object generation, determining which rows and
     *  columns are printed. If number-valued, specifies the row limit.
     * @return {this} The table instance.
     */
    print(options = {}) {
      const opt = isNumber(options)
        ? { limit: +options }
        // @ts-ignore
        : { ...options, limit: 10 };

      const obj = this.objects({ ...opt, grouped: false });
      const msg = `${this[Symbol.toStringTag]}. Showing ${obj.length} rows.`;

      console.log(msg);   // eslint-disable-line no-console
      console.table(obj); // eslint-disable-line no-console
      return this;
    }

    /**
     * Returns an array of indices for all rows passing the table filter.
     * @param {boolean} [order=true] A flag indicating if the returned
     *  indices should be sorted if this table is ordered. If false, the
     *  returned indices may or may not be sorted.
     * @return {Uint32Array} An array of row indices.
     */
    indices(order = true) {
      if (this._index) return this._index;

      const n = this.numRows();
      const index = new Uint32Array(n);
      const ordered = this.isOrdered();
      const bits = this.mask();
      let row = -1;

      // inline the following for performance:
      // this.scan(row => index[++i] = row);
      if (bits) {
        for (let i = bits.next(0); i >= 0; i = bits.next(i + 1)) {
          index[++row] = i;
        }
      } else {
        for (let i = 0; i < n; ++i) {
          index[++row] = i;
        }
      }

      // sort index vector
      if (order && ordered) {
        const { _order, _data } = this;
        index.sort((a, b) => _order(a, b, _data));
      }

      // save indices if they reflect table metadata
      if (order || !ordered) {
        this._index = index;
      }

      return index;
    }

    /**
     * Returns an array of indices for each group in the table.
     * If the table is not grouped, the result is the same as
     * the *indices* method, but wrapped within an array.
     * @param {boolean} [order=true] A flag indicating if the returned
     *  indices should be sorted if this table is ordered. If false, the
     *  returned indices may or may not be sorted.
     * @return {number[][] | Uint32Array[]} An array of row index arrays, one
     *  per group. The indices will be filtered if the table is filtered.
     */
    partitions(order = true) {
      // return partitions if already generated
      if (this._partitions) {
        return this._partitions;
      }

      // if not grouped, return a single partition
      if (!this.isGrouped()) {
        return [ this.indices(order) ];
      }

      // generate partitions
      const { keys, size } = this._group;
      const part = repeat(size, () => []);

      // populate partitions, don't sort if indices don't exist
      // inline the following for performance:
      // this.scan(row => part[keys[row]].push(row), sort);
      const sort = this._index;
      const bits = this.mask();
      const n = this.numRows();
      if (sort && this.isOrdered()) {
        for (let i = 0, r; i < n; ++i) {
          r = sort[i];
          part[keys[r]].push(r);
        }
      } else if (bits) {
        for (let i = bits.next(0); i >= 0; i = bits.next(i + 1)) {
          part[keys[i]].push(i);
        }
      } else {
        for (let i = 0; i < n; ++i) {
          part[keys[i]].push(i);
        }
      }

      // if ordered but not yet sorted, sort partitions directly
      if (order && !sort && this.isOrdered()) {
        const compare = this._order;
        const data = this._data;
        for (let i = 0; i < size; ++i) {
          part[i].sort((a, b) => compare(a, b, data));
        }
      }

      // save partitions if they reflect table metadata
      if (order || !this.isOrdered()) {
        this._partitions = part;
      }

      return part;
    }

    /**
     * Create a new fully-materialized instance of this table.
     * All filter and orderby settings are removed from the new table.
     * Instead, the backing data itself is filtered and ordered as needed.
     * @param {number[]} [indices] Ordered row indices to materialize.
     *  If unspecified, all rows passing the table filter are used.
     * @return {this} A reified table.
     */
    reify(indices) {
      const nrows = indices ? indices.length : this.numRows();
      const names = this._names;
      let data, groups;

      if (!indices && !this.isOrdered()) {
        if (!this.isFiltered()) {
          return this; // data already reified
        } else if (nrows === this.totalRows()) {
          data = this.data(); // all rows pass filter, skip copy
        }
      }

      if (!data) {
        const scan = indices ? f => indices.forEach(f) : f => this.scan(f, true);
        const ncols = names.length;
        data = {};

        for (let i = 0; i < ncols; ++i) {
          const name = names[i];
          const prev = this.column(name);
          const curr = data[name] = new (arrayType$1(prev))(nrows);
          let r = -1;
          // optimize array access
          isArrayType(prev)
            ? scan(row => curr[++r] = prev[row])
            : scan(row => curr[++r] = prev.at(row));
        }

        if (this.isGrouped()) {
          groups = reindex(this.groups(), scan, !!indices, nrows);
        }
      }

      return this.create({ data, names, groups, filter: null, order: null });
    }

    /**
     * Callback function to cancel a table scan.
     * @callback ScanStop
     * @return {void}
     */

    /**
     * Callback function invoked for each row of a table scan.
     * @callback ScanVisitor
     * @param {number} [row] The table row index.
     * @param {import('./types.js').ColumnData} [data]
     *  The backing table data store.
     * @param {ScanStop} [stop] Function to stop the scan early.
     *  Callees can invoke this function to prevent future calls.
     * @return {void}
     */

    /**
     * Perform a table scan, visiting each row of the table.
     * If this table is filtered, only rows passing the filter are visited.
     * @param {ScanVisitor} fn Callback invoked for each row of the table.
     * @param {boolean} [order=false] Indicates if the table should be
     *  scanned in the order determined by *orderby*. This
     *  argument has no effect if the table is unordered.
     * @property {number} [limit=Infinity] The maximum number of rows to scan.
     * @property {number} [offset=0] The row offset indicating how many
     *  initial rows to skip.
     */
    scan(fn, order, limit = Infinity, offset = 0) {
      const filter = this._mask;
      const nrows = this._nrows;
      const data = this._data;

      let i = offset || 0;
      if (i > nrows) return;

      const n = Math.min(nrows, i + limit);
      const stop = () => i = this._total;

      if (order && this.isOrdered() || filter && this._index) {
        const index = this.indices();
        const data = this._data;
        for (; i < n; ++i) {
          fn(index[i], data, stop);
        }
      } else if (filter) {
        let c = n - i + 1;
        for (i = filter.nth(i); --c && i > -1; i = filter.next(i + 1)) {
          fn(i, data, stop);
        }
      } else {
        for (; i < n; ++i) {
          fn(i, data, stop);
        }
      }
    }
  };

  function objectBuilder$1(table) {
    let b = table._builder;

    if (!b) {
      const createRow = rowObjectBuilder(table);
      const data = table.data();
      if (table.isOrdered() || table.isFiltered()) {
        const indices = table.indices();
        b = row => createRow(indices[row], data);
      } else {
        b = row => createRow(row, data);
      }
      table._builder = b;
    }

    return b;
  }

  function assign(table, ...others) {
    others = others.flat();
    const nrows = table.numRows();
    const base = table.reify();
    const cols = columnSet(base).groupby(base.groups());
    others.forEach(input => {
      input = input instanceof Table$1 ? input : new Table$1(input);
      if (input.numRows() !== nrows) error('Assign row counts do not match');
      input = input.reify();
      input.columnNames(name => cols.add(name, input.column(name)));
    });
    return cols.new(table);
  }

  function concat(table, ...others) {
    others = others.flat();
    const trows = table.numRows();
    const nrows = trows + others.reduce((n, t) => n + t.numRows(), 0);
    if (trows === nrows) return table;

    const tables = [table, ...others];
    const cols = columnSet();

    table.columnNames().forEach(name => {
      const arr = Array(nrows);
      let row = 0;
      tables.forEach(table => {
        const col = table.column(name) || { at: () => NULL };
        table.scan(trow => arr[row++] = col.at(trow));
      });
      cols.add(name, arr);
    });

    return cols.new(table);
  }

  function relocate(table, columns, {
    before = undefined,
    after = undefined
  } = {}) {
    const bef = before != null;
    const aft = after != null;

    if (!(bef || aft)) {
      error('relocate requires a before or after option.');
    }
    if (bef && aft) {
      error('relocate accepts only one of the before or after options.');
    }

    columns = resolve(table, columns);
    const anchors = [...resolve(table, bef ? before : after).keys()];
    const anchor = bef ? anchors[0] : anchors.pop();
    const select = new Map();

    // marshal inputs to select in desired order
    table.columnNames().forEach(name => {
      // check if we should assign the current column
      const assign = !columns.has(name);

      // at anchor column, insert relocated columns
      if (name === anchor) {
        if (aft && assign) select.set(name, name);
        for (const [key, value] of columns) {
          select.set(key, value);
        }
        if (aft) return; // exit if current column has been handled
      }

      if (assign) select.set(name, name);
    });

    return _select(table, select);
  }

  function bisector(compare) {
    return {
      left(a, x, lo, hi) {
        if (lo == null) lo = 0;
        if (hi == null) hi = a.length;
        while (lo < hi) {
          const mid = lo + hi >>> 1;
          if (compare(a[mid], x) < 0) lo = mid + 1;
          else hi = mid;
        }
        return lo;
      },
      right(a, x, lo, hi) {
        if (lo == null) lo = 0;
        if (hi == null) hi = a.length;
        while (lo < hi) {
          const mid = lo + hi >>> 1;
          if (compare(a[mid], x) > 0) hi = mid;
          else lo = mid + 1;
        }
        return lo;
      }
    };
  }

  const bisect$1 = bisector(ascending);

  function windowState(data, frame, adjust, ops, aggrs) {
    let rows, peer, cells, result, key;
    const isPeer = index => peer[index - 1] === peer[index];
    const numOps = ops.length;
    const numAgg = aggrs.length;

    const evaluate = ops.length
      ? unroll$1(
          ['w', 'r', 'k'],
          '{' + concat$1(ops, (_, i) => `r[_${i}.id][k]=_${i}.value(w,_${i}.get);`) + '}',
          ops
        )
      : () => {};

    const w = {
      i0: 0,
      i1: 0,
      index: 0,
      size: 0,
      peer: isPeer,

      init(partition, peers, results, group) {
        w.index = w.i0 = w.i1 = 0;
        w.size = peers.length;
        rows = partition;
        peer = peers;
        result = results;
        key = group;

        // initialize aggregates
        cells = aggrs ? aggrs.map(aggr => aggr.init()) : null;

        // initialize window ops
        for (let i = 0; i < numOps; ++i) {
          ops[i].init();
        }

        return w;
      },

      value(index, get) {
        return get(rows[index], data);
      },

      step(idx) {
        const [f0, f1] = frame;
        const n = w.size;
        const p0 = w.i0;
        const p1 = w.i1;

        w.i0 = f0 != null ? Math.max(0, idx - Math.abs(f0)) : 0;
        w.i1 = f1 != null ? Math.min(n, idx + Math.abs(f1) + 1) : n;
        w.index = idx;

        if (adjust) {
          if (w.i0 > 0 && isPeer(w.i0)) {
            w.i0 = bisect$1.left(peer, peer[w.i0]);
          }
          if (w.i1 < n && isPeer(w.i1)) {
            w.i1 = bisect$1.right(peer, peer[w.i1 - 1]);
          }
        }

        // evaluate aggregates
        for (let i = 0; i < numAgg; ++i) {
          const aggr = aggrs[i];
          const cell = cells[i];
          for (let j = p0; j < w.i0; ++j) {
            aggr.rem(cell, rows[j], data);
          }
          for (let j = p1; j < w.i1; ++j) {
            aggr.add(cell, rows[j], data);
          }
          aggr.write(cell, result, key);
        }

        // evaluate window ops
        evaluate(w, result, key);

        return result;
      }
    };

    return w;
  }

  const frameValue = op =>
    (op.frame || [null, null]).map(v => Number.isFinite(v) ? Math.abs(v) : null);

  const peersValue = op => !!op.peers;

  function windowOp(spec) {
    const { id, name, fields = [], params = [] } = spec;
    return {
      ...getWindow(name).create(...params),
      get: fields.length ? fields[0] : null,
      id
    };
  }

  function window(table, cols, exprs, result = {}, ops) {
    // instantiate window states
    const data = table.data();
    const states = windowStates(ops, data);
    const nstate = states.length;

    const write = unroll$1(
      ['r', 'd', 'op'],
      '{' + concat$1(cols, (_, i) => `_${i}[r] = $${i}(r, d, op);`) + '}',
      cols, exprs
    );

    // scan each ordered partition
    table.partitions().forEach((rows, key) => {
      const size = rows.length;
      const peers = windowPeers(table, rows);

      // initialize window states
      for (let i = 0; i < nstate; ++i) {
        states[i].init(rows, peers, result, key);
      }

      // calculate window values per-row
      const op = id => result[id][key];
      for (let index = 0; index < size; ++index) {
        // advance window frame, updates result object
        for (let i = 0; i < nstate; ++i) {
          states[i].step(index);
        }
        write(rows[index], data, op);
      }
    });
  }

  function windowStates(ops, data) {
    const map = {};

    // group operations by window frame parameters
    ops.forEach(op => {
      const frame = frameValue(op);
      const peers = peersValue(op);
      const key = `${frame},${peers}`;
      const { aggOps, winOps } = map[key] || (map[key] = {
        frame,
        peers,
        aggOps: [],
        winOps: []
      });
      hasAggregate(op.name)
        ? aggOps.push(op)
        : winOps.push(windowOp(op));
    });

    return Object.values(map).map(_ => windowState(
      data, _.frame, _.peers, _.winOps,
      reducers(_.aggOps, _.frame[0] != null ? -1 : 1)
    ));
  }

  function windowPeers(table, rows) {
    if (table.isOrdered()) {
      // generate peer ids for sort equality checking
      const compare = table.comparator();
      const data = table.data();
      const nrows = rows.length;
      const peers = new Uint32Array(nrows);
      for (let i = 1, index = 0; i < nrows; ++i) {
        peers[i] = compare(rows[i - 1], rows[i], data) ? ++index : index;
      }
      return peers;
    } else {
      // no sort, no peers: reuse row indices as peer ids
      return rows;
    }
  }

  function isWindowed(op) {
    return hasWindow(op.name) ||
      op.frame && (
        Number.isFinite(op.frame[0]) ||
        Number.isFinite(op.frame[1])
      );
  }

  function derive(table, values, options = {}) {
    const dt = _derive(table, parse$2(values, { table }), options);

    return options.drop || (options.before == null && options.after == null)
      ? dt
      : relocate(
          dt,
          Object.keys(values).filter(name => !table.column(name)),
          options
        );
  }

  function _derive(table, { names, exprs, ops = [] }, options = {}) {
    // instantiate output data
    const total = table.totalRows();
    const cols = columnSet(options.drop ? null : table);
    const data = names.map(name => cols.add(name, Array(total)));

    // analyze operations, compute non-windowed aggregates
    const [ aggOps, winOps ] = segmentOps(ops);

    const size = table.isGrouped() ? table.groups().size : 1;
    const result = aggregate(
      table, aggOps,
      repeat(ops.length, () => Array(size))
    );

    // perform table scans to generate output values
    winOps.length
      ? window(table, data, exprs, result, winOps)
      : output$1(table, data, exprs, result);

    return cols.derive(table);
  }

  function segmentOps(ops) {
    const aggOps = [];
    const winOps = [];
    const n = ops.length;

    for (let i = 0; i < n; ++i) {
      const op = ops[i];
      op.id = i;
      (isWindowed(op) ? winOps : aggOps).push(op);
    }

    return [aggOps, winOps];
  }

  function output$1(table, cols, exprs, result) {
    const bits = table.mask();
    const data = table.data();
    const { keys } = table.groups() || {};
    const op = keys
      ? (id, row) => result[id][keys[row]]
      : id => result[id][0];

    const m = cols.length;
    for (let j = 0; j < m; ++j) {
      const get = exprs[j];
      const col = cols[j];

      // inline the following for performance:
      // table.scan((i, data) => col[i] = get(i, data, op));
      if (bits) {
        for (let i = bits.next(0); i >= 0; i = bits.next(i + 1)) {
          col[i] = get(i, data, op);
        }
      } else {
        const n = table.totalRows();
        for (let i = 0; i < n; ++i) {
          col[i] = get(i, data, op);
        }
      }
    }
  }

  function filter$1(table, criteria) {
    const test = parse$2({ p: criteria }, { table });
    let predicate = test.exprs[0];
    if (test.ops.length) {
      const data = _derive(table, test, { drop: true }).column('p');
      predicate = row => data.at(row);
    }
    return _filter(table, predicate);
  }

  function _filter(table, predicate) {
    const n = table.totalRows();
    const bits = table.mask();
    const data = table.data();
    const filter = new BitSet(n);

    // inline the following for performance:
    // table.scan((row, data) => { if (predicate(row, data)) filter.set(row); });
    if (bits) {
      for (let i = bits.next(0); i >= 0; i = bits.next(i + 1)) {
        if (predicate(i, data)) filter.set(i);
      }
    } else {
      for (let i = 0; i < n; ++i) {
        if (predicate(i, data)) filter.set(i);
      }
    }

    return table.create({ filter });
  }

  function dedupe(table, ...keys) {
    keys = keys.flat();
    const gt = groupby(table, keys.length ? keys : table.columnNames());
    return filter$1(gt, 'row_number() === 1').ungroup().reify();
  }

  function rowLookup(table, hash) {
    const lut = new Map();
    table.scan((row, data) => {
      const key = hash(row, data);
      if (key != null && key === key) {
        lut.set(key, row);
      }
    });
    return lut;
  }

  function indexLookup(idx, data, hash) {
    const lut = new Map();
    const n = idx.length;
    for (let i = 0; i < n; ++i) {
      const row = idx[i];
      const key = hash(row, data);
      if (key != null && key === key) {
        lut.has(key)
          ? lut.get(key).push(i)
          : lut.set(key, [i]);
      }
    }
    return lut;
  }

  function parseKey(name, table, params) {
    const exprs = new Map();

    toArray(params).forEach((param, i) => {
      param = isNumber(param) ? table.columnName(param) : param;
      isString(param) ? exprs.set(i, field$1(param))
        : isFunction(param) || isObject(param) && param.expr ? exprs.set(i, param)
        : error(`Invalid ${name} key value: ${param+''}`);
    });

    const fn = parse$2(exprs, { table, aggregate: false, window: false });
    return keyFunction(fn.exprs, true);
  }

  function intersect$1(a, b) {
    const set = new Set(b);
    return a.filter(x => set.has(x));
  }

  function inferKeys(tableL, tableR, on) {
    if (!on) {
      // perform natural join if join condition not provided
      const isect = intersect$1(tableL.columnNames(), tableR.columnNames());
      if (!isect.length) error('Natural join requires shared column names.');
      on = [isect, isect];
    } else if (isString(on)) {
      on = [on, on];
    } else if (isArray$2(on) && on.length === 1) {
      on = [on[0], on[0]];
    }

    return on;
  }

  function keyPredicate(tableL, tableR, onL, onR) {
    if (onL.length !== onR.length) {
      error('Mismatched number of join keys');
    }
    return [
      parseKey('join', tableL, onL),
      parseKey('join', tableR, onR)
    ];
  }

  function semijoin(tableL, tableR, on) {
    return join_filter(tableL, tableR, on, { anti: false });
  }

  function antijoin(tableL, tableR, on) {
    return join_filter(tableL, tableR, on, { anti: true });
  }

  function join_filter(tableL, tableR, on, options) {
    on = inferKeys(tableL, tableR, on);

    const predicate = isArray$2(on)
      ? keyPredicate(tableL, tableR, ...on.map(toArray))
      : parse$2({ on }, { join: [tableL, tableR] }).exprs[0];

    return _join_filter(tableL, tableR, predicate, options);
  }

  function _join_filter(tableL, tableR, predicate, options = {}) {
    // calculate semi-join filter mask
    const filter = new BitSet(tableL.totalRows());
    const join = isArray$2(predicate) ? hashSemiJoin : loopSemiJoin;
    join(filter, tableL, tableR, predicate);

    // if anti-join, negate the filter
    if (options.anti) {
      filter.not().and(tableL.mask());
    }

    return tableL.create({ filter });
  }

  function hashSemiJoin(filter, tableL, tableR, [keyL, keyR]) {
    // build lookup table
    const lut = rowLookup(tableR, keyR);

    // scan table, update filter with matches
    tableL.scan((rowL, data) => {
      const rowR = lut.get(keyL(rowL, data));
      if (rowR >= 0) filter.set(rowL);
    });
  }

  function loopSemiJoin(filter, tableL, tableR, predicate) {
    const nL = tableL.numRows();
    const nR = tableR.numRows();
    const dataL = tableL.data();
    const dataR = tableR.data();

    if (tableL.isFiltered() || tableR.isFiltered()) {
      // use indices as at least one table is filtered
      const idxL = tableL.indices(false);
      const idxR = tableR.indices(false);
      for (let i = 0; i < nL; ++i) {
        const rowL = idxL[i];
        for (let j = 0; j < nR; ++j) {
          if (predicate(rowL, dataL, idxR[j], dataR)) {
            filter.set(rowL);
            break;
          }
        }
      }
    } else {
      // no filters, enumerate row indices directly
      for (let i = 0; i < nL; ++i) {
        for (let j = 0; j < nR; ++j) {
          if (predicate(i, dataL, j, dataR)) {
            filter.set(i);
            break;
          }
        }
      }
    }
  }

  function except(table, ...others) {
    others = others.flat();
    if (others.length === 0) return table;
    const names = table.columnNames();
    return dedupe(others.reduce((a, b) => antijoin(a, b.select(names)), table));
  }

  function unroll(table, values, options) {
    return _unroll(
      table,
      parse$1('unroll', table, values),
      options && options.drop
        ? { ...options, drop: parse$1('unroll', table, options.drop).names }
        : options
    );
  }

  function _unroll(table, { names = [], exprs = [], ops = [] }, options = {}) {
    if (!names.length) return table;

    const limit = options.limit > 0 ? +options.limit : Infinity;
    const index = options.index
      ? options.index === true ? 'index' : options.index + ''
      : null;
    const drop = new Set(options.drop);
    const get = aggregateGet(table, ops, exprs);

    // initialize output columns
    const cols = columnSet();
    const nset = new Set(names);
    const priors = [];
    const copies = [];
    const unroll = [];

    // original and copied columns
    table.columnNames().forEach(name => {
      if (!drop.has(name)) {
        const col = cols.add(name, []);
        if (!nset.has(name)) {
          priors.push(table.column(name));
          copies.push(col);
        }
      }
    });

    // unrolled output columns
    names.forEach(name => {
      if (!drop.has(name)) {
        if (!cols.has(name)) cols.add(name, []);
        unroll.push(cols.data[name]);
      }
    });

    // index column, if requested
    const icol = index ? cols.add(index, []) : null;

    let start = 0;
    const m = priors.length;
    const n = unroll.length;

    const copy = (row, maxlen) => {
      for (let i = 0; i < m; ++i) {
        copies[i].length = start + maxlen;
        copies[i].fill(priors[i].at(row), start, start + maxlen);
      }
    };

    const indices = icol
      ? (row, maxlen) => {
          for (let i = 0; i < maxlen; ++i) {
            icol[row + i] = i;
          }
        }
      : () => {};

    if (n === 1) {
      // optimize common case of one array-valued column
      const fn = get[0];
      const col = unroll[0];

      table.scan((row, data) => {
        // extract array data
        const array = toArray(fn(row, data));
        const maxlen = Math.min(array.length, limit);

        // copy original table data
        copy(row, maxlen);

        // copy unrolled array data
        for (let j = 0; j < maxlen; ++j) {
          col[start + j] = array[j];
        }

        // fill in array indices
        indices(start, maxlen);

        start += maxlen;
      });
    } else {
      table.scan((row, data) => {
        let maxlen = 0;

        // extract parallel array data
        const arrays = get.map(fn => {
          const value = toArray(fn(row, data));
          maxlen = Math.min(Math.max(maxlen, value.length), limit);
          return value;
        });

        // copy original table data
        copy(row, maxlen);

        // copy unrolled array data
        for (let i = 0; i < n; ++i) {
          const col = unroll[i];
          const arr = arrays[i];
          for (let j = 0; j < maxlen; ++j) {
            col[start + j] = arr[j];
          }
        }

        // fill in array indices
        indices(start, maxlen);

        start += maxlen;
      });
    }

    return cols.new(table);
  }

  function fold(table, values, options) {
    return _fold(table, parse$1('fold', table, values), options);
  }

  function _fold(table, { names = [], exprs = [], ops = [] }, options = {}) {
    if (names.length === 0) return table;

    const [k = 'key', v = 'value'] = options.as || [];
    const vals = aggregateGet(table, ops, exprs);

    return _unroll(
      table,
      {
        names: [k, v],
        exprs: [() => names, (row, data) => vals.map(fn => fn(row, data))]
      },
      { ...options, drop: names }
    );
  }

  function ungroup(table) {
    return table.isGrouped()
      ? table.create({ groups: null })
      : table;
  }

  function impute(table, values, options = {}) {
    values = parse$2(values, { table });

    values.names.forEach(name =>
      table.column(name) ? 0 : error(`Invalid impute column ${toString$1(name)}`)
    );

    if (options.expand) {
      const opt = { preparse: preparse$1, window: false, aggronly: true };
      const params = parse$1('impute', table, options.expand, opt);
      const result = _rollup(ungroup(table), params);
      return _impute(
        table, values, params.names,
        params.names.map(name => result.get(name, 0))
      );
    } else {
      return _impute(table, values);
    }
  }

  // map direct field reference to "unique" aggregate
  function preparse$1(map) {
    map.forEach((value, key) =>
      value.field ? map.set(key, array_agg_distinct(value + '')) : 0
    );
  }

  function _impute(table, values, keys, arrays) {
    const write = keys && keys.length;
    table = write ? expand(table, keys, arrays) : table;
    const { names, exprs, ops } = values;
    const gets = aggregateGet(table, ops, exprs);
    const cols = write ? null : columnSet(table);
    const rows = table.totalRows();

    names.forEach((name, i) => {
      const col = table.column(name);
      const out = write ? col : cols.add(name, Array(rows));
      const get = gets[i];

      table.scan(idx => {
        const v = col.at(idx);
        out[idx] = !isValid(v) ? get(idx) : v;
      });
    });

    return write ? table : table.create(cols);
  }

  function expand(table, keys, values) {
    const groups = table.groups();
    const data = table.data();

    // expansion keys and accessors
    const keyNames = (groups ? groups.names : []).concat(keys);
    const keyGet = (groups ? groups.get : [])
      .concat(keys.map(key => table.getter(key)));

    // build hash of existing rows
    const hash = new Set();
    const keyTable = keyFunction(keyGet);
    table.scan((idx, data) => hash.add(keyTable(idx, data)));

    // initialize output table data
    const names = table.columnNames();
    const cols = columnSet();
    const out = names.map(name => cols.add(name, []));
    names.forEach((name, i) => {
      const old = data[name];
      const col = out[i];
      table.scan(row => col.push(old.at(row)));
    });

    // enumerate expanded value sets and augment output table
    const keyEnum = keyFunction(keyGet.map((k, i) => a => a[i]));
    const set = unroll$1(
      'v',
      '{' + out.map((_, i) => `_${i}.push(v[$${i}]);`).join('') + '}',
      out, names.map(name => keyNames.indexOf(name))
    );

    if (groups) {
      let row = groups.keys.length;
      const prod = values.reduce((p, a) => p * a.length, groups.size);
      const keys = new Uint32Array(prod + (row - hash.size));
      keys.set(groups.keys);
      enumerate(groups, values, (vec, idx) => {
        if (!hash.has(keyEnum(vec))) {
          set(vec);
          keys[row++] = idx[0];
        }
      });
      cols.groupby({ ...groups, keys });
    } else {
      enumerate(groups, values, vec => {
        if (!hash.has(keyEnum(vec))) set(vec);
      });
    }

    return cols.new(table);
  }

  function enumerate(groups, values, callback) {
    const offset = groups ? groups.get.length : 0;
    const pad = groups ? 1 : 0;
    const len = pad + values.length;
    const lens = new Int32Array(len);
    const idxs = new Int32Array(len);
    const set = [];

    if (groups) {
      const { get, rows, size } = groups;
      lens[0] = size;
      set.push((vec, idx) => {
        const row = rows[idx];
        for (let i = 0; i < offset; ++i) {
          vec[i] = get[i](row);
        }
      });
    }

    values.forEach((a, i) => {
      const j = i + offset;
      lens[i + pad] = a.length;
      set.push((vec, idx) => vec[j] = a[idx]);
    });

    const vec = Array(offset + values.length);

    // initialize value vector
    for (let i = 0; i < len; ++i) {
      set[i](vec, 0);
    }
    callback(vec, idxs);

    // enumerate all combinations of values
    for (let i = len - 1; i >= 0;) {
      const idx = ++idxs[i];
      if (idx < lens[i]) {
        set[i](vec, idx);
        callback(vec, idxs);
        i = len - 1;
      } else {
        idxs[i] = 0;
        set[i](vec, 0);
        --i;
      }
    }
  }

  function intersect(table, ...others) {
    others = others.flat();
    const names = table.columnNames();
    return others.length
      ? dedupe(others.reduce((a, b) => semijoin(a, b.select(names)), table))
      : table.reify([]);
  }

  const OPT_L = { aggregate: false, window: false };
  const OPT_R = { ...OPT_L, index: 1 };
  const NONE = -Infinity;

  function cross(table, other, values, options) {
    return join(
      table,
      other,
      () => true,
      values,
      { ...options, left: true, right: true }
    );
  }

  function join(tableL, tableR, on, values, options = {}) {
    on = inferKeys(tableL, tableR, on);
    const optParse = { join: [tableL, tableR] };
    let predicate;

    if (isArray$2(on)) {
      const [onL, onR] = on.map(toArray);
      predicate = keyPredicate(tableL, tableR, onL, onR);

      if (!values) {
        // infer output columns, suppress duplicated key columns
        values = inferValues(tableL, onL, onR, options);
      }
    } else {
      predicate = parse$2({ on }, optParse).exprs[0];

      if (!values) {
        // include all table columns if values not provided
        values = [all(), all()];
      }
    }

    return _join(
      tableL, tableR, predicate,
      parseValues$1(tableL, tableR, values, optParse, options && options.suffix),
      options
    );
  }

  function inferValues(tableL, onL, onR, options) {
    const isect = [];
    onL.forEach((s, i) => isString(s) && s === onR[i] ? isect.push(s) : 0);
    const vR = not(isect);

    if (options.left && options.right) {
      // for full join, merge shared key columns together
      const shared = new Set(isect);
      return [
        tableL.columnNames().map(s => {
          const c = `[${toString$1(s)}]`;
          return shared.has(s)
            ? { [s]: `(a, b) => a${c} == null ? b${c} : a${c}` }
            : s;
        }),
        vR
      ];
    }

    return options.right ? [vR, all()] : [all(), vR];
  }

  function parseValues$1(tableL, tableR, values, optParse, suffix = []) {
    if (isArray$2(values)) {
      let vL, vR, vJ, n = values.length;
      vL = vR = vJ = { names: [], exprs: [] };

      if (n--) {
        vL = parse$1('join', tableL, values[0], optParse);
      }
      if (n--) {
        vR = parse$1('join', tableR, values[1], OPT_R);
      }
      if (n--) {
        vJ = parse$2(values[2], optParse);
      }

      // handle name collisions
      const rename = new Set();
      const namesL = new Set(vL.names);
      vR.names.forEach(name => {
        if (namesL.has(name)) {
          rename.add(name);
        }
      });
      if (rename.size) {
        suffix[0] !== '' && rekey(vL.names, rename, suffix[0] || '_1');
        suffix[1] !== '' && rekey(vR.names, rename, suffix[1] || '_2');
      }

      return {
        names: vL.names.concat(vR.names, vJ.names),
        exprs: vL.exprs.concat(vR.exprs, vJ.exprs)
      };
    } else {
      return parse$2(values, optParse);
    }
  }

  function rekey(names, rename, suffix) {
    names.forEach((name, i) => rename.has(name)
      ? (names[i] = name + suffix)
      : 0);
  }

  function emitter(columns, getters) {
    const args = ['i', 'a', 'j', 'b'];
    return unroll$1(
      args,
      '{' + concat$1(columns, (_, i) => `_${i}.push($${i}(${args}));`) + '}',
      columns, getters
    );
  }

  function _join(tableL, tableR, predicate, { names, exprs }, options = {}) {
    // initialize data for left table
    const dataL = tableL.data();
    const idxL = tableL.indices(false);
    const nL = idxL.length;
    const hitL = new Int32Array(nL);

    // initialize data for right table
    const dataR = tableR.data();
    const idxR = tableR.indices(false);
    const nR = idxR.length;
    const hitR = new Int32Array(nR);

    // initialize output data
    const ncols = names.length;
    const cols = columnSet();
    const columns = Array(ncols);
    const getters = Array(ncols);
    for (let i = 0; i < names.length; ++i) {
      columns[i] = cols.add(names[i], []);
      getters[i] = exprs[i];
    }
    const emit = emitter(columns, getters);

    // perform join
    const join = isArray$2(predicate) ? hashJoin : loopJoin;
    join(emit, predicate, dataL, dataR, idxL, idxR, hitL, hitR, nL, nR);

    if (options.left) {
      for (let i = 0; i < nL; ++i) {
        if (!hitL[i]) {
          emit(idxL[i], dataL, NONE, dataR);
        }
      }
    }

    if (options.right) {
      for (let j = 0; j < nR; ++j) {
        if (!hitR[j]) {
          emit(NONE, dataL, idxR[j], dataR);
        }
      }
    }

    return cols.new(tableL);
  }

  function loopJoin(emit, predicate, dataL, dataR, idxL, idxR, hitL, hitR, nL, nR) {
    // perform nested-loops join
    for (let i = 0; i < nL; ++i) {
      const rowL = idxL[i];
      for (let j = 0; j < nR; ++j) {
        const rowR = idxR[j];
        if (predicate(rowL, dataL, rowR, dataR)) {
          emit(rowL, dataL, rowR, dataR);
          hitL[i] = 1;
          hitR[j] = 1;
        }
      }
    }
  }

  function hashJoin(emit, [keyL, keyR], dataL, dataR, idxL, idxR, hitL, hitR, nL, nR) {
    // determine which table to hash
    let dataScan, keyScan, hitScan, idxScan;
    let dataHash, keyHash, hitHash, idxHash;
    let emitScan = emit;
    if (nL >= nR) {
      dataScan = dataL; keyScan = keyL; hitScan = hitL; idxScan = idxL;
      dataHash = dataR; keyHash = keyR; hitHash = hitR; idxHash = idxR;
    } else {
      dataScan = dataR; keyScan = keyR; hitScan = hitR; idxScan = idxR;
      dataHash = dataL; keyHash = keyL; hitHash = hitL; idxHash = idxL;
      emitScan = (i, a, j, b) => emit(j, b, i, a);
    }

    // build lookup table
    const lut = indexLookup(idxHash, dataHash, keyHash);

    // scan other table
    const m = idxScan.length;
    for (let j = 0; j < m; ++j) {
      const rowScan = idxScan[j];
      const list = lut.get(keyScan(rowScan, dataScan));
      if (list) {
        const n = list.length;
        for (let k = 0; k < n; ++k) {
          const i = list[k];
          emitScan(rowScan, dataScan, idxHash[i], dataHash);
          hitHash[i] = 1;
        }
        hitScan[j] = 1;
      }
    }
  }

  function lookup(tableL, tableR, on, ...values) {
    on = inferKeys(tableL, tableR, on);
    return _lookup(
      tableL,
      tableR,
      [ parseKey('lookup', tableL, on[0]), parseKey('lookup', tableR, on[1]) ],
      parse$1('lookup', tableR, values.flat())
    );
  }

  function _lookup(tableL, tableR, [keyL, keyR], { names, exprs, ops = [] }) {
    // instantiate output data
    const cols = columnSet(tableL);
    const total = tableL.totalRows();
    names.forEach(name => cols.add(name, Array(total).fill(NULL)));

    // build lookup table
    const lut = rowLookup(tableR, keyR);

    // generate setter function for lookup match
    const set = unroll$1(
      ['lr', 'rr', 'data'],
      '{' + concat$1(names, (_, i) => `_[${i}][lr] = $[${i}](rr, data);`) + '}',
      names.map(name => cols.data[name]),
      aggregateGet(tableR, ops, exprs)
    );

    // find matching rows, set values on match
    const dataR = tableR.data();
    tableL.scan((lrow, data) => {
      const rrow = lut.get(keyL(lrow, data));
      if (rrow >= 0) set(lrow, rrow, dataR);
    });

    return cols.derive(tableL);
  }

  // generate code to compare a single field
  const _compare = (u, v, lt, gt) => `((u = ${u}) < (v = ${v}) || u == null) && v != null ? ${lt} : (u > v || v == null) && u != null ? ${gt} : ((v = v instanceof Date ? +v : v), (u = u instanceof Date ? +u : u)) !== u && v === v ? ${lt} : v !== v && u === u ? ${gt} : `;
  const _collate = (u, v, lt, gt, f) => `(v = ${v}, (u = ${u}) == null && v == null) ? 0 : v == null ? ${gt} : u == null ? ${lt} : (u = ${f}(u,v)) ? u : `;

  function parse(table, fields) {
    // parse expressions, generate code for both a and b values
    const names = [];
    const exprs = [];
    const fn = [];
    let keys = null, opA = '0', opB = '0';
    if (table.isGrouped()) {
      keys = table.groups().keys;
      opA = 'ka';
      opB = 'kb';
    }
    const { ops } = parse$2(fields, {
      table,
      value: (name, node) => {
        names.push(name);
        if (node.escape) {
          // if an escaped function, invoke it directly
          const f = i => `fn[${fn.length}](${i}, data)`;
          exprs.push([f('a'), f('b')]);
          fn.push(node.escape);
        } else {
          // generate code to extract values to compare
          exprs.push([
            codegen(node, { index: 'a', op: opA }),
            codegen(node, { index: 'b', op: opB })
          ]);
        }
      },
      window: false
    });

    // calculate aggregate values if needed
    const result = aggregate(table, ops);
    const op = (id, row) => result[id][row];

    // generate comparison code for each field
    const n = names.length;
    let code = 'return (a, b) => {'
      + (op && table.isGrouped() ? 'const ka = keys[a], kb = keys[b];' : '')
      + 'let u, v; return ';
    for (let i = 0; i < n; ++i) {
      const field = fields.get(names[i]);
      const o = field.desc ? -1 : 1;
      const [u, v] = exprs[i];
      if (field.collate) {
        code += _collate(u, v, -o, o, `${o < 0 ? '-' : ''}fn[${fn.length}]`);
        fn.push(field.collate);
      } else {
        code += _compare(u, v, -o, o);
      }
    }
    code += '0;};';

    // instantiate and return comparator function
    return Function('op', 'keys', 'fn', 'data', code)(op, keys, fn, table.data());
  }

  function orderby(table, ...values) {
    return _orderby(table, parseValues(table, values.flat()));
  }

  function parseValues(table, params) {
    let index = -1;
    const exprs = new Map();
    const add = val => exprs.set(++index + '', val);

    params.forEach(param => {
      const expr = param.expr != null ? param.expr : param;

      if (isObject(expr) && !isFunction(expr)) {
        for (const key in expr) add(expr[key]);
      } else {
        add(
          isNumber(expr) ? field$1(param, table.columnName(expr))
            : isString(expr) ? field$1(param)
            : isFunction(expr) ? param
            : error(`Invalid orderby field: ${param+''}`)
        );
      }
    });

    return parse(table, exprs);
  }

  function _orderby(table, comparator) {
    return table.create({ order: comparator });
  }

  // TODO: enforce aggregates only (no output changes) for values
  function pivot(table, on, values, options) {
    return _pivot(
      table,
      parse$1('fold', table, on),
      parse$1('fold', table, values, { preparse, window: false, aggronly: true }),
      options
    );
  }

  // map direct field reference to "any" aggregate
  function preparse(map) {
    map.forEach((value, key) =>
      value.field ? map.set(key, any(value + '')) : 0
    );
  }

  const opt = (value, defaultValue) => value != null ? value : defaultValue;

  function _pivot(table, on, values, options = {}) {
    const { keys, keyColumn } = pivotKeys(table, on, options);
    const vsep = opt(options.valueSeparator, '_');
    const namefn = values.names.length > 1
      ? (i, name) => name + vsep + keys[i]
      : i => keys[i];

    // perform separate aggregate operations for each key
    // if keys do not match, emit NaN so aggregate skips it
    // use custom toString method for proper field resolution
    const results = keys.map(
      k => aggregate(table, values.ops.map(op => {
        if (op.name === 'count') { // fix #273
          const fn = r => k === keyColumn[r] ? 1 : NaN;
          fn.toString = () => k + ':1';
          return { ...op, name: 'sum', fields: [fn] };
        }
        const fields = op.fields.map(f => {
          const fn = (r, d) => k === keyColumn[r] ? f(r, d) : NaN;
          fn.toString = () => k + ':' + f;
          return fn;
        });
        return { ...op, fields };
      }))
    );

    return output(values, namefn, table.groups(), results).new(table);
  }

  function pivotKeys(table, on, options) {
    const limit = options.limit > 0 ? +options.limit : Infinity;
    const sort = opt(options.sort, true);
    const ksep = opt(options.keySeparator, '_');

    // construct key accessor function
    const get = aggregateGet(table, on.ops, on.exprs);
    const key = get.length === 1
      ? get[0]
      : (row, data) => get.map(fn => fn(row, data)).join(ksep);

    // generate vector of per-row key values
    const kcol = Array(table.totalRows());
    table.scan((row, data) => kcol[row] = key(row, data));

    // collect unique key values
    const uniq = aggregate(
      ungroup(table),
      [ {
        id: 0,
        name: 'array_agg_distinct',
        fields: [(row => kcol[row])], params: []
      } ]
    )[0][0];

    // get ordered set of unique key values
    const keys = sort ? uniq.sort() : uniq;

    // return key values
    return {
      keys: Number.isFinite(limit) ? keys.slice(0, limit) : keys,
      keyColumn: kcol
    };
  }

  function output({ names, exprs }, namefn, groups, results) {
    const size = groups ? groups.size : 1;
    const cols = columnSet();
    const m = results.length;
    const n = names.length;

    let result;
    const op = (id, row) => result[id][row];

    // write groupby fields to output
    if (groups) groupOutput(cols, groups);

    // write pivot values to output
    for (let i = 0; i < n; ++i) {
      const get = exprs[i];
      if (get.field != null) {
        // if expression is op only, use aggregates directly
        for (let j = 0; j < m; ++j) {
          cols.add(namefn(j, names[i]), results[j][get.field]);
        }
      } else if (size > 1) {
        // if multiple groups, evaluate expression for each
        for (let j = 0; j < m; ++j) {
          result = results[j];
          const col = cols.add(namefn(j, names[i]), Array(size));
          for (let k = 0; k < size; ++k) {
            col[k] = get(k, null, op);
          }
        }
      } else {
        // if only one group, no need to loop
        for (let j = 0; j < m; ++j) {
          result = results[j];
          cols.add(namefn(j, names[i]), [ get(0, null, op) ]);
        }
      }
    }

    return cols;
  }

  function reduce(table, reducer) {
    const cols = columnSet();
    const groups = table.groups();

    // initialize groups
    const { get, names = [], rows, size = 1 } = groups || {};
    const counts = new Uint32Array(size + 1);
    names.forEach(name => cols.add(name, null));

    // compute reduced values
    const cells = groups
      ? reduceGroups(table, reducer, groups)
      : [ reduceFlat(table, reducer) ];

    // initialize output columns
    reducer.outputs().map(name => cols.add(name, []));

    // write reduced values to output columns
    const n = counts.length - 1;
    let len = 0;
    for (let i = 0; i < n; ++i) {
      len += counts[i + 1] = reducer.write(cells[i], cols.data, counts[i]);
    }

    // write group values to output columns
    if (groups) {
      const data = table.data();
      names.forEach((name, index) => {
        const column = cols.data[name] = Array(len);
        const getter = get[index];
        for (let i = 0, j = 0; i < size; ++i) {
          column.fill(getter(rows[i], data), j, j += counts[i + 1]);
        }
      });
    }

    return cols.new(table);
  }

  function rename(table, ...columns) {
    const map = new Map();
    table.columnNames(x => (map.set(x, x), 0));
    return _select(table, resolve(table, columns.flat(), map));
  }

  function sampleIndices(buffer, replace, index, weight) {
    return (
      replace
        ? (weight ? sampleRW : sampleRU)
        : (weight ? sampleNW : sampleNU)
    )(buffer.length, buffer, index, weight);
  }

  // uniform sampling with replacement
  // uses straightforward uniform sampling
  function sampleRU(size, buffer, index) {
    const n = index.length;
    for (let i = 0; i < size; ++i) {
      buffer[i] = index[(n * random()) | 0];
    }
    return buffer;
  }

  // weighted sampling with replacement
  // uses binary search lookup against cumulative weight
  function sampleRW(size, buffer, index, weight) {
    const n = index.length;
    const w = new Float64Array(n);

    let sum = 0;
    for (let i = 0; i < n; ++i) {
      w[i] = (sum += weight(index[i]));
    }

    const bisect = bisector(ascending).right;
    for (let i = 0; i < size; ++i) {
      buffer[i] = index[bisect(w, sum * random())];
    }
    return buffer;
  }

  // uniform sampling without replacement
  // uses reservoir sampling to build out the sample
  // https://en.wikipedia.org/wiki/Reservoir_sampling
  function sampleNU(size, buffer, index) {
    const n = index.length;
    if (size >= n) return index;

    for (let i = 0; i < size; ++i) {
      buffer[i] = index[i];
    }

    for (let i = size; i < n; ++i) {
      const j = i * random();
      if (j < size) {
        buffer[j | 0] = index[i];
      }
    }

    return buffer;
  }

  // weighted sample without replacement
  // uses method of Efraimidis and Spirakis
  // TODO: could use min-heap to improve efficiency
  function sampleNW(size, buffer, index, weight) {
    const n = index.length;
    if (size >= n) return index;

    const w = new Float32Array(n);
    const k = new Uint32Array(n);
    for (let i = 0; i < n; ++i) {
      k[i] = i;
      w[i] = -Math.log(random()) / weight(index[i]);
    }

    k.sort((a, b) => w[a] - w[b]);
    for (let i = 0; i < size; ++i) {
      buffer[i] = index[k[i]];
    }
    return buffer;
  }

  function shuffleIndices(array, lo = 0, hi = array.length) {
    let n = hi - (lo = +lo);

    while (n) {
      const i = random() * n-- | 0;
      const v = array[n + lo];
      array[n + lo] = array[i + lo];
      array[i + lo] = v;
    }

    return array;
  }

  function sample(table, size, options = {}) {
    return _sample(
      table,
      parseSize(table, size),
      parseWeight(table, options.weight),
      options
    );
  }

  const get = col => row => col.at(row) || 0;

  function parseSize(table, size) {
    return isNumber(size)
      ? () => size
      : get(_rollup(table, parse$2({ size }, { table, window: false })).column('size'));
  }

  function parseWeight(table, w) {
    if (w == null) return null;
    w = isNumber(w) ? table.columnName(w) : w;
    return get(
      isString(w)
        ? table.column(w)
        : _derive(table, parse$2({ w }, { table }), { drop: true }).column('w')
    );
  }

  function _sample(table, size, weight, options = {}) {
    const { replace, shuffle } = options;
    const parts = table.partitions(false);

    let total = 0;
    size = parts.map((idx, group) => {
      let s = size(group);
      total += (s = (replace ? s : Math.min(idx.length, s)));
      return s;
    });

    const samples = new Uint32Array(total);
    let curr = 0;

    parts.forEach((idx, group) => {
      const sz = size[group];
      const buf = samples.subarray(curr, curr += sz);

      if (!replace && sz === idx.length) {
        // sample size === data size, no replacement
        // no need to sample, just copy indices
        buf.set(idx);
      } else {
        sampleIndices(buf, replace, idx, weight);
      }
    });

    if (shuffle !== false && (parts.length > 1 || !replace)) {
      // sampling with replacement methods shuffle, so in
      // that case a single partition is already good to go
      shuffleIndices(samples);
    }

    return table.reify(samples);
  }

  /**
   * Generate a table expression that filters a table based on ordered row
   * indices from start to end (end not included), where start and end
   * represent per-group ordered row numbers in the table. The resulting
   * string can be used as the input to the filter verb.
   * @param {number} [start] Zero-based index at which to start extraction.
   *  A negative index indicates an offset from the end of the group.
   *  If start is undefined, slice starts from the index 0.
   * @param {number} [end] Zero-based index before which to end extraction.
   *  A negative index indicates an offset from the end of the group.
   *  If end is omitted, slice extracts through the end of the group.
   * @return {string} A table expression string for slicing values.
   * @example slice(1, -1)
   */
  function _slice(start = 0, end = Infinity) {
    return `${prep$1(start)} < row_number() && row_number() <= ${prep$1(end)}`;
  }

  function prep$1(index) {
    return index < 0 ? `count() + ${index}` : index;
  }

  function slice(table, start = 0, end = Infinity) {
    if (table.isGrouped()) {
      return filter$1(table, _slice(start, end)).reify();
    }

    // if not grouped, scan table directly
    const indices = [];
    const nrows = table.numRows();
    start = Math.max(0, start + (start < 0 ? nrows : 0));
    end = Math.min(nrows, Math.max(0, end + (end < 0 ? nrows : 0)));
    table.scan(row => indices.push(row), true, end - start, start);
    return table.reify(indices);
  }

  function spread(table, values, options) {
    return _spread(table, parse$1('spread', table, values), options);
  }

  function _spread(table, { names, exprs, ops = [] }, options = {}) {
    if (names.length === 0) return table;

    // ignore 'as' if there are multiple field names
    const as = (names.length === 1 && options.as) || [];
    const drop = options.drop == null ? true : !!options.drop;
    const limit = options.limit == null
      ? as.length || Infinity
      : Math.max(1, +options.limit || 1);

    const get = aggregateGet(table, ops, exprs);
    const cols = columnSet();
    const map = names.reduce((map, name, i) => map.set(name, i), new Map());

    const add = (index, name) => {
      const columns = spreadCols(table, get[index], limit);
      const n = columns.length;
      for (let i = 0; i < n; ++i) {
        cols.add(as[i] || `${name}_${i + 1}`, columns[i]);
      }
    };

    table.columnNames().forEach(name => {
      if (map.has(name)) {
        if (!drop) cols.add(name, table.column(name));
        add(map.get(name), name);
        map.delete(name);
      } else {
        cols.add(name, table.column(name));
      }
    });

    map.forEach(add);

    return cols.derive(table);
  }

  function spreadCols(table, get, limit) {
    const nrows = table.totalRows();
    const columns = [];

    table.scan((row, data) => {
      const values = toArray(get(row, data));
      const n = Math.min(values.length, limit);
      while (columns.length < n) {
        columns.push(Array(nrows).fill(NULL));
      }
      for (let i = 0; i < n; ++i) {
        columns[i][row] = values[i];
      }
    });

    return columns;
  }

  function union$1(table, ...others) {
    return dedupe(concat(table, others.flat()));
  }

  function unorder(table) {
    return table.isOrdered()
      ? table.create({ order: null })
      : table;
  }

  /** Magic bytes 'ARROW1' indicating the Arrow 'file' format. */
  const MAGIC = Uint8Array.of(65, 82, 82, 79, 87, 49);

  /**
   * Apache Arrow version.
   */
  const Version = /** @type {const} */ ({
    /** 0.1.0 (October 2016). */
    V1: 0,
    /** 0.2.0 (February 2017). Non-backwards compatible with V1. */
    V2: 1,
    /** 0.3.0 -> 0.7.1 (May - December 2017). Non-backwards compatible with V2. */
    V3: 2,
    /** >= 0.8.0 (December 2017). Non-backwards compatible with V3. */
    V4: 3,
    /**
     * >= 1.0.0 (July 2020). Backwards compatible with V4 (V5 readers can read V4
     * metadata and IPC messages). Implementations are recommended to provide a
     * V4 compatibility mode with V5 format changes disabled.
     *
     * Incompatible changes between V4 and V5:
     * - Union buffer layout has changed.
     *   In V5, Unions don't have a validity bitmap buffer.
     */
    V5: 4
  });

  /**
   * Endianness of Arrow-encoded data.
   */
  const Endianness = /** @type {const} */ ({
    Little: 0,
    Big: 1
  });

  /**
   * Message header type codes.
   */
  const MessageHeader = /** @type {const} */ ({
    NONE: 0,
    /**
     * A Schema describes the columns in a record batch.
     */
    Schema: 1,
    /**
     * For sending dictionary encoding information. Any Field can be
     * dictionary-encoded, but in this case none of its children may be
     * dictionary-encoded.
     * There is one vector / column per dictionary, but that vector / column
     * may be spread across multiple dictionary batches by using the isDelta
     * flag.
     */
    DictionaryBatch: 2,
    /**
     * A data header describing the shared memory layout of a "record" or "row"
     * batch. Some systems call this a "row batch" internally and others a "record
     * batch".
     */
    RecordBatch: 3,
    /**
     * EXPERIMENTAL: Metadata for n-dimensional arrays, aka "tensors" or
     * "ndarrays". Arrow implementations in general are not required to implement
     * this type.
     *
     * Not currently supported by Flechette.
     */
    Tensor: 4,
    /**
     * EXPERIMENTAL: Metadata for n-dimensional sparse arrays, aka "sparse
     * tensors". Arrow implementations in general are not required to implement
     * this type.
     *
     * Not currently supported by Flechette.
     */
    SparseTensor: 5
  });

  /**
   * Field data type ids.
   * Only non-negative values ever occur in IPC flatbuffer binary data.
   */
  const Type = /** @type {const} */ ({
    /**
     * Dictionary types compress data by using a set of integer indices to
     * lookup potentially repeated vales in a separate dictionary of values.
     *
     * This type entry is provided for API convenience, it does not occur
     * in actual Arrow IPC binary data.
     */
    Dictionary: -1,
    /** No data type. Included for flatbuffer compatibility. */
    NONE: 0,
    /** Null values only. */
    Null: 1,
    /** Integers, either signed or unsigned, with 8, 16, 32, or 64 bit widths. */
    Int: 2,
    /** Floating point numbers with 16, 32, or 64 bit precision. */
    Float: 3,
    /** Opaque binary data. */
    Binary: 4,
    /** Unicode with UTF-8 encoding. */
    Utf8: 5,
    /** Booleans represented as 8 bit bytes. */
    Bool: 6,
    /**
     * Exact decimal value represented as an integer value in two's complement.
     * Currently only 128-bit (16-byte) and 256-bit (32-byte) integers are used.
     * The representation uses the endianness indicated in the schema.
     */
    Decimal: 7,
    /**
     * Date is either a 32-bit or 64-bit signed integer type representing an
     * elapsed time since UNIX epoch (1970-01-01), stored in either of two units:
     * - Milliseconds (64 bits) indicating UNIX time elapsed since the epoch (no
     * leap seconds), where the values are evenly divisible by 86400000
     * - Days (32 bits) since the UNIX epoch
     */
    Date: 8,
    /**
     * Time is either a 32-bit or 64-bit signed integer type representing an
     * elapsed time since midnight, stored in either of four units: seconds,
     * milliseconds, microseconds or nanoseconds.
     *
     * The integer `bitWidth` depends on the `unit` and must be one of the following:
     * - SECOND and MILLISECOND: 32 bits
     * - MICROSECOND and NANOSECOND: 64 bits
     *
     * The allowed values are between 0 (inclusive) and 86400 (=24*60*60) seconds
     * (exclusive), adjusted for the time unit (for example, up to 86400000
     * exclusive for the MILLISECOND unit).
     * This definition doesn't allow for leap seconds. Time values from
     * measurements with leap seconds will need to be corrected when ingesting
     * into Arrow (for example by replacing the value 86400 with 86399).
     */
    Time: 9,
    /**
     * Timestamp is a 64-bit signed integer representing an elapsed time since a
     * fixed epoch, stored in either of four units: seconds, milliseconds,
     * microseconds or nanoseconds, and is optionally annotated with a timezone.
     *
     * Timestamp values do not include any leap seconds (in other words, all
     * days are considered 86400 seconds long).
     *
     * The timezone is an optional string for the name of a timezone, one of:
     *
     *  - As used in the Olson timezone database (the "tz database" or
     *    "tzdata"), such as "America/New_York".
     *  - An absolute timezone offset of the form "+XX:XX" or "-XX:XX",
     *    such as "+07:30".
     *
     * Whether a timezone string is present indicates different semantics about
     * the data.
     */
    Timestamp: 10,
    /**
     * A "calendar" interval which models types that don't necessarily
     * have a precise duration without the context of a base timestamp (e.g.
     * days can differ in length during day light savings time transitions).
     * All integers in the units below are stored in the endianness indicated
     * by the schema.
     *
     *  - YEAR_MONTH - Indicates the number of elapsed whole months, stored as
     *    4-byte signed integers.
     *  - DAY_TIME - Indicates the number of elapsed days and milliseconds (no
     *    leap seconds), stored as 2 contiguous 32-bit signed integers (8-bytes
     *    in total). Support of this IntervalUnit is not required for full arrow
     *    compatibility.
     *  - MONTH_DAY_NANO - A triple of the number of elapsed months, days, and
     *    nanoseconds. The values are stored contiguously in 16-byte blocks.
     *    Months and days are encoded as 32-bit signed integers and nanoseconds
     *    is encoded as a 64-bit signed integer. Nanoseconds does not allow for
     *    leap seconds. Each field is independent (e.g. there is no constraint
     *    that nanoseconds have the same sign as days or that the quantity of
     *    nanoseconds represents less than a day's worth of time).
     */
    Interval: 11,
    /**
     * List (vector) data supporting variably-sized lists.
     * A list has a single child data type for list entries.
     */
    List: 12,
    /**
     * A struct consisting of multiple named child data types.
     */
    Struct: 13,
    /**
     * A union is a complex type with parallel child data types. By default ids
     * in the type vector refer to the offsets in the children. Optionally
     * typeIds provides an indirection between the child offset and the type id.
     * For each child `typeIds[offset]` is the id used in the type vector.
     */
    Union: 14,
    /**
     * Binary data where each entry has the same fixed size.
     */
    FixedSizeBinary: 15,
    /**
     * List (vector) data where every list has the same fixed size.
     * A list has a single child data type for list entries.
     */
    FixedSizeList: 16,
    /**
     * A Map is a logical nested type that is represented as
     * List<entries: Struct<key: K, value: V>>
     *
     * In this layout, the keys and values are each respectively contiguous. We do
     * not constrain the key and value types, so the application is responsible
     * for ensuring that the keys are hashable and unique. Whether the keys are sorted
     * may be set in the metadata for this field.
     *
     * In a field with Map type, the field has a child Struct field, which then
     * has two children: key type and the second the value type. The names of the
     * child fields may be respectively "entries", "key", and "value", but this is
     * not enforced.
     *
     * Map
     * ```text
     *   - child[0] entries: Struct
     *   - child[0] key: K
     *   - child[1] value: V
     *  ```
     * Neither the "entries" field nor the "key" field may be nullable.
     *
     * The metadata is structured so that Arrow systems without special handling
     * for Map can make Map an alias for List. The "layout" attribute for the Map
     * field must have the same contents as a List.
     */
    Map: 17,
    /**
     * An absolute length of time unrelated to any calendar artifacts. For the
     * purposes of Arrow implementations, adding this value to a Timestamp
     * ("t1") naively (i.e. simply summing the two numbers) is acceptable even
     * though in some cases the resulting Timestamp (t2) would not account for
     * leap-seconds during the elapsed time between "t1" and "t2". Similarly,
     * representing the difference between two Unix timestamp is acceptable, but
     * would yield a value that is possibly a few seconds off from the true
     * elapsed time.
     *
     * The resolution defaults to millisecond, but can be any of the other
     * supported TimeUnit values as with Timestamp and Time types. This type is
     * always represented as an 8-byte integer.
     */
    Duration: 18,
    /**
     * Same as Binary, but with 64-bit offsets, allowing representation of
     * extremely large data values.
     */
    LargeBinary: 19,
    /**
     * Same as Utf8, but with 64-bit offsets, allowing representation of
     * extremely large data values.
     */
    LargeUtf8: 20,
    /**
     * Same as List, but with 64-bit offsets, allowing representation of
     * extremely large data values.
     */
    LargeList: 21,
    /**
     * Contains two child arrays, run_ends and values. The run_ends child array
     * must be a 16/32/64-bit integer array which encodes the indices at which
     * the run with the value in each corresponding index in the values child
     * array ends. Like list/struct types, the value array can be of any type.
     */
    RunEndEncoded: 22,
    /**
     * Logically the same as Binary, but the internal representation uses a view
     * struct that contains the string length and either the string's entire data
     * inline (for small strings) or an inlined prefix, an index of another buffer,
     * and an offset pointing to a slice in that buffer (for non-small strings).
     *
     * Since it uses a variable number of data buffers, each Field with this type
     * must have a corresponding entry in `variadicBufferCounts`.
     */
    BinaryView: 23,
    /**
     * Logically the same as Utf8, but the internal representation uses a view
     * struct that contains the string length and either the string's entire data
     * inline (for small strings) or an inlined prefix, an index of another buffer,
     * and an offset pointing to a slice in that buffer (for non-small strings).
     *
     * Since it uses a variable number of data buffers, each Field with this type
     * must have a corresponding entry in `variadicBufferCounts`.
     */
    Utf8View: 24,
    /**
     * Represents the same logical types that List can, but contains offsets and
     * sizes allowing for writes in any order and sharing of child values among
     * list values.
     */
    ListView: 25,
    /**
     * Same as ListView, but with 64-bit offsets and sizes, allowing to represent
     * extremely large data values.
     */
    LargeListView: 26
  });

  /**
   * Floating point number precision.
   */
  const Precision = /** @type {const} */ ({
    /** 16-bit floating point number. */
    HALF: 0,
    /** 32-bit floating point number. */
    SINGLE: 1,
    /** 64-bit floating point number. */
    DOUBLE: 2
  });

  /**
   * Date units.
   */
  const DateUnit = /** @type {const} */ ({
    /* Days (as 32 bit int) since the UNIX epoch. */
    DAY: 0,
    /**
     * Milliseconds (as 64 bit int) indicating UNIX time elapsed since the epoch
     * (no leap seconds), with values evenly divisible by 86400000.
     */
    MILLISECOND: 1
  });

  /**
   * Time units.
   */
  const TimeUnit = /** @type {const} */ ({
    /** Seconds. */
    SECOND: 0,
    /** Milliseconds. */
    MILLISECOND: 1,
    /** Microseconds. */
    MICROSECOND: 2,
    /** Nanoseconds. */
    NANOSECOND: 3
  });

  /**
   * Date/time interval units.
   */
  const IntervalUnit = /** @type {const} */ ({
    /**
     * Indicates the number of elapsed whole months, stored as 4-byte signed
     * integers.
     */
    YEAR_MONTH: 0,
    /**
     * Indicates the number of elapsed days and milliseconds (no leap seconds),
     * stored as 2 contiguous 32-bit signed integers (8-bytes in total). Support
     * of this IntervalUnit is not required for full arrow compatibility.
     */
    DAY_TIME: 1,
    /**
     * A triple of the number of elapsed months, days, and nanoseconds.
     * The values are stored contiguously in 16-byte blocks. Months and days are
     * encoded as 32-bit signed integers and nanoseconds is encoded as a 64-bit
     * signed integer. Nanoseconds does not allow for leap seconds. Each field is
     * independent (e.g. there is no constraint that nanoseconds have the same
     * sign as days or that the quantity of nanoseconds represents less than a
     * day's worth of time).
     */
    MONTH_DAY_NANO: 2
  });

  /**
   * Union type modes.
   */
  const UnionMode = /** @type {const} */ ({
    /** Sparse union layout with full arrays for each sub-type. */
    Sparse: 0,
    /** Dense union layout with offsets into value arrays. */
    Dense: 1
  });

  const uint8Array = Uint8Array;
  const uint16Array = Uint16Array;
  const uint32Array = Uint32Array;
  const uint64Array = BigUint64Array;
  const int8Array = Int8Array;
  const int16Array = Int16Array;
  const int32Array = Int32Array;
  const int64Array = BigInt64Array;
  const float32Array = Float32Array;
  const float64Array = Float64Array;

  /**
   * Return the appropriate typed array constructor for the given
   * integer type metadata.
   * @param {number} bitWidth The integer size in bits.
   * @param {boolean} signed Flag indicating if the integer is signed.
   * @returns {import('../types.js').IntArrayConstructor}
   */
  function intArrayType(bitWidth, signed) {
    const i = Math.log2(bitWidth) - 3;
    return (
      signed
        ? [int8Array, int16Array, int32Array, int64Array]
        : [uint8Array, uint16Array, uint32Array, uint64Array]
    )[i];
  }

  /** Shared prototype for typed arrays. */
  const TypedArray = Object.getPrototypeOf(Int8Array);

  /**
   * Check if a value is a typed array.
   * @param {*} value The value to check.
   * @returns {value is import('../types.js').TypedArray}
   *  True if value is a typed array, false otherwise.
   */
  function isTypedArray(value) {
    return value instanceof TypedArray;
  }

  /**
   * Check if a value is either a standard array or typed array.
   * @param {*} value The value to check.
   * @returns {value is (Array | import('../types.js').TypedArray)}
   *  True if value is an array, false otherwise.
   */
  function isArray(value) {
    return Array.isArray(value) || isTypedArray(value);
  }

  /**
   * Check if a value is an array type (constructor) for 64-bit integers,
   * one of BigInt64Array or BigUint64Array.
   * @param {*} value The value to check.
   * @returns {value is import('../types.js').Int64ArrayConstructor}
   *  True if value is a 64-bit array type, false otherwise.
   */
  function isInt64ArrayType(value) {
    return value === int64Array || value === uint64Array;
  }

  /**
   * Determine the correct index into an offset array for a given
   * full column row index. Assumes offset indices can be manipulated
   * as 32-bit signed integers.
   * @param {import('../types.js').IntegerArray} offsets The offsets array.
   * @param {number} index The full column row index.
   */
  function bisect(offsets, index) {
    let a = 0;
    let b = offsets.length;
    if (b <= 2147483648) { // 2 ** 31
      // fast version, use unsigned bit shift
      // array length fits within 32-bit signed integer
      do {
        const mid = (a + b) >>> 1;
        if (offsets[mid] <= index) a = mid + 1;
        else b = mid;
      } while (a < b);
    } else {
      // slow version, use division and truncate
      // array length exceeds 32-bit signed integer
      do {
        const mid = Math.trunc((a + b) / 2);
        if (offsets[mid] <= index) a = mid + 1;
        else b = mid;
      } while (a < b);
    }
    return a;
  }

  /**
   * Compute a 64-bit aligned buffer size.
   * @param {number} length The starting size.
   * @param {number} bpe Bytes per element.
   * @returns {number} The aligned size.
   */
  function align64(length, bpe = 1) {
    return (((length * bpe) + 7) & ~7) / bpe;
  }

  /**
   * Return a 64-bit aligned version of the array.
   * @template {import('../types.js').TypedArray} T
   * @param {T} array The array.
   * @param {number} length The current array length.
   * @returns {T} The aligned array.
   */
  function align(array, length = array.length) {
    const alignedLength = align64(length, array.BYTES_PER_ELEMENT);
    return array.length > alignedLength ? /** @type {T} */ (array.subarray(0, alignedLength))
      : array.length < alignedLength ? resize(array, alignedLength)
      : array;
  }

  /**
   * Resize a typed array to exactly the specified length.
   * @template {import('../types.js').TypedArray} T
   * @param {T} array The array.
   * @param {number} newLength The new length.
   * @param {number} [offset] The offset at which to copy the old array.
   * @returns {T} The resized array.
   */
  function resize(array, newLength, offset = 0) {
    // @ts-ignore
    const newArray = new array.constructor(newLength);
    newArray.set(array, offset);
    return newArray;
  }

  /**
   * Grow a typed array to accommdate a minimum index. The array size is
   * doubled until it exceeds the minimum index.
   * @template {import('../types.js').TypedArray} T
   * @param {T} array The array.
   * @param {number} index The minimum index.
   * @param {boolean} [shift] Flag to shift copied bytes to back of array.
   * @returns {T} The resized array.
   */
  function grow(array, index, shift) {
    while (array.length <= index) {
      array = resize(array, array.length << 1, shift ? array.length : 0);
    }
    return array;
  }

  /**
   * Check if a value is a Date instance
   * @param {*} value The value to check.
   * @returns {value is Date} True if value is a Date, false otherwise.
   */
  function isDate(value) {
    return value instanceof Date;
  }

  /**
   * Check if a value is iterable.
   * @param {*} value The value to check.
   * @returns {value is Iterable} True if value is iterable, false otherwise.
   */
  function isIterable(value) {
    return typeof value[Symbol.iterator] === 'function';
  }

  /**
   * Return the input value if it passes a test.
   * Otherwise throw an error using the given message generator.
   * @template T
   * @param {T} value he value to check.
   * @param {(value: T) => boolean} test The test function.
   * @param {(value: *) => string} message Message generator.
   * @returns {T} The input value.
   * @throws if the value does not pass the test
   */
  function check$2(value, test, message) {
    if (test(value)) return value;
    throw new Error(message(value));
  }

  /**
   * Return the input value if it exists in the provided set.
   * Otherwise throw an error using the given message generator.
   * @template T
   * @param {T} value The value to check.
   * @param {T[] | Record<string,T>} set The set of valid values.
   * @param {(value: *) => string} [message] Message generator.
   * @returns {T} The input value.
   * @throws if the value is not included in the set
   */
  function checkOneOf(value, set, message) {
    set = Array.isArray(set) ? set : Object.values(set);
    return check$2(
      value,
      (value) => set.includes(value),
      message ?? (() => `${value} must be one of ${set}`)
    );
  }

  /**
   * Return the first object key that pairs with the given value.
   * @param {Record<string,any>} object The object to search.
   * @param {any} value The value to lookup.
   * @returns {string} The first matching key, or '<Unknown>' if not found.
   */
  function keyFor(object, value) {
    for (const [key, val] of Object.entries(object)) {
      if (val === value) return key;
    }
    return '<Unknown>';
  }

  /**
   * @typedef {import('./types.js').Field | import('./types.js').DataType} FieldInput
   */

  const invalidDataType = (typeId) =>
    `Unsupported data type: "${keyFor(Type, typeId)}" (id ${typeId})`;

  /**
   * Return a new field instance for use in a schema or type definition. A field
   * represents a field name, data type, and additional metadata. Fields are used
   * to represent child types within nested types like List, Struct, and Union.
   * @param {string} name The field name.
   * @param {import('./types.js').DataType} type The field data type.
   * @param {boolean} [nullable=true] Flag indicating if the field is nullable
   *  (default `true`).
   * @param {Map<string,string>|null} [metadata=null] Custom field metadata
   *  annotations (default `null`).
   * @returns {import('./types.js').Field} The field instance.
   */
  const field = (name, type, nullable = true, metadata = null) => ({
    name,
    type,
    nullable,
    metadata
  });

  /**
   * Checks if a value is a field instance.
   * @param {any} value
   * @returns {value is import('./types.js').Field}
   */
  function isField(value) {
    return Object.hasOwn(value, 'name') && isDataType(value.type)
  }

  /**
   * Checks if a value is a data type instance.
   * @param {any} value
   * @returns {value is import('./types.js').DataType}
   */
  function isDataType(value) {
    return typeof value?.typeId === 'number';
  }

  /**
   * Return a field instance from a field or data type input.
   * @param {FieldInput} value
   *  The value to map to a field.
   * @param {string} [defaultName] The default field name.
   * @param {boolean} [defaultNullable=true] The default nullable value.
   * @returns {import('./types.js').Field} The field instance.
   */
  function asField(value, defaultName = '', defaultNullable = true) {
    return isField(value)
      ? value
      : field(
          defaultName,
          check$2(value, isDataType, () => `Data type expected.`),
          defaultNullable
        );
  }

  /////

  /**
   * Return a basic type with only a type id.
   * @template {typeof Type[keyof typeof Type]} T
   * @param {T} typeId The type id.
   */
  const basicType = (typeId) => ({ typeId });

  /**
   * Return a Dictionary data type instance.  A dictionary type consists of a
   * dictionary of values (which may be of any type) and corresponding integer
   * indices that reference those values. If values are repeated, a dictionary
   * encoding can provide substantial space savings. In the IPC format,
   * dictionary indices reside alongside other columns in a record batch, while
   * dictionary values are written to special dictionary batches, linked by a
   * unique dictionary *id*.
   * @param {import('./types.js').DataType} type The data type of dictionary
   *  values.
   * @param {import('./types.js').IntType} [indexType] The data type of
   *  dictionary indices. Must be an integer type (default `int32`).
   * @param {boolean} [ordered=false] Indicates if dictionary values are
   *  ordered (default `false`).
   * @param {number} [id=-1] The dictionary id. The default value (-1) indicates
   *  the dictionary applies to a single column only. Provide an explicit id in
   *  order to reuse a dictionary across columns when building, in which case
   *  different dictionaries *must* have different unique ids. All dictionary
   *  ids are later resolved (possibly to new values) upon IPC encoding.
   * @returns {import('./types.js').DictionaryType}
   */
  const dictionary$1 = (type, indexType, ordered = false, id = -1) => ({
    typeId: Type.Dictionary,
    id,
    dictionary: type,
    indices: indexType || int32(),
    ordered
  });

  /**
   * Return a Null data type instance. Null data requires no storage and all
   * extracted values are `null`.
   * @returns {import('./types.js').NullType} The null data type.
   */
  const nullType = () => basicType(Type.Null);

  /**
   * Return an Int data type instance.
   * @param {import('./types.js').IntBitWidth} [bitWidth=32] The integer bit width.
   *  One of `8`, `16`, `32` (default), or `64`.
   * @param {boolean} [signed=true] Flag for signed or unsigned integers
   *  (default `true`).
   * @returns {import('./types.js').IntType} The integer data type.
   */
  const int = (bitWidth = 32, signed = true) => ({
    typeId: Type.Int,
    bitWidth: checkOneOf(bitWidth, [8, 16, 32, 64]),
    signed,
    values: intArrayType(bitWidth, signed)
  });
  /**
   * Return an Int data type instance for 8 bit signed integers.
   * @returns {import('./types.js').IntType} The integer data type.
   */
  const int8 = () => int(8);
  /**
   * Return an Int data type instance for 16 bit signed integers.
   * @returns {import('./types.js').IntType} The integer data type.
   */
  const int16 = () => int(16);
  /**
   * Return an Int data type instance for 32 bit signed integers.
   * @returns {import('./types.js').IntType} The integer data type.
   */
  const int32 = () => int(32);
  /**
   * Return an Int data type instance for 64 bit signed integers.
   * @returns {import('./types.js').IntType} The integer data type.
   */
  const int64 = () => int(64);
  /**
   * Return an Int data type instance for 8 bit unsigned integers.
   * @returns {import('./types.js').IntType} The integer data type.
   */
  const uint8 = () => int(8, false);
  /**
   * Return an Int data type instance for 16 bit unsigned integers.
   * @returns {import('./types.js').IntType} The integer data type.
   */
  const uint16 = () => int(16, false);
  /**
   * Return an Int data type instance for 32 bit unsigned integers.
   * @returns {import('./types.js').IntType} The integer data type.
   */
  const uint32 = () => int(32, false);
  /**
   * Return an Int data type instance for 64 bit unsigned integers.
   * @returns {import('./types.js').IntType} The integer data type.
   */
  const uint64 = () => int(64, false);

  /**
   * Return a Float data type instance for floating point numbers.
   * @param {import('./types.js').Precision_} [precision=2] The floating point
   *  precision. One of `Precision.HALF` (16-bit), `Precision.SINGLE` (32-bit)
   *  or `Precision.DOUBLE` (64-bit, default).
   * @returns {import('./types.js').FloatType} The floating point data type.
   */
  const float = (precision = 2) => ({
    typeId: Type.Float,
    precision: checkOneOf(precision, Precision),
    values: [uint16Array, float32Array, float64Array][precision]
  });
  /**
   * Return a Float data type instance for single-precision (32 bit) numbers.
   * @returns {import('./types.js').FloatType} The floating point data type.
   */
  const float32 = () => float(Precision.SINGLE);
  /**
   * Return a Float data type instance for double-precision (64 bit) numbers.
   * @returns {import('./types.js').FloatType} The floating point data type.
   */
  const float64 = () => float(Precision.DOUBLE);

  /**
   * Return a Binary data type instance for variably-sized opaque binary data
   * with 32-bit offsets.
   * @returns {import('./types.js').BinaryType} The binary data type.
   */
  const binary = () => ({
    typeId: Type.Binary,
    offsets: int32Array
  });

  /**
   * Return a Utf8 data type instance for Unicode string data.
   * [UTF-8](https://en.wikipedia.org/wiki/UTF-8) code points are stored as
   * binary data.
   * @returns {import('./types.js').Utf8Type} The utf8 data type.
   */
  const utf8 = () => ({
    typeId: Type.Utf8,
    offsets: int32Array
  });

  /**
   * Return a Bool data type instance. Bool values are stored compactly in
   * bitmaps with eight values per byte.
   * @returns {import('./types.js').BoolType} The bool data type.
   */
  const bool = () => basicType(Type.Bool);

  /**
   * Return a Decimal data type instance. Decimal values are represented as 128
   * or 256 bit integers in two's complement. Decimals are fixed point numbers
   * with a set *precision* (total number of decimal digits) and *scale*
   * (number of fractional digits). For example, the number `35.42` can be
   * represented as `3542` with *precision* ≥ 4 and *scale* = 2.
   * @param {number} precision The decimal precision: the total number of
   *  decimal digits that can be represented.
   * @param {number} scale The number of fractional digits, beyond the
   *  decimal point.
   * @param {128 | 256} [bitWidth] The decimal bit width.
   *  One of 128 (default) or 256.
   * @returns {import('./types.js').DecimalType} The decimal data type.
   */
  const decimal = (precision, scale, bitWidth = 128) => ({
    typeId: Type.Decimal,
    precision,
    scale,
    bitWidth: checkOneOf(bitWidth, [128, 256]),
    values: uint64Array
  });

  /**
   * Return a Date data type instance. Date values are 32-bit or 64-bit signed
   * integers representing elapsed time since the UNIX epoch (Jan 1, 1970 UTC),
   * either in units of days (32 bits) or milliseconds (64 bits, with values
   * evenly divisible by 86400000).
   * @param {import('./types.js').DateUnit_} unit The date unit.
   *  One of `DateUnit.DAY` or `DateUnit.MILLISECOND`.
   * @returns {import('./types.js').DateType} The date data type.
   */
  const date = (unit) => ({
    typeId: Type.Date,
    unit: checkOneOf(unit, DateUnit),
    values: unit === DateUnit.DAY ? int32Array : int64Array
  });
  /**
   * Return a Date data type instance with units of days.
   * @returns {import('./types.js').DateType} The date data type.
   */
  const dateDay = () => date(DateUnit.DAY);

  /**
   * Return a Time data type instance, stored in one of four *unit*s: seconds,
   * milliseconds, microseconds or nanoseconds. The integer *bitWidth* depends
   * on the *unit* and must be 32 bits for seconds and milliseconds or 64 bits
   * for microseconds and nanoseconds. The allowed values are between 0
   * (inclusive) and 86400 (=24*60*60) seconds (exclusive), adjusted for the
   * time unit (for example, up to 86400000 exclusive for the
   * `DateUnit.MILLISECOND` unit.
   *
   * This definition doesn't allow for leap seconds. Time values from
   * measurements with leap seconds will need to be corrected when ingesting
   * into Arrow (for example by replacing the value 86400 with 86399).
   * @param {import('./types.js').TimeUnit_} unit The time unit.
   *  One of `TimeUnit.SECOND`, `TimeUnit.MILLISECOND` (default),
   *  `TimeUnit.MICROSECOND`, or `TimeUnit.NANOSECOND`.
   * @param {32 | 64} bitWidth The time bit width. One of `32` (for seconds
   *  and milliseconds) or `64` (for microseconds and nanoseconds).
   * @returns {import('./types.js').TimeType} The time data type.
   */
  const time = (unit = TimeUnit.MILLISECOND, bitWidth = 32) => ({
    typeId: Type.Time,
    unit: checkOneOf(unit, TimeUnit),
    bitWidth: checkOneOf(bitWidth, [32, 64]),
    values: bitWidth === 32 ? int32Array : int64Array
  });

  /**
   * Return a Timestamp data type instance. Timestamp values are 64-bit signed
   * integers representing an elapsed time since a fixed epoch, stored in either
   * of four units: seconds, milliseconds, microseconds or nanoseconds, and are
   * optionally annotated with a timezone. Timestamp values do not include any
   * leap seconds (in other words, all days are considered 86400 seconds long).
   * @param {import('./types.js').TimeUnit_} [unit] The time unit.
   *  One of `TimeUnit.SECOND`, `TimeUnit.MILLISECOND` (default),
   *  `TimeUnit.MICROSECOND`, or `TimeUnit.NANOSECOND`.
   * @param {string|null} [timezone=null] An optional string for the name of a
   *  timezone. If provided, the value should either be a string as used in the
   *  Olson timezone database (the "tz database" or "tzdata"), such as
   *  "America/New_York", or an absolute timezone offset of the form "+XX:XX" or
   *  "-XX:XX", such as "+07:30".Whether a timezone string is present indicates
   *  different semantics about the data.
   * @returns {import('./types.js').TimestampType} The time data type.
   */
  const timestamp = (unit = TimeUnit.MILLISECOND, timezone = null) => ({
    typeId: Type.Timestamp,
    unit: checkOneOf(unit, TimeUnit),
    timezone,
    values: int64Array
  });

  /**
   * Return an Interval type instance. Values represent calendar intervals stored
   * as integers for each date part. The supported *unit*s are year/moth,
   * day/time, and month/day/nanosecond intervals.
   *
   * `IntervalUnit.YEAR_MONTH` indicates the number of elapsed whole months,
   * stored as 32-bit signed integers.
   *
   * `IntervalUnit.DAY_TIME` indicates the number of elapsed days and
   * milliseconds (no leap seconds), stored as 2 contiguous 32-bit signed
   * integers (8-bytes in total).
   *
   * `IntervalUnit.MONTH_DAY_NANO` is a triple of the number of elapsed months,
   * days, and nanoseconds. The values are stored contiguously in 16-byte blocks.
   * Months and days are encoded as 32-bit signed integers and nanoseconds is
   * encoded as a 64-bit signed integer. Nanoseconds does not allow for leap
   * seconds. Each field is independent (e.g. there is no constraint that
   * nanoseconds have the same sign as days or that the quantity of nanoseconds
   * represents less than a day's worth of time).
   * @param {import('./types.js').IntervalUnit_} unit  The interval unit.
   *  One of `IntervalUnit.YEAR_MONTH`, `IntervalUnit.DAY_TIME`, or
   *  `IntervalUnit.MONTH_DAY_NANO` (default).
   * @returns {import('./types.js').IntervalType} The interval data type.
   */
  const interval = (unit = IntervalUnit.MONTH_DAY_NANO) => ({
    typeId: Type.Interval,
    unit: checkOneOf(unit, IntervalUnit),
    values: unit === IntervalUnit.MONTH_DAY_NANO ? undefined : int32Array
  });

  /**
   * Return a List data type instance, representing variably-sized lists
   * (arrays) with 32-bit offsets. A list has a single child data type for
   * list entries. Lists are represented using integer offsets that indicate
   * list extents within a single child array containing all list values.
   * @param {FieldInput} child The child (list item) field or data type.
   * @returns {import('./types.js').ListType} The list data type.
   */
  const list = (child) => ({
    typeId: Type.List,
    children: [ asField(child) ],
    offsets: int32Array
  });

  /**
   * Return a Struct data type instance. A struct consists of multiple named
   * child data types. Struct values are stored as parallel child batches, one
   * per child type, and extracted to standard JavaScript objects.
   * @param {import('./types.js').Field[] | Record<string, import('./types.js').DataType>} children
   *  An array of property fields, or an object mapping property names to data
   *  types. If an object, the instantiated fields are assumed to be nullable
   *  and have no metadata.
   * @returns {import('./types.js').StructType} The struct data type.
   */
  const struct = (children) => ({
    typeId: Type.Struct,
    children: Array.isArray(children) && isField(children[0])
      ? /** @type {import('./types.js').Field[]} */ (children)
      : Object.entries(children).map(([name, type]) => field(name, type))
  });

  /**
   * Return a Union type instance. A union is a complex type with parallel
   * *children* data types. Union values are stored in either a sparse
   * (`UnionMode.Sparse`) or dense (`UnionMode.Dense`) layout *mode*. In a
   * sparse layout, child types are stored in parallel arrays with the same
   * lengths, resulting in many unused, empty values. In a dense layout, child
   * types have variable lengths and an offsets array is used to index the
   * appropriate value.
   *
   * By default, ids in the type vector refer to the index in the children
   * array. Optionally, *typeIds* provide an indirection between the child
   * index and the type id. For each child, `typeIds[index]` is the id used
   * in the type vector. The *typeIdForValue* argument provides a lookup
   * function for mapping input data to the proper child type id, and is
   * required if using builder methods.
   * @param {import('./types.js').UnionMode_} mode The union mode.
   *  One of `UnionMode.Sparse` or `UnionMode.Dense`.
   * @param {FieldInput[]} children The children fields or data types.
   *  Types are mapped to nullable fields with no metadata.
   * @param {number[]} [typeIds]  Children type ids, in the same order as the
   *  children types. Type ids provide a level of indirection over children
   *  types. If not provided, the children indices are used as the type ids.
   * @param {(value: any, index: number) => number} [typeIdForValue]
   *  A function that takes an arbitrary value and a row index and returns a
   *  correponding union type id. Required by builder methods.
   * @returns {import('./types.js').UnionType} The union data type.
   */
  const union = (mode, children, typeIds, typeIdForValue) => {
    typeIds ??= children.map((v, i) => i);
    return {
      typeId: Type.Union,
      mode: checkOneOf(mode, UnionMode),
      typeIds,
      typeMap: typeIds.reduce((m, id, i) => ((m[id] = i), m), {}),
      children: children.map((v, i) => asField(v, `_${i}`)),
      typeIdForValue,
      offsets: int32Array,
    };
  };

  /**
   * Create a FixedSizeBinary data type instance for opaque binary data where
   * each entry has the same fixed size.
   * @param {number} stride The fixed size in bytes.
   * @returns {import('./types.js').FixedSizeBinaryType} The fixed size binary data type.
   */
  const fixedSizeBinary = (stride) => ({
    typeId: Type.FixedSizeBinary,
    stride
  });

  /**
   * Return a FixedSizeList type instance for list (array) data where every list
   * has the same fixed size. A list has a single child data type for list
   * entries. Fixed size lists are represented as a single child array containing
   * all list values, indexed using the known stride.
   * @param {FieldInput} child The list item data type.
   * @param {number} stride The fixed list size.
   * @returns {import('./types.js').FixedSizeListType} The fixed size list data type.
   */
  const fixedSizeList = (child, stride) => ({
    typeId: Type.FixedSizeList,
    stride,
    children: [ asField(child) ]
  });

  /**
   * Internal method to create a Map type instance.
   * @param {boolean} keysSorted Flag indicating if the map keys are sorted.
   * @param {import('./types.js').Field} child The child fields.
   * @returns {import('./types.js').MapType} The map data type.
   */
  const mapType = (keysSorted, child) => ({
    typeId: Type.Map,
    keysSorted,
    children: [child],
    offsets: int32Array
  });

  /**
   * Return a Duration data type instance. Durations represent an absolute length
   * of time unrelated to any calendar artifacts. The resolution defaults to
   * millisecond, but can be any of the other `TimeUnit` values. This type is
   * always represented as a 64-bit integer.
   * @param {import('./types.js').TimeUnit_} unit
   * @returns {import('./types.js').DurationType} The duration data type.
   */
  const duration = (unit = TimeUnit.MILLISECOND) => ({
    typeId: Type.Duration,
    unit: checkOneOf(unit, TimeUnit),
    values: int64Array
  });

  /**
   * Return a LargeBinary data type instance for variably-sized opaque binary
   * data with 64-bit offsets, allowing representation of extremely large data
   * values.
   * @returns {import('./types.js').LargeBinaryType} The large binary data type.
   */
  const largeBinary = () => ({
    typeId: Type.LargeBinary,
    offsets: int64Array
  });

  /**
   * Return a LargeUtf8 data type instance for Unicode string data of variable
   * length with 64-bit offsets, allowing representation of extremely large data
   * values. [UTF-8](https://en.wikipedia.org/wiki/UTF-8) code points are stored
   * as binary data.
   * @returns {import('./types.js').LargeUtf8Type} The large utf8 data type.
   */
  const largeUtf8 = () => ({
    typeId: Type.LargeUtf8,
    offsets: int64Array
  });

  /**
   * Return a LargeList data type instance, representing variably-sized lists
   * (arrays) with 64-bit offsets, allowing representation of extremely large
   * data values. A list has a single child data type for list entries. Lists
   * are represented using integer offsets that indicate list extents within a
   * single child array containing all list values.
   * @param {FieldInput} child The child (list item) field or data type.
   * @returns {import('./types.js').LargeListType} The large list data type.
   */
  const largeList = (child) => ({
    typeId: Type.LargeList,
    children: [ asField(child) ],
    offsets: int64Array
  });

  /**
   * Return a RunEndEncoded data type instance, which compresses data by
   * representing consecutive repeated values as a run. This data type uses two
   * child arrays, `run_ends` and `values`. The `run_ends` child array must be
   * a 16, 32, or 64 bit integer array which encodes the indices at which the
   * run with the value in each corresponding index in the values child array
   * ends. Like list and struct types, the `values` array can be of any type.
   * @param {FieldInput} runsField The run-ends field or data type.
   * @param {FieldInput} valuesField The values field or data type.
   * @returns {import('./types.js').RunEndEncodedType} The large list data type.
   */
  const runEndEncoded = (runsField, valuesField) => ({
    typeId: Type.RunEndEncoded,
    children: [
      check$2(
        asField(runsField, 'run_ends'),
        (field) => field.type.typeId === Type.Int,
        () => 'Run-ends must have an integer type.'
      ),
      asField(valuesField, 'values')
    ]
  });

  /**
   * Return a ListView data type instance, representing variably-sized lists
   * (arrays) with 32-bit offsets. ListView data represents the same logical
   * types that List can, but contains both offsets and sizes allowing for
   * writes in any order and sharing of child values among list values.
   *
   * Flechette can encode and decode ListView data; however, Flechette does not
   * currently support building ListView columns from JavaScript values.
   * @param {FieldInput} child The child (list item) field or data type.
   * @returns {import('./types.js').ListViewType} The list view data type.
   */
  const listView = (child) => ({
    typeId: Type.ListView,
    children: [ asField(child, 'value') ],
    offsets: int32Array
  });

  /**
   * Return a LargeListView data type instance, representing variably-sized lists
   * (arrays) with 64-bit offsets, allowing representation of extremely large
   * data values. LargeListView data represents the same logical types that
   * LargeList can, but contains both offsets and sizes allowing for writes
   * in any order and sharing of child values among list values.
   *
   * Flechette can encode and decode LargeListView data; however, Flechette does
   * not currently support building LargeListView columns from JavaScript values.
   * @param {FieldInput} child The child (list item) field or data type.
   * @returns {import('./types.js').LargeListViewType} The large list view data type.
   */
  const largeListView = (child) => ({
    typeId: Type.LargeListView,
    children: [ asField(child, 'value') ],
    offsets: int64Array
  });

  // typed arrays over a shared buffer to aid binary conversion
  const f64 = new float64Array(2);
  const buf = f64.buffer;
  const i64 = new int64Array(buf);
  const u32 = new uint32Array(buf);
  const i32 = new int32Array(buf);
  const u8 = new uint8Array(buf);

  /**
   * Return a value unchanged.
   * @template T
   * @param {T} value The value.
   * @returns {T} The value.
   */
  function identity$1(value) {
    return value;
  }

  /**
   * Return a value coerced to a BigInt.
   * @param {*} value The value.
   * @returns {bigint} The BigInt value.
   */
  function toBigInt(value) {
    return BigInt(value);
  }

  /**
   * Return an offset conversion method for the given data type.
   * @param {{ offsets: import('../types.js').TypedArray}} type The array type.
   */
  function toOffset(type) {
    return isInt64ArrayType(type) ? toBigInt : identity$1;
  }

  /**
   * Return the number of days from a millisecond timestamp.
   * @param {number} value The millisecond timestamp.
   * @returns {number} The number of days.
   */
  function toDateDay(value) {
    return (value / 864e5) | 0;
  }

  /**
   * Return a timestamp conversion method for the given time unit.
   * @param {import('../types.js').TimeUnit_} unit The time unit.
   * @returns {(value: number) => bigint} The conversion method.
   */
  function toTimestamp(unit) {
    return unit === TimeUnit.SECOND ? value => toBigInt(value / 1e3)
      : unit === TimeUnit.MILLISECOND ? toBigInt
      : unit === TimeUnit.MICROSECOND ? value => toBigInt(value * 1e3)
      : value => toBigInt(value * 1e6);
  }

  /**
   * Write month/day/nanosecond interval to a byte buffer.
   * @param {Array | Float64Array} interval The interval data.
   * @returns {Uint8Array} A byte buffer with the interval data.
   *  The returned buffer is reused across calls, and so should be
   *  copied to a target buffer immediately.
   */
  function toMonthDayNanoBytes([m, d, n]) {
    i32[0] = m;
    i32[1] = d;
    i64[1] = toBigInt(n);
    return u8;
  }

  /**
   * Coerce a bigint value to a number. Throws an error if the bigint value
   * lies outside the range of what a number can precisely represent.
   * @param {bigint} value The value to check and possibly convert.
   * @returns {number} The converted number value.
   */
  function toNumber(value) {
    if (value > Number.MAX_SAFE_INTEGER || value < Number.MIN_SAFE_INTEGER) {
      throw Error(`BigInt exceeds integer number representation: ${value}`);
    }
    return Number(value);
  }

  /**
   * Divide one BigInt value by another, and return the result as a number.
   * The division may involve unsafe integers and a loss of precision.
   * @param {bigint} num The numerator.
   * @param {bigint} div The divisor.
   * @returns {number} The result of the division as a floating point number.
   */
  function divide(num, div) {
    return Number(num / div) + Number(num % div) / Number(div);
  }

  /**
   * Convert a floating point number or bigint to decimal bytes.
   * @param {number|bigint} value The number to encode. If a bigint, we assume
   *  it already represents the decimal in integer form with the correct scale.
   *  Otherwise, we assume a float that requires scaled integer conversion.
   * @param {BigUint64Array} buf The uint64 array to write to.
   * @param {number} offset The starting index offset into the array.
   * @param {number} stride The stride of an encoded decimal, in 64-bit steps.
   * @param {number} scale The scale mapping fractional digits to integers.
   */
  function toDecimal(value, buf, offset, stride, scale) {
    const v = typeof value === 'bigint'
      ? value
      : toBigInt(Math.trunc(value * scale));
    // assignment into uint64array performs needed truncation for us
    buf[offset] = v;
    buf[offset + 1] = (v >> 64n);
    if (stride > 2) {
      buf[offset + 2] = (v >> 128n);
      buf[offset + 3] = (v >> 192n);
    }
  }

  // helper method to extract uint64 values from bigints
  const asUint64 = v => BigInt.asUintN(64, v);

  /**
   * Convert a 128-bit decimal value to a bigint.
   * @param {BigUint64Array} buf The uint64 array containing the decimal bytes.
   * @param {number} offset The starting index offset into the array.
   * @returns {bigint} The converted decimal as a bigint, such that all
   *  fractional digits are scaled up to integers (for example, 1.12 -> 112).
   */
  function fromDecimal128(buf, offset) {
    const i = offset << 1;
    let x;
    if (BigInt.asIntN(64, buf[i + 1]) < 0) {
      x = asUint64(~buf[i]) | (asUint64(~buf[i + 1]) << 64n);
      x = -(x + 1n);
    } else {
      x = buf[i] | (buf[i + 1] << 64n);
    }
    return x;
  }

  /**
   * Convert a 256-bit decimal value to a bigint.
   * @param {BigUint64Array} buf The uint64 array containing the decimal bytes.
   * @param {number} offset The starting index offset into the array.
   * @returns {bigint} The converted decimal as a bigint, such that all
   *  fractional digits are scaled up to integers (for example, 1.12 -> 112).
   */
  function fromDecimal256(buf, offset) {
    const i = offset << 2;
    let x;
    if (BigInt.asIntN(64, buf[i + 3]) < 0) {
      x = asUint64(~buf[i])
        | (asUint64(~buf[i + 1]) << 64n)
        | (asUint64(~buf[i + 2]) << 128n)
        | (asUint64(~buf[i + 3]) << 192n);
      x = -(x + 1n);
    } else {
      x = buf[i]
        | (buf[i + 1] << 64n)
        | (buf[i + 2] << 128n)
        | (buf[i + 3] << 192n);
    }
    return x;
  }

  /**
   * Convert a number to a 16-bit float as integer bytes..
   * Inspired by numpy's `npy_double_to_half`:
   * https://github.com/numpy/numpy/blob/5a5987291dc95376bb098be8d8e5391e89e77a2c/numpy/core/src/npymath/halffloat.c#L43
   * Adapted from https://github.com/apache/arrow/blob/main/js/src/util/math.ts
   * @param {number} value The 64-bit floating point number to convert.
   * @returns {number} The converted 16-bit integer.
   */
  function toFloat16(value) {
    if (value !== value) return 0x7E00; // NaN
    f64[0] = value;

    // Magic numbers:
    // 0x80000000 = 10000000 00000000 00000000 00000000 -- masks the 32nd bit
    // 0x7ff00000 = 01111111 11110000 00000000 00000000 -- masks the 21st-31st bits
    // 0x000fffff = 00000000 00001111 11111111 11111111 -- masks the 1st-20th bit
    const sign = (u32[1] & 0x80000000) >> 16 & 0xFFFF;
    let expo = (u32[1] & 0x7FF00000), sigf = 0x0000;

    if (expo >= 0x40F00000) {
      //
      // If exponent overflowed, the float16 is either NaN or Infinity.
      // Rules to propagate the sign bit: mantissa > 0 ? NaN : +/-Infinity
      //
      // Magic numbers:
      // 0x40F00000 = 01000000 11110000 00000000 00000000 -- 6-bit exponent overflow
      // 0x7C000000 = 01111100 00000000 00000000 00000000 -- masks the 27th-31st bits
      //
      // returns:
      // qNaN, aka 32256 decimal, 0x7E00 hex, or 01111110 00000000 binary
      // sNaN, aka 32000 decimal, 0x7D00 hex, or 01111101 00000000 binary
      // +inf, aka 31744 decimal, 0x7C00 hex, or 01111100 00000000 binary
      // -inf, aka 64512 decimal, 0xFC00 hex, or 11111100 00000000 binary
      //
      // If mantissa is greater than 23 bits, set to +Infinity like numpy
      if (u32[0] > 0) {
        expo = 0x7C00;
      } else {
        expo = (expo & 0x7C000000) >> 16;
        sigf = (u32[1] & 0x000FFFFF) >> 10;
      }
    } else if (expo <= 0x3F000000) {
      //
      // If exponent underflowed, the float is either signed zero or subnormal.
      //
      // Magic numbers:
      // 0x3F000000 = 00111111 00000000 00000000 00000000 -- 6-bit exponent underflow
      //
      sigf = 0x100000 + (u32[1] & 0x000FFFFF);
      sigf = 0x100000 + (sigf << ((expo >> 20) - 998)) >> 21;
      expo = 0;
    } else {
      //
      // No overflow or underflow, rebase the exponent and round the mantissa
      // Magic numbers:
      // 0x200 = 00000010 00000000 -- masks off the 10th bit
      //
      // Ensure the first mantissa bit (the 10th one) is 1 and round
      expo = (expo - 0x3F000000) >> 10;
      sigf = ((u32[1] & 0x000FFFFF) + 0x200) >> 10;
    }
    return sign | expo | sigf & 0xFFFF;
  }

  const textDecoder = new TextDecoder('utf-8');
  const textEncoder = new TextEncoder();

  /**
   * Return a UTF-8 string decoded from a byte buffer.
   * @param {Uint8Array} buf The byte buffer.
   * @returns {string} The decoded string.
   */
  function decodeUtf8(buf) {
    return textDecoder.decode(buf);
  }

  /**
   * Return a byte buffer encoded from a UTF-8 string.
   * @param {string } str The string to encode.
   * @returns {Uint8Array} The encoded byte buffer.
   */
  function encodeUtf8(str) {
    return textEncoder.encode(str);
  }

  /**
   * Return a string-coercible key value that uniquely identifies a value.
   * @param {*} value The input value.
   * @returns {string} The key string.
   */
  function keyString(value) {
    const val = typeof value !== 'object' || !value ? (value ?? null)
      : isDate(value) ? +value
      // @ts-ignore
      : isArray(value) ? `[${value.map(keyString)}]`
      : objectKey(value);
    return `${val}`;
  }

  function objectKey(value) {
    let s = '';
    let i = -1;
    for (const k in value) {
      if (++i > 0) s += ',';
      s += `"${k}":${keyString(value[k])}`;
    }
    return `{${s}}`;
  }

  /** The size in bytes of a 32-bit integer. */
  const SIZEOF_INT = 4;

  /** The size in bytes of a 16-bit integer. */
  const SIZEOF_SHORT = 2;

  /**
   * Return a boolean for a single bit in a bitmap.
   * @param {Uint8Array} bitmap The bitmap.
   * @param {number} index The bit index to read.
   * @returns {boolean} The boolean bitmap value.
   */
  function decodeBit(bitmap, index) {
    return (bitmap[index >> 3] & 1 << (index % 8)) !== 0;
  }

  /**
   * Lookup helper for flatbuffer object (table) entries.
   * @param {Uint8Array} buf The byte buffer.
   * @param {number} index The base index of the object.
   */
  function readObject(buf, index) {
    const pos = index + readInt32(buf, index);
    const vtable = pos - readInt32(buf, pos);
    const size = readInt16(buf, vtable);
    /**
     * Retrieve a value from a flatbuffer table layout.
     * @template T
     * @param {number} index The table entry index.
     * @param {(buf: Uint8Array, offset: number) => T} read Read function to invoke.
     * @param {T} [fallback=null] The default fallback value.
     * @returns {T}
     */
    return (index, read, fallback = null) => {
      if (index < size) {
        const off = readInt16(buf, vtable + index);
        if (off) return read(buf, pos + off);
      }
      return fallback;
    };
  }

  /**
   * Return a buffer offset value.
   * @param {Uint8Array} buf
   * @param {number} offset
   * @returns {number}
   */
  function readOffset(buf, offset) {
    return offset;
  }

  /**
   * Return a boolean value.
   * @param {Uint8Array} buf
   * @param {number} offset
   * @returns {boolean}
   */
  function readBoolean(buf, offset) {
    return !!readInt8(buf, offset);
  }

  /**
   * Return a signed 8-bit integer value.
   * @param {Uint8Array} buf
   * @param {number} offset
   * @returns {number}
   */
  function readInt8(buf, offset) {
    return readUint8(buf, offset) << 24 >> 24;
  }

  /**
   * Return an unsigned 8-bit integer value.
   * @param {Uint8Array} buf
   * @param {number} offset
   * @returns {number}
   */
  function readUint8(buf, offset) {
    return buf[offset];
  }

  /**
   * Return a signed 16-bit integer value.
   * @param {Uint8Array} buf
   * @param {number} offset
   * @returns {number}
   */
  function readInt16(buf, offset) {
    return readUint16(buf, offset) << 16 >> 16;
  }

  /**
   * Return an unsigned 16-bit integer value.
   * @param {Uint8Array} buf
   * @param {number} offset
   * @returns {number}
   */
  function readUint16(buf, offset) {
    return buf[offset] | buf[offset + 1] << 8;
  }

  /**
   * Return a signed 32-bit integer value.
   * @param {Uint8Array} buf
   * @param {number} offset
   * @returns {number}
   */
  function readInt32(buf, offset) {
    return buf[offset]
      | buf[offset + 1] << 8
      | buf[offset + 2] << 16
      | buf[offset + 3] << 24;
  }

  /**
   * Return an unsigned 32-bit integer value.
   * @param {Uint8Array} buf
   * @param {number} offset
   * @returns {number}
   */
  function readUint32(buf, offset) {
    return readInt32(buf, offset) >>> 0;
  }

  /**
   * Return a signed 64-bit integer value coerced to a JS number.
   * Throws an error if the value exceeds what a JS number can represent.
   * @param {Uint8Array} buf
   * @param {number} offset
   * @returns {number}
   */
  function readInt64(buf, offset) {
    return toNumber(BigInt.asIntN(
      64,
      BigInt(readUint32(buf, offset)) +
        (BigInt(readUint32(buf, offset + SIZEOF_INT)) << 32n)
    ));
  }

  /**
   * Create a JavaScript string from UTF-8 data stored inside the FlatBuffer.
   * This allocates a new string and converts to wide chars upon each access.
   * @param {Uint8Array} buf The byte buffer.
   * @param {number} index The index of the string entry.
   * @returns {string} The decoded string.
   */
  function readString(buf, index) {
    let offset = index + readInt32(buf, index); // get the string offset
    const length = readInt32(buf, offset);  // get the string length
    offset += SIZEOF_INT; // skip length value
    return decodeUtf8(buf.subarray(offset, offset + length));
  }

  /**
   * Extract a flatbuffer vector to an array.
   * @template T
   * @param {Uint8Array} buf The byte buffer.
   * @param {number} offset The offset location of the vector.
   * @param {number} stride The stride between vector entries.
   * @param {(buf: Uint8Array, pos: number) => T} extract Vector entry extraction function.
   * @returns {T[]} The extracted vector entries.
   */
  function readVector(buf, offset, stride, extract) {
    if (!offset) return [];

    // get base position by adding offset delta
    const base = offset + readInt32(buf, offset);

    // read vector size, extract entries
    return Array.from(
      { length: readInt32(buf, base) },
      (_, i) => extract(buf, base + SIZEOF_INT + i * stride)
    );
  }

  const RowIndex = Symbol('rowIndex');

  /**
   * Returns a row proxy object factory. The resulting method takes a
   * batch-level row index as input and returns an object that proxies
   * access to underlying batches.
   * @param {string[]} names The column (property) names
   * @param {import('../batch.js').Batch[]} batches The value batches.
   * @returns {(index: number) => Record<string, any>}
   */
  function proxyFactory(names, batches) {
    class RowObject {
      /**
       * Create a new proxy row object representing a struct or table row.
       * @param {number} index The record batch row index.
       */
      constructor(index) {
        this[RowIndex] = index;
      }

      /**
       * Return a JSON-compatible object representation.
       */
      toJSON() {
        return structObject(names, batches, this[RowIndex]);
      }
    }
    // prototype for row proxy objects
    const proto = RowObject.prototype;

    for (let i = 0; i < names.length; ++i) {
      // skip duplicated column names
      if (Object.hasOwn(proto, names[i])) continue;

      // add a getter method for the current batch
      const batch = batches[i];
      Object.defineProperty(proto, names[i], {
        get() { return batch.at(this[RowIndex]); },
        enumerable: true
      });
    }

    return index => new RowObject(index);
  }

  /**
   * Returns a row object factory. The resulting method takes a
   * batch-level row index as input and returns an object whose property
   * values have been extracted from the batches.
   * @param {string[]} names The column (property) names
   * @param {import('../batch.js').Batch[]} batches The value batches.
   * @returns {(index: number) => Record<string, any>}
   */
  function objectFactory(names, batches) {
    return index => structObject(names, batches, index);
  }

  /**
   * Return a vanilla object representing a struct (row object) type.
   * @param {string[]} names The column (property) names
   * @param {import('../batch.js').Batch[]} batches The value batches.
   * @param {number} index The record batch row index.
   * @returns {Record<string, any>}
   */
  function structObject(names, batches, index) {
    const obj = {};
    for (let i = 0; i < names.length; ++i) {
      obj[names[i]] = batches[i].at(index);
    }
    return obj;
  }

  /**
   * Check if the input is a batch that supports direct access to
   * binary data in the form of typed arrays.
   * @param {Batch<any>?} batch The data batch to check.
   * @returns {boolean} True if a direct batch, false otherwise.
   */
  function isDirectBatch(batch) {
    return batch instanceof DirectBatch;
  }

  /**
   * Column values from a single record batch.
   * A column may contain multiple batches.
   * @template T
   */
  class Batch {
    /**
     * The array type to use when extracting data from the batch.
     * A null value indicates that the array type should match
     * the type of the batch's values array.
     * @type {ArrayConstructor | import('./types.js').TypedArrayConstructor | null}
     */
    static ArrayType = null;

    /**
     * Create a new column batch.
     * @param {object} options
     * @param {number} options.length The length of the batch
     * @param {number} options.nullCount The null value count
     * @param {import('./types.js').DataType} options.type The data type.
     * @param {Uint8Array} [options.validity] Validity bitmap buffer
     * @param {import('./types.js').TypedArray} [options.values] Values buffer
     * @param {import('./types.js').OffsetArray} [options.offsets] Offsets buffer
     * @param {import('./types.js').OffsetArray} [options.sizes] Sizes buffer
     * @param {Batch[]} [options.children] Children batches
     */
    constructor({
      length,
      nullCount,
      type,
      validity,
      values,
      offsets,
      sizes,
      children
    }) {
      this.length = length;
      this.nullCount = nullCount;
      this.type = type;
      this.validity = validity;
      this.values = values;
      this.offsets = offsets;
      this.sizes = sizes;
      this.children = children;

      // optimize access if this batch has no null values
      // some types (like union) may have null values in
      // child batches, but no top-level validity buffer
      if (!nullCount || !this.validity) {
        /** @type {(index: number) => T | null} */
        this.at = index => this.value(index);
      }
    }

    /**
     * Provide an informative object string tag.
     */
    get [Symbol.toStringTag]() {
      return 'Batch';
    }

    /**
     * Return the value at the given index.
     * @param {number} index The value index.
     * @returns {T | null} The value.
     */
    at(index) {
      return this.isValid(index) ? this.value(index) : null;
    }

    /**
     * Check if a value at the given index is valid (non-null).
     * @param {number} index The value index.
     * @returns {boolean} True if valid, false otherwise.
     */
    isValid(index) {
      return decodeBit(this.validity, index);
    }

    /**
     * Return the value at the given index. This method does not check the
     * validity bitmap and is intended primarily for internal use. In most
     * cases, callers should use the `at()` method instead.
     * @param {number} index The value index
     * @returns {T} The value, ignoring the validity bitmap.
     */
    value(index) {
      return /** @type {T} */ (this.values[index]);
    }

    /**
     * Extract an array of values within the given index range. Unlike
     * Array.slice, all arguments are required and may not be negative indices.
     * @param {number} start The starting index, inclusive
     * @param {number} end The ending index, exclusive
     * @returns {import('./types.js').ValueArray<T?>} The slice of values
     */
    slice(start, end) {
      const n = end - start;
      const values = Array(n);
      for (let i = 0; i < n; ++i) {
        values[i] = this.at(start + i);
      }
      return values;
    }

    /**
     * Return an iterator over the values in this batch.
     * @returns {Iterator<T?>}
     */
    *[Symbol.iterator]() {
      for (let i = 0; i < this.length; ++i) {
        yield this.at(i);
      }
    }
  }

  /**
   * A batch whose value buffer can be used directly, without transformation.
   * @template T
   * @extends {Batch<T>}
   */
  class DirectBatch extends Batch {
    /**
     * Create a new column batch with direct value array access.
     * @param {object} options
     * @param {number} options.length The length of the batch
     * @param {number} options.nullCount The null value count
     * @param {import('./types.js').DataType} options.type The data type.
     * @param {Uint8Array} [options.validity] Validity bitmap buffer
     * @param {import('./types.js').TypedArray} options.values Values buffer
     */
    constructor(options) {
      super(options);
      // underlying buffers may be padded, exceeding the logical batch length
      // we trim the values array so we can safely access it directly
      const { length, values } = this;
      this.values = values.subarray(0, length);
    }

    /**
     * Extract an array of values within the given index range. Unlike
     * Array.slice, all arguments are required and may not be negative indices.
     * When feasible, a zero-copy subarray of a typed array is returned.
     * @param {number} start The starting index, inclusive
     * @param {number} end The ending index, exclusive
     * @returns {import('./types.js').ValueArray<T?>} The slice of values
     */
    slice(start, end) {
      // @ts-ignore
      return this.nullCount
        ? super.slice(start, end)
        : this.values.subarray(start, end);
    }

    /**
     * Return an iterator over the values in this batch.
     * @returns {Iterator<T?>}
     */
    [Symbol.iterator]() {
      return this.nullCount
        ? super[Symbol.iterator]()
        : /** @type {Iterator<T?>} */ (this.values[Symbol.iterator]());
    }
  }

  /**
   * A batch whose values are transformed to 64-bit numbers.
   * @extends {Batch<number>}
   */
  class NumberBatch extends Batch {
    static ArrayType = float64Array;
  }

  /**
   * A batch whose values should be returned in a standard array.
   * @template T
   * @extends {Batch<T>}
   */
  class ArrayBatch extends Batch {
    static ArrayType = Array;
  }

  /**
   * A batch of null values only.
   * @extends {ArrayBatch<null>}
   */
  class NullBatch extends ArrayBatch {
    /**
     * @param {number} index The value index
     * @returns {null}
     */
    value(index) { // eslint-disable-line no-unused-vars
      return null;
    }
  }

  /**
   * A batch that coerces BigInt values to 64-bit numbers.
   * * @extends {NumberBatch}
   */
  class Int64Batch extends NumberBatch {
    /**
     * @param {number} index The value index
     */
    value(index) {
      return toNumber(/** @type {bigint} */ (this.values[index]));
    }
  }

  /**
   * A batch of 16-bit floating point numbers, accessed as unsigned
   * 16-bit ints and transformed to 64-bit numbers.
   */
  class Float16Batch extends NumberBatch {
    /**
     * @param {number} index The value index
     */
    value(index) {
      const v = /** @type {number} */ (this.values[index]);
      const expo = (v & 0x7C00) >> 10;
      const sigf = (v & 0x03FF) / 1024;
      const sign = (-1) ** ((v & 0x8000) >> 15);
      switch (expo) {
        case 0x1F: return sign * (sigf ? Number.NaN : 1 / 0);
        case 0x00: return sign * (sigf ? 6.103515625e-5 * sigf : 0);
      }
      return sign * (2 ** (expo - 15)) * (1 + sigf);
    }
  }

  /**
   * A batch of boolean values stored as a bitmap.
   * @extends {ArrayBatch<boolean>}
   */
  class BoolBatch extends ArrayBatch {
    /**
     * @param {number} index The value index
     */
    value(index) {
      return decodeBit(/** @type {Uint8Array} */ (this.values), index);
    }
  }

  /**
   * An abstract class for a batch of 128- or 256-bit decimal numbers,
   * accessed in strided BigUint64Arrays.
   * @template T
   * @extends {Batch<T>}
   */
  class DecimalBatch extends Batch {
    constructor(options) {
      super(options);
      const { bitWidth, scale } = /** @type {import('./types.js').DecimalType} */ (this.type);
      this.decimal = bitWidth === 128 ? fromDecimal128 : fromDecimal256;
      this.scale = 10n ** BigInt(scale);
    }
  }

  /**
   * A batch of 128- or 256-bit decimal numbers, returned as converted
   * 64-bit numbers. The number coercion may be lossy if the decimal
   * precision can not be represented in a 64-bit floating point format.
   * @extends {DecimalBatch<number>}
   */
  class DecimalNumberBatch extends DecimalBatch {
    static ArrayType = float64Array;
    /**
     * @param {number} index The value index
     */
    value(index) {
      return divide(
        this.decimal(/** @type {BigUint64Array} */ (this.values), index),
        this.scale
      );
    }
  }

  /**
   * A batch of 128- or 256-bit decimal numbers, returned as scaled
   * bigint values, such that all fractional digits have been shifted
   * to integer places by the decimal type scale factor.
   * @extends {DecimalBatch<bigint>}
   */
  class DecimalBigIntBatch extends DecimalBatch {
    static ArrayType = Array;
    /**
     * @param {number} index The value index
     */
    value(index) {
      return this.decimal(/** @type {BigUint64Array} */ (this.values), index);
    }
  }

  /**
   * A batch of date or timestamp values that are coerced to UNIX epoch timestamps
   * and returned as JS Date objects. This batch wraps a source batch that provides
   * timestamp values.
   * @extends {ArrayBatch<Date>}
   */
  class DateBatch extends ArrayBatch {
    /**
     * Create a new date batch.
     * @param {Batch<number>} batch A batch of timestamp values.
     */
    constructor(batch) {
      super(batch);
      this.source = batch;
    }

    /**
     * @param {number} index The value index
     */
    value(index) {
      return new Date(this.source.value(index));
    }
  }

  /**
   * A batch of dates as day counts, coerced to timestamp numbers.
   */
  class DateDayBatch extends NumberBatch {
    /**
     * @param {number} index The value index
     * @returns {number}
     */
    value(index) {
      // epoch days to milliseconds
      return 86400000 * /** @type {number} */ (this.values[index]);
    }
  }

  /**
   * A batch of dates as millisecond timestamps, coerced to numbers.
   */
  const DateDayMillisecondBatch = Int64Batch;

  /**
   * A batch of timestaps in seconds, coerced to millisecond numbers.
   */
  class TimestampSecondBatch extends Int64Batch {
    /**
     * @param {number} index The value index
     */
    value(index) {
      return super.value(index) * 1e3; // seconds to milliseconds
    }
  }

  /**
   * A batch of timestaps in milliseconds, coerced to numbers.
   */
  const TimestampMillisecondBatch = Int64Batch;

  /**
   * A batch of timestaps in microseconds, coerced to millisecond numbers.
   */
  class TimestampMicrosecondBatch extends Int64Batch {
    /**
     * @param {number} index The value index
     */
    value(index) {
      // microseconds to milliseconds
      return divide(/** @type {bigint} */ (this.values[index]), 1000n);
    }
  }

  /**
   * A batch of timestaps in nanoseconds, coerced to millisecond numbers.
   */
  class TimestampNanosecondBatch extends Int64Batch {
    /**
     * @param {number} index The value index
     */
    value(index) {
      // nanoseconds to milliseconds
      return divide(/** @type {bigint} */ (this.values[index]), 1000000n);
    }
  }

  /**
   * A batch of day/time intervals, returned as two-element 32-bit int arrays.
   * @extends {ArrayBatch<Int32Array>}
   */
  class IntervalDayTimeBatch extends ArrayBatch {
    /**
     * @param {number} index The value index
     * @returns {Int32Array}
     */
    value(index) {
      const values = /** @type {Int32Array} */ (this.values);
      return values.subarray(index << 1, (index + 1) << 1);
    }
  }

  /**
   * A batch of month/day/nanosecond intervals, returned as three-element arrays.
   * @extends {ArrayBatch<Float64Array>}
   */
  class IntervalMonthDayNanoBatch extends ArrayBatch {
    /**
     * @param {number} index The value index
     */
    value(index) {
      const values = /** @type {Uint8Array} */ (this.values);
      const base = index << 4;
      return Float64Array.of(
        readInt32(values, base),
        readInt32(values, base + 4),
        readInt64(values, base + 8)
      );
    }
  }

  const offset32 = ({values, offsets}, index) => values.subarray(offsets[index], offsets[index + 1]);
  const offset64 = ({values, offsets}, index) => values.subarray(toNumber(offsets[index]), toNumber(offsets[index + 1]));

  /**
   * A batch of binary blobs with variable offsets, returned as byte buffers of
   * unsigned 8-bit integers. The offsets are 32-bit ints.
   * @extends {ArrayBatch<Uint8Array>}
   */
  class BinaryBatch extends ArrayBatch {
    /**
     * @param {number} index
     * @returns {Uint8Array}
     */
    value(index) {
      return offset32(this, index);
    }
  }

  /**
   * A batch of binary blobs with variable offsets, returned as byte buffers of
   * unsigned 8-bit integers. The offsets are 64-bit ints. Value extraction will
   * fail if an offset exceeds `Number.MAX_SAFE_INTEGER`.
   * @extends {ArrayBatch<Uint8Array>}
   */
  class LargeBinaryBatch extends ArrayBatch {
    /**
     * @param {number} index
     * @returns {Uint8Array}
     */
    value(index) {
      return offset64(this, index);
    }
  }

  /**
   * A batch of UTF-8 strings with variable offsets. The offsets are 32-bit ints.
   * @extends {ArrayBatch<string>}
   */
  class Utf8Batch extends ArrayBatch {
    /**
     * @param {number} index
     */
    value(index) {
      return decodeUtf8(offset32(this, index));
    }
  }

  /**
   * A batch of UTF-8 strings with variable offsets. The offsets are 64-bit ints.
   * Value extraction will fail if an offset exceeds `Number.MAX_SAFE_INTEGER`.
   * @extends {ArrayBatch<string>}
   */
  class LargeUtf8Batch extends ArrayBatch {
    /**
     * @param {number} index
     */
    value(index) {
      return decodeUtf8(offset64(this, index));
    }
  }

  /**
   * A batch of list (array) values of variable length. The list offsets are
   * 32-bit ints.
   * @template V
   * @extends {ArrayBatch<import('./types.js').ValueArray<V>>}
   */
  class ListBatch extends ArrayBatch {
    /**
     * @param {number} index
     * @returns {import('./types.js').ValueArray<V>}
     */
    value(index) {
      const offsets = /** @type {Int32Array} */ (this.offsets);
      return this.children[0].slice(offsets[index], offsets[index + 1]);
    }
  }

  /**
   * A batch of list (array) values of variable length. The list offsets are
   * 64-bit ints. Value extraction will fail if an offset exceeds
   * `Number.MAX_SAFE_INTEGER`.
   * @template V
   * @extends {ArrayBatch<import('./types.js').ValueArray<V>>}
   */
  class LargeListBatch extends ArrayBatch {
    /**
     * @param {number} index
     * @returns {import('./types.js').ValueArray<V>}
     */
    value(index) {
      const offsets = /** @type {BigInt64Array} */ (this.offsets);
      return this.children[0].slice(toNumber(offsets[index]), toNumber(offsets[index + 1]));
    }
  }

  /**
   * A batch of list (array) values of variable length. The list offsets and
   * sizes are 32-bit ints.
   * @template V
   * @extends {ArrayBatch<import('./types.js').ValueArray<V>>}
   */
  class ListViewBatch extends ArrayBatch {
    /**
     * @param {number} index
     * @returns {import('./types.js').ValueArray<V>}
     */
    value(index) {
      const a = /** @type {number} */ (this.offsets[index]);
      const b = a + /** @type {number} */ (this.sizes[index]);
      return this.children[0].slice(a, b);
    }
  }

  /**
   * A batch of list (array) values of variable length. The list offsets and
   * sizes are 64-bit ints. Value extraction will fail if an offset or size
   * exceeds `Number.MAX_SAFE_INTEGER`.
   * @template V
   * @extends {ArrayBatch<import('./types.js').ValueArray<V>>}
   */
  class LargeListViewBatch extends ArrayBatch {
    /**
     * @param {number} index
     * @returns {import('./types.js').ValueArray<V>}
     */
    value(index) {
      const a = /** @type {bigint} */ (this.offsets[index]);
      const b = a + /** @type {bigint} */ (this.sizes[index]);
      return this.children[0].slice(toNumber(a), toNumber(b));
    }
  }

  /**
   * A batch with a fixed stride.
   * @template T
   * @extends {ArrayBatch<T>}
   */
  class FixedBatch extends ArrayBatch {
    constructor(options) {
      super(options);
      /** @type {number} */
      // @ts-ignore
      this.stride = this.type.stride;
    }
  }

  /**
   * A batch of binary blobs of fixed size, returned as byte buffers of unsigned
   * 8-bit integers.
   * @extends {FixedBatch<Uint8Array>}
   */
  class FixedBinaryBatch extends FixedBatch {
    /**
     * @param {number} index
     * @returns {Uint8Array}
     */
    value(index) {
      const { stride, values } = this;
      return /** @type {Uint8Array} */ (values)
        .subarray(index * stride, (index + 1) * stride);
    }
  }

  /**
   * A batch of list (array) values of fixed length.
   * @template V
   * @extends {FixedBatch<import('./types.js').ValueArray<V>>}
   */
  class FixedListBatch extends FixedBatch {
    /**
     * @param {number} index
     * @returns {import('./types.js').ValueArray<V>}
     */
    value(index) {
      const { children, stride } = this;
      return children[0].slice(index * stride, (index + 1) * stride);
    }
  }

  /**
   * Extract Map key-value pairs from parallel child batches.
   */
  function pairs({ children, offsets }, index) {
    const [ keys, vals ] = children[0].children;
    const start = offsets[index];
    const end = offsets[index + 1];
    const entries = [];
    for (let i = start; i < end; ++i) {
      entries.push([keys.at(i), vals.at(i)]);
    }
    return entries;
  }

  /**
   * A batch of map (key, value) values. The map is represented as a list of
   * key-value structs.
   * @template K, V
   * @extends {ArrayBatch<[K, V][]>}
   */
  class MapEntryBatch extends ArrayBatch {
    /**
     * Return the value at the given index.
     * @param {number} index The value index.
     * @returns {[K, V][]} The map entries as an array of [key, value] arrays.
     */
    value(index) {
      return /** @type {[K, V][]} */ (pairs(this, index));
    }
  }

  /**
   * A batch of map (key, value) values. The map is represented as a list of
   * key-value structs.
   * @template K, V
   * @extends {ArrayBatch<Map<K, V>>}
   */
  class MapBatch extends ArrayBatch {
    /**
     * Return the value at the given index.
     * @param {number} index The value index.
     * @returns {Map<K, V>} The map value.
     */
    value(index) {
      return new Map(/** @type {[K, V][]} */ (pairs(this, index)));
    }
  }

  /**
   * A batch of union-type values with a sparse layout, enabling direct
   * lookup from the child value batches.
   * @template T
   * @extends {ArrayBatch<T>}
   */
  class SparseUnionBatch extends ArrayBatch {
    /**
     * Create a new column batch.
     * @param {object} options
     * @param {number} options.length The length of the batch
     * @param {number} options.nullCount The null value count
     * @param {import('./types.js').DataType} options.type The data type.
     * @param {Uint8Array} [options.validity] Validity bitmap buffer
     * @param {Int32Array} [options.offsets] Offsets buffer
     * @param {Batch[]} options.children Children batches
     * @param {Int8Array} options.typeIds Union type ids buffer
     * @param {Record<string, number>} options.map A typeId to children index map
     */
    constructor({ typeIds, ...options }) {
      super(options);
      /** @type {Int8Array} */
      this.typeIds = typeIds;
      /** @type {Record<string, number>} */
      // @ts-ignore
      this.typeMap = this.type.typeMap;
    }

    /**
     * @param {number} index The value index.
     */
    value(index, offset = index) {
      const { typeIds, children, typeMap } = this;
      return children[typeMap[typeIds[index]]].at(offset);
    }
  }

  /**
   * A batch of union-type values with a dense layout, reqiring offset
   * lookups from the child value batches.
   * @template T
   * @extends {SparseUnionBatch<T>}
   */
  class DenseUnionBatch extends SparseUnionBatch {
    /**
     * @param {number} index The value index.
     */
    value(index) {
      return super.value(index, /** @type {number} */ (this.offsets[index]));
    }
  }

  /**
   * A batch of struct values, containing a set of named properties.
   * Struct property values are extracted and returned as JS objects.
   * @extends {ArrayBatch<Record<string, any>>}
   */
  class StructBatch extends ArrayBatch {
    constructor(options, factory = objectFactory) {
      super(options);
      /** @type {string[]} */
      // @ts-ignore
      this.names = this.type.children.map(child => child.name);
      this.factory = factory(this.names, this.children);
    }

    /**
     * @param {number} index The value index.
     * @returns {Record<string, any>}
     */
    value(index) {
      return this.factory(index);
    }
  }

  /**
   * A batch of struct values, containing a set of named properties.
   * Structs are returned as proxy objects that extract data directly
   * from underlying Arrow batches.
   * @extends {StructBatch}
   */
  class StructProxyBatch extends StructBatch {
    constructor(options) {
      super(options, proxyFactory);
    }
  }

  /**
   * A batch of run-end-encoded values.
   * @template T
   * @extends {ArrayBatch<T>}
   */
  class RunEndEncodedBatch extends ArrayBatch {
    /**
     * @param {number} index The value index.
     */
    value(index) {
      const [ { values: runs }, vals ] = this.children;
      return vals.at(
        bisect(/** @type {import('./types.js').IntegerArray} */(runs), index)
      );
    }
  }

  /**
   * A batch of dictionary-encoded values.
   * @template T
   * @extends {ArrayBatch<T>}
   */
  class DictionaryBatch extends ArrayBatch {
    /**
     * Register the backing dictionary. Dictionaries are added
     * after batch creation as the complete dictionary may not
     * be finished across multiple record batches.
     * @param {import('./column.js').Column<T>} dictionary
     * The dictionary of column values.
     */
    setDictionary(dictionary) {
      this.dictionary = dictionary;
      this.cache = dictionary.cache();
      return this;
    }

    /**
     * @param {number} index The value index.
     */
    value(index) {
      return this.cache[this.key(index)];
    }

    /**
     * @param {number} index The value index.
     * @returns {number} The dictionary key
     */
    key(index) {
      return /** @type {number} */ (this.values[index]);
    }
  }

  /**
   * @template T
   * @extends {ArrayBatch<T>}
   */
  class ViewBatch extends ArrayBatch {
    /**
     * Create a new view batch.
     * @param {object} options Batch options.
     * @param {number} options.length The length of the batch
     * @param {number} options.nullCount The null value count
     * @param {import('./types.js').DataType} options.type The data type.
     * @param {Uint8Array} [options.validity] Validity bitmap buffer
     * @param {Uint8Array} options.values Values buffer
     * @param {Uint8Array[]} options.data View data buffers
     */
    constructor({ data, ...options }) {
      super(options);
      this.data = data;
    }

    /**
     * Get the binary data at the provided index.
     * @param {number} index The value index.
     * @returns {Uint8Array}
     */
    view(index) {
      const { values, data } = this;
      const offset = index << 4; // each entry is 16 bytes
      let start = offset + 4;
      let buf = /** @type {Uint8Array} */ (values);
      const length = readInt32(buf, offset);
      if (length > 12) {
        // longer strings are in a data buffer
        start = readInt32(buf, offset + 12);
        buf = data[readInt32(buf, offset + 8)];
      }
      return buf.subarray(start, start + length);
    }
  }

  /**
   * A batch of binary blobs from variable data buffers, returned as byte
   * buffers of unsigned 8-bit integers.
   * @extends {ViewBatch<Uint8Array>}
   */
  class BinaryViewBatch extends ViewBatch {
    /**
     * @param {number} index The value index.
     */
    value(index) {
      return this.view(index);
    }
  }

  /**
   * A batch of UTF-8 strings from variable data buffers.
   * @extends {ViewBatch<string>}
   */
  class Utf8ViewBatch extends ViewBatch {
    /**
     * @param {number} index The value index.
     */
    value(index) {
      return decodeUtf8(this.view(index));
    }
  }

  /**
   * Build up a column from batches.
   */
  function columnBuilder(type) {
    let data = [];
    return {
      add(batch) { data.push(batch); return this; },
      clear: () => data = [],
      done: () => new Column(data, type)
    };
  }

  /**
   * A data column. A column provides a view over one or more value batches,
   * each drawn from an Arrow record batch. While this class supports random
   * access to column values by integer index; however, extracting arrays using
   * `toArray()` or iterating over values (`for (const value of column) {...}`)
   * provide more efficient ways for bulk access or scanning.
   * @template T
   */
  class Column {
    /**
     * Create a new column instance.
     * @param {import('./batch.js').Batch<T>[]} data The value batches.
     * @param {import('./types.js').DataType} [type] The column data type.
     *  If not specified, the type is extracted from the batches.
     */
    constructor(data, type = data[0]?.type) {
      /**
       * The column data type.
       * @type {import('./types.js').DataType}
       * @readonly
       */
      this.type = type;
      /**
       * The column length.
       * @type {number}
       * @readonly
       */
      this.length = data.reduce((m, c) => m + c.length, 0);
      /**
       * The count of null values in the column.
       * @type {number}
       * @readonly
       */
      this.nullCount = data.reduce((m, c) => m + c.nullCount, 0);
      /**
       * An array of column data batches.
       * @type {readonly import('./batch.js').Batch<T>[]}
       * @readonly
       */
      this.data = data;

      const n = data.length;
      const offsets = new Int32Array(n + 1);
      if (n === 1) {
        const [ batch ] = data;
        offsets[1] = batch.length;
        // optimize access to single batch
        this.at = index => batch.at(index);
      } else {
        for (let i = 0, s = 0; i < n; ++i) {
          offsets[i + 1] = (s += data[i].length);
        }
      }

      /**
       * Index offsets for data batches.
       * Used to map a column row index to a batch-specific index.
       * @type {Int32Array}
       * @readonly
       */
      this.offsets = offsets;
    }

    /**
     * Provide an informative object string tag.
     */
    get [Symbol.toStringTag]() {
      return 'Column';
    }

    /**
     * Return an iterator over the values in this column.
     * @returns {Iterator<T?>}
     */
    [Symbol.iterator]() {
      const data = this.data;
      return data.length === 1
        ? data[0][Symbol.iterator]()
        : batchedIterator(data);
    }

    /**
     * Return the column value at the given index. If a column has multiple
     * batches, this method performs binary search over the batch lengths to
     * determine the batch from which to retrieve the value. The search makes
     * lookup less efficient than a standard array access. If making a full
     * scan of a column, consider extracting arrays via `toArray()` or using an
     * iterator (`for (const value of column) {...}`).
     * @param {number} index The row index.
     * @returns {T | null} The value.
     */
    at(index) {
      // NOTE: if there is only one batch, this method is replaced with an
      // optimized version in the Column constructor.
      const { data, offsets } = this;
      const i = bisect(offsets, index) - 1;
      return data[i]?.at(index - offsets[i]); // undefined if out of range
    }

    /**
     * Return the column value at the given index. This method is the same as
     * `at()` and is provided for better compatibility with Apache Arrow JS.
     * @param {number} index The row index.
     * @returns {T | null} The value.
     */
    get(index) {
      return this.at(index);
    }

    /**
     * Extract column values into a single array instance. When possible,
     * a zero-copy subarray of the input Arrow data is returned.
     * @returns {import('./types.js').ValueArray<T?>}
     */
    toArray() {
      const { length, nullCount, data } = this;
      const copy = !nullCount && isDirectBatch(data[0]);
      const n = data.length;

      if (copy && n === 1) {
        // use batch array directly
        // @ts-ignore
        return data[0].values;
      }

      // determine output array type
      const ArrayType = !n || nullCount > 0 ? Array
        // @ts-ignore
        : (data[0].constructor.ArrayType ?? data[0].values.constructor);

      const array = new ArrayType(length);
      return copy ? copyArray(array, data) : extractArray(array, data);
    }

    /**
     * Return an array of cached column values.
     * Used internally to accelerate dictionary types.
     */
    cache() {
      return this._cache ?? (this._cache = this.toArray());
    }
  }

  function *batchedIterator(data) {
    for (let i = 0; i < data.length; ++i) {
      const iter = data[i][Symbol.iterator]();
      for (let next = iter.next(); !next.done; next = iter.next()) {
        yield next.value;
      }
    }
  }

  function copyArray(array, data) {
    for (let i = 0, offset = 0; i < data.length; ++i) {
      const { values } = data[i];
      array.set(values, offset);
      offset += values.length;
    }
    return array;
  }

  function extractArray(array, data) {
    let index = -1;
    for (let i = 0; i < data.length; ++i) {
      const batch = data[i];
      for (let j = 0; j < batch.length; ++j) {
        array[++index] = batch.at(j);
      }
    }
    return array;
  }

  /**
   * A table consists of a collection of named columns (or 'children').
   * To work with table data directly in JavaScript, use `toColumns()`
   * to extract an object that maps column names to extracted value arrays,
   * or `toArray()` to extract an array of row objects. For random access
   * by row index, use `getChild()` to access data for a specific column.
   */
  class Table {
    /**
     * Create a new table with the given schema and columns (children).
     * @param {import('./types.js').Schema} schema The table schema.
     * @param {import('./column.js').Column[]} children The table columns.
     * @param {boolean} [useProxy=false] Flag indicating if row proxy
     *  objects should be used to represent table rows (default `false`).
     */
    constructor(schema, children, useProxy = false) {
      const names = schema.fields.map(f => f.name);

      /** @readonly */
      this.schema = schema;
      /** @readonly */
      this.names = names;
      /**
       * @type {import('./column.js').Column[]}
       * @readonly
       */
      this.children = children;
      /**
       * @type {import('./types.js').StructFactory}
       * @readonly
       */
      this.factory = useProxy ? proxyFactory : objectFactory;

      // lazily created row object generators
      const gen = [];

      /**
       * Returns a row object generator for the given batch index.
       * @private
       * @readonly
       * @param {number} b The batch index.
       * @returns {(index: number) => Record<string,any>}
       */
      this.getFactory = b => gen[b]
        ?? (gen[b] = this.factory(names, children.map(c => c.data[b])));
    }

    /**
     * Provide an informative object string tag.
     */
    get [Symbol.toStringTag]() {
      return 'Table';
    }

    /**
     * The number of columns in this table.
     * @return {number} The number of columns.
     */
    get numCols() {
      return this.names.length;
    }

    /**
     * The number of rows in this table.
     * @return {number} The number of rows.
     */
    get numRows() {
      return this.children[0]?.length ?? 0;
    }

    /**
     * Return the child column at the given index position.
     * @param {number} index The column index.
     * @returns {import('./column.js').Column<any>}
     */
    getChildAt(index) {
      return this.children[index];
    }

    /**
     * Return the first child column with the given name.
     * @param {string} name The column name.
     * @returns {import('./column.js').Column<any>}
     */
    getChild(name) {
      const i = this.names.findIndex(x => x === name);
      return i > -1 ? this.children[i] : undefined;
    }

    /**
     * Construct a new table containing only columns at the specified indices.
     * The order of columns in the new table matches the order of input indices.
     * @param {number[]} indices The indices of columns to keep.
     * @param {string[]} [as] Optional new names for selected columns.
     * @returns {Table} A new table with columns at the specified indices.
     */
    selectAt(indices, as = []) {
      const { children, factory, schema } = this;
      const { fields } = schema;
      return new Table(
        {
          ...schema,
          fields: indices.map((i, j) => renameField(fields[i], as[j]))
        },
        indices.map(i => children[i]),
        factory === proxyFactory
      );
    }

    /**
     * Construct a new table containing only columns with the specified names.
     * If columns have duplicate names, the first (with lowest index) is used.
     * The order of columns in the new table matches the order of input names.
     * @param {string[]} names Names of columns to keep.
     * @param {string[]} [as] Optional new names for selected columns.
     * @returns {Table} A new table with columns matching the specified names.
     */
    select(names, as) {
      const all = this.names;
      const indices = names.map(name => all.indexOf(name));
      return this.selectAt(indices, as);
    }

    /**
     * Return an object mapping column names to extracted value arrays.
     * @returns {Record<string, import('./types.js').ValueArray<any>>}
     */
    toColumns() {
      const { children, names } = this;
      /** @type {Record<string, import('./types.js').ValueArray<any>>} */
      const cols = {};
      names.forEach((name, i) => cols[name] = children[i]?.toArray() ?? [] );
      return cols;
    }

    /**
     * Return an array of objects representing the rows of this table.
     * @returns {Record<string, any>[]}
     */
    toArray() {
      const { children, getFactory, numRows } = this;
      const data = children[0]?.data ?? [];
      const output = Array(numRows);
      for (let b = 0, row = -1; b < data.length; ++b) {
        const f = getFactory(b);
        for (let i = 0; i < data[b].length; ++i) {
          output[++row] = f(i);
        }
      }
      return output;
    }

    /**
     * Return an iterator over objects representing the rows of this table.
     * @returns {Generator<Record<string, any>, any, null>}
     */
    *[Symbol.iterator]() {
      const { children, getFactory } = this;
      const data = children[0]?.data ?? [];
      for (let b = 0; b < data.length; ++b) {
        const f = getFactory(b);
        for (let i = 0; i < data[b].length; ++i) {
          yield f(i);
        }
      }
    }

    /**
     * Return a row object for the given index.
     * @param {number} index The row index.
     * @returns {Record<string, any>} The row object.
     */
    at(index) {
      const { children, getFactory, numRows } = this;
      if (index < 0 || index >= numRows) return null;
      const [{ offsets }] = children;
      const b = bisect(offsets, index) - 1;
      return getFactory(b)(index - offsets[b]);
    }

    /**
     * Return a row object for the given index. This method is the same as
     * `at()` and is provided for better compatibility with Apache Arrow JS.
     * @param {number} index The row index.
     * @returns {Record<string, any>} The row object.
     */
    get(index) {
      return this.at(index);
    }
  }

  function renameField(field, name) {
    return (name != null && name !== field.name)
      ? { ...field, name }
      : field;
  }

  function batchType(type, options = {}) {
    const { typeId, bitWidth, precision, unit } = type;
    const { useBigInt, useDate, useDecimalBigInt, useMap, useProxy } = options;

    switch (typeId) {
      case Type.Null: return NullBatch;
      case Type.Bool: return BoolBatch;
      case Type.Int:
      case Type.Time:
      case Type.Duration:
        return useBigInt || bitWidth < 64 ? DirectBatch : Int64Batch;
      case Type.Float:
        return precision ? DirectBatch : Float16Batch;
      case Type.Date:
        return wrap(
          unit === DateUnit.DAY ? DateDayBatch : DateDayMillisecondBatch,
          useDate && DateBatch
        );
      case Type.Timestamp:
        return wrap(
          unit === TimeUnit.SECOND ? TimestampSecondBatch
            : unit === TimeUnit.MILLISECOND ? TimestampMillisecondBatch
            : unit === TimeUnit.MICROSECOND ? TimestampMicrosecondBatch
            : TimestampNanosecondBatch,
          useDate && DateBatch
        );
      case Type.Decimal:
        return useDecimalBigInt ? DecimalBigIntBatch : DecimalNumberBatch;
      case Type.Interval:
        return unit === IntervalUnit.DAY_TIME ? IntervalDayTimeBatch
          : unit === IntervalUnit.YEAR_MONTH ? DirectBatch
          : IntervalMonthDayNanoBatch;
      case Type.FixedSizeBinary: return FixedBinaryBatch;
      case Type.Utf8: return Utf8Batch;
      case Type.LargeUtf8: return LargeUtf8Batch;
      case Type.Binary: return BinaryBatch;
      case Type.LargeBinary: return LargeBinaryBatch;
      case Type.BinaryView: return BinaryViewBatch;
      case Type.Utf8View: return Utf8ViewBatch;
      case Type.List: return ListBatch;
      case Type.LargeList: return LargeListBatch;
      case Type.Map: return useMap ? MapBatch : MapEntryBatch;
      case Type.ListView: return ListViewBatch;
      case Type.LargeListView: return LargeListViewBatch;
      case Type.FixedSizeList: return FixedListBatch;
      case Type.Struct: return useProxy ? StructProxyBatch : StructBatch;
      case Type.RunEndEncoded: return RunEndEncodedBatch;
      case Type.Dictionary: return DictionaryBatch;
      case Type.Union: return type.mode ? DenseUnionBatch : SparseUnionBatch;
    }
    throw new Error(invalidDataType(typeId));
  }

  function wrap(BaseClass, WrapperClass) {
    return WrapperClass
      ? class WrapBatch extends WrapperClass {
          constructor(options) {
            super(new BaseClass(options));
          }
        }
      : BaseClass;
  }

  /**
   * Decode a block that points to messages within an Arrow 'file' format.
   * @param {Uint8Array} buf A byte buffer of binary Arrow IPC data
   * @param {number} index The starting index in the byte buffer
   * @returns The file block.
   */
  function decodeBlock(buf, index) {
    //  0: offset
    //  8: metadataLength
    // 16: bodyLength
    return {
      offset: readInt64(buf, index),
      metadataLength: readInt32(buf, index + 8),
      bodyLength: readInt64(buf, index + 16)
    }
  }

  /**
   * Decode a vector of blocks.
   * @param {Uint8Array} buf
   * @param {number} index
   * @returns An array of file blocks.
   */
  function decodeBlocks(buf, index) {
    return readVector(buf, index, 24, decodeBlock);
  }

  /**
   * Decode a record batch.
   * @param {Uint8Array} buf A byte buffer of binary Arrow IPC data
   * @param {number} index The starting index in the byte buffer
   * @param {import('../types.js').Version_} version Arrow version value
   * @returns {import('../types.js').RecordBatch} The record batch
   */
  function decodeRecordBatch(buf, index, version) {
    //  4: length
    //  6: nodes
    //  8: buffers
    // 10: compression (not supported)
    // 12: variadicBuffers (buffer counts for view-typed fields)
    const get = readObject(buf, index);
    if (get(10, readOffset, 0)) {
      throw new Error('Record batch compression not implemented');
    }

    // If an Arrow buffer was written before version 4,
    // advance 8 bytes to skip the now-removed page_id field
    const offset = version < Version.V4 ? 8 : 0;

    return {
      length: get(4, readInt64, 0),
      nodes: readVector(buf, get(6, readOffset), 16, (buf, pos) => ({
        length: readInt64(buf, pos),
        nullCount: readInt64(buf, pos + 8)
      })),
      regions: readVector(buf, get(8, readOffset), 16 + offset, (buf, pos) => ({
        offset: readInt64(buf, pos + offset),
        length: readInt64(buf, pos + offset + 8)
      })),
      variadic: readVector(buf, get(12, readOffset), 8, readInt64)
    };
  }

  /**
   * Decode a dictionary batch.
   * @param {Uint8Array} buf A byte buffer of binary Arrow IPC data
   * @param {number} index The starting index in the byte buffer
   * @param {import('../types.js').Version_} version Arrow version value
   * @returns {import('../types.js').DictionaryBatch} The dictionary batch
   */
  function decodeDictionaryBatch(buf, index, version) {
    //  4: id
    //  6: data
    //  8: isDelta
    const get = readObject(buf, index);
    return {
      id: get(4, readInt64, 0),
      data: get(6, (buf, off) => decodeRecordBatch(buf, off, version)),
      /**
       * If isDelta is true the values in the dictionary are to be appended to a
       * dictionary with the indicated id. If isDelta is false this dictionary
       * should replace the existing dictionary.
       */
      isDelta: get(8, readBoolean, false)
    };
  }

  /**
   * Decode a data type definition for a field.
   * @param {Uint8Array} buf A byte buffer of binary Arrow IPC data.
   * @param {number} index The starting index in the byte buffer.
   * @param {number} typeId The data type id.
   * @param {import('../types.js').Field[]} [children] A list of parsed child fields.
   * @returns {import('../types.js').DataType} The data type.
   */
  function decodeDataType(buf, index, typeId, children) {
    checkOneOf(typeId, Type, invalidDataType);
    const get = readObject(buf, index);

    switch (typeId) {
      // types without flatbuffer objects
      case Type.Binary: return binary();
      case Type.Utf8: return utf8();
      case Type.LargeBinary: return largeBinary();
      case Type.LargeUtf8: return largeUtf8();
      case Type.List: return list(children[0]);
      case Type.ListView: return listView(children[0]);
      case Type.LargeList: return largeList(children[0]);
      case Type.LargeListView: return largeListView(children[0]);
      case Type.Struct: return struct(children);
      case Type.RunEndEncoded: return runEndEncoded(children[0], children[1]);

      // types with flatbuffer objects
      case Type.Int: return int(
        // @ts-ignore
        get(4, readInt32, 0), // bitwidth
        get(6, readBoolean, false) // signed
      );
      case Type.Float: return float(
        // @ts-ignore
        get(4, readInt16, Precision.HALF) // precision
      );
      case Type.Decimal: return decimal(
        get(4, readInt32, 0), // precision
        get(6, readInt32, 0), // scale
        // @ts-ignore
        get(8, readInt32, 128) // bitwidth
      );
      case Type.Date: return date(
        // @ts-ignore
        get(4, readInt16, DateUnit.MILLISECOND) // unit
      );
      case Type.Time: return time(
        // @ts-ignore
        get(4, readInt16, TimeUnit.MILLISECOND), // unit
        get(6, readInt32, 32) // bitWidth
      );
      case Type.Timestamp: return timestamp(
        // @ts-ignore
        get(4, readInt16, TimeUnit.SECOND), // unit
        get(6, readString) // timezone
      );
      case Type.Interval: return interval(
        // @ts-ignore
        get(4, readInt16, IntervalUnit.YEAR_MONTH) // unit
      );
      case Type.Duration: return duration(
        // @ts-ignore
        get(4, readInt16, TimeUnit.MILLISECOND) // unit
      );

      case Type.FixedSizeBinary: return fixedSizeBinary(
        get(4, readInt32, 0) // stride
      );
      case Type.FixedSizeList: return fixedSizeList(
        children[0],
        get(4, readInt32, 0), // stride
      );
      case Type.Map: return mapType(
        get(4, readBoolean, false), // keysSorted
        children[0]
      );

      case Type.Union: return union(
        // @ts-ignore
        get(4, readInt16, UnionMode.Sparse), // mode
        children,
        readVector(buf, get(6, readOffset), 4, readInt32) // type ids
      );
    }
    // case Type.NONE:
    // case Type.Null:
    // case Type.Bool:
    // case Type.BinaryView:
    // case Type.Utf8View:
    // @ts-ignore
    return { typeId };
  }

  /**
   * Decode custom metadata consisting of key-value string pairs.
   * @param {Uint8Array} buf A byte buffer of binary Arrow IPC data
   * @param {number} index The starting index in the byte buffer
   * @returns {import('../types.js').Metadata | null} The custom metadata map
   */
  function decodeMetadata(buf, index) {
    const entries = readVector(buf, index, 4, (buf, pos) => {
      const get = readObject(buf, pos);
      return /** @type {[string, string]} */ ([
        get(4, readString), // 4: key (string)
        get(6, readString)  // 6: key (string)
      ]);
    });
    return entries.length ? new Map(entries) : null;
  }

  /**
   * Decode a table schema describing the fields and their data types.
   * @param {Uint8Array} buf A byte buffer of binary Arrow IPC data
   * @param {number} index The starting index in the byte buffer
   * @param {import('../types.js').Version_} version Arrow version value
   * @returns {import('../types.js').Schema} The schema
   */
  function decodeSchema(buf, index, version) {
    //  4: endianness (int16)
    //  6: fields (vector)
    //  8: metadata (vector)
    // 10: features (int64[])
    const get = readObject(buf, index);
    return {
      version,
      endianness: /** @type {import('../types.js').Endianness_} */ (get(4, readInt16, 0)),
      fields: get(6, decodeSchemaFields, []),
      metadata: get(8, decodeMetadata)
    };
  }

  /**
   * @returns {import('../types.js').Field[] | null}
   */
  function decodeSchemaFields(buf, fieldsOffset) {
    return readVector(buf, fieldsOffset, 4, decodeField);
  }

  /**
   * @returns {import('../types.js').Field}
   */
  function decodeField(buf, index) {
    //  4: name (string)
    //  6: nullable (bool)
    //  8: type id (uint8)
    // 10: type (union)
    // 12: dictionary (table)
    // 14: children (vector)
    // 16: metadata (vector)
    const get = readObject(buf, index);
    const typeId = get(8, readUint8, Type.NONE);
    const typeOffset = get(10, readOffset, 0);
    const dict = get(12, decodeDictionary);
    const children = get(14, (buf, off) => decodeFieldChildren(buf, off));

    let type = decodeDataType(buf, typeOffset, typeId, children);
    if (dict) {
      dict.dictionary = type;
      type = dict;
    }

    return {
      name: get(4, readString),
      type,
      nullable: get(6, readBoolean, false),
      metadata: get(16, decodeMetadata)
    };
  }

  /**
   * @returns {import('../types.js').Field[] | null}
   */
  function decodeFieldChildren(buf, fieldOffset) {
    const children = readVector(buf, fieldOffset, 4, decodeField);
    return children.length ? children : null;
  }

  /**
   * @param {Uint8Array} buf
   * @param {number} index
   * @returns {import('../types.js').DictionaryType}
   */
  function decodeDictionary(buf, index) {
    if (!index) return null;
    //  4: id (int64)
    //  6: indexType (Int type)
    //  8: isOrdered (boolean)
    // 10: kind (int16) currently only dense array is supported
    const get = readObject(buf, index);
    return dictionary$1(
      null, // data type will be populated by caller
      get(6, decodeInt, int32()), // index type
      get(8, readBoolean, false), // ordered
      get(4, readInt64, 0), // id
    );
  }

  /**
   * Decode an integer data type.
   * @param {Uint8Array} buf A byte buffer of binary Arrow IPC data.
   * @param {number} index The starting index in the byte buffer.
   * @returns {import('../types.js').IntType}
   */
  function decodeInt(buf, index) {
    return /** @type {import('../types.js').IntType} */ (
      decodeDataType(buf, index, Type.Int)
    );
  }

  const invalidMessageMetadata = (expected, actual) =>
    `Expected to read ${expected} metadata bytes, but only read ${actual}.`;

  const invalidMessageBodyLength = (expected, actual) =>
    `Expected to read ${expected} bytes for message body, but only read ${actual}.`;

  const invalidMessageType = (type) =>
    `Unsupported message type: ${type} (${keyFor(MessageHeader, type)})`;

  /**
   * A "message" contains a block of Apache Arrow data, such as a schema,
   * record batch, or dictionary batch. This message decodes a single
   * message, returning its associated metadata and content.
   * @param {Uint8Array} buf A byte buffer of binary Arrow IPC data
   * @param {number} index The starting index in the byte buffer
   * @returns {import('../types.js').Message} The decoded message.
   */
  function decodeMessage(buf, index) {
    // get message start
    let metadataLength = readInt32(buf, index) || 0;
    index += SIZEOF_INT;

    // ARROW-6313: If the first 4 bytes are continuation indicator (-1), read
    // the next 4 for the 32-bit metadata length. Otherwise, assume this is a
    // pre-v0.15 message, where the first 4 bytes are the metadata length.
    if (metadataLength === -1) {
      metadataLength = readInt32(buf, index) || 0;
      index += SIZEOF_INT;
    }
    if (metadataLength === 0) return null;

    const head = buf.subarray(index, index += metadataLength);
    if (head.byteLength < metadataLength) {
      throw new Error(invalidMessageMetadata(metadataLength, head.byteLength));
    }

    // decode message metadata
    //  4: version
    //  6: headerType
    //  8: headerIndex
    // 10: bodyLength
    const get = readObject(head, 0);
    const version = /** @type {import('../types.js').Version_} */
      (get(4, readInt16, Version.V1));
    const type = /** @type {import('../types.js').MessageHeader_} */
      (get(6, readUint8, MessageHeader.NONE));
    const offset = get(8, readOffset, 0);
    const bodyLength = get(10, readInt64, 0);
    let content;

    if (offset) {
      // decode message header
      const decoder = type === MessageHeader.Schema ? decodeSchema
        : type === MessageHeader.DictionaryBatch ? decodeDictionaryBatch
        : type === MessageHeader.RecordBatch ? decodeRecordBatch
        : null;
      if (!decoder) throw new Error(invalidMessageType(type));
      content = decoder(head, offset, version);

      // extract message body
      if (bodyLength > 0) {
        const body = buf.subarray(index, index += bodyLength);
        if (body.byteLength < bodyLength) {
          throw new Error(invalidMessageBodyLength(bodyLength, body.byteLength));
        }
        // @ts-ignore
        content.body = body;
      }
    }

    return { version, type, index, content };
  }

  /**
   * Decode [Apache Arrow IPC data][1] and return parsed schema, record batch,
   * and dictionary batch definitions. The input binary data may be either
   * an `ArrayBuffer` or `Uint8Array`. For Arrow data in the IPC 'stream' format,
   * an array of `Uint8Array` instances is also supported.
   *
   * This method stops short of generating views over field buffers. Use the
   * `createData()` method on the result to enable column data access.
   *
   * [1]: https://arrow.apache.org/docs/format/Columnar.html#serialization-and-interprocess-communication-ipc
   * @param {ArrayBuffer | Uint8Array | Uint8Array[]} data
   *  The source byte buffer, or an array of buffers. If an array, each byte
   *  array may contain one or more self-contained messages. Messages may NOT
   *  span multiple byte arrays.
   * @returns {import('../types.js').ArrowData}
   */
  function decodeIPC(data) {
    const source = data instanceof ArrayBuffer
      ? new Uint8Array(data)
      : data;
    return source instanceof Uint8Array && isArrowFileFormat(source)
      ? decodeIPCFile(source)
      : decodeIPCStream(source);
  }

  /**
   * @param {Uint8Array} buf
   * @returns {boolean}
   */
  function isArrowFileFormat(buf) {
    if (!buf || buf.length < 4) return false;
    for (let i = 0; i < 6; ++i) {
      if (MAGIC[i] !== buf[i]) return false;
    }
    return true;
  }

  /**
   * Decode data in the [Arrow IPC 'stream' format][1].
   *
   * [1]: https://arrow.apache.org/docs/format/Columnar.html#ipc-streaming-format
   * @param {Uint8Array | Uint8Array[]} data The source byte buffer, or an
   *  array of buffers. If an array, each byte array may contain one or more
   *  self-contained messages. Messages may NOT span multiple byte arrays.
   * @returns {import('../types.js').ArrowData}
   */
  function decodeIPCStream(data) {
    const stream = [data].flat();

    let schema;
    const records = [];
    const dictionaries = [];

    // consume each message in the stream
    for (const buf of stream) {
      if (!(buf instanceof Uint8Array)) {
        throw new Error(`IPC data batch was not a Uint8Array.`);
      }
      let offset = 0;

      // decode all messages in current buffer
      while (true) {
        const m = decodeMessage(buf, offset);
        if (m === null) break; // end of messages
        offset = m.index;
        if (!m.content) continue;
        switch (m.type) {
          case MessageHeader.Schema:
            // ignore repeated schema messages
            if (!schema) schema = m.content;
            break;
          case MessageHeader.RecordBatch:
            records.push(m.content);
            break;
          case MessageHeader.DictionaryBatch:
            dictionaries.push(m.content);
            break;
        }
      }
    }

    return /** @type {import('../types.js').ArrowData} */ (
      { schema, dictionaries, records, metadata: null }
    );
  }

  /**
   * Decode data in the [Arrow IPC 'file' format][1].
   *
   * [1]: https://arrow.apache.org/docs/format/Columnar.html#ipc-file-format
   * @param {Uint8Array} data The source byte buffer.
   * @returns {import('../types.js').ArrowData}
   */
  function decodeIPCFile(data) {
    // find footer location
    const offset = data.byteLength - (MAGIC.length + 4);
    const length = readInt32(data, offset);

    // decode file footer
    //  4: version
    //  6: schema
    //  8: dictionaries (vector)
    // 10: batches (vector)
    // 12: metadata
    const get = readObject(data, offset - length);
    const version = /** @type {import('../types.js').Version_} */
      (get(4, readInt16, Version.V1));
    const dicts = get(8, decodeBlocks, []);
    const recs = get(10, decodeBlocks, []);

    return /** @type {import('../types.js').ArrowData} */ ({
      schema: get(6, (buf, index) => decodeSchema(buf, index, version)),
      dictionaries: dicts.map(({ offset }) => decodeMessage(data, offset).content),
      records: recs.map(({ offset }) => decodeMessage(data, offset).content),
      metadata: get(12, decodeMetadata)
    });
  }

  /**
   * Decode [Apache Arrow IPC data][1] and return a new Table. The input binary
   * data may be either an `ArrayBuffer` or `Uint8Array`. For Arrow data in the
   * [IPC 'stream' format][2], an array of `Uint8Array` values is also supported.
   *
   * [1]: https://arrow.apache.org/docs/format/Columnar.html#serialization-and-interprocess-communication-ipc
   * [2]: https://arrow.apache.org/docs/format/Columnar.html#ipc-streaming-format
   * @param {ArrayBuffer | Uint8Array | Uint8Array[]} data
   *  The source byte buffer, or an array of buffers. If an array, each byte
   *  array may contain one or more self-contained messages. Messages may NOT
   *  span multiple byte arrays.
   * @param {import('../types.js').ExtractionOptions} [options]
   *  Options for controlling how values are transformed when extracted
   *  from an Arrow binary representation.
   * @returns {Table} A Table instance.
   */
  function tableFromIPC(data, options) {
    return createTable(decodeIPC(data), options);
  }

  /**
   * Create a table from parsed IPC data.
   * @param {import('../types.js').ArrowData} data
   *  The IPC data, as returned by parseIPC.
   * @param {import('../types.js').ExtractionOptions} [options]
   *  Options for controlling how values are transformed when extracted
   *  from am Arrow binary representation.
   * @returns {Table} A Table instance.
   */
  function createTable(data, options = {}) {
    const { schema = { fields: [] }, dictionaries, records } = data;
    const { version, fields } = schema;
    const dictionaryMap = new Map;
    const context = contextGenerator(options, version, dictionaryMap);

    // build dictionary type map
    const dictionaryTypes = new Map;
    visitSchemaFields(schema, field => {
      const type = field.type;
      if (type.typeId === Type.Dictionary) {
        dictionaryTypes.set(type.id, type.dictionary);
      }
    });

    // decode dictionaries, build dictionary column map
    const dicts = new Map;
    for (const dict of dictionaries) {
      const { id, data, isDelta, body } = dict;
      const type = dictionaryTypes.get(id);
      const batch = visit$1(type, context({ ...data, body }));
      if (!dicts.has(id)) {
        if (isDelta) {
          throw new Error('Delta update can not be first dictionary batch.');
        }
        dicts.set(id, columnBuilder(type).add(batch));
      } else {
        const dict = dicts.get(id);
        if (!isDelta) dict.clear();
        dict.add(batch);
      }
    }
    dicts.forEach((value, key) => dictionaryMap.set(key, value.done()));

    // decode column fields
    const cols = fields.map(f => columnBuilder(f.type));
    for (const batch of records) {
      const ctx = context(batch);
      fields.forEach((f, i) => cols[i].add(visit$1(f.type, ctx)));
    }

    return new Table(schema, cols.map(c => c.done()), options.useProxy);
  }

  /**
   * Visit all fields within a schema.
   * @param {import('../types.js').Schema} schema
   * @param {(field: import('../types.js').Field) => void} visitor
   */
  function visitSchemaFields(schema, visitor) {
    schema.fields.forEach(function visitField(field) {
      visitor(field);
      // @ts-ignore
      field.type.dictionary?.children?.forEach(visitField);
      // @ts-ignore
      field.type.children?.forEach(visitField);
    });
  }

  /**
   * Context object generator for field visitation and buffer definition.
   */
  function contextGenerator(options, version, dictionaryMap) {
    const base = {
      version,
      options,
      dictionary: id => dictionaryMap.get(id),
    };

    /**
     * Return a context generator.
     * @param {import('../types.js').RecordBatch} batch
     */
    return batch => {
      const { length, nodes, regions, variadic, body } = batch;
      let nodeIndex = -1;
      let bufferIndex = -1;
      let variadicIndex = -1;
      return {
        ...base,
        length,
        node: () => nodes[++nodeIndex],
        buffer: (ArrayType) => {
          const { length, offset } = regions[++bufferIndex];
          return ArrayType
            ? new ArrayType(body.buffer, body.byteOffset + offset, length / ArrayType.BYTES_PER_ELEMENT)
            : body.subarray(offset, offset + length)
        },
        variadic: () => variadic[++variadicIndex],
        visit(children) { return children.map(f => visit$1(f.type, this)); }
      };
    };
  }

  /**
   * Visit a field, instantiating views of buffer regions.
   */
  function visit$1(type, ctx) {
    const { typeId } = type;
    const { length, options, node, buffer, variadic, version } = ctx;
    const BatchType = batchType(type, options);

    if (typeId === Type.Null) {
      // no field node, no buffers
      return new BatchType({ length, nullCount: length, type });
    }

    // extract the next { length, nullCount } field node
    const base = { ...node(), type };

    switch (typeId) {
      // validity and data value buffers
      case Type.Bool:
      case Type.Int:
      case Type.Time:
      case Type.Duration:
      case Type.Float:
      case Type.Decimal:
      case Type.Date:
      case Type.Timestamp:
      case Type.Interval:
      case Type.FixedSizeBinary:
        return new BatchType({
          ...base,
          validity: buffer(),
          values: buffer(type.values)
        });

      // validity, offset, and value buffers
      case Type.Utf8:
      case Type.LargeUtf8:
      case Type.Binary:
      case Type.LargeBinary:
        return new BatchType({
          ...base,
          validity: buffer(),
          offsets: buffer(type.offsets),
          values: buffer()
        });

      // views with variadic buffers
      case Type.BinaryView:
      case Type.Utf8View:
        return new BatchType({
          ...base,
          validity: buffer(),
          values: buffer(), // views buffer
          data: Array.from({ length: variadic() }, () => buffer()) // data buffers
        });

      // validity, offset, and list child
      case Type.List:
      case Type.LargeList:
      case Type.Map:
        return new BatchType({
          ...base,
          validity: buffer(),
          offsets: buffer(type.offsets),
          children: ctx.visit(type.children)
        });

      // validity, offset, size, and list child
      case Type.ListView:
      case Type.LargeListView:
        return new BatchType({
          ...base,
          validity: buffer(),
          offsets: buffer(type.offsets),
          sizes: buffer(type.offsets),
          children: ctx.visit(type.children)
        });

      // validity and children
      case Type.FixedSizeList:
      case Type.Struct:
        return new BatchType({
          ...base,
          validity: buffer(),
          children: ctx.visit(type.children)
        });

      // children only
      case Type.RunEndEncoded:
        return new BatchType({
          ...base,
          children: ctx.visit(type.children)
        });

      // dictionary
      case Type.Dictionary: {
        const { id, indices } = type;
        return new BatchType({
          ...base,
          validity: buffer(),
          values: buffer(indices.values),
        }).setDictionary(ctx.dictionary(id));
      }

      // union
      case Type.Union: {
        if (version < Version.V5) {
          buffer(); // skip unused null bitmap
        }
        return new BatchType({
          ...base,
          typeIds: buffer(int8Array),
          offsets: type.mode === UnionMode.Sparse ? null : buffer(type.offsets),
          children: ctx.visit(type.children)
        });
      }

      // unsupported type
      default:
        throw new Error(invalidDataType(typeId));
    }
  }

  function writeInt32(buf, index, value) {
    buf[index] = value;
    buf[index + 1] = value >> 8;
    buf[index + 2] = value >> 16;
    buf[index + 3] = value >> 24;
  }

  const INIT_SIZE = 1024;

  /** Flatbuffer binary builder. */
  class Builder {
    /**
     * Create a new builder instance.
     * @param {import('./sink.js').Sink} sink The byte consumer.
     */
    constructor(sink) {
      /**
       * Sink that consumes built byte buffers;
       * @type {import('./sink.js').Sink}
       */
      this.sink = sink;
      /**
       * Minimum alignment encountered so far.
       * @type {number}
       */
      this.minalign = 1;
      /**
       * Current byte buffer.
       * @type {Uint8Array}
       */
      this.buf = new Uint8Array(INIT_SIZE);
      /**
       * Remaining space in the current buffer.
       * @type {number}
       */
      this.space = INIT_SIZE;
      /**
       * List of offsets of all vtables. Used to find and
       * reuse tables upon duplicated table field schemas.
       * @type {number[]}
       */
      this.vtables = [];
      /**
       * Total bytes written to sink thus far.
       */
      this.outputBytes = 0;
    }

    /**
     * Returns the flatbuffer offset, relative to the end of the current buffer.
     * @returns {number} Offset relative to the end of the buffer.
     */
    offset() {
      return this.buf.length - this.space;
    }

    /**
     * Write a flatbuffer int8 value at the current buffer position
     * and advance the internal cursor.
     * @param {number} value
     */
    writeInt8(value) {
      this.buf[this.space -= 1] = value;
    }

    /**
     * Write a flatbuffer int16 value at the current buffer position
     * and advance the internal cursor.
     * @param {number} value
     */
    writeInt16(value) {
      this.buf[this.space -= 2] = value;
      this.buf[this.space + 1] = value >> 8;
    }

    /**
     * Write a flatbuffer int32 value at the current buffer position
     * and advance the internal cursor.
     * @param {number} value
     */
    writeInt32(value) {
      writeInt32(this.buf, this.space -= 4, value);
    }

    /**
     * Write a flatbuffer int64 value at the current buffer position
     * and advance the internal cursor.
     * @param {number} value
     */
    writeInt64(value) {
      const v = BigInt(value);
      this.writeInt32(Number(BigInt.asIntN(32, v >> BigInt(32))));
      this.writeInt32(Number(BigInt.asIntN(32, v)));
    }

    /**
     * Add a flatbuffer int8 value, properly aligned,
     * @param value The int8 value to add the buffer.
     */
    addInt8(value) {
      prep(this, 1, 0);
      this.writeInt8(value);
    }

    /**
     * Add a flatbuffer int16 value, properly aligned,
     * @param value The int16 value to add the buffer.
     */
    addInt16(value) {
      prep(this, 2, 0);
      this.writeInt16(value);
    }

    /**
     * Add a flatbuffer int32 value, properly aligned,
     * @param value The int32 value to add the buffer.
     */
    addInt32(value) {
      prep(this, 4, 0);
      this.writeInt32(value);
    }

    /**
     * Add a flatbuffer int64 values, properly aligned.
     * @param value The int64 value to add the buffer.
     */
    addInt64(value) {
      prep(this, 8, 0);
      this.writeInt64(value);
    }

    /**
     * Add a flatbuffer offset, relative to where it will be written.
     * @param {number} offset The offset to add.
     */
    addOffset(offset) {
      prep(this, SIZEOF_INT, 0); // Ensure alignment is already done.
      this.writeInt32(this.offset() - offset + SIZEOF_INT);
    }

    /**
     * Add a flatbuffer object (vtable).
     * @param {number} numFields The maximum number of fields
     *  this object may include.
     * @param {(tableBuilder: ReturnType<objectBuilder>) => void} [addFields]
     *  A callback function that writes all fields using an object builder.
     * @returns {number} The object offset.
     */
    addObject(numFields, addFields) {
      const b = objectBuilder(this, numFields);
      addFields?.(b);
      return b.finish();
    }

    /**
     * Add a flatbuffer vector (list).
     * @template T
     * @param {T[]} items An array of items to write.
     * @param {number} itemSize The size in bytes of a serialized item.
     * @param {number} alignment The desired byte alignment value.
     * @param {(builder: this, item: T) => void} writeItem A callback
     *  function that writes a vector item to this builder.
     * @returns {number} The vector offset.
     */
    addVector(items, itemSize, alignment, writeItem) {
      const n = items?.length;
      if (!n) return 0;
      prep(this, SIZEOF_INT, itemSize * n);
      prep(this, alignment, itemSize * n); // Just in case alignment > int.
      for (let i = n; --i >= 0;) {
        writeItem(this, items[i]);
      }
      this.writeInt32(n);
      return this.offset();
    }

    /**
     * Convenience method for writing a vector of byte buffer offsets.
     * @param {number[]} offsets
     * @returns {number} The vector offset.
     */
    addOffsetVector(offsets) {
      return this.addVector(offsets, 4, 4, (b, off) => b.addOffset(off));
    }

    /**
     * Add a flatbuffer UTF-8 string.
     * @param {string} s The string to encode.
     * @return {number} The string offset.
     */
    addString(s) {
      if (s == null) return 0;
      const utf8 = encodeUtf8(s);
      const n = utf8.length;
      this.addInt8(0); // string null terminator
      prep(this, SIZEOF_INT, n);
      this.buf.set(utf8, this.space -= n);
      this.writeInt32(n);
      return this.offset();
    }

    /**
     * Finish the current flatbuffer by adding a root offset.
     * @param {number} rootOffset The root offset.
     */
    finish(rootOffset) {
      prep(this, this.minalign, SIZEOF_INT);
      this.addOffset(rootOffset);
    }

    /**
     * Flush the current flatbuffer byte buffer content to the sink,
     * and reset the flatbuffer builder state.
     */
    flush() {
      const { buf, sink } = this;
      const bytes = buf.subarray(this.space, buf.length);
      sink.write(bytes);
      this.outputBytes += bytes.byteLength;
      this.minalign = 1;
      this.vtables = [];
      this.buf = new Uint8Array(INIT_SIZE);
      this.space = INIT_SIZE;
    }

    /**
     * Add a byte buffer directly to the builder sink. This method bypasses
     * any unflushed flatbuffer state and leaves it unchanged, writing the
     * buffer to the sink *before* the flatbuffer.
     * The buffer will be padded for 64-bit (8-byte) alignment as needed.
     * @param {Uint8Array} buffer The buffer to add.
     * @returns {number} The total byte count of the buffer and padding.
     */
    addBuffer(buffer) {
      const size = buffer.byteLength;
      if (!size) return 0;
      this.sink.write(buffer);
      this.outputBytes += size;
      const pad = ((size + 7) & ~7) - size;
      this.addPadding(pad);
      return size + pad;
    }

    /**
     * Write padding bytes directly to the builder sink. This method bypasses
     * any unflushed flatbuffer state and leaves it unchanged, writing the
     * padding bytes to the sink *before* the flatbuffer.
     * @param {number} byteCount The number of padding bytes.
     */
    addPadding(byteCount) {
      if (byteCount > 0) {
        this.sink.write(new Uint8Array(byteCount));
        this.outputBytes += byteCount;
      }
    }
  }

  /**
   * Prepare to write an element of `size` after `additionalBytes` have been
   * written, e.g. if we write a string, we need to align such the int length
   * field is aligned to 4 bytes, and the string data follows it directly. If all
   * we need to do is alignment, `additionalBytes` will be 0.
   * @param {Builder} builder The builder to prep.
   * @param {number} size The size of the new element to write.
   * @param {number} additionalBytes Additional padding size.
   */
  function prep(builder, size, additionalBytes) {
    let { buf, space, minalign } = builder;

    // track the biggest thing we've ever aligned to
    if (size > minalign) {
      builder.minalign = size;
    }

    // find alignment needed so that `size` aligns after `additionalBytes`
    const bufSize = buf.length;
    const used = bufSize - space + additionalBytes;
    const alignSize = (~used + 1) & (size - 1);

    // reallocate the buffer if needed
    buf = grow(buf, used + alignSize + size - 1, true);
    space += buf.length - bufSize;

    // add padding
    for (let i = 0; i < alignSize; ++i) {
      buf[--space] = 0;
    }

    // update builder state
    builder.buf = buf;
    builder.space = space;
  }

  /**
   * Returns a builder object for flatbuffer objects (vtables).
   * @param {Builder} builder The underlying flatbuffer builder.
   * @param {number} numFields The expected number of fields, not
   *  including the standard size fields.
   */
  function objectBuilder(builder, numFields) {
    /** @type {number[]} */
    const vtable = Array(numFields).fill(0);
    const startOffset = builder.offset();

    function slot(index) {
      vtable[index] = builder.offset();
    }

    return {
      /**
       * Add an int8-valued table field.
       * @param {number} index
       * @param {number} value
       * @param {number} defaultValue
       */
      addInt8(index, value, defaultValue) {
        if (value != defaultValue) {
          builder.addInt8(value);
          slot(index);
        }
      },

      /**
       * Add an int16-valued table field.
       * @param {number} index
       * @param {number} value
       * @param {number} defaultValue
       */
      addInt16(index, value, defaultValue) {
        if (value != defaultValue) {
          builder.addInt16(value);
          slot(index);
        }
      },

      /**
       * Add an int32-valued table field.
       * @param {number} index
       * @param {number} value
       * @param {number} defaultValue
       */
      addInt32(index, value, defaultValue) {
        if (value != defaultValue) {
          builder.addInt32(value);
          slot(index);
        }
      },

      /**
       * Add an int64-valued table field.
       * @param {number} index
       * @param {number} value
       * @param {number} defaultValue
       */
      addInt64(index, value, defaultValue) {
        if (value != defaultValue) {
          builder.addInt64(value);
          slot(index);
        }
      },

      /**
       * Add a buffer offset-valued table field.
       * @param {number} index
       * @param {number} value
       * @param {number} defaultValue
       */
      addOffset(index, value, defaultValue) {
        if (value != defaultValue) {
          builder.addOffset(value);
          slot(index);
        }
      },

      /**
       * Write the vtable to the buffer and return the table offset.
       * @returns {number} The buffer offset to the vtable.
       */
      finish() {
        // add offset entry, will overwrite later with actual offset
        builder.addInt32(0);
        const vtableOffset = builder.offset();

        // trim zero-valued fields (indicating default value)
        let i = numFields;
        while (--i >= 0 && vtable[i] === 0) {} // eslint-disable-line no-empty
        const size = i + 1;

        // Write out the current vtable.
        for (; i >= 0; --i) {
          // Offset relative to the start of the table.
          builder.addInt16(vtable[i] ? (vtableOffset - vtable[i]) : 0);
        }

        const standardFields = 2; // size fields
        builder.addInt16(vtableOffset - startOffset);
        const len = (size + standardFields) * SIZEOF_SHORT;
        builder.addInt16(len);

        // Search for an existing vtable that matches the current one.
        let existingTable = 0;
        const { buf, vtables, space: vt1 } = builder;
      outer_loop:
        for (i = 0; i < vtables.length; ++i) {
          const vt2 = buf.length - vtables[i];
          if (len == readInt16(buf, vt2)) {
            for (let j = SIZEOF_SHORT; j < len; j += SIZEOF_SHORT) {
              if (readInt16(buf, vt1 + j) != readInt16(buf, vt2 + j)) {
                continue outer_loop;
              }
            }
            existingTable = vtables[i];
            break;
          }
        }

        if (existingTable) {
          // Found a match: remove the current vtable.
          // Point table to existing vtable.
          builder.space = buf.length - vtableOffset;
          writeInt32(buf, builder.space, existingTable - vtableOffset);
        } else {
          // No match: add the location of the current vtable to the vtables list.
          // Point table to current vtable.
          const off = builder.offset();
          vtables.push(off);
          writeInt32(buf, buf.length - vtableOffset, off - vtableOffset);
        }

        return vtableOffset;
      }
    }
  }

  /**
   * @param {import('./builder.js').Builder} builder
   * @param {import('../types.js').RecordBatch} batch
   * @returns {number}
   */
  function encodeRecordBatch(builder, batch) {
    const { nodes, regions, variadic } = batch;
    const nodeVector = builder.addVector(nodes, 16, 8,
      (builder, node) => {
        builder.writeInt64(node.nullCount);
        builder.writeInt64(node.length);
        return builder.offset();
      }
    );
    const regionVector = builder.addVector(regions, 16, 8,
      (builder, region) => {
        builder.writeInt64(region.length);
        builder.writeInt64(region.offset);
        return builder.offset();
      }
    );
    const variadicVector = builder.addVector(variadic, 8, 8,
      (builder, count) => builder.addInt64(count)
    );
    return builder.addObject(5, b => {
      b.addInt64(0, nodes[0].length, 0);
      b.addOffset(1, nodeVector, 0);
      b.addOffset(2, regionVector, 0);
      // NOT SUPPORTED: 3, compression offset
      b.addOffset(4, variadicVector, 0);
    });
  }

  /**
   * @param {import('./builder.js').Builder} builder
   * @param {import('../types.js').DictionaryBatch} dictionaryBatch
   * @returns {number}
   */
  function encodeDictionaryBatch(builder, dictionaryBatch) {
    const dataOffset = encodeRecordBatch(builder, dictionaryBatch.data);
    return builder.addObject(3, b => {
      b.addInt64(0, dictionaryBatch.id, 0);
      b.addOffset(1, dataOffset, 0);
      b.addInt8(2, +dictionaryBatch.isDelta, 0);
    });
  }

  /**
   * @param {import('./builder.js').Builder} builder
   * @param {Map<string, string>} metadata
   * @returns {number}
   */
  function encodeMetadata(builder, metadata) {
    return metadata?.size > 0
       ? builder.addOffsetVector(Array.from(metadata, ([k, v]) => {
          const key = builder.addString(`${k}`);
          const val = builder.addString(`${v}`);
          return builder.addObject(2, b => {
            b.addOffset(0, key, 0);
            b.addOffset(1, val, 0);
          });
        }))
      : 0;
  }

  /**
   * Encode a data type into a flatbuffer.
   * @param {import('./builder.js').Builder} builder
   * @param {import('../types.js').DataType} type
   * @returns {number} The offset at which the data type is written.
   */
  function encodeDataType(builder, type) {
    const typeId = checkOneOf(type.typeId, Type, invalidDataType);

    switch (typeId) {
      case Type.Dictionary:
        return encodeDictionary(builder, type);
      case Type.Int:
        return encodeInt(builder, type);
      case Type.Float:
        return encodeFloat(builder, type);
      case Type.Decimal:
        return encodeDecimal(builder, type);
      case Type.Date:
        return encodeDate(builder, type);
      case Type.Time:
        return encodeTime(builder, type);
      case Type.Timestamp:
        return encodeTimestamp(builder, type);
      case Type.Interval:
        return encodeInterval(builder, type);
      case Type.Duration:
        return encodeDuration(builder, type);
      case Type.FixedSizeBinary:
      case Type.FixedSizeList:
        return encodeFixedSize(builder, type);
      case Type.Map:
        return encodeMap(builder, type);
      case Type.Union:
        return encodeUnion(builder, type);
    }
    // case Type.Null:
    // case Type.Binary:
    // case Type.LargeBinary:
    // case Type.BinaryView:
    // case Type.Bool:
    // case Type.Utf8:
    // case Type.Utf8View:
    // case Type.LargeUtf8:
    // case Type.List:
    // case Type.ListView:
    // case Type.LargeList:
    // case Type.LargeListView:
    // case Type.RunEndEncoded:
    // case Type.Struct:
    return builder.addObject(0);
  }

  function encodeDate(builder, type) {
    return builder.addObject(1, b => {
      b.addInt16(0, type.unit, DateUnit.MILLISECOND);
    });
  }

  function encodeDecimal(builder, type) {
    return builder.addObject(3, b => {
      b.addInt32(0, type.precision, 0);
      b.addInt32(1, type.scale, 0);
      b.addInt32(2, type.bitWidth, 128);
    });
  }

  function encodeDuration(builder, type) {
    return builder.addObject(1, b => {
      b.addInt16(0, type.unit, TimeUnit.MILLISECOND);
    });
  }

  function encodeFixedSize(builder, type) {
    return builder.addObject(1, b => {
      b.addInt32(0, type.stride, 0);
    });
  }

  function encodeFloat(builder, type) {
    return builder.addObject(1, b => {
      b.addInt16(0, type.precision, Precision.HALF);
    });
  }

  function encodeInt(builder, type) {
    return builder.addObject(2, b => {
      b.addInt32(0, type.bitWidth, 0);
      b.addInt8(1, +type.signed, 0);
    });
  }

  function encodeInterval(builder, type) {
    return builder.addObject(1, b => {
      b.addInt16(0, type.unit, IntervalUnit.YEAR_MONTH);
    });
  }

  function encodeMap(builder, type) {
    return builder.addObject(1, b => {
      b.addInt8(0, +type.keysSorted, 0);
    });
  }

  function encodeTime(builder, type) {
    return builder.addObject(2, b => {
      b.addInt16(0, type.unit, TimeUnit.MILLISECOND);
      b.addInt32(1, type.bitWidth, 32);
    });
  }

  function encodeTimestamp(builder, type) {
    const timezoneOffset = builder.addString(type.timezone);
    return builder.addObject(2, b => {
      b.addInt16(0, type.unit, TimeUnit.SECOND);
      b.addOffset(1, timezoneOffset, 0);
    });
  }

  function encodeUnion(builder, type) {
    const typeIdsOffset = builder.addVector(
      type.typeIds, 4, 4,
      (builder, value) => builder.addInt32(value)
    );
    return builder.addObject(2, b => {
      b.addInt16(0, type.mode, UnionMode.Sparse);
      b.addOffset(1, typeIdsOffset, 0);
    });
  }

  function encodeDictionary(builder, type) {
    const keyTypeOffset = isInt32(type.indices)
      ? 0
      : encodeDataType(builder, type.indices);
    return builder.addObject(4, b => {
      b.addInt64(0, type.id, 0);
      b.addOffset(1, keyTypeOffset, 0);
      b.addInt8(2, +type.ordered, 0);
      // NOT SUPPORTED: 3, dictionaryKind (defaults to dense array)
    });
  }

  function isInt32(type) {
    return type.typeId === Type.Int && type.bitWidth === 32 && type.signed;
  }

  const isLittleEndian = new Uint16Array(new Uint8Array([1, 0]).buffer)[0] === 1;

  /**
   * @param {import('./builder.js').Builder} builder
   * @param {import('../types.js').Schema} schema
   * @returns {number}
   */
  function encodeSchema(builder, schema) {
    const { fields, metadata } = schema;
    const fieldOffsets = fields.map(f => encodeField(builder, f));
    const fieldsVectorOffset = builder.addOffsetVector(fieldOffsets);
    const metadataOffset = encodeMetadata(builder, metadata);
    return builder.addObject(4, b => {
      b.addInt16(0, +(!isLittleEndian), 0);
      b.addOffset(1, fieldsVectorOffset, 0);
      b.addOffset(2, metadataOffset, 0);
      // NOT SUPPORTED: 3, features
    });
  }

  /**
   * @param {import('./builder.js').Builder} builder
   * @param {import('../types.js').Field} field
   * @returns {number}
   */
  function encodeField(builder, field) {
    const { name, nullable, type, metadata } = field;
    let { typeId } = type;

    // encode field data type
    let typeOffset = 0;
    let dictionaryOffset = 0;
    if (typeId !== Type.Dictionary) {
      typeOffset = encodeDataType(builder, type);
    } else {
      const dict = /** @type {import('../types.js').DictionaryType} */ (type).dictionary;
      typeId = dict.typeId;
      dictionaryOffset = encodeDataType(builder, type);
      typeOffset = encodeDataType(builder, dict);
    }

    // encode children, metadata, name, and field object
    // @ts-ignore
    const childOffsets = (type.children || []).map(f => encodeField(builder, f));
    const childrenVectorOffset = builder.addOffsetVector(childOffsets);
    const metadataOffset = encodeMetadata(builder, metadata);
    const nameOffset = builder.addString(name);
    return builder.addObject(7, b => {
      b.addOffset(0, nameOffset, 0);
      b.addInt8(1, +nullable, +false);
      b.addInt8(2, typeId, Type.NONE);
      b.addOffset(3, typeOffset, 0);
      b.addOffset(4, dictionaryOffset, 0);
      b.addOffset(5, childrenVectorOffset, 0);
      b.addOffset(6, metadataOffset, 0);
    });
  }

  /**
   * Write a file footer.
   * @param {import('./builder.js').Builder} builder The binary builder.
   * @param {import('../types.js').Schema} schema The table schema.
   * @param {import('../types.js').Block[]} dictBlocks Dictionary batch file blocks.
   * @param {import('../types.js').Block[]} recordBlocks Record batch file blocks.
   * @param {Map<string,string> | null} metadata File-level metadata.
   */
  function writeFooter(builder, schema, dictBlocks, recordBlocks, metadata) {
    // encode footer flatbuffer
    const metadataOffset = encodeMetadata(builder, metadata);
    const recsOffset = builder.addVector(recordBlocks, 24, 8, encodeBlock);
    const dictsOffset = builder.addVector(dictBlocks, 24, 8, encodeBlock);
    const schemaOffset = encodeSchema(builder, schema);
    builder.finish(
      builder.addObject(5, b => {
        b.addInt16(0, Version.V5, Version.V1);
        b.addOffset(1, schemaOffset, 0);
        b.addOffset(2, dictsOffset, 0);
        b.addOffset(3, recsOffset, 0);
        b.addOffset(4, metadataOffset, 0);
      })
    );
    const size = builder.offset();

    // add eos with continuation indicator
    builder.addInt32(0);
    builder.addInt32(-1);

    // write builder contents
    builder.flush();

    // write file tail
    builder.sink.write(new Uint8Array(Int32Array.of(size).buffer));
    builder.sink.write(MAGIC);
  }

  /**
   * Encode a file pointer block.
   * @param {import('./builder.js').Builder} builder
   * @param {import('../types.js').Block} block
   * @returns {number} the current block offset
   */
  function encodeBlock(builder, { offset, metadataLength, bodyLength }) {
    builder.writeInt64(bodyLength);
    builder.writeInt32(0);
    builder.writeInt32(metadataLength);
    builder.writeInt64(offset);
    return builder.offset();
  }

  /**
   * Write an IPC message to the builder sink.
   * @param {import('./builder.js').Builder} builder
   * @param {import('../types.js').MessageHeader_} headerType
   * @param {number} headerOffset
   * @param {number} bodyLength
   * @param {import('../types.js').Block[]} [blocks]
   */
  function writeMessage(builder, headerType, headerOffset, bodyLength, blocks) {
    builder.finish(
      builder.addObject(5, b => {
        b.addInt16(0, Version.V5, Version.V1);
        b.addInt8(1, headerType, MessageHeader.NONE);
        b.addOffset(2, headerOffset, 0);
        b.addInt64(3, bodyLength, 0);
        // NOT SUPPORTED: 4, message-level metadata
      })
    );

    const prefixSize = 8; // continuation indicator + message size
    const messageSize = builder.offset();
    const alignedSize = (messageSize + prefixSize + 7) & ~7;

    // track blocks for file footer
    blocks?.push({
      offset: builder.outputBytes,
      metadataLength: alignedSize,
      bodyLength
    });

    // write size prefix (including padding)
    builder.addInt32(alignedSize - prefixSize);

    // write the stream continuation indicator
    builder.addInt32(-1);

    // flush the builder content
    builder.flush();

    // add alignment padding as needed
    builder.addPadding(alignedSize - messageSize - prefixSize);
  }

  class Sink {
    /**
     * Write bytes to this sink.
     * @param {Uint8Array} bytes The byte buffer to write.
     */
    write(bytes) { // eslint-disable-line no-unused-vars
    }

    /**
     * Write padding bytes (zeroes) to this sink.
     * @param {number} byteCount The number of padding bytes.
     */
    pad(byteCount) {
      this.write(new Uint8Array(byteCount));
    }

    /**
     * @returns {Uint8Array | null}
     */
    finish() {
      return null;
    }
  }

  class MemorySink extends Sink {
    /**
     * A sink that collects bytes in memory.
     */
    constructor() {
      super();
      this.buffers = [];
    }

    /**
     * Write bytes
     * @param {Uint8Array} bytes
     */
    write(bytes) {
      this.buffers.push(bytes);
    }

    /**
     * @returns {Uint8Array}
     */
    finish() {
      const bufs = this.buffers;
      const size = bufs.reduce((sum, b) => sum + b.byteLength, 0);
      const buf = new Uint8Array(size);
      for (let i = 0, off = 0; i < bufs.length; ++i) {
        buf.set(bufs[i], off);
        off += bufs[i].byteLength;
      }
      return buf;
    }
  }

  const STREAM = 'stream';
  const FILE = 'file';

  /**
   * Encode assembled data into Arrow IPC binary format.
   * @param {any} data Assembled table data.
   * @param {object} options Encoding options.
   * @param {import('./sink.js').Sink} [options.sink] IPC byte consumer.
   * @param {'stream' | 'file'} [options.format] Arrow stream or file format.
   * @returns {import('./sink.js').Sink} The sink that was passed in.
   */
  function encodeIPC(data, { sink, format = STREAM } = {}) {
    if (format !== STREAM && format !== FILE) {
      throw new Error(`Unrecognized Arrow IPC format: ${format}`);
    }
    const { schema, dictionaries = [], records = [], metadata } = data;
    const builder = new Builder(sink || new MemorySink());
    const file = format === FILE;
    const dictBlocks = [];
    const recordBlocks = [];

    if (file) {
      builder.addBuffer(MAGIC);
    } else if (schema) {
      writeMessage(
        builder,
        MessageHeader.Schema,
        encodeSchema(builder, schema),
        0
      );
    }

    for (const dict of dictionaries) {
      const { data } = dict;
      writeMessage(
        builder,
        MessageHeader.DictionaryBatch,
        encodeDictionaryBatch(builder, dict),
        data.byteLength,
        dictBlocks
      );
      writeBuffers(builder, data.buffers);
    }

    for (const batch of records) {
      writeMessage(
        builder,
        MessageHeader.RecordBatch,
        encodeRecordBatch(builder, batch),
        batch.byteLength,
        recordBlocks
      );
      writeBuffers(builder, batch.buffers);
    }

    if (file) {
      writeFooter(builder, schema, dictBlocks, recordBlocks, metadata);
    }

    return builder.sink;
  }

  /**
   * Write byte buffers to the builder sink.
   * Buffers are aligned to 64 bits (8 bytes) as needed.
   * @param {import('./builder.js').Builder} builder
   * @param {Uint8Array[]} buffers
   */
  function writeBuffers(builder, buffers) {
    for (let i = 0; i < buffers.length; ++i) {
      builder.addBuffer(buffers[i]); // handles alignment for us
    }
  }

  /**
   * Encode an Arrow table into Arrow IPC binary format.
   * @param {import('../table.js').Table} table The Arrow table to encode.
   * @param {object} options Encoding options.
   * @param {import('./sink.js').Sink} [options.sink] IPC byte consumer.
   * @param {'stream' | 'file'} [options.format] Arrow stream or file format.
   * @returns {Uint8Array | null} The generated bytes (for an in-memory sink)
   *  or null (if using a sink that writes bytes elsewhere).
   */
  function tableToIPC(table, options) {
    // accept a format string option for Arrow-JS compatibility
    if (typeof options === 'string') {
      options = { format: options };
    }

    const columns = table.children;
    checkBatchLengths(columns);

    const { dictionaries, idMap } = assembleDictionaryBatches(columns);
    const records = assembleRecordBatches(columns);
    const schema = assembleSchema(table.schema, idMap);
    const data = { schema, dictionaries, records };
    return encodeIPC(data, options).finish();
  }

  function checkBatchLengths(columns) {
    const n = columns[0]?.data.map(d => d.length);
    columns.forEach(({ data }) => {
      if (data.length !== n.length || data.some((b, i) => b.length !== n[i])) {
        throw new Error('Columns have inconsistent batch sizes.');
      }
    });
  }

  /**
   * Create a new assembly context.
   */
  function assembleContext() {
    let byteLength = 0;
    const nodes = [];
    const regions = [];
    const buffers = [];
    const variadic = [];
    return {
      /**
       * @param {number} length
       * @param {number} nullCount
       */
      node(length, nullCount) {
        nodes.push({ length, nullCount });
      },
      /**
       * @param {import('../types.js').TypedArray} b
       */
      buffer(b) {
        const size = b.byteLength;
        const length = ((size + 7) & ~7);
        regions.push({ offset: byteLength, length });
        byteLength += length;
        buffers.push(new Uint8Array(b.buffer, b.byteOffset, size));
      },
      /**
       * @param {number} length
       */
      variadic(length) {
        variadic.push(length);
      },
      /**
       * @param {import('../types.js').DataType} type
       * @param {import('../batch.js').Batch} batch
       */
      children(type, batch) {
        // @ts-ignore
        type.children.forEach((field, index) => {
          visit(field.type, batch.children[index], this);
        });
      },
      /**
       * @returns {import('../types.js').RecordBatch}
       */
      done() {
        return { byteLength, nodes, regions, variadic, buffers };
      }
    };
  }

  /**
   * Assemble dictionary batches and their unique ids.
   * @param {import('../column.js').Column[]} columns The table columns.
   * @returns {{
   *    dictionaries: import('../types.js').DictionaryBatch[],
   *    idMap: Map<import('../types.js').DataType, number>
   *  }}
   *  The assembled dictionary batches and a map from dictionary column
   *  instances to dictionary ids.
   */
  function assembleDictionaryBatches(columns) {
    const dictionaries = [];
    const dictMap = new Map;
    const idMap = new Map;
    let id = -1;

    // track dictionaries, key by dictionary column, assign ids
    const visitor = dictionaryColumn => {
      if (!dictMap.has(dictionaryColumn)) {
        dictMap.set(dictionaryColumn, ++id);
        for (let i = 0; i < dictionaryColumn.data.length; ++i) {
          dictionaries.push({
            id,
            isDelta: i > 0,
            data: assembleRecordBatch([dictionaryColumn], i)
          });
        }
        idMap.set(dictionaryColumn.type, id);
      } else {
        idMap.set(dictionaryColumn.type, dictMap.get(dictionaryColumn));
      }
    };

    // recurse through column batches to find dictionaries
    // it is sufficient to visit the first batch only,
    // as all batches have the same dictionary column
    columns.forEach(col => visitDictionaries(col.data[0], visitor));

    return { dictionaries, idMap };
  }

  /**
   * Traverse column batches to visit dictionary columns.
   * @param {import('../batch.js').Batch} batch
   * @param {(column: import('../column.js').Column) => void} visitor
   */
  function visitDictionaries(batch, visitor) {
    if (batch?.type.typeId === Type.Dictionary) {
      // @ts-ignore - batch has type DictionaryBatch
      const dictionary = batch.dictionary;
      visitor(dictionary);
      visitDictionaries(dictionary.data[0], visitor);
    }
    batch?.children?.forEach(child => visitDictionaries(child, visitor));
  }

  /**
   * Assemble a schema with resolved dictionary ids.
   * @param {import('../types.js').Schema} schema The schema.
   * @param {Map<import('../types.js').DataType, number>} idMap A map
   *  from dictionary value types to dictionary ids.
   * @returns {import('../types.js').Schema} A new schema with resolved
   *  dictionary ids. If there are no dictionaries, the input schema is
   *  returned unchanged.
   */
  function assembleSchema(schema, idMap) {
    // early exit if no dictionaries
    if (!idMap.size) return schema;

    const visit = type => {
      if (type.typeId === Type.Dictionary) {
        type.id = idMap.get(type.dictionary); // lookup and set id
        visitDictType(type);
      }
      if (type.children) {
        (type.children = type.children.slice()).forEach(visitFields);
      }
    };

    // visit a field in a field array
    const visitFields = (field, index, array) => {
      const type = { ...field.type };
      array[index] = { ...field, type };
      visit(type);
    };

    // visit a dictionary values type
    const visitDictType = (parentType) => {
      const type = { ...parentType.dictionary };
      parentType.dictionary = type;
      visit(type);
    };

    schema = { ...schema, fields: schema.fields.slice() };
    schema.fields.forEach(visitFields);
    return schema;
  }

  /**
   * Assemble record batches with marshalled buffers.
   * @param {import('../column.js').Column[]} columns The table columns.
   * @returns {import('../types.js').RecordBatch[]} The assembled record batches.
   */
  function assembleRecordBatches(columns) {
    return (columns[0]?.data || [])
      .map((_, index) => assembleRecordBatch(columns, index));
  }

  /**
   * Assemble a record batch with marshalled buffers.
   * @param {import('../column.js').Column[]} columns The table columns.
   * @param {number} batchIndex The batch index.
   * @returns {import('../types.js').RecordBatch} The assembled record batch.
   */
  function assembleRecordBatch(columns, batchIndex = 0) {
    const ctx = assembleContext();
    columns.forEach(column => {
      visit(column.type, column.data[batchIndex], ctx);
    });
    return ctx.done();
  }

  /**
   * Visit a column batch, assembling buffer data.
   * @param {import('../types.js').DataType} type The data type.
   * @param {import('../batch.js').Batch} batch The column batch.
   * @param {ReturnType<assembleContext>} ctx The assembly context.
   */
  function visit(type, batch, ctx) {
    const { typeId } = type;

    // no field node, no buffers
    if (typeId === Type.Null) return;

    // record field node info
    ctx.node(batch.length, batch.nullCount);

    switch (typeId) {
      // validity and value buffers
      // backing dictionaries handled elsewhere
      case Type.Bool:
      case Type.Int:
      case Type.Time:
      case Type.Duration:
      case Type.Float:
      case Type.Date:
      case Type.Timestamp:
      case Type.Decimal:
      case Type.Interval:
      case Type.FixedSizeBinary:
      case Type.Dictionary: // dict key values
        ctx.buffer(batch.validity);
        ctx.buffer(batch.values);
        return;

      // validity, offset, and value buffers
      case Type.Utf8:
      case Type.LargeUtf8:
      case Type.Binary:
      case Type.LargeBinary:
        ctx.buffer(batch.validity);
        ctx.buffer(batch.offsets);
        ctx.buffer(batch.values);
        return;

      // views with variadic buffers
      case Type.BinaryView:
      case Type.Utf8View:
        ctx.buffer(batch.validity);
        ctx.buffer(batch.values);
        // @ts-ignore
        ctx.variadic(batch.data.length);
        // @ts-ignore
        batch.data.forEach(b => ctx.buffer(b));
        return;

      // validity, offset, and list child
      case Type.List:
      case Type.LargeList:
      case Type.Map:
        ctx.buffer(batch.validity);
        ctx.buffer(batch.offsets);
        ctx.children(type, batch);
        return;

      // validity, offset, size, and list child
      case Type.ListView:
      case Type.LargeListView:
        ctx.buffer(batch.validity);
        ctx.buffer(batch.offsets);
        ctx.buffer(batch.sizes);
        ctx.children(type, batch);
        return;

      // validity and children
      case Type.FixedSizeList:
      case Type.Struct:
        ctx.buffer(batch.validity);
        ctx.children(type, batch);
        return;

      // children only
      case Type.RunEndEncoded:
        ctx.children(type, batch);
        return;

      // union
      case Type.Union: {
        // @ts-ignore
        ctx.buffer(batch.typeIds);
        if (type.mode === UnionMode.Dense) {
          ctx.buffer(batch.offsets);
        }
        ctx.children(type, batch);
        return;
      }

      // unsupported type
      default:
        throw new Error(invalidDataType(typeId));
    }
  }

  /**
   * Create a new resizable buffer instance.
   * @param {import('../types.js').TypedArrayConstructor} [arrayType]
   *  The array type.
   * @returns {Buffer} The buffer.
   */
  function buffer(arrayType) {
    return new Buffer(arrayType);
  }

  /**
   * Resizable byte buffer.
   */
  class Buffer {
    /**
     * Create a new resizable buffer instance.
     * @param {import('../types.js').TypedArrayConstructor} arrayType
     */
    constructor(arrayType = uint8Array) {
      this.buf = new arrayType(512);
    }
    /**
     * Return the underlying data as a 64-bit aligned array of minimum size.
     * @param {number} size The desired minimum array size.
     * @returns {import('../types.js').TypedArray} The 64-bit aligned array.
     */
    array(size) {
      return align(this.buf, size);
    }
    /**
     * Prepare for writes to the given index, resizing as necessary.
     * @param {number} index The array index to prepare to write to.
     */
    prep(index) {
      if (index >= this.buf.length) {
        this.buf = grow(this.buf, index);
      }
    }
    /**
     * Return the value at the given index.
     * @param {number} index The array index.
     */
    get(index) {
      return this.buf[index];
    }
    /**
     * Set a value at the given index.
     * @param {number | bigint} value The value to set.
     * @param {number} index The index to write to.
     */
    set(value, index) {
      this.prep(index);
      this.buf[index] = value;
    }
    /**
     * Write a byte array at the given index. The method should be called
     * only when the underlying buffer is of type Uint8Array.
     * @param {Uint8Array} bytes The byte array.
     * @param {number} index The starting index to write to.
     */
    write(bytes, index) {
      this.prep(index + bytes.length);
      /** @type {Uint8Array} */ (this.buf).set(bytes, index);
    }
  }

  /**
   * Create a new resizable bitmap instance.
   * @returns {Bitmap} The bitmap buffer.
   */
  function bitmap() {
    return new Bitmap();
  }

  /**
   * Resizable bitmap buffer.
   */
  class Bitmap extends Buffer {
    /**
     * Set a bit to true at the given bitmap index.
     * @param {number} index The index to write to.
     */
    set(index) {
      const i = index >> 3;
      this.prep(i);
      /** @type {Uint8Array} */ (this.buf)[i] |= (1 << (index % 8));
    }
  }

  /**
   * Abstract class for building a column data batch.
   */
  class BatchBuilder {
    constructor(type, ctx) {
      this.type = type;
      this.ctx = ctx;
      this.batchClass = ctx.batchType(type);
    }

    /**
     * Initialize the builder state.
     * @returns {this} This builder.
     */
    init() {
      this.index = -1;
      return this;
    }

    /**
     * Write a value to the builder.
     * @param {*} value
     * @param {number} index
     * @returns {boolean | void}
     */
    set(value, index) {
      this.index = index;
      return false;
    }

    /**
     * Returns a batch constructor options object.
     * Used internally to marshal batch data.
     * @returns {Record<string, any>}
     */
    done() {
      return null;
    }

    /**
     * Returns a completed batch and reinitializes the builder state.
     * @returns {import('../../batch.js').Batch}
     */
    batch() {
      const b = new this.batchClass(this.done());
      this.init();
      return b;
    }
  }

  /**
   * Builder for validity bitmaps within batches.
   */
  class ValidityBuilder extends BatchBuilder {
    constructor(type, ctx) {
      super(type, ctx);
    }

    init() {
      this.nullCount = 0;
      this.validity = bitmap();
      return super.init();
    }

    /**
     * @param {*} value
     * @param {number} index
     * @returns {boolean | void}
     */
    set(value, index) {
      this.index = index;
      const isValid = value != null;
      if (isValid) {
        this.validity.set(index);
      } else {
        this.nullCount++;
      }
      return isValid;
    }

    done() {
      const { index, nullCount, type, validity } = this;
      return {
        length: index + 1,
        nullCount,
        type,
        validity: nullCount
          ? validity.array((index >> 3) + 1)
          : new uint8Array(0)
      };
    }
  }

  /**
   * Create a context object for managing dictionary builders.
   */
  function dictionaryContext() {
    const idMap = new Map;
    const dicts = new Set;
    return {
      /**
       * Get a dictionary values builder for the given dictionary type.
       * @param {import('../../types.js').DictionaryType} type
       *  The dictionary type.
       * @param {*} ctx The builder context.
       * @returns {ReturnType<dictionaryValues>}
       */
      get(type, ctx) {
        // if a dictionary has a non-negative id, assume it was set
        // intentionally and track it for potential reuse across columns
        // otherwise the dictionary is used for a single column only
        const id = type.id;
        if (id >= 0 && idMap.has(id)) {
          return idMap.get(id);
        } else {
          const dict = dictionaryValues(type, ctx);
          if (id >= 0) idMap.set(id, dict);
          dicts.add(dict);
          return dict;
        }
      },
      /**
       * Finish building dictionary values columns and assign them to
       * their corresponding dictionary batches.
       * @param {import('../../types.js').ExtractionOptions} options
       */
      finish(options) {
        dicts.forEach(dict => dict.finish(options));
      }
    };
  }

  /**
   * Builder helper for creating dictionary values.
   * @param {import('../../types.js').DictionaryType} type
   *  The dictionary data type.
   * @param {ReturnType<import('../builder.js').builderContext>} ctx
   *  The builder context.
   */
  function dictionaryValues(type, ctx) {
    const keys = Object.create(null);
    const values = ctx.builder(type.dictionary);
    const batches = [];

    values.init();
    let index = -1;

    return {
      type,
      values,

      add(batch) {
        batches.push(batch);
        return batch;
      },

      key(value) {
        const v = keyString(value);
        let k = keys[v];
        if (k === undefined) {
          keys[v] = k = ++index;
          values.set(value, k);
        }
        return k;
      },

      finish(options) {
        const valueType = type.dictionary;
        const batch = new (batchType(valueType, options))(values.done());
        const dictionary = new Column([batch]);
        batches.forEach(batch => batch.setDictionary(dictionary));
      }
    };
  }

  /**
   * Builder for dictionary-typed data batches.
   */
  class DictionaryBuilder extends ValidityBuilder {
    constructor(type, ctx) {
      super(type, ctx);
      this.dict = ctx.dictionary(type);
    }

    init() {
      this.values = buffer(this.type.indices.values);
      return super.init();
    }

    set(value, index) {
      if (super.set(value, index)) {
        this.values.set(this.dict.key(value), index);
      }
    }

    done() {
      return {
        ...super.done(),
        values: this.values.array(this.index + 1)
      };
    }

    batch() {
      // register batch with dictionary
      // batch will be updated when the dictionary is finished
      return this.dict.add(super.batch());
    }
  }

  /**
   * Infer the data type for a given input array.
   * @param {(visitor: (value: any) => void) => void} visit
   *  A function that applies a callback to successive data values.
   * @returns {import('../types.js').DataType} The data type.
   */
  function inferType(visit) {
    const profile = profiler();
    visit(value => profile.add(value));
    return profile.type();
  }

  function profiler() {
    let length = 0;
    let nullCount = 0;
    let boolCount = 0;
    let numberCount = 0;
    let intCount = 0;
    let bigintCount = 0;
    let dateCount = 0;
    let dayCount = 0;
    let stringCount = 0;
    let arrayCount = 0;
    let structCount = 0;
    let min = Infinity;
    let max = -Infinity;
    let minLength = Infinity;
    let maxLength = -Infinity;
    let minBigInt;
    let maxBigInt;
    let arrayProfile;
    let structProfiles = {};

    return {
      add(value) {
        length++;
        if (value == null) {
          nullCount++;
          return;
        }
        switch (typeof value) {
          case 'string':
            stringCount++;
            break;
          case 'number':
            numberCount++;
            if (value < min) min = value;
            if (value > max) max = value;
            if (Number.isInteger(value)) intCount++;
            break;
          case 'bigint':
            bigintCount++;
            if (minBigInt === undefined) {
              minBigInt = maxBigInt = value;
            } else {
              if (value < minBigInt) minBigInt = value;
              if (value > maxBigInt) maxBigInt = value;
            }
            break;
          case 'boolean':
            boolCount++;
            break;
          case 'object':
            if (value instanceof Date) {
              dateCount++;
              // 1 day = 1000ms * 60s * 60min * 24hr = 86400000
              if ((+value % 864e5) === 0) dayCount++;
            } else if (isArray(value)) {
              arrayCount++;
              const len = value.length;
              if (len < minLength) minLength = len;
              if (len > maxLength) maxLength = len;
              arrayProfile ??= profiler();
              value.forEach(arrayProfile.add);
            } else {
              structCount++;
              for (const key in value) {
                const fieldProfiler = structProfiles[key]
                  ?? (structProfiles[key] = profiler());
                fieldProfiler.add(value[key]);
              }
            }
        }
      },
      type() {
        const valid = length - nullCount;
        return valid === 0 ? nullType()
          : intCount === valid ? intType(min, max)
          : numberCount === valid ? float64()
          : bigintCount === valid ? bigintType(minBigInt, maxBigInt)
          : boolCount === valid ? bool()
          : dayCount === valid ? dateDay()
          : dateCount === valid ? timestamp()
          : stringCount === valid ? dictionary$1(utf8())
          : arrayCount === valid ? arrayType(arrayProfile.type(), minLength, maxLength)
          : structCount === valid ? struct(
              Object.entries(structProfiles).map(_ => field(_[0], _[1].type()))
            )
          : unionType();
      }
    };
  }

  /**
   * Return a list or fixed list type.
   * @param {import('../types.js').DataType} type The child data type.
   * @param {number} minLength The minumum list length.
   * @param {number} maxLength The maximum list length.
   * @returns {import('../types.js').DataType} The data type.
   */
  function arrayType(type, minLength, maxLength) {
    return maxLength === minLength
      ? fixedSizeList(type, minLength)
      : list(type);
  }

  /**
   * @param {number} min
   * @param {number} max
   * @returns {import('../types.js').DataType}
   */
  function intType(min, max) {
    const v = Math.max(Math.abs(min) - 1, max);
    return v < (1 << 7) ? int8()
      : v < (1 << 15) ? int16()
      : v < (2 ** 31) ? int32()
      : float64();
  }

  /**
   * @param {bigint} min
   * @param {bigint} max
   * @returns {import('../types.js').IntType}
   */
  function bigintType(min, max) {
    const v = -min > max ? -min - 1n : max;
    if (v >= 2 ** 63) {
      throw new Error(`BigInt exceeds 64 bits: ${v}`);
    }
    return int64();
  }

  /**
   * @returns {import('../types.js').UnionType}
   */
  function unionType() {
    throw new Error('Mixed types detected, please define a union type.');
  }

  /**
   * Builder for batches of binary-typed data.
   */
  class BinaryBuilder extends ValidityBuilder {
    constructor(type, ctx) {
      super(type, ctx);
      this.toOffset = toOffset(type.offsets);
    }

    init() {
      this.offsets = buffer(this.type.offsets);
      this.values = buffer();
      this.pos = 0;
      return super.init();
    }

    set(value, index) {
      const { offsets, values, toOffset } = this;
      if (super.set(value, index)) {
        values.write(value, this.pos);
        this.pos += value.length;
      }
      offsets.set(toOffset(this.pos), index + 1);
    }

    done() {
      return {
        ...super.done(),
        offsets: this.offsets.array(this.index + 2),
        values: this.values.array(this.pos + 1)
      };
    }
  }

  /**
   * Builder for batches of bool-typed data.
   */
  class BoolBuilder extends ValidityBuilder {
    constructor(type, ctx) {
      super(type, ctx);
    }

    init() {
      this.values = bitmap();
      return super.init();
    }

    set(value, index) {
      super.set(value, index);
      if (value) this.values.set(index);
    }

    done() {
      return {
        ...super.done(),
        values: this.values.array((this.index >> 3) + 1)
      }
    }
  }

  /**
   * Builder for batches of decimal-typed data.
   */
  class DecimalBuilder extends ValidityBuilder {
    constructor(type, ctx) {
      super(type, ctx);
      this.scale = 10 ** type.scale;
      this.stride = type.bitWidth >> 6;
    }

    init() {
      this.values = buffer(this.type.values);
      return super.init();
    }

    set(value, index) {
      const { scale, stride, values } = this;
      if (super.set(value, index)) {
        values.prep((index + 1) * stride);
        // @ts-ignore
        toDecimal(value, values.buf, index * stride, stride, scale);
      }
    }

    done() {
      const { index, stride, values } = this;
      return {
        ...super.done(),
        values: values.array((index + 1) * stride)
      };
    }
  }

  /**
   * Builder for fixed-size-binary-typed data batches.
   */
  class FixedSizeBinaryBuilder extends ValidityBuilder {
    constructor(type, ctx) {
      super(type, ctx);
      this.stride = type.stride;
    }

    init() {
      this.values = buffer();
      return super.init();
    }

    set(value, index) {
      if (super.set(value, index)) {
        this.values.write(value, index * this.stride);
      }
    }

    done() {
      const { stride, values } = this;
      return {
        ...super.done(),
        values: values.array(stride * (this.index + 1))
      };
    }
  }

  /**
   * Builder for fixed-size-list-typed data batches.
   */
  class FixedSizeListBuilder extends ValidityBuilder {
    constructor(type, ctx) {
      super(type, ctx);
      this.child = ctx.builder(this.type.children[0].type);
      this.stride = type.stride;
    }

    init() {
      this.child.init();
      return super.init();
    }

    set(value, index) {
      const { child, stride } = this;
      const base = index * stride;
      if (super.set(value, index)) {
        for (let i = 0; i < stride; ++i) {
          child.set(value[i], base + i);
        }
      } else {
        child.index = base + stride;
      }
    }

    done() {
      const { child } = this;
      return {
        ...super.done(),
        children: [ child.batch() ]
      };
    }
  }

  /**
   * Builder for day/time interval-typed data batches.
   */
  class IntervalDayTimeBuilder extends ValidityBuilder {
    init() {
      this.values = buffer(this.type.values);
      return super.init();
    }

    set(value, index) {
      if (super.set(value, index)) {
        const i = index << 1;
        this.values.set(value[0], i);
        this.values.set(value[1], i + 1);
      }
    }

    done() {
      return {
        ...super.done(),
        values: this.values.array((this.index + 1) << 1)
      }
    }
  }

  /**
   * Builder for month/day/nano interval-typed data batches.
   */
  class IntervalMonthDayNanoBuilder extends ValidityBuilder {
    init() {
      this.values = buffer();
      return super.init();
    }

    set(value, index) {
      if (super.set(value, index)) {
        this.values.write(toMonthDayNanoBytes(value), index << 4);
      }
    }

    done() {
      return {
        ...super.done(),
        values: this.values.array((this.index + 1) << 4)
      }
    }
  }

  /**
   * Abstract class for building list data batches.
   */
  class AbstractListBuilder extends ValidityBuilder {
    constructor(type, ctx, child) {
      super(type, ctx);
      this.child = child;
    }

    init() {
      this.child.init();
      const offsetType = this.type.offsets;
      this.offsets = buffer(offsetType);
      this.toOffset = toOffset(offsetType);
      this.pos = 0;
      return super.init();
    }

    done() {
      return {
        ...super.done(),
        offsets: this.offsets.array(this.index + 2),
        children: [ this.child.batch() ]
      };
    }
  }

  /**
   * Builder for list-typed data batches.
   */
  class ListBuilder extends AbstractListBuilder {
    constructor(type, ctx) {
      super(type, ctx, ctx.builder(type.children[0].type));
    }

    set(value, index) {
      const { child, offsets, toOffset } = this;
      if (super.set(value, index)) {
        value.forEach(v => child.set(v, this.pos++));
      }
      offsets.set(toOffset(this.pos), index + 1);
    }
  }

  /**
   * Abstract class for building list-typed data batches.
   */
  class AbstractStructBuilder extends ValidityBuilder {
    constructor(type, ctx) {
      super(type, ctx);
      this.children = type.children.map(c => ctx.builder(c.type));
    }

    init() {
      this.children.forEach(c => c.init());
      return super.init();
    }

    done() {
      const { children } = this;
      children.forEach(c => c.index = this.index);
      return {
        ...super.done(),
        children: children.map(c => c.batch())
      };
    }
  }

  /**
   * Builder for struct-typed data batches.
   */
  class StructBuilder extends AbstractStructBuilder {
    constructor(type, ctx) {
      super(type, ctx);
      this.setters = this.children.map((child, i) => {
        const name = type.children[i].name;
        return (value, index) => child.set(value?.[name], index);
      });
    }

    set(value, index) {
      super.set(value, index);
      const setters = this.setters;
      for (let i = 0; i < setters.length; ++i) {
        setters[i](value, index);
      }
    }
  }

  /**
   * Builder for map-typed data batches.
   */
  class MapBuilder extends AbstractListBuilder {
    constructor(type, ctx) {
      super(type, ctx, new MapStructBuilder(type.children[0].type, ctx));
    }

    set(value, index) {
      const { child, offsets, toOffset } = this;
      if (super.set(value, index)) {
        for (const keyValuePair of value) {
          child.set(keyValuePair, this.pos++);
        }
      }
      offsets.set(toOffset(this.pos), index + 1);
    }
  }

  /**
   * Builder for key-value struct batches within a map.
   */
  class MapStructBuilder extends AbstractStructBuilder {
    set(value, index) {
      super.set(value, index);
      const [key, val] = this.children;
      key.set(value[0], index);
      val.set(value[1], index);
    }
  }

  const NO_VALUE = {}; // empty object that fails strict equality

  /**
   * Builder for run-end-encoded-typed data batches.
   */
  class RunEndEncodedBuilder extends BatchBuilder {
    constructor(type, ctx) {
      super(type, ctx);
      this.children = type.children.map(c => ctx.builder(c.type));
    }

    init() {
      this.pos = 0;
      this.key = null;
      this.value = NO_VALUE;
      this.children.forEach(c => c.init());
      return super.init();
    }

    next() {
      const [runs, vals] = this.children;
      runs.set(this.index + 1, this.pos);
      vals.set(this.value, this.pos++);
    }

    set(value, index) {
      // perform fast strict equality test
      if (value !== this.value) {
        // if no match, fallback to key string test
        const key = keyString(value);
        if (key !== this.key) {
          // if key doesn't match, write prior run and update
          if (this.key) this.next();
          this.key = key;
          this.value = value;
        }
      }
      this.index = index;
    }

    done() {
      this.next();
      const { children, index, type } = this;
      return {
        length: index + 1,
        nullCount: 0,
        type,
        children: children.map(c => c.batch())
      };
    }
  }

  /**
   * Abstract class for building union-typed data batches.
   */
  class AbstractUnionBuilder extends BatchBuilder {
    constructor(type, ctx) {
      super(type, ctx);
      this.children = type.children.map(c => ctx.builder(c.type));
      this.typeMap = type.typeMap;
      this.lookup = type.typeIdForValue;
    }

    init() {
      this.nullCount = 0;
      this.typeIds = buffer(int8Array);
      this.children.forEach(c => c.init());
      return super.init();
    }

    set(value, index) {
      const { children, lookup, typeMap, typeIds } = this;
      this.index = index;
      const typeId = lookup(value, index);
      const child = children[typeMap[typeId]];
      typeIds.set(typeId, index);
      if (value == null) ++this.nullCount;
      // @ts-ignore
      this.update(value, index, child);
    }

    done() {
      const { children, nullCount, type, typeIds } = this;
      const length = this.index + 1;
      return {
        length,
        nullCount,
        type,
        typeIds: typeIds.array(length),
        children: children.map(c => c.batch())
      };
    }
  }

  /**
   * Builder for sparse union-typed data batches.
   */
  class SparseUnionBuilder extends AbstractUnionBuilder {
    update(value, index, child) {
      // update selected child with value
      // then set all other children to null
      child.set(value, index);
      this.children.forEach(c => { if (c !== child) c.set(null, index); });
    }
  }

  /**
   * Builder for dense union-typed data batches.
   */
  class DenseUnionBuilder extends AbstractUnionBuilder {
    init() {
      this.offsets = buffer(this.type.offsets);
      return super.init();
    }

    update(value, index, child) {
      const offset = child.index + 1;
      child.set(value, offset);
      this.offsets.set(offset, index);
    }

    done() {
      return {
        ...super.done(),
        offsets: this.offsets.array(this.index + 1)
      };
    }
  }

  /**
   * Builder for utf8-typed data batches.
   */
  class Utf8Builder extends BinaryBuilder {
    set(value, index) {
      super.set(value && encodeUtf8(value), index);
    }
  }

  /**
   * Builder for data batches that can be accessed directly as typed arrays.
   */
  class DirectBuilder extends ValidityBuilder {
    constructor(type, ctx) {
      super(type, ctx);
      this.values = buffer(type.values);
    }

    init() {
      this.values = buffer(this.type.values);
      return super.init();
    }

    /**
     * @param {*} value
     * @param {number} index
     * @returns {boolean | void}
     */
    set(value, index) {
      if (super.set(value, index)) {
        this.values.set(value, index);
      }
    }
    done() {
      return {
        ...super.done(),
        values: this.values.array(this.index + 1)
      };
    }
  }

  /**
   * Builder for int64/uint64 data batches written as bigints.
   */
  class Int64Builder extends DirectBuilder {
    set(value, index) {
      super.set(value == null ? value : toBigInt(value), index);
    }
  }

  /**
   * Builder for data batches whose values must pass through a transform
   * function prior to be written to a backing buffer.
   */
  class TransformBuilder extends DirectBuilder {
    constructor(type, ctx, transform) {
      super(type, ctx);
      this.transform = transform;
    }
    set(value, index) {
      super.set(value == null ? value : this.transform(value), index);
    }
  }

  /**
   * Create a context object for shared builder state.
   * @param {import('../types.js').ExtractionOptions} [options]
   *  Batch extraction options.
  * @param {ReturnType<dictionaryContext>} [dictionaries]
   *  Context object for tracking dictionaries.
   */
  function builderContext(
    options = {},
    dictionaries = dictionaryContext()
  ) {
    return {
      batchType: type => batchType(type, options),
      builder(type) { return builder(type, this); },
      dictionary(type) { return dictionaries.get(type, this); },
      finish: () => dictionaries.finish(options)
    };
  }

  /**
   * Returns a batch builder for the given type and builder context.
   * @param {import('../types.js').DataType} type A data type.
   * @param {ReturnType<builderContext>} [ctx] A builder context.
   * @returns {import('./builders/batch.js').BatchBuilder}
   */
  function builder(type, ctx = builderContext()) {
    const { typeId } = type;
    switch (typeId) {
      case Type.Int:
      case Type.Time:
      case Type.Duration:
        return isInt64ArrayType(type.values)
          ? new Int64Builder(type, ctx)
          : new DirectBuilder(type, ctx);
      case Type.Float:
        return type.precision
          ? new DirectBuilder(type, ctx)
          : new TransformBuilder(type, ctx, toFloat16)
      case Type.Binary:
      case Type.LargeBinary:
        return new BinaryBuilder(type, ctx);
      case Type.Utf8:
      case Type.LargeUtf8:
        return new Utf8Builder(type, ctx);
      case Type.Bool:
        return new BoolBuilder(type, ctx);
      case Type.Decimal:
        return new DecimalBuilder(type, ctx);
      case Type.Date:
        return new TransformBuilder(type, ctx, type.unit ? toBigInt : toDateDay);
      case Type.Timestamp:
        return new TransformBuilder(type, ctx, toTimestamp(type.unit));
      case Type.Interval:
        switch (type.unit) {
          case IntervalUnit.DAY_TIME:
            return new IntervalDayTimeBuilder(type, ctx);
          case IntervalUnit.MONTH_DAY_NANO:
            return new IntervalMonthDayNanoBuilder(type, ctx);
        }
        // IntervalUnit.YEAR_MONTH:
        return new DirectBuilder(type, ctx);
      case Type.List:
      case Type.LargeList:
        return new ListBuilder(type, ctx);
      case Type.Struct:
        return new StructBuilder(type, ctx);
      case Type.Union:
        return type.mode
          ? new DenseUnionBuilder(type, ctx)
          : new SparseUnionBuilder(type, ctx);
      case Type.FixedSizeBinary:
        return new FixedSizeBinaryBuilder(type, ctx);
      case Type.FixedSizeList:
        return new FixedSizeListBuilder(type, ctx);
      case Type.Map:
        return new MapBuilder(type, ctx);
      case Type.RunEndEncoded:
        return new RunEndEncodedBuilder(type, ctx);

      case Type.Dictionary:
        return new DictionaryBuilder(type, ctx);
    }
    // case Type.BinaryView:
    // case Type.Utf8View:
    // case Type.ListView:
    // case Type.LargeListView:
    throw new Error(invalidDataType(typeId));
  }

  /**
   * Create a new column by iterating over provided values.
   * @template T
   * @param {Iterable | ((callback: (value: any) => void) => void)} values
   *  Either an iterable object or a visitor function that applies a callback
   *  to successive data values (akin to Array.forEach).
   * @param {import('../types.js').DataType} [type] The data type.
   * @param {import('../types.js').ColumnBuilderOptions} [options]
   *  Builder options for the generated column.
   * @param {ReturnType<
   *    import('./builders/dictionary.js').dictionaryContext
   *  >} [dicts] Dictionary context object, for internal use only.
   * @returns {Column<T>} The generated column.
   */
  function columnFromValues(values, type, options = {}, dicts) {
    const visit = isIterable(values)
      ? callback => { for (const value of values) callback(value); }
      : values;

    type ??= inferType(visit);
    const { maxBatchRows = Infinity, ...opt } = options;
    let data;

    if (type.typeId === Type.Null) {
      let length = 0;
      visit(() => ++length);
      data = nullBatches(type, length, maxBatchRows);
    } else {
      const ctx = builderContext(opt, dicts);
      const b = builder(type, ctx).init();
      const next = b => data.push(b.batch());
      data = [];

      let row = 0;
      visit(value => {
        b.set(value, row++);
        if (row >= maxBatchRows) {
          next(b);
          row = 0;
        }
      });
      if (row) next(b);

      // resolve dictionaries
      ctx.finish();
    }

    return new Column(data, type);
  }

  /**
   * Create null batches with the given batch size limit.
   * @param {import('../types.js').NullType} type The null data type.
   * @param {number} length The total column length.
   * @param {number} limit The maximum batch size.
   * @returns {import('../batch.js').NullBatch[]} The null batches.
   */
  function nullBatches(type, length, limit) {
    const data = [];
    const batch = length => new NullBatch({ length, nullCount: length, type });
    const numBatches = Math.floor(length / limit);
    for (let i = 0; i < numBatches; ++i) {
      data.push(batch(limit));
    }
    const rem = length % limit;
    if (rem) data.push(batch(rem));
    return data;
  }

  /**
   * Create a new column from a provided data array.
   * @template T
   * @param {Array | import('../types.js').TypedArray} array The input data.
   * @param {import('../types.js').DataType} [type] The data type.
   *  If not specified, type inference is attempted.
   * @param {import('../types.js').ColumnBuilderOptions} [options]
   *  Builder options for the generated column.
   * @param {ReturnType<import('./builders/dictionary.js').dictionaryContext>} [dicts]
   *  Builder context object, for internal use only.
   * @returns {Column<T>} The generated column.
   */
  function columnFromArray(array, type, options = {}, dicts) {
    return !type && isTypedArray(array)
      ? columnFromTypedArray(array, options)
      : columnFromValues(v => array.forEach(v), type, options, dicts);
  }

  /**
   * Create a new column from a typed array input.
   * @template T
   * @param {import('../types.js').TypedArray} values The input data.
   * @param {import('../types.js').ColumnBuilderOptions} options
   *  Builder options for the generated column.
   * @returns {Column<T>} The generated column.
   */
  function columnFromTypedArray(values, { maxBatchRows, useBigInt }) {
    const arrayType = /** @type {import('../types.js').TypedArrayConstructor} */ (
      values.constructor
    );
    const type = typeForTypedArray(arrayType);
    const length = values.length;
    const limit = Math.min(maxBatchRows || Infinity, length);
    const numBatches = Math.floor(length / limit);

    const batches = [];
    const batchType = isInt64ArrayType(arrayType) && !useBigInt ? Int64Batch : DirectBatch;
    const add = (start, end) => batches.push(new batchType({
      length: end - start,
      nullCount: 0,
      type,
      validity: new uint8Array(0),
      values: values.subarray(start, end)
    }));

    let idx = 0;
    for (let i = 0; i < numBatches; ++i) add(idx, idx += limit);
    if (idx < length) add(idx, length);

    return new Column(batches);
  }

  /**
   * Return an Arrow data type for a given typed array type.
   * @param {import('../types.js').TypedArrayConstructor} arrayType
   *  The typed array type.
   * @returns {import('../types.js').DataType} The data type.
   */
  function typeForTypedArray(arrayType) {
    switch (arrayType) {
      case float32Array: return float32();
      case float64Array: return float64();
      case int8Array: return int8();
      case int16Array: return int16();
      case int32Array: return int32();
      case int64Array: return int64();
      case uint8Array: return uint8();
      case uint16Array: return uint16();
      case uint32Array: return uint32();
      case uint64Array: return uint64();
    }
  }

  /**
   * Create a new table from a collection of columns. Columns are assumed
   * to have the same record batch sizes.
   * @param {[string, import('../column.js').Column][]
   *  | Record<string, import('../column.js').Column>} data The columns,
   *  as an object with name keys, or an array of [name, column] pairs.
   * @param {boolean} [useProxy] Flag indicating if row proxy
   *  objects should be used to represent table rows (default `false`).
   * @returns {Table} The new table.
   */
  function tableFromColumns(data, useProxy) {
    const fields = [];
    const entries = Array.isArray(data) ? data : Object.entries(data);
    const length = entries[0]?.[1].length;

    const columns = entries.map(([name, col]) => {
      if (col.length !== length) {
        throw new Error('All columns must have the same length.');
      }
      fields.push(field(name, col.type));
      return col;
    });

    const schema = {
      version: Version.V5,
      endianness: Endianness.Little,
      fields,
      metadata: null
    };

    return new Table(schema, columns, useProxy);
  }

  function isExactDateUTC(d) {
    return d.getUTCHours() === 0
      && d.getUTCMinutes() === 0
      && d.getUTCSeconds() === 0
      && d.getUTCMilliseconds() === 0;
  }

  function inferFormat(scan, options = {}) {
    let count = 0;
    let nulls = 0;
    let dates = 0;
    let dutcs = 0;
    let nums = 0;
    let digits = 0;

    scan(value => {
      ++count;
      if (value == null) {
        ++nulls;
        return;
      }

      const type = typeof value;
      if (type === 'object' && isDate$1(value)) {
        ++dates;
        if (isExactDateUTC(value)) ++dutcs;
      } else if (type === 'number') {
        ++nums;
        if (value === value &&  (value | 0) !== value) {
          const s = value + '';
          const p = s.indexOf('.');
          if (p >= 0) {
            const e = s.indexOf('e');
            const l = e > 0 ? e : s.length;
            digits = Math.max(digits, l - p - 1);
          }
        }
      }
    });

    return {
      align:  (nulls + nums + dates) / count > 0.5 ? 'r' : 'l',
      format: {
        utc:    dates === dutcs,
        digits: Math.min(digits, options.maxdigits || 6)
      }
    };
  }

  var identity = x => x;

  /**
   * Column selection function.
   * @typedef {(table: import('../table/Table.js').Table) => string[]} ColumnSelectFunction
   */

  /**
   * Column selection options.
   * @typedef {string[]|ColumnSelectFunction} ColumnSelectOptions
   */

  /**
   * Column format options. The object keys should be column names.
   * The object values should be formatting functions or objects.
   * If specified, these override any automatically inferred options.
   * @typedef {Object.<string, import('./value.js').ValueFormatOptions>} ColumnFormatOptions
   */

  /**
   * Column alignment options. The object keys should be column names.
   * The object values should be aligment strings, one of 'l' (left),
   * 'c' (center), or 'r' (right).
   * If specified, these override any automatically inferred options.
   * @typedef {Object.<string, 'l'|'c'|'r'>} ColumnAlignOptions
   */

  /**
   * Return a potentially filtered list of column names.
   * @param {import('../table/Table.js').Table} table A data table.
   * @param {ColumnSelectOptions} names The column names to select.
   * @returns {string[]} The selected column names.
   */
  function columns(table, names) {
    // @ts-ignore
    return isFunction(names) ? names(table)
      : names || table.columnNames();
  }

  function formats(table, names, options) {
    const formatOpt = options.format || {};
    const alignOpt = options.align || {};
    const format = {};
    const align = {};

    names.forEach(name => {
      const auto = inferFormat(values(table, name), options);
      align[name] = alignOpt[name] || auto.align;
      format[name] = formatOpt[name] || auto.format;
    });

    return { align, format };
  }

  function values(table, columnName) {
    const column = table.column(columnName);
    return fn => table.scan(row => fn(column.at(row)));
  }

  function scan(table, names, limit = 100, offset, ctx) {
    const { start = identity, cell, end = identity } = ctx;
    const data = table.data();
    const n = names.length;
    table.scan(row => {
      start(row);
      for (let i = 0; i < n; ++i) {
        const name = names[i];
        cell(data[name].at(row), name, i);
      }
      end(row);
    }, true, limit, offset);
  }

  /**
   * Create an Apache Arrow table for an input table.
   * @param {import('../table/Table.js').Table} table
   *  An input Arquero table to convert to Arrow format.
   * @param {import('./types.js').ArrowFormatOptions} [options]
   *  Encoding options, including column data types.
   * @return {import('@uwdata/flechette').Table} An Arrow Table instance.
   */
  function toArrow(table, options = {}) {
    const { columns: columns$1, limit = Infinity, offset = 0, types = {}, ...opt } = options;
    const names = columns(table, columns$1);
    const data = table.data();

    // make a full table scan with no indirection?
    const fullScan = offset === 0
      && table.numRows() <= limit
      && !table.isFiltered()
      && !table.isOrdered();

    return tableFromColumns(names.map(name => {
      const values = data[name];
      const type = types[name];
      const isArray = isArrayType(values);
      let col;
      if (fullScan && (isArray || isFunction(values.toArray))) {
        // @ts-ignore - use faster path, takes advantange of typed arrays
        col = columnFromArray(isArray ? values : values.toArray(), type, opt);
      } else {
        // use table scan method to visit column values
        const get = isArray
          ? row => values[row]
          : row => values.at(row);
        col = columnFromValues(
          visit => table.scan(row => visit(get(row)), true, limit, offset),
          type,
          opt
        );
      }
      return [name, col];
    }));
  }

  /**
   * Format a table as binary data in the Apache Arrow IPC format.
   * @param {import('../table/Table.js').Table} data The table data
   * @param {import('./types.js').ArrowIPCFormatOptions} [options]
   *  The Arrow IPC formatting options. Set the *format* option to `'stream'`
   *  or `'file'` to specify the IPC format.
   * @return {Uint8Array} A new Uint8Array of Arrow-encoded binary data.
   */
  function toArrowIPC(data, options = {}) {
    const { format = 'stream', ...toArrowOptions } = options;
    return tableToIPC(toArrow(data, toArrowOptions), { format });
  }

  /**
   * Options for CSV formatting.
   * @typedef {object} CSVFormatOptions
   * @property {string} [delimiter=','] The delimiter between values.
   * @property {boolean} [header=true] Flag to specify presence of header row.
   *  If true, includes a header row with column names.
   *  If false, the header is omitted.
   * @property {number} [limit=Infinity] The maximum number of rows to print.
   * @property {number} [offset=0] The row offset indicating how many initial rows to skip.
   * @property {import('./util.js').ColumnSelectOptions} [columns] Ordered list
   *  of column names to include. If function-valued, the function should
   *  accept a table as input and return an array of column name strings.
   * @property {Object.<string, (value: any) => any>} [format] Object of column
   *  format options. The object keys should be column names. The object values
   *  should be formatting functions to invoke to transform column values prior
   *  to output. If specified, these override automatically inferred options.
   */

  /**
   * Format a table as a comma-separated values (CSV) string. Other
   * delimiters, such as tabs or pipes ('|'), can be specified using
   * the options argument.
   * @param {import('../table/Table.js').Table} table The table to format.
   * @param {CSVFormatOptions} options The formatting options.
   * @return {string} A delimited-value format string.
   */
  function toCSV(table, options = {}) {
    const names = columns(table, options.columns);
    const format = options.format || {};
    const delim = options.delimiter || ',';
    const header = options.header ?? true;
    const reFormat = new RegExp(`["${delim}\n\r]`);

    const formatValue = value => value == null ? ''
      : isDate$1(value) ? formatUTCDate(value, true)
      : reFormat.test(value += '') ? '"' + value.replace(/"/g, '""') + '"'
      : value;

    const vals = names.map(formatValue);
    let text = header ? (vals.join(delim) + '\n') : '';

    scan(table, names, options.limit || Infinity, options.offset, {
      cell(value, name, index) {
        vals[index] = formatValue(format[name] ? format[name](value) : value);
      },
      end() {
        text += vals.join(delim) + '\n';
      }
    });

    return text;
  }

  /**
   * Column format object.
   * @typedef {object} ValueFormatObject
   * @property {boolean} [utc=false] If true, format dates in UTC time.
   * @property {number} [digits=0] The number of fractional digits to include
   *  when formatting numbers.
   * @property {number} [maxlen=30] The maximum string length for formatting
   *  nested object or array values.
   */

  /**
   * @callback ValueFormatFunction
   * @param {*} value The value to format.
   * @return {*} A string-coercible or JSON-compatible formatted value.
   */

  /**
   * Value format options.
   * @typedef {ValueFormatObject|ValueFormatFunction} ValueFormatOptions
   */

  /**
   * Format a value as a string.
   * @param {*} v The value to format.
   * @param {ValueFormatOptions} options Formatting options.
   * @return {string} The formatted string.
   */
  function formatValue(v, options = {}) {
    if (isFunction(options)) {
      // @ts-ignore
      return options(v) + '';
    }

    const type = typeof v;

    if (type === 'object') {
      if (isDate$1(v)) {
        // @ts-ignore
        return options.utc ? formatUTCDate(v) : formatDate(v);
      } else {
        const s = JSON.stringify(
          v,
          // @ts-ignore
          (k, v) => isTypedArray$1(v) ? Array.from(v) : v
        );
        // @ts-ignore
        const maxlen = options.maxlen || 30;
        return s.length > maxlen
          ? s.slice(0, 28) + '\u2026' + (s[0] === '[' ? ']' : '}')
          : s;
      }
    } else if (type === 'number') {
      // @ts-ignore
      const digits = options.digits || 0;
      let a;
      return v !== 0 && ((a = Math.abs(v)) >= 1e18 || a < Math.pow(10, -digits))
        ? v.toExponential(digits)
        : v.toFixed(digits);
    } else {
      return v + '';
    }
  }

  function mapObject(obj, fn, output = {}) {
    for (const key in obj) {
      output[key] = fn(obj[key], key);
    }
    return output;
  }

  /**
   * Null format function.
   * @callback NullFormat
   * @param {null|undefined} [value] The value to format.
   * @return {string} The formatted HTML string.
   */

  /**
   * CSS style function.
   * @callback StyleFunction
   * @param {string} name The column name.
   * @param {number} row The table row index.
   * @return {string} A CSS style string.
   */

  /**
   * CSS style options.
   * @typedef {Object.<string, string | StyleFunction>} StyleOptions
   */

  /**
   * Options for HTML formatting.
   * @typedef {object} HTMLFormatOptions
   * @property {number} [limit=Infinity] The maximum number of rows to print.
   * @property {number} [offset=0] The row offset indicating how many initial rows to skip.
   * @property {import('./util.js').ColumnSelectOptions} [columns] Ordered list
   *  of column names to include. If function-valued, the function should
   *  accept a table as input and return an array of column name strings.
   * @property {import('./util.js').ColumnAlignOptions} [align] Object of column
   *  alignment options. The object keys should be column names. The object
   *  values should be aligment strings, one of 'l' (left), 'c' (center), or
   *  'r' (right). If specified, these override automatically inferred options.
   * @property {import('./util.js').ColumnFormatOptions} [format] Object of column
   *  format options. The object keys should be column names. The object values
   *  should be formatting functions or specification objects. If specified,
   *  these override automatically inferred options.
   * @property {NullFormat} [null] Format function for null or undefined values.
   *  If specified, this function will be invoked with the null or undefined
   *  value as the sole input, and the return value will be used as the HTML
   *  output for the value.
   * @property {StyleOptions} [style] CSS styles to include in HTML output.
   *  The object keys should be HTML table tag names: 'table', 'thead',
   *  'tbody', 'tr', 'th', or 'td'. The object values should be strings of
   *  valid CSS style directives (such as "font-weight: bold;") or functions
   *  that take a column name and row as inputs and return a CSS string.
   * @property {number} [maxdigits=6] The maximum number of fractional digits
   *  to include when formatting numbers. This option is passed to the format
   *  inference method and is overridden by any explicit format options.
   */

  /**
   * Format a table as an HTML table string.
   * @param {import('../table/Table.js').Table} table The table to format.
   * @param {HTMLFormatOptions} options The formatting options.
   * @return {string} An HTML table string.
   */
  function toHTML(table, options = {}) {
    const names = columns(table, options.columns);
    const { align, format } = formats(table, names, options);
    const style = styles(options);
    const nullish = options.null;

    const alignValue = a => a === 'c' ? 'center' : a === 'r' ? 'right' : 'left';
    const escape = s => s.replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;');
    const baseFormat = (value, opt) => escape(formatValue(value, opt));
    const formatter = nullish
      ? (value, opt) => value == null ? nullish(value) : baseFormat(value, opt)
      : baseFormat;

    let r = -1;
    let idx = -1;

    const tag = (tag, name, shouldAlign) => {
      const a = shouldAlign ? alignValue(align[name]) : '';
      const s = style[tag] ? (style[tag](name, idx, r) || '') : '';
      const css = (a ? (`text-align: ${a};` + (s ? ' ' : '')) : '') + s;
      return `<${tag}${css ? ` style="${css}"` : ''}>`;
    };

    let text = tag('table')
      + tag('thead')
      + tag('tr', r)
      + names.map(name => `${tag('th', name, 1)}${name}</th>`).join('')
      + '</tr></thead>'
      + tag('tbody');

    scan(table, names, options.limit, options.offset, {
      start(row) {
        r = row;
        ++idx;
        text += tag('tr');
      },
      cell(value, name) {
        text += tag('td', name, 1)
          + formatter(value, format[name])
          + '</td>';
      },
      end() {
        text += '</tr>';
      }
    });

    return text + '</tbody></table>';
  }

  function styles(options) {
    return mapObject(
      options.style,
      value => isFunction(value) ? value : () => value
    );
  }

  function defaultTrue(value, trueValue = true, falseValue = false) {
    return (value === undefined || value) ? trueValue : falseValue;
  }

  /**
   * Options for JSON formatting.
   * @typedef {object} JSONFormatOptions
   * @property {number} [limit=Infinity] The maximum number of rows to print.
   * @property {number} [offset=0] The row offset indicating how many initial
   *  rows to skip.
   * @property {boolean} [schema=true] Flag indicating if table schema metadata
   *  should be included in the JSON output. If false, only the data payload
   *  is included.
   * @property {import('./util.js').ColumnSelectOptions} [columns] Ordered list
   *  of column names to include. If function-valued, the function should
   *  accept a table as input and return an array of column name strings.
   * @property {Object.<string, (value: any) => any>} [format] Object of column
   *  format options. The object keys should be column names. The object values
   *  should be formatting functions to invoke to transform column values prior
   *  to output. If specified, these override automatically inferred options.
   */

  const defaultFormatter = value => isDate$1(value)
    ? formatUTCDate(value, true)
    : value;

  /**
   * Format a table as a JavaScript Object Notation (JSON) string.
   * @param {import('../table/Table.js').Table} table The table to format.
   * @param {JSONFormatOptions} options The formatting options.
   * @return {string} A JSON string.
   */
  function toJSON(table, options = {}) {
    const schema = defaultTrue(options.schema);
    const format = options.format || {};
    const names = columns(table, options.columns);
    let text = '{';

    if (schema) {
      text += '"schema":{"fields":'
        + JSON.stringify(names.map(name => ({ name })))
        + '},"data":{';
    }

    names.forEach((name, i) => {
      text += (i ? ',' : '') + JSON.stringify(name) + ':[';

      const column = table.column(name);
      const formatter = format[name] || defaultFormatter;
      let r = -1;
      table.scan(row => {
        const value = column.at(row);
        text += (++r ? ',' : '') + JSON.stringify(formatter(value));
      }, true, options.limit, options.offset);

      text += ']';
    });

    return text + '}' + (schema ? '}' : '');
  }

  /**
   * Options for Markdown formatting.
   * @typedef {object} MarkdownFormatOptions
   * @property {number} [limit=Infinity] The maximum number of rows to print.
   * @property {number} [offset=0] The row offset indicating how many initial
   *  rows to skip.
   * @property {import('./util.js').ColumnSelectOptions} [columns] Ordered list
   *  of column names to include. If function-valued, the function should
   *  accept a table as input and return an array of column name strings.
   * @property {import('./util.js').ColumnAlignOptions} [align] Object of column
   *  alignment options. The object keys should be column names. The object
   *  values should be aligment strings, one of 'l' (left), 'c' (center), or
   *  'r' (right). If specified, these override automatically inferred options.
   * @property {import('./util.js').ColumnFormatOptions} [format] Object of column
   *  format options. The object keys should be column names. The object values
   *  should be formatting functions or specification objects. If specified,
   *  these override automatically inferred options.
   * @property {number} [maxdigits=6] The maximum number of fractional digits
   *  to include when formatting numbers. This option is passed to the format
   *  inference method and is overridden by any explicit format options.
   */

  /**
   * Format a table as a GitHub-Flavored Markdown table string.
   * @param {import('../table/Table.js').Table} table The table to format.
   * @param {MarkdownFormatOptions} options The formatting options.
   * @return {string} A GitHub-Flavored Markdown table string.
   */
  function toMarkdown(table, options = {}) {
    const names = columns(table, options.columns);
    const { align, format } = formats(table, names, options);

    const alignValue = a => a === 'c' ? ':-:' : a === 'r' ? '-:' : ':-';
    const escape = s => s.replace(/\|/g, '\\|');

    let text = '|'
      + names.map(escape).join('|')
      + '|\n|'
      + names.map(name => alignValue(align[name])).join('|')
      + '|\n';

    scan(table, names, options.limit, options.offset, {
      start() {
        text += '|';
      },
      cell(value, name) {
        text += escape(formatValue(value, format[name])) + '|';
      },
      end() {
        text += '\n';
      }
    });

    return text;
  }

  /**
   * A data table with transformation verbs.
   */
  class ColumnTable extends Table$1 {
    /**
     * Create a new table with additional columns drawn from one or more input
     * tables. All tables must have the same numer of rows and are reified
     * prior to assignment. In the case of repeated column names, input table
     * columns overwrite existing columns.
     * @param {...(Table|import('./types.js').ColumnData)} tables
     *  The tables to merge with this table.
     * @return {this} A new table with merged columns.
     * @example table.assign(table1, table2)
     */
    assign(...tables) {
      return assign(this, ...tables);
    }

    /**
     * Count the number of values in a group. This method is a shorthand
     * for *rollup* with a count aggregate function.
     * @param {import('./types.js').CountOptions} [options]
     *  Options for the count.
     * @return {this} A new table with groupby and count columns.
     * @example table.groupby('colA').count()
     * @example table.groupby('colA').count({ as: 'num' })
     */
    count(options = {}) {
      const { as = 'count' } = options;
      return rollup(this, { [as]: count() });
    }

    /**
     * Derive new column values based on the provided expressions. By default,
     * new columns are added after (higher indices than) existing columns. Use
     * the before or after options to place new columns elsewhere.
     * @param {import('./types.js').ExprObject} values
     *  Object of name-value pairs defining the columns to derive. The input
     *  object should have output column names for keys and table expressions
     *  for values.
     * @param {import('./types.js').DeriveOptions} [options]
     *  Options for dropping or relocating derived columns. Use either a before
     *  or after property to indicate where to place derived columns. Specifying
     *  both before and after is an error. Unlike the *relocate* verb, this
     *  option affects only new columns; updated columns with existing names
     *  are excluded from relocation.
     * @return {this} A new table with derived columns added.
     * @example table.derive({ sumXY: d => d.x + d.y })
     * @example table.derive({ z: d => d.x * d.y }, { before: 'x' })
     */
    derive(values, options) {
      return derive(this, values, options);
    }

    /**
     * Filter a table to a subset of rows based on the input criteria.
     * The resulting table provides a filtered view over the original data; no
     * data copy is made. To create a table that copies only filtered data to
     * new data structures, call *reify* on the output table.
     * @param {import('./types.js').TableExpr} criteria
     *  Filter criteria as a table expression. Both aggregate and window
     *  functions are permitted, taking into account *groupby* or *orderby*
     *  settings.
     * @return {this} A new table with filtered rows.
     * @example table.filter(d => abs(d.value) < 5)
     */
    filter(criteria) {
      return filter$1(this, criteria);
    }

    /**
     * Extract rows with indices from start to end (end not included), where
     * start and end represent per-group ordered row numbers in the table.
     * @param {number} [start] Zero-based index at which to start extraction.
     *  A negative index indicates an offset from the end of the group.
     *  If start is undefined, slice starts from the index 0.
     * @param {number} [end] Zero-based index before which to end extraction.
     *  A negative index indicates an offset from the end of the group.
     *  If end is omitted, slice extracts through the end of the group.
     * @return {this} A new table with sliced rows.
     * @example table.slice(1, -1)
     */
    slice(start, end) {
      return slice(this, start, end);
    }

    /**
     * Group table rows based on a set of column values.
     * Subsequent operations that are sensitive to grouping (such as
     * aggregate functions) will operate over the grouped rows.
     * To undo grouping, use *ungroup*.
     * @param  {...import('./types.js').ExprList} keys
     *  Key column values to group by. The keys may be specified using column
     *  name strings, column index numbers, value objects with output column
     *  names for keys and table expressions for values, or selection helper
     *  functions.
     * @return {this} A new table with grouped rows.
     * @example table.groupby('colA', 'colB')
     * @example table.groupby({ key: d => d.colA + d.colB })
     */
    groupby(...keys) {
      return groupby(this, ...keys);
    }

    /**
     * Order table rows based on a set of column values. Subsequent operations
     * sensitive to ordering (such as window functions) will operate over sorted
     * values. The resulting table provides an view over the original data,
     * without any copying. To create a table with sorted data copied to new
     * data strucures, call *reify* on the result of this method. To undo
     * ordering, use *unorder*.
     * @param  {...import('./types.js').OrderKeys} keys
     *  Key values to sort by, in precedence order.
     *  By default, sorting is done in ascending order.
     *  To sort in descending order, wrap values using *desc*.
     *  If a string, order by the column with that name.
     *  If a number, order by the column with that index.
     *  If a function, must be a valid table expression; aggregate functions
     *  are permitted, but window functions are not.
     *  If an object, object values must be valid values parameters
     *  with output column names for keys and table expressions
     *  for values (the output names will be ignored).
     *  If an array, array values must be valid key parameters.
     * @return {this} A new ordered table.
     * @example table.orderby('a', desc('b'))
     * @example table.orderby({ a: 'a', b: desc('b') )})
     * @example table.orderby(desc(d => d.a))
     */
    orderby(...keys) {
      return orderby(this, ...keys);
    }

    /**
     * Relocate a subset of columns to change their positions, also
     * potentially renaming them.
     * @param {import('./types.js').Select} columns
     *  An ordered selection of columns to relocate.
     *  The input may consist of column name strings, column integer indices,
     *  rename objects with current column names as keys and new column names
     *  as values, or functions that take a table as input and returns a valid
     *  selection parameter (typically the output of selection helper functions
     *  such as *all*, *not*, or *range*).
     * @param {import('./types.js').RelocateOptions} options
     *  Options for relocating. Must include either the before or after property
     *  to indicate where to place the relocated columns. Specifying both before
     *  and after is an error.
     * @return {this} A new table with relocated columns.
     * @example table.relocate(['colY', 'colZ'], { after: 'colX' })
     * @example table.relocate(not('colB', 'colC'), { before: 'colA' })
     * @example table.relocate({ colA: 'newA', colB: 'newB' }, { after: 'colC' })
     */
    relocate(columns, options) {
      return relocate(this, toArray(columns), options);
    }

    /**
     * Rename one or more columns, preserving column order.
     * @param {...import('./types.js').Select} columns
     *  One or more rename objects with current column names as keys and new
     *  column names as values.
     * @return {this} A new table with renamed columns.
     * @example table.rename({ oldName: 'newName' })
     * @example table.rename({ a: 'a2', b: 'b2' })
     */
    rename(...columns) {
      return rename(this, ...columns);
    }

    /**
     * Reduce a table, processing all rows to produce a new table.
     * To produce standard aggregate summaries, use the rollup verb.
     * This method allows the use of custom reducer implementations,
     * for example to produce multiple rows for an aggregate.
     * @param {import('../verbs/reduce/reducer.js').default} reducer
     *  The reducer to apply.
     * @return {this} A new table of reducer outputs.
     */
    reduce(reducer) {
      return reduce(this, reducer);
    }

    /**
     * Rollup a table to produce an aggregate summary.
     * Often used in conjunction with *groupby*.
     * To produce counts only, *count* is a shortcut.
     * @param {import('./types.js').ExprObject} [values]
     *  Object of name-value pairs defining aggregate output columns. The input
     *  object should have output column names for keys and table expressions
     *  for values. The expressions must be valid aggregate expressions: window
     *  functions are not allowed and column references must be arguments to
     *  aggregate functions.
     * @return {this} A new table of aggregate summary values.
     * @example table.groupby('colA').rollup({ mean: d => mean(d.colB) })
     * @example table.groupby('colA').rollup({ mean: op.median('colB') })
     */
    rollup(values) {
      return rollup(this, values);
    }

    /**
     * Generate a table from a random sample of rows.
     * If the table is grouped, performs a stratified sample by
     * sampling from each group separately.
     * @param {number | import('./types.js').TableExpr} size
     *  The number of samples to draw per group.
     *  If number-valued, the same sample size is used for each group.
     *  If function-valued, the input should be an aggregate table
     *  expression compatible with *rollup*.
     * @param {import('./types.js').SampleOptions} [options]
     *  Options for sampling.
     * @return {this} A new table with sampled rows.
     * @example table.sample(50)
     * @example table.sample(100, { replace: true })
     * @example table.groupby('colA').sample(() => op.floor(0.5 * op.count()))
     */
    sample(size, options) {
      return sample(this, size, options);
    }

    /**
     * Select a subset of columns into a new table, potentially renaming them.
     * @param {...import('./types.js').Select} columns
     *  An ordered selection of columns.
     *  The input may consist of column name strings, column integer indices,
     *  rename objects with current column names as keys and new column names
     *  as values, or functions that take a table as input and returns a valid
     *  selection parameter (typically the output of selection helper functions
     *  such as *all*, *not*, or *range*.).
     * @return {this} A new table of selected columns.
     * @example table.select('colA', 'colB')
     * @example table.select(not('colB', 'colC'))
     * @example table.select({ colA: 'newA', colB: 'newB' })
     */
    select(...columns) {
      return select(this, ...columns);
    }

    /**
     * Ungroup a table, removing any grouping criteria.
     * Undoes the effects of *groupby*.
     * @return {this} A new ungrouped table, or this table if not grouped.
     * @example table.ungroup()
     */
    ungroup() {
      return ungroup(this);
    }

    /**
     * Unorder a table, removing any sorting criteria.
     * Undoes the effects of *orderby*.
     * @return {this} A new unordered table, or this table if not ordered.
     * @example table.unorder()
     */
    unorder() {
      return unorder(this);
    }

    // -- Cleaning Verbs ------------------------------------------------------

    /**
     * De-duplicate table rows by removing repeated row values.
     * @param {...import('./types.js').ExprList} keys
     *  Key columns to check for duplicates.
     *  Two rows are considered duplicates if they have matching values for
     *  all keys. If keys are unspecified, all columns are used.
     *  The keys may be specified using column name strings, column index
     *  numbers, value objects with output column names for keys and table
     *  expressions for values, or selection helper functions.
     * @return {this} A new de-duplicated table.
     * @example table.dedupe()
     * @example table.dedupe('a', 'b')
     * @example table.dedupe({ abs: d => op.abs(d.a) })
     */
    dedupe(...keys) {
      return dedupe(this, ...keys);
    }

    /**
     * Impute missing values or rows. Accepts a set of column-expression pairs
     * and evaluates the expressions to replace any missing (null, undefined,
     * or NaN) values in the original column.
     * If the expand option is specified, imputes new rows for missing
     * combinations of values. All combinations of key values (a full cross
     * product) are considered for each level of grouping (specified by
     * *groupby*). New rows will be added for any combination
     * of key and groupby values not already contained in the table. For all
     * non-key and non-group columns the new rows are populated with imputation
     * values (first argument) if specified, otherwise undefined.
     * If the expand option is specified, any filter or orderby settings are
     * removed from the output table, but groupby settings persist.
     * @param {import('./types.js').ExprObject} values
     *  Object of name-value pairs for the column values to impute. The input
     *  object should have existing column names for keys and table expressions
     *  for values. The expressions will be evaluated to determine replacements
     *  for any missing values.
     * @param {import('./types.js').ImputeOptions} [options] Imputation options.
     *  The expand property specifies a set of column values to consider for
     *  imputing missing rows. All combinations of expanded values are
     *  considered, and new rows are added for each combination that does not
     *  appear in the input table.
     * @return {this} A new table with imputed values and/or rows.
     * @example table.impute({ v: () => 0 })
     * @example table.impute({ v: d => op.mean(d.v) })
     * @example table.impute({ v: () => 0 }, { expand: ['x', 'y'] })
     */
    impute(values, options) {
      return impute(this, values, options);
    }

    // -- Reshaping Verbs -----------------------------------------------------

    /**
     * Fold one or more columns into two key-value pair columns.
     * The fold transform is an inverse of the *pivot* transform.
     * The resulting table has two new columns, one containing the column
     * names (named "key") and the other the column values (named "value").
     * The number of output rows equals the original row count multiplied
     * by the number of folded columns.
     * @param {import('./types.js').ExprList} values The columns to fold.
     *  The columns may be specified using column name strings, column index
     *  numbers, value objects with output column names for keys and table
     *  expressions for values, or selection helper functions.
     * @param {import('./types.js').FoldOptions} [options] Options for folding.
     * @return {this} A new folded table.
     * @example table.fold('colA')
     * @example table.fold(['colA', 'colB'])
     * @example table.fold(range(5, 8))
     */
    fold(values, options) {
      return fold(this, values, options);
    }

    /**
     * Pivot columns into a cross-tabulation.
     * The pivot transform is an inverse of the *fold* transform.
     * The resulting table has new columns for each unique combination
     * of the provided *keys*, populated with the provided *values*.
     * The provided *values* must be aggregates, as a single set of keys may
     * include more than one row. If string-valued, the *any* aggregate is used.
     * If only one *values* column is defined, the new pivoted columns will
     * be named using key values directly. Otherwise, input value column names
     * will be included as a component of the output column names.
     * @param {import('./types.js').ExprList} keys
     *  Key values to map to new column names. The keys may be specified using
     *  column name strings, column index numbers, value objects with output
     *  column names for keys and table expressions for values, or selection
     *  helper functions.
     * @param {import('./types.js').ExprList} values Output values for pivoted
     *  columns. Column references will be wrapped in an *any* aggregate. If
     *  object-valued, the input object should have output value names for keys
     *  and aggregate table expressions for values.
     * @param {import('./types.js').PivotOptions} [options]
     *  Options for pivoting.
     * @return {this} A new pivoted table.
     * @example table.pivot('key', 'value')
     * @example table.pivot(['keyA', 'keyB'], ['valueA', 'valueB'])
     * @example table.pivot({ key: d => d.key }, { value: d => op.sum(d.value) })
     */
    pivot(keys, values, options) {
      return pivot(this, keys, values, options);
    }

    /**
     * Spread array elements into a set of new columns.
     * Output columns are named based on the value key and array index.
     * @param {import('./types.js').ExprList} values
     *  The column values to spread. The values may be specified using column
     *  name strings, column index numbers, value objects with output column
     *  names for keys and table expressions for values, or selection helper
     *  functions.
     * @param {import('./types.js').SpreadOptions } [options]
     *  Options for spreading.
     * @return {this} A new table with the spread columns added.
     * @example table.spread({ a: d => op.split(d.text, '') })
     * @example table.spread('arrayCol', { limit: 100 })
     */
    spread(values, options) {
      return spread(this, values, options);
    }

    /**
     * Unroll one or more array-valued columns into new rows.
     * If more than one array value is used, the number of new rows
     * is the smaller of the limit and the largest length.
     * Values for all other columns are copied over.
     * @param {import('./types.js').ExprList} values
     *  The column values to unroll. The values may be specified using column
     *  name strings, column index numbers, value objects with output column
     *  names for keys and table expressions for values, or selection helper
     *  functions.
     * @param {import('./types.js').UnrollOptions} [options]
     *  Options for unrolling.
     * @return {this} A new unrolled table.
     * @example table.unroll('colA', { limit: 1000 })
     */
    unroll(values, options) {
      return unroll(this, values, options);
    }

    // -- Joins ---------------------------------------------------------------

    /**
     * Lookup values from a secondary table and add them as new columns.
     * A lookup occurs upon matching key values for rows in both tables.
     * If the secondary table has multiple rows with the same key, only
     * the last observed instance will be considered in the lookup.
     * Lookup is similar to *join_left*, but with a simpler
     * syntax and the added constraint of allowing at most one match only.
     * @param {import('./types.js').TableRef} other
     *  The secondary table to look up values from.
     * @param {import('./types.js').JoinKeys} [on]
     *  Lookup keys (column name strings or table expressions) for this table
     *  and the secondary table, respectively.
     * @param {...import('./types.js').ExprList} values
     *  The column values to add from the secondary table. Can be column name
     *  strings or objects with column names as keys and table expressions as
     *  values.
     * @return {this} A new table with lookup values added.
     * @example table.lookup(other, ['key1', 'key2'], 'value1', 'value2')
     */
    lookup(other, on, ...values) {
      return lookup(this, other, on, ...values);
    }

    /**
     * Join two tables, extending the columns of one table with
     * values from the other table. The current table is considered
     * the "left" table in the join, and the new table input is
     * considered the "right" table in the join. By default an inner
     * join is performed, removing all rows that do not match the
     * join criteria. To perform left, right, or full outer joins, use
     * the *join_left*, *join_right*, or *join_full* methods, or provide
     * an options argument.
     * @param {import('./types.js').TableRef} other
     *  The other (right) table to join with.
     * @param {import('./types.js').JoinPredicate} [on]
     *  The join criteria for matching table rows. If unspecified, the values of
     *  all columns with matching names are compared.
     *  If array-valued, a two-element array should be provided, containing
     *  the columns to compare for the left and right tables, respectively.
     *  If a one-element array or a string value is provided, the same
     *  column names will be drawn from both tables.
     *  If function-valued, should be a two-table table expression that
     *  returns a boolean value. When providing a custom predicate, note that
     *  join key values can be arrays or objects, and that normal join
     *  semantics do not consider null or undefined values to be equal (that is,
     *  null !== null). Use the op.equal function to handle these cases.
     * @param {import('./types.js').JoinValues} [values]
     *  The columns to include in the join output.
     *  If unspecified, all columns from both tables are included; paired
     *  join keys sharing the same column name are included only once.
     *  If array-valued, a two element array should be provided, containing
     *  the columns to include for the left and right tables, respectively.
     *  Array input may consist of column name strings, objects with output
     *  names as keys and single-table table expressions as values, or the
     *  selection helper functions *all*, *not*, or *range*.
     *  If object-valued, specifies the key-value pairs for each output,
     *  defined using two-table table expressions.
     * @param {import('./types.js').JoinOptions} [options]
     *  Options for the join.
     * @return {this} A new joined table.
     * @example table.join(other, ['keyL', 'keyR'])
     * @example table.join(other, (a, b) => op.equal(a.keyL, b.keyR))
     */
    join(other, on, values, options) {
      return join(this, other, on, values, options);
    }

    /**
     * Perform a left outer join on two tables. Rows in the left table
     * that do not match a row in the right table will be preserved.
     * This is a convenience method with fixed options for *join*.
     * @param {import('./types.js').TableRef} other
     *  The other (right) table to join with.
     * @param {import('./types.js').JoinPredicate} [on]
     *  The join criteria for matching table rows.
     *  If unspecified, the values of all columns with matching names
     *  are compared.
     *  If array-valued, a two-element array should be provided, containing
     *  the columns to compare for the left and right tables, respectively.
     *  If a one-element array or a string value is provided, the same
     *  column names will be drawn from both tables.
     *  If function-valued, should be a two-table table expression that
     *  returns a boolean value. When providing a custom predicate, note that
     *  join key values can be arrays or objects, and that normal join
     *  semantics do not consider null or undefined values to be equal (that is,
     *  null !== null). Use the op.equal function to handle these cases.
     * @param {import('./types.js').JoinValues} [values]
     *  he columns to include in the join output.
     *  If unspecified, all columns from both tables are included; paired
     *  join keys sharing the same column name are included only once.
     *  If array-valued, a two element array should be provided, containing
     *  the columns to include for the left and right tables, respectively.
     *  Array input may consist of column name strings, objects with output
     *  names as keys and single-table table expressions as values, or the
     *  selection helper functions *all*, *not*, or *range*.
     *  If object-valued, specifies the key-value pairs for each output,
     *  defined using two-table table expressions.
     * @param {import('./types.js').JoinOptions} [options]
     *  Options for the join. With this method, any options will be
     *  overridden with `{left: true, right: false}`.
     * @return {this} A new joined table.
     * @example table.join_left(other, ['keyL', 'keyR'])
     * @example table.join_left(other, (a, b) => op.equal(a.keyL, b.keyR))
     */
    join_left(other, on, values, options) {
      const opt = { ...options, left: true, right: false };
      return join(this, other, on, values, opt);
    }

    /**
     * Perform a right outer join on two tables. Rows in the right table
     * that do not match a row in the left table will be preserved.
     * This is a convenience method with fixed options for *join*.
     * @param {import('./types.js').TableRef} other
     *  The other (right) table to join with.
     * @param {import('./types.js').JoinPredicate} [on]
     *  The join criteria for matching table rows.
     *  If unspecified, the values of all columns with matching names
     *  are compared.
     *  If array-valued, a two-element array should be provided, containing
     *  the columns to compare for the left and right tables, respectively.
     *  If a one-element array or a string value is provided, the same
     *  column names will be drawn from both tables.
     *  If function-valued, should be a two-table table expression that
     *  returns a boolean value. When providing a custom predicate, note that
     *  join key values can be arrays or objects, and that normal join
     *  semantics do not consider null or undefined values to be equal (that is,
     *  null !== null). Use the op.equal function to handle these cases.
     * @param {import('./types.js').JoinValues} [values]
     *  The columns to include in the join output.
     *  If unspecified, all columns from both tables are included; paired
     *  join keys sharing the same column name are included only once.
     *  If array-valued, a two element array should be provided, containing
     *  the columns to include for the left and right tables, respectively.
     *  Array input may consist of column name strings, objects with output
     *  names as keys and single-table table expressions as values, or the
     *  selection helper functions *all*, *not*, or *range*.
     *  If object-valued, specifies the key-value pairs for each output,
     *  defined using two-table table expressions.
     * @param {import('./types.js').JoinOptions} [options]
     *  Options for the join. With this method, any options will be overridden
     *  with `{left: false, right: true}`.
     * @return {this} A new joined table.
     * @example table.join_right(other, ['keyL', 'keyR'])
     * @example table.join_right(other, (a, b) => op.equal(a.keyL, b.keyR))
     */
    join_right(other, on, values, options) {
      const opt = { ...options, left: false, right: true };
      return join(this, other, on, values, opt);
    }

    /**
     * Perform a full outer join on two tables. Rows in either the left or
     * right table that do not match a row in the other will be preserved.
     * This is a convenience method with fixed options for *join*.
     * @param {import('./types.js').TableRef} other
     *  The other (right) table to join with.
     * @param {import('./types.js').JoinPredicate} [on]
     *  The join criteria for matching table rows.
     *  If unspecified, the values of all columns with matching names
     *  are compared.
     *  If array-valued, a two-element array should be provided, containing
     *  the columns to compare for the left and right tables, respectively.
     *  If a one-element array or a string value is provided, the same
     *  column names will be drawn from both tables.
     *  If function-valued, should be a two-table table expression that
     *  returns a boolean value. When providing a custom predicate, note that
     *  join key values can be arrays or objects, and that normal join
     *  semantics do not consider null or undefined values to be equal (that is,
     *  null !== null). Use the op.equal function to handle these cases.
     * @param {import('./types.js').JoinValues} [values]
     *  The columns to include in the join output.
     *  If unspecified, all columns from both tables are included; paired
     *  join keys sharing the same column name are included only once.
     *  If array-valued, a two element array should be provided, containing
     *  the columns to include for the left and right tables, respectively.
     *  Array input may consist of column name strings, objects with output
     *  names as keys and single-table table expressions as values, or the
     *  selection helper functions *all*, *not*, or *range*.
     *  If object-valued, specifies the key-value pairs for each output,
     *  defined using two-table table expressions.
     * @param {import('./types.js').JoinOptions} [options]
     *  Options for the join. With this method, any options will be overridden
     *  with `{left: true, right: true}`.
     * @return {this} A new joined table.
     * @example table.join_full(other, ['keyL', 'keyR'])
     * @example table.join_full(other, (a, b) => op.equal(a.keyL, b.keyR))
     */
    join_full(other, on, values, options) {
      const opt = { ...options, left: true, right: true };
      return join(this, other, on, values, opt);
    }

    /**
     * Produce the Cartesian cross product of two tables. The output table
     * has one row for every pair of input table rows. Beware that outputs
     * may be quite large, as the number of output rows is the product of
     * the input row counts.
     * This is a convenience method for *join* in which the
     * join criteria is always true.
     * @param {import('./types.js').TableRef} other
     *  The other (right) table to join with.
     * @param {import('./types.js').JoinValues} [values]
     *  The columns to include in the output.
     *  If unspecified, all columns from both tables are included.
     *  If array-valued, a two element array should be provided, containing
     *  the columns to include for the left and right tables, respectively.
     *  Array input may consist of column name strings, objects with output
     *  names as keys and single-table table expressions as values, or the
     *  selection helper functions *all*, *not*, or *range*.
     *  If object-valued, specifies the key-value pairs for each output,
     *  defined using two-table table expressions.
     * @param {import('./types.js').JoinOptions} [options]
     *  Options for the join.
     * @return {this} A new joined table.
     * @example table.cross(other)
     * @example table.cross(other, [['leftKey', 'leftVal'], ['rightVal']])
     */
    cross(other, values, options) {
      return cross(this, other, values, options);
    }

    /**
     * Perform a semi-join, filtering the left table to only rows that
     * match a row in the right table.
     * @param {import('./types.js').TableRef} other
     *  The other (right) table to join with.
     * @param {import('./types.js').JoinPredicate} [on]
     *  The join criteria for matching table rows.
     *  If unspecified, the values of all columns with matching names
     *  are compared.
     *  If array-valued, a two-element array should be provided, containing
     *  the columns to compare for the left and right tables, respectively.
     *  If a one-element array or a string value is provided, the same
     *  column names will be drawn from both tables.
     *  If function-valued, should be a two-table table expression that
     *  returns a boolean value. When providing a custom predicate, note that
     *  join key values can be arrays or objects, and that normal join
     *  semantics do not consider null or undefined values to be equal (that is,
     *  null !== null). Use the op.equal function to handle these cases.
     * @return {this} A new filtered table.
     * @example table.semijoin(other)
     * @example table.semijoin(other, ['keyL', 'keyR'])
     * @example table.semijoin(other, (a, b) => op.equal(a.keyL, b.keyR))
     */
    semijoin(other, on) {
      return semijoin(this, other, on);
    }

    /**
     * Perform an anti-join, filtering the left table to only rows that
     * do *not* match a row in the right table.
     * @param {import('./types.js').TableRef} other
     *  The other (right) table to join with.
     * @param {import('./types.js').JoinPredicate} [on]
     *  The join criteria for matching table rows.
     *  If unspecified, the values of all columns with matching names
     *  are compared.
     *  If array-valued, a two-element array should be provided, containing
     *  the columns to compare for the left and right tables, respectively.
     *  If a one-element array or a string value is provided, the same
     *  column names will be drawn from both tables.
     *  If function-valued, should be a two-table table expression that
     *  returns a boolean value. When providing a custom predicate, note that
     *  join key values can be arrays or objects, and that normal join
     *  semantics do not consider null or undefined values to be equal (that is,
     *  null !== null). Use the op.equal function to handle these cases.
     * @return {this} A new filtered table.
     * @example table.antijoin(other)
     * @example table.antijoin(other, ['keyL', 'keyR'])
     * @example table.antijoin(other, (a, b) => op.equal(a.keyL, b.keyR))
     */
    antijoin(other, on) {
      return antijoin(this, other, on);
    }

    // -- Set Operations ------------------------------------------------------

    /**
     * Concatenate multiple tables into a single table, preserving all rows.
     * This transformation mirrors the UNION_ALL operation in SQL.
     * Only named columns in this table are included in the output.
     * @param  {...import('./types.js').TableRefList} tables
     *  A list of tables to concatenate.
     * @return {this} A new concatenated table.
     * @example table.concat(other)
     * @example table.concat(other1, other2)
     * @example table.concat([other1, other2])
     */
    concat(...tables) {
      return concat(this, ...tables);
    }

    /**
     * Union multiple tables into a single table, deduplicating all rows.
     * This transformation mirrors the UNION operation in SQL. It is
     * similar to *concat* but suppresses duplicate rows with
     * values identical to another row.
     * Only named columns in this table are included in the output.
     * @param  {...import('./types.js').TableRefList} tables
     *  A list of tables to union.
     * @return {this} A new unioned table.
     * @example table.union(other)
     * @example table.union(other1, other2)
     * @example table.union([other1, other2])
     */
    union(...tables) {
      return union$1(this, ...tables);
    }

    /**
     * Intersect multiple tables, keeping only rows whose with identical
     * values for all columns in all tables, and deduplicates the rows.
     * This transformation is similar to a series of *semijoin*.
     * calls, but additionally suppresses duplicate rows.
     * @param  {...import('./types.js').TableRefList} tables
     *  A list of tables to intersect.
     * @return {this} A new filtered table.
     * @example table.intersect(other)
     * @example table.intersect(other1, other2)
     * @example table.intersect([other1, other2])
     */
    intersect(...tables) {
      return intersect(this, ...tables);
    }

    /**
     * Compute the set difference with multiple tables, keeping only rows in
     * this table that whose values do not occur in the other tables.
     * This transformation is similar to a series of *anitjoin*
     * calls, but additionally suppresses duplicate rows.
     * @param  {...import('./types.js').TableRefList} tables
     *  A list of tables to difference.
     * @return {this} A new filtered table.
     * @example table.except(other)
     * @example table.except(other1, other2)
     * @example table.except([other1, other2])
     */
    except(...tables) {
      return except(this, ...tables);
    }

    // -- Table Output Formats ------------------------------------------------

    /**
     * Format this table as a Flechette Arrow table.
     * @param {import('../format/types.js').ArrowFormatOptions} [options]
     *  The Arrow formatting options.
     * @return {import('@uwdata/flechette').Table} A Flechette Arrow table.
     */
    toArrow(options) {
      return toArrow(this, options);
    }

    /**
     * Format this table as binary data in the Apache Arrow IPC format.
     * @param {import('../format/types.js').ArrowIPCFormatOptions} [options]
     *  The Arrow IPC formatting options.
     * @return {Uint8Array} A new Uint8Array of Arrow-encoded binary data.
     */
    toArrowIPC(options) {
      return toArrowIPC(this, options);
    }

    /**
     * Format this table as a comma-separated values (CSV) string. Other
     * delimiters, such as tabs or pipes ('|'), can be specified using
     * the options argument.
     * @param {import('../format/to-csv.js').CSVFormatOptions} [options]
     *   The CSV formatting options.
     * @return {string} A delimited value string.
     */
    toCSV(options) {
      return toCSV(this, options);
    }

    /**
     * Format this table as an HTML table string.
     * @param {import('../format/to-html.js').HTMLFormatOptions} [options]
     *  The HTML formatting options.
     * @return {string} An HTML table string.
     */
    toHTML(options) {
      return toHTML(this, options);
    }

    /**
     * Format this table as a JavaScript Object Notation (JSON) string.
     * @param {import('../format/to-json.js').JSONFormatOptions} [options]
     *  The JSON formatting options.
     * @return {string} A JSON string.
     */
    toJSON(options) {
      return toJSON(this, options);
    }

    /**
     * Format this table as a GitHub-Flavored Markdown table string.
     * @param {import('../format/to-markdown.js').MarkdownFormatOptions} [options]
     *  The Markdown formatting options.
     * @return {string} A GitHub-Flavored Markdown table string.
     */
    toMarkdown(options) {
      return toMarkdown(this, options);
    }
  }

  /**
   * Create a new table backed by an Apache Arrow table instance.
   * @param {import('./types.js').ArrowInput} input
   *  An Apache Arrow data table or Arrow IPC byte buffer.
   * @param {import('./types.js').ArrowOptions} [options]
   *  Options for Arrow import.
   * @return {ColumnTable} A new table containing the imported values.
   */
  function fromArrow(input, options) {
    const { columns = all(), ...rest } = options || {};
    const arrow = input instanceof ArrayBuffer || input instanceof Uint8Array
      ? tableFromIPC(input, { useDate: true, ...rest })
      : input;

    const { fields } = arrow.schema;

    // resolve column selection
    const names = fields.map(f => f.name);
    const sel = resolve({
      columnNames: test => test ? names.filter(test) : names.slice(),
      columnIndex: name => names.indexOf(name)
    }, columns);

    // build Arquero columns for backing Arrow columns
    const cols = columnSet();
    sel.forEach((name, key) => {
      const col = /** @type {import('./types.js').ArrowColumn} */ (arrow.getChild(key));
      cols.add(name, col.type.typeId === -1 ? dictionary(col) : col);
    });

    return new ColumnTable(cols.data, cols.names);
  }

  function dictionary(column) {
    const { data, length, nullCount } = column;
    const batch = data[data.length - 1];
    // support both flechette and arrow-js
    const cache = batch.cache ?? batch.dictionary.toArray();
    const size = cache.length;
    const keys = dictKeys(data, length, nullCount, size);

    const get = nullCount
      ? (k => k === size ? null : cache[k])
      : (k => cache[k]);

    return {
      length,
      nullCount,
      at: row => get(keys[row]),
      key: row => keys[row],
      keyFor(value) {
        if (value === null) return nullCount ? size : -1;
        for (let i = 0; i < size; ++i) {
          if (cache[i] === value) return i;
        }
        return -1;
      },
      groups(names) {
        const s = size + (nullCount ? 1 : 0);
        return {
          keys,
          get: [get],
          names,
          rows: sequence(0, s),
          size: s
        };
      },
      [Symbol.iterator]: () => column[Symbol.iterator](),
      toArray: () => column.toArray()
    };
  }

  /**
   * Generate a dictionary key array.
   * @param {readonly any[]} data Arrow column batches
   * @param {number} length The length of the Arrow column
   * @param {number} nulls The count of column null values
   * @param {number} size The backing dictionary size
   */
  function dictKeys(data, length, nulls, size) {
    const v = data.length > 1 || nulls
      ? flatten(data, length)
      : data[0].values;
    return nulls ? nullKeys(data, v, size) : v;
  }

  /**
   * Flatten Arrow column chunks into a single array.
   */
  function flatten(data, length) {
    const type = data[0].values.constructor;
    const array = new type(length);
    const n = data.length;
    for (let i = 0, idx = 0, len; i < n; ++i) {
      len = data[i].length;
      array.set(data[i].values.subarray(0, len), idx);
      idx += len;
    }
    return array;
  }

  /**
   * Encode null values as an additional dictionary key.
   * Returns a new key array with null values added.
   * TODO: safeguard against integer overflow?
   */
  function nullKeys(data, keys, key) {
    // iterate over null bitmaps, encode null values as key
    const n = data.length;
    for (let i = 0, idx = 0, byte; i < n; ++i) {
      const batch = data[i];
      const { length } = batch;
      // support both flechette and arrow-js
      const validity = batch.validity ?? batch.nullBitmap;
      const m = length >> 3;
      if (validity && validity.length) {
        for (let j = 0; j <= m; ++j) {
          if ((byte = validity[j]) !== 255) {
            const base = idx + (j << 3);
            if ((byte & (1 << 0)) === 0) keys[base + 0] = key;
            if ((byte & (1 << 1)) === 0) keys[base + 1] = key;
            if ((byte & (1 << 2)) === 0) keys[base + 2] = key;
            if ((byte & (1 << 3)) === 0) keys[base + 3] = key;
            if ((byte & (1 << 4)) === 0) keys[base + 4] = key;
            if ((byte & (1 << 5)) === 0) keys[base + 5] = key;
            if ((byte & (1 << 6)) === 0) keys[base + 6] = key;
            if ((byte & (1 << 7)) === 0) keys[base + 7] = key;
          }
        }
      }
      idx += length;
    }
    return keys;
  }

  const parseBoolean = [ // boolean
    v => (v === 'true') || (v === 'false'),
    v => v === 'false' ? false : true
  ];

  const parseNumber = [ // number
    v => v === 'NaN' || (v = +v) === v,
    v => +v
  ];

  const parseDate = [ // iso date
    isISODateString,
    v => new Date(Date.parse(v))
  ];

  function numberParser(options) {
    const { decimal } = options;
    return decimal && decimal !== '.'
      ? parseNumber.map(f => s => f(s && s.replace(decimal, '.')))
      : parseNumber;
  }

  function valueParser(values, options) {
    const types = [parseBoolean, numberParser(options), parseDate];
    const n = types.length;
    for (let i = 0; i < n; ++i) {
      const [test, parser] = types[i];
      if (check$1(values, test)) {
        return parser;
      }
    }
    return identity;
  }

  function check$1(values, test) {
    const n = values.length;
    for (let i = 0; i < n; ++i) {
      const v = values[i];
      if (v != null && !test(v)) {
        return false;
      }
    }
    return true;
  }

  function defaultNames(n, off = 0) {
    return repeat(n - off, i => `col${i + off + 1}`);
  }

  function fromTextRows(next, names, options) {
    let row = next();
    const n = row.length;
    const automax = +options.autoMax || 1000;
    const values = repeat(n, () => []);
    names = names
      ? names.length < n ? [...names, defaultNames(n, names.length)] : names
      : defaultNames(n);

    // read in initial rows to guess types
    let idx = 0;
    for (; idx < automax && row; ++idx, row = next()) {
      for (let i = 0; i < n; ++i) {
        values[i].push(row[i] === '' ? null : row[i]);
      }
    }

    // initialize parsers
    const parsers = getParsers(names, values, options);

    // apply parsers
    parsers.forEach((parse, i) => {
      if (parse === identity) return;
      const v = values[i];
      for (let r = 0; r < idx; ++r) {
        if (v[r] != null) v[r] = parse(v[r]);
      }
    });

    // parse remainder of file
    for (; row; row = next()) {
      for (let i = 0; i < n; ++i) {
        values[i].push(row[i] ? parsers[i](row[i]) : null);
      }
    }

    /** @type {import('../table/types.js').ColumnData} */
    const columns = {};
    names.forEach((name, i) => columns[name] = values[i]);
    return new ColumnTable(columns, names);
  }

  function getParsers(names, values, options) {
    const { parse = {} } = options;
    const noParse = options.autoType === false;

    return names.map(
      (name, i) => isFunction(parse[name]) ? parse[name]
        : noParse ? identity
        : valueParser(values[i], options)
    );
  }

  const EOL = {};
  const EOF = {};
  const QUOTE = 34;
  const NEWLINE = 10;
  const RETURN = 13;

  function filter(read, skip, drop) {
    // skip initial lines, if requested
    let s = +skip || 0;
    while (--s >= 0) read();

    // return filtered stream
    return drop ? () => {
      let line;
      while (!line) {
        if (drop(line = read())) line = null;
        else return line;
      }
    } : read;
  }

  // Adapted from d3-dsv: https://github.com/d3/d3-dsv/blob/master/src/dsv.js
  // Copyright 2013-2016 Mike Bostock
  // All rights reserved.
  // Redistribution and use in source and binary forms, with or without modification,
  // are permitted provided that the following conditions are met:
  // * Redistributions of source code must retain the above copyright notice, this
  //   list of conditions and the following disclaimer.
  // * Redistributions in binary form must reproduce the above copyright notice,
  //   this list of conditions and the following disclaimer in the documentation
  //   and/or other materials provided with the distribution.
  // * Neither the name of the author nor the names of contributors may be used to
  //   endorse or promote products derived from this software without specific prior
  //   written permission.
  // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  // ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  // WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  // ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  // (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  // LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
  // ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  // (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  // SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  function parseDelimited(text, {
    delimiter = ',',
    skip = 0,
    comment = undefined
  }) {
    if (delimiter.length !== 1) {
      error(`Text "delimiter" should be a single character, found "${delimiter}"`);
    }
    const delimCode = delimiter.charCodeAt(0);

    let N = text.length;
    let I = 0; // current character index
    let t; // current token
    let eof = N <= 0; // current token followed by EOF?
    let eol = false; // current token followed by EOL?

    // Strip the trailing newline.
    if (text.charCodeAt(N - 1) === NEWLINE) --N;
    if (text.charCodeAt(N - 1) === RETURN) --N;

    function token() {
      if (eof) return EOF;
      if (eol) return eol = false, EOL;

      // Unescape quotes.
      const j = I;
      let i, c;
      if (text.charCodeAt(j) === QUOTE) {
        while (I++ < N && text.charCodeAt(I) !== QUOTE || text.charCodeAt(++I) === QUOTE);
        if ((i = I) >= N) eof = true;
        else if ((c = text.charCodeAt(I++)) === NEWLINE) eol = true;
        else if (c === RETURN) { eol = true; if (text.charCodeAt(I) === NEWLINE) ++I; }
        return text.slice(j + 1, i - 1).replace(/""/g, '"');
      }

      // Find next delimiter or newline.
      while (I < N) {
        if ((c = text.charCodeAt(i = I++)) === NEWLINE) eol = true;
        else if (c === RETURN) { eol = true; if (text.charCodeAt(I) === NEWLINE) ++I; }
        else if (c !== delimCode) continue;
        return text.slice(j, i);
      }

      // Return last token before EOF.
      return eof = true, text.slice(j, N);
    }

    function next() {
      if ((t = token()) !== EOF) {
        const row = [];
        while (t !== EOL && t !== EOF) row.push(t), t = token();
        return row;
      }
    }

    return filter(
      next, skip,
      comment && (x => (x && x[0] || '').startsWith(comment))
    );
  }

  /**
   * Options for CSV parsing.
   * @typedef {object} CSVParseOptions
   * @property {string} [delimiter=','] Single-character delimiter between values.
   * @property {string} [decimal='.'] Single-character numeric decimal separator.
   * @property {boolean} [header=true] Flag to specify presence of header row.
   *  If true, assumes the CSV contains a header row with column names. If false,
   *  indicates the CSV does not contain a header row; columns are given the
   *  names 'col1', 'col2', etc unless the *names* option is specified.
   * @property {string[]} [names] An array of column names to use for header-less
   *  CSV files. This option is ignored if the header option is true.
   * @property {number} [skip=0] The number of lines to skip before reading data.
   * @property {string} [comment] A string used to identify comment lines. Any
   *  lines that start with the comment pattern are skipped.
   * @property {boolean} [autoType=true] Flag for automatic type inference.
   * @property {number} [autoMax=1000] Maximum number of initial values to use
   *  for type inference.
   * @property {Object.<string, (value: string) => any>} [parse] Object of
   *  column parsing options. The object keys should be column names. The object
   *  values should be parsing functions that transform values upon input.
   */

  /**
   * Parse a comma-separated values (CSV) string into a table. Other
   * delimiters, such as tabs or pipes ('|'), can be specified using
   * the options argument. By default, automatic type inference is performed
   * for input values; string values that match the ISO standard
   * date format are parsed into JavaScript Date objects. To disable this
   * behavior, set the autoType option to false. To perform custom parsing
   * of input column values, use the parse option.
   * @param {string} text A string in a delimited-value format.
   * @param {CSVParseOptions} [options] The formatting options.
   * @return {import('../table/ColumnTable.js').ColumnTable} A new table
   *  containing the parsed values.
   */
  function fromCSV(text, options = {}) {
    const next = parseDelimited(text, options);
    return fromTextRows(
      next,
      options.header !== false ? next() : options.names,
      options
    );
  }

  function parseLines(text, { skip = 0, comment = undefined }) {
    let N = text.length;
    let I = 0; // current character index

    // Strip the trailing newline.
    if (text.charCodeAt(N - 1) === NEWLINE) --N;
    if (text.charCodeAt(N - 1) === RETURN) --N;

    function read() {
      if (I >= N) return;

      const j = I;
      let eol = false;
      let i, c;

      // Find next newline.
      while (I < N) {
        if ((c = text.charCodeAt(i = I++)) === NEWLINE) eol = true;
        else if (c === RETURN) { eol = true; if (text.charCodeAt(I) === NEWLINE) ++I; }
        if (eol) return text.slice(j, i);
      }

      // Return last line before EOF.
      return text.slice(j, N);
    }

    return filter(
      read, skip,
      comment && (x => (x || '').startsWith(comment))
    );
  }

  /**
   * Options for fixed width file parsing.
   * @typedef {object} FixedParseOptions
   * @property {[number, number][]} [positions] Array of start, end indices for
   *  fixed-width columns.
   * @property {number[]} [widths] Array of fixed column widths. This option is
   *  ignored if the positions property is specified.
   * @property {string[]} [names] An array of column names. The array length
   *  should match the length of the positions array. If not specified or
   *  shorter than the positions array, default column names are generated.
   * @property {string} [decimal='.'] Single-character numeric decimal separator.
   * @property {number} [skip=0] The number of lines to skip before reading data.
   * @property {string} [comment] A string used to identify comment lines. Any
   *  lines that start with the comment pattern are skipped.
   * @property {boolean} [autoType=true] Flag for automatic type inference.
   * @property {number} [autoMax=1000] Maximum number of initial values to use
   *  for type inference.
   * @property {Object.<string, (value: string) => any>} [parse] Object of
   *  column parsing options. The object keys should be column names. The object
   *  values should be parsing functions that transform values upon input.
   */

  /**
   * Parse a fixed-width file (FWF) string into a table. By default, automatic
   * type inference is performed for input values; string values that match the
   * ISO standard date format are parsed into JavaScript Date objects. To
   * disable this behavior, set the autoType option to false. To perform custom
   * parsing of input column values, use the parse option.
   * @param {string} text A string in a fixed-width file format.
   * @param {FixedParseOptions} options The formatting options.
   * @return {import('../table/ColumnTable.js').ColumnTable} A new table
   *  containing the parsed values.
   */
  function fromFixed(text, options = {}) {
    const read = parseLines(text, options);
    const p = positions(options);
    return fromTextRows(
      () => {
        const line = read();
        if (line) {
          return p.map(([i, j]) => line.slice(i, j).trim());
        }
      },
      options.names,
      options
    );
  }

  function positions({ positions = undefined, widths = undefined }) {
    if (!positions && !widths) {
      error('Fixed width files require a "positions" or "widths" option');
    }
    let i = 0;
    return positions || widths.map(w => [i, i += w]);
  }

  function isDigitString(value) {
    const n = value.length;
    for (let i = 0; i < n; ++i) {
      const c = value.charCodeAt(i);
      if (c < 48 || c > 57) return false;
    }
    return true;
  }

  /**
   * Options for JSON parsing.
   * @typedef {object} JSONParseOptions
   * @property {boolean} [autoType=true] Flag controlling automatic type
   *  inference. If false, date parsing for input JSON strings is disabled.
   * @property {Object.<string, (value: any) => any>} [parse] Object of column
   *  parsing options. The object keys should be column names. The object values
   *  should be parsing functions that transform values upon input.
   */

  /**
   * Parse JavaScript Object Notation (JSON) data into a table.
   * The expected JSON data format is an object with column names for keys
   * and column value arrays for values. By default string values that match
   * the ISO standard date format are parsed into JavaScript Date objects.
   * To disable this behavior, set the autoType option to false. To perform
   * custom parsing of input column values, use the parse option. Auto-type
   * parsing is not performed for columns with custom parse options.
   * The data payload can also be provided as the "data" property of an
   * enclosing object, with an optional "schema" property containing table
   * metadata such as a "fields" array of ordered column information.
   * @param {string|object} json A string in JSON format, or pre-parsed object.
   * @param {JSONParseOptions} options The formatting options.
   * @return {ColumnTable} A new table containing the parsed values.
   */
  function fromJSON(json, options = {}) {
    const autoType = defaultTrue(options.autoType);

    // parse string input
    if (isString(json)) {
      json = JSON.parse(json);
    }

    // separate schema and data, as needed
    let data = json.data, names;
    if (isObject(data) && !isArrayType(data)) {
      if (json.schema && json.schema.fields) {
        names = json.schema.fields.map(f => f.name);
      }
    } else {
      data = json;
    }

    // parse values as necessary
    if (autoType || options.parse) {
      const parsers = options.parse || {};
      for (const name in data) {
        const col = data[name];
        const len = col.length;
        if (parsers[name]) {
          // apply custom parser
          for (let i = 0; i < len; ++i) {
            col[i] = parsers[name](col[i]);
          }
        } else if (autoType) {
          // apply autoType parser
          for (let i = 0; i < len; ++i) {
            const val = col[i];
            if (isString(val) && isISODateString(val) && !isDigitString(val)) {
              col[i] = new Date(val);
            }
          }
        }
      }
    }

    return new ColumnTable(data, names);
  }

  /**
   * Options for binning number values.
   * @typedef {object} BinOptions
   * @property {number} [maxbins] The maximum number of bins.
   * @property {number} [minstep] The minimum step size between bins.
   * @property {number} [step] The exact step size to use between bins.
   *  If specified, the maxbins and minstep options are ignored.
   * @property {boolean} [nice=true] Flag indicating if bins should
   *  snap to "nice" human-friendly values such as multiples of ten.
   * @property {number} [offset=0] Step offset for bin boundaries.
   *  The default floors to the lower bin boundary. A value of 1 snaps
   *  one step higher to the upper bin boundary, and so on.
   */

  /**
   * Generate a table expression that performs uniform binning of
   * number values. The resulting string can be used as part of the
   * input to table transformation verbs.
   * @param {string} name The name of the column to bin.
   * @param {BinOptions} [options] Binning scheme options.
   * @return {string} A table expression string for binned values.
   * @example bin('colA', { maxbins: 20 })
   */
  function bin(name, options = {}) {
    const field = `d[${JSON.stringify(name)}]`;
    const { maxbins, nice, minstep, step, offset } = options;
    const args = [maxbins, nice, minstep, step];

    let n = args.length;
    while (n && args[--n] == null) args.pop();
    const a = args.length ? ', ' + args.map(a => a + '').join(', ') : '';

    return `d => op.bin(${field}, ...op.bins(${field}${a}), ${offset || 0})`;
  }

  /**
   * Escape a function or value to prevent it from being parsed and recompiled.
   * This helper can be used in lieu of single-table table expressions (which
   * are internally parsed and rewritten) to apply a JavaScript function as-is,
   * including support for closures. It can also be used to pass a constant,
   * literal value as a table expression, bypassing the parser.
   * @param {*} value A function or value to escape.
   * @return {object} A wrapper object representing the escaped value.
   * @example escape(d => d.a.toFixed(2))
   * @example escape(d => d.a * -d.b)
   */
  function escape(value) {
    return wrap$1(value, {
      escape: true,
      toString() { error('Escaped values can not be serialized.'); }
    });
  }

  /**
   * Annotate a table expression with collation metadata, indicating how
   * expression values should be compared and sorted. The orderby verb uses
   * collation metadata to determine sort order. The collation information can
   * either take the form a standard two-argument comparator function, or as
   * locale and option arguments compatible with `Intl.Collator`.
   * @param {string|Function|object} expr The table expression to annotate
   *  with collation metadata.
   * @param {Intl.LocalesArgument | ((a: any, b: any) => number)} comparator
   *  A comparator function or the locale(s) to collate by.
   * @param {Intl.CollatorOptions} [options] Collation options, applicable
   *  with locales only.
   * @return {object} A wrapper object representing the collated value.
   * @example orderby(collate('colA', 'de'))
   */
  function collate(expr, comparator, options) {
    return wrap$1(expr, {
      collate: isFunction(comparator)
        ? comparator
        : new Intl.Collator(comparator, options).compare
    });
  }

  /**
   * Annotate a table expression to indicate descending sort order.
   * @param {string|Function|object} expr The table expression to annotate.
   * @return {object} A wrapped expression indicating descending sort.
   * @example desc('colA')
   * @example desc(d => d.colA)
   */
  function desc(expr) {
    return wrap$1(expr, { desc: true });
  }

  /**
   * Generate a table expression that computes the number of rows
   * corresponding to a given fraction for each group. The resulting
   * string can be used as part of the input to the sample verb.
   * @param {number} fraction The fractional value.
   * @return {string} A table expression string for computing row counts.
   * @example frac(0.5)
   */
  function frac(fraction) {
    return `() => op.round(${+fraction} * op.count())`;
  }

  /**
   * Select columns by index and rename them to the provided names. Returns a
   * selection helper function that takes a table as input and produces a
   * rename map as output. If the number of provided names is less than the
   * number of table columns, the rename map will only include entries for the
   * provided names. If the number of table columns is less than then number of
   * provided names, only the rename map will only include entries that cover
   * the existing columns.
   * @param {...(string|string[])} names An ordered list of column names.
   * @return {Function} Selection function compatible with {@link Table#select}.
   * @example table.rename(aq.names('a', 'b', 'c'))
   * @example table.select(aq.names(['a', 'b', 'c']))
   */
  function names(...names) {
    names = names.flat();
    return table => {
      const m = new Map();
      const n = Math.min(names.length, table.numCols());
      for (let i = 0; i < n; ++i) {
        m.set(table.columnName(i), names[i]);
      }
      return m;
    };
  }

  /**
   * Annotate a table expression to compute rolling aggregate or window
   * functions within a sliding window frame. For example, to specify a
   * rolling 7-day average centered on the current day, use rolling with
   * a frame value of [-3, 3].
   * @param {string|Function|object} expr The table expression to annotate.
   * @param {[number?, number?]} [frame=[-Infinity, 0]] The sliding window frame
   *  offsets. Each entry indicates an offset from the current value. If an
   *  entry is non-finite, the frame will be unbounded in that direction,
   *  including all preceding or following values. If unspecified, the frame
   *  will include the current values and all preceding values.
   * @param {boolean} [includePeers=false] Indicates if the sliding window frame
   *  should ignore peer (tied) values. If false (the default), the window frame
   *  boundaries are insensitive to peer values. If `true`, the window frame
   *  expands to include all peers. This parameter only affects operations that
   *  depend on the window frame: aggregate functions and the first_value,
   *  last_value, and nth_value window functions.
   * @return A new wrapped expression annotated with rolling window parameters.
   * @example rolling(d => mean(d.colA), [-3, 3])
   * @example rolling(d => last_value(d.colA), null, true)
   */
  function rolling(expr, frame, includePeers) {
    return wrap$1(expr, {
      window: {
        frame: frame || [-Infinity, 0],
        peers: !!includePeers
      }
    });
  }

  /**
   * Convenience function for computing a single aggregate value for
   * a table. Equivalent to ungrouping a table, applying a rollup verb
   * for a single aggregate, and extracting the resulting value.
   * @param {import('../../table/Table.js').Table} table A table instance.
   * @param {import('../../table/types.js').TableExpr} expr An
   *   aggregate table expression to evaluate.
   * @return {import('../../table/types.js').DataValue} The aggregate value.
   * @example agg(table, op.max('colA'))
   * @example agg(table, d => [op.min('colA'), op.max('colA')])
   */
  function agg(table, expr) {
    return rollup(ungroup(table), { _: expr }).get('_');
  }

  const onIllegal = (name, type) =>
    error(`Illegal ${type} name: ${toString$1(name)}`);

  const onDefined = (name, type) =>
    error(`The ${type} ${toString$1(name)} is already defined. Use override option?`);

  const onReserve = (name, type) =>
    error(`The ${type} name ${toString$1(name)} is reserved and can not be overridden.`);

  function check(name, options, obj = ops, type = 'function') {
    if (!name) onIllegal(name, type);
    if (!options.override && has(obj, name)) onDefined(name, type);
  }

  function verifyFunction(name, def, object, options) {
    return object[name] === def || check(name, options);
  }

  /**
   * Register an aggregate or window operation.
   * @param {string} name The name of the operation
   * @param {AggregateDef|WindowDef} def The operation definition.
   * @param {object} object The registry object to add the definition to.
   * @param {RegisterOptions} [options] Registration options.
   */
  function addOp(name, def, object, options = {}) {
    if (verifyFunction(name, def, object, options)) return;
    const [nf = 0, np = 0] = def.param; // num fields, num params
    object[name] = def;
    ops[name] = (...params) => op(
      name,
      params.slice(0, nf),
      params.slice(nf, nf + np)
    );
  }

  /**
   * Register a custom aggregate function.
   * @param {string} name The name to use for the aggregate function.
   * @param {AggregateDef} def The aggregate operator definition.
   * @param {RegisterOptions} [options] Function registration options.
   * @throws If a function with the same name is already registered and
   *  the override option is not specified.
   */
  function addAggregateFunction(name, def, options) {
    addOp(name, def, aggregateFunctions, options);
  }

  /**
   * Register a custom window function.
   * @param {string} name The name to use for the window function.
   * @param {WindowDef} def The window operator definition.
   * @param {RegisterOptions} [options] Function registration options.
   * @throws If a function with the same name is already registered and
   *  the override option is not specified.
   */
  function addWindowFunction(name, def, options) {
    addOp(name, def, windowFunctions, options);
  }

  /**
   * Register a function for use within table expressions.
   * If only a single argument is provided, it will be assumed to be a
   * function and the system will try to extract its name.
   * @param {string} name The name to use for the function.
   * @param {Function} fn A standard JavaScript function.
   * @param {RegisterOptions} [options] Function registration options.
   * @throws If a function with the same name is already registered and
   *  the override option is not specified, or if no name is provided
   *  and the input function is anonymous.
   */
  function addFunction(name, fn, options = {}) {
    if (arguments.length === 1) {
      // @ts-ignore
      fn = name;
      name = fn.name;
      if (name === '' || name === 'anonymous') {
        error('Anonymous function provided, please include a name argument.');
      } else if (name === ROW_OBJECT) {
        onReserve(ROW_OBJECT, 'function');
      }
    }
    if (verifyFunction(name, fn, functions, options)) return;
    functions[name] = fn;
    ops[name] = fn;
  }

  /**
   * Aggregate function definition.
   * @typedef {import('./aggregate-functions.js').AggregateDef} AggregateDef
   */

  /**
   * Window function definition.
   * @typedef {import('./window-functions.js').WindowDef} WindowDef
   */

  /**
   * Options for registering new functions.
   * @typedef {object} RegisterOptions
   * @property {boolean} [override=false] Flag indicating if the added
   *  function can override an existing function with the same name.
   */

  /**
   * @return {import('./types.js').ColumnData}
   */
  function columnsFrom(values, names) {
    const raise = type => error(`Illegal argument type: ${type || typeof values}`);
    // @ts-ignore
    return values instanceof Map ? fromKeyValuePairs(values.entries(), names)
      : isDate$1(values) ? raise('Date')
      : isRegExp(values) ? raise('RegExp')
      : isString(values) ? raise()
      : isArray$2(values) ? fromArray(values, names)
      : isFunction(values[Symbol.iterator]) ? fromIterable(values, names)
      : isObject(values) ? fromKeyValuePairs(Object.entries(values), names)
      : raise();
  }

  function fromKeyValuePairs(entries, names = ['key', 'value']) {
    const keys = [];
    const vals = [];

    for (const [key, val] of entries) {
      keys.push(key);
      vals.push(val);
    }

    const columns = {};
    if (names[0]) columns[names[0]] = keys;
    if (names[1]) columns[names[1]] = vals;
    return columns;
  }

  function fromArray(values, names) {
    const len = values.length;
    const columns = {};
    const add = name => columns[name] = Array(len);

    if (len) {
      names = names || Object.keys(values[0]);
      const cols = names.map(add);
      const n = cols.length;
      for (let idx = 0; idx < len; ++idx) {
        const row = values[idx];
        for (let i = 0; i < n; ++i) {
          cols[i][idx] = row[names[i]];
        }
      }
    } else if (names) {
      names.forEach(add);
    }

    return columns;
  }

  function fromIterable(values, names) {
    const columns = {};
    const add = name => columns[name] = [];

    let cols;
    let n;
    for (const row of values) {
      if (!cols) {
        names = names || Object.keys(row);
        cols = names.map(add);
        n = cols.length;
      }
      for (let i = 0; i < n; ++i) {
        cols[i].push(row[names[i]]);
      }
    }

    if (!cols && names) {
      names.forEach(add);
    }

    return columns;
  }

  /**
   * Create a new table for a set of named columns.
   * @param {object|Map} columns
   *  The set of named column arrays. Keys are column names.
   *  The enumeration order of the keys determines the column indices,
   *  unless the names parameter is specified.
   *  Values must be arrays (or array-like values) of identical length.
   * @param {string[]} [names] Ordered list of column names. If specified,
   *  this array determines the column indices. If not specified, the
   *  key enumeration order of the columns object is used.
   * @return {ColumnTable} the instantiated table
   * @example table({ colA: ['a', 'b', 'c'], colB: [3, 4, 5] })
   */
  function table(columns, names) {
    if (columns instanceof ColumnTable) return columns;
    /** @type {import('./types.js').ColumnData} */
    const data = {};
    const keys = [];
    for (const [key, value] of entries(columns)) {
      data[key] = value;
      keys.push(key);
    }
    return new ColumnTable(data, names || keys);
  }

  /**
   * Create a new table from an existing object, such as an array of
   * objects or a set of key-value pairs.
   * @param {object|Array|Map} values Data values to populate the table.
   *  If array-valued or iterable, imports rows for each non-null value,
   *  using the provided column names as keys for each row object. If no
   *  names are provided, the first non-null object's own keys are used.
   *  If object- or Map-valued, create columns for the keys and values.
   * @param {string[]} [names] Column names to include.
   *  For object or Map values, specifies the key and value column names.
   *  Otherwise, specifies the keys to look up on each row object.
   * @return {ColumnTable} the instantiated table.
   * @example from([ { colA: 1, colB: 2 }, { colA: 3, colB: 4 } ])
   */
  function from(values, names) {
    return new ColumnTable(columnsFrom(values, names), names);
  }

  /**
   * @typedef {import('../table/ColumnTable.js').ColumnTable} ColumnTable
   */

  /**
   * Options for file loading.
   * @typedef {object} LoadOptions
   * @property {'arrayBuffer'|'text'|'json'} [as='text'] A string indicating
   *  the data type of the file. One of 'arrayBuffer', 'json', or 'text'.
   * @property {(data: *, options?: object) => ColumnTable} [using] A function
   *  that accepts a data payload (e.g., string or buffer) and an options object
   *  as input and returns an Arquero table (such as fromCSV or fromJSON).
   * @property {object} [fetch] Options to pass to the HTTP fetch method
   *  when loading a URL.
   */

  /**
   * Load data from a file and return a Promise for an Arquero table.
   * A specific format parser can be provided with the *using* option,
   * otherwise CSV format is assumed. The options to this method are
   * passed as the second argument to the format parser.
   * @param {string} url The URL to load.
   * @param {LoadOptions & object} options The loading and formatting options.
   * @return {Promise<ColumnTable>} A Promise for an Arquero table.
   * @example aq.load('data/table.csv')
   * @example aq.load('data/table.json', { using: aq.fromJSON })
   * @example aq.load('data/table.json', { using: aq.from })
   */
  function load(url, options = {}) {
    const parse = options.using || fromCSV;
    return fetch(url, options.fetch)
      .then(res => res[options.as || 'text']())
      .then(data => parse(data, options));
  }

  /**
   * Load an Arrow file from a URL and return a Promise for an Arquero table.
   * @param {string} url The URL to load.
   * @param {LoadOptions & import('./types.js').ArrowOptions} [options]
   *  Arrow format options.
   * @return {Promise<ColumnTable>} A Promise for an Arquero table.
   * @example aq.loadArrow('data/table.arrow')
   */
  function loadArrow(url, options) {
    return load(url, { ...options, as: 'arrayBuffer', using: fromArrow });
  }

  /**
   * Load a CSV file from a URL and return a Promise for an Arquero table.
   * @param {string} url The URL to load.
   * @param {LoadOptions & import('./from-csv.js').CSVParseOptions} [options]
   *  CSV format options.
   * @return {Promise<ColumnTable>} A Promise for an Arquero table.
   * @example aq.loadCSV('data/table.csv')
   * @example aq.loadTSV('data/table.tsv', { delimiter: '\t' })
   */
  function loadCSV(url, options) {
    return load(url, { ...options, as: 'text', using: fromCSV });
  }

  /**
   * Load a fixed width file from a URL and return a Promise for an Arquero table.
   * @param {string} url The URL to load.
   * @param {LoadOptions & import('./from-fixed.js').FixedParseOptions} [options]
   *  Fixed width format options.
   * @return {Promise<ColumnTable>} A Promise for an Arquero table.
   * @example aq.loadFixedWidth('data/table.txt', { names: ['name', 'city', state'], widths: [10, 20, 2] })
   */
   function loadFixed(url, options) {
    return load(url, { ...options, as: 'text', using: fromFixed });
  }

  /**
   * Load a JSON file from a URL and return a Promise for an Arquero table.
   * If the loaded JSON is array-valued, an array-of-objects format is assumed
   * and the aq.from method is used to construct the table. Otherwise, a
   * column object format is assumed and aq.fromJSON is applied.
   * @param {string} url The URL to load.
   * @param {LoadOptions & import('./from-json.js').JSONParseOptions} [options]
   *  JSON format options.
   * @return {Promise<ColumnTable>} A Promise for an Arquero table.
   * @example aq.loadJSON('data/table.json')
   */
  function loadJSON(url, options) {
    return load(url, { ...options, as: 'json', using: parseJSON });
  }

  function parseJSON(data, options) {
    return isArray$2(data) ? from(data) : fromJSON(data, options);
  }

  exports.BitSet = BitSet;
  exports.ColumnTable = ColumnTable;
  exports.Reducer = Reducer;
  exports.Table = Table$1;
  exports.addAggregateFunction = addAggregateFunction;
  exports.addFunction = addFunction;
  exports.addWindowFunction = addWindowFunction;
  exports.agg = agg;
  exports.all = all;
  exports.bin = bin;
  exports.collate = collate;
  exports.desc = desc;
  exports.endswith = endswith;
  exports.escape = escape;
  exports.field = field$1;
  exports.frac = frac;
  exports.from = from;
  exports.fromArrow = fromArrow;
  exports.fromCSV = fromCSV;
  exports.fromFixed = fromFixed;
  exports.fromJSON = fromJSON;
  exports.load = load;
  exports.loadArrow = loadArrow;
  exports.loadCSV = loadCSV;
  exports.loadFixed = loadFixed;
  exports.loadJSON = loadJSON;
  exports.matches = matches;
  exports.names = names;
  exports.not = not;
  exports.op = ops;
  exports.parse = parse$2;
  exports.range = range;
  exports.rolling = rolling;
  exports.seed = seed;
  exports.startswith = startswith;
  exports.table = table;
  exports.toArrow = toArrow;
  exports.toArrowIPC = toArrowIPC;
  exports.toCSV = toCSV;
  exports.toHTML = toHTML;
  exports.toJSON = toJSON;
  exports.toMarkdown = toMarkdown;
  exports.walk_ast = walk;

}));
