/**
 * @typedef {import('../table/ColumnTable.js').ColumnTable} ColumnTable
 */
/**
 * Options for file loading.
 * @typedef {object} LoadOptions
 * @property {'arrayBuffer'|'text'|'json'} [as='text'] A string indicating
 *  the data type of the file. One of 'arrayBuffer', 'json', or 'text'.
 * @property {(data: *, options?: object) => ColumnTable} [using] A function
 *  that accepts a data payload (e.g., string or buffer) and an options object
 *  as input and returns an Arquero table (such as fromCSV or fromJSON).
 * @property {object} [fetch] Options to pass to the HTTP fetch method
 *  when loading a URL.
 */
/**
 * Load data from a file and return a Promise for an Arquero table.
 * A specific format parser can be provided with the *using* option,
 * otherwise CSV format is assumed. The options to this method are
 * passed as the second argument to the format parser.
 * @param {string} url The URL to load.
 * @param {LoadOptions & object} options The loading and formatting options.
 * @return {Promise<ColumnTable>} A Promise for an Arquero table.
 * @example aq.load('data/table.csv')
 * @example aq.load('data/table.json', { using: aq.fromJSON })
 * @example aq.load('data/table.json', { using: aq.from })
 */
export function load(url: string, options?: LoadOptions & object): Promise<ColumnTable>;
/**
 * Load an Arrow file from a URL and return a Promise for an Arquero table.
 * @param {string} url The URL to load.
 * @param {LoadOptions & import('./types.js').ArrowOptions} [options]
 *  Arrow format options.
 * @return {Promise<ColumnTable>} A Promise for an Arquero table.
 * @example aq.loadArrow('data/table.arrow')
 */
export function loadArrow(url: string, options?: LoadOptions & import("./types.js").ArrowOptions): Promise<ColumnTable>;
/**
 * Load a CSV file from a URL and return a Promise for an Arquero table.
 * @param {string} url The URL to load.
 * @param {LoadOptions & import('./from-csv.js').CSVParseOptions} [options]
 *  CSV format options.
 * @return {Promise<ColumnTable>} A Promise for an Arquero table.
 * @example aq.loadCSV('data/table.csv')
 * @example aq.loadTSV('data/table.tsv', { delimiter: '\t' })
 */
export function loadCSV(url: string, options?: LoadOptions & import("./from-csv.js").CSVParseOptions): Promise<ColumnTable>;
/**
 * Load a fixed width file from a URL and return a Promise for an Arquero table.
 * @param {string} url The URL to load.
 * @param {LoadOptions & import('./from-fixed.js').FixedParseOptions} [options]
 *  Fixed width format options.
 * @return {Promise<ColumnTable>} A Promise for an Arquero table.
 * @example aq.loadFixedWidth('data/table.txt', { names: ['name', 'city', state'], widths: [10, 20, 2] })
 */
export function loadFixed(url: string, options?: LoadOptions & import("./from-fixed.js").FixedParseOptions): Promise<ColumnTable>;
/**
 * Load a JSON file from a URL and return a Promise for an Arquero table.
 * If the loaded JSON is array-valued, an array-of-objects format is assumed
 * and the aq.from method is used to construct the table. Otherwise, a
 * column object format is assumed and aq.fromJSON is applied.
 * @param {string} url The URL to load.
 * @param {LoadOptions & import('./from-json.js').JSONParseOptions} [options]
 *  JSON format options.
 * @return {Promise<ColumnTable>} A Promise for an Arquero table.
 * @example aq.loadJSON('data/table.json')
 */
export function loadJSON(url: string, options?: LoadOptions & import("./from-json.js").JSONParseOptions): Promise<ColumnTable>;
export type ColumnTable = import("../table/ColumnTable.js").ColumnTable;
/**
 * Options for file loading.
 */
export type LoadOptions = {
    /**
     * A string indicating
     * the data type of the file. One of 'arrayBuffer', 'json', or 'text'.
     */
    as?: "arrayBuffer" | "text" | "json";
    /**
     * A function
     * that accepts a data payload (e.g., string or buffer) and an options object
     * as input and returns an Arquero table (such as fromCSV or fromJSON).
     */
    using?: (data: any, options?: object) => ColumnTable;
    /**
     * Options to pass to the HTTP fetch method
     * when loading a URL.
     */
    fetch?: object;
};
