/**
 * Base class representing a column-oriented data table.
 */
export class Table {
    /**
     * Instantiate a Table instance.
     * @param {import('./types.js').ColumnData} columns
     *  An object mapping column names to values.
     * @param {string[]} [names]
     *  An ordered list of column names.
     * @param {import('./BitSet.js').BitSet} [filter]
     *  A filtering BitSet.
     * @param {import('./types.js').GroupBySpec} [group]
     *  A groupby specification.
     * @param {import('./types.js').RowComparator} [order]
     *  A row comparator function.
     * @param {import('./types.js').Params} [params]
     *  An object mapping parameter names to values.
     */
    constructor(columns: import("./types.js").ColumnData, names?: string[], filter?: import("./BitSet.js").BitSet, group?: import("./types.js").GroupBySpec, order?: import("./types.js").RowComparator, params?: import("./types.js").Params);
    /**
     * @private
     * @type {readonly string[]}
     */
    private _names;
    /**
     * @private
     * @type {import('./types.js').ColumnData}
     */
    private _data;
    /**
     * @private
     * @type {number}
     */
    private _total;
    /**
     * @private
     * @type {number}
     */
    private _nrows;
    /**
     * @private
     * @type {import('./BitSet.js').BitSet}
     */
    private _mask;
    /**
     * @private
     * @type {import('./types.js').GroupBySpec}
     */
    private _group;
    /**
     * @private
     * @type {import('./types.js').RowComparator}
     */
    private _order;
    /**
     * @private
     * @type {import('./types.js').Params}
     */
    private _params;
    /**
     * @private
     * @type {Uint32Array}
     */
    private _index;
    /**
     * @private
     * @type {number[][] | Uint32Array[]}
     */
    private _partitions;
    /**
     * Create a new table with the same type as this table.
     * The new table may have different data, filter, grouping, or ordering
     * based on the values of the optional configuration argument. If a
     * setting is not specified, it is inherited from the current table.
     * @param {import('./types.js').CreateOptions} [options]
     *  Creation options for the new table.
     * @return {this} A newly created table.
     */
    create({ data, names, filter, groups, order }?: import("./types.js").CreateOptions): this;
    /**
     * Get or set table expression parameter values.
     * If called with no arguments, returns the current parameter values
     * as an object. Otherwise, adds the provided parameters to this
     * table's parameter set and returns the table. Any prior parameters
     * with names matching the input parameters are overridden.
     * @param {import('./types.js').Params} [values]
     *  The parameter values.
     * @return {this|import('./types.js').Params}
     *  The current parameter values (if called with no arguments) or this table.
     */
    params(values?: import("./types.js").Params, ...args: any[]): this | import("./types.js").Params;
    /**
     * Indicates if the table has a filter applied.
     * @return {boolean} True if filtered, false otherwise.
     */
    isFiltered(): boolean;
    /**
     * Indicates if the table has a groupby specification.
     * @return {boolean} True if grouped, false otherwise.
     */
    isGrouped(): boolean;
    /**
     * Indicates if the table has a row order comparator.
     * @return {boolean} True if ordered, false otherwise.
     */
    isOrdered(): boolean;
    /**
     * Get the backing column data for this table.
     * @return {import('./types.js').ColumnData}
     *  Object of named column instances.
     */
    data(): import("./types.js").ColumnData;
    /**
     * Returns the filter bitset mask, if defined.
     * @return {import('./BitSet.js').BitSet} The filter bitset mask.
     */
    mask(): import("./BitSet.js").BitSet;
    /**
     * Returns the groupby specification, if defined.
     * @return {import('./types.js').GroupBySpec} The groupby specification.
     */
    groups(): import("./types.js").GroupBySpec;
    /**
     * Returns the row order comparator function, if specified.
     * @return {import('./types.js').RowComparator}
     *  The row order comparator function.
     */
    comparator(): import("./types.js").RowComparator;
    /**
     * The total number of rows in this table, counting both
     * filtered and unfiltered rows.
     * @return {number} The number of total rows.
     */
    totalRows(): number;
    /**
     * The number of active rows in this table. This number may be
     * less than the *totalRows* if the table has been filtered.
     * @return {number} The number of rows.
     */
    numRows(): number;
    /**
     * The number of active rows in this table. This number may be
     * less than the *totalRows* if the table has been filtered.
     * @return {number} The number of rows.
     */
    get size(): number;
    /**
     * The number of columns in this table.
     * @return {number} The number of columns.
     */
    numCols(): number;
    /**
     * Filter function invoked for each column name.
     * @callback NameFilter
     * @param {string} name The column name.
     * @param {number} index The column index.
     * @param {string[]} array The array of names.
     * @return {boolean} Returns true to retain the column name.
     */
    /**
     * The table column names, optionally filtered.
     * @param {NameFilter} [filter] An optional filter function.
     *  If unspecified, all column names are returned.
     * @return {string[]} An array of matching column names.
     */
    columnNames(filter?: (name: string, index: number, array: string[]) => boolean): string[];
    /**
     * The column name at the given index.
     * @param {number} index The column index.
     * @return {string} The column name,
     *  or undefined if the index is out of range.
     */
    columnName(index: number): string;
    /**
     * The column index for the given name.
     * @param {string} name The column name.
     * @return {number} The column index, or -1 if the name is not found.
     */
    columnIndex(name: string): number;
    /**
     * Get the column instance with the given name.
     * @param {string} name The column name.
     * @return {import('./types.js').ColumnType | undefined}
     *  The named column, or undefined if it does not exist.
     */
    column(name: string): import("./types.js").ColumnType<any> | undefined;
    /**
     * Get the column instance at the given index position.
     * @param {number} index The zero-based column index.
     * @return {import('./types.js').ColumnType | undefined}
     *  The column, or undefined if it does not exist.
     */
    columnAt(index: number): import("./types.js").ColumnType<any> | undefined;
    /**
     * Get an array of values contained in a column. The resulting array
     * respects any table filter or orderby criteria.
     * @param {string} name The column name.
     * @param {ArrayConstructor | import('./types.js').TypedArrayConstructor} [constructor=Array]
     *  The array constructor for instantiating the output array.
     * @return {import('./types.js').DataValue[] | import('./types.js').TypedArray}
     *  The array of column values.
     */
    array(name: string, constructor?: ArrayConstructor | import("./types.js").TypedArrayConstructor): import("./types.js").DataValue[] | import("./types.js").TypedArray;
    /**
     * Get the value for the given column and row.
     * @param {string} name The column name.
     * @param {number} [row=0] The row index, defaults to zero if not specified.
     * @return {import('./types.js').DataValue} The table value at (column, row).
     */
    get(name: string, row?: number): import("./types.js").DataValue;
    /**
     * Returns an accessor ("getter") function for a column. The returned
     * function takes a row index as its single argument and returns the
     * corresponding column value.
     * @param {string} name The column name.
     * @return {import('./types.js').ColumnGetter} The column getter function.
     */
    getter(name: string): import("./types.js").ColumnGetter;
    /**
     * Returns an object representing a table row.
     * @param {number} [row=0] The row index, defaults to zero if not specified.
     * @return {object} A row object with named properties for each column.
     */
    object(row?: number): object;
    /**
     * Returns an array of objects representing table rows.
     * @param {import('./types.js').ObjectsOptions} [options]
     *  The options for row object generation.
     * @return {object[]} An array of row objects.
     */
    objects(options?: import("./types.js").ObjectsOptions): object[];
    /**
     * Returns an iterator over column values.
     * @return {Iterator<object>} An iterator over row objects.
     */
    values(name: any): Iterator<object>;
    /**
     * Print the contents of this table using the console.table() method.
     * @param {import('./types.js').PrintOptions|number} options
     *  The options for row object generation, determining which rows and
     *  columns are printed. If number-valued, specifies the row limit.
     * @return {this} The table instance.
     */
    print(options?: import("./types.js").PrintOptions | number): this;
    /**
     * Returns an array of indices for all rows passing the table filter.
     * @param {boolean} [order=true] A flag indicating if the returned
     *  indices should be sorted if this table is ordered. If false, the
     *  returned indices may or may not be sorted.
     * @return {Uint32Array} An array of row indices.
     */
    indices(order?: boolean): Uint32Array;
    /**
     * Returns an array of indices for each group in the table.
     * If the table is not grouped, the result is the same as
     * the *indices* method, but wrapped within an array.
     * @param {boolean} [order=true] A flag indicating if the returned
     *  indices should be sorted if this table is ordered. If false, the
     *  returned indices may or may not be sorted.
     * @return {number[][] | Uint32Array[]} An array of row index arrays, one
     *  per group. The indices will be filtered if the table is filtered.
     */
    partitions(order?: boolean): number[][] | Uint32Array[];
    /**
     * Create a new fully-materialized instance of this table.
     * All filter and orderby settings are removed from the new table.
     * Instead, the backing data itself is filtered and ordered as needed.
     * @param {number[]} [indices] Ordered row indices to materialize.
     *  If unspecified, all rows passing the table filter are used.
     * @return {this} A reified table.
     */
    reify(indices?: number[]): this;
    /**
     * Callback function to cancel a table scan.
     * @callback ScanStop
     * @return {void}
     */
    /**
     * Callback function invoked for each row of a table scan.
     * @callback ScanVisitor
     * @param {number} [row] The table row index.
     * @param {import('./types.js').ColumnData} [data]
     *  The backing table data store.
     * @param {ScanStop} [stop] Function to stop the scan early.
     *  Callees can invoke this function to prevent future calls.
     * @return {void}
     */
    /**
     * Perform a table scan, visiting each row of the table.
     * If this table is filtered, only rows passing the filter are visited.
     * @param {ScanVisitor} fn Callback invoked for each row of the table.
     * @param {boolean} [order=false] Indicates if the table should be
     *  scanned in the order determined by *orderby*. This
     *  argument has no effect if the table is unordered.
     * @property {number} [limit=Infinity] The maximum number of rows to scan.
     * @property {number} [offset=0] The row offset indicating how many
     *  initial rows to skip.
     */
    scan(fn: (row?: number, data?: import("./types.js").ColumnData, stop?: () => void) => void, order?: boolean, limit?: number, offset?: number): void;
    /**
     * Provide an informative object string tag.
     */
    get [Symbol.toStringTag](): string;
    /**
     * Returns an iterator over objects representing table rows.
     * @return {Iterator<object>} An iterator over row objects.
     */
    [Symbol.iterator](): Iterator<object>;
}
