"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.render = void 0;
/**
 * Normalizes given props with render children to string and save value to `childrenContent` prop
 * @private
 * @param {Any} props
 * @returns {Any} normalized props.
 */
function normalizeProps(props) {
    return __assign(__assign({}, props), { childrenContent: render(props.children) });
}
/**
 * Executes a `render` method on a given component (in the case of a class component)
 * or executes the component itself (a functional component) to get pure string or complex value for the next operations
 * @private
 * @param {React.ReactElement} element a given component or complex form
 * @returns {(React.ReactElement | string)}
 */
function createElement(element) {
    if (!element) {
        return "";
    }
    var typeOf = typeof element.type;
    if (typeOf === 'string') {
        // HTML (also not standard) tags case
        throw new Error("HTML tags is not supported yet.");
    }
    else if (typeOf === 'symbol') {
        // internal React types like Fragments, Portal etc. We skip them.
        return render(element.props.children);
    }
    else if (typeOf === 'function') {
        // custom components case
        var type = element.type;
        var prototype = type.prototype;
        // Class component case
        if (prototype && typeof prototype.isReactComponent === "object") {
            var clazzComp = new type(normalizeProps(element.props));
            return createElement(clazzComp.render());
        }
        // Function component case
        return createElement(type(normalizeProps(element.props)));
    }
    return render(element) || "";
}
/**
 * Renders given component to string
 *
 * @param {ReactNode} component a given component to rendering
 * @example
 * function Component({ textProp }) {
 *   return <>{textProp}</>
 * }
 * render(<Component textProp="someText" />)
 * @returns {string}
 */
function render(component) {
    var content = "";
    var typeOf = typeof component;
    if (typeOf === 'string') {
        content += component;
    }
    else if (Array.isArray(component)) {
        content += component.map(function (child) {
            var childValue = createElement(child);
            return render(childValue);
        }).join("");
    }
    else if (typeOf === "object") {
        content += createElement(component);
    }
    return content;
}
exports.render = render;
//# sourceMappingURL=renderer.js.map