"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.withIndendation = exports.IndentationTypes = void 0;
/**
 * Type of indentation to use
 * @readonly
 * @enum {string}
 * @property {string}  TABS   - indicate to use tabs as separator
 * @property {string}  SPACES - indicate to use spaces as separator
 */
var IndentationTypes;
(function (IndentationTypes) {
    IndentationTypes["TABS"] = "tabs";
    IndentationTypes["SPACES"] = "spaces";
})(IndentationTypes = exports.IndentationTypes || (exports.IndentationTypes = {}));
/**
 * Ensures indentations are prepended to content.
 * @private
 * @param {string} content to prepend the indentation
 * @param {number} size the number of indendations to use
 * @param {IndentationTypes} type the type of indendations to use. SPACES by default.
 * @returns {string}
 */
function withIndendation(content, size, type) {
    if (content === void 0) { content = ''; }
    if (type === void 0) { type = IndentationTypes.SPACES; }
    if (size < 1) {
        return content;
    }
    // if the content includes new lines ensure that they have the added indentation as well.
    if (content.includes('\n')) {
        var newLineArray = content.split('\n');
        return newLineArray.reduce(function (accumulator, value) {
            var newValue = value.trim() === '' ? value : "" + getIndentation(size, type) + value;
            return accumulator === "" ? newValue : accumulator + "\n" + newValue;
        }, "");
    }
    return "" + getIndentation(size, type) + content;
}
exports.withIndendation = withIndendation;
/**
 * Get the indendation string based on how many and which type of indentation are requested.
 * @private
 * @param {number} size the number of indendations to use
 * @param {IndentationTypes} type the type of indendations to use. SPACES by default.
 * @returns {string}
 */
function getIndentation(size, type) {
    if (size === void 0) { size = 0; }
    if (type === void 0) { type = IndentationTypes.SPACES; }
    var whitespaceChar = type === IndentationTypes.SPACES ? ' ' : '\t';
    return Array(size).fill(whitespaceChar).join("");
}
//# sourceMappingURL=withIndendation.js.map