const OPTION_PREFIX = '(buf.validate.field)';
export function visit(obj, field) {
    const parsedOption = findRootOption(field);
    if (parsedOption !== null) {
        protocGenValidate(parsedOption, obj);
    }
}
function findRootOption(field) {
    if (field.parsedOptions && field.parsedOptions[OPTION_PREFIX]) {
        return field.parsedOptions[OPTION_PREFIX];
    }
    else if (field.parsedOptions && Array.isArray(field.parsedOptions)) {
        for (const parsedOption of field.parsedOptions) {
            if (parsedOption[OPTION_PREFIX]) {
                return parsedOption[OPTION_PREFIX];
            }
        }
    }
    return null;
}
export function protocGenValidate(option, obj) {
    for (const [optionKey, value] of Object.entries(option)) {
        switch (optionKey) {
            case 'float':
            case 'double':
            case 'int32':
            case 'int64':
            case 'uint32':
            case 'uint64':
            case 'sint32':
            case 'sint64':
            case 'fixed32':
            case 'fixed64':
            case 'sfixed32':
            case 'sfixed64':
                ProtocGenNumeric.handle(obj, value);
                break;
            case 'bool':
                ProtocGenBool.handle(obj, value);
                break;
            case 'string':
            case 'bytes':
                ProtocGenString.handle(obj, value);
                break;
            case 'repeated':
                ProtocGenRepeated.handle(obj, value);
                break;
        }
    }
}
// https://github.com/bufbuild/protovalidate/blob/main/examples/option_number_range.proto
class ProtocGenNumeric {
    static handle(obj, option) {
        for (const [optionKey, value] of Object.entries(option)) {
            switch (optionKey) {
                case 'const':
                    ProtocGenGeneric.constValue(obj, value);
                    break;
                case 'lt':
                    ProtocGenNumeric.lessThan(obj, value);
                    break;
                case 'lte':
                    ProtocGenNumeric.lessEqualThan(obj, value);
                    break;
                case 'gt':
                    ProtocGenNumeric.greaterThan(obj, value);
                    break;
                case 'gte':
                    ProtocGenNumeric.greaterEqualThan(obj, value);
                    break;
                case 'ignore_empty':
                    // implemented via isOptional
                    break;
                case 'in':
                    ProtocGenGeneric.inArray(obj, value);
                    break;
                case 'not_in':
                    ProtocGenGeneric.notInArray(obj, value);
                    break;
            }
        }
    }
    // x must equal `value` less than
    static lessThan(obj, value) {
        delete obj.maximum;
        obj.exclusiveMaximum = value;
    }
    // x must be greater less or equal to `value`
    static lessEqualThan(obj, value) {
        obj.maximum = value;
    }
    // x must equal `value` greater than
    static greaterThan(obj, value) {
        delete obj.minimum;
        obj.exclusiveMinimum = value;
    }
    // x must be greater than or equal to `value`
    static greaterEqualThan(obj, value) {
        obj.minimum = value;
    }
}
// https://github.com/bufbuild/protoc-gen-validate/blob/main/tests/harness/cases/bool.proto
class ProtocGenBool {
    static handle(obj, option) {
        for (const [optionKey, value] of Object.entries(option)) {
            switch (optionKey) {
                case 'const':
                    ProtocGenGeneric.constValue(obj, value);
                    break;
            }
        }
    }
}
// https://github.com/bufbuild/protoc-gen-validate/blob/main/tests/harness/cases/strings.proto
class ProtocGenString {
    static handle(obj, option) {
        for (const [optionKey, value] of Object.entries(option)) {
            switch (optionKey) {
                case 'const':
                    ProtocGenGeneric.constValue(obj, value);
                    break;
                case 'in':
                    ProtocGenGeneric.inArray(obj, value);
                    break;
                case 'not_in':
                    ProtocGenGeneric.notInArray(obj, value);
                    break;
                case 'len':
                    ProtocGenString.len(obj, value);
                    break;
                case 'min_len':
                    ProtocGenString.minLen(obj, value);
                    break;
                case 'max_len':
                    ProtocGenString.maxLen(obj, value);
                    break;
                case 'len_bytes':
                    ProtocGenString.len(obj, value);
                    break;
                case 'min_bytes':
                    ProtocGenString.minLen(obj, value);
                    break;
                case 'max_bytes':
                    ProtocGenString.maxLen(obj, value);
                    break;
                case 'pattern':
                    ProtocGenString.pattern(obj, value);
                    break;
                case 'prefix':
                    ProtocGenString.prefix(obj, value);
                    break;
                case 'contains':
                    ProtocGenString.contains(obj, value);
                    break;
                case 'not_contains':
                    ProtocGenString.notContains(obj, value);
                    break;
                case 'suffix':
                    ProtocGenString.suffix(obj, value);
                    break;
                case 'email':
                    if (value) {
                        ProtocGenString.email(obj);
                    }
                    break;
                case 'address':
                    if (value) {
                        ProtocGenString.address(obj);
                    }
                    break;
                case 'hostname':
                    if (value) {
                        ProtocGenString.hostname(obj);
                    }
                    break;
                case 'ip':
                    if (value) {
                        ProtocGenString.ip(obj);
                    }
                    break;
                case 'ipv4':
                    if (value) {
                        ProtocGenString.ipv4(obj);
                    }
                    break;
                case 'ipv6':
                    if (value) {
                        ProtocGenString.ipv6(obj);
                    }
                    break;
                case 'uri':
                    if (value) {
                        ProtocGenString.uri(obj);
                    }
                    break;
                case 'uri_ref':
                    if (value) {
                        ProtocGenString.uriRef(obj);
                    }
                    break;
                case 'uuid':
                    if (value) {
                        ProtocGenString.uuid(obj);
                    }
                    break;
                case 'well_known_regex':
                    ProtocGenString.wellKnownRegex(obj, value);
                    break;
                case 'ignore_empty':
                    // implemented via isOptional
                    break;
            }
        }
    }
    static len(obj, value) {
        obj.minLength = value;
        obj.maxLength = value;
    }
    static minLen(obj, value) {
        obj.minLength = value;
    }
    static maxLen(obj, value) {
        obj.maxLength = value;
    }
    static pattern(obj, value) {
        obj.pattern = value;
    }
    static prefix(obj, value) {
        obj.pattern = `^${escapeRegExp(value)}.*`;
    }
    static contains(obj, value) {
        obj.pattern = `.*${escapeRegExp(value)}.*`;
    }
    static notContains(obj, value) {
        obj.pattern = `^((?!${escapeRegExp(value)}).)*$`;
    }
    static suffix(obj, value) {
        obj.pattern = `.*${escapeRegExp(value)}$`;
    }
    static email(obj) {
        obj.format = 'email';
    }
    static hostname(obj) {
        obj.format = 'hostname';
    }
    static address(obj) {
        obj.anyOf = [
            { format: 'hostname' },
            { format: 'ipv4' },
            { format: 'ipv6' },
        ];
    }
    static ip(obj) {
        obj.anyOf = [
            { format: 'ipv4' },
            { format: 'ipv6' },
        ];
    }
    static ipv4(obj) {
        obj.format = 'ipv4';
    }
    static ipv6(obj) {
        obj.format = 'ipv6';
    }
    static uri(obj) {
        obj.format = 'uri';
    }
    static uriRef(obj) {
        obj.format = 'uri-reference';
    }
    static uuid(obj) {
        obj.format = 'uuid';
    }
    static wellKnownRegex(obj, value) {
        switch (value) {
            case 'HTTP_HEADER_NAME':
                obj.pattern = '^:?[0-9a-zA-Z!#$%&\'*+-.^_|~\x60]+$';
                break;
            case 'HTTP_HEADER_VALUE':
                obj.pattern = '^[^\u0000-\u0008\u000A-\u001F\u007F]*$';
                break;
        }
    }
}
// https://github.com/bufbuild/protoc-gen-validate/blob/main/tests/harness/cases/repeated.proto
class ProtocGenRepeated {
    static handle(obj, option) {
        for (const [optionKey, value] of Object.entries(option)) {
            switch (optionKey) {
                case 'min_items':
                    ProtocGenRepeated.minLen(obj, value);
                    break;
                case 'max_items':
                    ProtocGenRepeated.maxLen(obj, value);
                    break;
                case 'unique':
                    if (value) {
                        ProtocGenRepeated.unique(obj);
                    }
                    break;
                case 'items':
                    if (obj.items) { // avoid null pointer
                        protocGenValidate(value, obj.items);
                    }
                    break;
            }
        }
    }
    static minLen(obj, value) {
        obj.minItems = value;
    }
    static maxLen(obj, value) {
        obj.maxItems = value;
    }
    static unique(obj) {
        obj.uniqueItems = true;
    }
}
class ProtocGenGeneric {
    // x must equal `value` exactly
    static constValue(obj, value) {
        obj.const = value;
        delete obj.maximum;
        delete obj.minimum;
    }
    static inArray(obj, value) {
        if (!Array.isArray(value)) {
            throw new Error(`Expect value to be an array: ${value}`);
        }
        obj.oneOf = value.map(val => {
            const subSchema = {
                const: val
            };
            return subSchema;
        });
    }
    static notInArray(obj, value) {
        if (!Array.isArray(value)) {
            throw new Error(`Expect value to be an array: ${value}`);
        }
        obj.not = {
            oneOf: value.map(val => {
                const subSchema = {
                    const: val
                };
                return subSchema;
            })
        };
    }
}
export function isOptional(field) {
    const parsedOption = findRootOption(field);
    if (parsedOption !== null) {
        for (const [dataType, options] of Object.entries(parsedOption)) {
            if (dataType === 'repeated') {
                if (options.items) {
                    for (const [key, val] of Object.entries(options.items)) {
                        if (key === 'ignore_empty' && val) {
                            return true;
                        }
                    }
                }
            }
            else {
                for (const [key, val] of Object.entries(options)) {
                    if (key === 'ignore_empty' && val) {
                        return true;
                    }
                }
            }
        }
    }
    return false;
}
function escapeRegExp(str) {
    return str.replace(/[-[\]/{}()*+?.\\^$|]/g, '\\$&');
}
