"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const asynciterator_1 = require("asynciterator");
/**
 * An iterator that takes elements from a given array or iterator of iterators in a round-robin manner.
 *
 * Based on LDF client's UnionIterator:
 * https://github.com/LinkedDataFragments/Client.js/blob/master/lib/sparql/UnionIterator.js
 */
class RoundRobinUnionIterator extends asynciterator_1.BufferedIterator {
    constructor(sources, options) {
        super(options || { autoStart: false });
        this.sourcedEnded = false;
        this.currentSource = 0;
        this.listenersAttached = false;
        this.sources = [];
        this.sourceIterator = Array.isArray(sources) ? new asynciterator_1.ArrayIterator(sources) : sources;
        this.sourceIterator.on('error', (error) => this.emit('error', error));
    }
    _read(count, done) {
        if (!this.sourcedEnded) {
            // Fill the buffer once the source iterator becomes readable
            if (!this.listenersAttached) {
                this.listenersAttached = true;
                this.sourceIterator.on('readable', () => this._fillBuffer());
            }
            // Poll for new sources
            let source;
            while (source = this.sourceIterator.read()) {
                source.on('error', (error) => this.emit('error', error));
                source.on('readable', () => this._fillBuffer());
                source.on('end', () => this._fillBuffer());
                this.sources.push(source);
            }
            if (this.sourceIterator.ended) {
                this.sourcedEnded = true;
            }
        }
        let item = null;
        let attempts = this.sources.length;
        // Iterate over all sources once
        while (this.sources.length && item === null && attempts--) {
            const source = this.sources[this.currentSource];
            // Read from the current source
            item = source.read();
            // Remove the source if it has ended, otherwise, increment our stored position
            if (source.ended) {
                this.sources.splice(this.currentSource, 1);
            }
            else {
                this.currentSource++;
            }
            this.currentSource = this.sources.length ? this.currentSource % this.sources.length : 0;
        }
        // Push to the buffer if we have an item
        if (item !== null) {
            this._push(item);
        }
        // Otherwise close
        if (!this.sources.length && this.sourcedEnded) {
            this.close();
        }
        done();
    }
}
exports.RoundRobinUnionIterator = RoundRobinUnionIterator;
//# sourceMappingURL=RoundRobinUnionIterator.js.map