import React, { createContext, useCallback, useContext, useEffect, useState } from 'react';
import { bind } from 'bind-event-listener';
import { setGlobalTheme } from '@atlaskit/tokens';
const defaultThemeSettings = {
  dark: 'dark',
  light: 'light',
  spacing: 'spacing'
};
const ColorModeContext = /*#__PURE__*/createContext(undefined);
const SetColorModeContext = /*#__PURE__*/createContext(undefined);
const ThemeContext = /*#__PURE__*/createContext(undefined);
const SetThemeContext = /*#__PURE__*/createContext(undefined);

/**
 * __useColorMode()__
 *
 * Returns the current color mode when inside the app provider.
 */
export function useColorMode() {
  const value = useContext(ColorModeContext);
  if (!value) {
    throw new Error('useColorMode must be used within AppProvider.');
  }
  return value;
}

/**
 * __useSetColorMode()__
 *
 * Returns the color mode setter when inside the app provider.
 */
export function useSetColorMode() {
  const value = useContext(SetColorModeContext);
  if (!value) {
    throw new Error('useSetColorMode must be used within AppProvider.');
  }
  return value;
}

/**
 * __useTheme()__
 *
 * Returns the current theme settings when inside the app provider.
 */
export function useTheme() {
  const value = useContext(ThemeContext);
  if (!value) {
    throw new Error('useTheme must be used within AppProvider.');
  }
  return value;
}

/**
 * __useSetTheme()__
 *
 * Returns the theme setter when inside the app provider.
 */
export function useSetTheme() {
  const value = useContext(SetThemeContext);
  if (!value) {
    throw new Error('useSetTheme must be used within AppProvider.');
  }
  return value;
}
const isMatchMediaAvailable = typeof window !== 'undefined' && 'matchMedia' in window;
const prefersDarkModeMql = isMatchMediaAvailable ? window.matchMedia('(prefers-color-scheme: dark)') : undefined;

// TODO: currently 'auto' color mode will always return 'light' in SSR.
// Additional work required: https://product-fabric.atlassian.net/browse/DSP-9781
function getReconciledColorMode(colorMode) {
  if (colorMode === 'auto') {
    return prefersDarkModeMql !== null && prefersDarkModeMql !== void 0 && prefersDarkModeMql.matches ? 'dark' : 'light';
  }
  return colorMode;
}
/**
 * __Theme provider__
 *
 * Provides global theming configuration.
 *
 * @internal
 */
export function ThemeProvider({
  children,
  defaultColorMode,
  defaultTheme: {
    dark = 'dark',
    light = 'light',
    spacing = 'spacing',
    typography,
    shape
  } = defaultThemeSettings
}) {
  const [chosenColorMode, setChosenColorMode] = useState(defaultColorMode);
  const [reconciledColorMode, setReconciledColorMode] = useState(getReconciledColorMode(defaultColorMode));
  const [theme, setTheme] = useState({
    dark,
    light,
    spacing,
    typography,
    shape
  });
  const setColorMode = useCallback(colorMode => {
    setChosenColorMode(colorMode);
    setReconciledColorMode(getReconciledColorMode(colorMode));
  }, []);
  const setPartialTheme = useCallback(nextTheme => {
    setTheme(theme => ({
      ...theme,
      ...nextTheme
    }));
  }, []);
  useEffect(() => {
    setGlobalTheme({
      ...theme,
      colorMode: reconciledColorMode
    });
  }, [theme, reconciledColorMode]);
  useEffect(() => {
    if (!prefersDarkModeMql) {
      return;
    }
    const unbindListener = bind(prefersDarkModeMql, {
      type: 'change',
      listener: event => {
        if (chosenColorMode === 'auto') {
          setReconciledColorMode(event.matches ? 'dark' : 'light');
        }
      }
    });
    return unbindListener;
  }, [chosenColorMode]);
  return /*#__PURE__*/React.createElement(ColorModeContext.Provider, {
    value: reconciledColorMode
  }, /*#__PURE__*/React.createElement(SetColorModeContext.Provider, {
    value: setColorMode
  }, /*#__PURE__*/React.createElement(ThemeContext.Provider, {
    value: theme
  }, /*#__PURE__*/React.createElement(SetThemeContext.Provider, {
    value: setPartialTheme
  }, children))));
}
export default ThemeProvider;