/** @jsx jsx */
import { forwardRef, memo, useCallback, useRef } from 'react';
import { css, jsx } from '@emotion/react';
import { usePlatformLeafEventHandler } from '@atlaskit/analytics-next/usePlatformLeafEventHandler';
import noop from '@atlaskit/ds-lib/noop';
import { DN90A, N100A } from '@atlaskit/theme/colors';
import { useGlobalTheme } from '@atlaskit/theme/components';
import { layers } from '@atlaskit/theme/constants';
const packageName = "@atlaskit/blanket";
const packageVersion = "12.4.4";
const analyticsAttributes = {
  componentName: 'blanket',
  packageName,
  packageVersion
};
const baseStyles = css({
  position: 'fixed',
  zIndex: layers.blanket(),
  inset: 0,
  overflowY: 'auto',
  pointerEvents: 'initial'
});
const shouldAllowClickThroughStyles = css({
  pointerEvents: 'none'
});
const invisibleStyles = css({
  backgroundColor: 'transparent'
});
const lightBgStyles = css({
  backgroundColor: `var(--ds-blanket, ${N100A})`
});
const darkBgStyles = css({
  backgroundColor: `var(--ds-blanket, ${DN90A})`
});
const backgroundStyle = {
  light: lightBgStyles,
  dark: darkBgStyles
};

/**
 * __Blanket__
 *
 * A Blanket provides the overlay layer for components such as a modal dialog or a tooltip
 *
 * - [Examples](https://atlaskit.atlassian.com/examples/design-system/blanket/basic-usage)
 */
const Blanket = /*#__PURE__*/memo( /*#__PURE__*/forwardRef(function Blanket({
  shouldAllowClickThrough = false,
  isTinted = false,
  onBlanketClicked = noop,
  testId,
  children,
  analyticsContext
}, ref) {
  const {
    mode
  } = useGlobalTheme();
  const mouseDownTarget = useRef(null);
  const onBlanketClickedWithAnalytics = usePlatformLeafEventHandler({
    fn: onBlanketClicked,
    action: 'clicked',
    analyticsData: analyticsContext,
    ...analyticsAttributes
  });
  const blanketClickOutsideChildren = useCallback(e => e.currentTarget === e.target && mouseDownTarget.current === e.target ? onBlanketClickedWithAnalytics(e) : undefined, [onBlanketClickedWithAnalytics]);
  const onClick = shouldAllowClickThrough ? undefined : blanketClickOutsideChildren;
  const onMouseDown = useCallback(e => {
    mouseDownTarget.current = e.target;
  }, []);
  return (
    // TODO: Remove role="presentation", since div's have no semantics anyway (DSP-11587)
    jsx("div", {
      role: "presentation",
      css: [baseStyles, shouldAllowClickThrough && shouldAllowClickThroughStyles, backgroundStyle[mode], !isTinted && invisibleStyles],
      onClick: onClick,
      onMouseDown: onMouseDown,
      "data-testid": testId,
      ref: ref
    }, children)
  );
}));
Blanket.displayName = 'Blanket';
export default Blanket;