/** @jsx jsx */
import { memo, useCallback, useMemo } from 'react';
import { css, jsx } from '@emotion/react';
import { useGlobalTheme } from '@atlaskit/theme/components';
import { useHighlightLines } from './internal/hooks/use-highlight';
import { getCodeBlockStyles, getCodeBlockTheme } from './internal/theme/styles';
import { normalizeLanguage } from './internal/utils/get-normalized-language';
import SyntaxHighlighter from './syntax-highlighter';

/**
 * __Code block__
 *
 * A code block highlights an entire block of code and keeps the formatting.
 *
 * - [Examples](https://atlassian.design/components/code/code-block/examples)
 * - [Code](https://atlassian.design/components/code/code-block/code)
 * - [Usage](https://atlassian.design/components/code/code-block/usage)
 */
var CodeBlock = /*#__PURE__*/memo(function CodeBlock(_ref) {
  var _ref$showLineNumbers = _ref.showLineNumbers,
    showLineNumbers = _ref$showLineNumbers === void 0 ? true : _ref$showLineNumbers,
    _ref$language = _ref.language,
    providedLanguage = _ref$language === void 0 ? 'text' : _ref$language,
    _ref$highlight = _ref.highlight,
    highlight = _ref$highlight === void 0 ? '' : _ref$highlight,
    _ref$highlightedStart = _ref.highlightedStartText,
    highlightedStartText = _ref$highlightedStart === void 0 ? 'Highlight start' : _ref$highlightedStart,
    _ref$highlightedEndTe = _ref.highlightedEndText,
    highlightedEndText = _ref$highlightedEndTe === void 0 ? 'Highlight end' : _ref$highlightedEndTe,
    testId = _ref.testId,
    text = _ref.text,
    _ref$codeBidiWarnings = _ref.codeBidiWarnings,
    codeBidiWarnings = _ref$codeBidiWarnings === void 0 ? true : _ref$codeBidiWarnings,
    codeBidiWarningLabel = _ref.codeBidiWarningLabel,
    _ref$codeBidiWarningT = _ref.codeBidiWarningTooltipEnabled,
    codeBidiWarningTooltipEnabled = _ref$codeBidiWarningT === void 0 ? true : _ref$codeBidiWarningT,
    _ref$shouldWrapLongLi = _ref.shouldWrapLongLines,
    shouldWrapLongLines = _ref$shouldWrapLongLi === void 0 ? false : _ref$shouldWrapLongLi;
  var numLines = (text || '').split('\n').length;
  var globalTheme = useGlobalTheme();
  var theme = useMemo(function () {
    return getCodeBlockTheme(globalTheme, numLines);
  }, [globalTheme, numLines]);
  var getStyles = useMemo(function () {
    return getCodeBlockStyles(theme);
  }, [theme]);
  var styles = useMemo(function () {
    return css(getStyles(highlightedStartText, highlightedEndText, showLineNumbers, shouldWrapLongLines));
  }, [highlightedStartText, highlightedEndText, showLineNumbers, shouldWrapLongLines, getStyles]);
  var _useHighlightLines = useHighlightLines({
      highlight: highlight,
      testId: testId
    }),
    getHighlightStyles = _useHighlightLines.getHighlightStyles,
    highlightedLines = _useHighlightLines.highlightedLines;
  var getLineProps = useCallback(function (line) {
    return getHighlightStyles(line, highlightedLines);
  }, [getHighlightStyles, highlightedLines]);
  var language = useMemo(function () {
    return normalizeLanguage(providedLanguage);
  }, [providedLanguage]);

  // https://product-fabric.atlassian.net/browse/DST-2472
  var languageToUse = text ? language : 'text';
  return jsx(SyntaxHighlighter, {
    "data-code-lang": language,
    "data-ds--code--code-block": "",
    testId: testId,
    language: languageToUse,
    css: styles,
    showLineNumbers: showLineNumbers,
    lineProps: getLineProps
    // shouldCreateParentElementForLines is needed to pass down props to each line.
    // This is necessary for both line highlighting and testId's, as each of
    // these rely on a data attribute being passed down to lines.
    ,
    shouldCreateParentElementForLines: highlight.length > 0 || !!testId,
    shouldWrapLongLines: shouldWrapLongLines,
    codeBidiWarnings: codeBidiWarnings,
    codeBidiWarningLabel: codeBidiWarningLabel,
    codeBidiWarningTooltipEnabled: codeBidiWarningTooltipEnabled,
    text: text
  });
});
CodeBlock.displayName = 'CodeBlock';
export default CodeBlock;