import _defineProperty from "@babel/runtime/helpers/defineProperty";
import _extends from "@babel/runtime/helpers/extends";
/* eslint-disable @repo/internal/react/use-noop */
/** @jsx jsx */
import { Component } from 'react';
import { css, jsx } from '@emotion/react';
// eslint-disable-next-line no-restricted-imports
import { format, isValid, lastDayOfMonth, parseISO } from 'date-fns';
import pick from 'lodash/pick';
import { createAndFireEvent, withAnalyticsContext, withAnalyticsEvents } from '@atlaskit/analytics-next';
import Calendar from '@atlaskit/calendar';
import CalendarIcon from '@atlaskit/icon/glyph/calendar';
import { createLocalizationProvider } from '@atlaskit/locale';
import { getBooleanFF } from '@atlaskit/platform-feature-flags';
import Select, { mergeStyles } from '@atlaskit/select';
import { N0, N50A, N60A } from '@atlaskit/theme/colors';
import { layers } from '@atlaskit/theme/constants';
import { defaultDateFormat, EmptyComponent, padToTwo, placeholderDatetime } from '../internal';
import FixedLayer from '../internal/fixed-layer';
import { convertTokens } from './utils';
const packageName = "@atlaskit/datetime-picker";
const packageVersion = "12.7.7";

/* eslint-disable react/no-unused-prop-types */

function getValidDate(iso) {
  const date = parseISO(iso);
  return isValid(date) ? {
    day: date.getDate(),
    month: date.getMonth() + 1,
    year: date.getFullYear()
  } : {};
}
function getShortISOString(date) {
  return format(date, convertTokens('YYYY-MM-DD'));
}
const menuStyles = css({
  zIndex: layers.dialog(),
  backgroundColor: `var(--ds-surface-overlay, ${N0})`,
  borderRadius: "var(--ds-border-radius, 3px)",
  boxShadow: `var(--ds-shadow-overlay, ${`0 4px 8px -2px ${N50A}, 0 0 1px ${N60A}`})`,
  overflow: 'hidden'
});
const Menu = ({
  selectProps,
  innerProps
}) => jsx(FixedLayer, {
  inputValue: selectProps.inputValue,
  containerRef: selectProps.calendarContainerRef,
  content: jsx("div", _extends({
    css: menuStyles
  }, innerProps), jsx(Calendar, _extends({}, getValidDate(selectProps.calendarValue), getValidDate(selectProps.calendarView), {
    disabled: selectProps.calendarDisabled,
    disabledDateFilter: selectProps.calendarDisabledDateFilter,
    minDate: selectProps.calendarMinDate,
    maxDate: selectProps.calendarMaxDate,
    nextMonthLabel: selectProps.nextMonthLabel,
    onChange: selectProps.onCalendarChange,
    onSelect: selectProps.onCalendarSelect,
    previousMonthLabel: selectProps.previousMonthLabel,
    calendarRef: selectProps.calendarRef,
    selected: [selectProps.calendarValue],
    locale: selectProps.calendarLocale,
    testId: selectProps.testId && `${selectProps.testId}--calendar`,
    weekStartDay: selectProps.calendarWeekStartDay,
    tabIndex: -1
  }))),
  testId: selectProps.testId
});
const datePickerDefaultProps = {
  appearance: 'default',
  autoFocus: false,
  defaultIsOpen: false,
  defaultValue: '',
  disabled: [],
  disabledDateFilter: _ => false,
  hideIcon: false,
  icon: CalendarIcon,
  id: '',
  innerProps: {},
  isDisabled: false,
  isInvalid: false,
  name: '',
  onBlur: event => {},
  onChange: value => {},
  onFocus: event => {},
  selectProps: {},
  spacing: 'default',
  locale: 'en-US'
  // Not including a default prop for value as it will
  // Make the component a controlled component
};

class DatePicker extends Component {
  constructor(props) {
    super(props);
    _defineProperty(this, "calendarRef", null);
    _defineProperty(this, "containerRef", null);
    // All state needs to be accessed via this function so that the state is mapped from props
    // correctly to allow controlled/uncontrolled usage.
    _defineProperty(this, "getSafeState", () => {
      return {
        ...this.state,
        ...pick(this.props, ['value', 'isOpen']),
        ...pick(this.props.selectProps, ['inputValue'])
      };
    });
    _defineProperty(this, "isDateDisabled", date => {
      return this.props.disabled.indexOf(date) > -1;
    });
    _defineProperty(this, "onCalendarChange", ({
      iso
    }) => {
      const [year, month, date] = iso.split('-');
      let newIso = iso;
      const parsedDate = parseInt(date, 10);
      const parsedMonth = parseInt(month, 10);
      const parsedYear = parseInt(year, 10);
      const lastDayInMonth = lastDayOfMonth(new Date(parsedYear, parsedMonth - 1) // This needs to be -1, because the Date constructor expects an index of the given month
      ).getDate();
      if (lastDayInMonth < parsedDate) {
        newIso = `${year}-${padToTwo(parsedMonth)}-${padToTwo(lastDayInMonth)}`;
      } else {
        newIso = `${year}-${padToTwo(parsedMonth)}-${padToTwo(parsedDate)}`;
      }
      this.setState({
        calendarValue: newIso
      });
    });
    _defineProperty(this, "onCalendarSelect", ({
      iso
    }) => {
      this.setState({
        selectInputValue: '',
        isOpen: false,
        calendarValue: iso,
        value: iso
      });
      this.props.onChange(iso);
    });
    _defineProperty(this, "onInputClick", () => {
      if (!this.props.isDisabled && !this.getSafeState().isOpen) {
        this.setState({
          isOpen: true
        });
      }
    });
    _defineProperty(this, "onSelectBlur", event => {
      if (this.getSafeState().clearingFromIcon) {
        // Don't close menu if blurring after the user has clicked clear
        this.setState({
          clearingFromIcon: false
        });
      } else {
        this.setState({
          isOpen: false,
          isFocused: false
        });
      }
      this.props.onBlur(event);
    });
    _defineProperty(this, "onSelectFocus", event => {
      const {
        clearingFromIcon,
        value
      } = this.getSafeState();
      if (clearingFromIcon) {
        // Don't open menu if focussing after the user has clicked clear
        this.setState({
          clearingFromIcon: false
        });
      } else {
        this.setState({
          isOpen: true,
          calendarValue: value,
          isFocused: true
        });
      }
      this.props.onFocus(event);
    });
    _defineProperty(this, "onTextInput", event => {
      const value = event.target.value;
      if (value) {
        const parsed = this.parseDate(value);
        // Only try to set the date if we have month & day
        if (parsed && isValid(parsed)) {
          // We format the parsed date to YYYY-MM-DD here because
          // this is the format expected by the @atlaskit/calendar component
          this.setState({
            calendarValue: getShortISOString(parsed)
          });
        }
      }
      this.setState({
        isOpen: true
      });
    });
    _defineProperty(this, "getSafeCalendarValue", calendarValue => {
      // If `calendarValue` has a year that is greater than 9999, default to
      // today's date
      const yearIsOverLimit = calendarValue.match(/^\d{5,}/);
      if (yearIsOverLimit) {
        return getShortISOString(new Date());
      }
      return calendarValue;
    });
    _defineProperty(this, "onInputKeyDown", event => {
      const {
        value,
        calendarValue
      } = this.getSafeState();
      const keyPressed = event.key.toLowerCase();
      switch (keyPressed) {
        case 'arrowup':
        case 'arrowdown':
          if (this.calendarRef) {
            event.preventDefault();
            const key = keyPressed === 'arrowup' ? 'up' : 'down';
            this.calendarRef.navigate(key);
          }
          this.setState({
            isOpen: true
          });
          break;
        case 'arrowleft':
        case 'arrowright':
          if (this.calendarRef) {
            event.preventDefault();
            const key = keyPressed === 'arrowleft' ? 'left' : 'right';
            this.calendarRef.navigate(key);
          }
          break;
        case 'escape':
        case 'tab':
          this.setState({
            isOpen: false
          });
          break;
        case 'backspace':
        case 'delete':
          {
            const inputCount = getBooleanFF('platform.design-system-team.date-picker-input-a11y-fix_cbbxs') ? 1 : 0;
            if (value && event.target instanceof HTMLInputElement && event.target.value.length <= inputCount) {
              // If being cleared from keyboard, don't change behaviour
              this.setState({
                clearingFromIcon: false,
                value: ''
              });
            }
            break;
          }
        case 'enter':
          if (!this.state.isOpen) {
            return;
          }
          // Prevent form submission when a date is selected
          // using enter. See https://product-fabric.atlassian.net/browse/DSP-2501
          // for more details.
          event.preventDefault();
          if (!this.isDateDisabled(calendarValue)) {
            const {
              value
            } = this.getSafeState();
            // Get a safe `calendarValue` in case the value exceeds the maximum
            // allowed by ISO 8601
            const safeCalendarValue = this.getSafeCalendarValue(calendarValue);
            const valueChanged = safeCalendarValue !== value;
            this.setState({
              selectInputValue: '',
              isOpen: false,
              value: safeCalendarValue,
              calendarValue: safeCalendarValue
            });
            if (valueChanged) {
              this.props.onChange(safeCalendarValue);
            }
          }
          break;
        default:
          break;
      }
    });
    _defineProperty(this, "onClear", () => {
      let changedState = {
        value: '',
        calendarValue: this.props.defaultValue || getShortISOString(new Date())
      };
      if (!this.props.hideIcon) {
        changedState = {
          ...changedState,
          clearingFromIcon: true
        };
      }
      this.setState(changedState);
      this.props.onChange('');
    });
    _defineProperty(this, "onSelectChange", (value, action) => {
      // Used for native clear event in React Select
      // Triggered when clicking ClearIndicator or backspace with no value
      if (action.action === 'clear') {
        this.onClear();
      }
    });
    _defineProperty(this, "refCalendar", ref => {
      this.calendarRef = ref;
    });
    _defineProperty(this, "handleSelectInputChange", (selectInputValue, actionMeta) => {
      const {
        onInputChange
      } = this.props.selectProps;
      if (onInputChange) {
        onInputChange(selectInputValue, actionMeta);
      }
      this.setState({
        selectInputValue
      });
    });
    _defineProperty(this, "getContainerRef", ref => {
      const oldRef = this.containerRef;
      this.containerRef = ref;
      // Cause a re-render if we're getting the container ref for the first time
      // as the layered menu requires it for dimension calculation
      if (oldRef == null && ref != null) {
        this.forceUpdate();
      }
    });
    /**
     * There are two props that can change how the date is parsed.
     * The priority of props used is:
     *   1. parseInputValue
     *   2. locale
     */
    _defineProperty(this, "parseDate", date => {
      const {
        parseInputValue,
        dateFormat
      } = this.props;
      if (parseInputValue) {
        return parseInputValue(date, dateFormat || defaultDateFormat);
      }
      const {
        l10n
      } = this.getSafeState();
      return l10n.parseDate(date);
    });
    /**
     * There are multiple props that can change how the date is formatted.
     * The priority of props used is:
     *   1. formatDisplayLabel
     *   2. dateFormat
     *   3. locale
     */
    _defineProperty(this, "formatDate", value => {
      const {
        formatDisplayLabel,
        dateFormat
      } = this.props;
      const {
        l10n
      } = this.getSafeState();
      if (formatDisplayLabel) {
        return formatDisplayLabel(value, dateFormat || defaultDateFormat);
      }
      const date = parseISO(value);
      return dateFormat ? format(date, convertTokens(dateFormat)) : l10n.formatDate(date);
    });
    _defineProperty(this, "getPlaceholder", () => {
      const {
        placeholder
      } = this.props;
      if (placeholder) {
        return placeholder;
      }
      const {
        l10n
      } = this.getSafeState();
      return l10n.formatDate(placeholderDatetime);
    });
    this.state = {
      isOpen: this.props.defaultIsOpen,
      isFocused: false,
      clearingFromIcon: false,
      selectInputValue: this.props.selectProps.inputValue,
      value: this.props.value || this.props.defaultValue,
      calendarValue: this.props.value || this.props.defaultValue || getShortISOString(new Date()),
      l10n: createLocalizationProvider(this.props.locale)
    };
  }
  UNSAFE_componentWillReceiveProps(nextProps) {
    if (this.props.locale !== nextProps.locale) {
      this.setState({
        l10n: createLocalizationProvider(nextProps.locale)
      });
    }
  }
  render() {
    const {
      appearance,
      autoFocus,
      disabled,
      hideIcon,
      icon,
      id,
      innerProps,
      isDisabled,
      disabledDateFilter,
      maxDate,
      minDate,
      isInvalid,
      name,
      nextMonthLabel,
      previousMonthLabel,
      selectProps,
      spacing,
      locale,
      testId,
      weekStartDay
    } = this.props;
    const {
      value,
      calendarValue,
      isOpen,
      selectInputValue
    } = this.getSafeState();
    let actualSelectInputValue;
    if (getBooleanFF('platform.design-system-team.date-picker-input-a11y-fix_cbbxs')) {
      actualSelectInputValue = selectInputValue || (value ? this.formatDate(value) : undefined);
    } else {
      actualSelectInputValue = selectInputValue;
    }
    const menuIsOpen = isOpen && !isDisabled;
    const showClearIndicator = Boolean((value || selectInputValue) && !hideIcon);
    const dropDownIcon = appearance === 'subtle' || hideIcon || showClearIndicator ? null : icon;
    const selectComponents = {
      DropdownIndicator: dropDownIcon,
      Menu,
      ...(!showClearIndicator && {
        ClearIndicator: EmptyComponent
      })
    };
    const {
      styles: selectStyles = {}
    } = selectProps;
    const disabledStyle = isDisabled ? {
      pointerEvents: 'none',
      color: "var(--ds-icon-disabled, inherit)"
    } : {};
    const calendarProps = {
      calendarContainerRef: this.containerRef,
      calendarRef: this.refCalendar,
      calendarDisabled: disabled,
      calendarDisabledDateFilter: disabledDateFilter,
      calendarMaxDate: maxDate,
      calendarMinDate: minDate,
      calendarValue: value && getShortISOString(parseISO(value)),
      calendarView: calendarValue,
      onCalendarChange: this.onCalendarChange,
      onCalendarSelect: this.onCalendarSelect,
      calendarLocale: locale,
      calendarWeekStartDay: weekStartDay,
      nextMonthLabel,
      previousMonthLabel
    };
    return (
      // TODO: Remove role="presentation", since div's have no semantics anyway
      // (DSP-11587)
      jsx("div", _extends({}, innerProps, {
        role: "presentation",
        onClick: this.onInputClick,
        onInput: this.onTextInput,
        onKeyDown: this.onInputKeyDown,
        ref: this.getContainerRef,
        "data-testid": testId && `${testId}--container`
      }), jsx("input", {
        name: name,
        type: "hidden",
        value: value,
        "data-testid": testId && `${testId}--input`
      }), jsx(Select, _extends({
        appearance: this.props.appearance,
        enableAnimation: false,
        menuIsOpen: menuIsOpen,
        closeMenuOnSelect: true,
        autoFocus: autoFocus,
        instanceId: id,
        isDisabled: isDisabled,
        onBlur: this.onSelectBlur,
        onFocus: this.onSelectFocus,
        inputValue: actualSelectInputValue,
        onInputChange: this.handleSelectInputChange,
        components: selectComponents,
        onChange: this.onSelectChange,
        styles: mergeStyles(selectStyles, {
          control: base => ({
            ...base,
            ...disabledStyle
          }),
          indicatorsContainer: base => ({
            ...base,
            paddingLeft: "var(--ds-space-025, 2px)",
            // ICON_PADDING = 2
            paddingRight: "var(--ds-space-075, 6px)" // 8 - ICON_PADDING = 6
          })
        }),

        placeholder: this.getPlaceholder(),
        value: value && {
          label: this.formatDate(value),
          value
        }
      }, selectProps, calendarProps, {
        isClearable: true,
        spacing: spacing,
        isInvalid: isInvalid,
        testId: testId
      })))
    );
  }
}
_defineProperty(DatePicker, "defaultProps", datePickerDefaultProps);
export { DatePicker as DatePickerWithoutAnalytics };
export default withAnalyticsContext({
  componentName: 'datePicker',
  packageName,
  packageVersion
})(withAnalyticsEvents({
  onChange: createAndFireEvent('atlaskit')({
    action: 'selectedDate',
    actionSubject: 'datePicker',
    attributes: {
      componentName: 'datePicker',
      packageName,
      packageVersion
    }
  })
})(DatePicker));