import _defineProperty from "@babel/runtime/helpers/defineProperty";
import _extends from "@babel/runtime/helpers/extends";
/* eslint-disable @repo/internal/react/use-noop */
import React from 'react';

// eslint-disable-next-line no-restricted-imports
import { format, isValid } from 'date-fns';
import pick from 'lodash/pick';
import { createAndFireEvent, withAnalyticsContext, withAnalyticsEvents } from '@atlaskit/analytics-next';
import { createLocalizationProvider } from '@atlaskit/locale';
import Select, { components, CreatableSelect, mergeStyles } from '@atlaskit/select';
// eslint-disable-next-line @atlaskit/design-system/no-deprecated-imports
import { gridSize } from '@atlaskit/theme/constants';
import { defaultTimeFormat, defaultTimes, EmptyComponent, placeholderDatetime } from '../internal';
import FixedLayer from '../internal/fixed-layer';
import parseTime from '../internal/parse-time';
import { convertTokens } from './utils';
const packageName = "@atlaskit/datetime-picker";
const packageVersion = "12.7.7";

/* eslint-disable react/no-unused-prop-types */

const menuStyles = {
  /* Need to remove default absolute positioning as that causes issues with position fixed */
  position: 'static',
  /* Need to add overflow to the element with max-height, otherwise causes overflow issues in IE11 */
  overflowY: 'auto',
  /* React-Popper has already offset the menu so we need to reset the margin, otherwise the offset value is doubled */
  margin: 0
};
const FixedLayerMenu = ({
  selectProps,
  ...rest
}) => /*#__PURE__*/React.createElement(FixedLayer, {
  inputValue: selectProps.inputValue,
  containerRef: selectProps.fixedLayerRef,
  content: /*#__PURE__*/React.createElement(components.Menu, _extends({}, rest, {
    menuShouldScrollIntoView: false
  })),
  testId: selectProps.testId
});
const timePickerDefaultProps = {
  appearance: 'default',
  autoFocus: false,
  defaultIsOpen: false,
  defaultValue: '',
  hideIcon: false,
  id: '',
  innerProps: {},
  isDisabled: false,
  isInvalid: false,
  name: '',
  onBlur: event => {},
  onChange: value => {},
  onFocus: event => {},
  parseInputValue: (time, timeFormat) => parseTime(time),
  selectProps: {},
  spacing: 'default',
  times: defaultTimes,
  timeIsEditable: false,
  locale: 'en-US'
  // Not including a default prop for value as it will
  // Make the component a controlled component
};

class TimePicker extends React.Component {
  constructor(...args) {
    super(...args);
    _defineProperty(this, "containerRef", null);
    _defineProperty(this, "state", {
      isOpen: this.props.defaultIsOpen,
      clearingFromIcon: false,
      value: this.props.defaultValue,
      isFocused: false,
      l10n: createLocalizationProvider(this.props.locale)
    });
    // All state needs to be accessed via this function so that the state is mapped from props
    // correctly to allow controlled/uncontrolled usage.
    _defineProperty(this, "getSafeState", () => {
      return {
        ...this.state,
        ...pick(this.props, ['value', 'isOpen'])
      };
    });
    _defineProperty(this, "onChange", (newValue, action) => {
      const rawValue = newValue ? newValue.value || newValue : '';
      const value = rawValue.toString();
      let changedState = {
        value
      };
      if (action && action.action === 'clear') {
        changedState = {
          ...changedState,
          clearingFromIcon: true
        };
      }
      this.setState(changedState);
      this.props.onChange(value);
    });
    /**
     * Only allow custom times if timeIsEditable prop is true
     */
    _defineProperty(this, "onCreateOption", inputValue => {
      if (this.props.timeIsEditable) {
        const {
          parseInputValue,
          timeFormat
        } = this.props;
        let sanitizedInput;
        try {
          sanitizedInput = parseInputValue(inputValue, timeFormat || defaultTimeFormat);
        } catch (e) {
          return; // do nothing, the main validation should happen in the form
        }

        const formattedValue = format(sanitizedInput, 'HH:mm') || '';
        this.setState({
          value: formattedValue
        });
        this.props.onChange(formattedValue);
      } else {
        this.onChange(inputValue);
      }
    });
    _defineProperty(this, "onMenuOpen", () => {
      // Don't open menu after the user has clicked clear
      if (this.getSafeState().clearingFromIcon) {
        this.setState({
          clearingFromIcon: false
        });
      } else {
        this.setState({
          isOpen: true
        });
      }
    });
    _defineProperty(this, "onMenuClose", () => {
      // Don't close menu after the user has clicked clear
      if (this.getSafeState().clearingFromIcon) {
        this.setState({
          clearingFromIcon: false
        });
      } else {
        this.setState({
          isOpen: false
        });
      }
    });
    _defineProperty(this, "setContainerRef", ref => {
      const oldRef = this.containerRef;
      this.containerRef = ref;
      // Cause a re-render if we're getting the container ref for the first time
      // as the layered menu requires it for dimension calculation
      if (oldRef == null && ref != null) {
        this.forceUpdate();
      }
    });
    _defineProperty(this, "onBlur", event => {
      this.setState({
        isFocused: false
      });
      this.props.onBlur(event);
    });
    _defineProperty(this, "onFocus", event => {
      this.setState({
        isFocused: true
      });
      this.props.onFocus(event);
    });
    _defineProperty(this, "onSelectKeyDown", event => {
      const {
        key
      } = event;
      const keyPressed = key.toLowerCase();
      if (this.getSafeState().clearingFromIcon && (keyPressed === 'backspace' || keyPressed === 'delete')) {
        // If being cleared from keyboard, don't change behaviour
        this.setState({
          clearingFromIcon: false
        });
      }
    });
    /**
     * There are multiple props that can change how the time is formatted.
     * The priority of props used is:
     *   1. formatDisplayLabel
     *   2. timeFormat
     *   3. locale
     */
    _defineProperty(this, "formatTime", time => {
      const {
        formatDisplayLabel,
        timeFormat
      } = this.props;
      const {
        l10n
      } = this.getSafeState();
      if (formatDisplayLabel) {
        return formatDisplayLabel(time, timeFormat || defaultTimeFormat);
      }
      const date = parseTime(time);
      if (!(date instanceof Date)) {
        return '';
      }
      if (!isValid(date)) {
        return time;
      }
      if (timeFormat) {
        return format(date, convertTokens(timeFormat));
      }
      return l10n.formatTime(date);
    });
    _defineProperty(this, "getPlaceholder", () => {
      const {
        placeholder
      } = this.props;
      if (placeholder) {
        return placeholder;
      }
      const {
        l10n
      } = this.getSafeState();
      return l10n.formatTime(placeholderDatetime);
    });
  }
  UNSAFE_componentWillReceiveProps(nextProps) {
    if (this.props.locale !== nextProps.locale) {
      this.setState({
        l10n: createLocalizationProvider(nextProps.locale)
      });
    }
  }
  getOptions() {
    return this.props.times.map(time => {
      return {
        label: this.formatTime(time),
        value: time
      };
    });
  }
  render() {
    const {
      autoFocus,
      hideIcon,
      id,
      innerProps,
      isDisabled,
      name,
      selectProps,
      spacing,
      testId,
      isInvalid,
      timeIsEditable
    } = this.props;
    const ICON_PADDING = 2;
    const {
      value = '',
      isOpen
    } = this.getSafeState();
    const {
      styles: selectStyles = {},
      ...otherSelectProps
    } = selectProps;
    const SelectComponent = timeIsEditable ? CreatableSelect : Select;
    const labelAndValue = value && {
      label: this.formatTime(value),
      value
    };
    const selectComponents = {
      DropdownIndicator: EmptyComponent,
      Menu: FixedLayerMenu,
      ...(hideIcon && {
        ClearIndicator: EmptyComponent
      })
    };
    const renderIconContainer = Boolean(!hideIcon && value);
    const mergedStyles = mergeStyles(selectStyles, {
      control: base => ({
        ...base
      }),
      menu: base => ({
        ...base,
        ...menuStyles,
        // Fixed positioned elements no longer inherit width from their parent, so we must explicitly set the
        // menu width to the width of our container
        width: this.containerRef ? this.containerRef.getBoundingClientRect().width : 'auto'
      }),
      indicatorsContainer: base => ({
        ...base,
        paddingLeft: renderIconContainer ? ICON_PADDING : 0,
        paddingRight: renderIconContainer ? gridSize() - ICON_PADDING : 0
      })
    });
    return (
      /*#__PURE__*/
      // eslint-disable-next-line jsx-a11y/no-static-element-interactions
      React.createElement("div", _extends({}, innerProps, {
        ref: this.setContainerRef,
        "data-testid": testId && `${testId}--container`,
        onKeyDown: this.onSelectKeyDown
      }), /*#__PURE__*/React.createElement("input", {
        name: name,
        type: "hidden",
        value: value,
        "data-testid": testId && `${testId}--input`
      }), /*#__PURE__*/React.createElement(SelectComponent, _extends({
        appearance: this.props.appearance,
        autoFocus: autoFocus,
        components: selectComponents,
        instanceId: id,
        isClearable: true,
        isDisabled: isDisabled,
        menuIsOpen: isOpen && !isDisabled,
        menuPlacement: "auto",
        openMenuOnFocus: true,
        onBlur: this.onBlur,
        onCreateOption: this.onCreateOption,
        onChange: this.onChange,
        options: this.getOptions(),
        onFocus: this.onFocus,
        onMenuOpen: this.onMenuOpen,
        onMenuClose: this.onMenuClose,
        placeholder: this.getPlaceholder(),
        styles: mergedStyles,
        value: labelAndValue,
        spacing: spacing
        // @ts-ignore caused by prop not part of @atlaskit/select
        ,
        fixedLayerRef: this.containerRef,
        isInvalid: isInvalid,
        testId: testId
      }, otherSelectProps)))
    );
  }
}
_defineProperty(TimePicker, "defaultProps", timePickerDefaultProps);
export { TimePicker as TimePickerWithoutAnalytics };
export default withAnalyticsContext({
  componentName: 'timePicker',
  packageName,
  packageVersion
})(withAnalyticsEvents({
  onChange: createAndFireEvent('atlaskit')({
    action: 'selectedTime',
    actionSubject: 'timePicker',
    attributes: {
      componentName: 'timePicker',
      packageName,
      packageVersion
    }
  })
})(TimePicker));