/** @jsx jsx */

import { useCallback, useRef } from 'react';
import { jsx } from '@emotion/react';
import ArrowLeft from '@atlaskit/icon/glyph/arrow-left';
import { ExitingPersistence, SlideIn, useExitingPersistence } from '@atlaskit/motion';
import { animationTimingFunction, transitionDurationMs } from '../../constants';
import { createExtender } from '../utils';
import ContentOverrides from './content';
import DrawerWrapper from './drawer-wrapper';
import FocusLock from './focus-lock';
import IconButton from './icon-button';
import SidebarOverrides from './sidebar';

// Misc.
// ------------------------------

const defaults = {
  Sidebar: SidebarOverrides,
  Content: ContentOverrides
};

/**
 * This wrapper is used to specify separate durations for enter and exit.
 */
const CustomSlideIn = ({
  children,
  onFinish
}) => {
  const {
    isExiting
  } = useExitingPersistence();

  /**
   * The actual duration should be the same for both enter and exit,
   * but motion halves the passed duration for exit animations,
   * so we double it when exiting.
   */
  const duration = isExiting ? transitionDurationMs * 2 : transitionDurationMs;
  return jsx(SlideIn, {
    animationTimingFunction: animationTimingFunction,
    duration: duration,
    enterFrom: "left",
    exitTo: "left",
    fade: "none",
    onFinish: onFinish
  }, children);
};
const DrawerPrimitive = ({
  children,
  icon: Icon,
  onClose,
  onCloseComplete,
  onOpenComplete,
  overrides,
  testId,
  in: isOpen,
  shouldReturnFocus,
  autoFocusFirstElem,
  isFocusLockEnabled,
  width
}) => {
  const getOverrides = createExtender(defaults, overrides);
  const {
    component: Sidebar,
    ...sideBarOverrides
  } = getOverrides('Sidebar');
  const {
    component: Content,
    ...contentOverrides
  } = getOverrides('Content');

  /**
   * A ref to point to our wrapper, passed to `onCloseComplete` and `onOpenComplete` callbacks.
   */
  const drawerRef = useRef(null);
  const onFinish = useCallback(state => {
    if (state === 'entering') {
      onOpenComplete === null || onOpenComplete === void 0 ? void 0 : onOpenComplete(drawerRef.current);
    } else if (state === 'exiting') {
      onCloseComplete === null || onCloseComplete === void 0 ? void 0 : onCloseComplete(drawerRef.current);
    }
  }, [onCloseComplete, onOpenComplete]);
  return jsx(ExitingPersistence, {
    appear: true
  }, isOpen && jsx(CustomSlideIn, {
    onFinish: onFinish
  }, ({
    className
  }) => jsx(FocusLock, {
    autoFocusFirstElem: autoFocusFirstElem,
    isFocusLockEnabled: isFocusLockEnabled,
    shouldReturnFocus: shouldReturnFocus
  }, jsx(DrawerWrapper, {
    className: className,
    width: width,
    testId: testId,
    drawerRef: drawerRef
  }, jsx(Sidebar, sideBarOverrides, jsx(IconButton, {
    onClick: onClose,
    testId: testId && 'DrawerPrimitiveSidebarCloseButton'
  }, Icon ? jsx(Icon, {
    size: "large"
  }) : jsx(ArrowLeft, {
    label: "Close drawer"
  }))), jsx(Content, contentOverrides, children)))));
};

// eslint-disable-next-line @repo/internal/react/require-jsdoc
export default DrawerPrimitive;