import _extends from "@babel/runtime/helpers/extends";
import React, { useCallback, useEffect, useRef } from 'react';
import { usePlatformLeafEventHandler } from '@atlaskit/analytics-next/usePlatformLeafEventHandler';
import noop from '@atlaskit/ds-lib/noop';
import Flag from './flag';
import { useFlagGroup } from './flag-group';
const packageName = "@atlaskit/flag";
const packageVersion = "15.2.14";
export const AUTO_DISMISS_SECONDS = 8;

/**
 * __Auto dismiss flag__
 *
 * An auto dismiss flag is dismissed automatically after eight seconds.
 *
 * - [Examples](https://atlassian.design/components/flag/auto-dismiss-flag/examples)
 * - [Code](https://atlassian.design/components/flag/auto-dismiss-flag/code)
 */
const AutoDismissFlag = props => {
  const {
    id,
    analyticsContext,
    onDismissed: onDismissedProp = noop
  } = props;
  const autoDismissTimer = useRef(null);
  const {
    onDismissed: onDismissedFromFlagGroup,
    isDismissAllowed
  } = useFlagGroup();
  const onDismissed = useCallback((id, analyticsEvent) => {
    onDismissedProp(id, analyticsEvent);
    onDismissedFromFlagGroup(id, analyticsEvent);
  }, [onDismissedProp, onDismissedFromFlagGroup]);
  const onDismissedAnalytics = usePlatformLeafEventHandler({
    fn: onDismissed,
    action: 'dismissed',
    analyticsData: analyticsContext,
    componentName: 'flag',
    packageName,
    packageVersion
  });
  const isAutoDismissAllowed = isDismissAllowed && onDismissed;
  const dismissFlag = useCallback(() => {
    if (isAutoDismissAllowed) {
      onDismissedAnalytics(id);
    }
  }, [id, onDismissedAnalytics, isAutoDismissAllowed]);
  const stopAutoDismissTimer = useCallback(() => {
    if (autoDismissTimer.current) {
      clearTimeout(autoDismissTimer.current);
      autoDismissTimer.current = null;
    }
  }, []);
  const startAutoDismissTimer = useCallback(() => {
    if (!isAutoDismissAllowed) {
      return;
    }
    stopAutoDismissTimer();
    autoDismissTimer.current = window.setTimeout(dismissFlag, AUTO_DISMISS_SECONDS * 1000);
  }, [dismissFlag, stopAutoDismissTimer, isAutoDismissAllowed]);
  useEffect(() => {
    startAutoDismissTimer();
    return stopAutoDismissTimer;
  }, [startAutoDismissTimer, stopAutoDismissTimer]);
  return /*#__PURE__*/React.createElement(Flag
  // eslint-disable-next-line @repo/internal/react/no-unsafe-spread-props
  , _extends({}, props, {
    onMouseOver: stopAutoDismissTimer,
    onFocus: stopAutoDismissTimer,
    onMouseOut: startAutoDismissTimer,
    onBlur: startAutoDismissTimer
  }));
};
export default AutoDismissFlag;