import _extends from "@babel/runtime/helpers/extends";
/** @jsx jsx */
import { useCallback, useEffect, useState } from 'react';
import { jsx, css } from '@emotion/react';
import { UNSAFE_Text as Text } from '@atlaskit/ds-explorations';
import { Inline, Stack, Box, xcss } from '@atlaskit/primitives';
import { usePlatformLeafEventHandler } from '@atlaskit/analytics-next/usePlatformLeafEventHandler';
import noop from '@atlaskit/ds-lib/noop';
import FocusRing from '@atlaskit/focus-ring';
import { DEFAULT_APPEARANCE } from './constants';
import { flagTextColor, flagBackgroundColor, flagIconColor } from './theme';
import Actions from './flag-actions';
import { useFlagGroup } from './flag-group';
import { Expander, DismissButton } from './internal';
const CSS_VAR_ICON_COLOR = '--flag-icon-color';
const iconWrapperStyles = css({
  display: 'flex',
  alignItems: 'start',
  flexShrink: 0,
  color: `var(${CSS_VAR_ICON_COLOR})`
});
const flagStyles = xcss({
  boxShadow: 'elevation.shadow.overlay',
  borderRadius: 'border.radius.100',
  overflow: 'hidden',
  zIndex: 'flag',
  width: '100%',
  transition: 'background-color 200ms'
});
const flagWrapperStyles = css({
  width: '100%'
});
const analyticsAttributes = {
  componentName: 'flag',
  packageName: "@atlaskit/flag",
  packageVersion: "15.2.14"
};
const transitionStyles = css({
  flexGrow: 1,
  transition: `gap 0.3s`
});

/**
 * __Flag__
 *
 * A flag is used for confirmations, alerts, and acknowledgments that require minimal user interaction,
 * often displayed using a flag group.
 *
 * - [Examples](https://atlassian.design/components/flag/examples)
 * - [Code](https://atlassian.design/components/flag/code)
 * - [Usage](https://atlassian.design/components/flag/usage)
 */
const Flag = props => {
  const {
    actions = [],
    appearance = DEFAULT_APPEARANCE,
    icon,
    title,
    description,
    linkComponent,
    onMouseOver,
    onFocus = noop,
    onMouseOut,
    onBlur = noop,
    onDismissed: onDismissedProp = noop,
    testId,
    id,
    analyticsContext
  } = props;
  const {
    onDismissed: onDismissedFromFlagGroup,
    isDismissAllowed
  } = useFlagGroup();
  const onDismissed = useCallback((id, analyticsEvent) => {
    onDismissedProp(id, analyticsEvent);
    onDismissedFromFlagGroup(id, analyticsEvent);
  }, [onDismissedProp, onDismissedFromFlagGroup]);
  const [isExpanded, setIsExpanded] = useState(false);
  const onDismissedAnalytics = usePlatformLeafEventHandler({
    fn: onDismissed,
    action: 'dismissed',
    analyticsData: analyticsContext,
    ...analyticsAttributes
  });
  const isBold = appearance !== DEFAULT_APPEARANCE;
  const toggleExpand = useCallback(() => {
    setIsExpanded(previous => !previous);
  }, []);
  const buttonActionCallback = useCallback(() => {
    if (isDismissAllowed) {
      onDismissedAnalytics(id);
    }
  }, [onDismissedAnalytics, id, isDismissAllowed]);
  useEffect(() => {
    // If buttons are removed as a prop, update isExpanded to be false
    if (isBold && isExpanded && !description && !actions.length) {
      setIsExpanded(false);
    }
  }, [actions.length, description, isBold, isExpanded]);
  const onFocusAnalytics = usePlatformLeafEventHandler({
    fn: onFocus,
    action: 'focused',
    analyticsData: analyticsContext,
    ...analyticsAttributes
  });
  const onBlurAnalytics = usePlatformLeafEventHandler({
    fn: onBlur,
    action: 'blurred',
    analyticsData: analyticsContext,
    ...analyticsAttributes
  });
  const autoDismissProps = {
    onMouseOver,
    onFocus: onFocusAnalytics,
    onMouseOut,
    onBlur: onBlurAnalytics
  };
  const textColor = flagTextColor[appearance];
  const iconColor = flagIconColor[appearance];
  const isDismissable = isBold || isDismissAllowed;
  const shouldRenderGap = !isBold && (description || actions.length) || isExpanded;
  return jsx(FocusRing, null, jsx("div", _extends({
    // eslint-disable-next-line jsx-a11y/no-noninteractive-tabindex
    tabIndex: 0,
    role: "alert",
    css: flagWrapperStyles,
    "data-testid": testId
  }, autoDismissProps), jsx(Box, {
    backgroundColor: flagBackgroundColor[appearance],
    padding: "space.200",
    xcss: flagStyles
  }, jsx(Inline, {
    alignBlock: "stretch",
    space: "space.200"
  }, jsx("div", {
    css: iconWrapperStyles,
    style: {
      [CSS_VAR_ICON_COLOR]: iconColor
    }
  }, icon), jsx("span", {
    css: transitionStyles
  }, jsx(Stack, {
    space: shouldRenderGap ? 'space.100' : 'space.0' // Gap exists even when not expanded due to Expander internals always being in the DOM
  }, jsx(Inline, {
    alignBlock: "stretch",
    space: "space.100",
    spread: "space-between"
  }, jsx(Box, {
    paddingBlockStart: "space.025"
  }, jsx(Text, {
    color: textColor,
    fontWeight: "semibold",
    UNSAFE_style: {
      overflowWrap: 'anywhere' // For cases where a single word is longer than the container (e.g. filenames)
    }
  }, title)), isDismissable ? !(isBold && !description && !actions.length) && jsx(DismissButton, {
    testId: testId,
    appearance: appearance,
    isBold: isBold,
    isExpanded: isExpanded,
    onClick: isBold ? toggleExpand : buttonActionCallback
  }) : null), jsx(Expander, {
    isExpanded: !isBold || isExpanded,
    testId: testId
  }, description && jsx(Text, {
    as: "div",
    color: textColor,
    UNSAFE_style: {
      maxHeight: 100,
      // height is defined as 5 lines maximum by design
      overflow: 'auto',
      overflowWrap: 'anywhere' // For cases where a single word is longer than the container (e.g. filenames)
    },

    testId: testId && `${testId}-description`
  }, description), jsx(Actions, {
    actions: actions,
    appearance: appearance,
    linkComponent: linkComponent,
    testId: testId
  }))))))));
};
export default Flag;