import _typeof from "@babel/runtime/helpers/typeof";
/** @jsx jsx */

import { Children, createContext, useContext, useMemo } from 'react';
import { css, jsx } from '@emotion/react';
import { easeIn, ExitingPersistence, SlideIn } from '@atlaskit/motion';
import VisuallyHidden from '@atlaskit/visually-hidden';
import noop from '@atlaskit/ds-lib/noop';
import Portal from '@atlaskit/portal';
// eslint-disable-next-line @atlaskit/design-system/no-deprecated-imports
import { gridSize as getGridSize, layers } from '@atlaskit/theme/constants';
var gridSize = getGridSize();
export var flagWidth = gridSize * 50;
export var flagAnimationTime = 400;
var defaultFlagGroupContext = {
  onDismissed: noop,
  isDismissAllowed: false
};

// eslint-disable-next-line @repo/internal/react/require-jsdoc
export var FlagGroupContext = /*#__PURE__*/createContext(defaultFlagGroupContext);

// eslint-disable-next-line @repo/internal/react/require-jsdoc
export function useFlagGroup() {
  return useContext(FlagGroupContext);
}

// transition: none is set on first-of-type to prevent a bug in Firefox
// that causes a broken transition
var baseStyles = css({
  width: flagWidth,
  position: 'absolute',
  bottom: 0,
  transition: "transform ".concat(flagAnimationTime, "ms ease-in-out"),
  '@media (max-width: 560px)': {
    width: '100vw'
  },
  ':first-of-type': {
    transform: "translate(0,0)",
    transition: 'none'
  },
  ':nth-of-type(n + 2)': {
    animationDuration: '0ms',
    transform: "translateX(0) translateY(100%) translateY(".concat(2 * gridSize, "px)")
  },
  ':nth-of-type(1)': {
    zIndex: 5
  },
  ':nth-of-type(2)': {
    zIndex: 4
  },
  '&:nth-of-type(n + 4)': {
    visibility: 'hidden'
  }
});

// Transform needed to push up while 1st flag is leaving
// Exiting time should match the exiting time of motion so is halved
var dismissAllowedStyles = css({
  // eslint-disable-next-line @repo/internal/styles/no-nested-styles
  '&& + *': {
    transform: "translate(0, 0)",
    transitionDuration: "".concat(flagAnimationTime / 2, "ms")
  }
});
var flagGroupContainerStyles = css({
  position: 'fixed',
  zIndex: 'flag',
  bottom: "var(--ds-space-600, 48px)",
  left: "var(--ds-space-1000, 80px)",
  '@media (max-width: 560px)': {
    bottom: 0,
    left: 0
  }
});

/**
 * __Flag group__
 *
 * A flag group is used to group a set of related flags, with entry and exit animations.
 *
 * - [Examples](https://atlassian.design/components/flag/flag-group/examples)
 * - [Code](https://atlassian.design/components/flag/flag-group/code)
 */
var FlagGroup = function FlagGroup(props) {
  var id = props.id,
    _props$label = props.label,
    label = _props$label === void 0 ? 'Flag notifications' : _props$label,
    _props$labelTag = props.labelTag,
    LabelTag = _props$labelTag === void 0 ? 'h2' : _props$labelTag,
    children = props.children,
    _props$onDismissed = props.onDismissed,
    onDismissed = _props$onDismissed === void 0 ? noop : _props$onDismissed;
  var hasFlags = Array.isArray(children) ? children.length > 0 : Boolean(children);
  var dismissFlagContext = useMemo(function () {
    return {
      onDismissed: onDismissed,
      isDismissAllowed: true
    };
  }, [onDismissed]);
  var renderChildren = function renderChildren() {
    return children && _typeof(children) === 'object' ? Children.map(children, function (flag, index) {
      var isDismissAllowed = index === 0;
      return jsx(SlideIn, {
        enterFrom: "left",
        fade: "inout",
        duration: flagAnimationTime,
        animationTimingFunction: function animationTimingFunction() {
          return easeIn;
        }
      }, function (_ref) {
        var className = _ref.className,
          ref = _ref.ref;
        return jsx("div", {
          css: [baseStyles, isDismissAllowed && dismissAllowedStyles],
          className: className,
          ref: ref
        }, jsx(FlagGroupContext.Provider, {
          value:
          // Only the first flag should be able to be dismissed.
          isDismissAllowed ? dismissFlagContext : defaultFlagGroupContext
        }, flag));
      });
    }) : false;
  };
  return jsx(Portal, {
    zIndex: layers.flag()
  }, jsx("div", {
    id: id,
    css: flagGroupContainerStyles
  }, hasFlags ? jsx(VisuallyHidden, null, jsx(LabelTag, null, label)) : null, jsx(ExitingPersistence, {
    appear: false
  }, renderChildren())));
};
export default FlagGroup;