/** @jsx jsx */

import { Children, createContext, useContext, useMemo } from 'react';
import { css, jsx } from '@emotion/react';
import { easeIn, ExitingPersistence, SlideIn } from '@atlaskit/motion';
import VisuallyHidden from '@atlaskit/visually-hidden';
import noop from '@atlaskit/ds-lib/noop';
import Portal from '@atlaskit/portal';
// eslint-disable-next-line @atlaskit/design-system/no-deprecated-imports
import { gridSize as getGridSize, layers } from '@atlaskit/theme/constants';
const gridSize = getGridSize();
export const flagWidth = gridSize * 50;
export const flagAnimationTime = 400;
const defaultFlagGroupContext = {
  onDismissed: noop,
  isDismissAllowed: false
};

// eslint-disable-next-line @repo/internal/react/require-jsdoc
export const FlagGroupContext = /*#__PURE__*/createContext(defaultFlagGroupContext);

// eslint-disable-next-line @repo/internal/react/require-jsdoc
export function useFlagGroup() {
  return useContext(FlagGroupContext);
}

// transition: none is set on first-of-type to prevent a bug in Firefox
// that causes a broken transition
const baseStyles = css({
  width: flagWidth,
  position: 'absolute',
  bottom: 0,
  transition: `transform ${flagAnimationTime}ms ease-in-out`,
  '@media (max-width: 560px)': {
    width: '100vw'
  },
  ':first-of-type': {
    transform: `translate(0,0)`,
    transition: 'none'
  },
  ':nth-of-type(n + 2)': {
    animationDuration: '0ms',
    transform: `translateX(0) translateY(100%) translateY(${2 * gridSize}px)`
  },
  ':nth-of-type(1)': {
    zIndex: 5
  },
  ':nth-of-type(2)': {
    zIndex: 4
  },
  '&:nth-of-type(n + 4)': {
    visibility: 'hidden'
  }
});

// Transform needed to push up while 1st flag is leaving
// Exiting time should match the exiting time of motion so is halved
const dismissAllowedStyles = css({
  // eslint-disable-next-line @repo/internal/styles/no-nested-styles
  '&& + *': {
    transform: `translate(0, 0)`,
    transitionDuration: `${flagAnimationTime / 2}ms`
  }
});
const flagGroupContainerStyles = css({
  position: 'fixed',
  zIndex: 'flag',
  bottom: "var(--ds-space-600, 48px)",
  left: "var(--ds-space-1000, 80px)",
  '@media (max-width: 560px)': {
    bottom: 0,
    left: 0
  }
});

/**
 * __Flag group__
 *
 * A flag group is used to group a set of related flags, with entry and exit animations.
 *
 * - [Examples](https://atlassian.design/components/flag/flag-group/examples)
 * - [Code](https://atlassian.design/components/flag/flag-group/code)
 */
const FlagGroup = props => {
  const {
    id,
    label = 'Flag notifications',
    labelTag: LabelTag = 'h2',
    children,
    onDismissed = noop
  } = props;
  const hasFlags = Array.isArray(children) ? children.length > 0 : Boolean(children);
  const dismissFlagContext = useMemo(() => ({
    onDismissed: onDismissed,
    isDismissAllowed: true
  }), [onDismissed]);
  const renderChildren = () => {
    return children && typeof children === 'object' ? Children.map(children, (flag, index) => {
      const isDismissAllowed = index === 0;
      return jsx(SlideIn, {
        enterFrom: "left",
        fade: "inout",
        duration: flagAnimationTime,
        animationTimingFunction: () => easeIn
      }, ({
        className,
        ref
      }) => jsx("div", {
        css: [baseStyles, isDismissAllowed && dismissAllowedStyles],
        className: className,
        ref: ref
      }, jsx(FlagGroupContext.Provider, {
        value:
        // Only the first flag should be able to be dismissed.
        isDismissAllowed ? dismissFlagContext : defaultFlagGroupContext
      }, flag)));
    }) : false;
  };
  return jsx(Portal, {
    zIndex: layers.flag()
  }, jsx("div", {
    id: id,
    css: flagGroupContainerStyles
  }, hasFlags ? jsx(VisuallyHidden, null, jsx(LabelTag, null, label)) : null, jsx(ExitingPersistence, {
    appear: false
  }, renderChildren())));
};
export default FlagGroup;