import _extends from "@babel/runtime/helpers/extends";
import React, { useCallback, useContext, useMemo, useState } from 'react';
import AutoDismissFlag from './auto-dismiss-flag';
import Flag from './flag';
import FlagGroup from './flag-group';
const FlagContext = /*#__PURE__*/React.createContext(null);

/**
 * useFlags is used to access the `showFlags` function which can be used to programatically display flags.
 * - [Examples](https://atlassian.design/components/flag/flags-provider/examples#using-showflags)
 */
export function useFlags() {
  const api = useContext(FlagContext);
  if (api == null) {
    throw new Error('Unable to find FlagProviderContext');
  }
  return api;
}
const getUniqueId = (() => {
  let count = 0;
  return () => `flag-provider-unique-id:${count++}`;
})();
export function FlagsProvider({
  children
}) {
  const [flags, setFlags] = useState([]);
  const removeFlag = useCallback(id => {
    setFlags(current => {
      return current.slice(0).filter(flag => flag.id !== id);
    });
  }, []);
  const api = useMemo(() => ({
    showFlag: function show(value) {
      const flag = {
        ...value,
        id: value.id || getUniqueId()
      };
      setFlags(current => {
        const index = current.findIndex(value => value.id === flag.id);

        // If flag is not found add it
        if (index === -1) {
          return [flag, ...current];
        }

        // If flag already exists with the same id, then replace it
        const shallow = [...current];
        shallow[index] = flag;
        return shallow;
      });
      return function dismiss() {
        removeFlag(flag.id);
      };
    }
  }), [removeFlag]);
  return /*#__PURE__*/React.createElement(React.Fragment, null, /*#__PURE__*/React.createElement(FlagContext.Provider, {
    value: api
  }, children), /*#__PURE__*/React.createElement(FlagGroup, {
    onDismissed: removeFlag
  }, flags.map(flag => {
    const {
      isAutoDismiss,
      ...restProps
    } = flag;
    const FlagType = isAutoDismiss ? AutoDismissFlag : Flag;
    return /*#__PURE__*/React.createElement(FlagType, _extends({}, restProps, {
      key: flag.id
    }));
  })));
}
export const withFlagsProvider = fn => /*#__PURE__*/React.createElement(FlagsProvider, null, fn());