import { css } from '@emotion/react';
import { UNSAFE_BREAKPOINTS_ORDERED_LIST } from './constants';
import { media, UNSAFE_media } from './media-helper';
/**
 * Build a map of breakpoints to css with media queries and nested styles.
 *
 * @internal Not intended to be used outside of DST at this stage.
 * @experimental Not intended to be used outside of DST at this stage.
 *
 * @example
 * A map to build optional `display:none` for consumption on a div.
 * ```ts
 * const hideMediaQueries = buildAboveMediaQueryCSS({ display: 'none' });
 *
 * const Component = ({ hideAtBreakpoints: ('xs' | 'sm')[], children: ReactNode }) => {
 *   return <div css={hideAtBreakpoints.map(b => hideMediaQueries[b])}>{children}</div>;
 * }
 * ```
 *
 * This roughly builds a map that will look roughly like this (if done manually):
 * ```ts
 * {
 *   xxs: css({ '@media all': { display: 'none' } }),
 *   xs: css({ '@media (min-width: 30rem)': { display: 'none' } }),
 *   sm: css({ '@media (min-width: 48rem)': { display: 'none' } }),
 * }
 * ```
 */
export const UNSAFE_buildAboveMediaQueryCSS = input => {
  return UNSAFE_BREAKPOINTS_ORDERED_LIST.reduce((acc, breakpoint) => ({
    ...acc,
    [breakpoint]: css({
      [media.above[breakpoint]]: typeof input === 'function' ? input(breakpoint) : input
    })
  }), {});
};

/**
 * Build a map of breakpoints to css with media queries and nested styles.
 *
 * @internal Not intended to be used outside of DST at this stage.
 * @experimental Not intended to be used outside of DST at this stage.
 *
 * @example
 * A map to build optional `display:none` for consumption on a div.
 * ```ts
 * const hideMediaQueries = buildBelowMediaQueryCSS({ display: 'none' });
 *
 * const Component = ({ hideAtBreakpoints: ('xs' | 'sm')[], children: ReactNode }) => {
 *   return <div css={hideAtBreakpoints.map(b => hideMediaQueries[b])}>{children}</div>;
 * }
 * ```
 *
 * This roughly builds a map that will look roughly like this (if done manually):
 * ```ts
 * {
 *   xs: css({ '@media not all and (min-width: 30rem)': { display: 'none' } }),
 *   sm: css({ '@media not all and (min-width: 48rem)': { display: 'none' } }),
 * }
 * ```
 */
export const UNSAFE_buildBelowMediaQueryCSS = input => {
  return UNSAFE_BREAKPOINTS_ORDERED_LIST.reduce((acc, breakpoint) => {
    if (breakpoint === 'xxs') {
      return acc;
    }
    return {
      ...acc,
      [breakpoint]: css({
        // eslint-disable-next-line @atlaskit/design-system/no-nested-styles
        [UNSAFE_media.below[breakpoint]]: typeof input === 'function' ? input(breakpoint) : input
      })
    };
  }, {});
};