import _extends from "@babel/runtime/helpers/extends";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
import React, { PureComponent } from 'react';
import { createPortal } from 'react-dom';
import FocusLock from 'react-focus-lock';
import Select, { mergeStyles } from 'react-select';
import { uid } from 'react-uid';
import { Manager, Reference, Popper } from 'react-popper';
import NodeResolver from 'react-node-resolver';
import shallowEqualObjects from 'shallow-equal/objects';
import { N80 } from '@atlaskit/theme/colors';
import { getBooleanFF } from '@atlaskit/platform-feature-flags';
import { MenuDialog, DummyControl, defaultComponents } from './components';
import baseStyles from '../styles';
import { bind } from 'bind-event-listener';
import memoizeOne from 'memoize-one';
/** Are we rendering on the client or server? */
const canUseDOM = () => Boolean(typeof window !== 'undefined' && window.document && window.document.createElement);

// ==============================
// Types
// ==============================
// ==============================
// Class
// ==============================
const modifiers = [{
  name: 'offset',
  options: {
    offset: [0, 8]
  }
}, {
  name: 'preventOverflow',
  enabled: true,
  options: {
    padding: 5,
    boundary: 'clippingParents',
    altAxis: true,
    altBoundary: true
  }
}];
const defaultPopperProps = {
  modifiers,
  placement: 'bottom-start'
};
const isEmpty = obj => Object.keys(obj).length === 0;
export default class PopupSelect extends PureComponent {
  constructor(...args) {
    var _this$defaultOpenStat, _this$defaultOpenStat2;
    super(...args);
    _defineProperty(this, "menuRef", null);
    _defineProperty(this, "selectRef", null);
    _defineProperty(this, "targetRef", null);
    _defineProperty(this, "unbindWindowClick", null);
    _defineProperty(this, "unbindWindowKeydown", null);
    _defineProperty(this, "defaultStyles", mergeStyles(baseStyles(this.props.validationState || (this.props.isInvalid ? 'error' : 'default'), this.props.spacing === 'compact', 'default'), {
      groupHeading: provided => ({
        ...provided,
        color: `var(--ds-text-subtlest, ${N80})`
      })
    }));
    _defineProperty(this, "isOpenControlled", this.props.isOpen !== undefined);
    _defineProperty(this, "defaultOpenState", this.isOpenControlled ? this.props.isOpen : this.props.defaultIsOpen);
    _defineProperty(this, "state", getBooleanFF('platform.design-system-team.popup-select-render-perf_i0s6m') ? {
      isOpen: (_this$defaultOpenStat = this.defaultOpenState) !== null && _this$defaultOpenStat !== void 0 ? _this$defaultOpenStat : false,
      mergedComponents: defaultComponents,
      mergedPopperProps: defaultPopperProps
    } : {
      focusLockEnabled: false,
      isOpen: (_this$defaultOpenStat2 = this.defaultOpenState) !== null && _this$defaultOpenStat2 !== void 0 ? _this$defaultOpenStat2 : false,
      mergedComponents: defaultComponents,
      mergedPopperProps: defaultPopperProps
    });
    _defineProperty(this, "popperWrapperId", `${uid({
      options: this.props.options
    })}-popup-select`);
    // Event Handlers
    // ==============================
    _defineProperty(this, "handleKeyDown", event => {
      switch (event.key) {
        case 'Escape':
        case 'Esc':
          this.close();
          break;
        default:
      }
      if (this.props.onKeyDown) {
        /* @ts-ignore - updating type of event React.KeyboardEvent effects the unbindWindowsKeyDown listener. Check if this can be fixed once the component gets refactor to functional */
        this.props.onKeyDown(event);
      }
    });
    _defineProperty(this, "handleClick", ({
      target
    }) => {
      const {
        isOpen
      } = this.state;
      // appease flow
      if (!(target instanceof Element)) {
        return;
      }

      // NOTE: Why not use the <Blanket /> component to close?
      // We don't want to interupt the user's flow. Taking this approach allows
      // user to click "through" to other elements and close the popout.
      if (isOpen && this.menuRef && !this.menuRef.contains(target)) {
        this.close();
      }

      // open on target click -- we can't trust consumers to spread the onClick
      // property to the target
      if (!isOpen && this.targetRef && this.targetRef.contains(target)) {
        this.open();
      }
    });
    _defineProperty(this, "handleSelectChange", (value, actionMeta) => {
      const {
        closeMenuOnSelect,
        onChange
      } = this.props;
      if (closeMenuOnSelect && actionMeta.action !== 'clear') {
        this.close();
      }
      if (onChange) {
        onChange(value, actionMeta);
      }
    });
    _defineProperty(this, "handleFirstPopperUpdate", () => {
      // When the popup opens it's focused into. Since the popup is inside a portal, it's position is
      // initially set to 0,0 - this causes the window scroll position to jump to the top. To prevent
      // this we defer enabling the focus-lock until after Popper has positioned the popup the first time.
      this.setState({
        focusLockEnabled: true
      });
    });
    // Internal Lifecycle
    // ==============================
    /**
     * Opens the popup
     *
     * @param options.controlOverride  - Force the popup to open when it's open state is being controlled
     */
    _defineProperty(this, "open", options => {
      const {
        onOpen
      } = this.props;
      if (!(options !== null && options !== void 0 && options.controlOverride) && this.isOpenControlled) {
        // Prevent popup opening if it's open state is already being controlled
        return;
      }
      if (onOpen) {
        onOpen();
      }
      this.setState({
        isOpen: true
      });
      if (this.selectRef) {
        this.selectRef.openMenu('first');
      }
      if (typeof window === 'undefined') {
        return;
      }
      this.unbindWindowKeydown = bind(window, {
        type: 'keydown',
        listener: this.handleKeyDown,
        options: {
          capture: true
        }
      });
    });
    /**
     * Closes the popup
     *
     * @param options.controlOverride  - Force the popup to close when it's open state is being controlled
     */
    _defineProperty(this, "close", options => {
      var _this$unbindWindowKey;
      const {
        onClose
      } = this.props;
      if (!(options !== null && options !== void 0 && options.controlOverride) && this.isOpenControlled) {
        // Prevent popup closing if it's open state is already being controlled
        return;
      }
      if (onClose) {
        onClose();
      }
      this.setState({
        isOpen: false
      });
      if (getBooleanFF('platform.design-system-team.popup-select-render-perf_i0s6m')) {
        // Do nothing… (the pff eslint just doesn't like `!getBooleanFF(…)`)
      } else {
        this.setState({
          focusLockEnabled: false
        });
      }
      if (this.targetRef != null) {
        this.targetRef.focus();
      }
      if (typeof window === 'undefined') {
        return;
      }
      (_this$unbindWindowKey = this.unbindWindowKeydown) === null || _this$unbindWindowKey === void 0 ? void 0 : _this$unbindWindowKey.call(this);
      this.unbindWindowKeydown = null;
    });
    // Refs
    // ==============================
    _defineProperty(this, "resolveTargetRef", popperRef => ref => {
      // avoid thrashing fn calls
      if (!this.targetRef && popperRef && ref) {
        this.targetRef = ref;
        if (typeof popperRef === 'function') {
          popperRef(ref);
        } else {
          popperRef.current = ref;
        }
      }
    });
    _defineProperty(this, "resolveMenuRef", popperRef => ref => {
      this.menuRef = ref;
      if (typeof popperRef === 'function') {
        popperRef(ref);
      } else {
        popperRef.current = ref;
      }
    });
    _defineProperty(this, "getSelectRef", ref => {
      this.selectRef = ref;
    });
    // Utils
    // ==============================
    // Get a memoized merge of the default styles and the prop's in styles
    _defineProperty(this, "getSelectStyles", memoizeOne((defaultStyles, propStyles) => {
      return mergeStyles(defaultStyles, propStyles || {});
    }));
    // Get a memoized override of our `<Select components={…}>` overrides.
    _defineProperty(this, "getSelectComponents", memoizeOne((mergedComponents, showSearchControl) => {
      if (!showSearchControl) {
        // When we have no search control, we use a dummy override to hide it visually.
        return {
          ...mergedComponents,
          Control: DummyControl
        };
      }
      return mergedComponents;
    }));
    // account for groups when counting options
    // this may need to be recursive, right now it just counts one level
    _defineProperty(this, "getItemCount", () => {
      const {
        options
      } = this.props;
      let count = 0;
      options.forEach(groupOrOption => {
        if (groupOrOption.options) {
          groupOrOption.options.forEach(() => count++);
        } else {
          count++;
        }
      });
      return count;
    });
    _defineProperty(this, "getMaxHeight", () => {
      const {
        maxMenuHeight
      } = this.props;
      if (!this.selectRef) {
        return maxMenuHeight;
      }

      // subtract the control height to maintain consistency
      const showSearchControl = this.showSearchControl();
      const {
        controlRef
      } = this.selectRef;
      const offsetHeight = showSearchControl && controlRef ? controlRef.offsetHeight : 0;
      const maxHeight = maxMenuHeight - offsetHeight;
      return maxHeight;
    });
    // if the threshold is exceeded, AND isSearchable is true, then display the search control
    _defineProperty(this, "showSearchControl", () => {
      const {
        searchThreshold,
        isSearchable
      } = this.props;
      return isSearchable && this.getItemCount() > searchThreshold;
    });
    // Renderers
    // ==============================
    _defineProperty(this, "renderSelect", () => {
      const {
        footer,
        maxMenuWidth,
        minMenuWidth,
        target,
        ...props
      } = this.props;
      // TODO: If `platform.design-system-team.popup-select-render-perf_i0s6m` is kept, `focusLockEnabled` should be fully removed as we're preferring `isReferenceHidden`
      const {
        focusLockEnabled,
        isOpen,
        mergedComponents,
        mergedPopperProps
      } = this.state;
      const showSearchControl = this.showSearchControl();
      const portalDestination = canUseDOM() ? document.body : null;
      if (!portalDestination || !isOpen) {
        return null;
      }

      // Memoized merge of defaultStyles and props.styles
      const selectStyles = getBooleanFF('platform.design-system-team.popup-select-render-perf_i0s6m') ? this.getSelectStyles(this.defaultStyles, props.styles) : mergeStyles(this.defaultStyles, props.styles || {});

      // Memoized variance of the default select components
      const selectComponents = getBooleanFF('platform.design-system-team.popup-select-render-perf_i0s6m') ? this.getSelectComponents(mergedComponents, showSearchControl) : {
        ...mergedComponents,
        Control: showSearchControl ? mergedComponents.Control : DummyControl
      };
      const popper = /*#__PURE__*/React.createElement(Popper, _extends({}, mergedPopperProps, getBooleanFF('platform.design-system-team.popup-select-render-perf_i0s6m') ? undefined : {
        onFirstUpdate: state => {
          var _mergedPopperProps$on;
          this.handleFirstPopperUpdate();
          (_mergedPopperProps$on = mergedPopperProps.onFirstUpdate) === null || _mergedPopperProps$on === void 0 ? void 0 : _mergedPopperProps$on.call(mergedPopperProps, state);
        }
      }), ({
        placement,
        ref,
        style,
        isReferenceHidden
      }) => {
        /**
         * The reference is not available yet, so the Popper and Portal is either being rendered at `0,0` (scrolled to the top)
         * or not at all.  There's no reason to render the Select or lock scrolling at the top of the page yet.
         */
        const readyToRenderSelect = getBooleanFF('platform.design-system-team.popup-select-render-perf_i0s6m') ? isReferenceHidden !== null : true;
        return /*#__PURE__*/React.createElement(NodeResolver, {
          innerRef: this.resolveMenuRef(ref)
        }, /*#__PURE__*/React.createElement(MenuDialog, {
          style: style,
          "data-placement": placement,
          minWidth: minMenuWidth,
          maxWidth: maxMenuWidth,
          id: this.popperWrapperId
        }, /*#__PURE__*/React.createElement(FocusLock
        /*
         * NOTE: We intentionally want the FocusLock to be disabled until the refs are populated in Popper.
         * Until then, the portal the Popper creates is at `0,0`, meaning the FocusLock forces the page to scroll to `0,0`.
         * We do not want the user to scroll to the top of the page when they open their PopupSelect, so we disable it.
         *
         *  WARNING: This causes additional renders, eg. ±5ms in our example, but unless
         * FocusLock has a better way to avoid scrolling, this is necessary.
         */, {
          disabled: getBooleanFF('platform.design-system-team.popup-select-render-perf_i0s6m') ? !readyToRenderSelect : !focusLockEnabled,
          returnFocus: true
        }, readyToRenderSelect && /*#__PURE__*/React.createElement(Select, _extends({
          backspaceRemovesValue: false,
          controlShouldRenderValue: false,
          isClearable: false,
          tabSelectsValue: false,
          menuIsOpen: true,
          ref: this.getSelectRef
        }, props, {
          isSearchable: showSearchControl,
          styles: selectStyles,
          maxMenuHeight: this.getMaxHeight(),
          components: selectComponents,
          onChange: this.handleSelectChange
        })), footer)));
      });
      return mergedPopperProps.strategy === 'fixed' ? popper : /*#__PURE__*/createPortal(popper, portalDestination);
    });
  }
  static getDerivedStateFromProps(props, state) {
    const newState = {};

    // Merge consumer and default popper props
    const mergedPopperProps = {
      ...defaultPopperProps,
      ...props.popperProps
    };
    if (!shallowEqualObjects(mergedPopperProps, state.mergedPopperProps)) {
      newState.mergedPopperProps = mergedPopperProps;
    }

    // Merge consumer and default components
    const mergedComponents = {
      ...defaultComponents,
      ...props.components
    };
    if (!shallowEqualObjects(mergedComponents, state.mergedComponents)) {
      newState.mergedComponents = mergedComponents;
    }
    if (!isEmpty(newState)) {
      return newState;
    }
    return null;
  }
  componentDidMount() {
    if (typeof window === 'undefined') {
      return;
    }
    this.unbindWindowClick = bind(window, {
      type: 'click',
      listener: this.handleClick,
      options: {
        capture: true
      }
    });
  }
  componentWillUnmount() {
    var _this$unbindWindowCli, _this$unbindWindowKey2;
    if (typeof window === 'undefined') {
      return;
    }
    (_this$unbindWindowCli = this.unbindWindowClick) === null || _this$unbindWindowCli === void 0 ? void 0 : _this$unbindWindowCli.call(this);
    this.unbindWindowClick = null;
    (_this$unbindWindowKey2 = this.unbindWindowKeydown) === null || _this$unbindWindowKey2 === void 0 ? void 0 : _this$unbindWindowKey2.call(this);
    this.unbindWindowKeydown = null;
  }
  componentDidUpdate(prevProps) {
    const {
      isOpen
    } = this.props;
    if (prevProps.isOpen !== isOpen) {
      if (isOpen === true) {
        this.open({
          controlOverride: true
        });
      } else if (isOpen === false) {
        this.close({
          controlOverride: true
        });
      }
    }
  }
  render() {
    const {
      target
    } = this.props;
    const {
      isOpen
    } = this.state;
    return /*#__PURE__*/React.createElement(Manager, null, /*#__PURE__*/React.createElement(Reference, null, ({
      ref
    }) => target && target({
      isOpen,
      ref: this.resolveTargetRef(ref),
      'aria-haspopup': 'true',
      'aria-expanded': isOpen,
      'aria-controls': isOpen ? this.popperWrapperId : undefined
    })), this.renderSelect());
  }
}
_defineProperty(PopupSelect, "defaultProps", {
  closeMenuOnSelect: true,
  components: {},
  maxMenuHeight: 300,
  maxMenuWidth: 440,
  minMenuWidth: 220,
  popperProps: {},
  isSearchable: true,
  searchThreshold: 5,
  styles: {},
  options: []
});