"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _react = require("react");
var _react2 = require("@emotion/react");
var _dsExplorations = require("@atlaskit/ds-explorations");
var _components = require("@atlaskit/theme/components");
var _hooks = require("../hooks");
var _context = require("../internal/context");
var _styles = require("../internal/styles");
var _utils = require("../internal/utils");
/** @jsx jsx */

/**
 * __TabList__
 *
 * A TabList groups `Tab` components together.
 *
 * - [Examples](https://atlassian.design/components/tabs/examples)
 * - [Code](https://atlassian.design/components/tabs/code)
 * - [Usage](https://atlassian.design/components/tabs/usage)
 */
var TabList = function TabList(props) {
  var children = props.children;
  var _useGlobalTheme = (0, _components.useGlobalTheme)(),
    mode = _useGlobalTheme.mode;
  var _useTabList = (0, _hooks.useTabList)(),
    tabsId = _useTabList.tabsId,
    selected = _useTabList.selected,
    onChange = _useTabList.onChange;
  var ref = /*#__PURE__*/(0, _react.createRef)();

  // Don't include any conditional children
  var childrenArray = _react.Children.toArray(children).filter(Boolean);
  var length = childrenArray.length;
  var tabListStyles = (0, _react.useMemo)(function () {
    return (0, _styles.getTabListStyles)(mode);
  }, [mode]);
  var selectTabByIndex = (0, _react.useCallback)(function (index) {
    var _ref$current;
    var newSelectedNode = (_ref$current = ref.current) === null || _ref$current === void 0 ? void 0 : _ref$current.querySelector("[id='".concat(tabsId, "-").concat(index, "']"));
    if (newSelectedNode) {
      newSelectedNode.focus();
    }
    onChange(index);
  }, [tabsId, ref, onChange]);
  var onKeyDown = (0, _react.useCallback)(function (e) {
    if (!['ArrowRight', 'ArrowLeft', 'Home', 'End'].includes(e.key)) {
      return;
    }

    // preventing horizontal or vertical scroll
    e.preventDefault();
    var lastTabIndex = length - 1;
    if (['Home', 'End'].includes(e.key)) {
      var _newSelected = e.key === 'Home' ? 0 : lastTabIndex;
      selectTabByIndex(_newSelected);
      return;
    }

    // We use aria-posinset so we don't rely on the selected variable
    // If we used the selected variable this would regenerate each time
    // and create an unstable reference
    var selectedIndex = parseInt(e.currentTarget.getAttribute('aria-posinset') || '0') - 1;
    var modifier = e.key === 'ArrowRight' ? 1 : -1;
    var newSelected = selectedIndex + modifier;
    if (newSelected < 0 || newSelected >= length) {
      // Cycling focus to move from last to first and from first to last
      newSelected = newSelected < 0 ? lastTabIndex : 0;
    }
    selectTabByIndex(newSelected);
  }, [length, selectTabByIndex]);

  // Memoized so the function isn't recreated each time
  var getTabWithContext = (0, _react.useCallback)(function (_ref) {
    var tab = _ref.tab,
      isSelected = _ref.isSelected,
      index = _ref.index;
    return (0, _react2.jsx)(_context.TabContext.Provider, {
      value: {
        onClick: function onClick() {
          return onChange(index);
        },
        onMouseDown: _utils.onMouseDownBlur,
        onKeyDown: onKeyDown,
        'aria-setsize': length,
        role: 'tab',
        id: "".concat(tabsId, "-").concat(index),
        'aria-posinset': index + 1,
        'aria-selected': isSelected,
        'aria-controls': "".concat(tabsId, "-").concat(index, "-tab"),
        tabIndex: isSelected ? 0 : -1
      },
      key: index
    }, tab);
  }, [length, onKeyDown, onChange, tabsId]);
  return (
    // Only styles that affect the TabList itself have been applied via primitives.
    // The other styles applied through the CSS prop are there for styling children
    // through inheritance. This is important for custom cases that use the useTab(),
    // which applies accessibility atributes that we use as a styling hook.
    (0, _react2.jsx)(_dsExplorations.UNSAFE_Box, {
      as: "div",
      role: "tablist",
      display: "flex",
      position: "relative",
      padding: "space.0",
      ref: ref
      // eslint-disable-next-line @atlaskit/design-system/consistent-css-prop-usage
      ,
      css: tabListStyles
    }, childrenArray.map(function (child, index) {
      return getTabWithContext({
        tab: child,
        index: index,
        isSelected: index === selected
      });
    }))
  );
};
var _default = TabList;
exports.default = _default;