/** @jsx jsx */
import { Children, createRef, useCallback, useMemo } from 'react';
import { jsx } from '@emotion/react';
import { UNSAFE_Box as Box } from '@atlaskit/ds-explorations';
import { useGlobalTheme } from '@atlaskit/theme/components';
import { useTabList } from '../hooks';
import { TabContext } from '../internal/context';
import { getTabListStyles } from '../internal/styles';
import { onMouseDownBlur } from '../internal/utils';
/**
 * __TabList__
 *
 * A TabList groups `Tab` components together.
 *
 * - [Examples](https://atlassian.design/components/tabs/examples)
 * - [Code](https://atlassian.design/components/tabs/code)
 * - [Usage](https://atlassian.design/components/tabs/usage)
 */
var TabList = function TabList(props) {
  var children = props.children;
  var _useGlobalTheme = useGlobalTheme(),
    mode = _useGlobalTheme.mode;
  var _useTabList = useTabList(),
    tabsId = _useTabList.tabsId,
    selected = _useTabList.selected,
    onChange = _useTabList.onChange;
  var ref = /*#__PURE__*/createRef();

  // Don't include any conditional children
  var childrenArray = Children.toArray(children).filter(Boolean);
  var length = childrenArray.length;
  var tabListStyles = useMemo(function () {
    return getTabListStyles(mode);
  }, [mode]);
  var selectTabByIndex = useCallback(function (index) {
    var _ref$current;
    var newSelectedNode = (_ref$current = ref.current) === null || _ref$current === void 0 ? void 0 : _ref$current.querySelector("[id='".concat(tabsId, "-").concat(index, "']"));
    if (newSelectedNode) {
      newSelectedNode.focus();
    }
    onChange(index);
  }, [tabsId, ref, onChange]);
  var onKeyDown = useCallback(function (e) {
    if (!['ArrowRight', 'ArrowLeft', 'Home', 'End'].includes(e.key)) {
      return;
    }

    // preventing horizontal or vertical scroll
    e.preventDefault();
    var lastTabIndex = length - 1;
    if (['Home', 'End'].includes(e.key)) {
      var _newSelected = e.key === 'Home' ? 0 : lastTabIndex;
      selectTabByIndex(_newSelected);
      return;
    }

    // We use aria-posinset so we don't rely on the selected variable
    // If we used the selected variable this would regenerate each time
    // and create an unstable reference
    var selectedIndex = parseInt(e.currentTarget.getAttribute('aria-posinset') || '0') - 1;
    var modifier = e.key === 'ArrowRight' ? 1 : -1;
    var newSelected = selectedIndex + modifier;
    if (newSelected < 0 || newSelected >= length) {
      // Cycling focus to move from last to first and from first to last
      newSelected = newSelected < 0 ? lastTabIndex : 0;
    }
    selectTabByIndex(newSelected);
  }, [length, selectTabByIndex]);

  // Memoized so the function isn't recreated each time
  var getTabWithContext = useCallback(function (_ref) {
    var tab = _ref.tab,
      isSelected = _ref.isSelected,
      index = _ref.index;
    return jsx(TabContext.Provider, {
      value: {
        onClick: function onClick() {
          return onChange(index);
        },
        onMouseDown: onMouseDownBlur,
        onKeyDown: onKeyDown,
        'aria-setsize': length,
        role: 'tab',
        id: "".concat(tabsId, "-").concat(index),
        'aria-posinset': index + 1,
        'aria-selected': isSelected,
        'aria-controls': "".concat(tabsId, "-").concat(index, "-tab"),
        tabIndex: isSelected ? 0 : -1
      },
      key: index
    }, tab);
  }, [length, onKeyDown, onChange, tabsId]);
  return (
    // Only styles that affect the TabList itself have been applied via primitives.
    // The other styles applied through the CSS prop are there for styling children
    // through inheritance. This is important for custom cases that use the useTab(),
    // which applies accessibility atributes that we use as a styling hook.
    jsx(Box, {
      as: "div",
      role: "tablist",
      display: "flex",
      position: "relative",
      padding: "space.0",
      ref: ref
      // eslint-disable-next-line @atlaskit/design-system/consistent-css-prop-usage
      ,
      css: tabListStyles
    }, childrenArray.map(function (child, index) {
      return getTabWithContext({
        tab: child,
        index: index,
        isSelected: index === selected
      });
    }))
  );
};
export default TabList;