import _extends from "@babel/runtime/helpers/extends";
/** @jsx jsx */
import React, { forwardRef, memo, useCallback, useEffect, useMemo, useRef } from 'react';
import { css, jsx } from '@emotion/react';
import { usePlatformLeafEventHandler } from '@atlaskit/analytics-next';
import { useGlobalTheme } from '@atlaskit/theme/components';
import { borderWidth, getBaseStyles, themeStyles } from './styles';
import { Theme } from './theme';
const packageName = "@atlaskit/textarea";
const packageVersion = "4.7.4";
const analyticsParams = {
  componentName: 'textArea',
  packageName,
  packageVersion
};
const setSmartHeight = el => {
  // Always reset height to auto before calculating new height
  el.style.height = 'auto';
  const borderHeight = borderWidth;
  const paddingBoxHeight = el.scrollHeight;
  const borderBoxHeight = paddingBoxHeight + borderHeight * 2;
  el.style.height = `${borderBoxHeight}px`;
};
const TextAreaWithTokens = /*#__PURE__*/forwardRef((props, ref) => {
  const ourRef = useRef(null);
  const {
    resize = 'smart',
    appearance = 'standard',
    isCompact = false,
    isRequired = false,
    isReadOnly = false,
    isDisabled = false,
    isInvalid = false,
    isMonospaced = false,
    minimumRows = 2,
    theme,
    testId,
    maxHeight = '50vh',
    onBlur,
    onFocus,
    onChange,
    tokens,
    value,
    ...rest
  } = props;
  useEffect(() => {
    const el = ourRef.current;
    if (resize === 'smart' && el) {
      setSmartHeight(el);
    }
  }, [resize, value]);
  const onBlurWithAnalytics = usePlatformLeafEventHandler({
    fn: event => {
      onBlur && onBlur(event);
    },
    action: 'blurred',
    ...analyticsParams
  });
  const onFocusWithAnalytics = usePlatformLeafEventHandler({
    fn: event => {
      onFocus && onFocus(event);
    },
    action: 'focused',
    ...analyticsParams
  });
  const getTextAreaRef = elementRef => {
    ourRef.current = elementRef;
    if (ref && typeof ref === 'object') {
      // @ts-ignore
      ref.current = elementRef;
    }
    if (ref && typeof ref === 'function') {
      ref(elementRef);
    }
  };
  const handleOnChange = useCallback(e => {
    const el = ourRef.current;
    if (resize === 'smart' && el) {
      setSmartHeight(el);
    }
    onChange && onChange(e);
  }, [onChange, resize]);
  const controlProps = {
    'data-invalid': isInvalid ? isInvalid : undefined,
    'data-compact': isCompact ? isCompact : undefined,
    'data-testid': testId ? testId : undefined
  };
  const baseStyles = useMemo(() => getBaseStyles({
    minimumRows,
    resize,
    appearance,
    isMonospaced,
    maxHeight
  }), [minimumRows, resize, appearance, isMonospaced, maxHeight]);
  const textAreaStyles = css([baseStyles,
  // not memoizing themeStyles as `tokens` is an unstable reference
  themeStyles(tokens, appearance)]);
  return jsx("textarea", _extends({}, controlProps, {
    value: value,
    disabled: isDisabled,
    readOnly: isReadOnly,
    required: isRequired,
    ref: getTextAreaRef,
    onChange: handleOnChange,
    onBlur: onBlurWithAnalytics,
    onFocus: onFocusWithAnalytics,
    rows: minimumRows
    // TODO refactor to follow emotion styling rules
    // see: https://product-fabric.atlassian.net/browse/DSP-6060
    // eslint-disable-next-line @atlaskit/design-system/consistent-css-prop-usage
    ,
    css: textAreaStyles
  }, rest));
});

/**
 * __Text area__
 *
 * A text area lets users enter long form text which spans over multiple lines.
 *
 * - [Examples](https://atlassian.design/components/textarea/examples)
 * - [Code](https://atlassian.design/components/textarea/code)
 * - [Usage](https://atlassian.design/components/textarea/usage)
 */
const TextArea = /*#__PURE__*/memo( /*#__PURE__*/forwardRef(function TextArea(props, ref) {
  const {
    mode
  } = useGlobalTheme();
  return jsx(Theme.Provider, {
    value: props.theme
  }, jsx(Theme.Consumer, {
    appearance: props.appearance || 'standard',
    mode: mode
  }, tokens => jsx(TextAreaWithTokens, _extends({
    ref: ref
  }, props, {
    tokens: tokens
  }))));
}));
TextArea.displayName = 'TextArea';
export default TextArea;