import _extends from "@babel/runtime/helpers/extends";
/** @jsx jsx */
import { forwardRef, memo, useMemo, useState } from 'react';
import { jsx } from '@emotion/react';
import { usePlatformLeafEventHandler } from '@atlaskit/analytics-next';
import __noop from '@atlaskit/ds-lib/noop';
import CloseIcon from '@atlaskit/icon/glyph/editor/close';
import CheckIcon from '@atlaskit/icon/glyph/editor/done';
import { useGlobalTheme } from '@atlaskit/theme/components';
import { getStyles } from './internal/styles';
export const getIconSize = size => size === 'large' ? 'large' : 'small';
const noop = __noop;
const analyticsAttributes = {
  componentName: 'toggle',
  packageName: "@atlaskit/toggle",
  packageVersion: "12.6.9"
};

/**
 * __Toggle__
 *
 * A toggle is used to view or switch between enabled or disabled states.
 *
 * - [Examples](https://atlassian.design/components/toggle/examples)
 * - [Code](https://atlassian.design/components/toggle/code)
 * - [Usage](https://atlassian.design/components/toggle/usage)
 */
const Toggle = /*#__PURE__*/memo( /*#__PURE__*/forwardRef((props, ref) => {
  const {
    defaultChecked = false,
    isDisabled = false,
    onBlur: providedOnBlur = noop,
    onChange: providedChange = noop,
    onFocus: providedFocus = noop,
    size = 'regular',
    name = '',
    value = '',
    isChecked,
    analyticsContext,
    id,
    testId,
    label
  } = props;
  const {
    mode
  } = useGlobalTheme();
  const isControlled = typeof isChecked === 'undefined';
  const [checked, setChecked] = useState(defaultChecked);
  const handleBlur = usePlatformLeafEventHandler({
    fn: providedOnBlur,
    action: 'blur',
    analyticsData: analyticsContext,
    ...analyticsAttributes
  });
  const handleFocus = usePlatformLeafEventHandler({
    fn: providedFocus,
    action: 'focus',
    analyticsData: analyticsContext,
    ...analyticsAttributes
  });
  const handleChange = usePlatformLeafEventHandler({
    fn: (event, analyticsEvent) => {
      if (isControlled) {
        setChecked(checked => !checked);
      }
      return providedChange(event, analyticsEvent);
    },
    action: 'change',
    analyticsData: analyticsContext,
    ...analyticsAttributes
  });
  const shouldChecked = isControlled ? checked : isChecked;
  const controlProps = {
    'data-checked': shouldChecked ? shouldChecked : undefined,
    'data-disabled': isDisabled ? isDisabled : undefined,
    'data-size': size,
    'data-testid': testId ? testId : undefined
  };
  const toggleStyles = useMemo(() => getStyles(size, mode), [size, mode]);
  return jsx("label", _extends({}, controlProps, {
    css: toggleStyles
  }), jsx("input", {
    ref: ref,
    checked: shouldChecked,
    disabled: isDisabled,
    id: id,
    name: name,
    onBlur: handleBlur,
    onChange: handleChange,
    onFocus: handleFocus,
    type: "checkbox",
    value: value,
    "data-testid": testId && `${testId}--input`,
    "aria-label": label
  }), jsx(CheckIcon, {
    label: "",
    size: getIconSize(size),
    testId: testId && `${testId}--toggle-check-icon`
  }), jsx(CloseIcon, {
    label: "",
    size: getIconSize(size),
    testId: testId && `${testId}--toggle-cross-icon`
  }));
}));
Toggle.displayName = 'Toggle';
export default Toggle;