import { COLOR_MODE_ATTRIBUTE, CUSTOM_THEME_ATTRIBUTE } from './constants';
import { themeStateDefaults } from './set-global-theme';
import { limitSizeOfCustomStyleElements, reduceTokenMap } from './utils/custom-theme-loading-utils';
import { generateColors, generateTokenMapWithContrastCheck } from './utils/generate-custom-color-ramp';
import { hash } from './utils/hash';
export const CUSTOM_STYLE_ELEMENTS_SIZE_THRESHOLD = 10;
/**
 *
 * @param themeSchema The schema of available themes
 * @returns a string with the CSS for the custom theme
 */
/**
 * Takes a color mode and custom branding options, and returns an array of objects for use in applying custom styles to the document head.
 * Only supplies the color themes necessary for initial render, based on the current themeState. I.e. if in light mode, dark mode themes are not returned.
 *
 * @param {Object<string, string>} themeState The themes and color mode that should be applied.
 * @param {string} themeState.colorMode Determines which color theme is applied
 * @param {Object} themeState.UNSAFE_themeOptions The custom branding options to be used for custom theme generation
 *
 * @returns A Promise of an object array, containing theme IDs, data-attributes to attach to the theme, and the theme CSS.
 * If an error is encountered while loading themes, the themes array will be empty.
 */
export async function getCustomThemeStyles(themeState) {
  var _themeState$UNSAFE_th;
  const brandColor = themeState === null || themeState === void 0 ? void 0 : (_themeState$UNSAFE_th = themeState.UNSAFE_themeOptions) === null || _themeState$UNSAFE_th === void 0 ? void 0 : _themeState$UNSAFE_th.brandColor;
  const mode = (themeState === null || themeState === void 0 ? void 0 : themeState.colorMode) || themeStateDefaults['colorMode'];
  const optionString = JSON.stringify(themeState === null || themeState === void 0 ? void 0 : themeState.UNSAFE_themeOptions);
  const uniqueId = hash(optionString);
  const themeRamp = generateColors(brandColor);

  // outputs object to generate to CSS from
  const themes = [];
  const tokenMaps = generateTokenMapWithContrastCheck(brandColor, mode, themeRamp);
  if ((mode === 'light' || mode === 'auto') && tokenMaps.light) {
    // Light mode theming
    themes.push({
      id: 'light',
      attrs: {
        'data-theme': 'light',
        'data-custom-theme': uniqueId
      },
      css: `
html[${CUSTOM_THEME_ATTRIBUTE}="${uniqueId}"][${COLOR_MODE_ATTRIBUTE}="light"][data-theme~="light:light"] {
  /* Branded tokens */
    ${reduceTokenMap(tokenMaps.light, themeRamp)}
}`
    });
  }
  if ((mode === 'dark' || mode === 'auto') && tokenMaps.dark) {
    // Dark mode theming
    themes.push({
      id: 'dark',
      attrs: {
        'data-theme': 'dark',
        'data-custom-theme': uniqueId
      },
      css: `
html[${CUSTOM_THEME_ATTRIBUTE}="${uniqueId}"][${COLOR_MODE_ATTRIBUTE}="dark"][data-theme~="dark:dark"] {
  /* Branded tokens */
    ${reduceTokenMap(tokenMaps.dark, themeRamp)}
}`
    });
  }
  return themes;
}
export async function loadAndAppendCustomThemeCss(themeState) {
  getCustomThemeStyles(themeState).then(themes => {
    limitSizeOfCustomStyleElements(CUSTOM_STYLE_ELEMENTS_SIZE_THRESHOLD);
    themes.map(theme => {
      const styleTag = document.createElement('style');
      document.head.appendChild(styleTag);
      styleTag.dataset.theme = theme.attrs['data-theme'];
      styleTag.dataset.customTheme = theme.attrs['data-custom-theme'];
      styleTag.textContent = theme.css;
    });
  });
}