import * as t from '@babel/types';
import tokenDefaultValues from '../artifacts/token-default-values';
import tokenNames from '../artifacts/token-names';
export default function plugin() {
  return {
    visitor: {
      Program: {
        enter: function enter(path, state) {
          var sourceFile = path.hub.file.opts.filename;
          if (sourceFile && sourceFile.includes('node_modules')) {
            return;
          }
          path.traverse({
            CallExpression: function (_CallExpression) {
              function CallExpression(_x) {
                return _CallExpression.apply(this, arguments);
              }
              CallExpression.toString = function () {
                return _CallExpression.toString();
              };
              return CallExpression;
            }(function (path) {
              var tokenImportScope = getTokenImportScope(path);
              if (!tokenImportScope) {
                return;
              }

              // Check arguments have correct format
              if (!path.node.arguments[0]) {
                throw new Error("token() requires at least one argument");
              } else if (!t.isStringLiteral(path.node.arguments[0])) {
                throw new Error("token() must have a string as the first argument");
              } else if (path.node.arguments.length > 2) {
                throw new Error("token() does not accept ".concat(path.node.arguments.length, " arguments"));
              }

              // Check the token exists
              var tokenName = path.node.arguments[0].value;
              var cssTokenValue = tokenNames[tokenName];
              if (!cssTokenValue) {
                throw new Error("token '".concat(tokenName, "' does not exist"));
              }
              var replacementNode;

              // if no fallback is set, optionally find one from the default theme
              if (path.node.arguments.length < 2) {
                if (state.opts.shouldUseAutoFallback) {
                  replacementNode = t.stringLiteral("var(".concat(cssTokenValue, ", ").concat(getDefaultFallback(tokenName), ")"));
                } else {
                  replacementNode = t.stringLiteral("var(".concat(cssTokenValue, ")"));
                }
              }

              // Handle fallbacks
              var fallback = path.node.arguments[1];
              if (t.isStringLiteral(fallback)) {
                // String literals can be concatenated into css variable call
                // Empty string fallbacks are ignored. For now, as the user did specify a fallback, no default is inserted
                replacementNode = t.stringLiteral(fallback.value ? "var(".concat(cssTokenValue, ", ").concat(fallback.value, ")") : "var(".concat(cssTokenValue, ")"));
              } else if (t.isExpression(fallback)) {
                // Expressions should be placed in a template string/literal
                replacementNode = t.templateLiteral([t.templateElement({
                  cooked: "var(".concat(cssTokenValue, ", "),
                  // Currently we create a "raw" value by inserting escape characters via regex (https://github.com/babel/babel/issues/9242)
                  raw: "var(".concat(cssTokenValue.replace(/\\|`|\${/g, '\\$&'), ", ")
                }, false), t.templateElement({
                  raw: ')',
                  cooked: ')'
                }, true)], [fallback]);
              }

              // Replace path and call scope.crawl() to refresh the scope bindings + references
              replacementNode && path.replaceWith(replacementNode);
              // @ts-ignore crawl is a valid property
              tokenImportScope.crawl();
            })
          });
        },
        exit: function exit(path) {
          path.traverse({
            ImportDeclaration: function ImportDeclaration(path) {
              // remove import of 'token'
              if (path.node.source.value !== '@atlaskit/tokens') {
                return;
              }
              path.get('specifiers').forEach(function (specifier) {
                if (!specifier.isImportSpecifier()) {
                  return;
                }
                if (getNonAliasedImportName(specifier.node) !== 'token') {
                  return;
                }
                var binding = path.scope.bindings[getAliasedImportName(specifier.node)];

                // if no longer used, remove
                if (!binding.referenced) {
                  specifier.remove();
                }
              });

              // remove '@atlaskit/tokens' import if it is no longer needed
              if (path.get('specifiers').length === 0) {
                path.remove();
              }
            }
          });
        }
      }
    }
  };
}
function getDefaultFallback(tokenName) {
  return tokenDefaultValues[tokenName];
}
function getNonAliasedImportName(node) {
  if (t.isIdentifier(node.imported)) {
    return node.imported.name;
  }
  return node.imported.value;
}
function getAliasedImportName(node) {
  return node.local.name;
}

/**
 * Determine if the current call is to a token function, and
 * return the relevant scope
 */
function getTokenImportScope(path) {
  var callee = path.node.callee;
  if (!t.isIdentifier(callee)) {
    return undefined;
  }
  var binding = getTokenBinding(path.scope, callee.name);
  if (!binding || !t.isImportSpecifier(binding.path.node)) {
    return undefined;
  }
  if (binding.path.parent && t.isImportDeclaration(binding.path.parent)) {
    if (binding.path.parent.source.value !== '@atlaskit/tokens') {
      return undefined;
    }
  }
  return getNonAliasedImportName(binding.path.node) === 'token' ? binding.scope : undefined;
}
function getTokenBinding(scope, name) {
  if (!scope) {
    return undefined;
  }
  if (scope.bindings[name]) {
    return scope.bindings[name];
  } else {
    return getTokenBinding(scope.parent, name);
  }
}